/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __ARTICLE_H__
#define __ARTICLE_H__

#include <time.h>

#include <glib/gtypes.h>
#include <glib/gslist.h>
#include <glib/garray.h>

#include <pan/base/group.h>
#include <pan/base/pan-callback.h>

#define ARTICLE(a) ((Article *)a)


enum {
	DECODE_STATE_NONE	= 0,
	DECODE_STATE_DECODED	= 1,
	DECODE_STATE_FAILED	= 2
};

enum {
	MULTIPART_STATE_NONE    = 0,
	MULTIPART_STATE_SOME    = 1,
	MULTIPART_STATE_ALL     = 2
};

typedef struct _Article
{
        guint is_new          : 1;    /* true if article is 'new' */
	guint passes_filter   : 1;    /* articlelist */
	guint error_flag      : 1;    /* error downloading this article */
	guint8 multipart_state;       /* none, some, or all */
	guint8 decode_state;          /* none, decoded, or failed */
	gint16 part;		      /* part #; 0 if no attachments */
	gint16 parts;		      /* # of parts in multipart message */
	gint16 score;		      /* scoring */
	guint16 linecount;	      /* # of lines in the body */
	guint16 unread_children;      /* articlelist: unread kids in thread */
	guint16 new_children;         /* articlelist: new kids in thread */
	gulong byte_qty;              /* # of bytes in the body */
	time_t score_date;            /* when this article was last scored */
	time_t date;                  /* date posted, as parsed into time_t */
	GSList * threads;             /* articlelist: list of child threads */
	struct _Article * parent;     /* articlelist: parent in threading */
	const char * xref;            /* article's xref header from XOVER */
	const char * subject;         /* article's subject */
	const char * author_addr;     /* article's author -- address*/
	const char * message_id;      /* article's message-id */
	const char * author_real;     /* (maybe NULL) article's author -- real name */
	const char * references;      /* (maybe NULL) references string for this article */

	/* At some point this may be an array for 'article-centric' */
	gulong number;               /* (PROTECTED) number for article */
	Group * group;               /* (PROTECTED) group this article is in */
}
Article;

/* Headers stored in their own fields but here for completeness */
#define HEADER_REFERENCES         "References"
#define HEADER_SUBJECT            "Subject"
#define HEADER_FROM               "From"
#define HEADER_MESSAGE_ID         "Message-Id"
#define HEADER_DATE               "Date"
#define HEADER_XREF               "Xref"
/* Headers not stored (must be retrieved from the body). */
#define HEADER_TO                 "To"
#define HEADER_ORGANIZATION       "Organization"
#define HEADER_SUPERSEDES         "Supersedes"
#define HEADER_FOLLOWUP_TO        "Followup-To"
#define HEADER_NEWSGROUPS         "Newsgroups"
#define HEADER_REPLY_TO           "Reply-To"
#define HEADER_USER_AGENT         "User-Agent"
#define HEADER_X_NEWSREADER       "X-Newsreader"
#define HEADER_X_MAILER           "X-Mailer"
/* Headers beginning with X-Pan- are for internal use. */
#define PAN_ATTRIBUTION           "X-Pan-Internal-Attribution"
#define PAN_ATTACH_FILE           "X-Pan-Internal-Attach-File"
#define PAN_SERVER                "X-Pan-Internal-Post-Server"
#define PAN_LINES_PER_PART        "X-Pan-Internal-Lines-Per-Part"
#define PAN_CHARSET               "X-Pan-Internal-CharSet"
#define PAN_NEWSGROUPS            "X-Pan-Internal-Sendlater-Newsgroups"
#define PAN_TO                    "X-Pan-Internal-Sendlater-To"

/**
***  PUBLIC LIFE CYCLE
**/

Article*     article_new                      (Group          * group);

void         article_destructor               (Article        * article);

/**
***  FLAGS
**/

int          article_get_decode_state         (const Article  * article);

void         articles_set_decode_state        (Article       ** articles,
                                               int              qty,
                                               int              decode_state);

int          article_get_multipart_state      (const Article  * article);

void         articles_set_multipart_state     (Article       ** articles,
                                               int              qty,
                                               int              multipart_state);

void         article_set_error_flag           (Article        * article,
                                               gboolean         error_flag);

/**
***  Headers
**/

const char*  article_get_subject              (const Article  * article);

void         article_set_subject              (Article        * article,
                                               const char     * subject);

const char*  article_get_message_id           (const Article  * article);

void         article_set_message_id           (Article        * article,
                                               const char     * message_id);

void         article_set_references           (Article        * article,
                                               const char     * references);

void         article_set_xrefs                (Article        * article,
                                               const char     * xrefs);

void         article_set_author               (Article        * article,
                                               const char     * header_from);

char*        article_get_author_str           (const Article  * article,
                                               char           * buf,
                                               int              bufsize);

char*        article_format_author_str        (const char     * author_addr,
                                               const char     * author_real,
                                               char           * buf,    
                                               int              bufsize);

int          article_format_short_author_str  (const char     * author_addr,
                                               const char     * author_real,
                                               char           * buf,    
                                               int              bufsize);

int          article_get_short_author_str     (const Article  * article,
                                               char           * buf,
                                               int              bufsize);

/**
 * Return the number of crossposts in the xref header,
 * or 1 if no xref header is present.
 */
int          article_get_crosspost_qty        (const Article  * article);

void         article_set_from_g_mime_message  (Article        * setme_article,
                                               GMimeMessage   * source);

Article*     article_get_root                 (const Article  * article);

/**
 * Returns TRUE if the header is one only meant for Pan's internal bookkeeping
 * and should never be shown to the user.
 */
gboolean     article_header_is_internal       (const char     * key);

/**
 * This is a wart specific to message-window which should eventually be
 * moved out of article.h.  It returns true if the header isn't
 * internal, nor is followup_to, newsgroups, organization, or reply_to.
 */
gboolean     article_header_is_extra          (const char     * key);


/**
 * This is the iterator function for article_xref_foreach.
 */
typedef void (ArticleXRefFunc)(Server*,Group*,gulong,gpointer);

/**
 * For each cross-reference specified in Article a's Xref: header,
 * the specified ArticleXRefFunc is invoked.
 *
 * If skip_group_a is TRUE, then the group "a->group" is not included
 * in this foreach, even if it's listed in the Xref header.
 */
void         article_xref_foreach             (const Article   * a,
                                               ArticleXRefFunc   func,
                                               gpointer          user_data,
                                               ServerGroupsType  set,
                                               gboolean          skip_group_a);

/**
***  Sanity Checks
**/

gboolean     article_is_valid                 (const Article  * article);

gboolean     articles_are_valid               (const Article ** articles,
                                               int              qty);

gboolean     articles_are_valid_in_group      (const Article ** articles,
                                               int              qty);


/**
***  READ
**/

/**
 * Returns true if this article is marked as read.
 */
gboolean     article_is_read                  (const Article  * article);

/**
 * Marks the specified articles as read or unread and fires a single
 * ARTICLE_CHANGED_READ event.
 */
void         articles_set_read_simple         (Article      ** articles,
                                               int             article_qty,
                                               gboolean        read);

/**
 * Marks the specified articles as read or unread, as well as parts
 * 2..n of any multiparts passed in in "articles", as well as any
 * crossposts, and fires a single ARTICLE_CHANGED_READ event for each
 * group in which Articles are changed.
 */
void         articles_set_read                (Article      ** articles,
                                               int             article_qty,
                                               gboolean        read);


/**
 * Returns true if this article is new, where new is defined as having
 * been retrieved from the news server the last time the user tried to
 * fetch new headers.
 */
gboolean     article_is_new                   (const Article  * article);

void         articles_set_new                 (Article       ** articles,
                                               int              article_qty,
                                               gboolean         is_new);

void         articles_set_dirty               (Article       ** articles,
                                               int              article_qty);

/**
***  THREADS
**/

/**
 * @param article the article whose thread is to be returned
 * @param GPtrArray a container to hold all the articles found in
 *                  this article's thread.
 */
void           article_get_thread            (const Article   * article,
                                              GPtrArray       * setme);

void           article_get_subthread         (const Article   * article,
                                              GPtrArray       * setme);

void           article_get_references        (const Article   * article,
                                              GPtrArray       * setme);

typedef enum
{
	GET_WHOLE_THREAD,
	GET_SUBTHREAD
}
ThreadGet;
 
GPtrArray*     article_get_unique_threads    (const GPtrArray    * articles,
                                              ThreadGet            thread_get);
 

char*          article_get_thread_message_id (const Article*);

extern const char * default_incoming_name_real;

extern const char * default_incoming_name_addr;

/***
****  Events
***/
 
typedef enum
{
	ARTICLE_CHANGED_READ,
	ARTICLE_CHANGED_NEW,
	ARTICLE_CHANGED_DIRTY
}
ArticleChangeType;
 
typedef struct
{
	Group * group;
	Article ** articles;
	int article_qty;
	ArticleChangeType type;
}
ArticleChangeEvent;
 
/**
 * @call_obj: ArticleChangeEvent*
 * @call_arg: NULL
 **/
PanCallback *     article_get_articles_changed_callback      (void);                                                            



#endif /* __ARTICLE_H__ */
