/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/identities/identity.h>
#include <pan/identities/identity-edit-ui.h>

#include <pan/pan-file-entry.h>
#include <pan/pan-charset-picker.h>

#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget * dialog;

	GtkWidget * name_entry;
	GtkWidget * default_news_tb;
	GtkWidget * default_mail_tb;

	GtkWidget * author_real_entry;
	GtkWidget * author_addr_entry;

	GtkWidget * reply_to_entry;
	GtkWidget * organization_entry;
	GtkWidget * signature_entry;
	GtkWidget * attribution_entry;
	GtkWidget * msg_id_fqdn_entry;
	GtkWidget * posting_charset_entry;

	GtkWidget * custom_headers_text;

	gchar     * identity_name;
}
IdentityEditDialog;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

extern GtkTooltips * ttips;

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
***********/

/*****
******
******  IDENTITY INFO
******
*****/

static GtkWidget*
create_identity_tab (IdentityEditDialog *d)
{
	GtkWidget * t;
	GtkWidget * w;
	GtkWidget * l;
	int row;
	char buf[512];

	row = 0;
	t = gtk_table_new (4, 15, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(t), 12);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

		/* title for profile info section */
		g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Profile Information"));
		l = gtk_label_new (buf);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
		++row;

		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+3, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+3, 0, 0, 0, 0);

		l = gtk_label_new_with_mnemonic (_("_Profile name:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->name_entry = gtk_entry_new ();
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		w = d->default_news_tb = gtk_check_button_new_with_mnemonic (_("Make this the default profile for _news"));
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		w = d->default_mail_tb = gtk_check_button_new_with_mnemonic (_("Make this the default profile for _mail"));
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	/* spacer between "profile information" and "required information" */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

		/* title for profile info section */
		g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Required Information"));
		l = gtk_label_new (buf);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
		++row;

		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+2, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+2, 0, 0, 0, 0);

		l = gtk_label_new_with_mnemonic (_("_Full Name:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->author_real_entry = gtk_entry_new ();
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		l = gtk_label_new_with_mnemonic (_("_Email Address:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->author_addr_entry = gtk_entry_new ();
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	/* spacer between "profile information" and "required information" */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

		/* title for profile info section */
		g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Optional Information"));
		l = gtk_label_new (buf);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
		++row;

		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+2, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+2, 0, 0, 0, 0);

		l = gtk_label_new_with_mnemonic (_("_Reply-To Header:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->reply_to_entry = gtk_entry_new ();
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		l = gtk_label_new_with_mnemonic (_("Or_ganization Header:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->organization_entry = gtk_entry_new ();
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		l = gtk_label_new_with_mnemonic (_("Message-ID _Domain Name:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->msg_id_fqdn_entry = gtk_entry_new ();
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, 
			_("The domain name to use for Message-ID generation"), "");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		l = gtk_label_new_with_mnemonic (_("_Attribution:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->attribution_entry = gtk_entry_new ();
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, 
			"%i for Message-ID\n"
			"%a for Author and Address\n"
			"%n for Author name\n%d for Date", "");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		l = gtk_label_new_with_mnemonic (_("_Signature File:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->signature_entry = pan_file_entry_new (_("Select Signature File"));
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		l = gtk_label_new_with_mnemonic (_("C_haracter Set:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = d->posting_charset_entry = pan_charset_picker_new ("");
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, 
			_("The default charset set for articles "
			  "posted with this profile"), "");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}

static void
populate_identity_tab (IdentityEditDialog * d,
                       const Identity     * id,
                       gboolean             default_news,
                       gboolean             default_mail)
{
	gboolean    b;
	GtkWidget * w;

	b = id!=NULL;

	/* Identity name */
	w = d->name_entry;
	pan_gtk_entry_set_text (w, b ? id->name : "");

	/* default news */
	w = d->default_news_tb;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), default_news);

	/* default mail */
	w = d->default_mail_tb;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), default_mail);

	/* author name */
	w = d->author_real_entry;
	pan_gtk_entry_set_text (w, 
		b && is_nonempty_string (id->author_real) ? 
		id->author_real : "");

	/* author address */
	w = d->author_addr_entry;
	pan_gtk_entry_set_text (w, 
		b && is_nonempty_string (id->author_addr) ?
		id->author_addr : "");

	/* reply to */
	w = d->reply_to_entry;
	pan_gtk_entry_set_text (w, 
		 b && is_nonempty_string (id->reply_to) ?
		id->reply_to : "");

	/* msg id fqdn */
	w = d->msg_id_fqdn_entry;
	pan_gtk_entry_set_text (w, 
		 b && is_nonempty_string (id->msg_id_fqdn) ?
		id->msg_id_fqdn : "");

	/* organization */
	w = d->organization_entry;
	pan_gtk_entry_set_text (w, 
		b && is_nonempty_string (id->organization) ?
		id->organization : "");

	/* attribution */
	w = d->attribution_entry;
	pan_gtk_entry_set_text (w, 
		b && is_nonempty_string (id->attribution) ?
		id->attribution : _("On %d, %n wrote:"));

	/* signature file */
	pan_file_entry_set (d->signature_entry, 
		b && is_nonempty_string (id->signature) ?
		id->signature : "");

	/* posting charset */
	w = d->posting_charset_entry;
	pan_charset_picker_set_charset (w,
		b && is_nonempty_string (id->posting_charset) ? 
		id->posting_charset : get_charset_from_locale ());
}

static void
identity_from_identity_tab (IdentityEditDialog * d, 
                            Identity           * id,
                            gboolean           * default_news,
                            gboolean           * default_mail)
{
	char * pch;
	const char * cpch;
	GtkWidget * w;

	g_return_if_fail (d!=NULL);
	g_return_if_fail (id!=NULL);
	g_return_if_fail (default_news!=NULL);
	g_return_if_fail (default_mail!=NULL);

	/* Identity name */
	w = d->name_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->name, pch);
	g_strstrip (id->name);

	/* default news */
	w = d->default_news_tb;
	*default_news = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	/* default mail */
	w = d->default_mail_tb;
	*default_mail = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	/* author name */
	w = d->author_real_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->author_real, pch);
	g_strstrip (id->author_real);

	/* author address */
	w = d->author_addr_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->author_addr, pch);
	g_strstrip (id->author_addr);

	/* reply to */
	w = d->reply_to_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->reply_to, pch);
	g_strstrip (id->reply_to);

	/* msg id fqdn */
	w = d->msg_id_fqdn_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->msg_id_fqdn, pch);
	g_strstrip (id->msg_id_fqdn);

	/* organization */
	w = d->organization_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->organization, pch);
	g_strstrip (id->organization);
	
	/* attribution */
	w = d->attribution_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->attribution, pch);
	g_strstrip (id->attribution);

	/* signature file */
	cpch = pan_file_entry_get (d->signature_entry);
	replace_gstr (&id->signature, g_strdup(cpch));
	g_strstrip (id->signature);

	/* posting charset */
	cpch = pan_charset_picker_get_charset (d->posting_charset_entry);
	replace_gstr (&id->posting_charset, g_strdup (cpch));
	g_strstrip (id->posting_charset);
}

/*****
******
******  CUSTOM HEADERS
******
*****/

static GtkWidget*
create_headers_tab (IdentityEditDialog *d)
{
	GtkWidget * h;
	GtkWidget * v;
	GtkWidget * w;
	GtkWidget * l;
	GtkWidget * scroll;
	char buf[512];

	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), 12);

		g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Extra Headers"));
		l = gtk_label_new (buf);
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
		gtk_box_pack_start (GTK_BOX(v), l, FALSE, FALSE, 0);

		h = gtk_hbox_new (FALSE, GUI_PAD);
		gtk_box_pack_start (GTK_BOX(v), h, TRUE, TRUE, 0);

			w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
			gtk_widget_set_usize (w, 18u, 0u);
			gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);

			w = d->custom_headers_text = gtk_text_view_new ();
			scroll = gtk_scrolled_window_new (NULL, NULL);
			gtk_container_set_border_width (GTK_CONTAINER(scroll), GUI_PAD_SMALL);
			gtk_container_add (GTK_CONTAINER(scroll), w);
			gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
							GTK_POLICY_AUTOMATIC,
							GTK_POLICY_AUTOMATIC);
			w = gtk_frame_new (NULL);
			gtk_frame_set_shadow_type (GTK_FRAME(w), GTK_SHADOW_IN);
			gtk_container_add (GTK_CONTAINER(w), scroll);

			gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	
	return v;
}

static void
populate_headers_tab (IdentityEditDialog *d, const Identity *id)
{
	g_return_if_fail (d!=NULL);

	if (id!=NULL && id->custom_headers)
	{
		GtkTextBuffer * buffer;
		GtkTextIter     start;
		GtkTextIter     end;
		GString       * headers = g_string_new (NULL);
		guint           i;

		for (i=0; i!=id->custom_headers->len; ++i) {
			const Header * h = g_ptr_array_index (id->custom_headers, i);
			g_string_append_printf (headers, "%s: %s\n", h->name, h->value);
		}

		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(d->custom_headers_text));
		gtk_text_buffer_get_bounds (buffer, &start, &end);
		gtk_text_buffer_delete (buffer, &start, &end);
		gtk_text_buffer_insert (buffer, &start, headers->str, headers->len);

		g_string_free (headers, TRUE);
	}
	
}

static void
identity_from_headers_tab (IdentityEditDialog *d, Identity *id)
{
	char  * text;
	int     i;
	char ** lines;

	g_return_if_fail (d!=NULL);
	g_return_if_fail (id!=NULL);
	g_return_if_fail (id->custom_headers!=NULL);

	/* Clear all current custom headers */
	for (i=0; i<id->custom_headers->len; i++)
	{
		Header * h = g_ptr_array_index (id->custom_headers, i);

		g_free (h->name);
		g_free (h->value);
		g_free (h);
		g_ptr_array_remove (id->custom_headers, h);
	}

	/* get custom headers from textview */
	if (1)
	{
		GtkTextBuffer * buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(d->custom_headers_text));
		GtkTextIter start;
		GtkTextIter end;
		gtk_text_buffer_get_bounds (buffer, &start, &end);
		text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	}

	lines = g_strsplit (text, "\n", -1);
	for (i=0; lines!=NULL && lines[i]!=NULL; i++)
	{
		gchar * delimit;

		g_strstrip (lines[i]);
		delimit = strchr (lines[i], ':');

		if (delimit != NULL)
		{
			Header * h = g_new0 (Header, 1);
			gchar  * n = g_strndup (lines[i], delimit - lines[i]);
			gchar  * v = g_strdup (delimit + 1);

			g_strstrip (n);
			g_strstrip (v);

			h->name = n;
			h->value = v;

			g_ptr_array_add (id->custom_headers, h);
		}
	}

	/* cleanup */
	g_free (text);
	g_strfreev (lines);
}

/*****
******
******  
******
*****/

static gint
dialog_destroy_cb (GtkDialog * dialog, gpointer data)
{
	IdentityEditDialog * d = (IdentityEditDialog*) data;
	g_free (d->identity_name);
	g_free (d);
	return FALSE;
}
 
GtkWidget*
identity_edit_dialog_new (GtkWindow      * window,
                          const Identity * id,
                          gboolean         default_news,
                          gboolean         default_mail)
{
	GtkWidget *w, *n, *d_v;
	IdentityEditDialog * d;

	d = g_new0 (IdentityEditDialog, 1);

	/* dialog */

	w = d->dialog = gtk_dialog_new_with_buttons (_("Pan: Edit Profile"), window,
	                                             GTK_DIALOG_DESTROY_WITH_PARENT,
	                                             GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                             GTK_STOCK_OK, GTK_RESPONSE_OK,
	                                             NULL);
	d_v = GTK_DIALOG(w)->vbox;
	gtk_window_set_policy (GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_signal_connect (GTK_OBJECT(w), "destroy", GTK_SIGNAL_FUNC(dialog_destroy_cb), d);
	gtk_object_set_user_data (GTK_OBJECT(w), d);

	/* notebook */
	n = gtk_notebook_new ();
	w = create_identity_tab (d);
	gtk_container_add (GTK_CONTAINER(n), w);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK(n), w, gtk_label_new_with_mnemonic (_("_User Profile")));
	w = create_headers_tab (d);
	gtk_container_add (GTK_CONTAINER(n), w);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK(n), w, gtk_label_new_with_mnemonic (_("Cus_tom Headers")));
	gtk_container_set_border_width (GTK_CONTAINER(n), 12);
	gtk_box_pack_start (GTK_BOX(d_v), n, TRUE, TRUE, GUI_PAD_SMALL);

	/* populate */
	populate_identity_tab (d, id, default_news, default_mail);
	populate_headers_tab (d, id);

	return d->dialog;
}

Identity*
identity_edit_dialog_get_identity (GtkWidget * w, 
                                   gboolean  * default_news,
                                   gboolean  * default_mail)
{
	Identity           * id = NULL;
	IdentityEditDialog * d  = 
		(IdentityEditDialog*) gtk_object_get_user_data (GTK_OBJECT(w));
	
	id = identity_new ();
	identity_from_identity_tab (d, id, default_news, default_mail);
	identity_from_headers_tab (d, id);

	/* make sure user didn't skip on any of the essential parts */
	if (!identity_is_valid (id)) {
		pan_object_unref (PAN_OBJECT(id));
		id = NULL;
	}

	/* double-quotes in the real name aren't allowed: translate to single */
	if (id!=NULL && pan_strstr (id->author_real, "\"") != NULL) {

		gchar * new_author_real;
	       
		new_author_real = pan_substitute (id->author_real, "\"", "'");
		replace_gstr (&id->author_real, new_author_real);

		pan_info_dialog (_("Your real name contained double quotes.\nConverting into single quotes."));
	}

	return id;
}
