/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/messageidset.h>

struct MsgIdSet
{
	GHashTable * hash;
	GMutex * lock;
};

MessageIdSet*
messageidset_new (void)
{
	MessageIdSet * mset = g_new0 (MessageIdSet, 1);
	mset->lock = g_mutex_new ();
	mset->hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	return mset;
}

void
messageidset_free (MessageIdSet * mset)
{
	g_return_if_fail (mset != NULL);

	g_hash_table_destroy (mset->hash);
	g_mutex_free (mset->lock);
	g_free (mset);
}

void
messageidset_clear (MessageIdSet * mset)
{
	g_return_if_fail (mset != NULL);

	g_mutex_lock (mset->lock);
	g_hash_table_destroy (mset->hash);
	mset->hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	g_mutex_unlock (mset->lock);
}

static void
maybe_add (MessageIdSet * mset, const char * message_id)
{
	g_return_if_fail (is_nonempty_string(message_id));

	g_hash_table_insert (mset->hash, g_strdup(message_id), GINT_TO_POINTER(1));
}

void
messageidset_add_message_ids  (MessageIdSet         * mset,
                               const char          ** message_ids,
                               int                    message_id_qty)
{
	int i;

	g_return_if_fail (mset != NULL);
	g_return_if_fail (message_ids != NULL);
	g_return_if_fail (message_id_qty >= 1);

	g_mutex_lock (mset->lock);
	for (i=0; i<message_id_qty; ++i)
		maybe_add (mset, message_ids[i]);
	g_mutex_unlock (mset->lock);
}

void
messageidset_add_articles (MessageIdSet            * mset,
                           const Article          ** articles,
                           int                       article_qty)
{
	int i;

	g_return_if_fail (mset != NULL);
	g_return_if_fail (article_qty >= 1);
	g_return_if_fail (articles_are_valid (articles, article_qty));

	g_mutex_lock (mset->lock);
	for (i=0; i<article_qty; ++i)
		maybe_add (mset, article_get_message_id(articles[i]));
	g_mutex_unlock (mset->lock);
}

void
messageidset_add_articles_and_ancestors  (MessageIdSet            * mset,
                                          const Article          ** articles,
                                          int                       article_qty)
{
	int i;

	g_return_if_fail (mset != NULL);
	g_return_if_fail (article_qty >= 1);
	g_return_if_fail (articles_are_valid (articles, article_qty));

	g_mutex_lock (mset->lock);
	for (i=0; i<article_qty; ++i) {
		const Article * a = articles[i];
		while (a != NULL) {
			maybe_add (mset, article_get_message_id(a));
			a = a->parent;
		}
	}
	g_mutex_unlock (mset->lock);
}


void
messageidset_add_articles_and_threads (MessageIdSet        * mset,
                                       const Article      ** articles,
                                       int                   article_qty)
{
	int i;
	GPtrArray * thread;

	g_return_if_fail (mset != NULL);
	g_return_if_fail (article_qty >= 1);
	g_return_if_fail (articles_are_valid (articles, article_qty));

	thread = g_ptr_array_sized_new (64);
	for (i=0; i<article_qty; ++i)
	{
		g_ptr_array_set_size (thread, 0);
		article_get_thread (ARTICLE(articles[i]), thread);
		messageidset_add_articles (mset, (const Article**)thread->pdata, thread->len);
	}
	g_ptr_array_free (thread, TRUE);
}


static void
messageidset_get_all_ghfunc (gpointer key, gpointer val, gpointer data)
{
	g_ptr_array_add ((GPtrArray*)data, g_strdup(key));
}


GPtrArray*
messageidset_get_ids (const MessageIdSet  * mset)
{
	GPtrArray * message_ids;

	g_return_val_if_fail (mset!=NULL, NULL);

	g_mutex_lock (mset->lock);
	message_ids = g_ptr_array_sized_new (g_hash_table_size (mset->hash));
	g_hash_table_foreach (mset->hash, messageidset_get_all_ghfunc, message_ids);
	g_mutex_unlock (mset->lock);

	return message_ids;
}
