/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <gmime/gmime-utils.h>

#include <pan/base/acache.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-mime.h>

#include <pan/nntp.h>
#include <pan/prefs.h>
#include <pan/task.h>
#include <pan/auth-spa.h>

/** deprecated, use nntp_command */
extern const char * sockwrite_err_msg;
extern const char * sockread_err_msg;

enum
{
	OK_GROUP				= 211,
	OK_AUTH					= 281,
	NEED_AUTHDATA				= 381,
	ERR_NOAUTH				= 480,
	ERR_AUTHREJ				= 482,
	ERR_ACCESS				= 502
};

enum
{
	AUTH_UNKNOWN,
	AUTH_MSN
};

/**
***
**/

static gint
nntp_get_response (PanSocket     * socket,
                   const gchar  ** setme_result)
{
	gint val;
	gint retval;
	const gchar * msg = NULL;
	debug_enter ("nntp_get_response");

	val = pan_socket_getline (socket, &msg);
	if (val)
	{
		if (msg == NULL)
			msg = sockread_err_msg;
		if (setme_result != NULL)
			*setme_result = msg;
		retval = val;
	}
	else
	{
		if (setme_result != NULL)
			*setme_result = msg;
		retval = atoi (msg);
	}

	debug_exit ("nntp_get_response");
	return retval;
}

gint
nntp_command (StatusItem           * status,
              PanSocket            * sock,
              const gchar         ** setme_response,
              const gchar          * command)
{
	int val;
	size_t len;

	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (is_nonempty_string(command), -1);

	/**
	***  Send a command...
	**/

	len = strlen (command);
	if (len>=2 && command[len-2]=='\r' && command[len-1]=='\n')
	{
		val = pan_socket_putline (sock, command);
	}
	else /* not terminated in \r\n, so add */
	{
		gchar * tmp = g_strdup_printf ("%s\r\n", command);
		val = pan_socket_putline (sock, tmp);
		g_free (tmp);
	}
	if (val)
	{
		if (setme_response != NULL)
			*setme_response = sockwrite_err_msg;
		return val;
	}

	/**
	***  Get a response...
	**/

	val = nntp_get_response (sock, setme_response);

	if (val == ERR_NOAUTH)
	{
		val = nntp_authenticate (status, sock);
		if (val != TASK_SUCCESS)
			return -1;

		return nntp_command (status, sock, setme_response, command);
	}
	else return val;
}
 
gint
nntp_command_va (StatusItem           * status,
                 PanSocket            * sock,
                 const gchar         ** setme_result,
                 const gchar          * command_va,
                 ...)
{
	va_list args;
	char* line;
	int retval;

	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (command_va!=NULL, -1);

	va_start (args, command_va);
	line = g_strdup_vprintf (command_va, args);
	va_end (args);

	retval = nntp_command (status, sock, setme_result, line);

	g_free (line);
	return retval;
}

/**
***
**/

void
nntp_disconnect (StatusItem   * status,
                 PanSocket    * socket)
{
	debug_enter ("nntp_disconnect");

	/* entry assertions */
	g_return_if_fail (socket!=NULL);

	/* try to disconnect.  We don't look for the response, since we don't
	 * know what's in this socket.  It could be dead, or it could still have
	 * crap in the read buffer, mangling the response number. */
	pan_socket_putline (socket, "QUIT\r\n");

	/* Flush the socket. This ensures any remaining data (like TCP closing 
	 * packets) are read between sending the NNTP command (which makes the 
	 * server close the socket), and us closing our end of the socket.
	 * This prevents the socket going into TIME_WAIT state (or worse, 
	 * FIN_WAIT2)
	 */
	pan_socket_flush (socket);

	/* disconnected successfully */
	debug_exit ("nntp_disconnect");
}

/**
***
**/

gint
nntp_handshake (StatusItem  * status,
                PanSocket   * socket,
                gboolean    * setme_posting_ok)
{
	gint val;
	const gchar * response = NULL;

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme_posting_ok!=NULL, TASK_FAIL_HOPELESS);
	debug_enter ("nntp_handshake");

	/* get the server's handshake message */
	val = nntp_get_response (socket, &response);
	if (val==200 || val==201) { /* these values are expected */
		val = TASK_SUCCESS;
		log_add_va (LOG_INFO, _("Handshake: %s"), response);
	} else {
		status_item_emit_error_va (status, _("NNTP handshake failed: %s"), response);
		log_add_va (LOG_ERROR, _("Handshake failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS; 
	}

	/* can we post? */
	*setme_posting_ok = val==200;

	/* yes, we also support authentication on demand, but it appears
	   many servers are happier if we ask for authentication up-front. */
	if (pan_socket_needs_auth (socket))
		val = nntp_authenticate (status, socket);

	/* Achim Settelmeir reports that his local INN doesn't accept GROUP
	   without MODE READER first */
	nntp_can_post (NULL, socket, setme_posting_ok);


	debug_exit ("nntp_handshake");
	return val;
}

static gint
nntp_spa_authenticate (StatusItem     * status,
                        PanSocket      * sock)
{
	/* Authentication for SPA scheme (msn.com, and others) */
	/* Added by Marc Prud'hommeaux <marc@apocalypse.org>, mostly */
	/* taken from the ftechmail SPA authentication, which, in */
	/* turn, was ripped out of the SAMBA project. */

	gint val;
	const gchar * response = NULL;
	const gchar * response_trim = NULL;
	gchar msgbuf[2048];
	SPAAuthRequest   spa_request;
	SPAAuthChallenge spa_challenge;
	SPAAuthResponse  spa_response;

	debug_enter ("nntp_spa_authenticate");

	memset (msgbuf, 0, sizeof (msgbuf));

	if (nntp_command (status, sock, &response, "AUTHINFO GENERIC MSN")
	!= NEED_AUTHDATA) {
		status_item_emit_error_va (status, _(
			"Authentication failed: bad handshake for SPA password"));
		return TASK_FAIL;
	}
	
	spa_build_auth_request (&spa_request, pan_socket_get_username(sock), NULL);
	spa_bits_to_base64 (msgbuf, (unsigned char*)&spa_request, 
		spa_request_length (&spa_request));

	val = nntp_command_va (status, sock, &response, 
		"AUTHINFO GENERIC %s", msgbuf);
	if (val != NEED_AUTHDATA) {
		status_item_emit_error_va (status, _(
			"Bad SPA handshake: %s"), response);
		debug_exit ("nntp_spa_authenticate");
		return TASK_FAIL;
	}

	if (!is_nonempty_string(pan_socket_get_password(sock))) {
		if (status != NULL)
			status_item_emit_error_va (status, _(
				"Authentication failed: need a password"));
		return TASK_FAIL;
	}

	/* response is now something like:					*/
	/* 381 TlRMTVNTUAACAAAAGAAYACAAAAABAg [...]			*/
	/* Trim off the first 4 bytes in our response.		*/
	response_trim = response + 4;
	spa_base64_to_bits ((unsigned char*)&spa_challenge, response_trim);

	spa_build_auth_response (&spa_challenge, &spa_response, 
		pan_socket_get_username(sock),
		pan_socket_get_password(sock));

	spa_bits_to_base64 (msgbuf, (unsigned char*)&spa_response, 
		spa_request_length (&spa_response));

	val = nntp_command_va (status, sock, &response, 
		"AUTHINFO GENERIC %s", msgbuf);

	if (val != OK_AUTH) {
		status_item_emit_error_va (status, _(
			"Authentication failed: %s"), response);
		debug_exit ("nntp_spa_authenticate");
		return TASK_FAIL;
	}

	debug_exit ("nntp_spa_authenticate");
	return TASK_SUCCESS;
}

gint
nntp_authenticate_types (StatusItem     * status,
                         PanSocket      * sock,
                         gboolean         check_auth_types)
{
	gint val;
	int authtype = AUTH_UNKNOWN;
	const gchar * response = NULL;
	debug_enter ("nntp_authenticate");

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (pan_socket_get_username(sock)!=NULL, TASK_FAIL_HOPELESS);

	/* check for authentication schemes */
	/* see: http://www.mibsoftware.com/userkt/nntpext/0035.htm */
	if (check_auth_types) {
		val = nntp_command_va (status, sock, &response, "AUTHINFO GENERIC");
		if (val==OK_AUTH) {
			/* now we should be getting a list of valid authentication */
			/* types terminated with a "." */
			while (nntp_get_response (sock, &response) != -9999
			&& response != NULL
			&& pan_strcmp (response, ".\r\n")) {
				/* supported authentication schemes follow */
				if (!pan_strcmp (response, "MSN\r\n")) {
					authtype = AUTH_MSN;
				}
				else {
					status_item_emit_error_va (status, _(
						"Unsupported authentication mechanism: %s"), response);
				}
			}
	
			switch (authtype)
			{
				case AUTH_MSN:
					return nntp_spa_authenticate (status, sock);
	
				/* Unknown authentication type. We will fall through	*/
				/* and try the normal authentication method anyway.		*/
				case AUTH_UNKNOWN:
				default:
					status_item_emit_error_va (status, _(
						"No supported authentication mechanism"));
			}
		}
	}

	/* set the username */
	val = nntp_command_va (status, sock, &response, "AUTHINFO USER %s", pan_socket_get_username(sock));

	/* On servers that require SPA authentication, sending			*/
	/* "AUTHINFO USER ..." seems to result in a 502 error. If we	*/
	/* have not yet tried other authentication methods, do so now.	*/
	/* Ideally, we would not need to do this, but since some news`	*/
	/* servers disconnect the user if they send "AUTHINFO GENERIC",	*/
	/* we need to first try simple authentication.					*/
	if (val==ERR_ACCESS && check_auth_types==FALSE) {
		return nntp_authenticate_types (status, sock, TRUE);
	}

	if (val!=OK_AUTH && val!=NEED_AUTHDATA) {
		if (status != NULL)
			status_item_emit_error_va (status, _("Authentication failed: %s"), response);
		return TASK_FAIL;
	}

	/* set the password, if required */
	if (val==NEED_AUTHDATA) {
		const char * password = pan_socket_get_password (sock);
		if (!is_nonempty_string(password)) {
			if (status != NULL)
				status_item_emit_error_va (status, _("Authentication failed: need a password"));
			return TASK_FAIL;
		}
		val = nntp_command_va (status, sock, &response, "AUTHINFO PASS %s", password);
	}

	/* if we failed, emit an error */	
	if (val!=OK_AUTH) {
		if (status != NULL)
			status_item_emit_error_va (status, _("Authentication failed: %s"), response);
		return TASK_FAIL;
	}

	debug_exit ("nntp_authenticate");
	return TASK_SUCCESS;
}

gint
nntp_authenticate (StatusItem     * status,
                   PanSocket      * sock)
{
	return nntp_authenticate_types (status, sock, FALSE);
}

gint
nntp_can_post (StatusItem   * status,
               PanSocket    * sock,
               gboolean     * setme_can_post)
{
	gint val;
	gint retval = 0;
	const gchar * response = NULL;
	debug_enter ("nntp_can_post");

	/* entry assertions */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme_can_post!=NULL, TASK_FAIL_HOPELESS);

	/* ask the server if we can post or not */
	val = nntp_command (status, sock, &response, "MODE READER");
	switch (val) {
		case 200:
			*setme_can_post = TRUE;
			retval = 0;
			break;
		case 201:
			*setme_can_post = FALSE;
			retval = 0;
			break;
		default:
			*setme_can_post = FALSE;
			if (status != NULL)
				log_add_va (LOG_ERROR, _("MODE READER check failed: %s"),  response);
			retval = val;
			break;
	}

	debug_exit ("nntp_can_post");
	return retval;
}

/**
***
**/

int
nntp_set_group (StatusItem   * status,
                PanSocket    * sock,
                const char   * group_name)
{
	gint val;
	const gchar * response = NULL;
	debug_enter ("nntp_set_group");

	/* sanity checks */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(group_name), TASK_FAIL_HOPELESS);

	/* we're already there */
	if (!pan_strcmp (pan_socket_get_current_group(sock), group_name))
		return TASK_SUCCESS;

	/* change to that group */
	val = nntp_command_va (status, sock, &response, "GROUP %s", group_name);
	if (val != OK_GROUP) {
		status_item_emit_error_va (status, _("Unable to set group \"%s\": %s"), group_name, response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* update this socket's current group */
	pan_socket_set_current_group (sock, group_name);
	return TASK_SUCCESS;
}

/**
***
**/

int
nntp_post (StatusItem    * status,
           PanSocket     * sock,
           const gchar   * msg)
{
	gint val;
	const gchar * response;
	debug_enter ("nntp_post");

	/* entry assertions */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(msg), TASK_FAIL_HOPELESS);

	/* if we're in mute mode, don't post */
	if (pan_mute)
	{
		g_message ("Mute: Your Message will not actually be posted.");
		printf ("\n\n\nYour message\n%s<end of message>\n\n\n", msg);
		fflush (NULL);
	}
	else
	{
		/* tell the server we want to post */
		val = nntp_command (status, sock, &response, "POST");
		if (val != 340) {
			status_item_emit_error_va (status, _("Posting failed.  Server said: %s"), response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		}

		/* post the article */
		val = nntp_command_va (status, sock, &response, "%s\r\n.\r\n", msg);
		if (val == 240) {
			log_add_va (LOG_INFO, _("Posting complete.  Server said: %s"), response);
		} else {
			status_item_emit_error_va (status, "Posting failed.  Server said: %s", response);
			return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
		} 
	}

	/* if we made it this far, we succeeded! */
	debug_exit ("nntp_post");
	return TASK_SUCCESS;
}



gint
nntp_noop (StatusItem    * status,
           PanSocket     * socket)
{
	gboolean can_post = FALSE;
	debug_enter ("nntp_noop");

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (socket!=NULL, TASK_FAIL);

	/* do work */
	status_item_emit_status_va (status, _("\"stay connected\" sent to %s"), pan_socket_get_server_name(socket));
	return nntp_can_post (status, socket, &can_post);
}

/**
***
**/

int
nntp_article_download (StatusItem         * status,
                       PanSocket          * sock,
                       MessageIdentifier  * mid,
                       const gboolean     * abort,
                       int                  verbose)
{
	const char * response;
	gboolean getting_headers;
	MessageSource * source;
	int retval;
	int result;
	int val;
	GString * line;
	GString * setme;
	debug_enter ("nntp_article_download");

	/* sanity clause */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (message_identifier_is_valid(mid), TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL);

	/* let the user know what we're doing */
	if (verbose & NNTP_VERBOSE_TEXT)
		status_item_emit_status (status, mid->message_id);

	/* change to the group we want */
	source = message_identifier_get_source_for_server (mid, pan_socket_get_server_name(sock));
	if (*abort || !source)
		return TASK_FAIL;
	result = nntp_set_group (status, sock, source->group_name);
	if (result != TASK_SUCCESS)
		return result;

	/* request the article from the server */
	val = nntp_command_va (status, sock, &response, "ARTICLE %lu", source->number);
	if (val != 220)
	{
		/* log the error */
		char * pch = g_strdup_printf (_("Getting article \"%s\" body failed: %s"), mid->readable_name, response);
		status_item_emit_error (status, pch);
		g_free (pch);

		/* if -1, just fail... we might be able to retry later */
		if (val==-1)
			return TASK_FAIL;

		/* if it can't be done, try to mark the article as `error' and return hopeless */
		else {
			Server * server = serverlist_get_named_server (source->server_name);
			Group * group = server_get_named_group (server, source->group_name);
			if (group_ref_articles_if_loaded (group)) {
				Article * a = group_get_article_by_message_id (group, mid->message_id);
				if (a != NULL)
					article_set_error_flag (a, TRUE);
				group_unref_articles (group, NULL);
			}
			return TASK_FAIL_HOPELESS;
		}
	}

	/* try to read the article... */
	getting_headers = TRUE;
	line = g_string_sized_new (80);

	setme = g_string_sized_new (80 * 1000);
	retval = TASK_SUCCESS;
	if (verbose & NNTP_VERBOSE_INIT_STEPS)
		status_item_emit_init_steps (status, mid->line_qty);

	for (;;)
	{
		const gchar * pch;

		/* check for end cases: user abort, socket error, and success */
		if (*abort || pan_socket_getline (sock, &response)) {
			retval = TASK_FAIL;
			break;
		}

		if (!is_nonempty_string(response))
			continue;

		if (!strncmp(response, ".\r\n", 3)) {
			retval = TASK_SUCCESS;
			break;
		}

		/* strip out the \r */
		g_string_assign (line, response);
		while ((pch = strstr (line->str, "\r\n")))
			g_string_erase (line, pch-line->str, 1);

		/* rfc 977: 2.4.1 */
		if (line->len>=2 && line->str[0]=='.' && line->str[1]=='.')
			g_string_erase (line, 0, 1);

		/* save the line */
		g_string_append (setme, line->str);
		if (verbose & NNTP_VERBOSE_NEXT_STEP)
			status_item_emit_next_step (status);


		/* save the headers that we want to save */
		if (getting_headers)
		{
			if (line->len==1 && !strcmp(line->str, "\n")) /* header/body separator */
				getting_headers = FALSE;
		}
	}

	if (verbose & NNTP_VERBOSE_NEXT_STEP)
		status_item_emit_next_step (status);

	/* save the message */
	if (retval == TASK_SUCCESS)
        	acache_set_message (ACACHE_DEFAULT_KEY, mid->message_id, setme->str, setme->len);

	g_string_free (line, TRUE);
	g_string_free (setme, TRUE);
	debug_exit ("nntp_article_download");
	return retval;
}


/**
***
**/

gint
nntp_get_group_info (StatusItem       * status,
                     PanSocket        * sock,
                     const char       * group_name,
                     gulong           * article_qty,
                     gulong           * low_num,
                     gulong           * high_num,
                     const gboolean   * abort)
{
	gint val;
	const gchar * response = NULL;
	debug_enter ("nntp_get_group_info");

	/* entry assertions */
	g_return_val_if_fail (status!=NULL, TASK_FAIL);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (is_nonempty_string(group_name), TASK_FAIL);
	g_return_val_if_fail (article_qty!=NULL, TASK_FAIL);
	g_return_val_if_fail (low_num!=NULL, TASK_FAIL);
	g_return_val_if_fail (high_num!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL);

	/* change to that group */
	val = nntp_command_va (status, sock, &response, "GROUP %s", group_name);
	if (val != OK_GROUP) {
		status_item_emit_error_va (status, _("Unable to set group \"%s\": %s"), group_name, response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/* success; parse the results */
	sscanf (response, "%*d %lu %lu %lu", article_qty, low_num, high_num);
	pan_socket_set_current_group (sock, group_name);
	return TASK_SUCCESS;
}

/**
***
**/

gint
nntp_download_headers (StatusItem       * status,
                       PanSocket        * sock,
                       Group            * group,
                       gulong             low,
                       gulong             high,
                       const gboolean   * abort,
                       const char       * progress_fmt,
                       GPtrArray        * setme)
{
	const char * buffer;
	const char * response;
	gulong total;
	gulong first;
	gulong last;
	gboolean read_status = 0;
	gint result;
	gint val;
	GString * buf;
	debug_enter ("nntp_download_headers");

	/* sanity checks */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (group_is_valid(group), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(progress_fmt), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (setme!=NULL, TASK_FAIL_HOPELESS);

	/**
	***  Get up-to-date information about this group
	**/

	total = first = last = 0;
	result = nntp_get_group_info (status,
	                              sock,
	                              group->name,
	                              &total,
	                              &first,
	                              &last,
	                              abort);
	if (result != TASK_SUCCESS)
		return result;


	/**
	***  If no articles to get, then we're already done
	**/

	if (total == 0) {
		const gchar * n = group_get_name (group);
		status_item_emit_status_va (status, _("No articles found for group \"%s\""), n);
		return TASK_SUCCESS;
	}

	/**
	***  Tell the server that we want a series of article headers...
	**/

	val = *abort ? -1 : nntp_command_va (status, sock, &response, "XOVER %lu-%lu", low, high);
	if (val != 224) {
		status_item_emit_status_va (status, _("Getting header list failed: %s"), response);
		return val==-1 ? TASK_FAIL : TASK_FAIL_HOPELESS;
	}

	/**
	***  Walk through all the articles headers that the server spits back
	**/

	buf = g_string_new (NULL);
	pan_g_ptr_array_reserve (setme, setme->len + (high-low));
	while (!*abort && !((read_status = pan_socket_getline (sock, &buffer))))
	{
		int part = 0;
	        int parts = 0;
		const char * pch;
		const char * s;
		Article * a;
		const char * march = NULL;
		const char * tok = NULL;
		int tok_len = 0;
		const char * charset = group_get_default_charset (group);
		char * cvt;

		/* handle end-of-list */
		if (!strncmp(buffer, ".\r\n", 3))
			break;

		/* create the article data */
		a = article_new (group);
		a->is_new = TRUE;

		/* setup for parsing */
		g_string_assign (buf, buffer);
		march = buf->str;

		/* get article number */
		a->number = get_next_token_ulong (march, '\t', &march);

		/* get subject */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((char*)tok)[tok_len] = '\0';
			g_strstrip ((char*)tok);
			cvt = pan_header_to_utf8 (tok, tok_len, charset);
			article_set_subject (a, cvt);
			g_free (cvt);
		}

		/* get author */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((char*)tok)[tok_len] = '\0';
			g_strstrip ((char*)tok);
			article_set_author (a, tok);
		}

		/* get date */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((char*)tok)[tok_len] = '\0';
			g_strstrip ((char*)tok);
			a->date = is_nonempty_string(tok)
				? g_mime_utils_header_decode_date (tok, NULL)
				: (time_t)0;
		}

		/* get message id */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((char*)tok)[tok_len] = '\0';
			g_strstrip ((char*)tok);
			if (is_nonempty_string(tok) && *tok=='<')
				article_set_message_id (a, tok);
		}

		/* get references */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((char*)tok)[tok_len] = '\0';
			g_strstrip ((char*)tok);
			if (is_nonempty_string(tok) && *tok=='<')
				article_set_references (a, tok);
		}

		/* get byte qty */
		a->byte_qty = get_next_token_ulong (march, '\t', &march);

		/* get line qty */
		a->linecount = get_next_token_int (march, '\t', &march);

		/* get crossref */
		if (get_next_token_run (march, '\t', &march, &tok, &tok_len)) {
			((char*)tok)[tok_len] = '\0';
			g_strstrip ((char*)tok);
			if (tok_len>6 && !strncmp(tok,"Xref: ", 6)) {
				const char * header = tok + 6;
				g_strstrip ((char*)header); /* remove trailing linefeed */
				article_set_xrefs (a, header);
			}
		}

		/**
		***  Validity Checking
		**/

		if (!article_is_valid (a)) {
			status_item_emit_error_va (status, 
				_("Corrupt header skipped: %s"), buffer);
			continue;
		}


		/* Look for the (n/N) or [n/N] construct in subject lines,
		 * starting at the end of the string and working backwards */
		part = 0;
		parts = 0;
		s = a->subject;
		pch = s + strlen(s) - 1;
		while (pch != s)
		{
			/* find the ']' of [n/N] */
			--pch;
			if ((pch[1]!=')' && pch[1]!=']') || !isdigit((guchar)*pch))
				continue;

			/* find the '/' of [n/N] */
			while (s!=pch && isdigit((guchar)*pch))
				--pch;
			if (s==pch || (*pch!='/' && *pch!='|'))
				continue;

			/* N -> parts */
			parts = atoi (pch+1);
			--pch;

			/* find the '[' of [n/N] */
			while (s!=pch && isdigit((guchar)*pch))
				--pch;
			if (s==pch || (*pch!='(' && *pch!='[')) {
				parts = 0;
				continue;
			}

			/* n -> part */
			part = atoi (pch+1);
			break;
		}

		/* if not a multipart yet, AND if it's a big message, AND
		   it's either in one of the pictures/fan/sex groups or it
		   has commonly-used image names in the subject, guess it's
		   a single-part binary */
		if (!parts
			&& a->linecount>400
			&& (((pan_strstr(group->name,"binaries")
					|| pan_strstr(group->name,"fan")
				        || pan_strstr(group->name,"mag")
					|| pan_strstr(group->name,"sex")))
				|| ((pan_strstr(s,".jpg") || pan_strstr(s,".JPG")
					|| pan_strstr(s,".jpeg") || pan_strstr(s,".JPEG")
					|| pan_strstr(s,".gif") || pan_strstr(s,".GIF")
					|| pan_strstr(s,".tiff") || pan_strstr(s,".TIFF")))))
			part = parts = 1;

		/* Verify Multipart info */
		if ((parts>=1) && (part<=parts)) {
			/* yay, we found one! */
			a->parts = parts;
			a->part = part;
		}
		else {
			a->parts = 0;
			a->part = 0;
		}

		/* Add the article to the article list */
		g_ptr_array_add (setme, a);

		/* update the count & progress ui */
		status_item_emit_next_step (status);
		if (!(setme->len % 100))
			status_item_emit_status_va (status, progress_fmt, setme->len, (high-low+1));
	}
	g_string_free (buf, TRUE);

	if (read_status!=0)
		status_item_emit_error (status, sockread_err_msg);
	if (read_status!=0 || *abort)
		return TASK_FAIL;

	/* if we've made it this far, we succeeded! */
	return TASK_SUCCESS;
}

/**
***
**/

int
nntp_download_bodies (StatusItem                * status,
                      PanSocket                 * sock,
                      const gboolean            * abort,
                      MessageIdentifier        ** mids,
                      int                         mid_qty,
                      int                       * index,
                      gboolean                    abort_if_one_fails)
{
	int result = TASK_SUCCESS;
	int i;
	debug_enter ("nntp_download_bodies");

	/* sanity checks pt 1 */
	g_return_val_if_fail (mids!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (mid_qty>0, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (index!=NULL, TASK_FAIL_HOPELESS);

	/**
	***  If we have the bodies, then just unflag 'em and maybe check them out
	**/

	/* find the first message we don't have cached */
	for (i=*index; i<mid_qty; ++i)
		if (!acache_has_message (ACACHE_DEFAULT_KEY, mids[i]->message_id))
			break;

	/* are we done? */
	if (i==mid_qty)
		return TASK_SUCCESS;

	/* sanity checks pt 2 */
	g_return_val_if_fail (status!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (sock!=NULL, TASK_FAIL);
	g_return_val_if_fail (abort!=NULL, TASK_FAIL_HOPELESS);

	/**
	***  Phooey, we have to download some of the bodies
	**/

	/* status item */
	if (1) {
		gulong line_qty = 0ul;
		gulong line_qty_so_far = 0ul;
		for (i=0; i<mid_qty; ++i) {
			line_qty += mids[i]->line_qty;
			if (i<*index)
				line_qty_so_far += mids[i]->line_qty;
		}
		status_item_emit_init_steps (status, line_qty);
		status_item_emit_set_step (status, line_qty_so_far);
	}

	/* download the articles */
	for (; *index<mid_qty && !*abort; ++(*index))
	{
		MessageIdentifier * mid = mids[*index];

		if (mid_qty == 1)
			status_item_emit_status_va (status, _("Getting \"%s\""),
							      mid->readable_name);
		else
			status_item_emit_status_va (status, _("Getting %d of %d"),
							      1+*index, mid_qty);

		/* download if we have to. */
		if (acache_has_message (ACACHE_DEFAULT_KEY, mid->message_id))
		{
			result = TASK_SUCCESS;
			status_item_emit_inc_step (status, mid->line_qty);
		}
		else
		{
			result = nntp_article_download (status,
			                                sock,
			                                mid,
			                                abort,
			                                NNTP_VERBOSE_NEXT_STEP);

			if (result!=TASK_SUCCESS && abort_if_one_fails)
				return result;
		}
	}

	return result;
}


int
nntp_cancel (StatusItem     * status,
             const char     * message_id,
             PanSocket      * sock)
{
	const char * newsgroups;
	const char * author;
	int retval;
	MessageIdentifier * mid;
	GMimeMessage * message;
	debug_enter ("nntp_cancel");

	/* sanity checks */
	g_return_val_if_fail (sock != NULL, TASK_FAIL);
	g_return_val_if_fail (status != NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(message_id), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (acache_has_message(ACACHE_DEFAULT_KEY, message_id), TASK_FAIL_HOPELESS);

	/* get info */
	retval = TASK_FAIL;
	mid = message_identifier_new (message_id);
	message = acache_get_message (ACACHE_DEFAULT_KEY, &mid, 1);
	newsgroups = g_mime_message_get_header (message, HEADER_NEWSGROUPS);
	author = g_mime_message_get_sender (message);
	if (is_nonempty_string(newsgroups) && is_nonempty_string(author))
	{
		GString * msg;

		/* let the user know what we're doing */
		status_item_emit_status_va (status, _("Canceling article"));

		/* build the message to post */
		msg = g_string_sized_new (1024);
		g_string_append_printf (msg, "From: %s\r\n", author);
		g_string_append_printf (msg, "Newsgroups: %s\r\n", newsgroups);
		g_string_append_printf (msg, "Subject: cancel %s\r\n", message_id);
		g_string_append_printf (msg, "Control: cancel %s\r\n", message_id);
		g_string_append (msg, "\r\n");
		g_string_append_printf (msg, "Ignore\r\nArticle canceled by Pan %s\r\n", VERSION);

		/* post the cancel message */
		retval = nntp_post (status, sock, msg->str);

		/* cleanup */
		g_string_free (msg, TRUE);
	}

	/* cleanup */
	g_object_unref (message);
	g_object_unref (mid);
	debug_exit ("nntp_cancel");
	return retval;
}
