/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

#include <pan/base/acache.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/base/text-massager.h>
#include <pan/base/util-file.h>

#include <pan/filters/filter-score.h>

#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/gui-paned.h>
#include <pan/nntp.h>
#include <pan/pan-color-picker.h>
#include <pan/pan-font-picker.h>
#include <pan/pan-file-entry.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/server-menu-ui.h>
#include <pan/task-headers.h>
#include <pan/text.h>
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

#define NORMAL_PANGO_DESCRIPTION "sans 11"
#define MONOSPACE_PANGO_DESCRIPTION  "courier 11"

#ifdef G_OS_WIN32
#define DEFAULT_WEB_BROWSER "\"c:\\program files\\internet explorer\\iexplore\" %s"
#define DEFAULT_EXTERNAL_EDITOR "notepad %t"
#define DEFAULT_SCORE_EDITOR_COMMAND "notepad %t"
#else
#define DEFAULT_WEB_BROWSER "mozilla %s"
#define DEFAULT_EXTERNAL_EDITOR "xterm -e vi %t"
#define DEFAULT_SCORE_EDITOR_COMMAND "xterm -e vi +%n %t"
#endif

typedef struct
{
	GtkWidget * dialog;
	GtkWidget * notebook;

	/* smtp server */
	GtkWidget * smtp_address;
	GtkWidget * smtp_port;

	/* cache */
	GtkWidget * cache_megs_sb;
	GtkWidget * flush_cache_on_exit_check;

	/* general */
	GtkWidget * online_on_startup_cbutton;
	GtkWidget * single_click_selects_groups_cbutton;
	GtkWidget * single_click_selects_headers_cbutton;
	GtkWidget * remove_failed_tasks_cbutton;
	GtkWidget * fetch_new_on_group_load_cbutton;
	GtkWidget * fetch_new_on_startup_cbutton;
	GtkWidget * fetch_new_and_bodies_on_startup_cbutton;
	GtkWidget * break_thread_when_subject_changes_cbutton;
	GtkWidget * expand_all_threads_by_default_cbutton;
	GtkWidget * hide_mpart_child_nodes_cbutton;
	GtkWidget * external_editor_combo;
	GtkWidget * external_browser_combo;

	/* directories */
	GtkWidget * dir_download;
	GtkWidget * dir_temp;
	GtkWidget * dir_data;

	/* fonts */
	GtkWidget * header_pane_custom_font_enabled_check;
	GtkWidget * header_pane_custom_font_gfp;
	GtkWidget * group_pane_custom_font_enabled_check;
	GtkWidget * group_pane_custom_font_gfp;
	GtkWidget * body_pane_custom_font_enabled_check;
	GtkWidget * body_pane_custom_font_gfp;
	GtkWidget * body_pane_monospace_font_gfp;

	/* colors */
	GtkWidget * read_pcp;
	GtkWidget * unread_pcp;
	GtkWidget * text_fg_pcp;
	GtkWidget * text_bg_pcp;
	GtkWidget * text_url_pcp;
	GtkWidget * text_quoted_pcp[3];
	GtkWidget * signature_pcp;
	GtkWidget * text_quoted_chars_entry;
	GtkWidget * thread_date_entry;
	GtkWidget * smooth_scrolling_check;
	GtkWidget * smooth_scrolling_speed_sb;

	/* layout */
	GtkWidget * layout_page;
	GtkWidget * body_pane_page;

	/* scoring */
	GtkWidget * scorefile_pfe;
	GtkWidget * score_editor_command_combo;

	GtkWidget * color_subject_column_check;
	GtkWidget * score_pcp[SCORE_COLOR_QTY][2];
	GtkWidget * score_lb[SCORE_COLOR_QTY][2];

	/* column order */
	GtkListStore * column_store;
	GtkWidget * column_tree_view;
	GtkWidget * column_up_cbutton;
	GtkWidget * column_down_cbutton;
}
PrefsWindow;


static char* layout_get_new_string (GtkWidget * layout_page);
static gulong get_header_flags (void);

static PrefsWindow * win;

extern GtkTooltips * ttips;

#define magic(s) #s
#define stringify(s) magic(s)

gboolean collapse_group_names                 = FALSE;
gboolean text_window_smooth_scrolling         = TRUE;
gboolean do_spellcheck                        = TRUE;
int      text_window_smooth_scrolling_speed   = 10;

#define MAX_MEGS_VALUE_DEFAULT 10

char * thread_date_format                     = NULL;
char * layout_str                             = NULL;
int mail_server_port                          = 0;
gboolean expand_all_threads_by_default        = FALSE;
gboolean hide_mpart_child_nodes               = TRUE;
gboolean pan_mute                             = FALSE;
gboolean fetch_new_on_group_load              = TRUE;
gboolean fetch_new_on_startup                 = FALSE;
gboolean fetch_new_and_bodies_on_startup      = FALSE;
gboolean header_pane_is_threaded              = TRUE;
gboolean single_click_selects_groups          = FALSE;
gboolean single_click_selects_headers         = FALSE;
gboolean show_group_pane                      = TRUE;
gboolean show_header_pane                     = TRUE;
gboolean show_body_pane                       = TRUE;
gboolean header_pane_color_subject_column     = FALSE;
gboolean header_pane_color_score_column       = TRUE;

gboolean  group_pane_custom_font_enabled      = FALSE;
char *   group_pane_custom_font               = NULL;
gboolean header_pane_custom_font_enabled      = FALSE;
char *   header_pane_custom_font              = NULL;
gboolean body_pane_custom_font_enabled        = FALSE;
char *   body_pane_custom_font                = NULL;
gboolean body_pane_monospace_font_enabled     = FALSE;
char *   body_pane_monospace_font             = NULL;

char * external_editor = NULL;
char * score_editor_command = NULL;
char * external_web_browser = NULL;
char * mail_server_address = NULL;

static void
set_color (GdkColor      * color,
           const char    * key,
           const char    * default_value)
{
	char * value;
	char buf[1024];

	g_snprintf (buf, sizeof(buf), "%s=%s", key, default_value);
	value = pan_config_get_string (buf);

	if (!gdk_color_parse (value, color))
		g_warning ("couldn't parse color \"%s\"", value);
	else if (!gdk_color_alloc (cmap, color))
		g_error ("couldn't allocate color \"%s\"", key);

	g_free (value);
}

static void
date_help_clicked_cb (void)
{
	const char * str = _("%a - abbreviated weekday name\n"
	                     "%A - full weekday name\n"
	                     "%b - abbreviated month name\n"
	                     "%B - full month name\n"
	                     "%c - local date & time\n"
	                     "%d - day of the month\n"
	                     "%H - hour (24-hour clock)\n"
	                     "%I - hour (12-hour clock)\n"
	                     "%j - day of the year (001-366)\n"
	                     "%m - month (01-12)\n"
	                     "%M - minute (00-59)\n"
	                     "%p - local equivalent of AM/PM\n"
	                     "%S - second (00-61)\n"
	                     "%x - local date\n"
	                     "%X - local time\n"
	                     "%y - two-digit year\n"
	                     "%Y - four-digit year\n"
	                     "%% - %");
	GtkWidget * w = gtk_message_dialog_new (GTK_WINDOW(win->dialog),
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_MESSAGE_INFO,
	                                        GTK_BUTTONS_CLOSE, "%s", str);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT (w));
	gtk_widget_show_all (w);
}

static void
pan_prefs_changed (GtkDialog * dialog)
{
        gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_OK, TRUE);
        gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_APPLY, TRUE);
}

#define connect_signal_to_prefs_changed(object,signal_name) \
	g_signal_connect_swapped (object, signal_name, G_CALLBACK(pan_prefs_changed), win->dialog)

/**
***  UPDATE UTILS
**/

static gboolean
update_entry_and_bool_from_toggle_button (gboolean * setme,
                                          const char * key,
                                          GtkWidget * toggle)
{
	gboolean changed = FALSE;
	const gboolean b = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON(toggle));

	if (b != *setme) {
		*setme = b;
		pan_config_set_bool (key, b);
		changed = TRUE;
	}

	return changed;
}

static void
set_config_from_editable (const char   * key,
                          GtkWidget    * entry)
{
	char * text = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	g_strstrip (text);
	if (is_nonempty_string(text))
		pan_config_set_string (key, text);
	else
		pan_config_clean_key (key);
	g_free (text);
}

static void
handle_editable (GtkWidget * e, char ** setme, const char * key)
{
	char * s = gtk_editable_get_chars (GTK_EDITABLE(e), 0, -1);
	g_strstrip (s);
	if (!pan_strcmp(*setme, s))
		g_free (s);
	else {
		pan_config_set_string (key, s);
		replace_gstr (setme, s);
	}
}

static gboolean
handle_spin (GtkWidget * spin, int * pvalue, const char * config_key)
{
	int i;
	gboolean changed = FALSE;

	/* sanity clause */
	g_return_val_if_fail (GTK_IS_SPIN_BUTTON(spin), FALSE);
	g_return_val_if_fail (pvalue != NULL, FALSE);
	g_return_val_if_fail (is_nonempty_string(config_key), FALSE);

	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spin));
	if (i != *pvalue) {
		pan_config_set_int (config_key, i);
		*pvalue = i;
		changed = TRUE;
	}

	return changed;
}

static gboolean
handle_font_picker (GtkWidget * pfp, char ** p_old_font, const char * config_key)
{
	gboolean changed = FALSE;

	/* get new... */
	char * new_font = pan_font_picker_get_font (pfp);

	/* compare old with new... */
	if (pan_strcmp (*p_old_font, new_font)) {
		replace_gstr (p_old_font, g_strdup(new_font));
		pan_config_set_string (config_key, new_font);
		changed = TRUE;
	}

	g_free (new_font);
	return changed;
}

static gboolean
handle_color_picker (const char    * key,
                     GdkColor      * color,
                     GtkWidget     * w)
{
	GdkColor picker_color;
	gboolean color_changed = FALSE;

	g_return_val_if_fail (w!=NULL, FALSE);

	pan_color_picker_get_color (w, &picker_color);

	color_changed = (color->red != picker_color.red)
		     || (color->green != picker_color.green)
		     || (color->blue != picker_color.blue);

	if (color_changed)
	{
		char value[1024];

		color_changed = TRUE;

		/* update the color */
		gdk_colormap_free_colors (cmap, color, 1);
		color->red = picker_color.red;
		color->green = picker_color.green;
		color->blue = picker_color.blue;
		if (!gdk_color_alloc (cmap, color))
			g_error ("couldn't allocate \"%s\" color", key);

		/* update the config */
		g_snprintf (value, sizeof(value), "#%04x%04x%04x",
		            picker_color.red,
		            picker_color.green,
		            picker_color.blue);
		pan_config_set_string (key, value);
	}

	return color_changed;
}

static gboolean
handle_articlelist_columns (GtkTreeModel * model,
		            int          * articlelist_columns,
			    int            articlelist_column_qty,
			    const char   * key)
{
	gboolean      columns_changed = FALSE;
	gboolean      ok;
	GtkTreeIter   iter;
	int           idx;
	GString     * val = g_string_new (NULL);

	ok = gtk_tree_model_get_iter_first (model, &iter);

	for (idx=0; ok && idx < articlelist_column_qty ; idx++)
	{
		int col;

		gtk_tree_model_get (model, &iter, 1, &col, -1);

		columns_changed|= articlelist_columns[idx] != col;
		articlelist_columns[idx] = col;

		g_string_append_printf (val, idx > 0 ? "|%u" : "%u", col+1);

		ok = gtk_tree_model_iter_next (model, &iter);
	}

	if (val->len > 0)
		pan_config_set_string (KEY_ARTICLELIST_COLUMNS, val->str);

	g_string_free (val, TRUE);

	return columns_changed;
}

/**
***  UPDATE EVERYTHING
**/

/* Ok or Apply pressed in the Preferences dialog, save all information */
static void
prefs_apply (void)
{
	gboolean articlelist_changed = FALSE;
	gboolean color_changed;
	gboolean b;

	/* Commit all of the data to the config file */

	/* No entry for these yet
	 * pan_config_set_int("/Pan/Articles/Expiration", 30);
	 */

	/* thread date format */
	if (1) {
		char * tmp = g_strdup (thread_date_format);
		handle_editable (win->thread_date_entry, &thread_date_format, KEY_THREAD_DATE_FORMAT);
		if (pan_strcmp (tmp, thread_date_format))
			articlelist_changed = TRUE;
		g_free (tmp);
	}

	set_config_from_editable (
		"/Pan/Paths/download_dir",
		pan_file_entry_gtk_entry(win->dir_download));

	/**
	***  Mail Server
	**/

	handle_editable (win->smtp_address, &mail_server_address, "/Pan/Mail/smtp_address");

	handle_spin (win->smtp_port, &mail_server_port, "/Pan/Mail/smtp_port");

	/**
	***  Scoring
	**/

	pan_config_set_string (KEY_SCOREFILE, pan_file_entry_get (win->scorefile_pfe));
	handle_editable (GTK_COMBO(win->score_editor_command_combo)->entry, &score_editor_command, KEY_APP_SCOREFILE_EDITOR);
	articlelist_changed |= update_entry_and_bool_from_toggle_button (&header_pane_color_subject_column,
	                                                                 KEY_COLOR_SUBJECT_COLUMN,
	                                                                 win->color_subject_column_check);
	articlelist_changed |= handle_color_picker (KEY_COLOR_WATCHED_BG,
	                                            &score_color[SCORE_COLOR_WATCHED][0],
	                                            win->score_pcp[SCORE_COLOR_WATCHED][0]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_WATCHED_FG,
	                                            &score_color[SCORE_COLOR_WATCHED][1],
	                                            win->score_pcp[SCORE_COLOR_WATCHED][1]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_HIGH_BG,
	                                            &score_color[SCORE_COLOR_HIGH][0],
	                                            win->score_pcp[SCORE_COLOR_HIGH][0]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_HIGH_FG,
	                                            &score_color[SCORE_COLOR_HIGH][1],
	                                            win->score_pcp[SCORE_COLOR_HIGH][1]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_MEDIUM_BG,
	                                            &score_color[SCORE_COLOR_MEDIUM][0],
	                                            win->score_pcp[SCORE_COLOR_MEDIUM][0]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_MEDIUM_FG,
	                                            &score_color[SCORE_COLOR_MEDIUM][1],
	                                            win->score_pcp[SCORE_COLOR_MEDIUM][1]);
#if 0
	articlelist_changed |= handle_color_picker (KEY_COLOR_ZERO_BG,
	                                            &score_color[SCORE_COLOR_ZERO][0],
	                                            win->score_pcp[SCORE_COLOR_ZERO][0]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_ZERO_FG,
	                                            &score_color[SCORE_COLOR_ZERO][1],
	                                            win->score_pcp[SCORE_COLOR_ZERO][1]);
#endif
	articlelist_changed |= handle_color_picker (KEY_COLOR_LOW_BG,
	                                            &score_color[SCORE_COLOR_LOW][0],
	                                            win->score_pcp[SCORE_COLOR_LOW][0]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_LOW_FG,
	                                            &score_color[SCORE_COLOR_LOW][1],
	                                            win->score_pcp[SCORE_COLOR_LOW][1]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_IGNORED_BG,
	                                            &score_color[SCORE_COLOR_IGNORED][0],
	                                            win->score_pcp[SCORE_COLOR_IGNORED][0]);
	articlelist_changed |= handle_color_picker (KEY_COLOR_IGNORED_FG,
	                                            &score_color[SCORE_COLOR_IGNORED][1],
	                                            win->score_pcp[SCORE_COLOR_IGNORED][1]);
	/**
	***  Display
	**/

	/* header pane colors */
	articlelist_changed |= handle_color_picker (KEY_COLOR_READ, &read_color, win->read_pcp);
	articlelist_changed |= handle_color_picker (KEY_COLOR_UNREAD, &unread_color, win->unread_pcp);

	/* body pane colors */
	b = 0;
	b |= handle_color_picker (KEY_COLOR_URL, &text_url_color, win->text_url_pcp);
	b |= handle_color_picker (KEY_COLOR_QUOTED_1, &text_quoted_color[0], win->text_quoted_pcp[0]);
	b |= handle_color_picker (KEY_COLOR_QUOTED_2, &text_quoted_color[1], win->text_quoted_pcp[1]);
	b |= handle_color_picker (KEY_COLOR_QUOTED_3, &text_quoted_color[2], win->text_quoted_pcp[2]);
	b |= handle_color_picker (KEY_COLOR_SIGNATURE, &signature_color, win->signature_pcp);
	color_changed = b;

	/* quote prefix characters */
	if (1)
	{
		char * old_quote_chars = NULL;
		char * new_quote_chars = NULL;
		TextMassager * tm;
	       
		tm = text_pane_get_text_massager ();
	       	old_quote_chars = text_massager_get_quote_chars (tm);
		handle_editable (win->text_quoted_chars_entry, &new_quote_chars, KEY_QUOTE_CHARS);

		if (pan_strcmp (old_quote_chars, new_quote_chars))
		{
			text_massager_set_quote_chars (tm, new_quote_chars);
			color_changed = TRUE;
		}

		g_free (new_quote_chars);
		g_free (old_quote_chars);
	}

	if (color_changed)
	{
		GtkStyle * style;
		GtkTextBuffer *text_buffer;

		style = gtk_rc_get_style (Pan.text);

		style = style != NULL ? gtk_style_copy (style) 
				      : gtk_style_new ();

		gtk_widget_set_style (Pan.text, style);

		text_set_font ();
		if (Pan.text)
		{
			text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(Pan.text));
			text_set_text_buffer_tags (text_buffer);
		}
	}

	if (update_entry_and_bool_from_toggle_button (&group_pane_custom_font_enabled,
	                                              KEY_GROUP_PANE_CUSTOM_FONT_ENABLED,
	                                              win->group_pane_custom_font_enabled_check))
	{
		grouplist_update_font ();
	}

	if (update_entry_and_bool_from_toggle_button (&header_pane_custom_font_enabled,
	                                              KEY_HEADER_PANE_CUSTOM_FONT_ENABLED,
	                                              win->header_pane_custom_font_enabled_check))
	{
		articlelist_changed = TRUE;
	}
	if (update_entry_and_bool_from_toggle_button (&body_pane_custom_font_enabled,
	                                              KEY_BODY_PANE_CUSTOM_FONT_ENABLED,
	                                              win->body_pane_custom_font_enabled_check))
	{
		text_set_font ();
	}

	if (handle_font_picker (win->group_pane_custom_font_gfp,
	                        &group_pane_custom_font,
	                        KEY_GROUP_PANE_CUSTOM_FONT))
		grouplist_update_font ();

	if (handle_font_picker (win->header_pane_custom_font_gfp,
	                        &header_pane_custom_font,
	                        KEY_HEADER_PANE_CUSTOM_FONT))
		articlelist_changed = TRUE;

	update_entry_and_bool_from_toggle_button (&text_window_smooth_scrolling,
	                                          KEY_SMOOTH_SCROLLING_ENABLED,
	                                          win->smooth_scrolling_check);

	if (1) {
		GtkSpinButton* w = GTK_SPIN_BUTTON(
			win->smooth_scrolling_speed_sb);
		const gint i = gtk_spin_button_get_value_as_int (w);
		if (i != text_window_smooth_scrolling_speed) {
			text_window_smooth_scrolling_speed = i;
			pan_config_set_int (KEY_SMOOTH_SCROLLING, i);

		}
	}

	b = handle_font_picker (win->body_pane_custom_font_gfp,
	                        &body_pane_custom_font,
	                        KEY_BODY_PANE_CUSTOM_FONT);

	b |= handle_font_picker (win->body_pane_monospace_font_gfp,
	                        &body_pane_monospace_font,
	                        KEY_BODY_PANE_MONOSPACE_FONT);

	if (b)
		text_set_font ();

	if (1) {
		char * str = layout_get_new_string (win->layout_page);
		if (pan_strcmp (str, layout_str)) {
			replace_gstr (&layout_str, g_strdup(str));
			pan_config_set_string (KEY_PANE_LAYOUT, layout_str);
			gui_layout_refresh ();
		}
		g_free (str);
	}

	/**
	***  General
	**/

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_group_load,
		KEY_FETCH_NEW_ON_GROUP_LOAD,
		win->fetch_new_on_group_load_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_startup,
		KEY_FETCH_NEW_FROM_SUBSCRIBED_ON_STARTUP,
		win->fetch_new_on_startup_cbutton);

	update_entry_and_bool_from_toggle_button (
		&single_click_selects_groups,
		KEY_SINGLE_CLICK_SELECTS_GROUPS,
		win->single_click_selects_groups_cbutton);

	update_entry_and_bool_from_toggle_button (
		&single_click_selects_headers,
		KEY_SINGLE_CLICK_SELECTS_HEADERS,
		win->single_click_selects_headers_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_and_bodies_on_startup,
	        KEY_FETCH_NEW_AND_BODIES_ON_STARTUP,
		win->fetch_new_and_bodies_on_startup_cbutton);

	if (update_entry_and_bool_from_toggle_button (
		&break_thread_when_subject_changes,
	        KEY_BREAK_THREAD_WHEN_SUBJECT_CHANGES,
		win->break_thread_when_subject_changes_cbutton))
			articlelist_changed = TRUE;

	if (update_entry_and_bool_from_toggle_button (
		&expand_all_threads_by_default,
	        KEY_EXPAND_ALL_THREADS_BY_DEFAULT,
		win->expand_all_threads_by_default_cbutton))
			articlelist_changed = TRUE;

	if (update_entry_and_bool_from_toggle_button (
		&hide_mpart_child_nodes,
		KEY_HIDE_MULTIPART_CHILDREN,
		win->hide_mpart_child_nodes_cbutton))
			articlelist_changed = TRUE;

	/* remove failed tasks */
	if (1) {
		const gboolean old_val = queue_get_remove_failed_tasks ();
		gboolean new_val = old_val;
		update_entry_and_bool_from_toggle_button (
			&new_val,
			KEY_REMOVE_FAILED_TASKS,
			win->remove_failed_tasks_cbutton);
		if (old_val != new_val)
			queue_set_remove_failed_tasks (new_val);
	}

	/**
	***  Cache
	**/

	/* cache */
	if (1) {
		acache_max_megs = gtk_spin_button_get_value_as_int (
			GTK_SPIN_BUTTON(win->cache_megs_sb));
		if (acache_max_megs == MAX_MEGS_VALUE_DEFAULT)
			pan_config_clean_key (KEY_CACHE_MAXIMUM_SIZE_MEGS);
		else
			pan_config_set_int (KEY_CACHE_MAXIMUM_SIZE_MEGS, acache_max_megs);
		acache_expire ();
	}

	pan_config_set_bool (KEY_CACHE_FLUSH_ON_EXIT,
	                     gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON( win->flush_cache_on_exit_check)));


	/* article headers */
	if (1) {
		gulong flags = get_header_flags ( );
		if (header_flags != flags) {
			pan_config_set_int ("/Pan/State/Headers", flags);
			header_flags = flags;
			text_refresh ();
		}
	}
	


	handle_editable (GTK_COMBO(win->external_editor_combo)->entry,
	                 &external_editor,
	                 KEY_APP_EDITOR);

	handle_editable (GTK_COMBO(win->external_browser_combo)->entry,
	                 &external_web_browser,
	                 KEY_APP_BROWSER);

	/* articlelist columns */

	if (handle_articlelist_columns (GTK_TREE_MODEL (win->column_store),
				    articlelist_columns,
				    COLUMN_TYPE_QTY,
				    KEY_ARTICLELIST_COLUMNS))
	{
		articlelist_update_columns ();
		articlelist_changed = TRUE;
	}

	/**
	**/

	if (articlelist_changed)
		articlelist_reset_style_nolock ();

	pan_config_sync ();

	prefs_init ();
	gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_CANCEL, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_APPLY, FALSE);
}


static void
ensure_trailing_slash (char ** pch)
{
	char * m = strrchr (*pch, G_DIR_SEPARATOR);
	if (!m || m[1]!='\0')
		replace_gstr (pch, g_strdup_printf ("%s%c", *pch, G_DIR_SEPARATOR));
}

/**
 * This is where all the user preferences get loaded in.
 */
void
prefs_init (void)
{
	char * pch;
	const char * march;
	int idx, col;
	debug_enter ("prefs_init");

	/* get the layout string */
	replace_gstr (&layout_str, pan_config_get_string (KEY_PANE_LAYOUT));
	if (!is_nonempty_string (layout_str))
		replace_gstr (&layout_str, g_strdup ("4gta"));

	/* get download directory */
	replace_gstr (&download_dir, pan_config_get_string ("/Pan/Paths/download_dir"));
	if (is_nonempty_string (download_dir))
		ensure_trailing_slash (&download_dir);
	else
		download_dir = g_build_filename (pan_get_home_dir(), "News", "Pan", NULL);


	acache_max_megs = pan_config_get_int (KEY_CACHE_MAXIMUM_SIZE_MEGS "=" stringify(MAX_MEGS_VALUE_DEFAULT));

	/* scoring preferences */
	replace_gstr (&score_editor_command, pan_config_get_string (KEY_APP_SCOREFILE_EDITOR "=" DEFAULT_SCORE_EDITOR_COMMAND));
	header_pane_color_subject_column = pan_config_get_bool (KEY_COLOR_SUBJECT_COLUMN "=FALSE");
	header_pane_color_score_column = pan_config_get_bool (KEY_COLOR_SCORE_COLUMN "=TRUE");

	/* display preferences */
	collapse_group_names = pan_config_get_bool(KEY_GROUP_PANE_COLLAPSE_GROUP_NAMES "=FALSE");
	text_set_wrap (pan_config_get_bool(KEY_BODY_PANE_WRAP_ENABLED "=FALSE"));
	show_group_pane  = pan_config_get_bool (KEY_GROUP_PANE_ENABLED "=TRUE");
	show_header_pane = pan_config_get_bool (KEY_HEADER_PANE_ENABLED "=TRUE");
	show_body_pane   = pan_config_get_bool (KEY_BODY_PANE_ENABLED "=TRUE");
	header_pane_is_threaded = pan_config_get_bool(KEY_HEADER_PANE_THREADING_ENABLED "=TRUE");
	text_window_smooth_scrolling = pan_config_get_bool(KEY_SMOOTH_SCROLLING_ENABLED "=TRUE");
	text_window_smooth_scrolling_speed = pan_config_get_int(KEY_SMOOTH_SCROLLING "=10");
	body_pane_monospace_font_enabled = pan_config_get_bool(KEY_BODY_PANE_MONOSPACE_FONT_ENABLED "=FALSE");
	group_pane_custom_font_enabled = pan_config_get_bool (KEY_GROUP_PANE_CUSTOM_FONT_ENABLED "=FALSE");
	header_pane_custom_font_enabled = pan_config_get_bool (KEY_HEADER_PANE_CUSTOM_FONT_ENABLED "=FALSE");
	body_pane_custom_font_enabled = pan_config_get_bool (KEY_BODY_PANE_CUSTOM_FONT_ENABLED "=FALSE");

	/* set the quote characters column */
	pch = pan_config_get_string (KEY_QUOTE_CHARS);
	if (!is_nonempty_string (pch))
		replace_gstr (&pch, g_strdup (">:}|"));
	text_massager_set_quote_chars (text_pane_get_text_massager(), pch);
	g_free (pch);

	/* don't let the user turn off _all_ the panes; that's too confusing. */
	if (!show_group_pane && !show_header_pane && !show_body_pane)
		show_group_pane = show_header_pane = show_body_pane = TRUE;

	/* behavior */
	text_massager_set_wrap_column (text_pane_get_text_massager(), pan_config_get_int (KEY_WRAP_COLUMN "=74"));
	hide_mpart_child_nodes = pan_config_get_bool (KEY_HIDE_MULTIPART_CHILDREN "=TRUE");
	expand_all_threads_by_default = pan_config_get_bool (KEY_EXPAND_ALL_THREADS_BY_DEFAULT "=FALSE");
	break_thread_when_subject_changes = pan_config_get_bool (KEY_BREAK_THREAD_WHEN_SUBJECT_CHANGES "=FALSE");
	fetch_new_and_bodies_on_startup = pan_config_get_bool (KEY_FETCH_NEW_AND_BODIES_ON_STARTUP "=FALSE");
	fetch_new_on_group_load = pan_config_get_bool (KEY_FETCH_NEW_ON_GROUP_LOAD "=TRUE");
	fetch_new_on_startup = pan_config_get_bool (KEY_FETCH_NEW_FROM_SUBSCRIBED_ON_STARTUP "=FALSE");
	single_click_selects_groups = pan_config_get_bool (KEY_SINGLE_CLICK_SELECTS_GROUPS "=FALSE");
	single_click_selects_headers = pan_config_get_bool (KEY_SINGLE_CLICK_SELECTS_HEADERS "=FALSE");


	header_flags = (gulong) pan_config_get_int ("/Pan/State/Headers=243");

	/**
	***  Unsorted
	**/

	do_spellcheck = pan_config_get_bool ("/Pan/Compose/do_spellcheck=true");

	replace_gstr (&thread_date_format, pan_config_get_string (KEY_THREAD_DATE_FORMAT));
	if (!is_nonempty_string (thread_date_format))
		replace_gstr (&thread_date_format, g_strdup ("%x %X"));

	replace_gstr (&external_editor, pan_config_get_string (KEY_APP_EDITOR));
	if (!external_editor)
		external_editor = g_strdup (DEFAULT_EXTERNAL_EDITOR);

	replace_gstr (&external_web_browser, pan_config_get_string (KEY_APP_BROWSER));
	if (!external_web_browser) {
		char * browser = getenv ("BROWSER");
		if (!is_nonempty_string(browser))
			browser = DEFAULT_WEB_BROWSER;
		external_web_browser = g_strdup (browser);
	}

	/* mail server preferences */
	replace_gstr (&mail_server_address, pan_config_get_string ("/Pan/Mail/smtp_address"));
	mail_server_port = pan_config_get_int ("/Pan/Mail/smtp_port=25");

	/**
	***  Fonts
	**/

	replace_gstr (&group_pane_custom_font, pan_config_get_string (PREFIX_DISPLAY "group_pane_custom_font"));
	if (!is_nonempty_string (group_pane_custom_font))
		replace_gstr (&group_pane_custom_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&header_pane_custom_font, pan_config_get_string (PREFIX_DISPLAY "header_pane_custom_font"));
	if (!is_nonempty_string (header_pane_custom_font))
		replace_gstr (&header_pane_custom_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&body_pane_custom_font, pan_config_get_string (PREFIX_DISPLAY "body_pane_custom_font"));
	if (!is_nonempty_string(body_pane_custom_font))
		replace_gstr (&body_pane_custom_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&body_pane_monospace_font, pan_config_get_string (PREFIX_DISPLAY "body_pane_monospace_font"));
	if (!is_nonempty_string(body_pane_monospace_font))
		replace_gstr (&body_pane_monospace_font, g_strdup(MONOSPACE_PANGO_DESCRIPTION));

	/**
	***  Colors
	**/

	set_color (&score_color[SCORE_COLOR_WATCHED][0],   KEY_COLOR_WATCHED_BG,  "#FFFF00"); /* Yellow */
	set_color (&score_color[SCORE_COLOR_WATCHED][1],   KEY_COLOR_WATCHED_FG,  "#FF0000"); /* Red */
	set_color (&score_color[SCORE_COLOR_HIGH][0],      KEY_COLOR_HIGH_BG,     "#00FF00"); /* Green */
	set_color (&score_color[SCORE_COLOR_HIGH][1],      KEY_COLOR_HIGH_FG,     "#000000"); /* Black */
	set_color (&score_color[SCORE_COLOR_MEDIUM][0],    KEY_COLOR_MEDIUM_BG,   "#FFFF00"); /* Green */
	set_color (&score_color[SCORE_COLOR_MEDIUM][1],    KEY_COLOR_MEDIUM_FG,   "#000000"); /* Black */
	set_color (&score_color[SCORE_COLOR_ZERO][0],      KEY_COLOR_ZERO_BG,     "#FFFFFF"); /* White */
	set_color (&score_color[SCORE_COLOR_ZERO][1],      KEY_COLOR_ZERO_FG,     "#000000"); /* Black */
	set_color (&score_color[SCORE_COLOR_LOW][0],       KEY_COLOR_LOW_BG,      "#545454"); /* Dim Slate Grey */
	set_color (&score_color[SCORE_COLOR_LOW][1],       KEY_COLOR_LOW_FG,      "#A8A8A8"); /* Light Grey */
	set_color (&score_color[SCORE_COLOR_IGNORED][0],   KEY_COLOR_IGNORED_BG,  "#000000"); /* Black */
	set_color (&score_color[SCORE_COLOR_IGNORED][1],   KEY_COLOR_IGNORED_FG,  "#A8A8A8"); /* Light Grey */

	set_color (&read_color,            KEY_COLOR_READ,            "#A8A8A8"); /* Light Grey*/
	set_color (&unread_color,          KEY_COLOR_UNREAD,          "#000000"); /* Black */
	set_color (&text_url_color,        KEY_COLOR_URL,             "#0000FF"); /* Blue */
	set_color (&text_quoted_color[0],  KEY_COLOR_QUOTED_1,        "#660066"); 
	set_color (&text_quoted_color[1],  KEY_COLOR_QUOTED_2,        "#990000"); /* Forest Green */
	set_color (&text_quoted_color[2],  KEY_COLOR_QUOTED_3,        "#000099"); /* Dark Purple */
	set_color (&signature_color, 	   KEY_COLOR_SIGNATURE,	      "#007777"); 

	/**
	***
	**/

	pch = pan_config_get_string (KEY_ARTICLELIST_COLUMNS "=1|2|4|3|5|6|7");
	march = pch;
	for (idx=0; (col = get_next_token_int (march, '|', &march)) != 0; idx++)
	{
		--col;
		if (col < COLUMN_ACTION_STATE || col >= COLUMN_TYPE_QTY)
			continue;

		articlelist_columns[idx] = col;
	}
	g_free (pch);

	base_prefs_init (download_dir,
	                 acache_max_megs,
			 pan_config_get_bool(KEY_CACHE_FLUSH_ON_EXIT "=FALSE"),
	                 break_thread_when_subject_changes);

	debug_exit ("prefs_init");
}

void
show_group_substitution_help_dialog (gpointer window)
{
	const char * str = _("%g - group as one directory (alt.binaries.pictures.trains)\n"
	                     "%G - group as nested directory (/alt/binaries/pictures/trains)\n"
	                     " \n"
	                     "\"/home/user/News/Pan/%g\" becomes\n"
	                     "\"/home/user/News/Pan/alt.binaries.pictures.trains\", and\n"
	                     "\"/home/user/News/Pan/%G\" becomes\n"
	                     "\"/home/user/News/Pan/alt/binaries/pictures/trains\",");
	GtkWidget * w = gtk_message_dialog_new (GTK_WINDOW(window),
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_MESSAGE_INFO,
	                                        GTK_BUTTONS_CLOSE, "%s", str);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT (w));
	gtk_widget_show_all (w);
}

/****
*****
*****   THE CONFIGURE PAGES
*****
*****   * font_page
*****   * color_page
*****   * layout_page
*****   * header_pane_page
*****   * body_pane_page
*****   * behavior_page
*****   * apps_and_mail_page
*****
****/

/**
***  Font Page
**/

static void
custom_font_check_toggled (GtkToggleButton * tb, gpointer user_data)
{
	GtkWidget * font_picker = GTK_WIDGET(user_data);
	const gboolean custom_font_enabled = gtk_toggle_button_get_active (tb);
	gtk_widget_set_sensitive (font_picker, custom_font_enabled);
}

static GtkWidget*
font_page (void)
{
	GtkWidget * w;
	GtkWidget * t;
	GtkWidget * l;
	int row;
	int items_in_section;
	char buf[512];

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Fonts"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* group pane font */
		l = gtk_check_button_new_with_mnemonic (_("Use custom font in Group Pane:"));
		win->group_pane_custom_font_enabled_check = l; 
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(l), group_pane_custom_font_enabled);
		connect_signal_to_prefs_changed (l, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->group_pane_custom_font_gfp = pan_font_picker_new ();
		g_signal_connect (l, "toggled", G_CALLBACK(custom_font_check_toggled), w);
		gtk_widget_set_sensitive (w, group_pane_custom_font_enabled);
		pan_font_picker_set_font (w, group_pane_custom_font);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* header pane font */
		l = gtk_check_button_new_with_mnemonic (_("Use custom font in Header Pane:"));
		win->header_pane_custom_font_enabled_check = l; 
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(l), header_pane_custom_font_enabled);
		connect_signal_to_prefs_changed (l, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->header_pane_custom_font_gfp = pan_font_picker_new ();
		g_signal_connect (l, "toggled", G_CALLBACK(custom_font_check_toggled), w);
		gtk_widget_set_sensitive (w, header_pane_custom_font_enabled);
		pan_font_picker_set_font (w, header_pane_custom_font);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* body pane font */
		l = gtk_check_button_new_with_mnemonic (_("Use custom font in Body Pane:"));
		win->body_pane_custom_font_enabled_check = l; 
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(l), body_pane_custom_font_enabled);
		connect_signal_to_prefs_changed (l, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->body_pane_custom_font_gfp = pan_font_picker_new ();
		gtk_widget_set_sensitive (w, body_pane_custom_font_enabled);
		g_signal_connect (l, "toggled", G_CALLBACK(custom_font_check_toggled), w);
		pan_font_picker_set_font (w, body_pane_custom_font);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* body pane monospace font */
		l = gtk_label_new_with_mnemonic (_("Monospac_e font:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);
		w = win->body_pane_monospace_font_gfp = pan_font_picker_new ();
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		pan_font_picker_set_font (w, body_pane_monospace_font);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}

/**
***  Colors Page
**/

static GtkWidget*
color_page (void)
{
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * t;
	GtkWidget * l;
	int row;
	int items_in_section;
	char buf[512];

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Header Pane"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* read threads */
		l = gtk_label_new_with_mnemonic (_("_Read Threads:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->read_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &read_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* unread threads */
		l = gtk_label_new_with_mnemonic (_("_Unread Threads:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->unread_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &unread_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Header Pane Scoring"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 6;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* color the subject column? */
		w = win->color_subject_column_check = gtk_check_button_new_with_mnemonic (_("Color the \"sub_ject\" column"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), header_pane_color_subject_column);
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* very high scoring articles */
		l = gtk_label_new (_("Articles with a Score of 9999 or higher:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);

			l = win->score_lb[SCORE_COLOR_WATCHED][1] = gtk_label_new (_("Text:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_WATCHED][1] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_WATCHED][1]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

			l = win->score_lb[SCORE_COLOR_WATCHED][0] = gtk_label_new (_("Background:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_WATCHED][0] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_WATCHED][0]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

		++row;

		/* high scoring articles */
		l = gtk_label_new (_("Articles with a Score between 5000 and 9998"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);

			l = win->score_lb[SCORE_COLOR_HIGH][1] = gtk_label_new (_("Text:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_HIGH][1] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_HIGH][1]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

			l = win->score_lb[SCORE_COLOR_HIGH][0] = gtk_label_new (_("Background:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_HIGH][0] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_HIGH][0]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

		++row;

		/* low scoring articles */
		l = gtk_label_new (_("Articles with a Score between 1 and 4999"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);

			l = win->score_lb[SCORE_COLOR_MEDIUM][1] = gtk_label_new (_("Text:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_MEDIUM][1] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_MEDIUM][1]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

			l = win->score_lb[SCORE_COLOR_MEDIUM][0] = gtk_label_new (_("Background:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_MEDIUM][0] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_MEDIUM][0]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

		++row;

		/* low scoring articles */
		l = gtk_label_new (_("Articles with a Score between -9998 and -1"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);

			l = win->score_lb[SCORE_COLOR_LOW][1] = gtk_label_new (_("Text:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_LOW][1] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_LOW][1]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

			l = win->score_lb[SCORE_COLOR_LOW][0] = gtk_label_new (_("Background:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_LOW][0] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_LOW][0]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

		++row;

		/* very low scoring articles */
		l = gtk_label_new (_("Articles with a Score of -9999 or lower"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);

			l = win->score_lb[SCORE_COLOR_IGNORED][1] = gtk_label_new (_("Text:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_IGNORED][1] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_IGNORED][1]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

			l = win->score_lb[SCORE_COLOR_IGNORED][0] = gtk_label_new (_("Background:"));
			gtk_box_pack_start_defaults (GTK_BOX(h), l);

			w = win->score_pcp[SCORE_COLOR_IGNORED][0] = pan_color_picker_new ();
			pan_color_picker_set_color (w, &score_color[SCORE_COLOR_IGNORED][0]);
			connect_signal_to_prefs_changed (w, "clicked");
			gtk_box_pack_start_defaults (GTK_BOX(h), w);

		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Body Pane"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 5;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* quoted text 1 */
		l = gtk_label_new_with_mnemonic (_("Quoted Text _1:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_quoted_pcp[0] = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_quoted_color[0]);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* quoted text 2 */
		l = gtk_label_new_with_mnemonic (_("Quoted Text _2:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_quoted_pcp[1] = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_quoted_color[1]);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* quoted text 3 */
		l = gtk_label_new_with_mnemonic (_("Quoted Text _3:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_quoted_pcp[2] = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_quoted_color[2]);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* signature */
		l = gtk_label_new_with_mnemonic (_("Signature"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->signature_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &signature_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;
		
		/* URLs */
		l = gtk_label_new_with_mnemonic (_("_URLs:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		w = win->text_url_pcp = pan_color_picker_new ();
		pan_color_picker_set_color (w, &text_url_color);
		connect_signal_to_prefs_changed (w, "clicked");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}

/**
***  Layout Page
**/

#define LAYOUT_QTY 6

static char*
layout_get_new_string (GtkWidget * page)
{
	int i;
	int layout = -1;
	const char * order = "tag";
	GtkWidget ** layout_buttons = (GtkWidget**) g_object_get_data (G_OBJECT(page), "layout_buttons");
	GSList * l = (GSList*) g_object_get_data (G_OBJECT(page), "layout_radio_group");

	/* get the layout number */
	for (i=0; i<LAYOUT_QTY; ++i) {
		GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
		if (gtk_toggle_button_get_active(tb)) {
			layout = i + 1;
			break;
		}
	}

	/* get the ordering string */
	for (; l!=NULL; l=l->next) {
		GtkRadioButton * rb = GTK_RADIO_BUTTON(l->data);
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(rb))) {
			order = (const char*) g_object_get_data (G_OBJECT(rb), "layout");
			g_assert (order != NULL);
			g_assert (strlen(order) == 3);
			break;
		}
	}

	/* build & return the string */
	return g_strdup_printf ("%d%s", layout, order);
}

static void
layout_changed_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	GtkWidget ** layout_buttons = (GtkWidget**) user_data;
	static gboolean dampen_feedback_loop = FALSE;

	if (!dampen_feedback_loop) {
		int i;
		dampen_feedback_loop = TRUE;
		for (i=0; i<LAYOUT_QTY; ++i) {
			GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
			gboolean active = togglebutton==tb;
			if (gtk_toggle_button_get_active(tb) != active)
				gtk_toggle_button_set_active (tb, active);
		}
		dampen_feedback_loop = FALSE;
	}
}


static GtkWidget*
layout_page (void)
{
	int i;
	GtkWidget * h;
	GtkWidget * w;
	GtkWidget * radio;
	GtkWidget * l;
	GtkWidget * t;
	const char * pch_group;
	const char * pch_header;
	const char * pch_body;
	const char lch = *layout_str;
	int row;
	char buf [256];
	const guint8 * inline_txt [LAYOUT_QTY] = {
		icon_layout_1, icon_layout_2, icon_layout_3,
		icon_layout_4, icon_layout_5, icon_layout_6 };
	GtkWidget ** layout_buttons = g_malloc (sizeof(GtkWidget*) * LAYOUT_QTY);
	int items_in_section;

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Pane Layout"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 1;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* pane layout */
		h = gtk_hbox_new (FALSE, 6);
		gtk_container_set_border_width (GTK_CONTAINER (h), 0);
		for (i=0; i<LAYOUT_QTY; ++i)
		{
			char buf[64];
			GdkPixbuf * pixbuf;
			GdkPixmap * pixmap;
			GdkBitmap * bitmap;

			g_snprintf (buf, sizeof(buf), "%d", i+1);
			w = gtk_toggle_button_new ();
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), *buf==lch);

			pixbuf = gdk_pixbuf_new_from_inline (-1, inline_txt[i], FALSE, NULL);
			gdk_pixbuf_render_pixmap_and_mask_for_colormap (pixbuf, gtk_widget_get_colormap(w), &pixmap, &bitmap, 128);
			gtk_container_add (GTK_CONTAINER(w), gtk_image_new_from_pixmap (pixmap, bitmap));

			g_object_unref (pixbuf);
			g_object_unref (pixmap);
			g_object_unref (bitmap);

			g_signal_connect (w, "toggled", G_CALLBACK(layout_changed_cb), layout_buttons);
			connect_signal_to_prefs_changed (w, "toggled");
			layout_buttons[i] = w;
			gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
		}
		gtk_table_attach_defaults (GTK_TABLE(t), h, 1, 2, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Pane Order"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 6;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* pane order */
		pch_group = _("Group");
		pch_header = _("Headers");
		pch_body = _("Body");
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_group, pch_header, pch_body);
		w = radio = gtk_radio_button_new_with_mnemonic_from_widget (NULL, buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"gta"));
		g_object_set_data (G_OBJECT(w), "layout", "gta");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_group, pch_body, pch_header);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"gat"));
		g_object_set_data (G_OBJECT(w), "layout", "gat");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_header, pch_group, pch_body);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"tga"));
		g_object_set_data (G_OBJECT(w), "layout", "tga");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_header, pch_body, pch_group);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"tag"));
		g_object_set_data (G_OBJECT(w), "layout", "tag");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_body, pch_group, pch_header);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"agt"));
		g_object_set_data (G_OBJECT(w), "layout", "agt");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);
		++row;
		g_snprintf (buf, sizeof(buf), "1=%s, 2=%s, 3=%s", pch_body, pch_header, pch_body);
		w = gtk_radio_button_new_with_mnemonic_from_widget (GTK_RADIO_BUTTON(w), buf);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !strcmp(layout_str+1,"atg"));
		g_object_set_data (G_OBJECT(w), "layout", "atg");
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);
		++row;

	g_object_set_data_full (G_OBJECT(t), "layout_buttons", layout_buttons, g_free);
	g_object_set_data (G_OBJECT(t), "layout_radio_group", gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio)));
	return t;
}

/**
***  Header Pane Page
**/

static const char *
articlelist_column_to_prefs_label (int type)
{
	switch (type)
	{
		case COLUMN_ACTION_STATE: return _("Action State");
		case COLUMN_ARTICLE_STATE: return _("Article State");
		case COLUMN_SCORE: return _("Score");
		case COLUMN_SUBJECT: return _("Subject");
		case COLUMN_LINES: return _("Lines");
		case COLUMN_AUTHOR: return _("Author");
		case COLUMN_DATE: return _("Date");
		default: break;
	}

	pan_warn_if_reached();
	return "BUG!!!";
}


static void
articlelist_column_move (int up)
{
	GtkTreeModel *model;
	GtkTreeIter iter1, iter2;
	GtkTreeSelection *selection;
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(win->column_tree_view));
	if (gtk_tree_selection_get_selected (selection, &model, &iter1))
	{
		GtkTreePath *old_path;
		GtkTreePath *new_path;

		/* ARG! Why doesn't gtk_tree_model have */
		/* an API to move up AND down easily??  */

		old_path = gtk_tree_model_get_path (model, &iter1);
		new_path = gtk_tree_model_get_path (model, &iter1);

		if (up)
			gtk_tree_path_prev (new_path);
		else
			gtk_tree_path_next (new_path);

		if (gtk_tree_path_compare (old_path, new_path) != 0 &&
		    gtk_tree_model_get_iter (model, &iter2, new_path))
		{
			int col;

			/* In GTK2.2, we could call gtk_list_store_swap () */

			gtk_tree_model_get (model, &iter1, 1, &col, -1);
			gtk_list_store_remove (win->column_store, &iter1);
		
			if (up)
				gtk_list_store_insert_before (win->column_store, &iter1, &iter2);
			else
				gtk_list_store_insert_after (win->column_store, &iter1, &iter2);

			gtk_list_store_set (win->column_store, &iter1,
				0, articlelist_column_to_prefs_label(col),
				1, col,
				-1);

			/* Keep the row selected */
			gtk_tree_selection_select_iter (selection, &iter1);
			gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(win->column_tree_view), new_path, NULL, TRUE, 0.5, 0.5);
		}

		gtk_tree_path_free (old_path);
		gtk_tree_path_free (new_path);
	}
}

static void
articlelist_column_up_button_clicked_cb (GtkButton * button, gpointer data)
{
	articlelist_column_move (1);

}

static void
articlelist_column_down_button_clicked_cb (GtkButton * button, gpointer data)
{
	articlelist_column_move (0);
}

static GtkWidget*
header_pane_page (void)
{
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * t;
	GtkWidget * l;
	GtkWidget * hbox;
	GtkWidget * bbox;
	GtkCellRenderer * renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection * selection;

	int idx;
	int row;
	int items_in_section;
	char buf[512];

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Column Layout"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 1;

		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* column ordering list */
		/* work area */
		hbox = gtk_hbox_new (FALSE, GUI_PAD);
		gtk_container_set_border_width (GTK_CONTAINER(hbox), 12);
		gtk_table_attach_defaults (GTK_TABLE(t), hbox, 1, 2, row, row+1);

		/* create list store & tree view */
		win->column_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_UINT);
		win->column_tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(win->column_store));
		gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(win->column_tree_view), FALSE);

		/* add the column to the view */
		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes (_("Column"),
				renderer, "text", 0, NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW (win->column_tree_view), column);
		
		/* set the selection mode */
		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(win->column_tree_view));
		gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

		/* populate the list store */

		for (idx=0; idx<COLUMN_TYPE_QTY; idx++)
		{
			GtkTreeIter iter;

			gtk_list_store_append (win->column_store, &iter);
			gtk_list_store_set (win->column_store, &iter,
				0, articlelist_column_to_prefs_label (articlelist_columns[idx]),
				1, articlelist_columns[idx], 
				-1);
		}

		/* containing widget */
		w = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(w), win->column_tree_view);
		gtk_widget_set_usize (w, 150u, 150u);
		gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);

		/* controlling buttons */
		bbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
		gtk_box_pack_start (GTK_BOX(hbox), bbox, FALSE, FALSE, 0);

		/* up button */
		w = gtk_button_new_from_stock (GTK_STOCK_GO_UP);
		gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
		g_signal_connect (w, "clicked", 
			G_CALLBACK(articlelist_column_up_button_clicked_cb), 
			NULL);
		connect_signal_to_prefs_changed (w, "clicked");
		win->column_up_cbutton = w;

		/* down button */
		w = gtk_button_new_from_stock (GTK_STOCK_GO_DOWN);
		gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
		g_signal_connect (w, "clicked", 
			G_CALLBACK(articlelist_column_down_button_clicked_cb), 
			NULL);
		connect_signal_to_prefs_changed (w, "clicked");
		win->column_down_cbutton = w;
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Other Settings"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* break threads when subject changes? */
		w = gtk_check_button_new_with_mnemonic (_("When a Followup subject header changes, show as _new thread"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), break_thread_when_subject_changes);
		connect_signal_to_prefs_changed (w, "toggled");
		win->break_thread_when_subject_changes_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* expand threads by default? */
		w = gtk_check_button_new_with_mnemonic (_("E_xpand all threads by default"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), expand_all_threads_by_default);
		connect_signal_to_prefs_changed (w, "toggled");
		win->expand_all_threads_by_default_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* multipart: one node or all */
		w = gtk_check_button_new_with_mnemonic (_("Show complete _multipart posts as a single article"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), hide_mpart_child_nodes);
		connect_signal_to_prefs_changed (w, "toggled");
		win->hide_mpart_child_nodes_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* date format label */
		l = gtk_label_new_with_mnemonic (_("Date Forma_t:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* date format control */
		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		w = gtk_entry_new_with_max_length (128);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		pan_gtk_entry_set_text (w, thread_date_format);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
		win->thread_date_entry = w;
		w = gtk_button_new_with_mnemonic (_("_Help"));
		g_signal_connect (w, "clicked",
				  G_CALLBACK (date_help_clicked_cb), NULL);
		gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);
		++row;

	return t;
}

/**
***  Body Pane Page
**/

static void
smooth_scrolling_check_toggled (GtkToggleButton * tb, gpointer user_data)
{
	gtk_widget_set_sensitive (GTK_WIDGET(win->smooth_scrolling_speed_sb), gtk_toggle_button_get_active(tb));
}

static gulong
get_header_flags (void)
{
	guint i;
	gulong flags = 0;
	GPtrArray * header_toggles = (GPtrArray*) g_object_get_data (G_OBJECT(win->body_pane_page), "header_toggles");

	for (i=0; i<header_toggles->len; ++i) {
		GtkWidget * w = GTK_WIDGET (g_ptr_array_index (header_toggles, i));
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w)))
			flags |= GPOINTER_TO_UINT (g_object_get_data (G_OBJECT(w), "bit"));
	}

	return flags;
}

typedef struct
{
	char* text;
	gulong flag;
}
MessageListThrowawayStruct;

static GtkWidget*
body_pane_page (void)
{
	GtkAdjustment * adj;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * t;
	GtkWidget * l;
	GPtrArray * header_toggles = g_ptr_array_new ();
	int row;
	int items_in_section;
	char * pch;
	char buf[512];
	const gulong flags = (gulong) pan_config_get_int ("/Pan/State/Headers=243");

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Headers"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 8;

		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

#define ADD_HEADER_TOGGLE(name,bit) \
		g_snprintf (buf, sizeof(buf), _("Show \"%s\" header"), name); \
		w = gtk_check_button_new_with_mnemonic (buf); \
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), (flags & bit?1:0)); \
		g_object_set_data (G_OBJECT(w), "bit", GUINT_TO_POINTER(bit)); \
		connect_signal_to_prefs_changed (w, "toggled"); \
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1); \
		g_ptr_array_add (header_toggles, w); \
		++row;
		ADD_HEADER_TOGGLE(_("From"), UI_HEADER_AUTHOR)
		ADD_HEADER_TOGGLE(_("Subject"), UI_HEADER_SUBJECT)
		ADD_HEADER_TOGGLE(_("Date"), UI_HEADER_DATE)
		ADD_HEADER_TOGGLE(_("Reply-To"), UI_HEADER_REPLY_TO)
		ADD_HEADER_TOGGLE(_("Followup-To"), UI_HEADER_FOLLOWUP_TO)
		ADD_HEADER_TOGGLE(_("Newsgroups"), UI_HEADER_NEWSGROUPS)
		ADD_HEADER_TOGGLE(_("Message-ID"), UI_HEADER_MESSAGE_ID)
		ADD_HEADER_TOGGLE(_("Newsreader"), UI_HEADER_NEWSREADER)
#undef ADD_HEADER_TOGGLE

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile misc section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Other Settings"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);


		/* quoted text label */
		l = gtk_label_new_with_mnemonic (_("Characters denoting _quoted text:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* quoted text control */
		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		w = gtk_entry_new_with_max_length (20);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		pch = text_massager_get_quote_chars (text_pane_get_text_massager());
		gtk_entry_set_text (GTK_ENTRY(w), pch);
		g_free (pch);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
		win->text_quoted_chars_entry = w;
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);
		++row;

		/* smooth scrolling label */
		w = gtk_check_button_new_with_mnemonic (_("S_mooth scrolling at speed:"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), text_window_smooth_scrolling);
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(smooth_scrolling_check_toggled), 0);
		win->smooth_scrolling_check = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 2, row, row+1);

		/* smooth scrolling control */
		adj = (GtkAdjustment *) gtk_adjustment_new (1.0, 1.0, 10.0, 1.0, 1.0, 0.0);
		w = gtk_spin_button_new (adj, 0, 0);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), text_window_smooth_scrolling_speed);
		connect_signal_to_prefs_changed (w, "changed");
		win->smooth_scrolling_speed_sb = w;
		gtk_widget_set_sensitive (w, text_window_smooth_scrolling);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;


	g_object_set_data (G_OBJECT(t), "header_toggles", header_toggles);
	return t;
}

/**
***  Behavior Page
**/

static GtkWidget*
behavior_page (void)
{
	int row;
	int items_in_section;
	GtkAdjustment * adj;
	GtkWidget * t;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * l;
	char buf[512];
	char * p;
	const gboolean flush_on_exit = pan_config_get_bool (KEY_CACHE_FLUSH_ON_EXIT "=FALSE");

	row = 0;
	t = gtk_table_new (4, 16, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Mouse"));
	w = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* mouse behavior - group pane */
		w = gtk_check_button_new_with_mnemonic (_("single-clic_k selects, rather than loads, groups"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), single_click_selects_groups);
		connect_signal_to_prefs_changed (w, "toggled");
		win->single_click_selects_groups_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* mouse behavior - header pane */
		w = gtk_check_button_new_with_mnemonic (_("single-click selects, rather than loads, heade_rs"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), single_click_selects_headers);
		connect_signal_to_prefs_changed (w, "toggled");
		win->single_click_selects_headers_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Automatic Downloads"));
	w = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

		items_in_section = 3;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* download new headers */
		w = gtk_check_button_new_with_mnemonic (
			_("Download new headers when loading a _group"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_group_load);
		connect_signal_to_prefs_changed (w, "toggled");
		win->fetch_new_on_group_load_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* download new headers on startup */
		w = gtk_check_button_new_with_mnemonic (
			_("Download new headers from s_ubscribed groups when starting Pan"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_startup);
		connect_signal_to_prefs_changed (w, "toggled");
		win->fetch_new_on_startup_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* download new headers and bodies on startup */
		w = gtk_check_button_new_with_mnemonic (
			_("Download new headers and bod_ies from subscribed groups when starting Pan"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_and_bodies_on_startup);
		connect_signal_to_prefs_changed (w, "toggled");
		win->fetch_new_and_bodies_on_startup_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Tasks"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 3;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* download directory */
		l = gtk_label_new_with_mnemonic (_("Save a_ttachments in directory:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
		w = win->dir_download = pan_file_entry_new (_("Download Directory"));
		if ((p = pan_config_get_string ("/Pan/Paths/download_dir")) == NULL)
			p = g_build_filename (pan_get_home_dir(), "News", "Pan", NULL);
		pan_file_entry_set (win->dir_download, p);
		g_free (p);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), pan_file_entry_gtk_entry(w));
		connect_signal_to_prefs_changed (pan_file_entry_gtk_entry(w), "changed");
		gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

		w = gtk_button_new_with_mnemonic (_("_Help"));
		g_signal_connect_swapped (w, "clicked", G_CALLBACK (show_group_substitution_help_dialog), win->dialog);
		gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
		gtk_table_attach_defaults (GTK_TABLE(t), h, 3, 4, row, row+1);
		++row;

		/* remove failed tasks */
		w = gtk_check_button_new_with_mnemonic (_("Automatically remove _failed tasks from the task manager"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), queue_get_remove_failed_tasks());
		connect_signal_to_prefs_changed (w, "toggled");
		win->remove_failed_tasks_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Article Cache"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* flush on exit */
		w = gtk_check_button_new_with_mnemonic (_("_Erase cache when exiting Pan"));
		win->flush_cache_on_exit_check = w;
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), flush_on_exit);
		connect_signal_to_prefs_changed (w, "toggled");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;

		/* max cache size */
		l = gtk_label_new_with_mnemonic (_("Ma_ximum Cache Size (Megs):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		adj = (GtkAdjustment*) gtk_adjustment_new  (acache_max_megs, 1.0, 20000, 1.0, 1.0, 1.0);
		w = win->cache_megs_sb = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 1.0, 0);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

/* this is an odd duck.  Let's comment it out and see if anyone requests to have it back. */
#if 0
	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Online"));
	w = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), w, 0, 4, row, row+1);
	++row;

		items_in_section = 1;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* download new headers */
		w = gtk_check_button_new_with_mnemonic (_("Start Pan in o_nline mode"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), pan_config_get_bool (PREFIX_GENERAL "=TRUE"));
		connect_signal_to_prefs_changed (w, "toggled");
		win->online_on_startup_cbutton = w;
		gtk_table_attach_defaults (GTK_TABLE(t), w, 1, 4, row, row+1);
		++row;
#endif

	return t;
}

/**
***  Apps and Mail Page
**/

static GtkWidget*
apps_and_mail_page (void)
{
	GtkWidget * w;
	GtkWidget * t;
	GtkWidget * l;
	GList * s;
	int row;
	int items_in_section;
	char buf[512];
	char * p;

	row = 0;
	
	t = gtk_table_new (4, 100, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(t), 6);

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Applications"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 4;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* external browser label */
		l = gtk_label_new_with_mnemonic (_("_Web Browser (%s is URL):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* external browser combo */
		w = win->external_browser_combo = gtk_combo_new ();
		gtk_widget_set_usize (w, 300u, 0u);
		s = NULL;
#ifdef G_OS_WIN32
		s = g_list_append (s, DEFAULT_WEB_BROWSER);
		s = g_list_append (s, "netscape '%s'");
		s = g_list_append (s, "netscape -remote 'openURL(%s,raise)'");
		s = g_list_append (s, "mozilla -remote 'openURL(%s)'");
#endif
#ifdef G_OS_UNIX
		s = g_list_append (s, "gnome-moz-remote --raise --newwin '%s'");
		s = g_list_append (s, "galeon --existing '%s'");
		s = g_list_append (s, "galeon --new-tab '%s'");
		s = g_list_append (s, "netscape '%s'");
		s = g_list_append (s, "netscape -remote 'openURL(%s,raise)'");
		s = g_list_append (s, "mozilla -remote 'openURL(%s)'");
		s = g_list_append (s, "konqueror '%s'");
		s = g_list_append (s, "xterm -e w3m \"%s\"");
		s = g_list_append (s, "xterm -e lynx \"%s\"");
		s = g_list_append (s, "xterm -e links \"%s\"");
		s = g_list_append (s, "konsole -e w3m \"%s\"");
		s = g_list_append (s, "konsole -e lynx \"%s\"");
		s = g_list_append (s, "konsole -e links \"%s\"");
		s = g_list_append (s, "gnome-terminal -e w3m \"%s\"");
		s = g_list_append (s, "gnome-terminal -e lynx \"%s\"");
		s = g_list_append (s, "gnome-terminal -e links \"%s\"");
#endif
		gtk_combo_set_popdown_strings (GTK_COMBO(w), s);
		g_list_free (s);
		pan_gtk_entry_set_text (GTK_COMBO(w)->entry, external_web_browser ? external_web_browser : NULL);
		connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), GTK_COMBO(w)->entry);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* external editor label */
		l = gtk_label_new_with_mnemonic (_("_Editor (%t is filename):"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* external editor combo */
		w = win->external_editor_combo = gtk_combo_new ();
		gtk_widget_set_usize (w, 300u, 0u);
		s = NULL;
#ifdef G_OS_WIN32
		s = g_list_append (s, "notepad %t");
		s = g_list_append (s, "pfe %t");
		s = g_list_append (s, "vim %t");
#endif
#ifdef G_OS_UNIX
		s = g_list_append (s, "gedit %t");
		s = g_list_append (s, "mgedit --no-fork %t");
		s = g_list_append (s, "emacs %t");
		s = g_list_append (s, "xemacs %t");
		s = g_list_append (s, "xterm -e jed %t");
		s = g_list_append (s, "xterm -e vi %t");
		s = g_list_append (s, "konsole -e jed %t");
		s = g_list_append (s, "konsole -e vi %t");
		s = g_list_append (s, "gnome-terminal -e jed %t");
		s = g_list_append (s, "gnome-terminal -e vi %t");
#endif
		gtk_combo_set_popdown_strings (GTK_COMBO(w), s);
		g_list_free (s);
		pan_gtk_entry_set_text (GTK_COMBO(w)->entry, external_editor);
		connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), GTK_COMBO(w)->entry);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* external scorefile editor label */
		l = gtk_label_new_with_mnemonic (_("Scorefile _Editor:\n(%t is filename, %n is line number)"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* external editor combo */
		w = win->score_editor_command_combo = gtk_combo_new ();
		//gtk_widget_set_usize (w, 300u, 0u);
		s = NULL;
#ifdef G_OS_WIN32
		s = g_list_append (s, "notepad %t");
		s = g_list_append (s, "pfe %t");
#endif
#ifdef G_OS_UNIX
		s = g_list_append (s, "xterm -e vi +%n %t");
		s = g_list_append (s, "gedit +%n %t");
		s = g_list_append (s, "emacs +%n %t");
		s = g_list_append (s, "xemacs +%n %t");
#endif
		gtk_combo_set_popdown_strings (GTK_COMBO(w), s);
		g_list_free (s);

		pan_gtk_entry_set_text (GTK_COMBO(w)->entry, score_editor_command);
		connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), GTK_COMBO(w)->entry);
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* scorefile label */
		l = gtk_label_new_with_mnemonic (_("Score_file:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* scorefile entry  */
		w = win->scorefile_pfe = pan_file_entry_new (_("Scorefile"));
		p = pan_config_get_string (KEY_SCOREFILE_DEFAULT);
		replace_gstr (&p, pan_file_normalize (p, NULL));
		pan_file_entry_set (w, p);
		g_free (p);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), pan_file_entry_gtk_entry(w));
		connect_signal_to_prefs_changed (pan_file_entry_gtk_entry(w), "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;



	/* section spacer */
	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 0u, 6u);
	gtk_table_attach (GTK_TABLE(t), w, 0, 4, row, row+1, 0, 0, 0, 0);
	++row;

	/* title for profile info section */
	g_snprintf (buf, sizeof(buf), "<b>%s</b>", _("Sending Mail"));
	l = gtk_label_new (buf);
	gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
	gtk_label_set_use_markup (GTK_LABEL(l), TRUE);
	gtk_table_attach_defaults (GTK_TABLE(t), l, 0, 4, row, row+1);
	++row;

		items_in_section = 2;
	
		/* spacer to move the fields a little to the right of the name header */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 18u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 0, 1, row, row+items_in_section, 0, 0, 0, 0);

		/* spacer between the controls and their labels */
		w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
		gtk_widget_set_usize (w, 12u, 0u);
		gtk_table_attach (GTK_TABLE(t), w, 2, 3, row, row+items_in_section, 0, 0, 0, 0);

		/* address label */
		l = gtk_label_new_with_mnemonic (_("Mail Server Add_ress:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* address control */
		w = win->smtp_address = gtk_entry_new ();
		if (is_nonempty_string(mail_server_address))
			pan_gtk_entry_set_text (w, mail_server_address);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

		/* address label */
		l = gtk_label_new_with_mnemonic (_("Ma_il Server Por_t:"));
		gtk_misc_set_alignment (GTK_MISC(l), 0.0f, 0.5f);
		gtk_table_attach_defaults (GTK_TABLE(t), l, 1, 2, row, row+1);

		/* address control */
		w = win->smtp_port = gtk_spin_button_new (
			GTK_ADJUSTMENT(gtk_adjustment_new (mail_server_port, 0, 65536, 1, 1, 1)), 1, 0);
		gtk_label_set_mnemonic_widget (GTK_LABEL(l), w);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach_defaults (GTK_TABLE(t), w, 3, 4, row, row+1);
		++row;

	return t;
}


/***
****  
****  
****  
***/

static void
prefs_destroy_cb (GtkObject * object, gpointer user_data)
{
	/* cleanup main window */
	g_free (win);
	win = NULL;
}

static void
prefs_response_cb (GtkDialog * dialog, int response, gpointer user_data)
{
	if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_APPLY)
		prefs_apply ();

	if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_CANCEL)
		gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
prefs_spawn (void)
{
	GtkWidget * w;
	GtkWidget * v;
	GtkWidget * c;

	/* There can be only one! (properties window) */
	if (win) {
		gdk_window_raise (GTK_WIDGET(win->dialog)->window);
		return;
	}

	win = g_new0 (PrefsWindow, 1);

	win->dialog = w = gtk_dialog_new_with_buttons (_("Pan Preferences"),
	                                               GTK_WINDOW(Pan.window),
	                                               GTK_DIALOG_DESTROY_WITH_PARENT,
	                                               GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                               GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
	                                               GTK_STOCK_OK, GTK_RESPONSE_OK,
	                                               NULL);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(w), GTK_RESPONSE_OK, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(w), GTK_RESPONSE_APPLY, FALSE);

	/* put a notebook in the dialog's workarea */
	w = win->notebook = gtk_notebook_new ();
	gtk_container_set_border_width (GTK_CONTAINER (w), 18);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(win->dialog)->vbox), w, TRUE, TRUE, 0);

	/* add the tabs */
#define add_tab(tab,func) \
	c = func(); \
	gtk_container_set_border_width (GTK_CONTAINER(c), 18); \
	v = gtk_vbox_new (FALSE, 0); \
	gtk_box_pack_start (GTK_BOX(v), c, FALSE, FALSE, 0); \
	gtk_notebook_append_page (GTK_NOTEBOOK(w), v, gtk_label_new_with_mnemonic (tab));
	add_tab (_("Beha_vior"), behavior_page)
	add_tab (_("Pane _Layout"), win->layout_page=layout_page)
	add_tab (_("Hea_der Pane"), header_pane_page)
	add_tab (_("Bod_y Pane"), win->body_pane_page=body_pane_page)
	add_tab (_("Fonts"), font_page)
	add_tab (_("Colors"), color_page)
	add_tab (_("A_pps & Mail"), apps_and_mail_page)
#undef add_tab

	/* callbacks */
	g_signal_connect (win->dialog, "response", G_CALLBACK(prefs_response_cb), NULL);
	g_signal_connect (win->dialog, "destroy", G_CALLBACK(prefs_destroy_cb), NULL);

	gtk_widget_show_all (win->dialog);
}
