#include <stdlib.h>

#include <string.h>

#include <glib.h>
#include <gmime/gmime.h>

#include <pan/base/article.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/group.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/message-identifier.h>
#include <pan/base/server.h>
#include <pan/base/serverlist.h>


static void
pan_init (void)
{
	char * base;
	char * download_dir;

        g_mime_init (0);

	base = getenv ("srcdir");
	base = g_strdup_printf ("%s/", base!=NULL ? base : getenv("PWD"));
	download_dir = g_strdup_printf ("%s/data/save/", base);
	base_prefs_init (download_dir, 1000, FALSE, FALSE);
	g_free (download_dir);
	g_free (base);
}


int
main (void)
{
	int test = 0;
	Server * s;
	Group * g;
	Group * groups[3];
	Article * a1;
	MessageIdentifier * mid;
	MessageSource * source;
	const char * cpch;
	const char * cpch_in;
	GPtrArray * tmp_array;

	g_thread_init (NULL);
	pan_init ();

	/* setup some scaffolding: news server */
	s = server_new ();
	server_set_name (s, "news.rebelbase.com");
	serverlist_add_server (s);

	/* set up some scaffolding: groups */
	groups[0] = group_new (s, "alt.fan.navel.lint");
	groups[1] = group_new (s, "rec.arts.twaddle");
	groups[2] = group_new (s, "sci.phrenology");
	server_init_groups (s, groups, G_N_ELEMENTS(groups), NULL, NULL);

	g = groups[2];
	group_ref_articles (g, NULL);
	a1 = article_new (g);
	a1->number = 999;
	article_set_subject (a1, "Aimlessly walking around the house with a mallet");
	article_set_message_id (a1, "<foobar.foo.bar>");
	article_set_xrefs (a1, "news.rebelbase.com alt.fan.navel.lint:1234 rec.arts.twaddle:5678 sci.phrenology:999");
	tmp_array = g_ptr_array_new ();
	g_ptr_array_add (tmp_array, a1);
	group_init_articles (g, tmp_array, NULL);
	g_ptr_array_free (tmp_array, TRUE);

	/**
	***  Test: message_identifier_new ()
	**/

	++test;
	mid = message_identifier_new (a1->message_id);
	if (mid == NULL) return test;
	if (pan_strcmp (a1->message_id, mid->message_id)) return test;
	if (a1->message_id == mid->message_id) return test;
	if (mid->readable_name != NULL) return test;
	if (mid->line_qty != 0ul) return test;
	if (mid->byte_qty != 0ul) return test;
	if (mid->message_sources == NULL) return test;
	if (mid->message_sources->len != 0u) return test;
	g_object_unref (G_OBJECT(mid));
	g_message ("PASS message_identifier_new (message_id_string)");

	/**
	***  Test: message_identifier_new_from_article ()
	**/

	++test;
	mid = message_identifier_new_from_article (a1);
	if (!message_identifier_is_valid(mid)) return test;
	if (pan_strcmp (a1->message_id, mid->message_id)) return test;
	if (a1->message_id == mid->message_id) return test;
	if (mid->readable_name == a1->subject) return test;
	if (pan_strcmp (mid->readable_name, a1->subject)) return test;
	if (mid->line_qty != 0ul) return test;
	if (mid->byte_qty != 0ul) return test;
	if (pan_strcmp (message_identifier_get_primary_group(mid), g->name)) return test;
	if (mid->message_sources == NULL) return test;
	if (mid->message_sources->len != 3u) return test;
	/* check the sources.  Note the primary always comes first ... */
	source = (MessageSource*) g_ptr_array_index (mid->message_sources, 0);
	if (source == NULL) return test;
	if (pan_strcmp (source->server_name, s->name)) return test;
	if (pan_strcmp (source->group_name, g->name)) return test;
	if (source->number != a1->number) return test;
	source = (MessageSource*) g_ptr_array_index (mid->message_sources, 1);
	if (source == NULL) return test;
	if (pan_strcmp (source->server_name, s->name)) return test;
	if (pan_strcmp (source->group_name, "alt.fan.navel.lint")) return test;
	if (source->number != 1234ul) return test;
	source = (MessageSource*) g_ptr_array_index (mid->message_sources, 2);
	if (source == NULL) return test;
	if (pan_strcmp (source->server_name, s->name)) return test;
	if (pan_strcmp (source->group_name, "rec.arts.twaddle")) return test;
	if (source->number != 5678ul) return test;
	g_message ("PASS message_identifier_new_from_article (a)");

	/**
	***  Test: message_identifier_get_source_for_server
	**/

	++test;
	source = message_identifier_get_source_for_server (mid, server_get_name (s));
	if (source == NULL) return test;
	if (pan_strcmp (source->server_name, s->name)) return test;
	if (pan_strcmp (source->group_name, g->name)) return test;
	if (source->number != a1->number) return test;
	g_message ("PASS message_identifier_get_source_for_server (s)");

	/**
	***  Test: message_identifier_get_primary_group
	**/

	++test;
	cpch = message_identifier_get_primary_group (mid);
	if (cpch == group_get_name(g)) return test;
	if (pan_strcmp (cpch, group_get_name(g))) return test;
	g_message ("PASS message_identifier_get_primary_group");

	/**
	***  Test: message_identifier_set_readable_name
	**/

	++test;
	cpch_in = "And you shall know us by the trail of the dead";
	message_identifier_set_readable_name (mid, cpch_in);
	cpch = message_identifier_get_readable_name (mid);
	if (cpch != mid->readable_name) return test;
	if (cpch == cpch_in) return test;
	if (pan_strcmp (cpch, cpch_in)) return test;
	g_message ("PASS message_identifier_{get,set}_readable_name");

	/**
	***  Test: message_identifiers_mark_read
	**/

	++test;
	if (article_is_read (a1)) return test;
	message_identifiers_mark_read ((const MessageIdentifier**)&mid, 1, TRUE, SERVER_GROUPS_ALL);
	if (!article_is_read (a1)) return test;
	message_identifiers_mark_read ((const MessageIdentifier**)&mid, 1, FALSE, SERVER_GROUPS_ALL);
	if (article_is_read (a1)) return test;
	g_message ("PASS message_identifiers_mark_read");
	g_object_unref (G_OBJECT(mid));

	/**
	***  Test: corrupt/duplicate xref entries
	**/

	++test;
	g_message ("there should be a warning here about message_identifier_get_source_from_group()==NULL");
	article_set_xrefs (a1, "news.rebelbase.com alt.fan.navel.lint:1234 rec.arts.twaddle:5678 sci.phrenology:999 rec.arts.twaddle:5678");
	mid = message_identifier_new_from_article (a1);
	if (mid == NULL) return test;
	if (mid->message_sources->len != 3u) return test;
	g_message ("PASS corrupt Xref header");
	g_object_unref (G_OBJECT(mid));

	/**
	***  Success
	**/

	g_message ("All message-identifier tests passed");
	return 0;
}
