/*
 * Copyright 2003-2006 Gentoo Foundation
 * Distributed under the terms of the GNU General Public License v2
 * $Header: /var/cvsroot/gentoo-projects/pax-utils/scanmach.c,v 1.88 2005/09/30 03:30:54 vapier Exp $
 *
 * Copyright 2003-2006 Ned Ludd        - <solar@gentoo.org>
 * Copyright 2004-2006 Mike Frysinger  - <vapier@gentoo.org>
 */

#include "paxinc.h"

static const char *rcsid = "$Id: scanmach.c,v 1.88 2005/09/30 03:30:54 vapier Exp $";
#define argv0 "scanmach"

#define IS_MODIFIER(c) (c == '%' || c == '#' || c == '!')


/* prototypes */
static void scanmacho_file(const char *filename);
static void scanmacho_dir(const char *path);
static void scanmacho_ldpath();
static void scanmacho_envpath();
static void usage(int status);
static void parseargs(int argc, char *argv[]);
static char *xstrdup(const char *s);
static void *xmalloc(size_t size);
static void xstrcat(char **dst, const char *src, size_t *curr_len);
static inline void xchrcat(char **dst, const char append, size_t *curr_len);

/* variables to control behavior */
static char *ldpaths[256];
static char scan_ldpath = 0;
static char scan_envpath = 0;
static char scan_symlink = 1;
static char dir_recurse = 0;
static char dir_crossmount = 1;
static char show_pax = 0;
static char show_phdr = 0;
static char show_textrel = 0;
static char show_rpath = 0;
static char show_needed = 0;
static char show_interp = 0;
static char show_bind = 0;
static char show_soname = 0;
static char show_textrels = 0;
static char show_banner = 1;
static char be_quiet = 0;
static char be_verbose = 0;
static char be_wewy_wewy_quiet = 0;
static char *find_sym = NULL, *versioned_symname = NULL;
static char *find_lib = NULL;
static char *out_format = NULL;
static char *search_path = NULL;
static char gmatch = 0;


/* scan an macho file and show all the fun stuff */
#define prints(str) write(fileno(stdout), str, strlen(str))
static void scanmacho_file(const char *filename)
{
	unsigned long i;
	char found_pax, found_phdr, found_relro, found_load, found_textrel, 
	     found_rpath, found_needed, found_interp, found_bind, found_soname, 
	     found_sym, found_lib, found_file, found_textrels;
	machoobj *macho;
	struct stat st;
	static char *out_buffer = NULL;
	static size_t out_len;

	/* make sure 'filename' exists */
	if (lstat(filename, &st) == -1) {
		if (be_verbose > 2) printf("%s: does not exist\n", filename);
		return;
	}
	/* always handle regular files and handle symlinked files if no -y */
	if (S_ISLNK(st.st_mode)) {
		if (!scan_symlink) return;
		stat(filename, &st);
	}
	if (!S_ISREG(st.st_mode)) {
		if (be_verbose > 2) printf("%s: skipping non-file\n", filename);
		return;
	}

	found_pax = found_phdr = found_relro = found_load = found_textrel = \
	found_rpath = found_needed = found_interp = found_bind = found_soname = \
	found_sym = found_lib = found_file = found_textrels = 0;

	/* verify this is real MACH-O */
	if ((macho = readmacho(filename)) == NULL) {
		if (be_verbose > 2) printf("%s: not a MACH-O\n", filename);
		return;
	}

	if (be_verbose > 1)
		printf("%s: scanning file\n", filename);
	else if (be_verbose)
		printf("%s: scanning file\n", filename);

	/* init output buffer */
	if (!out_buffer) {
		out_len = sizeof(char) * 80;
		out_buffer = (char*)xmalloc(out_len);
	}
	*out_buffer = '\0';

	/* show the header */
	if (!be_quiet && show_banner) {
		for (i = 0; out_format[i]; ++i) {
			if (!IS_MODIFIER(out_format[i])) continue;

			switch (out_format[++i]) {
			case '%': break;
			case '#': break;
			case '!': break;
			case 'F':
			case 'p':
			case 'f': prints("FILE "); found_file = 1; break;
			case 'o': prints(" TYPE   "); break;
			case 'x': prints(" PAX   "); break;
			case 'e': prints("STK/REL/PTL "); break;
			case 't': prints("TEXTREL "); break;
			case 'r': prints("RPATH "); break;
			case 'n': prints("NEEDED "); break;
			case 'i': prints("INTERP "); break;
			case 'b': prints("BIND "); break;
			case 'S': prints("SONAME "); break;
			case 's': prints("SYM "); break;
			case 'N': prints("LIB "); break;
			case 'T': prints("TEXTRELS "); break;
			default: warnf("'%c' has no title ?", out_format[i]);
			}
		}
		if (!found_file) prints("FILE ");
		prints("\n");
		found_file = 0;
		show_banner = 0;
	}

	/* dump all the good stuff */
	for (i = 0; out_format[i]; ++i) {
		const char *out;
		const char *tmp;

		/* make sure we trim leading spaces in quiet mode */
		if (be_quiet && *out_buffer == ' ' && !out_buffer[1])
			*out_buffer = '\0';

		if (!IS_MODIFIER(out_format[i])) {
			xchrcat(&out_buffer, out_format[i], &out_len);
			continue;
		}

		out = NULL;
		be_wewy_wewy_quiet = (out_format[i] == '#');
		switch (out_format[++i]) {
		case '%':
		case '#':
			xchrcat(&out_buffer, out_format[i], &out_len); break;
		case '!': break;
		case 'F':
			found_file = 1;
			if (be_wewy_wewy_quiet) break;
			xstrcat(&out_buffer, filename, &out_len);
			break;
		case 'p':
			found_file = 1;
			if (be_wewy_wewy_quiet) break;
			tmp = filename;
			if (search_path) {
				ssize_t len_search = strlen(search_path);
				ssize_t len_file = strlen(filename);
				if (!strncmp(filename, search_path, len_search) && \
				    len_file > len_search)
					tmp += len_search;
				if (*tmp == '/' && search_path[len_search-1] == '/') tmp++;
			}
			xstrcat(&out_buffer, tmp, &out_len);
			break;
		case 'f':
			found_file = 1;
			if (be_wewy_wewy_quiet) break;
			tmp = strrchr(filename, '/');
			tmp = (tmp == NULL ? filename : tmp+1);
			xstrcat(&out_buffer, tmp, &out_len);
			break;
#if 0
		case 'o': out = get_machoetype(macho); break;
		case 'x': out = scanmacho_file_pax(macho, &found_pax); break;
		case 'e': out = scanmacho_file_phdr(macho, &found_phdr, &found_relro, &found_load); break;
		case 't': out = scanmacho_file_textrel(macho, &found_textrel); break;
		case 'T': out = scanmacho_file_textrels(macho, &found_textrels, &found_textrel); break;
		case 'r': scanmacho_file_rpath(macho, &found_rpath, &out_buffer, &out_len); break;
		case 'n':
		case 'N': out = scanmacho_file_needed_lib(macho, &found_needed, &found_lib, (out_format[i]=='N'), &out_buffer, &out_len); break;
		case 'i': out = scanmacho_file_interp(macho, &found_interp); break;
		case 'b': out = scanmacho_file_bind(macho, &found_bind); break;
		case 'S': out = scanmacho_file_soname(macho, &found_soname); break;
		case 's': out = scanmacho_file_sym(macho, &found_sym); break;
#endif
		default: warnf("'%c' has no scan code?", out_format[i]);
		}
		if (out) xstrcat(&out_buffer, out, &out_len);
	}

#define FOUND_SOMETHING() \
	(found_pax || found_phdr || found_relro || found_load || found_textrel || \
	 found_rpath || found_needed || found_interp || found_bind || \
	 found_soname || found_sym || found_lib || found_textrels)

	if (!found_file && (!be_quiet || (be_quiet && FOUND_SOMETHING()))) {
		xchrcat(&out_buffer, ' ', &out_len);
		xstrcat(&out_buffer, filename, &out_len);
	}
	if (!be_quiet || (be_quiet && FOUND_SOMETHING())) {
		puts(out_buffer);
		fflush(stdout);
	}

	unreadmacho(macho);
}

/* scan a directory for ET_EXEC files and print when we find one */
static void scanmacho_dir(const char *path)
{
	register DIR *dir;
	register struct dirent *dentry;
	struct stat st_top, st;
	char buf[__PAX_UTILS_PATH_MAX];
	size_t pathlen = 0, len = 0;

	/* make sure path exists */
	if (lstat(path, &st_top) == -1) {
		if (be_verbose > 2) printf("%s: does not exist\n", path);
		return;
	}

	/* ok, if it isn't a directory, assume we can open it */
	if (!S_ISDIR(st_top.st_mode)) {
		scanmacho_file(path);
		return;
	}

	/* now scan the dir looking for fun stuff */
	if ((dir = opendir(path)) == NULL) {
		warnf("could not opendir %s: %s", path, strerror(errno));
		return;
	}
	if (be_verbose) printf("%s: scanning dir\n", path);

	pathlen = strlen(path);
	while ((dentry = readdir(dir))) {
		if (!strcmp(dentry->d_name, ".") || !strcmp(dentry->d_name, ".."))
			continue;
		len = (pathlen + 1 + strlen(dentry->d_name) + 1);
		if (len >= sizeof(buf)) {
			warnf("Skipping '%s': len > sizeof(buf); %lu > %lu\n", path,
			      (unsigned long)len, (unsigned long)sizeof(buf));
			continue;
		}
		sprintf(buf, "%s/%s", path, dentry->d_name);
		if (lstat(buf, &st) != -1) {
			if (S_ISREG(st.st_mode))
				scanmacho_file(buf);
			else if (dir_recurse && S_ISDIR(st.st_mode)) {
				if (dir_crossmount || (st_top.st_dev == st.st_dev))
					scanmacho_dir(buf);
			}
		}
	}
	closedir(dir);
}

static int scanmacho_from_file(char *filename)
{
	FILE *fp = NULL;
	char *p;
	char path[__PAX_UTILS_PATH_MAX];

	if (((strcmp(filename, "-")) == 0) && (ttyname(0) == NULL))
		fp = stdin;
	else if ((fp = fopen(filename, "r")) == NULL)
		return 1;

	while ((fgets(path, __PAX_UTILS_PATH_MAX, fp)) != NULL) {
		if ((p = strchr(path, '\n')) != NULL)
			*p = 0;
		search_path = path;
		scanmacho_dir(path);
	}
	if (fp != stdin)
		fclose(fp);
	return 0;
}

static void load_ld_so_conf()
{
	FILE *fp = NULL;
	char *p;
	char path[__PAX_UTILS_PATH_MAX];
	int i = 0;

	if ((fp = fopen("/etc/ld.so.conf", "r")) == NULL)
		return;

	while ((fgets(path, __PAX_UTILS_PATH_MAX, fp)) != NULL) {
		if (*path != '/')
			continue;

		if ((p = strrchr(path, '\r')) != NULL)
			*p = 0;
		if ((p = strchr(path, '\n')) != NULL)
			*p = 0;

		ldpaths[i++] = xstrdup(path);

		if (i + 1 == sizeof(ldpaths) / sizeof(*ldpaths))
			break;
	}
	ldpaths[i] = NULL;

	fclose(fp);
}

/* scan /etc/ld.so.conf for paths */
static void scanmacho_ldpath()
{
	char scan_l, scan_ul, scan_ull;
	int i = 0;

	if (!ldpaths[0])
		err("Unable to load any paths from ld.so.conf");

	scan_l = scan_ul = scan_ull = 0;

	while (ldpaths[i]) {
		if (!scan_l   && !strcmp(ldpaths[i], "/lib")) scan_l = 1;
		if (!scan_ul  && !strcmp(ldpaths[i], "/usr/lib")) scan_ul = 1;
		if (!scan_ull && !strcmp(ldpaths[i], "/usr/local/lib")) scan_ull = 1;
		scanmacho_dir(ldpaths[i]);
		++i;
	}

	if (!scan_l)   scanmacho_dir("/lib");
	if (!scan_ul)  scanmacho_dir("/usr/lib");
	if (!scan_ull) scanmacho_dir("/usr/local/lib");
}

/* scan env PATH for paths */
static void scanmacho_envpath()
{
	char *path, *p;

	path = getenv("PATH");
	if (!path)
		err("PATH is not set in your env !");
	path = xstrdup(path);

	while ((p = strrchr(path, ':')) != NULL) {
		scanmacho_dir(p + 1);
		*p = 0;
	}

	free(path);
}



/* usage / invocation handling functions */
#define PARSE_FLAGS "plRmyxetrnibSs:gN:TaqvF:f:o:BhV"
#define a_argument required_argument
static struct option const long_opts[] = {
	{"path",      no_argument, NULL, 'p'},
	{"ldpath",    no_argument, NULL, 'l'},
	{"recursive", no_argument, NULL, 'R'},
	{"mount",     no_argument, NULL, 'm'},
	{"symlink",   no_argument, NULL, 'y'},
	{"pax",       no_argument, NULL, 'x'},
	{"header",    no_argument, NULL, 'e'},
	{"textrel",   no_argument, NULL, 't'},
	{"rpath",     no_argument, NULL, 'r'},
	{"needed",    no_argument, NULL, 'n'},
	{"interp",    no_argument, NULL, 'i'},
	{"bind",      no_argument, NULL, 'b'},
	{"soname",    no_argument, NULL, 'S'},
	{"symbol",     a_argument, NULL, 's'},
	{"lib",        a_argument, NULL, 'N'},
	{"gmatch",    no_argument, NULL, 'g'},
	{"textrels",  no_argument, NULL, 'T'},
	{"all",       no_argument, NULL, 'a'},
	{"quiet",     no_argument, NULL, 'q'},
	{"verbose",   no_argument, NULL, 'v'},
	{"format",     a_argument, NULL, 'F'},
	{"from",       a_argument, NULL, 'f'},
	{"file",       a_argument, NULL, 'o'},
	{"nobanner",  no_argument, NULL, 'B'},
	{"help",      no_argument, NULL, 'h'},
	{"version",   no_argument, NULL, 'V'},
	{NULL,        no_argument, NULL, 0x0}
};

static const char *opts_help[] = {
	"Scan all directories in PATH environment",
	"Scan all directories in /etc/ld.so.conf",
	"Scan directories recursively",
	"Don't recursively cross mount points",
	"Don't scan symlinks\n",
	"Print PaX markings",
	"Print GNU_STACK/PT_LOAD markings",
	"Print TEXTREL information",
	"Print RPATH information",
	"Print NEEDED information",
	"Print INTERP information",
	"Print BIND information",
	"Print SONAME information",
	"Find a specified symbol",
	"Find a specified library",
	"Use strncmp to match libraries. (use with -N)",
	"Locate cause of TEXTREL",
	"Print all scanned info (-x -e -t -r -b)\n",
	"Only output 'bad' things",
	"Be verbose (can be specified more than once)",
	"Use specified format for output",
	"Read input stream from a filename",
	"Write output stream to a filename",
	"Don't display the header",
	"Print this help and exit",
	"Print version and exit",
	NULL
};

/* display usage and exit */
static void usage(int status)
{
	unsigned long i;
	printf("* Scan MACHO binaries for stuff\n\n"
	       "Usage: %s [options] <dir1/file1> [dir2 dirN fileN ...]\n\n", argv0);
	printf("Options: -[%s]\n", PARSE_FLAGS);
	for (i = 0; long_opts[i].name; ++i)
		if (long_opts[i].has_arg == no_argument)
			printf("  -%c, --%-13s* %s\n", long_opts[i].val, 
			       long_opts[i].name, opts_help[i]);
		else
			printf("  -%c, --%-6s <arg> * %s\n", long_opts[i].val,
			       long_opts[i].name, opts_help[i]);

	if (status != EXIT_SUCCESS)
		exit(status);

	puts("\nThe format modifiers for the -F option are:");
	puts(" F Filename \tx PaX Flags \te STACK/RELRO");
	puts(" t TEXTREL  \tr RPATH     \tn NEEDED");
	puts(" i INTERP   \tb BIND      \ts symbol");
	puts(" N library  \to Type      \tT TEXTRELs");
	puts(" S SONAME");
	puts(" p filename (with search path removed)");
	puts(" f filename (short name/basename)");
	puts("Prefix each modifier with '%' (verbose) or '#' (silent)");
	puts("Insert '!' between %/# and modifier to negate the check");

	exit(status);
}

/* parse command line arguments and preform needed actions */
static void parseargs(int argc, char *argv[])
{
	int i;
	char *from_file = NULL;

	opterr = 0;
	while ((i=getopt_long(argc, argv, PARSE_FLAGS, long_opts, NULL)) != -1) {
		switch (i) {

		case 'V':
			printf("pax-utils-%s: %s compiled %s\n%s\n"
			       "%s written for Gentoo by <solar and vapier @ gentoo.org>\n",
			       VERSION, __FILE__, __DATE__, rcsid, argv0);
			exit(EXIT_SUCCESS);
			break;
		case 'h': usage(EXIT_SUCCESS); break;
		case 'f':
			if (from_file) err("Don't specify -f twice");
			from_file = xstrdup(optarg);
			break;
		case 'o': {
			if (freopen(optarg, "w", stdout) == NULL)
				err("Could not open output stream '%s': %s", optarg, strerror(errno));
			break;
		}

		case 's': {
			size_t len;
			if (find_sym) err("Don't specify -s twice");
			find_sym = xstrdup(optarg);
			len = strlen(find_sym) + 1;
			versioned_symname = (char*)xmalloc(sizeof(char) * (len+1));
			sprintf(versioned_symname, "%s@", find_sym);
			break;
		}
		case 'N': {
			if (find_lib) err("Don't specify -N twice");
			find_lib = xstrdup(optarg);
			break;
		}

		case 'F': {
			if (out_format) err("Don't specify -F twice");
			out_format = xstrdup(optarg);
			break;
		}

		case 'g': gmatch = 1;
		case 'y': scan_symlink = 0; break;
		case 'B': show_banner = 0; break;
		case 'l': scan_ldpath = 1; break;
		case 'p': scan_envpath = 1; break;
		case 'R': dir_recurse = 1; break;
		case 'm': dir_crossmount = 0; break;
		case 'x': show_pax = 1; break;
		case 'e': show_phdr = 1; break;
		case 't': show_textrel = 1; break;
		case 'r': show_rpath = 1; break;
		case 'n': show_needed = 1; break;
		case 'i': show_interp = 1; break;
		case 'b': show_bind = 1; break;
		case 'S': show_soname = 1; break;
		case 'T': show_textrels = 1; break;
		case 'q': be_quiet = 1; break;
		case 'v': be_verbose = (be_verbose % 20) + 1; break;
		case 'a': show_pax = show_phdr = show_textrel = show_rpath = show_bind = 1; break;

		case ':':
			err("Option missing parameter\n");
		case '?':
			err("Unknown option\n");
		default:
			err("Unhandled option '%c'", i);
		}
	}

	/* let the format option override all other options */
	if (out_format) {
		show_pax = show_phdr = show_textrel = show_rpath = \
		show_needed = show_interp = show_bind = show_soname = \
		show_textrels = 0;
		for (i = 0; out_format[i]; ++i) {
			if (!IS_MODIFIER(out_format[i])) continue;

			switch (out_format[++i]) {
			case '%': break;
			case '#': break;
			case 'F': break;
			case 'p': break;
			case 'f': break;
			case 's': break;
			case 'N': break;
			case 'o': break;
			case 'x': show_pax = 1; break;
			case 'e': show_phdr = 1; break;
			case 't': show_textrel = 1; break;
			case 'r': show_rpath = 1; break;
			case 'n': show_needed = 1; break;
			case 'i': show_interp = 1; break;
			case 'b': show_bind = 1; break;
			case 'S': show_soname = 1; break;
			case 'T': show_textrels = 1; break;
			default:
				err("Invalid format specifier '%c' (byte %i)", 
				    out_format[i], i+1);
			}
		}

	/* construct our default format */
	} else {
		size_t fmt_len = 30;
		out_format = (char*)xmalloc(sizeof(char) * fmt_len);
		if (!be_quiet)     xstrcat(&out_format, "%o ", &fmt_len);
		if (show_pax)      xstrcat(&out_format, "%x ", &fmt_len);
		if (show_phdr)     xstrcat(&out_format, "%e ", &fmt_len);
		if (show_textrel)  xstrcat(&out_format, "%t ", &fmt_len);
		if (show_rpath)    xstrcat(&out_format, "%r ", &fmt_len);
		if (show_needed)   xstrcat(&out_format, "%n ", &fmt_len);
		if (show_interp)   xstrcat(&out_format, "%i ", &fmt_len);
		if (show_bind)     xstrcat(&out_format, "%b ", &fmt_len);
		if (show_soname)   xstrcat(&out_format, "%S ", &fmt_len);
		if (show_textrels) xstrcat(&out_format, "%T ", &fmt_len);
		if (find_sym)      xstrcat(&out_format, "%s ", &fmt_len);
		if (find_lib)      xstrcat(&out_format, "%N ", &fmt_len);
		if (!be_quiet)     xstrcat(&out_format, "%F ", &fmt_len);
	}
	if (be_verbose > 2) printf("Format: %s\n", out_format);

	/* now lets actually do the scanning */
	if (scan_ldpath || (show_rpath && be_quiet))
		load_ld_so_conf();
	if (scan_ldpath) scanmacho_ldpath();
	if (scan_envpath) scanmacho_envpath();
	if (from_file) {
		scanmacho_from_file(from_file);
		free(from_file);
		from_file = *argv;
	}
	if (optind == argc && !scan_ldpath && !scan_envpath && !from_file)
		err("Nothing to scan !?");
	while (optind < argc) {
		search_path = argv[optind++];
		scanmacho_dir(search_path);
	}

	/* clean up */
	if (find_sym) {
		free(find_sym);
		free(versioned_symname);
	}
	if (find_lib) free(find_lib);
	if (out_format) free(out_format);
	for (i = 0; ldpaths[i]; ++i)
		free(ldpaths[i]);
}



/* utility funcs */
static char *xstrdup(const char *s)
{
	char *ret = strdup(s);
	if (!ret) err("Could not strdup(): %s", strerror(errno));
	return ret;
}

static void *xmalloc(size_t size)
{
	void *ret = malloc(size);
	if (!ret) err("Could not malloc() %li bytes", (unsigned long)size);
	return ret;
}

static void xstrcat(char **dst, const char *src, size_t *curr_len)
{
	size_t new_len;

	new_len = strlen(*dst) + strlen(src);
	if (*curr_len <= new_len) {
		*curr_len = new_len + (*curr_len / 2);
		*dst = realloc(*dst, *curr_len);
		if (!*dst)
			err("could not realloc %li bytes", (unsigned long)*curr_len);
	}

	strcat(*dst, src);
}

static inline void xchrcat(char **dst, const char append, size_t *curr_len)
{
	static char my_app[2];
	my_app[0] = append;
	my_app[1] = '\0';
	xstrcat(dst, my_app, curr_len);
}


int main(int argc, char *argv[])
{
	if (argc < 2)
		usage(EXIT_FAILURE);
	parseargs(argc, argv);
	fclose(stdout);
#ifdef __BOUNDS_CHECKING_ON
	warn("The calls to add/delete heap should be off by 1 due to the out_buffer not being freed in scanmacho_file()");
#endif
	return EXIT_SUCCESS;
}
