/***************************************************************************
 *   Copyright (C) 2005-2006 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "about.h"

#if defined(Q_WS_WIN)
#  define SVN_REVISION "windows"
#else
#  include "svn_revision.h"
#endif

#include "process.h"

//---------------------------------------------------------------------------
const Piklab::UrlType::Data Piklab::UrlType::DATA[Nb_Types] = {
  { "http://piklab.sourceforge.net"                                        },
  { "http://piklab.sourceforge.net/wiki/index.php/FAQ"                     },
  { "http://sourceforge.net/tracker/?func=add&group_id=138852&atid=743140" }
};

//-----------------------------------------------------------------------------
Piklab::OptionList::OptionList(const KCmdLineOptions *options)
: _options(NULL)
{
  for (uint i=0; options[i].name; i++) append(options[i]);
}

const KCmdLineOptions *Piklab::OptionList::ptr() const
{
  delete[] _options;
  _options = new KCmdLineOptions[count()+1];
  for (uint i=0; i<uint(count()); i++) {
    _options[i] = *at(i);
    Q_ASSERT(_options[i].name != NULL);
  }
  _options[count()].name = NULL;
  return _options;
}

//-----------------------------------------------------------------------------
void Piklab::initArgs(const KAboutData& about, int argc, char **argv, const KCmdLineOptions& options)
{
  KCmdLineArgs::init(argc, argv, &about);
  KCmdLineArgs::addCmdLineOptions(&options);
}

void Piklab::createApplication(const char* executable, int argc, char **argv,
                               bool gui, Log::Generic* log)
{
#if defined(NO_KDE)
#  if QT_VERSION<0x040000
  (void)new QApplication(argc, argv, QApplication::Tty);
#  else
  (void)new QCoreApplication(argc, argv);
#  endif
  Q_UNUSED(gui);
#else
  if (!gui) KApplication::disableAutoDcopRegistration();
  (void)new KApplication(gui, gui);
  Q_UNUSED(argc);
  Q_UNUSED(argv);
#endif

  QString version = executable;
#if defined(NO_KDE)
#  if defined(Q_OS_WIN)
  version += " \"win32\"";
#  else
  version += " \"qt-only\"";
#  endif
#endif
  version += QString(": version ") + VERSION + " (rev. " + SVN_REVISION + ")";
  if (log != NULL) {
    log->log(Log::LineType::Comment, version);
    Process::LineOutput process;
    QStringList args;
    args += "-srmpo";
    process.setup("uname", args, false);
    Process::State state = Process::runSynchronously(process, Process::Start, 100);
    if (state == Process::Exited) {
      QStringList lines = process.lines(Process::OutputType::Stdout);
      if (!lines.isEmpty() && !lines[0].isEmpty())
        log->log(Log::LineType::Comment, lines[0], Log::LogFileOnly);
    }
    process.setup("date", QStringList(), false);
    state = Process::runSynchronously(process, Process::Start, 100);
    if (state == Process::Exited) {
      QStringList lines = process.lines(Process::OutputType::Stdout);
      if (!lines.isEmpty() && !lines[0].isEmpty())
        log->log(Log::LineType::Comment, lines[0], Log::LogFileOnly);
    }
  } else {
    version += "\n";
    fprintf(stdout, "%s", stringToChar(version));
  }
}

//---------------------------------------------------------------------------
Piklab::AboutData::AboutData(const char *executable, const char *name,
                             const char *description)
: KAboutData(executable, name, VERSION, description, KAboutData::License_GPL,
             "(c) 2005-2009 Nicolas Hadacek\n(c) 2002-2005 Alain Gibaud\n(c) 2003-2004 Stephen Landamore\n(c) 2005 Lorenz Möenlechner and Matthias Kranz\n(c) 2001-2005 Craig Franklin",
             0, UrlType(UrlType::Homepage).data().url, UrlType(UrlType::BugReport).data().url)
{
  addAuthor("Nicolas Hadacek", I18N_NOOP("Author and maintainer."), "hadacek@kde.org");
  addAuthor("Alain Gibaud", I18N_NOOP("Author of PiKdev."), "alain.gibaud@free.fr");
  addAuthor("Stephen Landamore", I18N_NOOP("Author of LPLAB (original Microchip programmer support)."), "webmaster@landamore.com");
  addAuthor("Craig Franklin", I18N_NOOP("Author of gputils"), "craigfranklin@users.sourceforge.net");
  addAuthor("Sébastien Laoût", I18N_NOOP("Author of likeback"), "slaout@linux62.org");

  addCredit("Brian C. Lane", I18N_NOOP("Original code for direct programming."), 0);
  addCredit("Manwlis \"Manos\" Giannos", I18N_NOOP("Direct programming for PIC18F devices."), "mgiannos2000@yahoo.gr");
  addCredit("Sean A. Walberg", I18N_NOOP("Direct programming for 16F676/630."), "sean@ertw.com");
  addCredit("Mirko Panciri", I18N_NOOP("Support for direct programmers with bidirectionnal buffers."), "mirko.panciri@adept.it");
  addCredit("Keith Baker", I18N_NOOP("Direct programming for 16F73/74/76/77."), "susyzygy@pubnix.org" );
  addCredit("Lorenz Möenlechner and Matthias Kranz", I18N_NOOP("USB support for ICD2 programmer."), "icd2linux@hcilab.org");
  addCredit("Xiaofan Chen", I18N_NOOP("Test of PICkit2 and ICD2 programmer."), "xiaofanc@gmail.com");
  addCredit("Homer Reid", I18N_NOOP("Direct programming for dsPICs is inspired from his program \"dspicprg\"."), "homer@homerreid.com");
  addCredit("Frank Damgaard", I18N_NOOP("Direct programming for 24C EEPROM is inspired from his program \"prog84\"."), "frda@post3.tele.dk");
}
