/***************************************************************************
 *   Copyright (C) 2008 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "process_kde.h"
#include "process_kde.moc"

#include <kprocess.h>

//----------------------------------------------------------------------------
Process::ProcessIO::ProcessIO(QObject *parent)
  : QObject(parent)
{
  _kproc.reset(new KProcess);
  connect(_kproc.get(), SIGNAL(receivedStdout(KProcess*, char*, int)), SLOT(receivedStdout(KProcess*, char*, int)));
  connect(_kproc.get(), SIGNAL(receivedStderr(KProcess*, char*, int)), SLOT(receivedStderr(KProcess*, char*, int)));
  connect(_kproc.get(), SIGNAL(processExited(KProcess*)), SIGNAL(processExited()));
}

Process::ProcessIO::~ProcessIO()
{
}

void Process::ProcessIO::setArguments(const QStringList &args)
{
  _kproc->clearArguments();
  for (QStringList::const_iterator it = args.begin();
       it!=args.end();
       ++it) {
      *_kproc << *it;
  }
}

void Process::ProcessIO::setWorkingDirectory(const QDir &dir)
{
  _kproc->setWorkingDirectory(dir.path());
}

bool Process::ProcessIO::start(QStringList *env)
{
  if (env) {
    for (QStringList::const_iterator it = env->begin();
         it!=env->end();
         ++it) {
      QStringList list = QStringList::split('=', *it);
      if ( list.count()==2 ) _kproc->setEnvironment(list[0], list[1]);
    }
  }
  return _kproc->start(KProcess::NotifyOnExit, KProcess::All);
}

bool Process::ProcessIO::isRunning() const
{
  return _kproc->isRunning();
}

bool Process::ProcessIO::normalExit() const
{
  return _kproc->normalExit();
}

int Process::ProcessIO::exitStatus() const
{
  return _kproc->exitStatus();
}

QByteArray Process::ProcessIO::readStdout()
{
  QByteArray tmp = _stdout.copy();
  _stdout.resize(0);
  return tmp;
}

QByteArray Process::ProcessIO::readStderr()
{
  QByteArray tmp = _stderr.copy();
  _stderr.resize(0);
  return tmp;
}

Q_LONG Process::ProcessIO::processIdentifier() const
{
  return _kproc->pid();
}

void Process::ProcessIO::kill() const
{
  _kproc->kill();
}

void Process::ProcessIO::writeToStdin(const QByteArray &buf)
{
  _kproc->writeStdin(buf.data(), buf.size());
}

void Process::ProcessIO::receivedStdout(KProcess*, char *buf, int len)
{
  uint n = _stdout.size();
  _stdout.resize(n + len);
  for (int i=0; i<len; i++) _stdout[n+i] = buf[i] != 0 ? buf[i] : ' ';
  emit readyReadStdout();
}

void Process::ProcessIO::receivedStderr(KProcess*, char *buf, int len)
{
  uint n = _stderr.size();
  _stderr.resize(n + len);
  for (int i=0; i<len; i++) _stderr[n+i] = buf[i] != 0 ? buf[i] : ' ';
  emit readyReadStderr();
}
