/***************************************************************************
 *   Copyright (C) 2006-2010 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "dialog.h"
#include "dialog.moc"

#include <qtimer.h>
#include <qlabel.h>

#include "misc_gui.h"
#include "common/common/synchronous.h"

//-----------------------------------------------------------------------------
int Dialog::spacingHint()
{
#ifdef NO_KDE
  return 10;
#else
  return KDialogBase::spacingHint();
#endif
}

Dialog::Dialog(QWidget *parent, const char *name, bool modal,
               const QString &caption, int buttonMask, ButtonCode defaultButton,
               bool separator, const QSize &defaultSize)
  : KDialogBase(parent, name, modal, caption, buttonMask, defaultButton, separator),
    _defaultSize(defaultSize)
{
  _busyCursor.reset(new BusyCursorStarter);
  Q_ASSERT(name);
  QWidget *main = new QWidget(this);
  setMainWidget(main);

  QTimer::singleShot(0, this, SLOT(updateSize()));
}

Dialog::Dialog(DialogType type, const QString &caption, int buttonMask, ButtonCode defaultButton,
               QWidget *parent, const char *name, bool modal, bool separator, const QSize &defaultSize)
  : KDialogBase(type, caption, buttonMask, defaultButton, parent, name, modal, separator),
    _defaultSize(defaultSize)
{
  _busyCursor.reset(new BusyCursorStarter);
  Q_ASSERT(name);
  QTimer::singleShot(0, this, SLOT(updateSize()));
}

Dialog::~Dialog()
{
  GuiConfig gc;
  gc.writeEntry(QString(name()) + "_size", size());
}

void Dialog::updateSize()
{
  GuiConfig gc;
  resize(gc.readSizeEntry(QString(name()) + "_size", &_defaultSize));
  _busyCursor.reset(NULL);
}

//-----------------------------------------------------------------------------
TreeListDialog::Item::Item(const QString &label, QWidget *page, const QString &title, Q3ListView *listview)
  : KListViewItem(listview, label), _page(page), _title(title)
{}

TreeListDialog::Item::Item(const QString &label, QWidget *page, const QString &title, Q3ListViewItem *item)
  : KListViewItem(item, label), _page(page), _title(title)
{}

TreeListDialog::TreeListDialog(QWidget *parent, const char *name, bool modal,
                               const QString &caption, int buttonMask, ButtonCode defaultButton,
                               bool separator)
  : Dialog(parent, name, modal, caption, buttonMask, defaultButton, separator)
{
  QVBoxLayout *top = new QVBoxLayout(mainWidget(), 0, 10);

  // list view
  QValueList<int> widths;
  widths += 80;
  widths += 500;
  Splitter *splitter = new Splitter(widths, Qt::Horizontal, mainWidget(), name);
  top->addWidget(splitter);
  _listView = new ListView(splitter);
  connect(_listView, SIGNAL(currentChanged(Q3ListViewItem *)), SLOT(currentChanged(Q3ListViewItem *)));
  _listView->setAllColumnsShowFocus(true);
  _listView->setRootIsDecorated(true);
  _listView->setSorting(0);
  _listView->addColumn(QString::null);
  _listView->header()->hide();
  _listView->setResizeMode(Q3ListView::LastColumn);

  // pages
  _frame = new QFrame(splitter);
  QVBoxLayout *vbox = new QVBoxLayout(_frame, 10, 10);
  _titleBox = new QHBoxLayout(vbox);
  _label = new QLabel(_frame);
  _titleBox->addWidget(_label);
  _stack = new Q3WidgetStack(_frame);
  connect(_stack, SIGNAL(aboutToShow(QWidget *)), SIGNAL(aboutToShowPage(QWidget *)));
  vbox->addWidget(_stack);
  vbox->addStretch(1);
}

QWidget *TreeListDialog::addPage(const QStringList &labels)
{
  Q_ASSERT( !labels.isEmpty() );

  QWidget *page = NULL;
  Q3ListViewItem *item = NULL;
  Q3ListViewItemIterator it(_listView);
  for (; it.current(); ++it) {
    if ( it.current()->text(0)==labels[0] ) {
      item = it.current();
      break;
    }
  }
  if ( item==0 ) {
    page = new QWidget(_stack);
    connect(page, SIGNAL(destroyed(QObject *)), SLOT(pageDestroyed(QObject *)));
    _stack->addWidget(page);
    item = new Item(labels[0], page, labels[0], _listView);
    item->setOpen(true);
    bool last = ( labels.count()==1 );
    item->setSelectable(last);
  }
  for (uint i=1; i<labels.count(); i++) {
    Q3ListViewItem *parent = item;
    item = NULL;
    Q3ListViewItemIterator iti(parent);
    for (; it.current(); ++it) {
      if ( it.current()->text(0)==labels[i] ) {
        item = it.current();
        break;
      }
    }
    if ( item==NULL ) {
      page = new QWidget(_stack);
      connect(page, SIGNAL(destroyed(QObject *)), SLOT(pageDestroyed(QObject *)));
      _stack->addWidget(page);
      item = new Item(labels[i], page, labels[i], parent);
      item->setOpen(true);
      bool last = ( labels.count()==i+1 );
      item->setSelectable(last);
    }
  }

  return page;
}

void TreeListDialog::currentChanged(Q3ListViewItem *lvitem)
{
  if ( lvitem==0 ) return;
  Item *item = static_cast<Item *>(lvitem);
  _listView->ensureItemVisible(item);
  _label->setText(item->_title);
  _stack->raiseWidget(item->_page);
}

void TreeListDialog::showPage(QWidget *page)
{
  Q3ListViewItemIterator it(_listView);
  for (; it.current(); ++it) {
    Item *item = static_cast<Item *>(it.current());
    if ( item->_page==page ) {
      _listView->setCurrentItem(item);
      currentChanged(item);
      break;
    }
  }
}

int TreeListDialog::pageIndex(QWidget *page) const
{
  return _stack->id(page);
}

int TreeListDialog::activePageIndex() const
{
  const Item *item = static_cast<const Item *>(_listView->currentItem());
  if ( item==0 ) return -1;
  return pageIndex(item->_page);
}

void TreeListDialog::pageDestroyed(QObject *object)
{
  Q3ListViewItemIterator it(_listView);
  for (; it.current(); ++it) {
    Item *item = static_cast<Item *>(it.current());
    if ( item->_page!=object ) continue;
    delete item;
    break;
  }
}

//-----------------------------------------------------------------------------
TextEditorDialog::TextEditorDialog(const QString &text, const QString &caption,
                                   bool wrapAtWidgetWidth, QWidget *parent)
  : Dialog(parent, "text_editor_dialog", true, caption, Close, Close, false, QSize(200, 100))
{
  KTextEdit *w = new KTextEdit(text, QString::null, this);
  w->setReadOnly(true);
  w->setWordWrap(wrapAtWidgetWidth ? KTextEdit::WidgetWidth : KTextEdit::NoWrap);
  setMainWidget(w);
}

//-----------------------------------------------------------------------------
Process::SynchronousDialog::SynchronousDialog(const QString &msg)
: Dialog(qApp->mainWidget(), "sync_process_dialog", true, i18n("Executing..."),
         Cancel, Cancel, false, QSize(200, 100))
{
  QLabel* label = new QLabel(msg, this);
  setMainWidget(label);
}

void Process::SynchronousDialog::finished()
{
  emit accept();
}

Process::State Process::runSynchronouslyDialog(Base &process, RunActions actions, const QString& msg)
{
  // run process for at most 1s without dialog
  Synchronous sync(1000);
  QObject::connect(&process, SIGNAL(done(int)), &sync, SLOT(done()));
  QObject::connect(&process, SIGNAL(requestSynchronousStop()), &sync, SLOT(done()));
  if ( (actions & Start) && !process.start(0) ) return process.state();
  if (sync.enterLoop()) return process.state();

  // continue running the process with a dialog
  SynchronousDialog dialog(msg);
  QObject::connect(&process, SIGNAL(done(int)), &dialog, SLOT(finished()));
  QObject::connect(&process, SIGNAL(requestSynchronousStop()), &dialog, SLOT(finished()));
  int res = dialog.exec();
  if (res == QDialog::Rejected) {
    process.kill();
  }
  return process.state();
}
