/***************************************************************************
 *   Copyright (C) 2005 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "log_view.h"
#include "log_view.moc"

#include <qeventloop.h>

#include "global_config.h"
#include "common/gui/purl_gui.h"
#include "common/gui/misc_gui.h"

Log::Widget::Widget(QWidget *parent, const char *name)
  : KTextEdit(parent, name), _selectedLine(0)
{
  setTextFormat(Qt::RichText);
  setWordWrap(WidgetWidth);
  setWrapPolicy(Anywhere);
  setUndoRedoEnabled(false);
  QPalette p = palette();
  setReadOnly(true);
  setPalette(p);
  setMinimumWidth(300);
}

void Log::Widget::updateDebugLevel()
{
  setDebugLevel(GlobalConfig::debugLevel());
}

void Log::Widget::logExtra(const QString &text)
{
  _text += text;
  if ( GlobalConfig::logOutputType()==GuiConsole ) fprintf(stdout, "%s", text.latin1());
}

void Log::Widget::doLog(LineType type, const QString &text, Action action)
{
  doLog(text, type.data().color, type.data().bold, action);
}

void Log::Widget::doLog(DebugLevel level, const QString &text, Action action)
{
  doLog(text, level.data().color, false, action);
}

void Log::Widget::doLog(const QString &text, const QString &color, bool bold, Action action)
{
  if (action == LogFileOnly) return;
  logExtra(text + "\n");
  QString s = QString("<font color=%1>").arg(color);
  if (bold) s += "<b>";
  s += escapeXml(text) + "&nbsp;";
  if (bold) s += "</b>";
  s += "</font>";
  append(s);
  // ### DANGER
  if ( action==Immediate)
#ifdef NO_KDE
    QCoreApplication::processEvents(QEventLoop::ExcludeUserInput);
#else
    QApplication::eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
#endif
}

Q3PopupMenu *Log::Widget::createPopupMenu(const QPoint &pos)
{
  updateDebugLevel();
  _popup = KTextEdit::createPopupMenu(pos);
  QPixmap icon = IconLoader::loadIcon("filesave", IconLoader::Small);
  _popup->insertItem(icon, "Save As...", this, SLOT(saveAs()));
  icon = IconLoader::loadIcon("fileclose", IconLoader::Small);
  _popup->insertItem(icon, "Clear", this, SLOT(clear()));
  _popup->insertSeparator();
  FOR_EACH(DebugLevel, level) {
    _id[level.type()] = _popup->insertItem(level.label());
    _popup->setItemChecked(_id[level.type()], _debugLevel==level);
  }
  _popup->insertSeparator();
  int id = _popup->insertItem(i18n("Output in console"), this, SLOT(toggleConsoleOutput()));
  _popup->setItemChecked(id, GlobalConfig::logOutputType()==GuiConsole);
  connect(_popup, SIGNAL(activated(int)), SLOT(toggleVisible(int)));
  return _popup;
}

void Log::Widget::toggleVisible(int id)
{
  FOR_EACH(DebugLevel, level) {
    if ( _id[level.type()]==id ) {
      _debugLevel = level;
      GlobalConfig::writeDebugLevel(level);
      break;
    }
  }
}

void Log::Widget::toggleConsoleOutput()
{
  GlobalConfig::writeLogOutputType(GlobalConfig::logOutputType()==GuiOnly ? GuiConsole : GuiOnly);
}

void Log::Widget::sorry(const QString &message, const QString &details)
{
  logExtra(message + " [" + details + "]\n");
  MessageBox::detailedSorry(message, details, Log::Show);
}

bool Log::Widget::askContinue(const QString &message)
{
  bool ok = MessageBox::askContinue(message);
  logExtra(message + " [" + (ok ? "continue" : "cancel") + "]\n");
  return ok;
}

void Log::Widget::clear()
{
  KTextEdit::clear();
  _text = QString::null;
}

void Log::Widget::saveAs()
{
  PURL::Url url = PURL::getSaveUrl(":save_log", "text/x-log", this, i18n("Save log to file"), PURL::AskOverwrite);
  if ( url.isEmpty() ) return;
  url.write(_text, *this);
}

QColor calculateAlternateBackgroundColor(const QColor& base)
{
    if (base == Qt::white)
        return QColor(238,246,255);
    else
    {
        int h, s, v;
        base.hsv( &h, &s, &v );
        if (v > 128)
            return base.dark(106);
        else if (base != Qt::black)
            return base.light(110);

        return QColor(32,32,32);
    }
}

void Log::Widget::contentsMouseReleaseEvent(QMouseEvent *e)
{
  KTextEdit::contentsMouseReleaseEvent(e);
  if ( e->button()!=Qt::LeftButton ) return;
  QColor color = colorGroup().base();
  setParagraphBackgroundColor(_selectedLine, color);
  _selectedLine = paragraphAt(e->pos());
  color = calculateAlternateBackgroundColor(color);
  setParagraphBackgroundColor(_selectedLine, color);
  lineClicked(_selectedLine);
}
