/***************************************************************************
 * Copyright (C) 2010 Nicolas Hadacek <hadacek@kde.org>                    *
 * Copyright (C) 2009-2010 Alberto Maccioni                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "hid_port.h"

#ifdef Q_OS_UNIX
#  include <sys/types.h>
#  include <sys/stat.h>
#  include <sys/ioctl.h>
#  include <fcntl.h>
#  include <linux/hiddev.h>
#  include <errno.h>
#  include <unistd.h>
#endif

#include "common/common/number.h"

//-----------------------------------------------------------------------------
bool Port::HID::isAvailable()
{
#ifdef Q_OS_UNIX
  return true;
#else
  return false;
#endif
}

QString Port::HID::devicePath(uint index)
{
  return QString("/dev/usb/hiddev%1").arg(index);
}

bool Port::HID::getDeviceIds(uint index, uint& vendorId, uint& productId)
{
#ifdef Q_OS_UNIX
  const QString path = devicePath(index);
  const int fd = ::open(path.latin1(), O_RDONLY);
  if (fd < 0) {
     return false;
  }
  hiddev_devinfo info;
  const bool ok = (ioctl(fd, HIDIOCGDEVINFO, &info) >= 0);
  if (ok) {
    vendorId = info.vendor;
    productId = info.product;
  }
  ::close(fd);
  return ok;
#else
  return false;
#endif
}

QStringList Port::HID::probedDeviceList()
{
  QStringList list;
#ifdef Q_OS_UNIX
  for (uint i=0; ; ++i) {
    uint vendorId = 0x0;
    uint productId = 0x0;
    if (getDeviceIds(i, vendorId, productId)) {
      const QString s = QString("%1 - Vendor Id: %2 - Product Id: %3").arg(devicePath(i))
                        .arg(toLabel(NumberBase::Hex, vendorId, 4)).arg(toLabel(NumberBase::Hex, productId, 4));
      list.append(s);
    } else if (i >= 10) { // check at least hiddev0 to hiddev9
      break;
    }
  }
#endif
  return list;
}

bool Port::HID::findDevice(uint vendorId, uint productId, uint& index)
{
#ifdef Q_OS_UNIX
  for (index=0; ; ++index) {
    uint vid = 0x0;
    uint pid = 0x0;
    if (getDeviceIds(index, vid, pid)) {
      if (vid == vendorId && pid == productId) {
        return true;
      }
    } else if (index >= 10) { // check at least hiddev0 to hiddev9
      break;
    }
  }
 #endif
  return false;
}

//-----------------------------------------------------------------------------
#ifdef Q_OS_UNIX

Port::HID::HID(Log::Base &base, uint vendorId, uint productId)
: Base(base), _vendorId(vendorId), _productId(productId),
 _handle(-1)
{
}

Port::HID::~HID()
{
  close();
}

void Port::HID::setSystemError(const QString &message)
{
  log(Log::LineType::Error, message + QString(" (err=%1).").arg(errno));
}

bool Port::HID::internalOpen()
{
  uint index = 0;
  if (!findDevice(_vendorId, _productId, index)) {
    log(Log::LineType::Error, i18n("Could not find HID device (vendor=%1 product=%2).")
             .arg(toLabel(NumberBase::Hex, _vendorId, 4)).arg(toLabel(NumberBase::Hex, _productId, 4)));
    return false;
  }
  QString path = devicePath(index);
  log(Log::DebugLevel::Extra, QString("found HID device as \"%1\"").arg(path));
  _handle = ::open(path.toAscii(), O_RDONLY);
  if (_handle < 0) {
    setSystemError(i18n("Error opening HID device."));
    return false;
  }
  return true;
}

void Port::HID::internalClose()
{
  if (_handle >= 0) {
    ::close(_handle);
    _handle = -1;
  }
}

#endif
