#lang scribble/doc
@(require scribble/manual
          "common.ss"
          (for-label scheme/base
                     scheme/unit
                     scheme/contract
                     launcher/launcher
                     launcher/launcher-sig
                     launcher/launcher-unit
                     compiler/embed
                     scheme/gui/base))

@title[#:tag "launcher"]{Installation-Specific Launchers for Scheme Code}

@defmodule[launcher/launcher]

The @schememodname[launcher/launcher] library provides functions for
creating @defterm{launchers}, which are similar to stand-alone
executables, but sometimes smaller because they depend permanently on
the local PLT Scheme installation. In the case of Unix, in particular,
a launcher is simply a shell script. The @|mzc| tool provides no
direct support for creating launchers.

@section{Creating Launchers}

@defproc[(make-mred-launcher [args (listof string?)]
                             [dest path-string?]
                             [aux (listof (cons/c symbol? any/c)) null])
         void?]{

Creates the launcher @scheme[dest], which starts MrEd with the
command-line arguments specified as strings in @scheme[args]. Extra
arguments passed to the launcher at run-time are appended (modulo
special Unix/X flag handling, as described below) to this list and
passed on to MrEd. If @scheme[dest] exists already, as either a file
or directory, it is replaced.

The optional @scheme[aux] argument is an association list for
platform-specific options (i.e., it is a list of pairs where the first
element of the pair is a key symbol and the second element is the
value for that key). See also @scheme[build-aux-from-path]. See
@scheme[create-embedding-executable] for a list that applies to both
stand-alone executables and launchers under Windows and Mac OS X MrEd;
the following additional associations apply to launchers:

@itemize[

 @item{@scheme['independent?] (Windows) --- a boolean; @scheme[#t]
       creates an old-style launcher that is independent of the
       MzScheme or MrEd binary, like @exec{setup-plt.exe}. No other
       @scheme[aux] associations are used for an old-style launcher.}

 @item{@scheme['exe-name] (Mac OS X, @scheme['script-3m] or
       @scheme['script-cgc] variant) --- provides the base name for a
       @scheme['3m]-/@scheme['cgc]-variant launcher, which the script
       will call ignoring @scheme[args]. If this name is not provided,
       the script will go through the MrEd executable as usual.}

 @item{@scheme['relative?] (all platforms) --- a boolean, where
        @scheme[#t] means that the generated launcher should find the
        base MrEd executable through a relative path.}

]

For Unix/X, the script created by @scheme[make-mred-launcher] detects
and handles X Windows flags specially when they appear as the initial
arguments to the script. Instead of appending these arguments to the
end of @scheme[args], they are spliced in after any X Windows flags
already listed listed in @scheme[args]. The remaining arguments (i.e.,
all script flags and arguments after the last X Windows flag or
argument) are then appended after the spliced @scheme[args].}


@defproc[(make-mzscheme-launcher [args (listof string?)]
                                 [dest path-string?]
                                 [aux (listof (cons/c symbol? any/c)) null])
         void?]{

Like @scheme[make-mred-launcher], but for starting MzScheme. Under Mac
OS X, the @scheme['exe-name] @scheme[aux] association is ignored.}


@defproc[(make-mred-program-launcher [file string?]
                                     [collection string?]
                                     [dest path-string?])
         void?]{

Calls @scheme[make-mred-launcher] with arguments that start the MrEd
program implemented by @scheme[file] in @scheme[collection]:
@scheme[(list "-l-" (string-append collection "/" file))]. The
@scheme[_aux] argument to @scheme[make-mred-launcher] is generated by
stripping the suffix (if any) from @scheme[file], adding it to the
path of @scheme[collection], and passing the result to
@scheme[build-aux-from-path].}


@defproc[(make-mzscheme-program-launcher [file string?]
                                         [collection string?]
                                         [dest path-string?])
        void?]{

Like @scheme[make-mred-program-launcher], but for
@scheme[make-mzscheme-launcher].}


@defproc[(install-mred-program-launcher [file string?]
                                        [collection string?]
                                        [name string?])
         void?]{

Same as 

@schemeblock[
(make-mred-program-launcher 
 file collection
 (mred-program-launcher-path name))
]}

@defproc[(install-mzscheme-program-launcher [file string?]
                                            [collection string?]
                                            [name string?])
         void?]{

Same as 

@schemeblock[
(make-mzscheme-program-launcher 
 file collection
 (mzscheme-program-launcher-path name))
]}

@; ----------------------------------------------------------------------

@section{Launcher Path and Platform Conventions}

@defproc[(mred-program-launcher-path [name string?]) path?]{

Returns a pathname for an executable in the PLT Scheme installation
called something like @scheme[name].  For Windows, the @filepath{.exe}
suffix is automatically appended to @scheme[name]. For Unix,
@scheme[name] is changed to lowercase, whitespace is changed to
@litchar{-}, and the path includes the @filepath{bin} subdirectory of
the PLT Scheme installation. For Mac OS X, the @filepath{.app} suffix
is appended to @scheme[name].}


@defproc[(mzscheme-program-launcher-path [name string?]) path?]{

Returns the same path as @scheme[(mred-program-launcher-path name)]
for Unix and Windows. For Mac OS X, the result is the same as for
Unix.}


@defproc[(mred-launcher-is-directory?) boolean?]{

Returns @scheme[#t] if MrEd launchers for the current platform are
directories from the user's perspective. For all currently supported
platforms, the result is @scheme[#f].}


@defproc[(mzscheme-launcher-is-directory?) boolean?]{

Like @scheme[mred-launcher-is-directory?], but for MzScheme
launchers.}


@defproc[(mred-launcher-is-actually-directory?) boolean?]{

Returns @scheme[#t] if MrEd launchers for the current platform are
implemented as directories from the filesystem's perspective. The
result is @scheme[#t] for Mac OS X, @scheme[#f] for all other
platforms.}


@defproc[(mzscheme-launcher-is-actually-directory?) boolean?]{

Like @scheme[mred-launcher-is-actuall-directory?], but for MzScheme
launchers. The result is @scheme[#f] for all platforms.}


@defproc[(mred-launcher-add-suffix [path-string? path]) path?]{

Returns a path with a suitable executable suffix added, if it's not
present already.}

@defproc[(mzscheme-launcher-add-suffix [path-string? path]) path?]{

Like @scheme[mred-launcher-add-suffix], but for MzScheme launchers.}


@defproc[(mred-launcher-put-file-extension+style+filters)
         (values (or/c string? false/c)
                 (listof (one-of/c 'packages 'enter-packages))
                 (listof (list/c string? string?)))]{

Returns three values suitable for use as the @scheme[extension],
@scheme[style], and @scheme[filters] arguments to @scheme[put-file],
respectively.

If MrEd launchers for the current platform were directories form the
user's perspective, the @scheme[style] result is suitable for use with
@scheme[get-directory], and the @scheme[extension] result may be a
string indicating a required extension for the directory name. }


@defproc[(mzscheme-launcher-put-file-extension+style+filters)
         (values (or/c string? false/c)
                 (listof (one-of/c 'packages 'enter-packages))
                 (listof (list/c string? string?)))]{

Like @scheme[mred-launcher-get-file-extension+style+filters], but for
MzScheme launchers.}


@; ----------------------------------------------------------------------

@section{Launcher Configuration}

@defproc[(mred-launcher-up-to-date? [dest path-string?]
                                    [aux (listof (cons/c symbol? any/c))])
         boolean?]{

Returns @scheme[#t] if the MrEd launcher @scheme[dest] does not need
to be updated, assuming that @scheme[dest] is a launcher and its
arguments have not changed.}

@defproc[(mzscheme-launcher-up-to-date? [dest path-string?]
                                        [aux (listof (cons/c symbol? any/c))])
         boolean?]{

Analogous to @scheme[mred-launcher-up-to-date?], but for a MzScheme
launcher.}

@defproc[(build-aux-from-path [path path-string?])
         (listof (cons/c symbol? any/c))]{

Creates an association list suitable for use with
@scheme[make-mred-launcher] or @scheme[create-embedding-executable].
It builds associations by adding to @scheme[path] suffixes, such as
@filepath{.icns}, and checking whether such a file exists.

The recognized suffixes are as follows:

@itemize[

 @item{@filepath{.icns} @'rarr @scheme['icns] file for use under Mac
       OS X}

 @item{@filepath{.ico} @'rarr @scheme['ico] file for use under
       Windows}

 @item{@filepath{.lch} @'rarr @scheme['independent?] as @scheme[#t]
       (the file content is ignored) for use under Windows}

 @item{@filepath{.creator} @'rarr @scheme['creator] as the initial
       four characters in the file for use under Mac OS X}

 @item{@filepath{.filetypes} @'rarr @scheme['file-types] as
       @scheme[read] content (a single S-expression), and
       @scheme['resource-files] as a list constructed by finding
       @scheme["CFBundleTypeIconFile"] entries in @scheme['file-types]
       (and filtering duplicates); for use under Mac OS X}

 @item{@filepath{.utiexports} @'rarr @scheme['uti-exports] as
       @scheme[read] content (a single S-expression); for use under
       Mac OS X}

]}

@defparam[current-launcher-variant variant symbol?]{

A parameter that indicates a variant of MzScheme or MrEd to use for
launcher creation and for generating launcher names. The default is
the result of @scheme[(system-type 'gc)]. Under Unix and Windows, the
possibilities are @scheme['cgc] and @scheme['3m]. Under Mac OS X, the
@scheme['script-3m] and @scheme['script-cgc] variants are also
available for MrEd launchers.}

@defproc[(available-mred-variants) (listof symbol?)]{

Returns a list of symbols corresponding to available variants of MrEd
in the current PLT Scheme installation. The list normally includes at
least one of @scheme['3m] or @scheme['cgc]--- whichever is the result
of @scheme[(system-type 'gc)]---and may include the other, as well as
@scheme['script-3m] and/or @scheme['script-cgc] under Mac OS X.}

@defproc[(available-mzscheme-variants) (listof symbol?)]{

Returns a list of symbols corresponding to available variants of
MzScheme in the current PLT Scheme installation. The list normally
includes at least one of @scheme['3m] or @scheme['cgc]---whichever is
the result of @scheme[(system-type 'gc)]---and may include the other.}

@; ----------------------------------------

@section{Launcher Creation Signature}

@defmodule[launcher/launcher-sig]

@defsignature/splice[launcher^ ()]{

Includes the identifiers provided by @schememodname[launcher/launcher].}

@; ----------------------------------------

@section{Launcher Creation Unit}

@defmodule[launcher/launcher-unit]

@defthing[launcher@ unit?]{

A unit that imports nothing and exports @scheme[launcher^].}
