/* dhcpd.c
 *
 * DHCP Client plugin for pppd
 *
 * Ben McKeegan <ben@netservers.co.uk> August 2002
 *
 * Portions derived from udhcp DHCP client by
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
 
#include <limits.h>

/* #define DHCP_TEST_SHORTLEASE */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/file.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <string.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <errno.h>

#include "pppd.h"

char pppd_version[] = VERSION;

#include "dhcpd.h"
#include "dhcpc.h"
#include "options.h"
#include "clientpacket.h"
#include "packet.h"
#include "socket.h"
#include "debug.h"

static void (*dhcp_old_ip_choose_hook)(u_int32_t *addrp);

static int dhcp_state;
static u_int32_t requested_ip; /* = 0 */
u_int32_t assigned_ip; /* value supplied to remote ppp */
static unsigned long server_addr;
unsigned long lease, renew_timeout;
unsigned long xid = 0;
static int packet_num; /* = 0 */
static int fd;

#define LISTEN_NONE 0
#define LISTEN_KERNEL 1
#define LISTEN_RAW 2
static int listen_mode;

#define DHCP_RX_POLL_INTERVAL 8

/* structure retained only for linking against clientpacket.c */
struct client_config_t client_config = {
	/* Default options. */
	abort_if_no_lease: 0,
	foreground: 0,
	quit_after_lease: 0,
	interface: "eth0",
	pidfile: NULL,
	script: NULL,
	clientid: NULL,
	hostname: NULL,
	ifindex: 0,
	arp: "\0\0\0\0\0\0", /* appease gcc-3.0 */
	giaddr: 0,
	siaddr: 0,
	subnet_selection: 0
};

static char *dhcp_server= NULL;
static char *dhcp_relay_address= NULL;
static char *dhcp_subnet_selection= NULL;
static int dhcp_relay_port= SERVER_PORT;

static option_t dhcpc_options[] = {
  { "dhcp-interface", o_string, &client_config.interface,
      "Interface to send DHCP requests on." },
  { "dhcp-server", o_string, &dhcp_server,
      "IP address of DHCP server (disable broadcasts)" },
  { "dhcp-relay-address", o_string, &dhcp_relay_address,
      "Our address (supplied to server as relay address, defaults to interface address)" },
  { "dhcp-subnet-selection", o_string, &dhcp_subnet_selection,
      "Subnet to request addresses for" },
  { "dhcp-relay-port", o_int, &dhcp_relay_port,
      "Port we listen on (normally bootps)" },
  { NULL }
};


void change_mode(int new_mode);
void dhcp_process_ack(struct dhcpMessage *packet);
void dhcp_rx(void *dummy);
void dhcp_release(void *ptr, int arg);
void dhcp_renew(void *dummy);
void dhcp_request_new();
void dhcp_ip_choose(u_int32_t *addrp);
void dhcp_read_options(void);


void change_mode(int new_mode)
{
  if (new_mode == LISTEN_RAW && client_config.giaddr)
    if (listen_mode == LISTEN_KERNEL)
      return;
    else
      new_mode = LISTEN_KERNEL;

  dbglog("DHCPC: entering %s listen mode on %s",
       new_mode ? (new_mode == LISTEN_KERNEL ? "kernel" : "raw") : "none",
	 client_config.siaddr ? "*" : client_config.interface);
  if (fd >= 0)
    close(fd);
  fd = -1;
  listen_mode = new_mode;
  if (listen_mode == LISTEN_KERNEL)
    fd = listen_socket(INADDR_ANY,
       dhcp_relay_port, client_config.siaddr ? NULL : client_config.interface);
  else if (listen_mode == LISTEN_RAW)
    fd = raw_socket(client_config.ifindex);
  
  if (listen_mode != LISTEN_NONE && fd < 0) {
    fatal("DHCPC: couldn't listen on socket, %s", sys_errlist[errno]);
  }
  
}

void dhcp_process_ack(struct dhcpMessage *packet) {
  unsigned char *temp;
  struct in_addr temp_addr;
  
  if (!(temp = get_option(packet, DHCP_LEASE_TIME))) {
    warn("DHCPC: No lease time with ACK, using 1 hour lease");
    lease = 60 * 60;
  } else {
    memcpy(&lease, temp, 4);
    lease = ntohl(lease);
  }

#ifdef DHCP_TEST_SHORTLEASE
  warn("DHCPC: Test mode: shortening lease to 60 seconds!");
  lease=60;
#endif
  
  /* enter bound state */
  temp_addr.s_addr = packet->yiaddr;
  info("DHCPC: Lease of %s obtained, lease time %ld", 
       inet_ntoa(temp_addr), lease);
  
  requested_ip = packet->yiaddr;
  
  dhcp_state = BOUND;
  change_mode(LISTEN_NONE);
}


void dhcp_rx(void *dummy) {
  unsigned char *temp, *message;
  unsigned long t2 = 0;
  fd_set rfds;
  int retval;
  struct timeval tv;
  int c, len;
  struct dhcpMessage packet;
  struct in_addr temp_addr;
  int pid_fd;
  time_t now;

  dbglog("DHCPC: Entering RX Polling function");

  if (listen_mode != LISTEN_KERNEL)
    return;	  
  if (dhcp_state != RENEWING && dhcp_state != REBINDING && dhcp_state != BOUND)
    return;

  tv.tv_sec = 0;  /* poll only, do not wait */
  tv.tv_usec = 0;
  FD_ZERO(&rfds);
  FD_SET(fd, &rfds);

  dbglog("DHCPC: Polling for new packets");
  
  while(select(fd + 1, &rfds, NULL, NULL, &tv)) { 
    len = get_packet(&packet, fd);
    
    if (len == -1 && errno != EINTR) {
      dbglog("DHCPC: error on read, %s, reopening socket", sys_errlist[errno]);
      change_mode(LISTEN_KERNEL);
    }
    if (len < 0) continue;
    
    if (packet.xid != xid) {
      dbglog("DHCPC: Ignoring XID %lx (our xid is %lx)",
	    (unsigned long) packet.xid, xid);
      continue;
    }
			
    if ((message = get_option(&packet, DHCP_MESSAGE_TYPE)) == NULL) {
      dbglog("DHCPC: Could not get option from packet -- ignoring");
      continue;
    }
			
    if ((*message == DHCPOFFER || *message == DHCPACK) &&
	  client_config.subnet_selection &&
	  get_option(&packet, DHCP_SUBNET_SELECTION) == NULL) {
  	  warn("DHCPC: server does not support subnet selection, discarding response");
	  continue;
    }
      
    if (dhcp_state == RENEWING || dhcp_state == REBINDING) {
      if (*message == DHCPACK) {
	dhcp_process_ack(&packet);
	if (requested_ip != assigned_ip) {
	  fatal("DHCPC: Terminating because address has changed!");
	}
       	untimeout(&dhcp_renew,NULL);
	timeout(&dhcp_renew, NULL, lease / 2,0);
	return;
      } else if (*message == DHCPNAK) {
	/* return to init state */
	dhcp_state = RELEASED;
	change_mode(LISTEN_NONE);
	fatal("DHCP Lease was NAK'd during renewal/rebinding!");
      }
    }
  }
  timeout(&dhcp_rx,NULL,DHCP_RX_POLL_INTERVAL,0);
  
}

void dhcp_release(void *ptr, int arg)
{
  /* send release packet */
  if (dhcp_state == BOUND || dhcp_state == RENEWING || dhcp_state == REBINDING)
    send_release(server_addr, requested_ip); /* unicast */
  
  change_mode(LISTEN_NONE);
  dhcp_state = RELEASED;
}


void dhcp_renew(void *dummy) {
  static long dhcp_renew_timeout;
  static int dhcp_packet_interval;

  dbglog("DHCPC: Entering renewal timer function");

  if (dhcp_state == BOUND) {
    dhcp_state = RENEWING;
    change_mode(LISTEN_KERNEL);
    timeout(&dhcp_rx,NULL,3,0);
    dbglog("DHCPC: Entering renew state");
    /* 1/2 of lease used on entry, timeout renew after another 3/8 of lease */
    dhcp_renew_timeout = (lease * 0x3) >> 3;
    dhcp_packet_interval = 2;
  }
  if (dhcp_state != RENEWING && dhcp_state != REBINDING)
    return;
  
  if (dhcp_renew_timeout <= 0) {
    switch(dhcp_state) {
    case REBINDING:
      fatal("DHCP Timed out rebinding");
    case RENEWING:
      /* 1/8 of lease remaining */
      dhcp_renew_timeout = (lease) >> 3;
      dhcp_state = REBINDING;
      dhcp_packet_interval = 2;
    }
  }

  if (dhcp_packet_interval < 64) 
    dhcp_packet_interval *= 2;

  send_renew(xid, dhcp_state==REBINDING ? client_config.siaddr : server_addr, requested_ip); 
  
  dhcp_renew_timeout-=dhcp_packet_interval;
  timeout(&dhcp_renew, NULL, dhcp_packet_interval,0);
}

void dhcp_request_new() {
  unsigned long request_timeout = 0;
  unsigned char *message,*temp;
  unsigned long t1 = 0, t2 = 0;
  fd_set rfds;
  int retval;
  struct timeval tv;
  int c, len;
  struct dhcpMessage packet;
  time_t now;
  
  if (dhcp_state == BOUND || dhcp_state == RENEWING || dhcp_state == REBINDING)
    return;

  dhcp_state = INIT_SELECTING;
  len=strlen(peer_authname);
  client_config.clientid = malloc(len + 3);
  client_config.clientid[OPT_CODE] = DHCP_CLIENT_ID;
  client_config.clientid[OPT_LEN] = len + 1;
  client_config.clientid[OPT_DATA] = 0;
  memcpy(client_config.clientid + 3, peer_authname, len);

  change_mode(LISTEN_KERNEL);
  for (;;) {
    
    tv.tv_sec = request_timeout - time(0);
    tv.tv_usec = 0;
    FD_ZERO(&rfds);
    
    if (fd >= 0) FD_SET(fd, &rfds);
    
    if (tv.tv_sec > 0) {
      dbglog("Waiting on select...\n");
      retval = select(fd + 1, &rfds, NULL, NULL, &tv);
    } else retval = 0; /* If we already timed out, fall through */
    
    now = time(0);
    if (retval == 0) {
      /* timeout dropped to zero */
      switch (dhcp_state) {
      case INIT_SELECTING:
	if (packet_num < 3) {
	  if (packet_num == 0)
	    xid = random_xid();
	  
	  /* send discover packet */
	  send_discover(xid, requested_ip); /* broadcast */
	  
	  request_timeout = now + ((packet_num == 2) ? 10 : 2);
	  packet_num++;
	} else {
	  info("DHCPC: No lease, failing.");
	  return;
	}
	break;
      case REQUESTING:
	if (packet_num < 3) {
	  /* send request packet */
	  send_selecting(xid, server_addr, requested_ip); /* broadcast */
	  
	  request_timeout = now + ((packet_num == 2) ? 10 : 2);
	  packet_num++;
	} else {
	  /* timed out, go back to init state */
	  dhcp_state = INIT_SELECTING;
	  request_timeout = now;
	  packet_num = 0;
	}
	break;
      }
    } else if (retval > 0 && listen_mode != LISTEN_NONE && FD_ISSET(fd, &rfds)) {
      /* a packet is ready, read it */
      
      if (listen_mode == LISTEN_KERNEL)
	len = get_packet(&packet, fd);
      else len = get_raw_packet(&packet, fd);
      
      if (len == -1 && errno != EINTR) {
	dbglog("DHCPC: error on read, %s, reopening socket", sys_errlist[errno]);
	change_mode(listen_mode); /* just close and reopen */
      }
      if (len < 0) continue;
      
      if (packet.xid != xid) {
	dbglog("DHCPC: Ignoring XID %lx (our xid is %lx)",
	      (unsigned long) packet.xid, xid);
	continue;
      }
      
      if ((message = get_option(&packet, DHCP_MESSAGE_TYPE)) == NULL) {
	warn("DHCPC: couldnt get option from packet -- ignoring");
	continue;
      }

      if ((*message == DHCPOFFER || *message == DHCPACK) &&
	  client_config.subnet_selection &&
	  get_option(&packet, DHCP_SUBNET_SELECTION) == NULL) {
  	  warn("DHCPC: server does not support subnet selection, discarding response");
	  continue;
      }
      
      switch (dhcp_state) {
      case INIT_SELECTING:
	/* Must be a DHCPOFFER to one of our xid's */
	if (*message == DHCPOFFER) {
	  if ((temp = get_option(&packet, DHCP_SERVER_ID))) {
	    memcpy(&server_addr, temp, 4);
	    xid = packet.xid;
	    requested_ip = packet.yiaddr;
	    
	    /* enter requesting state */
	    dhcp_state = REQUESTING;
	    request_timeout = now;
	    packet_num = 0;
	  } else {
	    dbglog("DHCPC: No server ID in message");
	  }
	}
	break;
      case REQUESTING:
	if (*message == DHCPACK) {
	  dhcp_process_ack(&packet);
	  assigned_ip = requested_ip;
	  dbglog("DHCPC: Setting renewal timer for %d seconds", lease /2);
  	  timeout(&dhcp_renew, NULL, lease / 2,0);
	  return;
	} else if (*message == DHCPNAK) {
	  /* return to init state */
	  info("DHCPC: Received DHCP NAK");
	  dhcp_state = INIT_SELECTING;
	  request_timeout = now;
	  requested_ip = 0;
	  packet_num = 0;
	  sleep(3); /* avoid excessive network traffic */
	}
	break;
      }					
    } else if (retval == -1 && errno == EINTR) {
      /* a signal was caught */
      dbglog("DHCPC: signal caught");
      
    } else {
      /* An error occured */
        dbglog("DHCPC: error on select, %s, reopening socket", sys_errlist[errno]);
        change_mode(listen_mode); /* just close and reopen */
    }
    
  }
  return;
}

void dhcp_ip_choose(u_int32_t *addrp) {
  u_int32_t entryvalue;

  dbglog("DHCPC: ip_choose_hook entered with peer name %s",peer_authname);

  if (dhcp_old_ip_choose_hook) {
    dbglog("DHCPC: calling ip_choose_hook for previously loaded module");
    entryvalue= *addrp;
    dhcp_old_ip_choose_hook(addrp);
    if (*addrp != entryvalue) {
      info("DHCPC: A previously loaded module has supplied an IP address.  Skipping DHCP.");
      return;
    }
  }

  dhcp_read_options();

  if (strlen(peer_authname))
    dhcp_request_new();

  if (dhcp_state == BOUND || dhcp_state == RENEWING || dhcp_state == REBINDING)
    *addrp=assigned_ip;
  else
    fatal("DHCPC: Failed to obtain an IP address.  Terminating connection.");

  return;

}


void plugin_init(void)
{
  dhcp_old_ip_choose_hook= ip_choose_hook;
  /* we save pointer to ip_choose_hook so we can defer to other modules
     that may specify an IP, e.g. radius */
  ip_choose_hook= dhcp_ip_choose;
  add_options(dhcpc_options);
  add_notifier(&exitnotify, dhcp_release, NULL);
  info("DHCPC: plugin initialized");

}


void dhcp_read_options(void)
{
  struct in_addr sa;
  
  if (read_interface(client_config.interface, &client_config.ifindex, 
		     &sa.s_addr, client_config.arp) < 0)
    fatal("DHCPC: Could not find interface");


  if (dhcp_relay_address && !inet_aton(dhcp_relay_address, &sa))
    fatal("DHCPC: Invalid relay address specified.");

  client_config.giaddr=sa.s_addr;
  info("DHCPC: Using relay address of '%s'", inet_ntoa(sa));

  if (dhcp_subnet_selection)
    if (inet_aton(dhcp_subnet_selection, &sa)) {
      client_config.subnet_selection= sa.s_addr;
      info("DHCPC: Requesting subnet '%s'", inet_ntoa(sa));
    } else
      fatal("DHCPC: Invalid address in subnet selection option");

  if (dhcp_server)
    if (inet_aton(dhcp_server, &sa)) {
      client_config.siaddr= sa.s_addr;
      info("DHCPC: Unicasting to server '%s' only", inet_ntoa(sa));
    } else
      fatal("DHCPC: Invalid server address specified.");
  else
    info("DHCPC: Broadcasting to servers on interface '%s'", client_config.interface);

}

