/* free.c - a /proc implementation of free */
/* Dec14/92 by Brian Edmonds */
/* Thanks to Rafal Maszkowski for the Total line */

#include "proc/sysinfo.h"
#include "proc/version.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <getopt.h>

/* set this to whatever you want by default */
int byteshift = 10;
int total = 0;

int
main(int argc, char **argv)
{
	int i, rtime = 0, old_fmt = 0;

	/* check startup flags */
	while ((i = getopt(argc, argv, "bkmos:tV")) != -1) {
		switch (i) {
		case 'b':
			byteshift = 0;
			break;
		case 'k':
			byteshift = 10;
			break;
		case 'm':
			byteshift = 20;
			break;
		case 'o':
			old_fmt = 1;
			break;
		case 's':
			rtime = 1000000 * atof(optarg);
			break;
		case 't':
			total = 1;
			break;
		case 'V':
			display_version();
			exit(0);
		default:
			fprintf(stderr,
				"usage: %s [-b|-k|-m] [-o] [-s delay] [-t] [-V]\n",
				argv[0]);
			return 1;
		}
	}

	/* redirect stdin to /proc/meminfo */
	if (linux_version_code < LINUX_VERSION(2, 0, 0)) {
		close(0);
		if (open("/proc/meminfo", O_RDONLY) < 0) {
			perror("open");
			return 2;
		}
	}

	do {
		struct meminfo_struct mem_info = MEMINFO_ZERO;

		meminfo(&mem_info);
		if (mem_info.mem.total == 0) {
			fprintf(stderr,
				"Cannot get size of memory from /proc/meminfo\n");
			exit(1);
		}

		printf("%18s %10s %10s %10s %10s %10s\n",
		       "total", "used", "free", "shared", "buffers", "cached");
		printf("%-7s %10Ld %10Ld %10Ld %10Ld %10Ld %10Ld\n", "Mem:",
		       mem_info.mem.total >> byteshift,
		       mem_info.mem.used >> byteshift,
		       mem_info.mem.free >> byteshift,
		       mem_info.mem.shared >> byteshift,
		       mem_info.mem.buffers >> byteshift,
		       mem_info.mem.cached >> byteshift);
		if (!old_fmt)
			printf("-/+ buffers/cache: %10Ld %10Ld\n",
			       (mem_info.mem.used
				- mem_info.mem.buffers
				- mem_info.mem.cached) >> byteshift,
			       (mem_info.mem.free
				+ mem_info.mem.buffers
				+ mem_info.mem.cached) >> byteshift);
		printf("%-7s %10Ld %10Ld %10Ld\n", "Swap:",
		       mem_info.swap.total >> byteshift,
		       mem_info.swap.used >> byteshift,
		       mem_info.swap.free >> byteshift);
		if (total == 1)
			printf("%-7s %10Ld %10Ld %10Ld\n", "Total:",
			       (mem_info.mem.total +
				mem_info.swap.total) >> byteshift,
			       (mem_info.mem.used +
				mem_info.swap.used) >> byteshift,
			       (mem_info.mem.free +
				mem_info.swap.free) >> byteshift);

		if (rtime) {
			fputc('\n', stdout);
			fflush(stdout);
			usleep(rtime);
		}
	} while (rtime);

	return 0;
}
