/*
    fchksum - python extension for fast file checksumming
    Copyright (C) 2000-2002  Matthew Mueller <donut@azstarnet.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "pfconfig.h"
#include "fchksum.h"

#include "Python.h"
#include "md5.h"
#include "cksum.h"
#include "sum.h"
#include <zlib.h>

static PyObject *ululToL(unsigned long l0, unsigned long l1){
	if (!l1)
		return PyLong_FromUnsignedLong(l0);
	else{
#ifdef HAVE_LONG_LONG
		return PyLong_FromUnsignedLongLong(l1*((unsigned long long)ULONG_MAX+1)+l0);
#else
		return PyLong_FromDouble(l1*((double)ULONG_MAX+1)+l0);/*icky, but doesn't seem to mess up until you get into many terabyte sized files. Most people will have long long anyway, I hope :)*/
#endif
	}
}

/*static PyObject *p_ululToL(PyObject *self, PyObject *args){
	PyObject *L0, *L1;
	if (!PyArg_ParseTuple(args, "OO", &L0,&L1))
		return NULL;
	return ululToL(PyLong_AsUnsignedLong(L0),PyLong_AsUnsignedLong(L1));
}*/

static FILE * sfopen(PyObject *args,char **fname){
	FILE *f;
	if (!PyArg_ParseTuple(args, "s", fname))
		return NULL;
	if (**fname=='\0')
		f=stdin;
	else
		f=fopen(*fname,"rb");
	if (!f){
		PyErr_SetFromErrnoWithFilename(PyExc_IOError,*fname);
		return NULL;
	}
	return f;
}

static PyObject *fcrc32_L(PyObject *args,uLong *res){
	char *fname;
	FILE *f;
	int fd;
	unsigned long size=0;
	unsigned long size1=0;
	char buffer[FCHKSUMBLOCKSIZE];
	ssize_t sum;
/*	size_t n;*/
	uLong crc = crc32(0L, Z_NULL, 0);

	if ((f=sfopen(args,&fname))==NULL) return NULL;
	fd = fileno(f);
	while (1) {
		/*well, for crc32 taking care of partial reads to fully fill the buffer before testing doesn't seem to help much at all.*/
		/* Read block.  Take care for partial reads.  */
/*		sum=0;
		do{
			n = fread (buffer + sum, 1, 4096 - sum, f);

			sum += n;
		}while (sum < 4096 && n != 0);*/

/*		sum = fread (buffer, 1, FCHKSUMBLOCKSIZE, f);*/
		sum = read (fd, buffer, FCHKSUMBLOCKSIZE);
		if (sum <= 0){
			if (sum < 0){
/*			if (ferror (f)){*/
				PyErr_SetFromErrnoWithFilename(PyExc_IOError,fname);
				fclose(f);
				return NULL;
			}
			fclose(f);
			break;
		}
		crc = crc32(crc, buffer, sum);
		size+=sum;
		if (size < sum)
			++size1;
	}
	*res=crc;
	return ululToL(size,size1);
}

static char fcrc32__doc__[] =
"filename -> checksum, size. Return the crc32 as an int."
;
static PyObject *fcrc32(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;

	if ((size=fcrc32_L(args,&res))==NULL)
		return NULL;
	return Py_BuildValue("(l,N)", res,size);
}

static char fcrc32t__doc__[] =
"filename -> checksum, size. Return the crc32 as a 8 char hex string."
;
static PyObject *fcrc32t(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;
	unsigned char out[9];

	if ((size=fcrc32_L(args,&res))==NULL)
		return NULL;

	sprintf(out,"%08lX",res);
	return Py_BuildValue("(s#,N)", out, 8, size);
}


static PyObject *fmd5_L(PyObject *args,unsigned char *res){
	char *fname;
	FILE *f;
	unsigned long size, size1;

	if ((f=sfopen(args,&fname))==NULL) return NULL;
	if (md5_stream(f,res,&size,&size1)){
		PyErr_SetFromErrnoWithFilename(PyExc_IOError,fname);
		fclose(f);
		return NULL;
	}
	fclose(f);
	return ululToL(size,size1);
}

static char fmd5t__doc__[] =
"filename -> checksum, size. Return the md5sum as a 32 char hex string."
;
static PyObject *fmd5t(PyObject *self,PyObject *args){
	unsigned char res[16];
	unsigned char out[33];
	int i;
	PyObject *size;

	if ((size=fmd5_L(args,res))==NULL)
		return NULL;
	for (i=0;i<16;i++){
		sprintf(out+i*2,"%02x",res[i]);
	}
	return Py_BuildValue("(s#,N)", out,32,size);
}

static char fmd5__doc__[] =
"filename -> checksum, size. Return the md5sum as a 16 byte string."
;
static PyObject *fmd5(PyObject *self,PyObject *args){
	unsigned char res[16];
	PyObject *size;

	if ((size=fmd5_L(args,res))==NULL)
		return NULL;
	return Py_BuildValue("(s#,N)", res,16,size);
}


static PyObject *fcksum_L(PyObject *args,uLong *res){
	char *fname;
	FILE *f;
	unsigned long size, size1;

	if ((f=sfopen(args,&fname))==NULL) return NULL;
	if (cksum_stream(f,res,&size,&size1)){
		PyErr_SetFromErrnoWithFilename(PyExc_IOError,fname);
		fclose(f);
		return NULL;
	}
	fclose(f);
	return ululToL(size,size1);
}

static char fcksumt__doc__[] =
"filename -> checksum, size. Return the cksum as a dec string."
;
static PyObject *fcksumt(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;
	unsigned char out[11];

	if ((size=fcksum_L(args,&res))==NULL)
		return NULL;

	sprintf(out,"%lu",res);
	return Py_BuildValue("(s,N)", out, size);
}

static char fcksum__doc__[] =
"filename -> checksum, size. Return the cksum as an int."
;
static PyObject *fcksum(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;

	if ((size=fcksum_L(args,&res))==NULL)
		return NULL;
/* if res would make a negative pyint, return as a pylong */
/*	if ((long)res<0)
		return Py_BuildValue("(N,N)", PyLong_FromUnsignedLong(res), size);*/
	return Py_BuildValue("(l,N)", res,size);
}


static PyObject *fbsdsum_L(PyObject *args,uLong *res){
	char *fname;
	FILE *f;
	unsigned long size, size1;

	if ((f=sfopen(args,&fname))==NULL) return NULL;
	if (bsd_sum_stream(f,res,&size,&size1)){
		PyErr_SetFromErrnoWithFilename(PyExc_IOError,fname);
		fclose(f);
		return NULL;
	}
	fclose(f);
	return ululToL(size,size1);
}

static char fbsdsumt__doc__[] =
"filename -> checksum, size. Return the bsd-style sum as a 5 char dec string."
;
static PyObject *fbsdsumt(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;
	unsigned char out[6];

	if ((size=fbsdsum_L(args,&res))==NULL)
		return NULL;

	sprintf(out,"%05lu",res);
	return Py_BuildValue("(s#,N)", out, 5, size);
}

static char fbsdsum__doc__[] =
"filename -> checksum, size. Return the bsd-style sum as an int."
;
static PyObject *fbsdsum(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;

	if ((size=fbsdsum_L(args,&res))==NULL)
		return NULL;
	return Py_BuildValue("(l,N)", res,size);
}


static PyObject *fsysvsum_L(PyObject *args,uLong *res){
	char *fname;
	FILE *f;
	unsigned long size, size1;

	if ((f=sfopen(args,&fname))==NULL) return NULL;
	if (sysv_sum_stream(f,res,&size,&size1)){
		PyErr_SetFromErrnoWithFilename(PyExc_IOError,fname);
		fclose(f);
		return NULL;
	}
	fclose(f);
	return ululToL(size,size1);
}

static char fsysvsumt__doc__[] =
"filename -> checksum, size. Return the sysv-style sum as a dec string."
;
static PyObject *fsysvsumt(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;
	unsigned char out[6];

	if ((size=fsysvsum_L(args,&res))==NULL)
		return NULL;

	sprintf(out,"%lu",res);
	return Py_BuildValue("(s,N)", out, size);
}

static char fsysvsum__doc__[] =
"filename -> checksum, size. Return the sysv-style sum as an int."
;
static PyObject *fsysvsum(PyObject *self,PyObject *args){
	uLong res;
	PyObject *size;

	if ((size=fsysvsum_L(args,&res))==NULL)
		return NULL;
	return Py_BuildValue("(l,N)", res,size);
}


static char version__doc__[] =
"Return the interface version number.\n\
\n\
v1.0-1.1 had no version() function\n\
v1.2 returned 2\n\
v1.3-1.6 returns 3\n\
";
static PyObject *version(PyObject *self,PyObject *args){
	return Py_BuildValue("i", 3);
}


static PyMethodDef fchksumMethods[] = {
	{"fmd5",   fmd5,  METH_VARARGS, fmd5__doc__},
	{"fmd5t",  fmd5t, METH_VARARGS, fmd5t__doc__},
	{"fcrc32", fcrc32,  METH_VARARGS, fcrc32__doc__},
	{"fcrc32t", fcrc32t,  METH_VARARGS, fcrc32t__doc__},
	{"fcksum", fcksum,  METH_VARARGS, fcksum__doc__},
	{"fcksumt", fcksumt,  METH_VARARGS, fcksumt__doc__},
	{"fbsdsum", fbsdsum,  METH_VARARGS, fbsdsum__doc__},
	{"fbsdsumt", fbsdsumt,  METH_VARARGS, fbsdsumt__doc__},
	{"fsysvsum", fsysvsum,  METH_VARARGS, fsysvsum__doc__},
	{"fsysvsumt", fsysvsumt,  METH_VARARGS, fsysvsumt__doc__},
	{"version", version,  METH_VARARGS, version__doc__},
	/*	{"ululToL", p_ululToL,  METH_VARARGS},*/
	{NULL,      NULL}        /* Sentinel */
};


static char fchksum_module__doc__[] ="\
This module provides quick and easy functions to find checksums of files.\n\
It supports md5, crc32, cksum, bsd-style sum, and sysv-style sum.\n\
The advantage of using fchksum over the python md5 and zlib(.crc32)\n\
modules is both ease of use and speed.  You only need to tell it the\n\
filename and the actual work is done by C code.  Compared to the\n\
implementing a read loop in python with the standard python modules,\n\
fchksum is up to 2.0x faster in md5 and 1.1x faster in crc32.\n\
\n\
All checksum functions take a filename as a string, and return a tuple\n\
(checksum, size).  An empty string may be substituted for filename to\n\
read from stdin.  The returned size is always a python Long, and the\n\
checksum return type varies depending on the function.\n\
";

void
initfchksum(void)
{
	(void) Py_InitModule3("fchksum", fchksumMethods, fchksum_module__doc__);
}

