
"""
__version__ = "$Revision: 1.23 $"
__date__ = "$Date: 2004/03/17 23:26:10 $"

wxStyledTextCtrl documentation
http://wiki.wxpython.org/index.cgi/wxStyledTextCtrl

"""

from wxPython import wx, stc
import os, sys

from PythonCardPrototype import binding, config, event, STCStyleEditor, registry, widget

LINE_NUMBER_MARGIN = 1
LINE_NUMBER_MARGIN_WIDTH = 40
CODE_FOLDING_MARGIN = 2
CODE_FOLDING_MARGIN_WIDTH = 12

class CodeEditorSpec(widget.WidgetSpec):
    def __init__(self):
        widget.WidgetSpec.__init__(self)
        
        self.name = 'CodeEditor'
        self.parent = 'Widget'
        self.parentName = self.parent
        self.events.extend([event.KeyPressEvent, 
                event.KeyDownEvent, 
                event.KeyUpEvent, 
                #event.TextEnterEvent,
                #event.TextUpdateEvent,
                #event.CloseFieldEvent
        ])
        self._attributes.update({
            'text' : { 'presence' : 'optional', 'default' : '' },
            'editable' : { 'presence' : 'optional', 'default' : 1 },
            #'border' : { 'presence' : 'optional', 'default' : '3d', 'values' : [ '3d', 'none' ] }
        })
        
        self.attributes = self.parseAttributes(self._attributes)
        self.requiredAttributes = self.parseRequiredAttributes()
        self.optionalAttributes = self.parseOptionalAttributes()


# POB 2002-05-04
# Borrowed stuff from PyCrust just to get this working.
if wx.wxPlatform == '__WXMSW__':
    faces = { 'times'  : 'Times New Roman',
              'mono'   : 'Courier New',
              'helv'   : 'Lucida Console',
              'lucida' : 'Lucida Console',
              'other'  : 'Comic Sans MS',
              'size'   : 10,
              'lnsize' : 9,
              'backcol': '#FFFFFF',
            }
else:  # GTK
    faces = { 'times'  : 'Times',
              'mono'   : 'Courier',
              'helv'   : 'Helvetica',
              'other'  : 'new century schoolbook',
              'size'   : 12,
              'lnsize' : 10,
              'backcol': '#FFFFFF',
            }

class CodeEditor(widget.Widget, stc.wxStyledTextCtrl):
    """
    A Python source code editor.
    """

    _spec = CodeEditorSpec()

    def __init__( self, aParent,  aResource ) :
        #attributes = ['_border']
        #self._createAttributes(attributes)
        #self._border = aResource.border
        widget.Widget.__init__( self, aParent, aResource )

        stc.wxStyledTextCtrl.__init__(
            self,
            aParent, 
            self.getId(), 
            #aResource.text, 
            wx.wxPoint( aResource.position[ 0 ], aResource.position[ 1 ] ), 
            wx.wxSize( aResource.size[ 0 ], aResource.size[ 1 ] ), 
            #style = wxTE_PROCESS_ENTER | borderStyle | wxCLIP_SIBLINGS,
            #style = borderStyle | wx.wxCLIP_SIBLINGS,
            style = wx.wxNO_FULL_REPAINT_ON_RESIZE | wx.wxCLIP_SIBLINGS,
            name = aResource.name )

        # POB 2002-05-04
        # Borrowed stuff from PyCrust just to get this working.
        self.SetMarginType(LINE_NUMBER_MARGIN, stc.wxSTC_MARGIN_NUMBER)
        self.SetMarginWidth(LINE_NUMBER_MARGIN, LINE_NUMBER_MARGIN_WIDTH)
        self.SetLexer(stc.wxSTC_LEX_PYTHON)
        import keyword
        self.SetKeyWords(0, ' '.join(keyword.kwlist))
        self._setStyles(faces)
        self.SetViewWhiteSpace(0)
        self.SetTabWidth(4)
        self.SetUseTabs(0)
        self.CallTipSetBackground(wx.wxColour(255, 255, 232))
        self.SetBackSpaceUnIndents(1)

        stc.EVT_STC_UPDATEUI(self, self.getId(), self.OnUpdateUI)


        # KEA 2002-12-16
        # code folding code taken from
        # wxStyledTextCtrl_2.py
        self.SetProperty("fold", "1")
        # Setup a margin to hold fold markers
        self.SetFoldFlags(16)  ###  WHAT IS THIS VALUE?  WHAT ARE THE OTHER FLAGS?  DOES IT MATTER?
        self.SetMarginType(CODE_FOLDING_MARGIN, stc.wxSTC_MARGIN_SYMBOL)
        self.SetMarginMask(CODE_FOLDING_MARGIN, stc.wxSTC_MASK_FOLDERS)
        self.SetMarginSensitive(CODE_FOLDING_MARGIN, 1)
        self.SetMarginWidth(CODE_FOLDING_MARGIN, CODE_FOLDING_MARGIN_WIDTH)
        # initially code folding should be off
        self.SetMarginWidth(CODE_FOLDING_MARGIN, 0)
        
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDEREND,     stc.wxSTC_MARK_BOXPLUSCONNECTED,  "white", "black")
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDEROPENMID, stc.wxSTC_MARK_BOXMINUSCONNECTED, "white", "black")
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDERMIDTAIL, stc.wxSTC_MARK_TCORNER,  "white", "black")
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDERTAIL,    stc.wxSTC_MARK_LCORNER,  "white", "black")
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDERSUB,     stc.wxSTC_MARK_VLINE,    "white", "black")
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDER,        stc.wxSTC_MARK_BOXPLUS,  "white", "black")
        self.MarkerDefine(stc.wxSTC_MARKNUM_FOLDEROPEN,    stc.wxSTC_MARK_BOXMINUS, "white", "black")

        stc.EVT_STC_MARGINCLICK(self, self.getId(), self.OnMarginClick)

        if not hasattr(self.__class__, '_getText'):
            self.__class__._getText = self.__class__.GetText
        #if not hasattr(self.__class__, '_setText'):
        #    self.__class__._setText = self.__class__.SetText
        if not hasattr(self.__class__, 'ClearSelection'):
            self.__class__.ClearSelection = self.__class__.Clear

        # KEA 2002-05-03
        # GetReadOnly and SetReadOnly are the opposite of
        # IsEditable and SetEditable in wxTextCtrl
        # so the methods are wrapped below
        #if not hasattr(self.__class__, '_getEditable'):
        #    self.__class__._getEditable = self.__class__.GetReadOnly
        #if not hasattr(self.__class__, '_setEditable'):
        #    self.__class__._setEditable = self.__class__.SetReadOnly

        self._setEditable( aResource.editable )

        # then call Widget._postInit which were the initialization
        # steps done after the _delegate was created
        widget.Widget._postInit(self, aParent, aResource)
        
    # POB 2002-05-04
    # Borrowed stuff from PyCrust just to get this working.
    def _setStyles(self, faces):
        """Configure font size, typeface and color for lexer."""
        
        # Default style
        self.StyleSetSpec(stc.wxSTC_STYLE_DEFAULT, "face:%(mono)s,size:%(size)d" % faces)

        self.StyleClearAll()

        # Built in styles
        self.StyleSetSpec(stc.wxSTC_STYLE_LINENUMBER, "back:#C0C0C0,face:%(mono)s,size:%(lnsize)d" % faces)
        self.StyleSetSpec(stc.wxSTC_STYLE_CONTROLCHAR, "face:%(mono)s" % faces)
        self.StyleSetSpec(stc.wxSTC_STYLE_BRACELIGHT, "fore:#0000FF,back:#FFFF88")
        self.StyleSetSpec(stc.wxSTC_STYLE_BRACEBAD, "fore:#FF0000,back:#FFFF88")

        # Python styles
        self.StyleSetSpec(stc.wxSTC_P_DEFAULT, "face:%(mono)s" % faces)
        self.StyleSetSpec(stc.wxSTC_P_COMMENTLINE, "fore:#007F00,face:%(mono)s" % faces)
        self.StyleSetSpec(stc.wxSTC_P_NUMBER, "")
        self.StyleSetSpec(stc.wxSTC_P_STRING, "fore:#7F007F,face:%(mono)s" % faces)
        self.StyleSetSpec(stc.wxSTC_P_CHARACTER, "fore:#7F007F,face:%(mono)s" % faces)
        self.StyleSetSpec(stc.wxSTC_P_WORD, "fore:#00007F,bold")
        self.StyleSetSpec(stc.wxSTC_P_TRIPLE, "fore:#7F0000")
        self.StyleSetSpec(stc.wxSTC_P_TRIPLEDOUBLE, "fore:#000033,back:#FFFFE8")
        self.StyleSetSpec(stc.wxSTC_P_CLASSNAME, "fore:#0000FF,bold")
        self.StyleSetSpec(stc.wxSTC_P_DEFNAME, "fore:#007F7F,bold")
        self.StyleSetSpec(stc.wxSTC_P_OPERATOR, "")
        self.StyleSetSpec(stc.wxSTC_P_IDENTIFIER, "")
        self.StyleSetSpec(stc.wxSTC_P_COMMENTBLOCK, "fore:#7F7F7F")
        self.StyleSetSpec(stc.wxSTC_P_STRINGEOL, "fore:#000000,face:%(mono)s,back:#E0C0E0,eolfilled" % faces)

        # almost the same as code in debug.py
        # and codeEditor sample
        # probably should have a util routine for setting
        # the style info
        configfile = os.path.join(config.homedir, 'stc-styles.rc.cfg')
        if not os.path.exists(configfile):
            configfile = os.path.join(config.default_homedir, 'stc-styles.rc.cfg')
        if os.path.exists(configfile):
            STCStyleEditor.initSTC(self, configfile, 'python')

    def _bindEvents(self):
        adapter = CodeEditorEventBinding(self)
        adapter.bindEvents()
        #pass

    def _getLineNumbersVisible(self):
        return self.GetMarginWidth(LINE_NUMBER_MARGIN) / LINE_NUMBER_MARGIN_WIDTH

    def _setLineNumbersVisible(self, aBoolean):
        self.SetMarginWidth(LINE_NUMBER_MARGIN, LINE_NUMBER_MARGIN_WIDTH * int(aBoolean))

    def _getCodeFoldingVisible(self):
        return self.GetMarginWidth(CODE_FOLDING_MARGIN) / CODE_FOLDING_MARGIN_WIDTH

    def _setCodeFoldingVisible(self, aBoolean):
        if not aBoolean:
            self.FoldAll(toggle=0)
        self.SetMarginWidth(CODE_FOLDING_MARGIN, CODE_FOLDING_MARGIN_WIDTH * int(aBoolean))

    def _getEditable(self):
        return not self.GetReadOnly()

    def _setEditable(self, aBoolean):
        self.SetReadOnly(not aBoolean)

    def _setText(self, text):
        if self.GetReadOnly():
            self.SetReadOnly(0)
            self.SetText(text)
            self.SetReadOnly(1)
        else:
            self.SetText(text)

    # KEA 2002-05-20
    # methods added to mimic wxTextCtrl

    def CanCopy(self):
        return 1

    def CanCut(self):
        return 1

    # KEA 2002-05-06
    # taken from wxPython\lib\pyshell.py
    def OnUpdateUI(self, evt):
        #document = self.components.document
        # check for matching braces
        braceAtCaret = -1
        braceOpposite = -1
        charBefore = None
        caretPos = self.GetCurrentPos()
        if caretPos > 0:
            charBefore = self.GetCharAt(caretPos - 1)
            # KEA 2002-05-29
            # see message from Jean-Michel Fauth 
            # http://aspn.activestate.com/ASPN/Mail/Message/wxPython-users/1218004
            #******
            if charBefore < 0:
                charBefore = 32 #mimic a space
            #******
            styleBefore = self.GetStyleAt(caretPos - 1)

        # check before
        if charBefore and chr(charBefore) in "[]{}()" and styleBefore == stc.wxSTC_P_OPERATOR:
            braceAtCaret = caretPos - 1

        # check after
        if braceAtCaret < 0:
            charAfter = self.GetCharAt(caretPos)
            # KEA 2002-05-29
            # see message from Jean-Michel Fauth 
            # http://aspn.activestate.com/ASPN/Mail/Message/wxPython-users/1218004
            #******
            if charAfter < 0:
                charAfter = 32 #mimic a space
            #******
            styleAfter = self.GetStyleAt(caretPos)
            if charAfter and chr(charAfter) in "[]{}()" and styleAfter == stc.wxSTC_P_OPERATOR:
                braceAtCaret = caretPos

        if braceAtCaret >= 0:
            braceOpposite = self.BraceMatch(braceAtCaret)

        if braceAtCaret != -1  and braceOpposite == -1:
            self.BraceBadLight(braceAtCaret)
        else:
            self.BraceHighlight(braceAtCaret, braceOpposite)

    # KEA 2002-12-16
    # code folding code taken from
    # wxStyledTextCtrl_2.py
    def OnMarginClick(self, evt):
        # fold and unfold as needed
        if evt.GetMargin() == 2:
            if evt.GetShift() and evt.GetControl():
                self.FoldAll()
            else:
                lineClicked = self.LineFromPosition(evt.GetPosition())
                if self.GetFoldLevel(lineClicked) & stc.wxSTC_FOLDLEVELHEADERFLAG:
                    if evt.GetShift():
                        self.SetFoldExpanded(lineClicked, 1)
                        self.Expand(lineClicked, 1, 1, 1)
                    elif evt.GetControl():
                        if self.GetFoldExpanded(lineClicked):
                            self.SetFoldExpanded(lineClicked, 0)
                            self.Expand(lineClicked, 0, 1, 0)
                        else:
                            self.SetFoldExpanded(lineClicked, 1)
                            self.Expand(lineClicked, 1, 1, 100)
                    else:
                        self.ToggleFold(lineClicked)


    def FoldAll(self, toggle=1):
        lineCount = self.GetLineCount()
        expanding = 1

        if toggle:
            # find out if we are folding or unfolding
            for lineNum in range(lineCount):
                if self.GetFoldLevel(lineNum) & stc.wxSTC_FOLDLEVELHEADERFLAG:
                    expanding = not self.GetFoldExpanded(lineNum)
                    break;

        lineNum = 0
        while lineNum < lineCount:
            level = self.GetFoldLevel(lineNum)
            if level & stc.wxSTC_FOLDLEVELHEADERFLAG and \
               (level & stc.wxSTC_FOLDLEVELNUMBERMASK) == stc.wxSTC_FOLDLEVELBASE:

                if expanding:
                    self.SetFoldExpanded(lineNum, 1)
                    lineNum = self.Expand(lineNum, 1)
                    lineNum = lineNum - 1
                else:
                    lastChild = self.GetLastChild(lineNum, -1)
                    self.SetFoldExpanded(lineNum, 0)
                    if lastChild > lineNum:
                        self.HideLines(lineNum+1, lastChild)

            lineNum = lineNum + 1



    def Expand(self, line, doExpand, force=0, visLevels=0, level=-1):
        lastChild = self.GetLastChild(line, level)
        line = line + 1
        while line <= lastChild:
            if force:
                if visLevels > 0:
                    self.ShowLines(line, line)
                else:
                    self.HideLines(line, line)
            else:
                if doExpand:
                    self.ShowLines(line, line)

            if level == -1:
                level = self.GetFoldLevel(line)

            if level & stc.wxSTC_FOLDLEVELHEADERFLAG:
                if force:
                    if visLevels > 1:
                        self.SetFoldExpanded(line, 1)
                    else:
                        self.SetFoldExpanded(line, 0)
                    line = self.Expand(line, doExpand, force, visLevels-1)

                else:
                    if doExpand and self.GetFoldExpanded(line):
                        line = self.Expand(line, 1, force, visLevels-1)
                    else:
                        line = self.Expand(line, 0, force, visLevels-1)
            else:
                line = line + 1;

        return line

    def getStyleConfigPath(self):
        configfile = os.path.join(config.homedir, 'stc-styles.rc.cfg')
        if not os.path.exists(configfile):
            configfile = os.path.join(config.default_homedir, 'stc-styles.rc.cfg')
        return configfile

    def setEditorStyle(self, ext=None):
        config = self.getStyleConfigPath()
        if config is not None:
            if ext is not None:
                ext = ext.lower().split('.')[-1]
            if ext in ['py', 'pyw']:
                style = 'python'
            elif ext in ['ini', 'text', 'txt']:
                style = 'text'
            elif ext in ['htm', 'html']:
                # 'html' style appears broken, just use xml for now
                style = 'xml'
            elif ext in ['rdf', 'xml']:
                style = 'xml'
            else:
                style = 'text'
            STCStyleEditor.initSTC(self, config, style)


# this is a copy of TextFieldEventBinding
# not all the events below are valid, but key presses do seem to
# be working
class CodeEditorEventBinding( binding.wxPython_EventBinding ) :
    """
    Bind the Events supported by event.TextField to wxPython.
    """
    def __init__( self, aComponent ) :

        binding.wxPython_EventBinding.__init__( self, aComponent )

    def bindEvent( self, aEventClass ) :

        parent = self._component._parent

        # KEA 2001-10-05
        # I don't think this event is used in TextField ?!
        #if aEventClass is TextEnterEvent :
        #    EVT_TEXT_ENTER( parent, self._component.getId(), self._dispatch )

        if aEventClass is event.KeyDownEvent :
            wx.EVT_KEY_DOWN( self._component, self._dispatch )

        if aEventClass is event.KeyUpEvent :
            wx.EVT_KEY_UP( self._component, self._dispatch )

        if aEventClass is event.KeyPressEvent :
            wx.EVT_CHAR( self._component, self._dispatch )

    def _dispatch(self, aWxEvent):
        # Call our superclass to dispatch the standard mouse
        # events that every widget should get.
        if binding.wxPython_EventBinding._dispatch(self, aWxEvent):
            return
        
        evt = None

        if aWxEvent.GetEventType() == event.wxEVT_CLOSE_FIELD:
            #print 'handling closeField'
            evt = self._createEvent(event.CloseFieldEvent, aWxEvent)

        if aWxEvent.GetEventType() == wx.wxEVT_COMMAND_TEXT_UPDATED :
            evt = self._createEvent( event.TextUpdateEvent, aWxEvent )

        #if aWxEvent.GetEventType() == wxEVT_COMMAND_TEXT_ENTER :
        #    event = self._createEvent( TextEnterEvent, aWxEvent )
            # this should be associated with losing focus, not textEnter
            #self._component._notifyEventListeners( CloseField( self._component ) )

        if aWxEvent.GetEventType() == wx.wxEVT_KEY_DOWN :
            #print '_dispatch wxEVT_KEY_DOWN'
            #evt = KeyDownEvent( self._component )
            evt = self._createEvent( event.KeyDownEvent, aWxEvent )

        if aWxEvent.GetEventType() == wx.wxEVT_KEY_UP :
            #print '_dispatch wxEVT_KEY_UP'
            #evt = KeyUpEvent( self._component )
            evt = self._createEvent( event.KeyUpEvent, aWxEvent )

        if aWxEvent.GetEventType() == wx.wxEVT_CHAR :
            #print '_dispatch wxEVT_CHAR'
            #evt = KeyPressEvent( self._component )
            evt = self._createEvent( event.KeyPressEvent, aWxEvent )

        if evt is not None :
            #print '_dispatch evt is not None'
            self._component._notifyEventListeners( evt )
            #print "after notify"
            
            # KEA 2001-10-05
            # there needs to be an Event.Skip() somewhere if the event was not handled
            # by user code
            # however, skip should normally only be called if the user code didn't process
            # the event, which will allow the user code to "eat" the event, suppressing
            # and/or changing the actions of certain keys
            # need to make a complete list of which events can be eaten and which can't
            if not evt.getUsed():
                aWxEvent.Skip()
                # KEA 2001-10-26
                # if textEnter wasn't handled then focus should move to the next
                # control in the panel, just like pressing tab



registry.getRegistry().register( sys.modules[__name__].CodeEditor )
