/****************************************************************************
 **
 ** Copyright (C) 2008-2009 Grigory A. Mozhaev.  All rights reserved.
 **
 ** This file may be used under the terms of the GNU General Public
 ** License version 2.0 as published by the Free Software Foundation
 ** and appearing in the file LICENSE.GPL included in the packaging of
 ** this file.  Please review the following information to ensure GNU
 ** General Public Licensing requirements will be met:
 ** http://www.trolltech.com/products/qt/opensource.html
 **
 ** If you are unsure which license is appropriate for your use, please
 ** review the following information:
 ** http://www.trolltech.com/products/qt/licensing.html or contact the
 ** sales department at sales@trolltech.com.
 **
 ** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 ** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 **
 ** AUTHOR: Grigory A. Mozhaev <zcrendel@mail.ru>
 **
 ****************************************************************************/
//! -*- coding: UTF-8 -*-
#define SOURCE_CODING "UTF-8"

#include "cdvdmodel.h"

class CBurnThreadPrivate {
public:
    QString mISOFileName;

    QString mDevice;
    int mSpeed;

    bool mInitialized;
    bool mEject;

    bool mSuccess;

    CDvdModel *mpModel;

    bool mpExit;

    CBurnThreadPrivate() {
        mISOFileName = QString::null;
        mSpeed = 4;
        mInitialized = false;
        mSuccess = false;

        mpModel = 0;
        mpExit = false;
    }

    ~CBurnThreadPrivate() {

    }
};

CBurnThread::CBurnThread() {
    mpData = new CBurnThreadPrivate;

    Q_ASSERT(mpData!=0);    
}

CBurnThread::~CBurnThread() {
    if (mpData)
        delete mpData;
}

void CBurnThread::init(CDvdModel *apModel) {
    if (!apModel)
        return ;

    if (apModel->isoFileName().isNull())
        return ;

    if (apModel->device().isNull())
        return ;

    if ((apModel->speed() < 0) || ((apModel->speed() % 2) != 0))
        return ;

    if (mpData) {
        mpData->mpModel = apModel;

        mpData->mISOFileName = apModel->isoFileName();
        mpData->mDevice = apModel->device();
        mpData->mSpeed = apModel->speed();
        mpData->mInitialized = true;
    }
}

void CBurnThread::stop() {
    if (mpData)
        mpData->mpExit = true;
}

bool CBurnThread::done() {
    return mpData->mSuccess;
}


// %%%%%%%%%%%%%%%%%%%%%%%% CBurnCDThread %%%%%%%%%%%%%%%%%%%%%% //
CBurnCDThread::CBurnCDThread() :
    CBurnThread() {
    
}

CBurnCDThread::~CBurnCDThread() {
    
}

void CBurnCDThread::run() {
    Q_ASSERT(mpData!=0);

    if (!mpData->mInitialized) {
        mpData->mSuccess = false;
        return ;
    }
    
    QString vProgram = "cdrecord";
    QStringList vArguments;

    vArguments << QLatin1String("-v")
               << QString("dev=%1").arg(mpData->mDevice)
               << QLatin1String("fs=16m")
               << QString("speed=%1").arg(mpData->mSpeed)
               << QDir::convertSeparators(mpData->mISOFileName);

    QProcess vProcess;

    vProcess.setReadChannel(QProcess::StandardOutput);
    vProcess.start(vProgram, vArguments);

    if (!vProcess.waitForStarted()) {
        return ;
    }
    
    while ((!vProcess.waitForFinished(500)) &&
           (vProcess.state()==QProcess::Running)) {

        if (mpData->mpExit) {

            vProcess.terminate();
            mpData->mSuccess = false;
        }

        QString vString = vProcess.readAll();

        if (!vString.isNull()) {
            QStringList vList = vString.split("\n",QString::SkipEmptyParts);
            QRegExp vRegExp(QLatin1String("Track.*:.*of.*MB.*written.*\\(fifo.*\\).*\\[buf.*\\].*x."));
            if (vList.size() > 0) {
                if (vList.last().indexOf(vRegExp, 0) >= 0) {
                    QString vParse = vList.last();               
                    QRegExp vRemoveRegExp(QLatin1String("Track.*:"));
                    vParse.remove(vRemoveRegExp);
                    vRemoveRegExp.setPattern(QLatin1String("MB.*written.*\\(fifo.*\\).*\\[buf.*\\]"));
                    vParse.remove(vRemoveRegExp);
                    vParse.remove(QLatin1String("of"));
                    QStringList vParams = vParse.split(" ",QString::SkipEmptyParts);
                    if (vParams.size() >= 3) {
                        int vPercent = (vParams.at(vParams.count()-2).toInt()*1024) / 100;
                        if (vPercent) {
                            int vCurrent = (vParams.at(vParams.count()-3).toInt()*1024) / vPercent;
                            emit(progress(vCurrent, 
                                          QString("Average speed %1").arg(vParams.at(vParams.count()-1))));
                        }
                    }
            } else 
                if (vList.last().indexOf("Fixating...") >= 0) {
                    emit(progress(100, QLatin1String("Fixating...")));
                }
            }
        }
    }

    bool vFlag = true;
    if (vProcess.exitStatus() == QProcess::CrashExit)
        vFlag = false; 
    else if (vProcess.exitCode() != 0) 
        vFlag = false;

    vProgram = QLatin1String("eject");
    vArguments.clear();
    vArguments << mpData->mDevice;

    vProcess.start(vProgram, vArguments);
    if (vProcess.waitForStarted()) {
        while ((!vProcess.waitForFinished(1000)) &&
               (vProcess.state()==QProcess::Running)) {       
        }
    }

    if (!mpData->mpExit) {
        mpData->mSuccess = vFlag;
    }

    return ;
}

// %%%%%%%%%%%%%%%%%%%%%%%% CBurnDVDThread %%%%%%%%%%%%%%%%%%%%%% //
CBurnDVDThread::CBurnDVDThread()
  : CBurnThread() {
    //
}

CBurnDVDThread::~CBurnDVDThread() {
    //
}
    
void CBurnDVDThread::run() {
    Q_ASSERT(mpData!=0);

    if (!mpData->mInitialized) {
        mpData->mSuccess = false;
        return ;
    }
   
    QString vProgram = "growisofs";
    QStringList vArguments;

    vArguments << QLatin1String("-dvd-compat")
               << QString("-Z %1=%2").arg(mpData->mDevice)
                                     .arg(QDir::convertSeparators(mpData->mISOFileName))
               << QString("-speed=%1").arg(mpData->mSpeed);

    QProcess vProcess;

    vProcess.setReadChannel(QProcess::StandardOutput);
    vProcess.start(vProgram, vArguments);

    if (!vProcess.waitForStarted()) {
        return ;
    }
    
    while ((!vProcess.waitForFinished(500)) &&
           (vProcess.state()==QProcess::Running)) {

        if (mpData->mpExit) {
            vProcess.terminate();
            mpData->mSuccess = false;
        }

        QString vString = vProcess.readAll();
        if (!vString.isNull()) {            
            QStringList vList = vString.split("\n",QString::SkipEmptyParts);
            // 36306944/4668260352 ( 0.8%) @4.0x, remaining 61:39 RBU 100.0% UBU 100.0%
            // 36306944/4668260352 @4.0x
            QRegExp vRegExp(QLatin1String(".*\\/.*\\(.*\\).*remaining.*RBU.*UBU.*\\%"));
            if (vList.size() > 0) {
                if (vList.last().indexOf(vRegExp, 0) >= 0) {
                    QString vParse = vList.last();               
                    QRegExp vRemoveRegExp(QLatin1String("\\(.*\\)"));
                    vParse.remove(vRemoveRegExp);
                    vRemoveRegExp.setPattern(QLatin1String("\\,.*"));
                    vParse.remove(vRemoveRegExp);
                    vParse.replace("/"," ").replace("@","");
                    qDebug() << vParse;
                    QStringList vParams = vParse.split(" ",QString::SkipEmptyParts);
                    if (vParams.size() >= 3) {
                        quint64 vPercent = vParams.at(vParams.count()-2).toULongLong() / 100;
                        if (vPercent) {
                            int vCurrent = vParams.at(vParams.count()-3).toULongLong() / vPercent;
                            emit(progress(vCurrent, 
                                          QString("Average speed %1").arg(vParams.at(vParams.count()-1))));
                        }
                    }
                } else {
                    vRegExp.setPattern(QLatin1String(".*: flushing cache"));
                    if (vList.last().indexOf(vRegExp,0) >= 0) {
                        emit(progress(100, QLatin1String("flushing cache...")));
                    }
                    
                    vRegExp.setPattern(QLatin1String(".*: closing track"));
                    if (vList.last().indexOf(vRegExp,0) >= 0) {
                        emit(progress(100, QLatin1String("closing track...")));
                    }
                    
                    vRegExp.setPattern(QLatin1String(".*: closing disc"));
                    if (vList.last().indexOf(vRegExp,0) >= 0) {
                        emit(progress(100, QLatin1String("closing disc...")));
                    }
                }
            }
        }
    }

    bool vFlag = true;
    if (vProcess.exitStatus() == QProcess::CrashExit)
        vFlag = false; 
    else if (vProcess.exitCode() != 0) 
        vFlag = false;

    vProgram = QLatin1String("eject");
    vArguments.clear();
    vArguments << mpData->mDevice;

    vProcess.start(vProgram, vArguments);
    if (vProcess.waitForStarted()) {
        while ((!vProcess.waitForFinished(500)) &&
               (vProcess.state()==QProcess::Running)) {       
        }
    }

    if (!mpData->mpExit) {
        mpData->mSuccess = vFlag;
    }
  
    return ;
}
