/****************************************************************************
 **
 ** Copyright (C) 2008-2009 Grigory A. Mozhaev.  All rights reserved.
 **
 ** This file may be used under the terms of the GNU General Public
 ** License version 2.0 as published by the Free Software Foundation
 ** and appearing in the file LICENSE.GPL included in the packaging of
 ** this file.  Please review the following information to ensure GNU
 ** General Public Licensing requirements will be met:
 ** http://www.trolltech.com/products/qt/opensource.html
 **
 ** If you are unsure which license is appropriate for your use, please
 ** review the following information:
 ** http://www.trolltech.com/products/qt/licensing.html or contact the
 ** sales department at sales@trolltech.com.
 **
 ** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 ** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 **
 ** AUTHOR: Grigory A. Mozhaev <zcrendel@mail.ru>
 **
 ****************************************************************************/
//! -*- coding: UTF-8 -*-
#define SOURCE_CODING "UTF-8"

#include "cdvdmodel.h"

#include <QtCore/QCryptographicHash>

class CCheckThreadPrivate {
public:
    QString mDevice;
    bool    mSuccess;
    bool    mInitialized;
    QCryptographicHash *mpHash;
    quint64 mPercent;

    CDvdModel *mpModel;
    bool mpExit;

    CCheckThreadPrivate() {
        mDevice = QString::null;
        mInitialized = false;        
        mpHash = new QCryptographicHash(QCryptographicHash::Md5);
        mSuccess = false;
        
        mpModel = 0;
        mpExit  = false;
    }
    ~CCheckThreadPrivate() {
        if (mpHash) {
            delete mpHash;
            mpHash = 0;
        }
    }
};

CCheckThread::CCheckThread() {
    mpData = new CCheckThreadPrivate;

    Q_ASSERT(mpData!=0);    
}
 
CCheckThread::~CCheckThread() {
    if (mpData) {
        delete mpData;
        mpData = 0;
    }
}


void 
CCheckThread::init(CDvdModel *apModel) {
    if (!apModel)
        return ;

    if (apModel->device().isNull())
        return ;

    if (mpData) {
        mpData->mpModel = apModel;

        mpData->mDevice = apModel->device();
        mpData->mPercent = apModel->capacity() / 100;

        mpData->mInitialized = true;
    }
}
    
void CCheckThread::stop() {
    if (mpData)
        mpData->mpExit = true;
}

void 
CCheckThread::run() {
    if (!mpData->mInitialized) {
        mpData->mSuccess = false;
        return;
    }

    sleep(2);

    QString vProgram;
    QStringList vArguments;
    QProcess vProcess;

    vProgram = QLatin1String("eject");
    vArguments << mpData->mDevice;
    vArguments << QLatin1String("-t");
    vProcess.start(vProgram, vArguments);
    if (vProcess.waitForStarted()) {
        while ((!vProcess.waitForFinished(1000)) &&
               (vProcess.state()==QProcess::Running)) {       
        }
    }

    int aCounter = 0;
    while (true) {
        if (mpData->mpModel->dvdStatus() == CDvdModel::dsDataCd)
            break;
        
        if (mpData->mpModel->dvdStatus() == CDvdModel::dsDataDvd)
            break;

        if (aCounter >= 25) {
            mpData->mSuccess = false;
            emit(progress(100));
            return ;
        }

        aCounter++;
        sleep(1);
    }

    //mpData->mpHash->reset();
    progress(0);

    QFile vFile(mpData->mDevice);
    if (!vFile.open(QIODevice::ReadOnly)) {
        mpData->mSuccess = false;
        emit(progress(100));
        return;
    }

    quint64 vRed = 0;
    quint64 vISOSize = mpData->mpModel->isoSize();
    int vTryCount = 0;
    bool vError = false;

    qDebug() << "ISO file size to check:" << vISOSize;

    while (!vFile.atEnd()) {
        if (mpData->mpExit) {
            mpData->mSuccess = false;
            break;
        }

        char buffer[4096];
        qint64 vRetVal = vFile.read(buffer, 4096);
        if (vRetVal > 0) {
            vTryCount=0;
            vRed += vRetVal;

            if (vISOSize > 0) {
                if (vISOSize == vRed) {
                    qDebug() << "Read from media equal ISO file size - quit from thread";
                    break;
                } else
                if (vRed > vISOSize) {
                    vRetVal -= (vRed - vISOSize);
                    mpData->mpHash->addData(buffer, vRetVal);
                    
                    qDebug() << "Media Size is Greater than ISO size!";
                    qDebug() << "Read from media:" << vRed;
                    qDebug() << "ISO file size:" << vISOSize;

                    break;
                }
            }

            mpData->mpHash->addData(buffer, vRetVal);
            emit(progress(vRed / mpData->mPercent));
        } else 
        if (vRetVal < 0) {
            sleep(1);
            vTryCount++;
            if (vTryCount > 2) {
                vError = true;
                qDebug() << "Error: Can't read from device";
                qDebug() << "Current md5sum is:" << 
                    mpData->mpHash->result().toHex();
                break;
            }
        } else {
            break;
        }       
    }
    vFile.close();
    emit(progress(100));

    qDebug() << "Check Sum Ended";

    qApp->processEvents();

    vProgram = QLatin1String("eject");
    vArguments.clear();
    vArguments << mpData->mDevice;

    vProcess.start(vProgram, vArguments);
    if (vProcess.waitForStarted()) {
        while ((!vProcess.waitForFinished(500)) &&
               (vProcess.state()==QProcess::Running)) {
        }
    }

    if ((!mpData->mpExit) && (!vError)) {
        mpData->mSuccess = true;
    } else {
        mpData->mSuccess = false;
    }
    
    qDebug() << "CD ejected";
}

bool CCheckThread::done() {
    return mpData->mSuccess;
}

QString CCheckThread::hash() {
    if (mpData) {
        if (mpData->mpHash) {
            return mpData->mpHash->result().toHex();
        }
    }

    return QString();
}
