/****************************************************************************
 **
 ** Author: Grigory A. Mozhaev
 ** Copyright (C) 2008-2009 Griland Company.
 ** Contact: Griland Company (griland.support@gmail.com)
 **
 ** This file may be used under the terms of the GNU General Public
 ** License version 2.0 as published by the Free Software Foundation
 ** and appearing in the file LICENSE.GPL included in the packaging of
 ** this file.  Please review the following information to ensure GNU
 ** General Public Licensing requirements will be met:
 ** http://www.trolltech.com/products/qt/opensource.html
 **
 ** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 ** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 **
 ****************************************************************************/
//! -*- coding: UTF-8 -*-
#define SOURCE_CODING "UTF-8"

#include <Qt/QtCore>
#include <Qt/QtDBus>
#include <QtDBus/QDBusInterface>

#include "cdvdmodel.h"

#define HAL_SERVICE         QLatin1String("org.freedesktop.Hal")
#define HAL_PATH_MANAGER    QLatin1String("/org/freedesktop/Hal/Manager")
#define HAL_PATH_DEVICE     QLatin1String("/org/freedesktop/Hal/Device")
#define HAL_OBJECT_MANAGER  QLatin1String("org.freedesktop.Hal.Manager")
#define HAL_OBJECT_DEVICE   QLatin1String("org.freedesktop.Hal.Device")

#define DBUS_SERVICE        QLatin1String("Q_MULTI_RECORD")

//!
class CDvdModelPrivate {
public:
    QString mDevice;
    QString mDeviceInfo;
    QString mDeviceHalVolume;

    QString mHalVolume;

    CDvdModel::DeviceType  mDeviceType;
    CDvdModel::DvdStatus   mStatus;    
    CDvdModel::ProcessMode mProcessMode;    

    quint64 mCapacity;
    
    // minor
    CBurnThread *mpBurnThread;

    CCheckThread *mpCheckThread;
    QString mHash;

    // very minor
    int mSpeed;
    QString mISOFileName;
    quint64 mISOSize;

    CDvdModelPrivate() {
        mCounter++;
        mDevice  = QString::null;
        mDeviceInfo = QString::null;
        mDeviceHalVolume = QString::null;
        mHalVolume  = QString::null;
        mStatus  = CDvdModel::dsUndefined;
        mDeviceType = CDvdModel::dtUndefined;
        mProcessMode = CDvdModel::pmNone;
        mpBurnThread = 0;
        mpCheckThread = 0;
        mHash = QString::null;

        mSpeed = -1;
        mISOFileName = QString::null;
        mISOSize = 0;
    }

    ~CDvdModelPrivate() {
    }

    static int mCounter;
    static QDBusConnection mDbusConnection;
};

int CDvdModelPrivate::mCounter = 0;
QDBusConnection  CDvdModelPrivate::mDbusConnection = QDBusConnection(DBUS_SERVICE);

//!
CDvdModel::CDvdModel() : mpData(0) {
    if (!mpData) {
        mpData = new CDvdModelPrivate;
        if (mpData) {
            mpData->mDbusConnection = QDBusConnection::connectToBus(QDBusConnection::SystemBus,
                                                                    DBUS_SERVICE);
        }
    }

    Q_ASSERT(mpData!=0);

    if (mpData->mDbusConnection.isConnected()) {
        mpData->mDbusConnection.connect( HAL_SERVICE, HAL_PATH_MANAGER,
                                         HAL_OBJECT_MANAGER, "DeviceAdded",
                                         this, SLOT( addVolume ( QString )));
        mpData->mDbusConnection.connect( HAL_SERVICE, HAL_PATH_MANAGER,
                                         HAL_OBJECT_MANAGER, "DeviceRemoved",
                                         this, SLOT( removeVolume ( QString )));
    }
}

//!
CDvdModel::~CDvdModel() {
    if (mpData) {
        mpData->mCounter--;
        if (mpData->mCounter == 0) {
            mpData->mDbusConnection.disconnectFromBus(DBUS_SERVICE);
        }
        delete mpData;
        mpData = 0;
    }
}

//!
bool isDvd(QDBusInterface &aDBusInterface) {
    QDBusReply<bool> vReply = aDBusInterface.call( "GetProperty", "volume.is_disc");
    if (vReply.value()) {
        QDBusReply<QString> vDiscType = aDBusInterface.call( "GetProperty", "volume.disc.type" );
        if (vDiscType.value().indexOf(QRegExp(".*dvd.*"), 0) >= 0) return true;
    }

    return false;
}

//!
bool isCdrom(QDBusInterface &aDBusInterface) {
    QDBusReply<bool> vReply = aDBusInterface.call( "GetProperty", "volume.is_disc");
    if ((vReply.isValid()) && (vReply.value())) {
        QDBusReply<QString> vDiscType = aDBusInterface.call( "GetProperty", "volume.disc.type" );
        if (vDiscType.value().indexOf(QRegExp(".*cd.*"),0) >= 0) return true;
    }

    return false;
}

//!
bool isDeviceCdrom(QDBusInterface &aDBusInterface) {
    QDBusReply<bool> vReply = aDBusInterface.call( "GetProperty", "storage.cdrom.cdr");
    if ((vReply.isValid()) && (vReply.value()))
        return true;

    return false;
}

//!
bool isDeviceDvdrom(QDBusInterface &aDBusInterface) {
    QDBusReply<bool> vReply = aDBusInterface.call( "GetProperty", "storage.cdrom.dvd");
    if ((vReply.isValid()) && (vReply.value()))
        return true;

    return false;
}

//!
bool isEmpty(QDBusInterface &aDBusInterface) {
    QDBusReply<bool> vReply = aDBusInterface.call( "GetProperty", "volume.disc.is_blank");
    if ((vReply.isValid()) && (vReply.value())) {
        vReply = aDBusInterface.call( "GetProperty", "volume.disc.has_data");
        if (!vReply.value()) 
            return true;
    }

    return false;
}

//!
quint64 capacity(QDBusInterface &aDBusInterface) {
    QDBusReply<quint64> vReply = aDBusInterface.call("GetProperty", "volume.disc.capacity");
    if (vReply.isValid()) {        
        return vReply.value();
    }
    return quint64(0);
}

//!
QMap<QString,QString> CDvdModel::allDevices() {
    QMap<QString,QString> vDevicesMap;

    QDBusInterface vDBusInterface( HAL_SERVICE, HAL_PATH_MANAGER, HAL_OBJECT_MANAGER,
                                   QDBusConnection::systemBus());

    QDBusReply<QStringList> vReply = vDBusInterface.call( "FindDeviceStringMatch",
                                                          "storage.drive_type",
                                                          "cdrom");
    if (vReply.isValid()) {
        foreach (QString vDevice, vReply.value()) {
            QDBusInterface vDBusDeviceInterface( HAL_SERVICE, 
                                           vDevice,
                                           HAL_OBJECT_DEVICE,
                                           QDBusConnection::systemBus() );

            if (isDeviceCdrom(vDBusDeviceInterface) || 
                isDeviceDvdrom(vDBusDeviceInterface)) {
                // vendor
                QDBusReply<QString> vDeviceName = vDBusDeviceInterface.call( "GetProperty", "block.device");
                if (vDeviceName.isValid()) {

                    QString vDeviceInfo;
                    // vendor
                    QDBusReply<QString> vVendor = vDBusDeviceInterface.call( "GetProperty", "storage.vendor");
                    if (vVendor.isValid())
                        vDeviceInfo += vVendor + "_";

                    // product
                    QDBusReply<QString> vProduct = vDBusDeviceInterface.call( "GetProperty", "info.product");
                    if (vProduct.isValid())
                        vDeviceInfo += vProduct;

                    vDevicesMap.insert(vDeviceName, vDeviceInfo);
                }
            }
        }
    }

    return vDevicesMap;
}

//!
void CDvdModel::addVolume(const QString &aVolume) {
    checkVolume(aVolume);
}

//!
void CDvdModel::checkVolume(const QString &aVolume) {
    if (mpData) {
        QDBusInterface vDBusInterface( HAL_SERVICE, aVolume, HAL_OBJECT_DEVICE,
                                       QDBusConnection::systemBus() );

        QDBusReply<QString> vBlock = vDBusInterface.call( "GetProperty", "block.device");
        if (vBlock.isValid() && (vBlock.value().compare(mpData->mDevice) == 0) &&
            (mpData->mDeviceType == CDvdModel::dtUndefined)) {
            QDBusReply<QString> vReply = vDBusInterface.call( "GetProperty", "block.storage_device");

            if (vReply.isValid()) {
                mpData->mDeviceHalVolume = vReply.value();

                QDBusInterface vDBusDvdromInterface( HAL_SERVICE, mpData->mDeviceHalVolume,
                                                     HAL_OBJECT_DEVICE,
                                                     QDBusConnection::systemBus() );

                bool vIsCdromDev = isDeviceCdrom(vDBusDvdromInterface);
                bool vIsDvdromDev = isDeviceDvdrom(vDBusDvdromInterface);

                if (vIsCdromDev || vIsDvdromDev) {
                    if (vIsCdromDev)  mpData->mDeviceType = CDvdModel::dtCdWriter;
                    if (vIsDvdromDev) mpData->mDeviceType = CDvdModel::dtDvdWriter;
                    mpData->mStatus = CDvdModel::dsNoDisc;
                    mpData->mCapacity = 0;
                } else {
                    mpData->mDeviceType = CDvdModel::dtUnsupported;
                }
            }
        }

        if ((mpData->mDeviceType == CDvdModel::dtCdWriter ||
             mpData->mDeviceType == CDvdModel::dtDvdWriter) && 
            (aVolume.compare(mpData->mDeviceHalVolume)!=0)) {

            QDBusReply<QString> vReply = vDBusInterface.call( "GetProperty", 
                                                              "block.storage_device");

            if (vReply.isValid() && 
                (vReply.value().compare(mpData->mDeviceHalVolume) == 0)) {
                bool vIsCdrom = isCdrom(vDBusInterface);
                bool vIsDvd   = isDvd(vDBusInterface);
                
                if (vIsCdrom || vIsDvd) {
                    mpData->mHalVolume = aVolume;                    
                    mpData->mCapacity = ::capacity(vDBusInterface);

                    bool vEmpty = isEmpty(vDBusInterface);
                    if (vEmpty) {
                        if (vIsCdrom) {
                            mpData->mStatus = CDvdModel::dsEmptyCd;
                        } else {
                            mpData->mStatus = CDvdModel::dsEmptyDvd;
                        }
                    } else {
                        if (vIsCdrom) {
                            mpData->mStatus = CDvdModel::dsDataCd;
                        } else {
                            mpData->mStatus = CDvdModel::dsDataDvd;
                        }
                    }
                    update();
                }
            }
        }
    }
}

//!
void CDvdModel::checkAllVolumes() {
    if (mpData) {
        QDBusInterface vDBusInterface( HAL_SERVICE, HAL_PATH_MANAGER, HAL_OBJECT_MANAGER,
                                       QDBusConnection::systemBus());

        QDBusReply<QStringList> vReply = vDBusInterface.call( "FindDeviceStringMatch",
                                                              "block.device",
                                                              mpData->mDevice);
        if (vReply.isValid()) {
            foreach (QString vDevice, vReply.value()) {
                qDebug() << "found device: " << vDevice;
                checkVolume(vDevice);
            }
        }
    }
}

//!
void CDvdModel::removeVolume(const QString &aVolume) {
    qDebug() << "removeVolume: " << aVolume;

    if (mpData) {
        if (mpData->mHalVolume.compare(aVolume) == 0) {
            mpData->mStatus = CDvdModel::dsNoDisc;
            update();
        }
    }
}


//!
void CDvdModel::setDevice(const QString &aDevice) {
    if (mpData) {
        mpData->mDevice = aDevice;
        mpData->mDeviceType = CDvdModel::dtUndefined;
        mpData->mStatus = CDvdModel::dsUndefined;

        checkAllVolumes();

        update();
    }
}


//!
QString CDvdModel::device() {
    if (mpData)
        return mpData->mDevice;

    return QString::null;
}

void CDvdModel::setDeviceInfo(const QString &aDeviceInfo) {
    if (mpData) {
        mpData->mDeviceInfo = aDeviceInfo;
        update();
    }
}


//!
QString CDvdModel::deviceInfo() {
    if (mpData)
        return mpData->mDeviceInfo;

    return QString::null;
}


//!
void CDvdModel::setSpeed(int aSpeed) {
    if (mpData)
        mpData->mSpeed = aSpeed;
}

//!
int CDvdModel::speed() {
    if (mpData)
        return mpData->mSpeed;

    return -1;
}

//!
quint64 CDvdModel::capacity() {
    if (mpData)
        return mpData->mCapacity;

    return -1;
}

//!
void CDvdModel::setISOFileName(const QString &aISOFileName) {
    if (mpData)
        mpData->mISOFileName = aISOFileName; 
}

//!
QString CDvdModel::isoFileName() {
    if (mpData)
        return mpData->mISOFileName; 

    return QString::null;
}

//!
void CDvdModel::setISOSize(quint64 aSize) {
    if (mpData)
        mpData->mISOSize = aSize; 
}


//!
quint64 CDvdModel::isoSize() {
    if (mpData)
        return mpData->mISOSize; 

    return quint64(0);
}

//!
CDvdModel::DvdStatus CDvdModel::dvdStatus() {
    if (mpData)
        return mpData->mStatus;

    return CDvdModel::dsUndefined;
}

//!
void CDvdModel::update() {    
    if (mpData) {      
        emit(statusChanged(QString("%1 : %2").arg(mpData->mDevice).arg(mpData->mDeviceInfo), 
                           mpData->mDeviceType, mpData->mStatus, 
                           mpData->mProcessMode, mpData->mCapacity, mpData->mHash));
    } else {
        qDebug() << "mpData is NULL!";
    }
}


//////////////// SLOTS /////////////////////////////

//!
void CDvdModel::burnISO(const QString aFileName) {
    if (mpData->mpBurnThread) return ;

    setISOFileName(aFileName);

    mpData->mpBurnThread = new CBurnCDThread();
    mpData->mpBurnThread->init(this);
  
    connect(mpData->mpBurnThread, SIGNAL(finished()),
            this, SLOT(burnThreadFinished()));

    connect(mpData->mpBurnThread, SIGNAL(started()),
            this, SLOT(burnThreadStarted()));

    connect(mpData->mpBurnThread, SIGNAL(terminated()),
            this, SLOT(burnThreadTerminated()));

    connect(mpData->mpBurnThread, SIGNAL(progress(int,const QString &)),
            this, SIGNAL(progress(int,const QString &)));

    mpData->mpBurnThread->start();
}

//!
void CDvdModel::cancel() {
    if (mpData->mpBurnThread) {
        mpData->mpBurnThread->stop();
    } else
    if (mpData->mpCheckThread) {
        mpData->mpCheckThread->stop();
    }
}


//!
void CDvdModel::burnThreadFinished() {
    qDebug() << "burnThreadFinished!";
    if (mpData->mpBurnThread) {
        if (mpData->mpBurnThread->done()) {
            mpData->mProcessMode = CDvdModel::pmWriteDone;
        } else {
            mpData->mProcessMode = CDvdModel::pmWriteFailed;
        }
        update();

        delete mpData->mpBurnThread;
        mpData->mpBurnThread = 0;
    }
}

//!
void CDvdModel::burnThreadStarted() {
    qDebug() << "burnThreadStarted!";
    
    mpData->mProcessMode = CDvdModel::pmWriting;
    update();
}

//!
void CDvdModel::burnThreadTerminated() {
    qDebug() << "burnThreadTerminated!";
    if (mpData->mpBurnThread) {
        mpData->mProcessMode = CDvdModel::pmWriteFailed;
        update();
        
        delete mpData->mpBurnThread;
        mpData->mpBurnThread = 0;
    }
}

//!
void CDvdModel::checkSum(quint64 aSize) {
    if (mpData->mpCheckThread) return ;

    mpData->mISOSize = aSize;
    mpData->mpCheckThread = new CCheckThread();
    mpData->mpCheckThread->init(this);
  
    connect(mpData->mpCheckThread, SIGNAL(finished()),
            this, SLOT(checkThreadFinished()));

    connect(mpData->mpCheckThread, SIGNAL(started()),
            this, SLOT(checkThreadStarted()));

    connect(mpData->mpCheckThread, SIGNAL(terminated()),
            this, SLOT(checkThreadTerminated()));

    connect(mpData->mpCheckThread, SIGNAL(progress(int,const QString &)),
            this, SIGNAL(progress(int,const QString &)));

    mpData->mpCheckThread->start();
}

//!
void CDvdModel::checkThreadFinished() {
    qDebug() << "checkThreadFinished!";
    mpData->mHash = QLatin1String("");

    if (mpData->mpCheckThread) {
        if (mpData->mpCheckThread->done()) {
            mpData->mHash = mpData->mpCheckThread->hash();
            mpData->mProcessMode = CDvdModel::pmCheckDone;
        } else {
            mpData->mProcessMode = CDvdModel::pmCheckFailed;
        }
        update();
        delete mpData->mpCheckThread;
        mpData->mpCheckThread = 0;
    }
}

//!
void CDvdModel::checkThreadStarted() {
    qDebug() << "checkThreadStarted!";

    mpData->mProcessMode = CDvdModel::pmChecking;
    update();
}

//!
void CDvdModel::checkThreadTerminated() {
    qDebug() << "checkThreadTerminated!";
    if (mpData->mpCheckThread) {
        mpData->mProcessMode = CDvdModel::pmCheckFailed;
        update();
        
        delete mpData->mpCheckThread;
        mpData->mpCheckThread = 0;
    }
}

