/****************************************************************************
 **
 ** Author: Grigory A. Mozhaev
 ** Copyright (C) 2008-2009 Griland Company.
 ** Contact: Griland Company (griland.support@gmail.com)
 **
 ** This file may be used under the terms of the GNU General Public
 ** License version 2.0 as published by the Free Software Foundation
 ** and appearing in the file LICENSE.GPL included in the packaging of
 ** this file.  Please review the following information to ensure GNU
 ** General Public Licensing requirements will be met:
 ** http://www.trolltech.com/products/qt/opensource.html
 **
 ** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 ** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 **
 ****************************************************************************/
//! -*- coding: UTF-8 -*-
#define SOURCE_CODING "UTF-8"

#include <Qt/QtCore>
#include <Qt/QtGui>

#include "capplication.h"

#include "cdvdview.h"
#include "ui_cdvdview.h"

static QMap<QString,QIcon> mIconMap;

QPixmap CDvdView::icon(const QString &aName) {
    if (aName.isNull())
        return QPixmap();

    if (mIconMap.contains(aName)) {
        return mIconMap.value(aName).pixmap(46,46);
    }

    QPixmap vTmp(QFile::decodeName(":/icons/%1.png").arg(aName));

    if (!vTmp.isNull()) {
        QIcon vIcon(vTmp);
        mIconMap.insert(aName,vIcon);
        return vIcon.pixmap(46,46);
    }

    return QPixmap();
}


CDvdView::CDvdView(QWidget* apParent, Qt::WindowFlags aFlags) : 
    QFrame(apParent, aFlags),  mpUi(new Ui::CDvdView) {

    Q_ASSERT(CApplication::global()!=0);

    mpModel = 0;
    mpTimer = 0;
    mISOSize = 0;
    mAnimateFlag = false;
    mCDSpeedIndex = -1;
    mDvdSpeedIndex = -1;
    mpMenu = 0;

    mpUi->setupUi(this);
    mpUi->mpWideStatusLabel->setText(QLatin1String(""));
    mpUi->mpIconLabel->setText(QLatin1String(""));
    mpUi->mpProgressBar->setValue(0);
    mpUi->mpHashLabel1->setText(QLatin1String(""));
    mpUi->mpHashLabel2->setText(QLatin1String(""));
    mpUi->mpCancelButton->setEnabled(false);
    
    mpModel = new CDvdModel();
    if (mpModel) {
        connect(mpModel, SIGNAL(statusChanged(const QString &, const CDvdModel::DeviceType  &,
                                              const CDvdModel::DvdStatus   &,
                                              const CDvdModel::ProcessMode &,
                                              quint64,
                                              const QString &)),
                this, SLOT(statusChanged(const QString &, const CDvdModel::DeviceType  &,
                                        const CDvdModel::DvdStatus   &,
                                        const CDvdModel::ProcessMode &,
                                        quint64,
                                        const QString &)));
     
        connect (mpModel, SIGNAL(progress(int,const QString &)),
                 this, SLOT(progress(int,const QString &)));
    }

    connect(CApplication::global(), SIGNAL(refreshRootDirs()), this, SLOT(refreshRootDirsSlot()));
}

CDvdView::~CDvdView() {
    if (mpMenu)
        delete mpMenu;

    if (mpModel)
        delete mpModel;
}

void 
CDvdView::setDevice(const QString& aDevice, const QString& aDeviceInfo) {
    if (mpModel) {
        mpModel->setDevice(aDevice);
        mpModel->setDeviceInfo(aDeviceInfo);
    }
}

void 
CDvdView::setLabel(const QString& aLabel) {
    mpUi->mpNumberLabel->setText(aLabel);
}

void CDvdView::statusChanged(const QString &aDevice, 
                             const CDvdModel::DeviceType  &aDeviceType,
                             const CDvdModel::DvdStatus   &aDvdStatus,
                             const CDvdModel::ProcessMode &aProcessMode,
                             quint64 aCapacity,
                             const QString &aHash) {

    mpUi->mpDeviceLabel->setText(aDevice);
    mpUi->mpHashLabel2->setText(aHash);
    mpUi->mpBlankDataLabel->setText("");
    mpUi->mpWideStatusLabel->setText("");
    mpUi->mpTaskStatusLabel->setText("");
    mpUi->mpTaskStatusLabel->setEnabled(true);

    switch (aDeviceType) {

        case CDvdModel::dtCdWriter: {
            if ((aProcessMode!=CDvdModel::pmWriting) &&
                (aProcessMode!=CDvdModel::pmChecking))
                mpUi->mpIconLabel->setPixmap(icon(QLatin1String("cdrom")));
            break;
        }
        case CDvdModel::dtDvdWriter: {
            if ((aProcessMode!=CDvdModel::pmWriting) &&
                (aProcessMode!=CDvdModel::pmChecking))
                mpUi->mpIconLabel->setPixmap(icon(QLatin1String("dvdrom")));
            break;
        }

        case CDvdModel::dtUndefined:
        case CDvdModel::dtUnsupported:
        default:  {
            disableGui();
            return ;
        }

    };

    switch (aDvdStatus) {

        case CDvdModel::dsUndefined : {
            mpUi->mpSpeedCombo->clear();
            mSpeedList = 0;

            mpUi->mpStatusLabel->setText("Undefined");
            disableWrite();
            disableCheck();
            break;
        }

        case CDvdModel::dsNoDisc : {
            mpUi->mpSpeedCombo->clear();
            mSpeedList = 0;

            mpUi->mpStatusLabel->setText("No disc");
            disableWrite();
            disableCheck();
            break;
        }            

        case CDvdModel::dsEmptyCd : {
            mpUi->mpBlankDataLabel->setText("BLANK");
            mpUi->mpStatusLabel->setText("CD-R/RW");

            if (mSpeedList != 1) {
                mpUi->mpSpeedCombo->clear();
                mpUi->mpSpeedCombo->addItem("4");
                mpUi->mpSpeedCombo->addItem("8");
                mpUi->mpSpeedCombo->addItem("12");
                mpUi->mpSpeedCombo->addItem("16");
                mpUi->mpSpeedCombo->addItem("24");
                mpUi->mpSpeedCombo->addItem("32");
                mpUi->mpSpeedCombo->addItem("48");
                mpUi->mpSpeedCombo->addItem("52");

                if (mCDSpeedIndex < 0) 
                    mCDSpeedIndex = 4;

                mSpeedList = 1;
                mpUi->mpSpeedCombo->setCurrentIndex(mCDSpeedIndex);
            }

            enableWrite();
            disableCheck();            
            break;
        }

        case CDvdModel::dsEmptyDvd : {
            mpUi->mpBlankDataLabel->setText("BLANK");
            mpUi->mpStatusLabel->setText("DVDR/RW");

            if (aDeviceType == CDvdModel::dtDvdWriter) {
                if (mSpeedList != 2) {
                    mpUi->mpSpeedCombo->addItem("1");
                    mpUi->mpSpeedCombo->addItem("2");
                    mpUi->mpSpeedCombo->addItem("4");
                    mpUi->mpSpeedCombo->addItem("6");
                    mpUi->mpSpeedCombo->addItem("8");
                    mpUi->mpSpeedCombo->addItem("10");
                    mpUi->mpSpeedCombo->addItem("12");

                    if (mDvdSpeedIndex < 0) 
                        mDvdSpeedIndex = 4;

                    mSpeedList = 2;
                    mpUi->mpSpeedCombo->setCurrentIndex(mDvdSpeedIndex);
                }

                enableWrite();
                disableCheck();
            } else {
                mpUi->mpWideStatusLabel->setText("Unsupported media by this device!");
                disableWrite();
                disableCheck();
            }
            break;
        }

        case CDvdModel::dsDataCd : {
            mpUi->mpSpeedCombo->clear();
            mSpeedList = 0;


            mpUi->mpStatusLabel->setText("CD-R/RW");
            mpUi->mpBlankDataLabel->setText("DATA");

            disableWrite();
            enableCheck();
            break;
        }

        case CDvdModel::dsDataDvd : {
            mpUi->mpSpeedCombo->clear();
            mSpeedList = 0;

            mpUi->mpStatusLabel->setText("DVDR/RW");
            mpUi->mpBlankDataLabel->setText("DATA");

            disableWrite();
            enableCheck();
            break;
        }
    };


    if ((aProcessMode != CDvdModel::pmWriting) &&
        (aProcessMode != CDvdModel::pmChecking)) {
        qDebug() << "aProcessMode: " << aProcessMode;
        enableCombos();
        stopAnimation();
    } else {
        disableCombos();
    }

    switch (aProcessMode) {
        case CDvdModel::pmNone: {
            if (aDvdStatus == CDvdModel::dsEmptyCd || aDvdStatus == CDvdModel::dsEmptyDvd) {
                mpUi->mpWideStatusLabel->setText(QString("ISO Size %1 M / Disk capacity %2 M")
                                                 .arg(mISOSize  / 1024 / 1024)
                                                 .arg(aCapacity  / 1024 / 1024));
            }

            if ((mISOSize == 0) || (mISOSize > aCapacity)) {
                disableWrite();
            }

            break;
        }

        case CDvdModel::pmWriting: {
            mpUi->mpHashLabel2->setText("Writing...");
            startBurnAnimation();
            disableWrite();
            disableCheck();
            enableCancel();
            break;
        }

        case CDvdModel::pmChecking: {
            mpUi->mpHashLabel2->setText("Calculating...");
            startCheckAnimation();
            disableWrite();
            disableCheck();
            enableCancel();
            break;
        }


        case CDvdModel::pmWaiting: {
            break;
        }

        case CDvdModel::pmWriteDone: {
            mpUi->mpTaskStatusLabel->setEnabled(true);
            mpUi->mpTaskStatusLabel->setText("Burn Done");

            if ((mpUi->mpDataCheck->checkState() == Qt::PartiallyChecked) ||
                (mpUi->mpDataCheck->checkState() == Qt::Checked)) {
                    mpModel->checkSum(mISOSize);
            }

            disableCancel();
            break;
        }

        case CDvdModel::pmCheckDone: {
            if ((aHash.size() > 0) && (mISOmd5sum.size() > 0)) {
                if (aHash.compare(mISOmd5sum)==0) {
                    mpUi->mpTaskStatusLabel->setEnabled(true);
                    mpUi->mpTaskStatusLabel->setText("md5 is correct");
                }
                else {
                    mpUi->mpTaskStatusLabel->setEnabled(false);
                    mpUi->mpTaskStatusLabel->setText("md5 is incorrect");
                }
            }
            disableCancel();
            break;
        }

        case CDvdModel::pmCheckFailed: {
            mpUi->mpTaskStatusLabel->setEnabled(false);
            mpUi->mpTaskStatusLabel->setText("check md5 failed");
            mpUi->mpHashLabel2->setText("I/O Error: can't read md5sum");
            disableCancel();
            break;
        }

        case CDvdModel::pmWriteFailed: {
            mpUi->mpTaskStatusLabel->setEnabled(false);
            mpUi->mpTaskStatusLabel->setText("Burn failed");
            mpUi->mpHashLabel2->setText("Canceled by user or I/O Error");
            disableCheck();
            disableCancel();
            break;
        }
    };
}

void CDvdView::enableWrite() {
    mpUi->mpBurnButton->setEnabled(true);
    mpUi->mpSpeedCombo->setEnabled(true);
}

void CDvdView::disableWrite() {
    mpUi->mpBurnButton->setEnabled(false);
    mpUi->mpSpeedCombo->setEnabled(false);
}

void CDvdView::enableCheck() {
    mpUi->mpCheckButton->setEnabled(true);
}

void CDvdView::disableCheck() {
    mpUi->mpCheckButton->setEnabled(false);
}

void CDvdView::enableCancel() {
    mpUi->mpCancelButton->setEnabled(true);
}

void CDvdView::disableCancel() {
    mpUi->mpCancelButton->setEnabled(false);
}

void CDvdView::enableCombos() {
    mpUi->mpCategoryCombo->setEnabled(true);
    mpUi->mpISOCombo->setEnabled(true);
}

void CDvdView::disableCombos() {
    mpUi->mpCategoryCombo->setEnabled(false);
    mpUi->mpISOCombo->setEnabled(false);
}

void CDvdView::disableGui() {
    setEnabled(false);
}

void CDvdView::startBurnAnimation() {
    if (!mpTimer) {
        mpTimer = new QTimer();

        mpTimer->start(1000);

        connect(mpTimer, SIGNAL(timeout()),
                this,  SLOT(burnAnimate()));
    }
}

void CDvdView::stopAnimation() {
    if (mpTimer) {
        mpTimer->stop();
        delete mpTimer;
        mpTimer = 0;
    }
}

void CDvdView::burnAnimate() {
    mAnimateFlag = !mAnimateFlag;
    if (mAnimateFlag) {
        mpUi->mpIconLabel->setPixmap(icon(QLatin1String("burn_on")));
    } else {
        mpUi->mpIconLabel->setPixmap(icon(QLatin1String("burn_off")));
    }
    mpUi->mpIconLabel->update();
}

void CDvdView::startCheckAnimation() {
    if (!mpTimer)
        mpTimer = new QTimer();

    mpTimer->start(1000);

    connect(mpTimer, SIGNAL(timeout()),
            this,  SLOT(checkAnimate()));
}

void CDvdView::checkAnimate() {
    mAnimateFlag = !mAnimateFlag;
    if (mAnimateFlag) {
        mpUi->mpIconLabel->setPixmap(icon(QLatin1String("check_on")));
    } else {
        mpUi->mpIconLabel->setPixmap(icon(QLatin1String("check_off")));
    }
}

void CDvdView::progress(int aProgress, const QString &aInfo) {
    mpUi->mpProgressBar->setValue(aProgress);
    mpUi->mpWideStatusLabel->setText(aInfo);
}

// Ui Slots

void 
CDvdView::on_mpDataCheck_stateChanged(int aState) {
}

void 
CDvdView::on_mpCategoryCombo_currentIndexChanged(QString aCategory) {
    mCategory = aCategory;

    QStringList vRootDirs = CApplication::global()->variable(QLatin1String("fs"), QLatin1String("rootDirs")).toStringList();   

    qDebug() << vRootDirs;

    mpUi->mpISOCombo->clear();
    foreach (QString vPath, vRootDirs) {
        QDir vDir(QString("%1/%2").arg(vPath).arg(mCategory));
        
        QStringList vNameFilters;
        vNameFilters << QLatin1String("*.[iI][sS][oO]");   
        
        QStringList vFileNames = vDir.entryList(vNameFilters, QDir::Files);
        foreach (QString vFileName, vFileNames) {
            if (mpUi->mpISOCombo->findText(vFileName) == -1)
                mpUi->mpISOCombo->addItem(vFileName);
        }
    }
}

void
CDvdView::on_mpISOCombo_currentIndexChanged(QString aISOFileName) {
    QStringList vRootDirs = CApplication::global()->variable(QLatin1String("fs"), QLatin1String("rootDirs")).toStringList();   
    mISOmd5sum = QLatin1String("");
    foreach (QString vPath, vRootDirs) {
        mISOFile = QDir::toNativeSeparators(QString("%1/%2/%3")
                                            .arg(vPath).arg(mCategory).arg(aISOFileName));
        QFileInfo vFileInfo(mISOFile);
        if (vFileInfo.exists()) {
            mISOSize = vFileInfo.size();
        
            QString vFileName = QDir::toNativeSeparators(QString("%1/%2/%3.md5sum")
                                                         .arg(vPath).arg(mCategory).arg(aISOFileName));
            vFileInfo.setFile(vFileName);
            if (vFileInfo.exists()) {
                QFile vFile(vFileName);
                if (vFile.open(QIODevice::ReadOnly)) {
                    QString vData = vFile.readAll();
                    if (vData == QLatin1String("calculating...")) {
                        mISOmd5sum = tr("calculating...");
                    } else {
                        QStringList vSplittedData = vData.split(QLatin1String(" "),
                                                                QString::SkipEmptyParts);
                        if (vSplittedData.size() == 2) {
                            mISOmd5sum = vSplittedData.at(0);
                        }
                    }
                    vFile.close();
                }
            }
            break;
        } else {
            mISOSize = 0;
        }
    }
    mpUi->mpHashLabel1->setText(mISOmd5sum);
    mpModel->update();
}


void 
CDvdView::on_mpSpeedCombo_currentIndexChanged(int aIndex) {
    if (aIndex >= 0) {
        if (mSpeedList == 1) {
            mCDSpeedIndex = aIndex;
            
        } else
        if (mSpeedList == 2) {
            mDvdSpeedIndex = aIndex;
        }
    }
}

void
CDvdView::on_mpCheckButton_clicked() {
    if (mpModel)
        mpModel->checkSum(mISOSize);

}

void 
CDvdView::on_mpBurnButton_clicked() {
    if (mpModel) {
        mpModel->setSpeed(mpUi->mpSpeedCombo->currentText().toInt());
        mpModel->burnISO(mISOFile);
    }
}

void 
CDvdView::on_mpCancelButton_clicked() {
    if (mpModel)
        mpModel->cancel();

}

void
CDvdView::on_mpRefreshButton_clicked() {    
    if (mpMenu) {
        delete mpMenu;
        mpMenu = 0;
    }

    mpMenu = new QMenu();
    QAction *vpAction = mpMenu->addAction(tr("only for this"));
    connect(vpAction, SIGNAL(triggered()), this, SLOT(refreshRootDirsSlot()));

    if (CApplication::global() != 0) {
         vpAction = mpMenu->addAction(tr("do it for all"));
         connect(vpAction, SIGNAL(triggered()), CApplication::global(), SLOT(updateRootDirs()));
    }

    mpMenu->popup(mpUi->mpRefreshButton->mapToGlobal(QPoint(0,0)));
}


void
CDvdView::refreshRootDirsSlot() {
    QStringList vRootDirs = CApplication::global()->variable(QLatin1String("fs"), QLatin1String("rootDirs")).toStringList();   

    mpUi->mpCategoryCombo->clear();
    foreach (QString vPath, vRootDirs) {
        QDir vDir(vPath);
        QStringList vCategories = vDir.entryList(QDir::AllDirs | QDir::NoDotAndDotDot);
        foreach (QString vCategory, vCategories) {
            if (mpUi->mpCategoryCombo->findText(vCategory) == -1)
                mpUi->mpCategoryCombo->addItem(vCategory);
        }
    }
}
