/****************************************************************************
 **
 ** Copyright (C) 2008-2009 Grigory A. Mozhaev.  All rights reserved.
 **
 ** This file may be used under the terms of the GNU General Public
 ** License version 2.0 as published by the Free Software Foundation
 ** and appearing in the file LICENSE.GPL included in the packaging of
 ** this file.  Please review the following information to ensure GNU
 ** General Public Licensing requirements will be met:
 ** http://www.trolltech.com/products/qt/opensource.html
 **
 ** If you are unsure which license is appropriate for your use, please
 ** review the following information:
 ** http://www.trolltech.com/products/qt/licensing.html or contact the
 ** sales department at sales@trolltech.com.
 **
 ** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 ** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 **
 ** AUTHOR: Grigory A. Mozhaev <zcrendel@mail.ru>
 **
 ****************************************************************************/
//! -*- coding: UTF-8 -*-
#define SOURCE_CODING "UTF-8"

#include "cgeneratethread.h"

#include <QtCore/QCryptographicHash>

class CGenerateThreadPrivate {
public:
    QCryptographicHash *mpHash;
    QMap<QString, int> mJobMap;
    QMap<QString, QString> mFullNames;

    bool mInitialized;

    quint64 mPercent;

    CGenerateThreadPrivate() {
        mInitialized = false;
        mpHash = new QCryptographicHash(QCryptographicHash::Md5);       
    }
    ~CGenerateThreadPrivate() {
        if (mpHash) {
            delete mpHash;
            mpHash = 0;
        }
    }
};

CGenerateThread::CGenerateThread() {
    mpData = new CGenerateThreadPrivate;

    Q_ASSERT(mpData!=0);    
}
 
CGenerateThread::~CGenerateThread() {
    if (mpData) {
        delete mpData;
        mpData = 0;
    }
}


void 
CGenerateThread::init(const QStringList &aJobList, const QStringList &aFullNames) {
    if (mpData) {
        for (int i=0; i < aJobList.count(); i++) {
            mpData->mJobMap.insert(aJobList.at(i), 1);            
            mpData->mFullNames.insert(aJobList.at(i), aFullNames.at(i));
        }

        mpData->mInitialized = true;
    }
}
    
void CGenerateThread::stop() {
    if (mpData) {
        mMutex.lock();
        QMap<QString, int>::Iterator vCurrentJob = mpData->mJobMap.begin();
        while (vCurrentJob != mpData->mJobMap.end()) {
            if (vCurrentJob.value() == 1)
                vCurrentJob.value() = 0;
            ++vCurrentJob;
        }
        mMutex.unlock();
    }
}

void 
CGenerateThread::run() {
    qDebug() << "CGenerateThread::run()";

    if (!mpData->mInitialized) {
        qDebug() << "CGenerateThread not initialized! exiting...";
        return;
    }

    QMap<QString, int>::Iterator vCurrentJob = mpData->mJobMap.begin();
    QString vPrevJob = QString::null; 
    while (vCurrentJob != mpData->mJobMap.end()) {
        switchJob(vPrevJob, vCurrentJob.key());

        QFile vFile(mpData->mFullNames.value(vCurrentJob.key()));
        if (vCurrentJob.value() == 1) {
            if (!vFile.open(QIODevice::ReadOnly)) {
                mMutex.lock();
                vCurrentJob.value() = 2;
                mMutex.unlock();
                progress(vCurrentJob.key(), 0);
                vPrevJob = vCurrentJob.key();
                ++vCurrentJob;
                continue;
            }
        } else {
            progress(vCurrentJob.key(), 0, tr("CANCELED"));
            switchJob(vPrevJob, vCurrentJob.key());
            vPrevJob = vCurrentJob.key();
            ++vCurrentJob;
            continue ;
        }

        progress(vCurrentJob.key(), 0);

        // calculate hash
        mpData->mpHash->reset();

        QFileInfo vFileInfo(mpData->mFullNames.value(vCurrentJob.key()));
        quint64   vPercent  = vFileInfo.size() / 100;
        quint64   vRed      = 0;
        int       vTryCount = 0;
        bool      vError    = false;
        int       vCurrentPercent = 0;
        int       vPrevPercent    = 0;

        // ******************
        while (!vFile.atEnd()) {
         
            if (vCurrentJob.value() == 0) {
                switchJob(vPrevJob, vCurrentJob.key());
                progress(vCurrentJob.key(), 0, tr("CANCELED"));
                mMutex.lock();
                vCurrentJob.value() = 0;
                mMutex.unlock();
                break;
            }

            char buffer[16384];
            qint64 vRetVal = vFile.read(buffer, 16384);
            if (vRetVal > 0) {
                vTryCount = 0;
                vRed += vRetVal;
                mpData->mpHash->addData(buffer, vRetVal);
                vCurrentPercent = vRed / vPercent;
                if (vCurrentPercent != vPrevPercent) {
                    progress(vCurrentJob.key(), vCurrentPercent, tr("progress..."));
                    vPrevPercent = vCurrentPercent;
                    qApp->processEvents();
                }
            } else 
                if (vRetVal < 0) {
                    vTryCount++;
                    if (vTryCount > 2) {
                        vError = true;
                        progress(vCurrentJob.key(), 0, tr("ERROR"));
                        mMutex.lock();
                        vCurrentJob.value() = 4;
                        mMutex.unlock();
                        qDebug() << "Error: Can't read from device";
                        qDebug() << "Current md5sum is:" << 
                            mpData->mpHash->result().toHex();
                        break;
                    }
                } else {
                    break;
                }
        }
        vFile.close();
        // ******************

        if ((!vError) && (vCurrentJob.value()==1)) {
            qDebug() << "DONE!";
            QFile vMd5File(QString("%1.md5sum").arg(mpData->mFullNames.value(vCurrentJob.key())));
            if (vMd5File.open(QIODevice::WriteOnly)) {
                QFileInfo vMd5FileInfo(mpData->mFullNames.value(vCurrentJob.key()));
                QTextStream vOut(&vMd5File);
                vOut << mpData->mpHash->result().toHex() << QLatin1String("  ") << vMd5FileInfo.fileName() << endl;
                vMd5File.close();
                progress(vCurrentJob.key(), 100, tr("DONE"));
                mMutex.lock();
                vCurrentJob.value() = 3;
                mMutex.unlock();
            } else {
                vError = true;
                progress(vCurrentJob.key(), 0, tr("ERROR"));
                mMutex.lock();
                vCurrentJob.value() = 4;
                mMutex.unlock();
            }
        }

        vPrevJob = vCurrentJob.key();
        ++vCurrentJob;
    }
    switchJob(vPrevJob, QString::null);

    qApp->processEvents();
}

int
CGenerateThread::jobStatus(const QString &aJobName) {
    if ((mpData) && (mpData->mJobMap.contains(aJobName)))
        return mpData->mJobMap.value(aJobName);

    return 0;
}

void CGenerateThread::cancelJob(const QString &aJobName) {
    mMutex.lock();
    mpData->mJobMap.insert(aJobName, 0);
    mMutex.unlock();
}
