/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SandboxExtension_h
#define SandboxExtension_h

#include <wtf/Forward.h>
#include <wtf/Noncopyable.h>
#include <wtf/PassRefPtr.h>
#include <wtf/RefCounted.h>
#include <wtf/text/WTFString.h>

#if ENABLE(WEB_PROCESS_SANDBOX)
typedef struct __WKSandboxExtension* WKSandboxExtensionRef;
#endif

namespace CoreIPC {
    class ArgumentEncoder;
    class ArgumentDecoder;
}

namespace WebKit {

class SandboxExtension : public RefCounted<SandboxExtension> {
public:
    enum Type {
        ReadOnly,
        WriteOnly,
        ReadWrite,
    };

    class Handle {
        WTF_MAKE_NONCOPYABLE(Handle);
    
    public:
        Handle();
        ~Handle();

        void encode(CoreIPC::ArgumentEncoder*) const;
        static bool decode(CoreIPC::ArgumentDecoder*, Handle&);

    private:
        friend class SandboxExtension;
#if ENABLE(WEB_PROCESS_SANDBOX)
        mutable WKSandboxExtensionRef m_sandboxExtension;
#endif
    };

    static PassRefPtr<SandboxExtension> create(const Handle&);
    static void createHandle(const String& path, Type type, Handle&);
    static String createHandleForTemporaryFile(const String& prefix, Type type, Handle&);
    ~SandboxExtension();

    bool invalidate();
    bool consume();
    bool consumePermanently();

private:
    explicit SandboxExtension(const Handle&);
                     
#if ENABLE(WEB_PROCESS_SANDBOX)
    mutable WKSandboxExtensionRef m_sandboxExtension;
#endif
};

#if !ENABLE(WEB_PROCESS_SANDBOX)
inline SandboxExtension::Handle::Handle() { }
inline SandboxExtension::Handle::~Handle() { }
inline void SandboxExtension::Handle::encode(CoreIPC::ArgumentEncoder*) const { }
inline bool SandboxExtension::Handle::decode(CoreIPC::ArgumentDecoder*, Handle&) { return true; }
inline PassRefPtr<SandboxExtension> SandboxExtension::create(const Handle&) { return 0; }
inline void SandboxExtension::createHandle(const String& path, Type type, Handle&) { }
inline String SandboxExtension::createHandleForTemporaryFile(const String& prefix, Type type, Handle&) {return String();}
inline SandboxExtension::~SandboxExtension() { }
inline bool SandboxExtension::invalidate() { return true; }
inline bool SandboxExtension::consume() { return true; }
inline bool SandboxExtension::consumePermanently() { return true; }
#endif

} // namespace WebKit


#endif // SandboxExtension_h
