/***************************************************************************
 *   Copyright (C) 2013~2013 by Yichao Yu                                  *
 *   yyc1992@gmail.com                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.              *
 ***************************************************************************/

#include "dirs.h"
#include "log.h"
#include "strs.h"
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>

static char *qtc_home = NULL;
static char *qtc_xdg_data_home = NULL;
static char *qtc_xdg_config_home = NULL;
static char *qtc_conf_dir = NULL;

QTC_EXPORT void
qtcMakePath(const char *path, int mode)
{
    if (qtcIsDir(path)) {
        return;
    }
    size_t len = strlen(path);
    char opath[len + 1];
    memcpy(opath, path, len + 1);
    while (opath[len - 1] == '/') {
        opath[len - 1] = '\0';
        len--;
    }
    char *p = opath + strspn(opath, "/");
    if (!p) {
        return;
    }
    p += 1;
    for (;*p;p++) {
        if (*p == '/') {
            *p = '\0';
            if (access(opath, F_OK)) {
                mkdir(opath, mode | 0300);
            }
            *p = '/';
        }
    }
    if (access(opath, F_OK)) {
        mkdir(opath, mode);
    }
}

static const char*
_qtcConfDir()
{
    if (qtcUnlikely(!qtc_conf_dir)) {
        const char *env_home = getenv("QTCURVE_CONFIG_DIR");
        if (env_home && *env_home == '/') {
            qtc_conf_dir = qtcCatStrs(env_home, "/");
        } else {
            qtc_conf_dir = qtcCatStrs(qtcGetXDGConfigHome(), "qtcurve/");
        }
    }
    return qtc_conf_dir;
}

QTC_EXPORT const char*
qtcConfDir()
{
    static const char *conf_dir = NULL;
    if (!conf_dir) {
        conf_dir = _qtcConfDir();
        qtcMakePath(conf_dir, 0700);
    }
    return conf_dir;
}

QTC_EXPORT const char*
qtcGetHome()
{
    if (qtcUnlikely(!qtc_home)) {
        const char *env_home = getenv("HOME");
        if (qtcUnlikely(env_home && *env_home == '/')) {
            qtc_home = qtcCatStrs(env_home, "/");
        } else {
            struct passwd *pw = getpwuid(getuid());
            if (qtcLikely(pw && pw->pw_dir && *pw->pw_dir == '/')) {
                qtc_home = qtcCatStrs(pw->pw_dir, "/");
            }
        }
        if (qtcUnlikely(!qtc_home)) {
            qtc_home = strdup("/tmp/");
        }
    }
    return qtc_home;
}

QTC_EXPORT const char*
qtcGetXDGDataHome()
{
    if (qtcUnlikely(!qtc_xdg_data_home)) {
        const char *env_home = getenv("XDG_DATA_HOME");
        if (env_home && *env_home == '/') {
            qtc_xdg_data_home = qtcCatStrs(env_home, "/");
        } else {
            qtc_xdg_data_home = qtcCatStrs(qtcGetHome(), ".local/share/");
        }
    }
    return qtc_xdg_data_home;
}

QTC_EXPORT const char*
qtcGetXDGConfigHome()
{
    if (qtcUnlikely(!qtc_xdg_config_home)) {
        const char *env_home = getenv("XDG_CONFIG_HOME");
        if (env_home && *env_home == '/') {
            qtc_xdg_config_home = qtcCatStrs(env_home, "/");
        } else {
            qtc_xdg_config_home = qtcCatStrs(qtcGetHome(), ".config/");
        }
    }
    return qtc_xdg_config_home;
}

QTC_EXPORT char*
(qtcGetConfFile)(const char *file, char *buff)
{
    if (file[0] == '/') {
        return qtcFillStrs(buff, file);
    }
    return qtcFillStrs(buff, qtcConfDir(), file);
}

__attribute__((constructor)) static void
_qtcGetDirs()
{
    qtcGetHome();
    qtcGetXDGDataHome();
    qtcGetXDGConfigHome();
    _qtcConfDir();
}

__attribute__((destructor)) static void
_qtcFreeDirs()
{
    qtcFree(qtc_home);
    qtc_home = NULL;
    qtcFree(qtc_xdg_data_home);
    qtc_xdg_data_home = NULL;
    qtcFree(qtc_xdg_config_home);
    qtc_xdg_config_home = NULL;
    qtcFree(qtc_conf_dir);
    qtc_conf_dir = NULL;
}
