/*
	world.c

	world query functions

	Copyright (C) 1996-1997  Id Software, Inc.

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

	See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to:

		Free Software Foundation, Inc.
		59 Temple Place - Suite 330
		Boston, MA  02111-1307, USA

*/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

static __attribute__ ((unused)) const char rcsid[] = 
	"$Id: world.c,v 1.51 2004/02/24 20:06:53 despair Exp $";

#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif

#include <stdio.h>

#include "QF/clip_hull.h"
#include "QF/console.h"
#include "QF/crc.h"
#include "QF/sys.h"

#include "compat.h"
#include "server.h"
#include "sv_progs.h"
#include "world.h"

/*
	entities never clip against themselves, or their owner
	line of sight checks trace->crosscontent, but bullets don't
*/

typedef struct {
	vec3_t       boxmins, boxmaxs;		// enclose the test object along
										// entire move
	const float *mins, *maxs;			// size of the moving object
	vec3_t       mins2, maxs2;			// size when clipping against
										// monsters
	const float *start, *end;
	trace_t      trace;
	int          type;
	edict_t    *passedict;
} moveclip_t;

/* HULL BOXES */

static hull_t box_hull;
static dclipnode_t box_clipnodes[6];
static mplane_t box_planes[6];


/*
	SV_InitHull SV_InitBoxHull

	Set up the planes and clipnodes so that the six floats of a bounding box
	can just be stored out and get a proper hull_t structure.
*/
void
SV_InitHull (hull_t *hull, dclipnode_t *clipnodes, mplane_t *planes)
{
	int			side, i;

	hull->clipnodes = clipnodes;
	hull->planes = planes;
	hull->firstclipnode = 0;
	hull->lastclipnode = 5;

	for (i = 0; i < 6; i++) {
		hull->clipnodes[i].planenum = i;

		side = i & 1;

		hull->clipnodes[i].children[side] = CONTENTS_EMPTY;
		if (i != 5)
			hull->clipnodes[i].children[side ^ 1] = i + 1;
		else
			hull->clipnodes[i].children[side ^ 1] = CONTENTS_SOLID;

		hull->planes[i].type = i >> 1;
		hull->planes[i].normal[i >> 1] = 1;
	}
}

static inline void
SV_InitBoxHull (void)
{
	SV_InitHull (&box_hull, box_clipnodes, box_planes);
}

/*
	SV_HullForBox

	To keep everything totally uniform, bounding boxes are turned into small
	BSP trees instead of being compared directly.
*/
static inline hull_t *
SV_HullForBox (const vec3_t mins, const vec3_t maxs)
{
	box_planes[0].dist = maxs[0];
	box_planes[1].dist = mins[0];
	box_planes[2].dist = maxs[1];
	box_planes[3].dist = mins[1];
	box_planes[4].dist = maxs[2];
	box_planes[5].dist = mins[2];

	return &box_hull;
}

/*
	SV_HullForEntity

	Returns a hull that can be used for testing or clipping an object of
	mins/maxs size.  Offset is filled in to contain the adjustment that
	must be added to the testing object's origin to get a point to use with
	the returned hull.
*/
hull_t *
SV_HullForEntity (edict_t *ent, const vec3_t mins, const vec3_t maxs,
				  vec3_t offset)
{
	hull_t     *hull = 0;
	int         hull_index = 0;
	model_t    *model;
	vec3_t		hullmins, hullmaxs, size;

	if ((sv_fields.rotated_bbox != -1
		 && SVinteger (ent, rotated_bbox))
		|| SVfloat (ent, solid) == SOLID_BSP) {
		VectorSubtract (maxs, mins, size);
		if (size[0] < 3)
			hull_index = 0;
		else if (size[0] <= 32)
			hull_index = 1;
		else
			hull_index = 2;
	}
	// decide which clipping hull to use, based on the size
	if (sv_fields.rotated_bbox != -1
		&& SVinteger (ent, rotated_bbox)) {
		int h = SVinteger (ent, rotated_bbox) - 1;
		hull = pf_hull_list[h]->hulls[hull_index];
	} if (SVfloat (ent, solid) == SOLID_BSP) {
		// explicit hulls in the BSP model
		if (SVfloat (ent, movetype) != MOVETYPE_PUSH)
			Sys_Error ("SOLID_BSP without MOVETYPE_PUSH");

		model = sv.models[(int) SVfloat (ent, modelindex)];

		if (!model || model->type != mod_brush)
			Sys_Error ("SOLID_BSP with a non bsp model");

		hull = &model->hulls[hull_index];
	}

	if (hull) {
		// calculate an offset value to center the origin
		VectorSubtract (hull->clip_mins, mins, offset);
		VectorAdd (offset, SVvector (ent, origin), offset);
	} else {					// create a temp hull from bounding box sizes
		VectorSubtract (SVvector (ent, mins), maxs, hullmins);
		VectorSubtract (SVvector (ent, maxs), mins, hullmaxs);
		hull = SV_HullForBox (hullmins, hullmaxs);

		VectorCopy (SVvector (ent, origin), offset);
	}

	return hull;
}

/* ENTITY AREA CHECKING */

areanode_t  sv_areanodes[AREA_NODES];
int         sv_numareanodes;

static areanode_t *
SV_CreateAreaNode (int depth, const vec3_t mins, const vec3_t maxs)
{
	areanode_t *anode;
	vec3_t      mins1, maxs1, mins2, maxs2, size;

	anode = &sv_areanodes[sv_numareanodes];
	sv_numareanodes++;

	ClearLink (&anode->trigger_edicts);
	ClearLink (&anode->solid_edicts);

	if (depth == AREA_DEPTH) {
		anode->axis = -1;
		anode->children[0] = anode->children[1] = NULL;
		return anode;
	}

	VectorSubtract (maxs, mins, size);
	if (size[0] > size[1])
		anode->axis = 0;
	else
		anode->axis = 1;

	anode->dist = 0.5 * (maxs[anode->axis] + mins[anode->axis]);
	VectorCopy (mins, mins1);
	VectorCopy (mins, mins2);
	VectorCopy (maxs, maxs1);
	VectorCopy (maxs, maxs2);

	maxs1[anode->axis] = mins2[anode->axis] = anode->dist;

	anode->children[0] = SV_CreateAreaNode (depth + 1, mins2, maxs2);
	anode->children[1] = SV_CreateAreaNode (depth + 1, mins1, maxs1);

	return anode;
}

void
SV_ClearWorld (void)
{
	SV_InitBoxHull ();

	memset (sv_areanodes, 0, sizeof (sv_areanodes));
	sv_numareanodes = 0;
	SV_CreateAreaNode (0, sv.worldmodel->mins, sv.worldmodel->maxs);
}

link_t        **sv_link_next;
link_t        **sv_link_prev;

void
SV_UnlinkEdict (edict_t *ent)
{
	if (!ent->area.prev)
		return;							// not linked in anywhere
	RemoveLink (&ent->area);
	if (sv_link_next && *sv_link_next == &ent->area)
		*sv_link_next = ent->area.next;
	if (sv_link_prev && *sv_link_prev == &ent->area)
		*sv_link_prev = ent->area.prev;
	ent->area.prev = ent->area.next = NULL;
}

static void
SV_TouchLinks (edict_t *ent, areanode_t *node)
{
	int         old_self, old_other;
	edict_t    *touch;
	link_t     *l, *next;

	// touch linked edicts
	sv_link_next = &next;
	for (l = node->trigger_edicts.next; l != &node->trigger_edicts; l = next) {
		next = l->next;
		touch = EDICT_FROM_AREA (l);
		if (touch == ent)
			continue;
		if (!SVfunc (touch, touch)
			|| SVfloat (touch, solid) != SOLID_TRIGGER)
			continue;
		if (SVvector (ent, absmin)[0] > SVvector (touch, absmax)[0]
			|| SVvector (ent, absmin)[1] > SVvector (touch, absmax)[1]
			|| SVvector (ent, absmin)[2] > SVvector (touch, absmax)[2]
			|| SVvector (ent, absmax)[0] < SVvector (touch, absmin)[0]
			|| SVvector (ent, absmax)[1] < SVvector (touch, absmin)[1]
			|| SVvector (ent, absmax)[2] < SVvector (touch, absmin)[2])
			continue;

		old_self = *sv_globals.self;
		old_other = *sv_globals.other;

		*sv_globals.time = sv.time;
		sv_pr_touch (touch, ent);

		*sv_globals.self = old_self;
		*sv_globals.other = old_other;
	}
	sv_link_next = 0;

	// recurse down both sides
	if (node->axis == -1)
		return;

	if (SVvector (ent, absmax)[node->axis] > node->dist)
		SV_TouchLinks (ent, node->children[0]);
	if (SVvector (ent, absmin)[node->axis] < node->dist)
		SV_TouchLinks (ent, node->children[1]);
}

static void
SV_FindTouchedLeafs (edict_t *ent, mnode_t *node)
{
	int			leafnum, sides;
	mleaf_t    *leaf;
	mplane_t   *splitplane;

	if (node->contents == CONTENTS_SOLID)
		return;

	// add an efrag if the node is a leaf
	if (node->contents < 0) {
		if (ent->num_leafs == MAX_ENT_LEAFS)
			return;

		leaf = (mleaf_t *) node;
		leafnum = leaf - sv.worldmodel->leafs - 1;

		ent->leafnums[ent->num_leafs] = leafnum;
		ent->num_leafs++;
		return;
	}

	// NODE_MIXED
	splitplane = node->plane;
	sides = BOX_ON_PLANE_SIDE (SVvector (ent, absmin),
							   SVvector (ent, absmax), splitplane);

	// recurse down the contacted sides
	if (sides & 1)
		SV_FindTouchedLeafs (ent, node->children[0]);

	if (sides & 2)
		SV_FindTouchedLeafs (ent, node->children[1]);
}

void
SV_LinkEdict (edict_t *ent, qboolean touch_triggers)
{
	areanode_t *node;

	if (ent->area.prev)
		SV_UnlinkEdict (ent);			// unlink from old position

	if (ent == sv.edicts)
		return;							// don't add the world
	if (ent->free)
		return;

	// set the abs box
	VectorAdd (SVvector (ent, origin), SVvector (ent, mins),
			   SVvector (ent, absmin));
	VectorAdd (SVvector (ent, origin), SVvector (ent, maxs),
			   SVvector (ent, absmax));

	// to make items easier to pick up and allow them to be grabbed off
	// of shelves, the abs sizes are expanded
	if ((int) SVfloat (ent, flags) & FL_ITEM) {
		SVvector (ent, absmin)[0] -= 15;
		SVvector (ent, absmin)[1] -= 15;
		SVvector (ent, absmax)[0] += 15;
		SVvector (ent, absmax)[1] += 15;
	} else {	// movement is clipped an epsilon away from actual edge, so we
				// must fully check even when bounding boxes don't quite touch
		SVvector (ent, absmin)[0] -= 1;
		SVvector (ent, absmin)[1] -= 1;
		SVvector (ent, absmin)[2] -= 1;
		SVvector (ent, absmax)[0] += 1;
		SVvector (ent, absmax)[1] += 1;
		SVvector (ent, absmax)[2] += 1;
	}

	// link to PVS leafs
	ent->num_leafs = 0;
	if (SVfloat (ent, modelindex))
		SV_FindTouchedLeafs (ent, sv.worldmodel->nodes);

	if (SVfloat (ent, solid) == SOLID_NOT)
		return;

	// find the first node that the ent's box crosses
	node = sv_areanodes;
	while (1) {
		if (node->axis == -1)
			break;
		if (SVvector (ent, absmin)[node->axis] > node->dist)
			node = node->children[0];
		else if (SVvector (ent, absmax)[node->axis] < node->dist)
			node = node->children[1];
		else
			break;						// crosses the node
	}

	// link it in
	if (SVfloat (ent, solid) == SOLID_TRIGGER)
		InsertLinkBefore (&ent->area, &node->trigger_edicts);
	else
		InsertLinkBefore (&ent->area, &node->solid_edicts);

	// if touch_triggers, touch all entities at this node and descend for more
	if (touch_triggers)
		SV_TouchLinks (ent, sv_areanodes);
}

/* POINT TESTING IN HULLS */

#ifndef USE_INTEL_ASM
int
SV_HullPointContents (hull_t *hull, int num, const vec3_t p)
{
	float        d;
	dclipnode_t *node;
	mplane_t    *plane;

	while (num >= 0) {
		if (num < hull->firstclipnode || num > hull->lastclipnode)
			Sys_Error ("SV_HullPointContents: bad node number");

		node = hull->clipnodes + num;
		plane = hull->planes + node->planenum;

		if (plane->type < 3)
			d = p[plane->type] - plane->dist;
		else
			d = DotProduct (plane->normal, p) - plane->dist;
		if (d < 0)
			num = node->children[1];
		else
			num = node->children[0];
	}

	return num;
}
#endif // !USE_INTEL_ASM

int
SV_PointContents (const vec3_t p)
{
	int         cont;

	cont = SV_HullPointContents (&sv.worldmodel->hulls[0], 0, p);
	if (cont <= CONTENTS_CURRENT_0 && cont >= CONTENTS_CURRENT_DOWN)
		cont = CONTENTS_WATER;
	return cont;
}

int
SV_TruePointContents (const vec3_t p)
{
	return SV_HullPointContents (&sv.worldmodel->hulls[0], 0, p);
}

/*
	SV_TestEntityPosition

	This could be a lot more efficient...
	A small wrapper around SV_BoxInSolidEntity that never clips against the
	supplied entity.
*/
edict_t *
SV_TestEntityPosition (edict_t *ent)
{
	trace_t		trace;

	trace =	SV_Move (SVvector (ent, origin),
					 SVvector (ent, mins),
					 SVvector (ent, maxs),
					 SVvector (ent, origin), 0, ent);

	if (trace.startsolid)
		return sv.edicts;
	return NULL;
}

/* LINE TESTING IN HULLS */

// 1/32 epsilon to keep floating point happy
#define	DIST_EPSILON	(0.03125)

qboolean
SV_RecursiveHullCheck (hull_t *hull, int num, float p1f, float p2f,
					   const vec3_t p1, const vec3_t p2, trace_t *trace)
{
	dclipnode_t *node;
	float       frac, midf, t1, t2;
	int         side, i;
	mplane_t   *plane;
	vec3_t      mid;

	// check for empty
	if (num < 0) {
		if (num != CONTENTS_SOLID) {
			trace->allsolid = false;
			if (num == CONTENTS_EMPTY)
				trace->inopen = true;
			else
				trace->inwater = true;
		} else
			trace->startsolid = true;
		return true;					// empty
	}

	// find the point distances
	node = hull->clipnodes + num;
	plane = hull->planes + node->planenum;

	if (plane->type < 3) {
		t1 = p1[plane->type] - plane->dist;
		t2 = p2[plane->type] - plane->dist;
	} else {
		t1 = DotProduct (plane->normal, p1) - plane->dist;
		t2 = DotProduct (plane->normal, p2) - plane->dist;
	}

	if (t1 >= 0 && t2 >= 0)
		return SV_RecursiveHullCheck (hull, node->children[0], p1f, p2f, p1,
									  p2, trace);
	if (t1 < 0 && t2 < 0)
		return SV_RecursiveHullCheck (hull, node->children[1], p1f, p2f, p1,
									  p2, trace);

	side = (t1 < 0);
	frac = t1 / (t1 - t2);
	//frac = bound (0, frac, 1); // is this needed?

	midf = p1f + (p2f - p1f) * frac;
	for (i = 0; i < 3; i++)
		mid[i] = p1[i] + frac * (p2[i] - p1[i]);

	// move up to the node
	if (!SV_RecursiveHullCheck (hull, node->children[side],
								p1f, midf, p1, mid, trace))
		return false;

	if (SV_HullPointContents (hull, node->children[side ^ 1], mid)
		!= CONTENTS_SOLID) {
		// go past the node
		return SV_RecursiveHullCheck (hull, node->children[side ^ 1], midf,
									  p2f, mid, p2, trace);
	}

	if (trace->allsolid)
		return false;					// never got out of the solid area

	// the other side of the node is solid, this is the impact point
	if (!side) {
		VectorCopy (plane->normal, trace->plane.normal);
		trace->plane.dist = plane->dist;
	} else {
		// invert plane paramterers
		trace->plane.normal[0] = -plane->normal[0];
		trace->plane.normal[1] = -plane->normal[1];
		trace->plane.normal[2] = -plane->normal[2];
		trace->plane.dist = -plane->dist;
	}

	// put the crosspoint DIST_EPSILON pixels on the near side to guarantee
	// mid is on the correct side of the plane
	if (side)
		frac = (t1 + DIST_EPSILON) / (t1 - t2);
	else
		frac = (t1 - DIST_EPSILON) / (t1 - t2);
	frac = bound (0, frac, 1);

	midf = p1f + (p2f - p1f) * frac;
	for (i = 0; i < 3; i++)
		mid[i] = p1[i] + frac * (p2[i] - p1[i]);

	trace->fraction = midf;
	VectorCopy (mid, trace->endpos);

	return false;
}

/*
	SV_ClipMoveToEntity

	Handles selection or creation of a clipping hull, and offseting (and
	eventually rotation) of the end points
*/
static trace_t
SV_ClipMoveToEntity (edict_t *touched, edict_t *mover, const vec3_t start,
					 const vec3_t mins, const vec3_t maxs, const vec3_t end)
{
	hull_t     *hull;
	trace_t     trace;
	vec3_t      offset, start_l, end_l;

	// fill in a default trace
	memset (&trace, 0, sizeof (trace_t));

	trace.fraction = 1;
	trace.allsolid = true;
	VectorCopy (end, trace.endpos);

	// get the clipping hull
	hull = SV_HullForEntity (touched, mins, maxs, offset);

	VectorSubtract (start, offset, start_l);
	VectorSubtract (end, offset, end_l);

	// trace a line through the apropriate clipping hull
	SV_RecursiveHullCheck (hull, hull->firstclipnode, 0, 1, start_l, end_l,
						   &trace);

	// fix trace up by the offset
	if (trace.fraction != 1)
		VectorAdd (trace.endpos, offset, trace.endpos);

	// did we clip the move?
	if (trace.fraction < 1 || trace.startsolid)
		trace.ent = touched;

	return trace;
}

/*
	SV_ClipToLinks

	Mins and maxs enclose the entire area swept by the move
*/
static void
SV_ClipToLinks (areanode_t *node, moveclip_t * clip)
{
	edict_t    *touch;
	link_t     *l, *next;
	trace_t		trace;

	// touch linked edicts
	for (l = node->solid_edicts.next; l != &node->solid_edicts; l = next) {
		next = l->next;
		touch = EDICT_FROM_AREA (l);
		if (SVfloat (touch, solid) == SOLID_NOT)
			continue;
		if (touch == clip->passedict)
			continue;
		if (SVfloat (touch, solid) == SOLID_TRIGGER)
			Sys_Error ("Trigger in clipping list");

		if (clip->type == MOVE_NOMONSTERS && SVfloat (touch, solid)
			!= SOLID_BSP)
			continue;

		if (clip->boxmins[0] > SVvector (touch, absmax)[0]
			|| clip->boxmins[1] > SVvector (touch, absmax)[1]
			|| clip->boxmins[2] > SVvector (touch, absmax)[2]
			|| clip->boxmaxs[0] < SVvector (touch, absmin)[0]
			|| clip->boxmaxs[1] < SVvector (touch, absmin)[1]
			|| clip->boxmaxs[2] < SVvector (touch, absmin)[2])
			continue;

		if (clip->passedict != 0 && SVvector (clip->passedict, size)[0]
			&& !SVvector (touch, size)[0])
			continue;					// points never interact

		// might intersect, so do an exact clip
		if (clip->trace.allsolid)
			return;
		if (clip->passedict) {
			if (PROG_TO_EDICT (&sv_pr_state, SVentity (touch, owner))
				== clip->passedict)
				continue;				// don't clip against own missiles
			if (PROG_TO_EDICT (&sv_pr_state,
							   SVentity (clip->passedict, owner)) == touch)
				continue;				// don't clip against owner
		}

		if ((int) SVfloat (touch, flags) & FL_MONSTER)
			trace = SV_ClipMoveToEntity (touch, clip->passedict, clip->start,
										 clip->mins2, clip->maxs2, clip->end);
		else
			trace = SV_ClipMoveToEntity (touch, clip->passedict, clip->start,
										 clip->mins, clip->maxs, clip->end);
		if (trace.allsolid || trace.startsolid
			|| trace.fraction < clip->trace.fraction) {
			trace.ent = touch;
			if (clip->trace.startsolid) {
				clip->trace = trace;
				clip->trace.startsolid = true;
			} else
				clip->trace = trace;
		} else if (trace.startsolid)
			clip->trace.startsolid = true;
	}

	// recurse down both sides
	if (node->axis == -1)
		return;

	if (clip->boxmaxs[node->axis] > node->dist)
		SV_ClipToLinks (node->children[0], clip);
	if (clip->boxmins[node->axis] < node->dist)
		SV_ClipToLinks (node->children[1], clip);
}

static void
SV_MoveBounds (const vec3_t start, const vec3_t mins, const vec3_t maxs,
			   const vec3_t end, vec3_t boxmins, vec3_t boxmaxs)
{
#if 0
	// debug to test against everything
	boxmins[0] = boxmins[1] = boxmins[2] = -9999;
	boxmaxs[0] = boxmaxs[1] = boxmaxs[2] = 9999;
#else
	int			i;

	for (i = 0; i < 3; i++) {
		if (end[i] > start[i]) {
			boxmins[i] = start[i] + mins[i] - 1;
			boxmaxs[i] = end[i] + maxs[i] + 1;
		} else {
			boxmins[i] = end[i] + mins[i] - 1;
			boxmaxs[i] = start[i] + maxs[i] + 1;
		}
	}
#endif
}

trace_t
SV_Move (const vec3_t start, const vec3_t mins, const vec3_t maxs,
		 const vec3_t end, int type, edict_t *passedict)
{
	int			i;
	moveclip_t  clip;

	memset (&clip, 0, sizeof (moveclip_t));

	// clip to world
	clip.trace = SV_ClipMoveToEntity (sv.edicts, passedict, start,
									  mins, maxs, end);

	clip.start = start;
	clip.end = end;
	clip.mins = mins;
	clip.maxs = maxs;
	clip.type = type;
	clip.passedict = passedict;

	if (type == MOVE_MISSILE) {
		for (i = 0; i < 3; i++) {
			clip.mins2[i] = -15;
			clip.maxs2[i] = 15;
		}
	} else {
		VectorCopy (mins, clip.mins2);
		VectorCopy (maxs, clip.maxs2);
	}

	// create the bounding box of the entire move
	SV_MoveBounds (start, clip.mins2, clip.maxs2, end, clip.boxmins,
				   clip.boxmaxs);

	// clip to entities
	SV_ClipToLinks (sv_areanodes, &clip);

	return clip.trace;
}

edict_t *
SV_TestPlayerPosition (edict_t *ent, const vec3_t origin)
{
	edict_t    *check;
	hull_t     *hull;
	int         e;
	vec3_t      boxmins, boxmaxs, offset;

	// check world first
	hull = &sv.worldmodel->hulls[1];
	if (SV_HullPointContents (hull, hull->firstclipnode, origin) !=
		CONTENTS_EMPTY) return sv.edicts;

	// check all entities
	VectorAdd (origin, SVvector (ent, mins), boxmins);
	VectorAdd (origin, SVvector (ent, maxs), boxmaxs);

	check = NEXT_EDICT (&sv_pr_state, sv.edicts);
	for (e = 1; e < sv.num_edicts; e++, check = NEXT_EDICT (&sv_pr_state,
															check)) {
		if (check->free)
			continue;
		if (check == ent)
			continue;

		if (SVfloat (check, solid) != SOLID_BSP
			&& SVfloat (check, solid) != SOLID_BBOX
			&& SVfloat (check, solid) != SOLID_SLIDEBOX)
			continue;

		if (boxmins[0] > SVvector (check, absmax)[0]
			|| boxmins[1] > SVvector (check, absmax)[1]
			|| boxmins[2] > SVvector (check, absmax)[2]
			|| boxmaxs[0] < SVvector (check, absmin)[0]
			|| boxmaxs[1] < SVvector (check, absmin)[1]
			|| boxmaxs[2] < SVvector (check, absmin)[2])
			continue;

		// get the clipping hull
		hull = SV_HullForEntity (check, SVvector (ent, mins),
								 SVvector (ent, maxs), offset);

		VectorSubtract (origin, offset, offset);

		// test the point
		if (SV_HullPointContents (hull, hull->firstclipnode, offset) !=
			CONTENTS_EMPTY)
			return check;
	}

	return NULL;
}
