/*
* $Id: nanotimer.c,v 1.2 2003/12/19 14:49:55 rgb Exp $
*
* See copyright in copyright.h and the accompanying file COPYING
*
*/

/*
 *========================================================================
 * This is a portable nanosecond timer.  It uses gettimeofday (wall clock
 * time) with the time of the first call subtracted off to keep intervals
 * from horribly overflowing the double with irrelevant numbers (causing
 * a loss of precision).  Note that my direct measurements show that
 * gettimeofday() itself takes about 2 usec to complete, so accuracy is
 * NOT to nanoseconds at all.
 *========================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

static struct timeval tv_first = {0,0};

double gtod_nanotimer()
{

 struct timeval tv_now;
 double nanotime;

 /*
  * This avoids potential precision problems by computing the starting
  * time as of the first call, and measuring all subsequent times
  * relative to this.  Gets rid of a LOT of seconds.
  */
 if((tv_first.tv_sec == 0) && (tv_first.tv_usec == 0)){
    gettimeofday(&tv_first, (struct timezone *) NULL);
 }
 
 gettimeofday(&tv_now, (struct timezone *) NULL);
 nanotime = (double)(tv_now.tv_sec - tv_first.tv_sec);
 nanotime += 1.0e-6*(double)(tv_now.tv_usec - tv_first.tv_usec);

 /* Cruft
 printf("tvnsec = %u tvfsec = %u tvnusec = %u tvfusec = %u\n",
    tv_now.tv_sec,tv_first.tv_sec,tv_now.tv_usec,tv_first.tv_usec);
 printf("nanotime = %20.10f\n",nanotime);
 */

 /* return nanoseconds */	  
 nanotime *= 1.e+9;
 /* Cruft
 printf("nanotime = %20.10f\n",nanotime);
 */
 return(nanotime);
 
}

static double nsec_per_cycle = 0.0;
static unsigned long ax_first,dx_first;
static unsigned long long count_first;

double init_nanotimer()
{

 int i,numfields;
 char statbuf[1024];
 char delim[2],*nextval;
 FILE *cpuinfo_fd;
 double mhz,my_nsec_per_cycle;

 delim[0] = ':';                /* separator */
 delim[1] = (char) NULL;        /* string terminator */
 cpuinfo_fd = fopen("/proc/cpuinfo","r");
 while(-1){

   /* Normal EOF causes break from while loop */
   if((fgets(statbuf,1024,cpuinfo_fd) == NULL)) break;

   if(strncmp(statbuf,"cpu MHz",7) == 0) {
     nextval = strtok(statbuf,delim);       /* first field skip */
     nextval = strtok((char *)NULL,delim);  /* second field is it */
     mhz = strtod(&nextval[1],0);
     my_nsec_per_cycle = 1000.0/mhz;
     break;
   }
 }

 fclose(cpuinfo_fd);

 if(my_nsec_per_cycle == 0.0){
   fprintf(stderr,"Error: Cannot parse out the cpu MHz from /proc/cpuinfo.\n");
   exit(0);
 }

 return(my_nsec_per_cycle);

}

void my_nanosleep(unsigned long nanodelay)
{
 unsigned long ax, dx;
 unsigned long long count,cycledelay;
 double nanostart,nanotime,nanodelta;

 if(nsec_per_cycle == 0.0) {
   nsec_per_cycle = init_nanotimer();
 }
 cycledelay = nanodelay/nsec_per_cycle;
 /*
  * We subtract off the time base to ensure that times of
  * order seconds (and then some) have resolvable differences
  * in double precision.
  */
 asm volatile("rdtsc" : "=a" (ax_first), "=d" (dx_first));
 count_first = dx_first;
 count_first = count_first<<32;
 count_first += ax_first;

 count = 0;
 while(count < cycledelay){
   asm volatile("rdtsc" : "=a" (ax), "=d" (dx));
   count = dx;
   count = count<<32;
   count += ax;
   count -= count_first;
 }
 
}

double nanotimer()
{
 unsigned long ax, dx;
 unsigned long long count;
 double my_nanotime;

 if(nsec_per_cycle == 0.0) {
   nsec_per_cycle = init_nanotimer();
   /* Cruft
   printf("init_nanotimer: nsec_per_cycle = %f\n",nsec_per_cycle);
   */
   /*
    * We subtract off the time base to ensure that times of
    * order seconds (and then some) have resolvable differences
    * in double precision.
    */
   asm volatile("rdtsc" : "=a" (ax_first), "=d" (dx_first));
   count_first = dx_first;
   count_first = count_first<<32;
   count_first += ax_first;
 }
 asm volatile("rdtsc" : "=a" (ax), "=d" (dx));
 count = dx;
 count = count<<32;
 count += ax;
 count -= count_first;

 my_nanotime = (double) count;
 /* Cruft
 printf("count = %f\n",my_nanotime);
 printf("init_nanotimer 2: nsec_per_cycle = %f\n",nsec_per_cycle);
 */
 my_nanotime *= nsec_per_cycle;
 /* Cruft
 printf("my_nanotime = %f\n",my_nanotime);
 */

 return(my_nanotime);
 
}

