#!/bin/env python
#Copyright ReportLab Europe Ltd. 2000-2017
#see license.txt for license details
__version__='3.3.0'
__doc__='Test script for reportlab.tables'
from reportlab.lib.testutils import setOutDir,makeSuiteForClasses, outputfile, printLocation
setOutDir(__name__)
import os,unittest
from reportlab.platypus import Spacer, SimpleDocTemplate, Table, TableStyle, FrameBG
from reportlab.platypus.paragraph import Paragraph
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch, cm
from reportlab.lib import colors
from reportlab.graphics.charts.linecharts import HorizontalLineChart
from reportlab.graphics.shapes import Drawing, _DrawingEditorMixin
from reportlab.graphics.charts.barcharts import VerticalBarChart

styleSheet = getSampleStyleSheet()

def getTable():
    t = Table((('','North','South','East','West'),
             ('Quarter 1',100,200,300,400),
             ('Quarter 2',100,400,600,800),
             ('Total',300,600,900,'1,200')),
             (72,36,36,36,36),
             (24, 16,16,18)
            )
    return t

def makeStyles():
    styles = []
    for i in range(5):
        styles.append(TableStyle([('ALIGN', (1,1), (-1,-1), 'RIGHT'),
                                  ('ALIGN', (0,0), (-1,0), 'CENTRE'),
                                  ('HREF', (0,0), (0,0), 'www.google.com'),
                                  ]))
    for style in styles[1:]:
        style.add('GRID', (0,0), (-1,-1), 0.25, colors.black)
    for style in styles[2:]:
        style.add('LINEBELOW', (0,0), (-1,0), 2, colors.black)
    for style in styles[3:]:
        style.add('LINEABOVE', (0, -1), (-1,-1), 2, colors.black)
    styles[-1].add('LINEBELOW',(1,-1), (-1, -1), 2, (0.5, 0.5, 0.5))
    return styles

def run():
    doc = SimpleDocTemplate(outputfile('test_platypus_tables.pdf'), pagesize=(8.5*inch, 11*inch), showBoundary=1)
    lst = []
    from reportlab import Version
    styNormal = styleSheet['Normal']
    styBackground = ParagraphStyle('background', parent=styNormal, backColor=colors.pink)
    styH1 = styleSheet['Heading1']
    lst.append(FrameBG(color=colors.red))
    lst.append(Paragraph("First, a test of how tables align their content...", styH1))
    lst.append(Paragraph("""Generated with version %s""" % Version,
                        styNormal))
    lst.append(Paragraph("""In release 2.3, cells with plain text positioned their
                         text differently to cells with Paragraphs using the
                         same font.  Hopefully now they are back on the same baseline""",
                        styNormal))
    lst.append(FrameBG(color=colors.blue))
    ts1 = TableStyle([
                ('ALIGN', (0,0), (-1,0), 'RIGHT'),
                ('BACKGROUND', (0,0), (-1,0), colors.lightgrey),
                ('VALIGN', (0,0), (-1,-1), 'TOP'),
                ('GRID', (0,0), (-1,-1), 0.25, colors.black),
                    ])
    t1 = Table([
        ('plain text','plain text','shortpara','plain text', 'long para'),
        ('Text','more text', Paragraph('Is this para level?', styBackground), 'Back to text', Paragraph('Short para again', styBackground)),
        ('Text',
            'more text',
            Paragraph('Is this level?', styBackground),
            'This is plain\ntext with line breaks\nto compare against\nthe para on right',
            Paragraph('Long paragraph we expect to wrap over several lines accurately', styBackground)),

        ])
    t1.setStyle(ts1)
    lst.append(t1)
    lst.append(FrameBG(start=False))
    lst.append(Spacer(0,10))
    lst.append(Paragraph("Now we make a table with just one cell containing a string...note how the text sits low", styNormal))
    lst.append(FrameBG(start=False))

    tsGrid = TableStyle([
                ('GRID', (0,0), (-1,-1), 0.25, colors.black),
                    ])
    lst.append(Table([['One cell of plain text']], style=tsGrid, colWidths=[200]))

    lst.append(Spacer(0,10))
    lst.append(Paragraph("Now we make a table with just one cell containing a para...should be same position.  Note that the overall bounding box is an approximation and lies - it always did.", styNormal))
    lst.append(Table([[Paragraph('One cell containing a paragraph.  &#196;&#201;&#8747;', styBackground)]], style=tsGrid, colWidths=[200]))

    lst.append(Spacer(0,10))
    lst.append(Paragraph("Paragraphs jumped up post 2.1.  Ideally they should align the same.", styNormal))


    lst.append(Spacer(0,30))
    lst.append(Paragraph("Now for all the tests we had before.  See also the much longer test_platypus_tables_2.pdf, which for reasons unknown was split into a separate file generated by the same script", styNormal))

    styles = makeStyles()
    for style in styles:
        t = getTable()
        t.setStyle(style)
##        print '--------------'
##        for rowstyle in t._cellstyles:
##            for s in rowstyle:
##                print s.alignment
        lst.append(t)
        lst.append(Spacer(0,12))

    t=Table([['VERTICAL Gradient Red top, grey bottom','Horizontal Gradient Blue left, green right'],
             ['HORIZONTAL Gradient Span grey left red right', ''],
             ['VERTICAL Gradiant Span Blue top green bottom',''],
             ['','CLEAR']],[3.5*inch, 2.7*inch])
    style=TableStyle([
    ('SPAN', (0,1),(1,1)),
    ('SPAN', (0,2),(0,3)),
    ('BACKGROUND',(0,0), (0,0),['VERTICAL', colors.grey, colors.red]),
    ('BACKGROUND',(1,0), (1,0),['HORIZONTAL', colors.blue, colors.green]),
    ('BACKGROUND',(0,1), (1,1),['HORIZONTAL', colors.grey, colors.red]),
    ('BACKGROUND',(0,2), (0,3),['VERTICAL', colors.blue, colors.green])
    ])
    t.setStyle(style)
    lst.append(t)

    #illustrate usage of minRowHeights idea from Jon Hinton inivatajon @ bitbucket.org
    t=Table([['VERTICAL Red --> grey  minRowHeights[0]=30','Horizontal Gradient Blue left, green right'],
             ['HORIZONTAL Gradient Span grey left red right', ''],
             ['VERTICAL Gradiant Span Blue top green bottom',''],
             ['','CLEAR']],[3.5*inch, 2.7*inch],minRowHeights=(30,), spaceBefore=15)
    style=TableStyle([
    ('SPAN', (0,1),(1,1)),
    ('SPAN', (0,2),(0,3)),
    ('VALIGN', (0,0),(-1,0),'MIDDLE'),
    ('BACKGROUND',(0,0), (0,0),['VERTICAL', colors.grey, colors.red]),
    ('BACKGROUND',(1,0), (1,0),['HORIZONTAL', colors.blue, colors.green]),
    ('BACKGROUND',(0,1), (1,1),['HORIZONTAL', colors.grey, colors.red]),
    ('BACKGROUND',(0,2), (0,3),['VERTICAL', colors.blue, colors.green])
    ])
    t.setStyle(style)

    t=Table([],[3.5*inch, 2.7*inch],minRowHeights=(30,), spaceBefore=15, style=style, emptyTableAction='ignore')
    lst.append(t)
    doc.build(lst)

class TableBarChart(_DrawingEditorMixin,Drawing):
    def __init__(self,width=400,height=200,*args,**kw):
        Drawing.__init__(self,width,height,*args,**kw)
        self.width = 136
        self.height = 140
        self._add(self,VerticalBarChart(),name='chart',validate=None,desc=None)
        self.chart.y = 20
        self.chart.width = self.width - 21
        self.chart.height = self.height - 24
        self.chart.categoryAxis.categoryNames = ['Spring','Summer','Autumn','Winter']
        self.chart.categoryAxis.labels.fontSize = 7

def old_tables_test():
    from reportlab.lib.units import inch, cm
    from reportlab.platypus.flowables import Image, PageBreak, Spacer, XBox
    from reportlab.platypus.paragraph import Paragraph
    from reportlab.platypus.xpreformatted import XPreformatted
    from reportlab.platypus.flowables import Preformatted
    from reportlab.platypus.doctemplate import SimpleDocTemplate
    from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
    from reportlab.platypus.tables import GRID_STYLE, BOX_STYLE, LABELED_GRID_STYLE, COLORED_GRID_STYLE, LIST_STYLE, LongTable
    rowheights = (24, 16, 16, 16, 16)
    rowheights2 = (24, 16, 16, 16, 30)
    colwidths = (50, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32)
    data = (
        ('', 'Jan', 'Feb', 'Mar','Apr','May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'),
        ('Mugs', 0, 4, 17, 3, 21, 47, 12, 33, 2, -2, 44, 89),
        ('T-Shirts', 0, 42, 9, -3, 16, 4, 72, 89, 3, 19, 32, 119),
        ('Key Ring', 0,0,0,0,0,0,1,0,0,0,2,13),
        ('Hats', 893, 912, '1,212', 643, 789, 159, 888, '1,298', 832, 453, '1,344','2,843')
        )
    data2 = (
        ('', 'Jan', 'Feb', 'Mar','Apr','May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'),
        ('Mugs', 0, 4, 17, 3, 21, 47, 12, 33, 2, -2, 44, 89),
        ('T-Shirts', 0, 42, 9, -3, 16, 4, 72, 89, 3, 19, 32, 119),
        ('Key Ring', 0,0,0,0,0,0,1,0,0,0,2,13),
        ('Hats\nLarge', 893, 912, '1,212', 643, 789, 159, 888, '1,298', 832, 453, '1,344','2,843')
        )
    lst = []
    lst_add = lst.append
    lst_add(Paragraph("Tables", styleSheet['Heading1']))
    lst_add(Paragraph(__doc__, styleSheet['BodyText']))
    lst_add(Paragraph("The Tables (shown in different styles below) were created using the following code:", styleSheet['BodyText']))
    lst_add(Preformatted("""
    colwidths = (50, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32)
    rowheights = (24, 16, 16, 16, 16)
    data = (
        ('', 'Jan', 'Feb', 'Mar','Apr','May', 'Jun',
           'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'),
        ('Mugs', 0, 4, 17, 3, 21, 47, 12, 33, 2, -2, 44, 89),
        ('T-Shirts', 0, 42, 9, -3, 16, 4, 72, 89, 3, 19, 32, 119),
        ('Key Ring', 0,0,0,0,0,0,1,0,0,0,2,13),
        ('Hats', 893, 912, '1,212', 643, 789, 159,
             888, '1,298', 832, 453, '1,344','2,843')
        )
    t = Table(data, colwidths, rowheights)
    """, styleSheet['Code'], dedent=4))
    lst_add(Paragraph("""
    You can then give the Table a TableStyle object to control its format. The first TableStyle used was
    created as follows:
    """, styleSheet['BodyText']))
    lst_add(Preformatted("""
GRID_STYLE = TableStyle(
    [('GRID', (0,0), (-1,-1), 0.25, colors.black),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT')]
    )
    """, styleSheet['Code']))
    lst_add(Paragraph("""
    TableStyles are created by passing in a list of commands. There are two types of commands - line commands
    and cell formatting commands. In all cases, the first three elements of a command are the command name,
    the starting cell and the ending cell.
    """, styleSheet['BodyText']))
    lst_add(Paragraph("""
    Line commands always follow this with the weight and color of the desired lines. Colors can be names,
    or they can be specified as a (R,G,B) tuple, where R, G and B are floats and (0,0,0) is black. The line
    command names are: GRID, BOX, OUTLINE, INNERGRID, LINEBELOW, LINEABOVE, LINEBEFORE
    and LINEAFTER. BOX and OUTLINE are equivalent, and GRID is the equivalent of applying both BOX and
    INNERGRID.
    """, styleSheet['BodyText']))
    lst_add(Paragraph("""
    Cell formatting commands are:
    """, styleSheet['BodyText']))
    lst_add(Paragraph("""
    FONT - takes fontname, fontsize and (optional) leading.
    """, styleSheet['Definition']))
    lst_add(Paragraph("""
    TEXTCOLOR - takes a color name or (R,G,B) tuple.
    """, styleSheet['Definition']))
    lst_add(Paragraph("""
    ALIGNMENT (or ALIGN) - takes one of LEFT, RIGHT, CENTRE (or CENTER) or DECIMAL.
    """, styleSheet['Definition']))
    lst_add(Paragraph("""
    LEFTPADDING - defaults to 6.
    """, styleSheet['Definition']))
    lst_add(Paragraph("""
    RIGHTPADDING - defaults to 6.
    """, styleSheet['Definition']))
    lst_add(Paragraph("""
    BOTTOMPADDING - defaults to 3.
    """, styleSheet['Definition']))
    lst_add(Paragraph("""
    A tablestyle is applied to a table by calling Table.setStyle(tablestyle).
    """, styleSheet['BodyText']))
    t = Table(data, colwidths, rowheights)
    t.setStyle(GRID_STYLE)
    lst_add(PageBreak())
    lst_add(Paragraph("This is GRID_STYLE\n", styleSheet['BodyText']))
    lst_add(t)

    t = Table(data, colwidths, rowheights)
    t.setStyle(BOX_STYLE)
    lst_add(Paragraph("This is BOX_STYLE\n", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Paragraph("""
    It was created as follows:
    """, styleSheet['BodyText']))
    lst_add(Preformatted("""
BOX_STYLE = TableStyle(
    [('BOX', (0,0), (-1,-1), 0.50, colors.black),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT')]
    )
    """, styleSheet['Code']))

    t = Table(data, colwidths, rowheights)
    t.setStyle(LABELED_GRID_STYLE)
    lst_add(Paragraph("This is LABELED_GRID_STYLE\n", styleSheet['BodyText']))
    lst_add(t)
    t = Table(data2, colwidths, rowheights2)
    t.setStyle(LABELED_GRID_STYLE)
    lst_add(Paragraph("This is LABELED_GRID_STYLE ILLUSTRATES EXPLICIT LINE SPLITTING WITH NEWLINE (different heights and data)\n", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Paragraph("""
    It was created as follows:
    """, styleSheet['BodyText']))
    lst_add(Preformatted("""
LABELED_GRID_STYLE = TableStyle(
    [('INNERGRID', (0,0), (-1,-1), 0.25, colors.black),
     ('BOX', (0,0), (-1,-1), 2, colors.black),
     ('LINEBELOW', (0,0), (-1,0), 2, colors.black),
     ('LINEAFTER', (0,0), (0,-1), 2, colors.black),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT')]
    )
    """, styleSheet['Code']))
    lst_add(PageBreak())

    t = Table(data, colwidths, rowheights)
    t.setStyle(COLORED_GRID_STYLE)
    lst_add(Paragraph("This is COLORED_GRID_STYLE\n", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Paragraph("""
    It was created as follows:
    """, styleSheet['BodyText']))
    lst_add(Preformatted("""
COLORED_GRID_STYLE = TableStyle(
    [('INNERGRID', (0,0), (-1,-1), 0.25, colors.black),
     ('BOX', (0,0), (-1,-1), 2, colors.red),
     ('LINEBELOW', (0,0), (-1,0), 2, colors.black),
     ('LINEAFTER', (0,0), (0,-1), 2, colors.black),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT')]
    )
    """, styleSheet['Code']))

    t = Table(data, colwidths, rowheights)
    t.setStyle(LIST_STYLE)
    lst_add(Paragraph("This is LIST_STYLE\n", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Paragraph("""
    It was created as follows:
    """, styleSheet['BodyText']))
    lst_add(Preformatted("""
LIST_STYLE = TableStyle(
    [('LINEABOVE', (0,0), (-1,0), 2, colors.green),
     ('LINEABOVE', (0,1), (-1,-1), 0.25, colors.black),
     ('LINEBELOW', (0,-1), (-1,-1), 2, colors.green),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT')]
    )
    """, styleSheet['Code']))

    t = Table(data, colwidths, rowheights)
    ts = TableStyle(
    [('LINEABOVE', (0,0), (-1,0), 2, colors.green),
     ('LINEABOVE', (0,1), (-1,-1), 0.25, colors.black),
     ('LINEBELOW', (0,-1), (-1,-1), 3, colors.green,'butt'),
     ('LINEBELOW', (0,-1), (-1,-1), 1, colors.white,'butt'),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT'),
     ('TEXTCOLOR', (0,1), (0,-1), colors.red),
     ('BACKGROUND', (0,0), (-1,0), colors.Color(0,0.7,0.7))]
    )
    t.setStyle(ts)
    lst_add(Paragraph("This is a custom style\n", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Paragraph("""
    It was created as follows:
    """, styleSheet['BodyText']))
    lst_add(Preformatted("""
   ts = TableStyle(
    [('LINEABOVE', (0,0), (-1,0), 2, colors.green),
     ('LINEABOVE', (0,1), (-1,-1), 0.25, colors.black),
     ('LINEBELOW', (0,-1), (-1,-1), 3, colors.green,'butt'),
     ('LINEBELOW', (0,-1), (-1,-1), 1, colors.white,'butt'),
     ('ALIGN', (1,1), (-1,-1), 'RIGHT'),
     ('TEXTCOLOR', (0,1), (0,-1), colors.red),
     ('BACKGROUND', (0,0), (-1,0), colors.Color(0,0.7,0.7))]
    )
    """, styleSheet['Code']))
    data = (
        ('', 'Jan\nCold', 'Feb\n', 'Mar\n','Apr\n','May\n', 'Jun\nHot', 'Jul\n', 'Aug\nThunder', 'Sep\n', 'Oct\n', 'Nov\n', 'Dec\n'),
        ('Mugs', 0, 4, 17, 3, 21, 47, 12, 33, 2, -2, 44, 89),
        ('T-Shirts', 0, 42, 9, -3, 16, 4, 72, 89, 3, 19, 32, 119),
        ('Key Ring', 0,0,0,0,0,0,1,0,0,0,2,13),
        ('Hats', 893, 912, '1,212', 643, 789, 159, 888, '1,298', 832, 453, '1,344','2,843')
        )
    c = list(colwidths)
    c[0] = None
    c[8] = None
    t = Table(data, c, [None]+list(rowheights[1:]))
    t.setStyle(LIST_STYLE)
    lst_add(Paragraph("""
        This is a LIST_STYLE table with the first rowheight set to None ie automatic.
        The top row cells are split at a newline '\\n' character. The first and August
        column widths were also set to None.
    """, styleSheet['BodyText']))
    lst_add(t)

    lst_add(Paragraph("""
        This demonstrates a number of features useful in financial statements. The first is decimal alignment;
        with ALIGN=DECIMAL the numbers align on the points; and the points are aligned based on
        the RIGHTPADDING, which is usually 3 points so you should set it higher.  The second is multiple lines;
        one can specify double or triple lines and control the separation if desired. Finally, the coloured
        negative numbers were (we regret to say) done in the style; we don't have a way to conditionally
        format numbers based on value yet.
    """, styleSheet['BodyText']))


    t = Table([['Corporate Assets','Amount'],
               ['Fixed Assets','1,234,567.89'],
               ['Company Vehicle','1,234.8901'],
               ['Petty Cash','42'],
               [u'Intellectual Property\u00ae','(42,078,231.56)'],
               ['Overdraft','(12,345)'],
               ['Boardroom Flat Screen','60 inches'],
               ['Net Position','Deep Sh*t.Really']
               ],
              [144,72])

    ts = TableStyle(
        [#first the top row
         ('ALIGN', (1,1), (-1,-1), 'CENTER'),
         ('LINEABOVE', (0,0), (-1,0), 1, colors.purple),
         ('LINEBELOW', (0,0), (-1,0), 1, colors.purple),
         ('FONT', (0,0), (-1,0), 'Times-Bold'),

        #bottom row has a line above, and two lines below
         ('LINEABOVE', (0,-1), (-1,-1), 1, colors.purple),  #last 2 are count, sep
         ('LINEBELOW', (0,-1), (-1,-1), 0.5, colors.purple, 1, None, None, 4,1),
         ('LINEBELOW', (0,-1), (-1,-1), 1, colors.red),
         ('FONT', (0,-1), (-1,-1), 'Times-Bold'),

        #numbers column
         ('ALIGN', (1,1), (-1,-1), 'DECIMAL'),
         ('RIGHTPADDING', (1,1), (-1,-1), 36),
         ('TEXTCOLOR', (1,4), (1,4), colors.red),

        #red cell
        ]
        )

    t.setStyle(ts)
    lst_add(t)
    lst_add(Spacer(36,36))
    lst_add(Paragraph("""
        The red numbers should be aligned LEFT &amp; BOTTOM, the blue RIGHT &amp; TOP
        and the green CENTER &amp; MIDDLE.
    """, styleSheet['BodyText']))
    XY  =   [['X00y', 'X01y', 'X02y', 'X03y', 'X04y'],
            ['X10y', 'X11y', 'X12y', 'X13y', 'X14y'],
            ['X20y', 'X21y', 'X22y', 'X23y', 'X24y'],
            ['X30y', 'X31y', 'X32y', 'X33y', 'X34y']]
    t=Table(XY, 5*[0.6*inch], 4*[0.6*inch])
    t.setStyle([('ALIGN',(1,1),(-2,-2),'LEFT'),
                ('TEXTCOLOR',(1,1),(-2,-2),colors.red),

                ('VALIGN',(0,0),(1,-1),'TOP'),
                ('ALIGN',(0,0),(1,-1),'RIGHT'),
                ('TEXTCOLOR',(0,0),(1,-1),colors.blue),

                ('ALIGN',(0,-1),(-1,-1),'CENTER'),
                ('VALIGN',(0,-1),(-1,-1),'MIDDLE'),
                ('TEXTCOLOR',(0,-1),(-1,-1),colors.green),
                ('INNERGRID', (0,0), (-1,-1), 0.25, colors.black),
                ('BOX', (0,0), (-1,-1), 0.25, colors.black),
                ])
    lst_add(t)
    data = [('alignment', 'align\012alignment'),
            ('bulletColor', 'bulletcolor\012bcolor'),
            ('bulletFontName', 'bfont\012bulletfontname'),
            ('bulletFontSize', 'bfontsize\012bulletfontsize'),
            ('bulletIndent', 'bindent\012bulletindent'),
            ('firstLineIndent', 'findent\012firstlineindent'),
            ('fontName', 'face\012fontname\012font'),
            ('fontSize', 'size\012fontsize'),
            ('leading', 'leading'),
            ('leftIndent', 'leftindent\012lindent'),
            ('rightIndent', 'rightindent\012rindent'),
            ('spaceAfter', 'spaceafter\012spacea'),
            ('spaceBefore', 'spacebefore\012spaceb'),
            ('textColor', 'fg\012textcolor\012color')]
    t = Table(data)
    t.setStyle([
            ('VALIGN',(0,0),(-1,-1),'TOP'),
            ('INNERGRID', (0,0), (-1,-1), 0.25, colors.black),
            ('BOX', (0,0), (-1,-1), 0.25, colors.black),
            ])
    lst_add(t)
    t = Table([ ('Attribute', 'Synonyms'),
                ('alignment', 'align, alignment'),
                ('bulletColor', 'bulletcolor, bcolor'),
                ('bulletFontName', 'bfont, bulletfontname'),
                ('bulletFontSize', 'bfontsize, bulletfontsize'),
                ('bulletIndent', 'bindent, bulletindent'),
                ('firstLineIndent', 'findent, firstlineindent'),
                ('fontName', 'face, fontname, font'),
                ('fontSize', 'size, fontsize'),
                ('leading', 'leading'),
                ('leftIndent', 'leftindent, lindent'),
                ('rightIndent', 'rightindent, rindent'),
                ('spaceAfter', 'spaceafter, spacea'),
                ('spaceBefore', 'spacebefore, spaceb'),
                ('textColor', 'fg, textcolor, color')])
    t.repeatRows = 1
    t.setStyle([
                ('FONT',(0,0),(-1,1),'Times-Bold',10,12),
                ('FONT',(0,1),(-1,-1),'Courier',8,8),
                ('VALIGN',(0,0),(-1,-1),'MIDDLE'),
                ('INNERGRID', (0,0), (-1,-1), 0.25, colors.black),
                ('BOX', (0,0), (-1,-1), 0.25, colors.black),
                ('BACKGROUND', (0, 0), (-1, 0), colors.green),
                ('BACKGROUND', (0, 1), (-1, -1), colors.pink),
                ('ALIGN', (0, 0), (-1, 0), 'CENTER'),
                ('ALIGN', (0, 1), (0, -1), 'LEFT'),
                ('ALIGN', (-1, 1), (-1, -1), 'RIGHT'),
                ('FONT', (0, 0), (-1, 0), 'Times-Bold', 12),
                ('ALIGN', (1, 1), (1, -1), 'CENTER'),
                ])
    lst_add(t)
    lst_add(Table(XY,
            style=[ ('FONT',(0,0),(-1,-1),'Times-Roman', 5,6),
                    ('GRID', (0,0), (-1,-1), 0.25, colors.blue),]))
    lst_add(Table(XY,
            style=[ ('FONT',(0,0),(-1,-1),'Times-Roman', 10,12),
                    ('GRID', (0,0), (-1,-1), 0.25, colors.black),]))
    lst_add(Table(XY,
            style=[ ('FONT',(0,0),(-1,-1),'Times-Roman', 20,24),
                    ('GRID', (0,0), (-1,-1), 0.25, colors.red),]))
    lst_add(PageBreak())
    data=  [['00', '01', '02', '03', '04'],
            ['10', '11', '12', '13', '14'],
            ['20', '21', '22', '23', '24'],
            ['30', '31', '32', '33', '34']]
    t=Table(data,style=[
                    ('GRID',(0,0),(-1,-1),0.5,colors.grey),
                    ('GRID',(1,1),(-2,-2),1,colors.green),
                    ('BOX',(0,0),(1,-1),2,colors.red),
                    ('BOX',(0,0),(-1,-1),2,colors.black),
                    ('LINEABOVE',(1,2),(-2,2),1,colors.blue),
                    ('LINEBEFORE',(2,1),(2,-2),1,colors.pink),
                    ('BACKGROUND', (0, 0), (0, 1), colors.pink),
                    ('BACKGROUND', (1, 1), (1, 2), colors.lavender),
                    ('BACKGROUND', (2, 2), (2, 3), colors.orange),
                    ('TEXTCOLOR',(0,-1),(-2,-1),colors.green),
                    ])
    lst_add(Paragraph("Illustrating splits: nosplit", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits: split(4in,30)", styleSheet['BodyText']))
    for s in t.split(4*inch,30):
        lst_add(s)
        lst_add(Spacer(0,6))
    lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits: split(4in,36)", styleSheet['BodyText']))
    for s in t.split(4*inch,36):
        lst_add(s)
        lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits: split(4in,56)", styleSheet['BodyText']))
    lst_add(Spacer(0,6))
    for s in t.split(4*inch,56):
        lst_add(s)
        lst_add(Spacer(0,6))

    lst_add(Paragraph("Illustrating splits: repeated split(4in,30)", styleSheet['BodyText']))
    lst_add(Spacer(0,6))
    S = t.split(4*inch,30)
    s = S.pop(-1)
    S.extend(s.split(4*inch,30))
    s = S.pop(-1)
    S.extend(s.split(4*inch,30))

    for s in S:
        lst_add(s)
        lst_add(Spacer(0,6))

    lst_add(PageBreak())
    data=  [['00', '01', '02', '03', '04'],
            ['', '11', '12', '13', '14'],
            ['20', '21', '22', '23', '24'],
            ['30', '31', '', '33', '34']]
    sty=[
                    ('GRID',(0,0),(-1,-1),0.5,colors.grey),
                    ('GRID',(1,1),(-2,-2),1,colors.green),
                    ('BOX',(0,0),(1,-1),2,colors.red),
                    ('BOX',(0,0),(-1,-1),2,colors.black),
                    ('LINEABOVE',(1,2),(-2,2),1,colors.blue),
                    ('LINEBEFORE',(2,1),(2,-2),1,colors.pink),
                    ('BACKGROUND', (0, 0), (0, 1), colors.pink),
                    ('SPAN',(0,0),(0,1)),
                    ('BACKGROUND', (2, 2), (2, 3), colors.orange),
                    ('SPAN',(2,2),(2,3)),
                    ]
    t=Table(data,style=sty)
    lst_add(Paragraph("Illustrating splits with spans: nosplit", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits with spans: split(4in,30)", styleSheet['BodyText']))
    for s in t.split(4*inch,30):
        lst_add(s)
        lst_add(Spacer(0,6))
    lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits with spans: split(4in,36)", styleSheet['BodyText']))
    for s in t.split(4*inch,36):
        lst_add(s)
        lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits with spans: split(4in,56)", styleSheet['BodyText']))
    lst_add(Spacer(0,6))
    for s in t.split(4*inch,56):
        lst_add(s)
        lst_add(Spacer(0,6))

    data=  [['00', '01', '02', '03', '04'],
            ['', '11', '12', '13', ''],
            ['20', '21', '22', '23', '24'],
            ['30', '31', '', '33', ''],
            ['40', '41', '', '43', '44']]
    sty=[
        ('GRID',(0,0),(-1,-1),0.5,colors.grey),
        ('GRID',(1,1),(-2,-2),1,colors.green),
        ('BOX',(0,0),(1,-1),2,colors.red),
        ('BOX',(0,0),(-1,-1),2,colors.black),
        ('LINEABOVE',(1,2),(-2,2),1,colors.blue),
        ('LINEBEFORE',(2,1),(2,-2),1,colors.pink),
        ('BACKGROUND', (0, 0), (0, 1), colors.pink),
        ('SPAN',(0,0),(0,1)),
        ('BACKGROUND',(-2,1),(-1,1),colors.palegreen),
        ('SPAN',(-2,1),(-1,1)),
        ('BACKGROUND',(-2,3),(-1,3),colors.yellow),
        ('SPAN',(-2,3),(-1,3)),
        ('BACKGROUND', (2, 3), (2, 4), colors.orange),
        ('SPAN',(2,3),(2,4)),
        ]

    t=Table(data,style=sty,repeatRows=2)
    lst_add(Paragraph("Illustrating splits with spans and repeatRows: nosplit", styleSheet['BodyText']))
    lst_add(t)
    lst_add(Spacer(0,6))
    if  1:
        lst_add(Paragraph("Illustrating splits with spans and repeatRows: split(4in,30)", styleSheet['BodyText']))
        for s in t.split(4*inch,30):
            lst_add(s)
            lst_add(Spacer(0,6))
        lst_add(Spacer(0,6))
        lst_add(Paragraph("Illustrating splits with spans and repeatRows: split(4in,36)", styleSheet['BodyText']))
        for s in t.split(4*inch,36):
            lst_add(s)
            lst_add(Spacer(0,6))
    lst_add(Paragraph("Illustrating splits with spans and repeatRows: split(4in,56)", styleSheet['BodyText']))
    lst_add(Spacer(0,6))
    for s in t.split(4*inch,56):
        lst_add(s)
        lst_add(Spacer(0,6))

    lst_add(PageBreak())
    from reportlab.lib.testutils import testsFolder
    I = Image(os.path.join(os.path.dirname(testsFolder),'tools','pythonpoint','demos','leftlogo.gif'))
    I.drawHeight = 1.25*inch*I.drawHeight / I.drawWidth
    I.drawWidth = 1.25*inch
    #I.drawWidth = 9.25*inch #uncomment to see better messaging
    P = Paragraph("<para align=center spaceb=3>The <b>ReportLab Left <font color=red>Logo</font></b> Image</para>", styleSheet["BodyText"])
    B = TableBarChart()
    BP = Paragraph("<para align=center spaceb=3>A bar chart in a cell.</para>", styleSheet["BodyText"])

    data=  [['A', 'B', 'C', Paragraph("<b>A pa<font color=red>r</font>a<i>graph</i></b><super><font color=yellow>1</font></super>",styleSheet["BodyText"]), 'D'],
            ['00', '01', '02', [I,P], '04'],
            ['10', '11', '12', [I,P], '14'],
            ['20', '21', '22', '23', '24'],
            ['30', '31', '32', '33', '34'],
            ['40', '41', '42', [B,BP], '44']]

    t=Table(data,style=[('GRID',(1,1),(-2,-2),1,colors.green),
                    ('BOX',(0,0),(1,-1),2,colors.red),
                    ('LINEABOVE',(1,2),(-2,2),1,colors.blue),
                    ('LINEBEFORE',(2,1),(2,-2),1,colors.pink),
                    ('BACKGROUND', (0, 0), (0, 1), colors.pink),
                    ('BACKGROUND', (1, 1), (1, 2), colors.lavender),
                    ('BACKGROUND', (2, 2), (2, 3), colors.orange),
                    ('BOX',(0,0),(-1,-1),2,colors.black),
                    ('GRID',(0,0),(-1,-1),0.5,colors.black),
                    ('VALIGN',(3,0),(3,0),'BOTTOM'),
                    ('BACKGROUND',(3,0),(3,0),colors.limegreen),
                    ('BACKGROUND',(3,1),(3,1),colors.khaki),
                    ('ALIGN',(3,1),(3,1),'CENTER'),
                    ('BACKGROUND',(3,2),(3,2),colors.beige),
                    ('ALIGN',(3,2),(3,2),'LEFT'),
                    ])

    t._argW[3]=1.5*inch
    lst_add(t)

    # now for an attempt at column spanning.
    lst_add(PageBreak())
    data=  [['A', 'BBBBB', 'C', 'D', 'E'],
            ['00', '01', '02', '03', '04'],
            ['10', '11', '12', '13', '14'],
            ['20', '21', '22', '23', '24'],
            ['30', '31', '32', '33', '34']]
    sty = [
            ('ALIGN',(0,0),(-1,-1),'CENTER'),
            ('VALIGN',(0,0),(-1,-1),'TOP'),
            ('GRID',(0,0),(-1,-1),1,colors.green),
            ('BOX',(0,0),(-1,-1),2,colors.red),

            #span 'BBBB' across middle 3 cells in top row
            ('SPAN',(1,0),(3,0)),
            #now color the first cell in this range only,
            #i.e. the one we want to have spanned.  Hopefuly
            #the range of 3 will come out khaki.
            ('BACKGROUND',(1,0),(1,0),colors.khaki),

            ('SPAN',(0,2),(-1,2)),


            #span 'AAA'down entire left column
            ('SPAN',(0,0), (0, 1)),
            ('BACKGROUND',(0,0),(0,0),colors.cyan),
            ('TEXTCOLOR', (0,'splitfirst'), (-1,'splitfirst'), colors.cyan),
            ('TEXTCOLOR', (0,'splitlast'), (-1,'splitlast'), colors.red),
            ('BACKGROUND', (0,'splitlast'), (-1,'splitlast'), colors.pink),
            ('LINEBELOW', (0,'splitlast'), (-1,'splitlast'), 1, colors.grey,'butt'),
           ]
    t=Table(data,style=sty, colWidths = [20] * 5, rowHeights = [20]*5)
    lst_add(t)
    lst_add(Spacer(18,18))

    t=Table(data,style=sty, colWidths = [20] * 5, rowHeights = [20]*5)
    for s in t.split(4*inch,72):
        lst_add(s)
        lst_add(Spacer(0,6))

    # now for an attempt at percentage widths
    lst_add(Spacer(18,18))
    lst_add(Paragraph("This table has colWidths=5*['14%']!", styleSheet['BodyText']))
    t=Table(data,style=sty, colWidths = ['14%'] * 5, rowHeights = [20]*5)
    lst_add(t)

    lst_add(Spacer(18,18))
    lst_add(Paragraph("This table has colWidths=['14%','10%','19%','22%','*']!", styleSheet['BodyText']))
    t=Table(data,style=sty, colWidths = ['14%','10%','19%','22%','*'], rowHeights = [20]*5)
    lst_add(t)

    # Mike's test example
    lst_add(Spacer(18,18))
    lst_add(Paragraph('Mike\'s Spanning Example', styleSheet['Heading1']))
    data=  [[Paragraph('World Domination: The First Five Years', styleSheet['BodyText']), ''],
            [Paragraph('World <font color="green">Domination</font>: The First Five Years', styleSheet['BodyText']),''],
            [Paragraph('World Domination: The First Five Years', styleSheet['BodyText']), ''],
            ]
    t=Table(data, style=[('SPAN',(0,0),(1,0)),('SPAN',(0,1),(1,1)),('SPAN',(0,2),(1,2)),], colWidths = [3*cm,8*cm], rowHeights = [None]*3)
    lst_add(t)

    lst_add(Spacer(18,18))
    lst_add(Paragraph('Mike\'s Non-spanning Example', styleSheet['Heading1']))
    data=  [[Paragraph('World Domination: The First Five Years', styleSheet['BodyText'])],
            [Paragraph('World <font color="magenta">Domination</font>: The First Five Years', styleSheet['BodyText'])],
            [Paragraph('World Domination: The First Five Years', styleSheet['BodyText'])],
            ]
    t=Table(data, style=[], colWidths = [11*cm], rowHeights = [None]*3)
    lst_add(t)

    lst_add(Spacer(18,18))
    lst_add(Paragraph('xpre example', styleSheet['Heading1']))
    data=  [    [
                XPreformatted('Account Details', styleSheet['Heading3']),
                '', XPreformatted('Client Details', styleSheet['Heading3']),
                ],  #end of row 0
            ]
    t=Table(data, style=[], colWidths = [80,230.0,80], rowHeights = [None]*1)
    lst_add(t)

    lst_add(PageBreak())

    lst_add(Paragraph('Trying colour cycling in background', styleSheet['Heading1']))
    lst_add(Paragraph("This should alternate pale blue and uncolored by row", styleSheet['BodyText']))
    data=  [['001', '01', '02', '03', '04', '05'],
            ['002', '01', '02', '03', '04', '05'],
            ['003', '01', '02', '03', '04', '05'],
            ['004', '01', '02', '03', '04', '05'],
            ['005', '01', '02', '03', '04', '05'],
            ['006', '01', '02', '03', '04', '05'],
            ['007', '01', '02', '03', '04', '05'],
            ['008', '01', '02', '03', '04', '05'],
            ['009', '01', '02', '03', '04', '05'],
            ['010', '01', '02', '03', '04', '05'],

            ]
    t=Table(data,style=[
                    ('GRID',(0,0),(-1,-1),0.5,colors.grey),
                    ('ROWBACKGROUNDS', (0, 0), (-1, -1), (0xD0D0FF, None)),
                    ])
    lst_add(t)
    lst_add(Spacer(0,6))
    lst_add(Paragraph("And this should pale blue, pale pink and None by column", styleSheet['BodyText']))
    t=Table(data,style=[
                    ('GRID',(0,0),(-1,-1),0.5,colors.grey),
                    ('COLBACKGROUNDS', (0, 0), (-1, -1), (0xD0D0FF, 0xFFD0D0, None)),
                    ])
    lst_add(t)

    lst_add(PageBreak())
    lst_add(Paragraph("This spanning example illustrates automatic removal of grids and lines in spanned cells!", styleSheet['BodyText']))
    lst_add(Spacer(0,6))
    data=  [['Top\nLeft', '', '02', '03', '04', '05', '06', '07'],
            ['', '', '12', 'Span (3,1) (6,2)', '','','','17'],
            ['20', '21', '22', '', '','','','27'],
            ['30', '31', '32', '33', '34','35','36','37'],
            ['40', 'In The\nMiddle', '', '', '44','45','46','47'],
            ['50', '', '', '', '54','55','56','57'],
            ['60', '', '', '','64', '65', 'Bottom\nRight', ''],
            ['70', '71', '72', '73','74', '75', '', '']]
    t=Table(data,style=[
            ('GRID',(0,0),(-1,-1),0.5,colors.grey),
            ('BACKGROUND',(0,0),(1,1),colors.palegreen),
            ('SPAN',(0,0),(1,1)),
            ('BACKGROUND',(-2,-2),(-1,-1), colors.pink),
            ('SPAN',(-2,-2),(-1,-1)),
            ('SPAN',(1,4),(3,6)),
            ('BACKGROUND',(1,4),(3,6), colors.lightblue),
            ('SPAN',(3,1),(6,2)),
            ('BACKGROUND',(3,1),(6,2), colors.peachpuff),
            ('VALIGN',(3,1),(6,2),'TOP'),
            ('LINEABOVE', (0,2),(-1,2), 1, colors.black, 0, None, None, 2, 2),
            ('LINEBEFORE', (3,0),(3,-1), 1, colors.black, 0, None, None, 2, 2),
            ])
    lst_add(t)

    lst_add(PageBreak())

    lst_add(Paragraph("und jetzt noch eine Tabelle mit 5000 Zeilen:", styleSheet['BodyText']))
    sty = [ ('GRID',(0,0),(-1,-1),1,colors.green),
            ('BOX',(0,0),(-1,-1),2,colors.red),
           ]
    data = [[str(i), Paragraph("xx "* (i%10), styleSheet["BodyText"]), Paragraph("blah "*(i%40), styleSheet["BodyText"])] for i in range(500)]
    t=LongTable(data, style=sty, colWidths = [50,100,200])
    lst_add(t)

    #Yuan Hong's bug tester
    lst_add(PageBreak())
    lst_add(Paragraph('Yian Hong\'s Bug Case (should not blow up)', styleSheet['Heading2']))
    data = ([['Col1', 'Col2', 'Col3', 'Col4', 'Col5']]+
                [['01', Paragraph('This is cell one that contains a paragraph.', styleSheet['Normal']), '02', '03', '04']
                    for i in range(50)])

    t = Table(data, ['20%']*5, repeatRows=1)
    t.setStyle(TableStyle([
        ('INNERGRID', (0,0), (-1,-1), 0.25, colors.black),
        ('BOX', (0,0), (-1,-1), 0.25, colors.black),
        ('SPAN', (0,50), (-2,50)),
        ]))

    lst_add(t)
    lst_add(PageBreak())

    #Volker Haas' example extended
    #the optimal row heights are the solution of an LP similar to
    #
    #Objective function
    #   min: 3*h0+3*h1+3*h2+2*h3;
    #
    #constraints
    #   h0>=12;
    #   h1>=12;
    #   h2>=12;
    #   h3>=12;
    #   h0+h1+h2>=48;
    #   h0+h1>=12;
    #   h2+h3>=60;
    #
    #the solution H=[12,12,24,36]
    def makeTable(x,y):
        return Table([
                ['00', '01', '02', '03', '04', '05\nline2\nline3\nline4'],
                ['', '11', '12', x, '',''],
                ['20', '21', y, '23', '24',''],
                ['30', '31', '', '33', '34','35'],
                ],
                style=[
                    ('TOPPADDING',(0,0),(-1,-1),0),
                    ('BOTTOMPADDING',(0,0),(-1,-1),0),
                    ('RIGHTPADDING',(0,0),(-1,-1),0),
                    ('LEFTPADDING',(0,0),(-1,-1),0),
                    ('GRID',(0,0),(-1,-1),0.5,colors.grey),
                    ('BACKGROUND', (0, 0), (0, 1), colors.pink),
                    ('SPAN',(0,0),(0,1)),
                    ('BACKGROUND', (2, 2), (2, 3), colors.orange),
                    ('SPAN',(2,2),(2,3)),
                    ('SPAN',(3,1),(4,1)),
                    ('SPAN',(5,0),(5,2)),
                ])
    p_style= ParagraphStyle('Normal')
    lst_add(makeTable(
            Paragraph('This is a string',p_style),
            Paragraph('22<br/>blub<br/>asfd<br/>afd<br/>asdfs', p_style)
            ))

    lst_add(Spacer(10,10))
    lst_add(makeTable(
            XPreformatted('This is a string',p_style),
            Paragraph('22<br/>blub<br/>asfd<br/>afd<br/>asdfs', p_style)
            ))
    lst_add(Spacer(10,10))
    lst_add(makeTable(
            'This is a string',
            '22\nblub\nasfd\nafd\nasdfs',
            ))
    lst_add(Spacer(10,10))
    lst_add(makeTable(
            'This is a string',
            Paragraph('22<br/>blub<br/>asfd<br/>afd<br/>asdfs', p_style)
            ))
    SimpleDocTemplate(outputfile('test_platypus_tables_2.pdf'), showBoundary=1).build(lst)

class TablesTestCase(unittest.TestCase):
    "Make documents with tables"

    def test0(self):
        "Make a document full of tables 0"
        run()

    def test1(self):
        "Make a document full of tables 1"
        old_tables_test()

    def test2(self):
        '''buggy table example from Lele Gaifax https://bitbucket.org/lele/
        should split to two pages with the blue box on page 1 complete
        '''
        from reportlab.lib.pagesizes import A4, landscape

        data = [
            ['Date', '08 AM', '', '', '', '09 AM', '', '', '', '10 AM', '', '', '', '11 AM', '', '', '', '12 PM', '', '', '', '01 PM', '', '', '', '02 PM', '', '', '', '03 PM', '', '', '', '04 PM', '', '', '', '05 PM', '', '', '', '06 PM', '', '', '', '07 PM', '', '', '', '08 PM', '', '', '', '09 PM', '', '', '', '10 PM', '', '', '', '11 PM', '', '', ''],
            ['04-30-2015', '', '', '', '', '09:00 AM\n:\n01:00 PM\nSupervision\nReception', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
            ['05-01-2015', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '01:00 PM\n:\n04:00 PM\nSupervision\nInfo point 1', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
            ['05-03-2015', '', '', '', '', '09:00 AM\n:\n01:00 PM\nSupervision\nReception', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
            ['05-04-2015', '', '', '', '', '', '', '', '', '10:00 AM\n:\n01:00 PM\nSupervision\nInfo point 2', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '03:00 PM\n:\n05:30 PM\nSupervision\nInfo point 3', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
            ['05-05-2015', '', '', '', '', '09:00 AM\n:\n01:00 PM\nSupervision\nReception', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
            ['05-08-2015', '', '', '', '', '', '', '', '', '10:00 AM\n:\n01:00 PM\nSupervision\nInfo point 2', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '02:30 PM\n:\n05:30 PM\nSupervision\nInfo point 3', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
            ['05-10-2015', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '04:00 PM\n:\n08:00 PM\nSupervision\nInfo point 2', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', ''],
         ]

        style = [
            ('SPAN', (1, 0), (4, 0)),
            ('SPAN', (5, 0), (8, 0)),
            ('SPAN', (9, 0), (12, 0)),
            ('SPAN', (13, 0), (16, 0)),
            ('SPAN', (17, 0), (20, 0)),
            ('SPAN', (21, 0), (24, 0)),
            ('SPAN', (25, 0), (28, 0)),
            ('SPAN', (29, 0), (32, 0)),
            ('SPAN', (33, 0), (36, 0)),
            ('SPAN', (37, 0), (40, 0)),
            ('SPAN', (41, 0), (44, 0)),
            ('SPAN', (45, 0), (48, 0)),
            ('SPAN', (49, 0), (52, 0)),
            ('SPAN', (53, 0), (56, 0)),
            ('SPAN', (57, 0), (60, 0)),
            ('SPAN', (61, 0), (64, 0)),
            ('SPAN', (1, -1), (4, -1)),
            ('SPAN', (5, -1), (8, -1)),
            ('SPAN', (9, -1), (12, -1)),
            ('SPAN', (13, -1), (16, -1)),
            ('SPAN', (17, -1), (20, -1)),
            ('SPAN', (21, -1), (24, -1)),
            ('SPAN', (25, -1), (28, -1)),
            ('SPAN', (29, -1), (32, -1)),
            ('SPAN', (33, -1), (36, -1)),
            ('SPAN', (37, -1), (40, -1)),
            ('SPAN', (41, -1), (44, -1)),
            ('SPAN', (45, -1), (48, -1)),
            ('SPAN', (49, -1), (52, -1)),
            ('SPAN', (53, -1), (56, -1)),
            ('SPAN', (57, -1), (60, -1)),
            ('SPAN', (61, -1), (64, -1)),
            ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
            ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
            ('SIZE', (0, 1), (0, -1), 8),
            ('SIZE', (1, 1), (-1, -1), 6),
            ('BOX', (5, 1), (20, 1), 1, colors.black), ('SPAN', (5, 1), (20, 1)),
            ('BOX', (21, 2), (32, 2), 1, colors.black), ('SPAN', (21, 2), (32, 2)),
            ('BOX', (5, 3), (20, 3), 1, colors.black), ('SPAN', (5, 3), (20, 3)),
            ('BOX', (9, 4), (20, 4), 1, colors.black), ('SPAN', (9, 4), (20, 4)),
            ('BOX', (29, 4), (38, 4), 1, colors.black), ('SPAN', (29, 4), (38, 4)),
            ('BOX', (5, 5), (20, 5), 1, colors.green), ('SPAN', (5, 5), (20, 5)),
            ('BOX', (9, 6), (20, 6), 1, colors.red), ('SPAN', (9, 6), (20, 6)),
            ('BOX', (27, 6), (38, 6), 1, colors.blue),('SPAN', (27, 6), (38, 6)),
            ('BOX', (33, 7), (48, 7), 1, colors.black),('SPAN', (33, 7), (48, 7)),
            ]
        t = Table(data, colWidths=None, rowHeights=None, style=style, repeatRows=1)
        doc = SimpleDocTemplate(outputfile('test_platypus_tables_issue74.pdf'), showBoundary=0, pagesize=landscape(A4))
        doc.build([t])

    data34 = [
            ['001', '01', '02', '03', '04', '05'],
            ['002', '01', '02', '03', '04', '05'],
            ['003', '01', '02', '03', '04', '05'],
            ['004', '01', '02', '03', '04', '05'],
            ['005', '01', '02', '03', '04', '05'],
            ['006', '01', '02', '03', '04', '05'],
            ['007', '01', '02', '03', '04', '05'],
            ['008', '01', '02', '03', '04', '05'],
            ['009', '01', '02', '03', '04', '05'],
            ['010', '01', '02', '03', '04', '05'],
            ['011', '01', '02', '03', '04', '05'],
            ['012', '01', '02', '03', '04', '05'],
            ]
    def test3(self):
        '''bug reported by David VanEe <david.vanee@convergent.ca>'''
        story = []
        story_add = story.append
        ts_tables = [
                 ('BACKGROUND',(0,0),(-1,0),colors.pink),
                 ('BACKGROUND',(0,1),(-1,1),colors.lightblue),
                 ('BACKGROUND',(0,3),(-1,3),colors.grey),
                 ('TEXTCOLOR',(0,0),(-1,0),colors.green),
                 ('TEXTCOLOR',(0,1),(-1,1),colors.red),
                 ('LINEABOVE', (0,0), (-1,0), 1, colors.purple),
                 ('LINEBELOW', (0,0), (-1,0), 2, colors.purple),
                 ('LINEABOVE', (0,1), (-1,1), 1, colors.orange),
                 ('LINEBELOW', (0,1), (-1,1), 2, colors.orange),
                 ('FONT', (2,2), (5,8), 'Times-Bold'),
                 ]
        data = self.data34
        from reportlab.platypus import Paragraph, Table, SimpleDocTemplate, PageBreak
        from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
        styleSheet = getSampleStyleSheet()
        bodyText = styleSheet['BodyText']

        story_add(Paragraph('The whole table',bodyText))
        t = Table(data, style=ts_tables, repeatRows=(1,3))
        story_add(t)
        t = Table(data, style=ts_tables, repeatRows=(1,3))
        T = t.split(4*72,90)
        story_add(Paragraph('The split table part 0',bodyText))
        story_add(T[0])
        story_add(Paragraph('The split table part 1',bodyText))
        story_add(T[1])
        self.assertIn(('BACKGROUND', (0, 0), (-1, 0), colors.lightblue),T[1]._bkgrndcmds)
        self.assertIn(('BACKGROUND', (0, 1), (-1, 1), colors.grey),T[1]._bkgrndcmds)
        self.assertEqual(len(T[1]._bkgrndcmds),2)

        # do the same again with repeatRows=1
        story_add(PageBreak())
        story_add(Paragraph('The whole table repeatRows=1',bodyText))
        t = Table(data, style=ts_tables, repeatRows=1)
        story_add(t)
        t = Table(data, style=ts_tables, repeatRows=1)
        T = t.split(4*72,60)
        story_add(Paragraph('The split table (repeatRows=1) part 0',bodyText))
        story_add(T[0])
        story_add(Paragraph('The split table (repeatRows=1) part 1',bodyText))
        story_add(T[1])
        self.assertIn(('BACKGROUND', (0, 0), (-1, 0), colors.pink),  T[1]._bkgrndcmds)
        self.assertIn(('BACKGROUND', (0, 1), (-1, 1), colors.grey), T[1]._bkgrndcmds)
        self.assertEqual(len(T[1]._bkgrndcmds),2)
        doc = SimpleDocTemplate(outputfile('test_platypus_tables_repeatrows_bgsplit.pdf'), showBoundary=0)
        doc.build(story)

    def test4(self):
        '''test splitting row colour cycles'''
        story = []
        story_add = story.append
        ts_tables = [
                 ('BACKGROUND',(0,0),(-1,0),colors.pink),
                 ('BACKGROUND',(0,1),(-1,1),colors.lightblue),
                 ('ROWBACKGROUNDS',(0,2),(-1,-1),(colors.lightgrey,None)),
                 ('TEXTCOLOR',(0,0),(-1,0),colors.green),
                 ('TEXTCOLOR',(0,1),(-1,1),colors.red),
                 ('LINEABOVE', (0,0), (-1,0), 1, colors.purple),
                 ('LINEBELOW', (0,0), (-1,0), 2, colors.purple),
                 ('LINEABOVE', (0,1), (-1,1), 1, colors.orange),
                 ('LINEBELOW', (0,1), (-1,1), 2, colors.orange),
                 ('FONT', (2,2), (5,8), 'Times-Bold'),
                 ]
        data = self.data34
        from reportlab.platypus import Paragraph, Table, SimpleDocTemplate, PageBreak
        from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
        styleSheet = getSampleStyleSheet()
        bodyText = styleSheet['BodyText']

        story_add(Paragraph('The whole table',bodyText))
        t = Table(data, style=ts_tables, repeatRows=2)
        story_add(t)
        t = Table(data, style=ts_tables, repeatRows=2)
        T = t.split(4*72,90)
        story_add(Paragraph('The split table part 0',bodyText))
        story_add(T[0])
        story_add(Paragraph('The split table part 1',bodyText))
        story_add(T[1])
        self.assertIn(('BACKGROUND', (0, 0), (-1, 0), colors.pink),T[1]._bkgrndcmds)
        self.assertIn(('BACKGROUND', (0, 1), (-1, 1), colors.lightblue),T[1]._bkgrndcmds)
        self.assertIn(('ROWBACKGROUNDS', (0, 2), (-1, 8), (colors.lightgrey,None)),T[1]._bkgrndcmds)
        self.assertEqual(len(T[1]._bkgrndcmds),3)
        doc = SimpleDocTemplate(outputfile('test_platypus_tables_repeatrows_bgsplit_1.pdf'), showBoundary=0)
        self.assertEqual(len(T[1]._bkgrndcmds),3)
        doc.build(story)

def makeSuite():
    return makeSuiteForClasses(TablesTestCase)


#noruntests
if __name__ == "__main__":
    unittest.TextTestRunner().run(makeSuite())
    printLocation()
