#ifndef __DMA_RIVA_H
#define __DMA_RIVA_H

#include <linux/interrupt.h>

/* DMA structure. */

struct rivatv_dma {
	u32 ht_size;		    /* Hashtable size in bytes.		*/
	u32 ht_base;		    /* Hashtable base address.		*/
	u32 ht_search;		    /* Hashtable search flag.		*/
	u32 ht_bits;		    /* Number of bits of the hashtable. */
	u8  *notifiers;		    /* Notifier data.			*/
	dma_addr_t addr;	    /* DMA (bus) address.		*/
	dma_addr_t notifiers_addr;  /* DMA notifier address.		*/
	struct tasklet_struct task; /* Task queue.			*/
};

/* Colour plane definition for converting DMA transfers. */

struct rivatv_direct_DMA_plane
{
	u32 src_offset; /* source skip in bytes               */
	u32 dst_offset; /* destination skip in bytes          */
	u32 src_pitch;  /* skip bytes per line in source      */
	u32 dst_pitch;  /* skip bytes per line in destination */
	u32 src_length; /* bytes per line in source           */
	u32 src_lines;  /* lines in source                    */
	u32 src_inc;    /* source increment                   */
	u32 dst_inc;    /* destination increment              */
	u32 planar;     /* flag indicating planar conversion  */
};

/* Colour conversion array for DMA transfers. */

struct rivatv_direct_DMA
{
	int valid;
	int nplanes;
	struct rivatv_direct_DMA_plane plane[3];
};

extern struct rivatv_direct_DMA rivatv_convert_DMA[17];

/* Notifier structure. */

#define NV_NOTIFICATION_SIZE 0x100

typedef volatile struct {
	u64 timeStamp; /* nanoseconds since Jan. 1, 1970 */
	u32 returnVal; /* NVX				 */
	u16 errorCode; /* NVX				 */
	u8  reserved;  /* zero				 */
	u8  status;    /* NV_NOTIFICATION_STATUS_*	 */
} NvNotification;

#define NV_NOTIFICATION_STATUS_DONE_SUCCESS		(0x00)
#define NV_NOTIFICATION_STATUS_IN_PROGRESS		(0xFF)

#define NV_NOTIFY_WRITE_LE				(0x00000000)
#define NV_NOTIFY_WRITE_LE_AWAKEN			(0x00000001)
#define NV_NOTIFY_WRITE_LE_AWAKEN_2			(0x00000002)
#define NV_NOTIFY_WRITE_LE_AWAKEN_3			(0x00000003)
#define NV_NOTIFY_WRITE_LE_AWAKEN_4			(0x00000004)
#define NV_NOTIFY_WRITE_LE_AWAKEN_5			(0x00000005)
#define NV_NOTIFY_WRITE_LE_AWAKEN_6			(0x00000006)
#define NV_NOTIFY_WRITE_LE_AWAKEN_7			(0x00000007)
#define NV_NOTIFY_WRITE_LE_AWAKEN_8			(0x00000008)
#define NV_NOTIFY_WRITE_LE_AWAKEN_9			(0x00000009)
#define NV_NOTIFY_WRITE_LE_AWAKEN_A			(0x0000000A)
#define NV_NOTIFY_WRITE_LE_AWAKEN_B			(0x0000000B)
#define NV_NOTIFY_WRITE_LE_AWAKEN_C			(0x0000000C)
#define NV_NOTIFY_WRITE_LE_AWAKEN_D			(0x0000000D)
#define NV_NOTIFY_WRITE_LE_AWAKEN_E			(0x0000000E)
#define NV_NOTIFY_WRITE_LE_AWAKEN_F			(0x0000000F)

/* Generic class (notify access). */

typedef volatile struct {
	u32 NoOperation;             /* ignored        */
	u32 Notify;                  /* NV_NOTIFY_*    */
	u32 Reserved00[0x01e];
	u32 SetContextDmaNotifies;   /* NV_CONTEXT_DMA */
	u32 Reserved02[0x79f];
} NvGeneric;

/* Class 039: Nv03MemoryToMemoryFormat. */

typedef volatile struct {
	u32 NoOperation;	     /* ignored			      */
	u32 Notify;		     /* NV_NOTIFY_STYLE*	      */
	u32 Reserved00[0x01e];
	u32 SetContextDmaNotifies;   /* NV_CONTEXT_DMA		      */
	u32 SetContextDmaBufferIn;   /* NV_CONTEXT_DMA		      */
	u32 SetContextDmaBufferOut;  /* NV_CONTEXT_DMA		      */
	u32 Reserved01[0x060];
	u32 SetOffsetIn;	     /* byte offset for input buffer  */
	u32 SetOffsetOut;	     /* byte offset for output buffer */
	u32 SetPitchIn;		     /*	input line pitch in bytes     */
	u32 SetPitchOut;	     /*	output line pitch in bytes    */
	u32 SetLineLengthIn;	     /*	length of a line in bytes     */
	u32 SetLineCount;	     /*	# of lines to copy	      */
	u32 SetFormat;		     /*	byte increment per byte	      */
	u32 SetBufNotify;	     /*	starts the transfer	      */
	u32 Reserved02[0x735];
} Nv03MemoryToMemoryFormat;

#define NV0039_SET_FORMAT_INPUT_INC_1			 0x00000001
#define NV0039_SET_FORMAT_INPUT_INC_2			 0x00000002
#define NV0039_SET_FORMAT_INPUT_INC_4			 0x00000004

#define NV0039_SET_FORMAT_OUTPUT_INC_1			 0x00000100
#define NV0039_SET_FORMAT_OUTPUT_INC_2			 0x00000200
#define NV0039_SET_FORMAT_OUTPUT_INC_4			 0x00000400

/* Class 063: Nv05ScaledImageFromMemory. */

typedef volatile struct {
	u32 NoOperation;             /* ignored                                     */
	u32 Notify;                  /* NV_NOTIFY_STYLE*                            */
	u32 Reserved00[0x01e];
	u32 SetContextDmaNotifies;   /* NV_CONTEXT_DMA                              */
	u32 SetContextDmaImage;      /* NV_CONTEXT_DMA                              */
	u32 SetContextPattern;
	u32 SetContextRop;
	u32 SetContextBeta1;
	u32 SetContextBeta4;
	u32 SetContextSurface;       /* the referenced destination surface          */
	u32 Reserved01[0x058];
	u32 SetColorConversion;      /* pixel conversion type                       */
	u32 SetColorFormat;          /* source colour format                        */
	u32 SetOperation;            /* general conversion type                     */
	u32 ClipPoint;               /* clipping point                              */
	u32 ClipSize;                /* clipping width and height                   */
	u32 ImageOutPoint;           /* destination out point                       */
	u32 ImageOutSize;            /* destination width and height                */
	u32 DuDx;                    /* x ratio                                     */
	u32 DvDy;                    /* y ratio                                     */
	u32 Reserved02[0x038];
	u32 ImageInSize;             /* source width and height                     */
	u32 ImageInFormat;           /* source pitch, origin and interpolation type */
	u32 ImageInOffset;           /* source offset                               */
	u32 ImageInPoint;            /* source in point                             */
	u32 Reserved03[0x6fc];
} Nv05ScaledImageFromMemory;

#define NV0063_SET_COLOR_CONVERSION_TYPE_DITHER		0x00000000
#define NV0063_SET_COLOR_CONVERSION_TYPE_TRUNCATE	0x00000001
#define NV0063_SET_COLOR_CONVERSION_TYPE_SUBTR_TRUNCATE 0x00000002

#define NV0063_SET_COLOR_FORMAT_LE_A1R5G5B5		0x00000001
#define NV0063_SET_COLOR_FORMAT_LE_X1R5G5B5		0x00000002
#define NV0063_SET_COLOR_FORMAT_LE_A8R8G8B8		0x00000003
#define NV0063_SET_COLOR_FORMAT_LE_X8R8G8B8		0x00000004
#define NV0063_SET_COLOR_FORMAT_LE_V8YB8U8YA8		0x00000005
#define NV0063_SET_COLOR_FORMAT_LE_YB8V8YA8U8		0x00000006
#define NV0063_SET_COLOR_FORMAT_LE_R5G6B5		0x00000007

#define NV0063_SET_OPERATION_MODE_SRCCOPY_AND		0x00000000
#define NV0063_SET_OPERATION_MODE_ROP_AND		0x00000001
#define NV0063_SET_OPERATION_MODE_BLEND_AND		0x00000002
#define NV0063_SET_OPERATION_MODE_SRCCOPY		0x00000003
#define NV0063_SET_OPERATION_MODE_SRCCOPY_PREMULT	0x00000004
#define NV0063_SET_OPERATION_MODE_BLEND_PREMULT		0x00000005

#define NV0063_FORMAT_ORIGIN_CENTER			0x00010000
#define NV0063_FORMAT_ORIGIN_CORNER			0x00020000

#define NV0063_FORMAT_INTERPOLATOR_ZOH			0x00000000
#define NV0063_FORMAT_INTERPOLATOR_FOH			0x01000000

/* Class 042: Nv04ContextSurfaces2d. */

typedef volatile struct {
	u32 NoOperation;             /* ignored                       */
	u32 Notify;                  /* NV_NOTIFY_STYLE*              */
	u32 Reserved00[0x01e];
	u32 SetContextDmaNotifies;   /* NV_CONTEXT_DMA                */
	u32 SetContextDmaSource;     /* NV_CONTEXT_DMA                */
	u32 SetContextDmaDestin;     /* NV_CONTEXT_DMA                */
	u32 Reserved01[0x05d];
	u32 SetFormat;               /* NV0042_SET_FORMAT_*           */
	u32 SetPitch;                /* destin_source U16_U16         */
	u32 SetOffsetSource;         /* byte offset of top-left pixel */
	u32 SetOffsetDestin;         /* byte offset of top-left pixel */
	u32 Reserved02[0x73c];
} Nv04ContextSurfaces2d;

#define NV0042_SET_FORMAT_LE_Y8                           0x00000001
#define NV0042_SET_FORMAT_LE_X1R5G5B5_Z1R5G5B5            0x00000002
#define NV0042_SET_FORMAT_LE_X1R5G5B5_O1R5G5B5            0x00000003
#define NV0042_SET_FORMAT_LE_R5G6B5                       0x00000004
#define NV0042_SET_FORMAT_LE_Y16                          0x00000005
#define NV0042_SET_FORMAT_LE_X8R8G8B8_Z8R8G8B8            0x00000006
#define NV0042_SET_FORMAT_LE_X8R8G8B8_O8R8G8B8            0x00000007
#define NV0042_SET_FORMAT_LE_X1A7R8G8B8_Z1A7R8G8B8        0x00000008
#define NV0042_SET_FORMAT_LE_X1A7R8G8B8_O1A7R8G8B8        0x00000009
#define NV0042_SET_FORMAT_LE_A8R8G8B8                     0x0000000a
#define NV0042_SET_FORMAT_LE_Y32                          0x0000000b

/* FIFO Channels structures. */

typedef volatile struct {
	u32 SetObject;	       /* current object, write only */
	u32 Reserved00[0x03];
	u16 Free;	       /* free count, read only	     */
	u16 Zero;	       /* zeroes, read only	     */
	u32 Reserved01[0x0b];
	u32 DmaPut;	       /* offset 28:2		     */
	u32 DmaGet;	       /* offset 28:2		     */
	u32 Reserved02[0x2e];
	union {		       /* start class methods, write only 0x100- */
		NvGeneric                 nvGeneric;
		Nv03MemoryToMemoryFormat  nv03MemoryToMemoryFormat;
		Nv04ContextSurfaces2d     nv04ContextSurfaces2d;
		Nv05ScaledImageFromMemory nv05ScaledImageFromMemory;
	} u;
} NvSubchannel;

#define NV_NUMBER_OF_SUBCHANNELS 8

typedef volatile struct {
	NvSubchannel sub[NV_NUMBER_OF_SUBCHANNELS];
} NvChannel;

#define NV03_MEMORY_TO_MEMORY_FORMAT 0x00000039
#define NV01_MEMORY_LOCAL_BANKED     0x0000003D
#define NV01_MEMORY_SYSTEM           0x0000003E
#define NV03_MEMORY_LOCAL_CURSOR     0x0000003F
#define NV01_MEMORY_LOCAL_LINEAR     0x00000040

/* Exported functions. */

struct rivatv_info;
void rivatv_checkAGP (struct rivatv_info *);
void rivatv_checkDMA (struct rivatv_info *);
void rivatv_cleanupDMA (struct rivatv_info *);
int  rivatv_startDMA (struct rivatv_info *, int);
int  rivatv_WaitOnDMA (struct rivatv_info *);
void rivatv_interrupt_DMA (struct rivatv_info *);
void rivatv_configureDMA (struct rivatv_info *);


/* --------------------------------------------------------------------------- */
/* This is what we possibly need to do DMA ourselves... bits of documentation. */
/* --------------------------------------------------------------------------- */


/* graphics object context:

   I think this is supposed to be the second word of the hash table entry.
   The format for that is (in NV4 and later):

   15- 0  instance_addr >> 4
   17-16  engine (here uses 1 = graphics)
   28-24  channel id
   31	  valid (use 1)

   For NV3 that is:

   15- 0  instance_addr >> 4
   22-16  object type (class)
   23     engine (here uses 1 = graphics)
   30-24  channel id
*/

/* hash table key:

   Used as offset in the instance RAM.
   That depends on the hash table size, which is different for the 'nvidia' (16K, 11 bits)
   and the 'nv' (4K, 9 bits) driver.  For the first, the bits xor'ed are

      31 30  29 28 27 26 25 24 23 22 
   21 20 19  18 17 16 25 14 13 12 11 
   10 09 08  07 06 05 04 03 02 01 00 
   ch ch ch  ch -- -- -- -- -- -- -- 

   For the 'nv' driver, it is

		31 30 29 28 27 
   26  25 24 23 22 21 20 19 18 
   17  16 25 14 13 12 11 10 09 
   08  07 06 05 04 03 02 01 00 
   ch  ch ch ch -- -- -- -- -- 
*/

/* Where is the hash table located:

   Base address and size can be calculated from this register:

   ht_base = 0x1000 *  GetBitField (pNv->PFIFO[0x0210/4],8:4);
   ht_size = 0x1000 << GetBitField (pNv->PFIFO[0x0210/4],17:16);

   and the hash table will be located between address PRAMIN + ht_base and
   PRAMIN + ht_base + ht_size.	Each hash table entry has two longwords.
*/

/* Can we determine a free channel + subchannel parsing the instance ram, 
   fifo context table and/or hash table ? 

   In principle, yes.  But it would be no use: The 'nv' server doesn't 
   initialize the hash table properly,	but we *know* which channels and
   subchannels it uses.	 For the 'nvidia' server, we cannot use a channel
   without going through the ioctl interface, anyway, because the internal
   data structures are not properly initialized.
*/

/* Where should be notifiers be located ?  What are notifiers ? 
   How do they get referenced ?

   Notifiers are system memory or card memory locations that get written
   to when some action is complete.  They are referenced as 'dma entries'
   in the hash table / instance ram.
*/

/* How do we wait for DMA completion (by notifiers) ?

   Either repeatedly read the notifier address and wait until it changes,
   or enable a 'wakeup' interrupt by writing NOTIFY_WRITE_LE_AWAKEN into
   the 'notify' field of the object in the channel.  My guess is that 
   this causes an interrupt in PGRAPH/NOTIFY as soon as the transfer is
   completed.  Clients probably can use poll on the nv* devices to get this 
   event.  All this is a guess.	 I don't know any details, and I have not
   tested is.  Also, I have no idea how the 'nvdriver' reacts if it gets 
   notify events that are not registered.

   Writing NV_NOTIFY_WRITE_LE_AWAKEN into the 'Notify' field of an object
   in a channel really causes an interrupt in the PGRAPH engine.  Thus
   we can determine whether a DMA transfer has finished in the interrupt
   handler.
*/

#endif /* __DMA_RIVA_H */
