/* ------------------------------------------------------------------------- */
/*   i2c-riva.c I2C adapter driver for NVIDIA display adapters		     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2000 Ferenc Bakonyi <fero@drama.obuda.kando.hu>
 *   Based on various sources by Simon G. Vogl
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>

#include "rivatv.h"
#include "i2c-riva.h"
#include "id.h"

/* ------------------------------------------------------------------------- *
 *
 * I2C bus
 *
 * ------------------------------------------------------------------------- */

#define BUS0_USED 0x01
#define BUS1_USED 0x02
#define BUS2_USED 0x04

static int __init rivatv_setup_i2c (struct rivatv_info *info,
				    const struct i2c_riva_initdata *i2c_init)
{
	int err;
	struct i2c_riva_info *i2c = info->i2c;
	struct i2c_algo_bit_data *algo = i2c_init->algo;
	struct i2c_adapter *bus0 = i2c_init->bus0;
	struct i2c_adapter *bus1 = i2c_init->bus1;
	struct i2c_adapter *bus2 = i2c_init->bus2;

	// No I2C bus used so far
	i2c->bus_used = 0;

	switch (info->chip.realarch) {
	case NV_ARCH_17:
	case NV_ARCH_18:
	case NV_ARCH_25:
	case NV_ARCH_28: // Bus 2 is used by GeForce4 & 4MX
	case NV_ARCH_30:
	case NV_ARCH_31:
	case NV_ARCH_32:
	case NV_ARCH_33:
	case NV_ARCH_34:
		memcpy (&i2c->i2c_bus2, bus2, sizeof (i2c->i2c_bus2));
		memcpy (&i2c->bit_riva_data2, algo, sizeof (i2c->bit_riva_data2));
		i2c->bus_data2.Read_Reg = 0x50;
		i2c->bus_data2.Write_Reg = 0x51;
		i2c->bus_data2.IO_Addr = info->chip.PCIO;
		i2c->bus_data2.chip = &info->chip;
		i2c->bit_riva_data2.data = &i2c->bus_data2;
		i2c->i2c_bus2.algo_data = &i2c->bit_riva_data2;
		i2c_set_adapdata (&i2c->i2c_bus2, i2c);
		err = i2c_init->bus_init (i2c->bit_riva_data2.data);
		if (!err) err = i2c_bit_add_bus (&i2c->i2c_bus2);
		if (!err) i2c->bus_used |= BUS2_USED;
		// fall through

	case NV_ARCH_04:
	case NV_ARCH_05:
	case NV_ARCH_10:
	case NV_ARCH_11:
	case NV_ARCH_15:
	case NV_ARCH_20: // Bus 1 is used by Riva TNT and up
		memcpy (&i2c->i2c_bus1, bus1, sizeof (i2c->i2c_bus1));
		memcpy (&i2c->bit_riva_data1, algo, sizeof (i2c->bit_riva_data1));
		i2c->bus_data1.Read_Reg = 0x36;
		i2c->bus_data1.Write_Reg = 0x37;
		i2c->bus_data1.IO_Addr = info->chip.PCIO;
		i2c->bus_data1.chip = &info->chip;
		i2c->bit_riva_data1.data = &i2c->bus_data1;
		i2c->i2c_bus1.algo_data = &i2c->bit_riva_data1;
		i2c_set_adapdata (&i2c->i2c_bus1, i2c);
		err = i2c_init->bus_init (i2c->bit_riva_data1.data);
		if (!err) err = i2c_bit_add_bus (&i2c->i2c_bus1);
		if (!err) i2c->bus_used |= BUS1_USED;
		// fall through

	case NV_ARCH_03: // Bus 0 is used by all architectures from Riva 128 up.
		memcpy (&i2c->i2c_bus0, bus0, sizeof (i2c->i2c_bus0));
		memcpy (&i2c->bit_riva_data0, algo, sizeof (i2c->bit_riva_data0));
		i2c->bus_data0.Read_Reg = 0x3e;
		i2c->bus_data0.Write_Reg = 0x3f;
		i2c->bus_data0.IO_Addr = info->chip.PCIO;
		i2c->bus_data0.chip = &info->chip;
		i2c->bit_riva_data0.data = &i2c->bus_data0;
		i2c->i2c_bus0.algo_data = &i2c->bit_riva_data0;
		i2c_set_adapdata (&i2c->i2c_bus0, i2c);
		err = i2c_init->bus_init (i2c->bit_riva_data0.data);
		if (!err) err = i2c_bit_add_bus (&i2c->i2c_bus0);
		if (!err) i2c->bus_used |= BUS0_USED;
		break;

	default:
		PRINTK_ERR ("init I2C failed: no such architecture NV%X\n", info->chip.realarch);
		break;
	}
	return i2c->bus_used ? 0 : -1;
}

static void __exit rivatv_release_i2c (struct rivatv_info *info)
{
	if (info->i2c->bus_used & BUS0_USED) i2c_bit_del_bus (&info->i2c->i2c_bus0);
	if (info->i2c->bus_used & BUS1_USED) i2c_bit_del_bus (&info->i2c->i2c_bus1);
	if (info->i2c->bus_used & BUS2_USED) i2c_bit_del_bus (&info->i2c->i2c_bus2);
	info->i2c->bus_used = 0;
}

static struct __initdata i2c_riva_initdata driver_init;

int __init rivatv_register_i2c (struct rivatv_info *info)
{
	int err = -ENOMEM;
	struct i2c_riva_info *i2c;

	if (info == NULL)
		return -ENODEV;

	PRINTK_INFO ("I2C adapter driver for NVIDIA cards\n");

	i2c = kmalloc (sizeof (struct i2c_riva_info), GFP_KERNEL);
	if (!i2c)
		goto err_out;
	memset (i2c, 0, sizeof (struct i2c_riva_info));

	info->i2c = i2c;
	err = rivatv_setup_i2c (info, &driver_init);
	if (err)
		goto err_out_kfree;

	return 0;

err_out_kfree:
	info->i2c = NULL;
	kfree (i2c);
err_out:
	return err;
}

void __exit rivatv_unregister_i2c (struct rivatv_info *info)
{
	struct i2c_riva_info *i2c;
	
	if (info == NULL)
		return;
	i2c = info->i2c;
	rivatv_release_i2c (info);
	info->i2c = NULL;
	kfree (i2c);
}

/* Check status of the tuner chip in TV-Boxes. */
static void __init rivatv_check_tuner (struct i2c_client *tuner)
{
	unsigned char status;

	if (tvbox) {
		i2c_master_recv (tuner, &status, 1);
		if (status & 0x80) {
			PRINTK_ERR ("tuner in TV-Box is available, but off\n");
		}
	}
}

/* Check status of the tuner chip in TV-Boxes. */
void rivatv_enable_tuner (struct i2c_client *irchip, unsigned char channel)
{
	unsigned char codes[1];

	if (irchip) {
		codes[0] = channel;
		i2c_master_send (irchip, codes, 1);
	}
}

/*---------------------------------------------------------------------- 
 *
 * I2C adapter driver
 *
 *----------------------------------------------------------------------*/

static void bit_riva_setscl (void *bus, int state);
static void bit_riva_setsda (void *bus, int state);
static int  bit_riva_getscl (void *bus);
static int  bit_riva_getsda (void *bus);
static int  bit_riva_reg (struct i2c_client *client);
static int  bit_riva_unreg (struct i2c_client *client);

static spinlock_t riva_reg_lock = SPIN_LOCK_UNLOCKED;

/* Riva I2C bus manipulation functions and data */
static struct __initdata i2c_algo_bit_data riva_bit_data = {
	.data	 = NULL,	/* struct riva_i2c_bus * */
	.setsda	 = bit_riva_setsda,
	.setscl	 = bit_riva_setscl,
	.getsda	 = bit_riva_getsda,
	.getscl	 = bit_riva_getscl,
	.udelay	 = 5,		/* Waits, Timeout: guessed values */
	.mdelay	 = 5,
	.timeout = 10,
};

/* For registering I2C adapter driver */
static struct __initdata i2c_adapter i2c_riva_bus0 = {
	I2C_DEVNAME ("NVIDIA display adapter bus 0"),
	.id                = I2C_HW_B_RIVA,
	.client_register   = bit_riva_reg,
	.client_unregister = bit_riva_unreg,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use           = i2c_adapter_inc_use,
	.dec_use	   = i2c_adapter_dec_use,
#else
	.owner             = THIS_MODULE,
#endif
#ifdef I2C_CLASS_TV_ANALOG
	.class             = I2C_CLASS_TV_ANALOG | I2C_CLASS_DDC,
#else
#ifdef I2C_ADAP_CLASS_TV_ANALOG
        .class             = I2C_ADAP_CLASS_TV_ANALOG | I2C_ADAP_CLASS_DDC,
#endif
#endif
};

static struct __initdata i2c_adapter i2c_riva_bus1 = {
	I2C_DEVNAME ("NVIDIA display adapter bus 1"),
	.id                = I2C_HW_B_RIVA,
	.client_register   = bit_riva_reg,
	.client_unregister = bit_riva_unreg,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use           = i2c_adapter_inc_use,
	.dec_use           = i2c_adapter_dec_use,
#else
	.owner             = THIS_MODULE,
#endif
#ifdef I2C_CLASS_TV_ANALOG
	.class             = I2C_CLASS_TV_ANALOG | I2C_CLASS_HWMON,
#else
#ifdef I2C_ADAP_CLASS_TV_ANALOG
        .class             = I2C_ADAP_CLASS_TV_ANALOG | I2C_ADAP_CLASS_SMBUS,
#endif
#endif
};

static struct __initdata i2c_adapter i2c_riva_bus2 = {
	I2C_DEVNAME ("NVIDIA display adapter bus 2"),
	.id                = I2C_HW_B_RIVA,
	.client_register   = bit_riva_reg,
	.client_unregister = bit_riva_unreg,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use           = i2c_adapter_inc_use,
	.dec_use           = i2c_adapter_dec_use,
#else
	.owner             = THIS_MODULE,
#endif
#ifdef I2C_CLASS_TV_ANALOG
	.class             = I2C_CLASS_TV_ANALOG | I2C_CLASS_HWMON,
#else
#ifdef I2C_ADAP_CLASS_TV_ANALOG
        .class             = I2C_ADAP_CLASS_TV_ANALOG | I2C_ADAP_CLASS_SMBUS,
#endif
#endif
};

/* Most likely, future nVidia chips will have no I2C support anymore,
   due to Philips suing companies for using their invention.
   Don't you just looooooove lawsuits! *SIGH* */

/*---------------------------------------------------------------------- 
 *
 * Low level hardware manipulation functions
 *
 *----------------------------------------------------------------------*/


/* Possibly: #define DUAL_HEAD_MASK (~0x00000110) */
#define DUAL_HEAD_MASK (~0x00000010)
#define SAVE_DUAL_HEAD_STATE(chip, head)		           \
	(head) = VID_RD32 ((chip)->PMC, 0x600860);	           \
	VID_WR32 ((chip)->PMC, 0x600860, (head) & DUAL_HEAD_MASK);
#define RESTORE_DUAL_HEAD_STATE(chip, head)		           \
	VID_WR32 ((chip)->PMC, 0x600860, (head));

/* Locking */
#define LOCK_I2C_RIVA(chip, flags)			  \
	spin_lock_irqsave (&riva_reg_lock, (flags));	  \
	lock_kernel ();					  \
	VID_WR32 ((chip)->PMC, 0x140, 0);
#define UNLOCK_I2C_RIVA(chip, flags)			  \
	VID_WR32 ((chip)->PMC, 0x140, 1);		  \
	unlock_kernel ();				  \
	spin_unlock_irqrestore (&riva_reg_lock, (flags));

#if defined (__i386__) && RIVATV_ENABLE_ASM
#define DELAY_I2C_RIVA(count)					    \
	__asm__ __volatile__ ("	  movl	%0, %%ecx   \n"		    \
			      "1: inb	$0x80, %%al \n"		    \
			      "	  decl	%%ecx	    \n"		    \
			      "	  jnz	1b	    \n"		    \
			      : : "i" (count) : "cc", "eax", "ecx")
#else
#define DELAY_I2C_RIVA(count) __asm__ __volatile__ ("" : : : "memory")
#endif

/* Set the clock line on the I2C bus */
static void bit_riva_setscl (void *bus, int state)
{
	struct riva_i2c_bus *rb = (struct riva_i2c_bus *) bus;
	unsigned char data;
	unsigned long flags;
	u32 head;

	LOCK_I2C_RIVA (rb->chip, flags);
	SAVE_DUAL_HEAD_STATE (rb->chip, head);

	rb->chip->lock (rb->chip, 0);
	VID_WR08 (rb->IO_Addr, 0x3d4, rb->Write_Reg);
	data = VID_RD08 (rb->IO_Addr, 0x3d5) & 0xf0;
	if (state)
		data |= 0x20;
	else
		data &= ~0x20;
	VID_WR08 (rb->IO_Addr, 0x3d4, rb->Write_Reg);
	VID_WR08 (rb->IO_Addr, 0x3d5, data | 0x01);

	RESTORE_DUAL_HEAD_STATE (rb->chip, head);
	UNLOCK_I2C_RIVA (rb->chip, flags);
}

/* Set the data line on the I2C bus */
static void bit_riva_setsda (void *bus, int state)
{
	struct riva_i2c_bus *rb = (struct riva_i2c_bus *) bus; 
	unsigned char data;
	unsigned long flags;
	u32 head;

	LOCK_I2C_RIVA (rb->chip, flags);
	SAVE_DUAL_HEAD_STATE (rb->chip, head);

	rb->chip->lock (rb->chip, 0);
	VID_WR08 (rb->IO_Addr, 0x3d4, rb->Write_Reg);
	data = VID_RD08 (rb->IO_Addr, 0x3d5) & 0xf0;
	if (state)
		data |= 0x10;
	else
		data &= ~0x10;
	VID_WR08 (rb->IO_Addr, 0x3d4, rb->Write_Reg);
	VID_WR08 (rb->IO_Addr, 0x3d5, data | 0x01);

	RESTORE_DUAL_HEAD_STATE (rb->chip, head);
	UNLOCK_I2C_RIVA (rb->chip, flags);
}

/* Read the clock line on the I2C bus */
static int bit_riva_getscl (void *bus)
{
	struct riva_i2c_bus *rb = (struct riva_i2c_bus *) bus; 
	int data = 0;
	unsigned long flags;
	u32 head;

	LOCK_I2C_RIVA (rb->chip, flags);
	SAVE_DUAL_HEAD_STATE (rb->chip, head);

	rb->chip->lock (rb->chip, 0);
	VID_WR08 (rb->IO_Addr, 0x3d4, rb->Read_Reg);
	if (VID_RD08 (rb->IO_Addr, 0x3d5) & 0x04)
		data = 1;

	RESTORE_DUAL_HEAD_STATE (rb->chip, head);
	UNLOCK_I2C_RIVA (rb->chip, flags);
	return data;
}

/* Read the data line on the I2C bus */
static int bit_riva_getsda (void *bus)
{
	struct riva_i2c_bus *rb = (struct riva_i2c_bus *) bus; 
	int data = 0;
	unsigned long flags;
	u32 head;

	LOCK_I2C_RIVA (rb->chip, flags);
	SAVE_DUAL_HEAD_STATE (rb->chip, head);

	rb->chip->lock (rb->chip, 0);
	VID_WR08 (rb->IO_Addr, 0x3d4, rb->Read_Reg);
	if (VID_RD08 (rb->IO_Addr, 0x3d5) & 0x08)
		data = 1;

	RESTORE_DUAL_HEAD_STATE (rb->chip, head);
	UNLOCK_I2C_RIVA (rb->chip, flags);
	return data;
}

/*---------------------------------------------------------------------- 
 *
 * I2C adapter driver interface functions
 *
 *----------------------------------------------------------------------*/

/* Reset the I2C bus lines once */
static int __init bit_riva_init (struct riva_i2c_bus *bus)
{
	bit_riva_setscl (bus, 1);
	bit_riva_setsda (bus, 1);
	return 0;
}

/* This is the top of the thingy */
static int bit_riva_reg (struct i2c_client *client)
{
	struct i2c_riva_info *i2c = i2c_get_adapdata (client->adapter);
	int id = client->driver->id;

	DPRINTK2 ("chip registration requested: %d (%s)\n", id, i2c_clientname (client));

	if (i2c->video_decoder == NULL)
		if (id == I2C_DRIVERID_SAA7111A || id == I2C_DRIVERID_SAA7113 ||
		    id == I2C_DRIVERID_SAA7108  || id == I2C_DRIVERID_VPX32XX ||
		    id == I2C_DRIVERID_TW98     || id == I2C_DRIVERID_SAA7174) {
			i2c->video_decoder = client;
			PRINTK_INFO ("video decoder chip registered\n");
			return 0;
		}
	if (i2c->tuner == NULL)
		if (id == I2C_DRIVERID_TUNER) {
			i2c->tuner = client;
			PRINTK_INFO ("video tuner chip registered\n");
			rivatv_check_tuner (client);
			return 0;
		}
	if (i2c->ir_chip == NULL)
		if (id == I2C_DRIVERID_EXP3 && (client->addr == 0x21 || client->addr == 0x23)) {
			i2c->ir_chip = client;
			PRINTK_INFO ("IR chip registered\n");
			rivatv_enable_tuner (client, RIVATV_TUNER_COMPOSITE);
			return 0;
		}
	if (i2c->audio_processor == NULL)
		/* I2C_DRIVERID_TDA8425 */
		if (id == I2C_DRIVERID_TVAUDIO && (client->addr == 0x41)) {
			i2c->audio_processor = client;
			PRINTK_INFO ("audio processor chip registered\n");
			return 0;
		}
	if (i2c->audio_decoder == NULL)
		/* I2C_DRIVERID_TDA9850, I2C_DRIVERID_TDA9873 and I2C_DRIVERID_TDA9874 */
		if (id == I2C_DRIVERID_TVAUDIO && 
		    (client->addr == 0x5a || client->addr == 0x5b || client->addr == 0x58)) {
			i2c->audio_decoder = client;
			PRINTK_INFO ("audio decoder chip registered\n");
			return 0;
		}
	return -ENODEV;
}

static int bit_riva_unreg (struct i2c_client *client)
{
	struct i2c_riva_info *i2c = i2c_get_adapdata (client->adapter);
	int id = client->driver->id;

	if (i2c->video_decoder != NULL)
		if (id == I2C_DRIVERID_SAA7111A || id == I2C_DRIVERID_SAA7113 ||
		    id == I2C_DRIVERID_SAA7108  || id == I2C_DRIVERID_VPX32XX ||
		    id == I2C_DRIVERID_TW98     || id == I2C_DRIVERID_SAA7174) {
			i2c->video_decoder = NULL;
			PRINTK_INFO ("video decoder chip unregistered\n");
		}
	if (i2c->tuner != NULL)
		if (id == I2C_DRIVERID_TUNER) {
			i2c->tuner = NULL;
			PRINTK_INFO ("video tuner chip unregistered\n");
		}
	if (i2c->ir_chip != NULL)
		if (id == I2C_DRIVERID_EXP3) {
			i2c->ir_chip = NULL;
			PRINTK_INFO ("IR chip unregistered\n");
		}
	if (i2c->audio_processor != NULL)
		if (id == I2C_DRIVERID_TDA8425) {
			i2c->audio_processor = NULL;
			PRINTK_INFO ("audio processor chip unregistered\n");
		}
	if (i2c->audio_decoder != NULL)
		if (id == I2C_DRIVERID_TDA9850 || id == I2C_DRIVERID_TDA9873 || id == I2C_DRIVERID_TDA9874) {
			i2c->audio_decoder = NULL;
			PRINTK_INFO ("audio decoder chip unregistered\n");
		}
	return 0;
}

/*---------------------------------------------------------------------- 
 *
 * Modularization
 *
 *----------------------------------------------------------------------*/

static struct __initdata i2c_riva_initdata driver_init = {
	.algo	  = &riva_bit_data,
	.bus0	  = &i2c_riva_bus0,
	.bus1	  = &i2c_riva_bus1,
	.bus2	  = &i2c_riva_bus2,
	.bus_init = bit_riva_init
};
