/* ------------------------------------------------------------------------- */
/*   procfs-riva.c procfs driver for NVIDIA display adapters		     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2002, 2003 Stefan Jahn <stefan@lkcc.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/init.h>

#include "i2c-riva.h"
#include "rivatv.h"

#ifdef CONFIG_PROC_FS

/* Read function - delivers info. */

static int rivatv_procfs_read (char *page, char **start,
			       off_t offset, int count,
			       int *eof, void *data)
{
	struct rivatv_info *info = (struct rivatv_info *) data;
	struct rivatv_stats *stats = &info->sched.stats;
	char tuner[33], vdecoder[33], adecoder[33], aprocessor[33];
	int len;

	/* No file positioning. */
	if (offset > 0)
		return 0;

	/* Get client names. */
	if (info->i2c->tuner) {
		strncpy (tuner, i2c_clientname (info->i2c->tuner), sizeof (tuner) - 1);
		tuner[sizeof (tuner) - 1] = '\0';
	} else {
		strcpy (tuner, "unavailable");
	}
	if (info->i2c->video_decoder) {
		strncpy (vdecoder, i2c_clientname (info->i2c->video_decoder), sizeof (vdecoder) - 1);
		vdecoder[sizeof (vdecoder) - 1] = '\0';
	} else {
		strcpy (vdecoder, "unavailable");
	}
	if (info->i2c->audio_decoder) {
		strncpy (adecoder, i2c_clientname (info->i2c->audio_decoder), sizeof (adecoder) - 1);
		adecoder[sizeof (adecoder) - 1] = '\0';
	} else {
		strcpy (adecoder, "unavailable");
	}
	if (info->i2c->audio_processor) {
		strncpy (aprocessor, i2c_clientname (info->i2c->audio_processor), sizeof (aprocessor) - 1);
		aprocessor[sizeof (aprocessor) - 1] = '\0';
	} else {
		strcpy (aprocessor, "unavailable");
	}

	spin_lock_irq (&info->sched.sched_lock);

	/* Fill the buffer and get its length. */
	len = sprintf (page,
                       "nVidia Chip:    %s\n"
                       "Model:          %s\n"
                       "Architecture:   NV%X (NV%X)\n"
                       "Access:         Control [0x%08lx-0x%08lx]\n"
                       "                FB      [0x%08lx-0x%08lx]\n"
                       "Interrupts:     %lu out of %lu (DMA: %lu, Overlay: %lu, Missing: %ld)\n"
                       "Device:         %s\n"
                       "VideoDecoder:   %s\n"
                       "Tuner:          %s\n"
                       "AudioDecoder:   %s\n"
                       "AudioProcessor: %s\n"
		       "IR chip:        %s\n",
		       info->driver,
		       info->nicecardname, info->chip.arch, info->chip.realarch,
		       info->base0_region, info->base0_region + info->base0_region_size - 1,
		       info->base1_region, info->base1_region + info->base1_region_size - 1,
		       stats->decoder_interrupts, stats->interrupts, 
		       stats->dma_interrupts, stats->overlay_interrupts,
		       (long) (stats->interrupts - stats->decoder_interrupts - 
			       stats->overlay_interrupts - stats->dma_interrupts),
		       info->device_busy ? "busy" : "available",
		       vdecoder, tuner, adecoder, aprocessor,
		       info->i2c->ir_chip ? "present" : "unavailable");

	spin_unlock_irq (&info->sched.sched_lock);

	/* Return the length. */
	return len;
}

#define SKIP_SPACES() do {                                                  \
	while (idx <= count && (buffer[idx] == ' ' || buffer[idx] == '\t')) \
		idx++;                                                      \
        if (idx > count) error++; } while (0)

#define PARSE_STRING(str,length) do {                       \
	int i = 0;                                          \
        SKIP_SPACES ();                                     \
	while (i < (length - 1) && idx <= count &&          \
               buffer[idx] != ' ' && buffer[idx] != '\t') { \
		str[i++] = buffer[idx++]; }                 \
        str[i] = 0;                                         \
        if (idx > count) error++; } while (0)

#define PARSE_HEX(hex) do {                               \
	unsigned char b;                                  \
        SKIP_SPACES ();                                   \
	hex = 0;                                          \
	b = buffer[idx++];                                \
	while (idx <= count && ((b >= '0' && b <= '9') || \
			       (b >= 'a' && b <= 'f') ||  \
			       (b >= 'A' && b <= 'F'))) { \
		hex <<= 4;                                \
		if (b >= '0' && b <= '9')                 \
			hex += b - '0';                   \
		else if (b >= 'a' && b <= 'f')            \
			hex += b - 'a' + 10;              \
		else if (b >= 'A' && b <= 'F')            \
			hex += b - 'A' + 10;              \
		b = buffer[idx++];                        \
	}                                                 \
        if (idx > count) error++; } while (0)

/* Write function.  Set values of registers... Be *very* careful! */

static int rivatv_procfs_write (struct file *file, const char *buffer,
				unsigned long count, void *data)
{
	struct rivatv_info *info = (struct rivatv_info *) data;
	unsigned long reg, value;
	int error = 0, idx = 0, retval = -EINVAL;
	char device[256], equal[16];

	PARSE_STRING (device, 256);
	if (error)
		goto parse_error;
	PARSE_HEX (reg);
	if (error)
		goto parse_error;
	PARSE_STRING (equal, 16);
	if (error)
		goto parse_error;
	PARSE_HEX (value);
	if (error)
		goto parse_error;
	
	if (!strcmp (device, "VideoDecoder:")) {
		if (info->i2c->video_decoder != NULL) {
			value = (value & 0xff) | ((reg & 0x0fff) << 8); /* also 12bit addresses */
			info->i2c->video_decoder->driver->command (info->i2c->video_decoder, WRITE_REGISTER, &value);
		}
	}
	else if (!strcmp (device, "Tuner:")) {
		if (info->i2c->tuner != NULL) {
			value = (value & 0xff) | ((reg & 0xff) << 8);
			info->i2c->tuner->driver->command (info->i2c->tuner, WRITE_REGISTER, &value);
		}
	}
	else if (!strcmp (device, "AudioDecoder:")) {
		if (info->i2c->audio_decoder != NULL) {
			value = (value & 0xff) | ((reg & 0xff) << 8);
			info->i2c->audio_decoder->driver->command (info->i2c->audio_decoder, WRITE_REGISTER, &value);
		}
	}
	else if (!strcmp (device, "AudioProcessor:")) {
		if (info->i2c->audio_processor != NULL) {
			value = (value & 0xff) | ((reg & 0xff) << 8);
			info->i2c->audio_processor->driver->command (info->i2c->audio_processor, WRITE_REGISTER, &value);
		}
	}
	else if (!strcmp (device, "PMC:")) {
		DPRINTK ("Writing Master Control Register 0x%08lx -> 0x%08lx\n", reg, value);
		VID_WR32 (info->chip.PMC, reg, value);
	}
	retval = count;

 parse_error:
	return retval;
}

/* Initialize the procfs entry - register the proc file. */

int __init rivatv_register_procfs (struct rivatv_info *info)
{
	static struct proc_dir_entry *rivatv_file;
	char filename[9];

	/* create using convenience function */
	sprintf (filename, "rivatv%d", info->nr);
	rivatv_file = create_proc_entry (filename, S_IFREG | S_IRUGO | S_IWUSR, proc_root_driver);

	if (rivatv_file == NULL)
		return -ENOMEM;

        rivatv_file->data = info;
        rivatv_file->read_proc = rivatv_procfs_read;
        rivatv_file->write_proc = rivatv_procfs_write;
        rivatv_file->size = 0;
	rivatv_file->owner = THIS_MODULE;

	/* single card backward compatibility */
	if (info->nr == 0)
		proc_symlink ("rivatv", proc_root_driver, "rivatv0");

	PRINTK_INFO ("procfs file registered for rivatv%d\n", info->nr);
	return 0;
}

/* Cleanup - unregister procfs file. */

void __exit rivatv_unregister_procfs (struct rivatv_info *info)
{
	char filename[9];

	sprintf (filename, "rivatv%d", info->nr);
	remove_proc_entry (filename, proc_root_driver);

	/* single card backward compatibility */
	if (info->nr == 0)
		remove_proc_entry ("rivatv", proc_root_driver);

	PRINTK_INFO ("procfs file unregistered for rivatv%d\n", info->nr);
}

#endif /* CONFIG_PROC_FS */
