/* ------------------------------------------------------------------------- */
/*   rivatv-driver.c Video4Linux driver for NVIDIA display adapters	     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2000 Ferenc Bakonyi <fero@drama.obuda.kando.hu>
 *   Based on various sources by Alan Cox
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/pci.h>
#include <linux/video_decoder.h>
#include <linux/videotext.h>

#include "i2c-riva.h"
#include "rivatv.h"
#include "tuner.h"
#include "audiochip.h"

/*----------------------------------------------------------------------
 *
 * Video4Linux video capture device driver part
 *
 *----------------------------------------------------------------------*/

/* Definition of available video sources. */

#define RIVATV_SOURCES	        2
#define RIVATV_COMPOSITE_SOURCE 0
#define RIVATV_SVIDEO_SOURCE    1
#define RIVATV_TUNER_SOURCE     2

static struct video_channel rivatv_channel[RIVATV_SOURCES + 1] = {
	{ 0, "Composite",  0, 0,	      VIDEO_TYPE_CAMERA, VIDEO_MODE_AUTO },
	{ 1, "S-Video",	   0, 0,	      VIDEO_TYPE_CAMERA, VIDEO_MODE_AUTO },
	{ 2, "Television", 1, VIDEO_VC_TUNER, VIDEO_TYPE_TV,	 VIDEO_MODE_AUTO }
};

static struct rivatv_source rivatv_source[] = {
	{ I2C_DRIVERID_SAA7111A, 
	  { 3, 4, -1 }, 
	  { VIDEO_PALETTE_UYVY, VIDEO_PALETTE_RGB24, VIDEO_PALETTE_RGB565, 
	    VIDEO_PALETTE_YUV422, VIDEO_PALETTE_YUYV, VIDEO_PALETTE_YUV411, -1 } ,
	  VIDEO_PALETTE_UYVY
	},
	{ I2C_DRIVERID_SAA7113,
	  { 3, 6, 1 },
	  { VIDEO_PALETTE_UYVY, -1 } ,
	  VIDEO_PALETTE_UYVY
	},
	{ I2C_DRIVERID_SAA7108,
	  { 0, 6, 0 },
	  { VIDEO_PALETTE_UYVY, -1 } ,
	  VIDEO_PALETTE_UYVY
	},
	{ I2C_DRIVERID_SAA7174,
	  { 0, 6, 0 },
	  { VIDEO_PALETTE_UYVY, -1 } ,
	  VIDEO_PALETTE_UYVY
	},
	{ I2C_DRIVERID_VPX32XX,
	  { 1, 7, -1 },
	  { VIDEO_PALETTE_UYVY, -1 } ,
	  VIDEO_PALETTE_UYVY
	},
	{ I2C_DRIVERID_TW98,
	  { 2, 5, -1 },
	  { VIDEO_PALETTE_UYVY, -1 } ,
	  VIDEO_PALETTE_UYVY
	},
	{ -1 }
};

/* Setup the video decoder source for a given I2C chip id. */
static int rivatv_setup_source (struct rivatv_info *info, int i2c_id)
{
	struct rivatv_source *source = rivatv_source;

	/* determine number of input source */
	info->sources = RIVATV_SOURCES;
	if (info->tuner_required != -1 && info->i2c->tuner)
		info->sources++;

	/* go through list of possible decoder chips */
	while (source->id != -1) {
		if (source->id == i2c_id) {
			info->source = source;
			/* copy default values for input modes if card has not 
			   been detected correctly */
			if (info->card.identified == 0)
				memcpy (info->card.mode, source->mode, sizeof (info->card.mode));
			return 0;
		}
		source++;
	}
	return -1;
}

/* Start/stop video decoder output. */
static void rivatv_decoder_output (struct rivatv_info *info, int enable)
{
	struct i2c_client *decoder = info->i2c->video_decoder;

	if (decoder) {
		down (&info->decoder_lock);
		decoder->driver->command (decoder, DECODER_ENABLE_OUTPUT, &enable);
		up (&info->decoder_lock);
	}
}

/* Standard character-device-driver functions */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static int rivatv_video_open (struct video_device *dev, int nb)
{
#else
static int rivatv_video_open  (struct inode *inode, struct file *file)
{
        struct video_device *dev = video_devdata (file);
#endif
	struct rivatv_info *info = video_get_drvdata (dev);

	/* open video device just once */
	if (info->device_busy) {
		DPRINTK ("video device busy: %d\n", info->device_busy);
		return -EBUSY;
	}

	/* check for the video decoder device */
	if (info->i2c == NULL || info->i2c->video_decoder == NULL) {
		DPRINTK ("no video decoder device registered\n");
		return -ENODEV;
	}

	info->device_busy++;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 0)
	file->private_data = dev;
#endif

	/* setup decoder source */
	if (info->source == NULL) {
		if (rivatv_setup_source (info, info->i2c->video_decoder->driver->id)) {
			PRINTK_ERR ("no source defined for decoder chip 0x%04x\n", 
				    info->i2c->video_decoder->driver->id);
		}
	}

	/* increment module usage counter of I2C clients */
	kcompat_i2c_client_get (info->i2c->video_decoder);
	if (info->tuner_required != -1 && info->i2c->tuner != NULL) {
		kcompat_i2c_client_get (info->i2c->tuner);
		info->tuner_used++;
		info->tuner_selected = 0;
	}
	if (info->audio_required != -1) {
		if (info->i2c->audio_decoder != NULL) {
			kcompat_i2c_client_get (info->i2c->audio_decoder);
			info->audio_decoder_used++;
		}
		if (info->i2c->audio_processor != NULL) {
			kcompat_i2c_client_get (info->i2c->audio_processor);
			info->audio_processor_used++;
		}
	}
	return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static void rivatv_video_close (struct video_device *dev)
{
#else
static int rivatv_video_close  (struct inode *inode, struct file *file)
{
        struct video_device *dev = video_devdata (file);
#endif
	struct rivatv_info *info = video_get_drvdata (dev);

	/* reset the busy counter (see open()) */
	if (info->device_busy > 0)
		info->device_busy--;

	rivatv_reset_queue (info);
	rivatv_overlay_stop (info);
	rivatv_decoder_output (info, 0);
	rivatv_video_stop (info);

	/* reset decoder source */
	info->source = NULL;

	/* decrement module usage counter of I2C clients */
	if (info->tuner_used) {
		kcompat_i2c_client_put (info->i2c->tuner);
		info->tuner_used--;
		info->tuner_selected = 0;
	}
	if (info->audio_decoder_used) {
		kcompat_i2c_client_put (info->i2c->audio_decoder);
		info->audio_decoder_used--;
	}
	if (info->audio_processor_used) {
		int input = AUDIO_MUTE;
		info->i2c->audio_processor->driver->command (info->i2c->audio_processor, AUDC_SET_INPUT, &input);
		info->audio.flags |= VIDEO_AUDIO_MUTE;
		kcompat_i2c_client_put (info->i2c->audio_processor);
		info->audio_processor_used--;
	}
	kcompat_i2c_client_put (info->i2c->video_decoder);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
	return;
#else
	file->private_data = NULL;
	return 0;
#endif
}

#if EVER_IMPLEMENTED
static long rivatv_video_read (struct video_device *vd, char *buf, unsigned long count, int nb)
{
	struct rivatv_info *info = video_get_drvdata (vd);
	struct i2c_client *client = info->i2c->video_decoder;
	char i2c_buf[1024];
	int c;

	if (client == NULL)
		return -ENODEV;

#if 0
	if (count > 0x1000)
		count = 0x1000;
	for (c = 0; c < count; c += 4, buf += 4) {
		u32 reg;
		reg = VID_RD32 (info->chip.PME, 0x200000 + c);
		if (copy_to_user (buf, &reg, sizeof (reg)))
			return -EFAULT;
	}
#else
	c = client->driver->command (client, READ_REGISTERS, i2c_buf);
	if (c < 0)
		return -EINVAL;

	if (count > c)
		count = c;
	
	if (copy_to_user (buf, i2c_buf, count))
		return -EFAULT;
#endif

	return count;
}


static long rivatv_video_write (struct video_device *vd, const char *buf, unsigned long count, int nb)
{
	struct rivatv_info *info = video_get_drvdata (vd);
	struct i2c_client *client = info->i2c->video_decoder;
	char i2c_buf[1024];
	int c;
	
	if (client == NULL)
		return -ENODEV;

	c = client->driver->command (client, GET_NR_OF_REGISTERS, NULL);
	if (count != c || count > 1024)
		return -EINVAL;

	if (copy_from_user (i2c_buf, buf, count))
		return -EFAULT;

	if (client->driver->command (client, WRITE_REGISTERS, i2c_buf) != c)
		return -EINVAL;

	return count;
}
#endif /* EVER_IMPLEMENTED */

/* For your convenience. */
#define COPY_TO_USER(value)                                 \
	if (copy_to_user (arg, &(value), sizeof (value)))   \
		return -EFAULT;
#define COPY_FROM_USER(value)                               \
	if (copy_from_user (&(value), arg, sizeof (value))) \
		return -EFAULT;


#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static int rivatv_video_ioctl (struct video_device *dev, unsigned int cmd, void *arg) 
{
#else
static int rivatv_video_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long a)
{
        struct video_device *dev = video_devdata (file);
	void *arg = (void *) a;
#endif
	struct rivatv_info *info = video_get_drvdata (dev);
	struct i2c_client *decoder = info->i2c->video_decoder;
	struct i2c_client *tuner = info->i2c->tuner;
	struct i2c_client *irchip = info->i2c->ir_chip;
	struct i2c_client *audio_decoder = info->i2c->audio_decoder;
	struct i2c_client *audio_processor = info->i2c->audio_processor;

	switch (cmd) {

	/* Get capabilities */
	case VIDIOCGCAP:
	{
		struct video_capability v;
#if 0
		struct video_decoder_capability d;

		down (&info->decoder_lock);
		decoder->driver->command (decoder, DECODER_GET_CAPABILITIES, &d);
		up (&info->decoder_lock);
#endif

		DPRINTK ("VIDIOCGCAP\n");
		strcpy (v.name, "rivatv video");
		v.type = VID_TYPE_CAPTURE;
		if (info->chip.arch == NV_ARCH_30 ||
		    info->chip.arch == NV_ARCH_20 || info->chip.arch == NV_ARCH_10 || 
		    info->chip.arch == NV_ARCH_04 || info->chip.arch == NV_ARCH_03)
			v.type |= VID_TYPE_OVERLAY | VID_TYPE_CHROMAKEY;
		v.channels  = info->sources;
		v.audios    = 0;
		v.minwidth  = RIVATV_MIN_VLD_WIDTH;
		v.minheight = RIVATV_MIN_VLD_HEIGHT;
		rivatv_get_capability (info, &v);

		COPY_TO_USER (v);
		break;
	}

	/* Get channel info (sources) */
	case VIDIOCGCHAN:
	{
		struct video_channel v;

		DPRINTK ("VIDIOCGCHAN\n");
		COPY_FROM_USER (v);
		if (v.channel >= info->sources)
			return -EINVAL;

		rivatv_channel[v.channel].flags &= ~VIDEO_VC_AUDIO;
		if (v.channel == RIVATV_TUNER_SOURCE) {
			strcpy (rivatv_channel[v.channel].name, tvbox ? "TV-Box" : "Television");
			if (info->audio_required != -1 && audio_processor != NULL)
				rivatv_channel[v.channel].flags |= VIDEO_VC_AUDIO;
		}
		rivatv_channel[v.channel].norm = info->format.norm;
		COPY_TO_USER (rivatv_channel[v.channel]);
		break;
	}

	/* Set channel */
	case VIDIOCSCHAN:
	{
		struct video_channel v;
		int val;

		COPY_FROM_USER (v);
		DPRINTK ("VIDIOCSCHAN: %u\n", v.channel);
		if (v.channel >= info->sources)
			return -EINVAL;

// --> Force channel here

		/* select type of tuner right here if necessary */
		if (v.channel == RIVATV_TUNER_SOURCE) {
			if (!info->tuner_selected) {
				int type = info->tuner_required;
				tuner->driver->command (tuner, TUNER_SET_TYPE, &type);
				info->tuner_selected++;
			}
			tuner->driver->command (tuner, cmd, &v);
		}

		/* set input source of audio chip */
		if (info->audio_required != -1) {
			int input = (v.channel == RIVATV_TUNER_SOURCE) ? AUDIO_TUNER : AUDIO_EXTERN;
			/* the tda8425 processor has 2 input channels, current driver uses 1 only */
			if (audio_processor != NULL)
				audio_processor->driver->command (audio_processor, AUDC_SET_INPUT, &input);
		}

		/* save selected video channel */
		info->channel = v.channel;

		down (&info->decoder_lock);

		/* set input mode of video decoder */
		val = info->card.mode[v.channel];
		decoder->driver->command (decoder, DECODER_SET_INPUT, &val);

		/* set input mode of tuner chip if necessary */
		if (info->ir_required != -1) {
			switch (v.channel) {
			case RIVATV_COMPOSITE_SOURCE:
				rivatv_enable_tuner (irchip, RIVATV_TUNER_COMPOSITE);
				break;
			case RIVATV_SVIDEO_SOURCE:
				rivatv_enable_tuner (irchip, RIVATV_TUNER_SVIDEO);
				break;
			case RIVATV_TUNER_SOURCE:
				rivatv_enable_tuner (irchip, RIVATV_TUNER_TV);
				break;
			}
		}

		/* set video norm of video decoder */
		val = v.norm;
		decoder->driver->command (decoder, DECODER_SET_NORM, &val);

		/* set output format of video decoder */
		val = info->source->format;
		decoder->driver->command (decoder, DECODER_ENABLE_OUTPUT, &val);

		/* obtain current video standard */
		val = 0;
		decoder->driver->command (decoder, DECODER_GET_STATUS, &val);
		if (val & DECODER_STATUS_PAL)
			info->format.norm = VIDEO_MODE_PAL;
		else if (val & DECODER_STATUS_NTSC)
			info->format.norm = VIDEO_MODE_NTSC;
		else if (val & DECODER_STATUS_SECAM)
			info->format.norm = VIDEO_MODE_SECAM;

		up (&info->decoder_lock);
		break;
	}

	/* Get tuner abilities */
	case VIDIOCGTUNER:
	{
		struct video_tuner v;
		int val = 0;

		DPRINTK ("VIDIOCGTUNER\n");
		COPY_FROM_USER (v);
		strcpy (v.name, rivatv_channel[RIVATV_TUNER_SOURCE].name);
		v.rangelow = 44 * 16;
		v.rangehigh = 958 * 16;
		v.flags = VIDEO_TUNER_PAL | VIDEO_TUNER_NTSC | VIDEO_TUNER_SECAM;
		v.mode = info->format.norm;

		down (&info->decoder_lock);
		decoder->driver->command (decoder, DECODER_GET_STATUS, &val);
		up (&info->decoder_lock);
		v.signal = (val & DECODER_STATUS_GOOD) ? 0xffff : 0;

		COPY_TO_USER (v);
		break;
	}

	/* Tune the tuner for the current channel */
	case VIDIOCSTUNER:
	{
		struct video_tuner v;

		COPY_FROM_USER (v);
		DPRINTK ("VIDIOCSTUNER: [%u:%s] %lu-%lu, 0x%08X 0x%04X %u\n",
			 v.tuner, v.name, v.rangelow, v.rangehigh, v.flags, v.mode, v.signal);

		if (v.tuner != 0)
			return -EINVAL;
				
		if (v.mode != VIDEO_MODE_PAL   && v.mode != VIDEO_MODE_NTSC && 
		    v.mode != VIDEO_MODE_SECAM && v.mode != VIDEO_MODE_AUTO)
			return -EOPNOTSUPP;

		if (info->format.norm != v.mode) {
			int val = v.mode;
			down (&info->decoder_lock);
			decoder->driver->command (decoder, DECODER_SET_NORM, &val);
			val = 0;
			decoder->driver->command (decoder, DECODER_GET_STATUS, &val);
			if (val & DECODER_STATUS_PAL)
				info->format.norm = VIDEO_MODE_PAL;
			else if (val & DECODER_STATUS_NTSC)
				info->format.norm = VIDEO_MODE_NTSC;
			else if (val & DECODER_STATUS_SECAM)
				info->format.norm = VIDEO_MODE_SECAM;
			up (&info->decoder_lock);
		}
		break;
	}

	/* Get picture properties */
	case VIDIOCGPICT:
	{
		struct video_picture *v = &info->picture;

		DPRINTK ("VIDIOCGPICT\n");
		COPY_TO_USER (*v);
		break;
	}

	/* Set picture properties */
	case VIDIOCSPICT:
	{
		struct video_picture v;

		COPY_FROM_USER (v);
		DPRINTK ("VIDIOCSPICT: BRI=%u HUE=%u COL=%u CON=%u WHI=%u DEP=%u PAL=%u\n",
			 v.brightness, v.hue, v.colour, v.contrast, v.whiteness, v.depth, v.palette);

#if V4L_APPS_CARE_CORRECTLY
		/* most V4L application don't really care (or even worse: different 
		   appplications care in variants!) */
		if (!rivatv_palette_supported (info, v.palette))
			return -EINVAL;
#endif

		info->picture = v;
		down (&info->decoder_lock);
		decoder->driver->command (decoder, DECODER_SET_PICTURE, &v);
		up (&info->decoder_lock);
		break;
	}

	/* Start, end capture */
	case VIDIOCCAPTURE:
	{
		int enable;
		struct video_buffer *buffer = &info->overlay.buffer;

		COPY_FROM_USER (enable);
		DPRINTK ("VIDIOCCAPTURE: %d\n", enable);

		if (buffer->base == 0)
			return -EINVAL;
                if (buffer->width == 0 || buffer->height == 0 || buffer->bytesperline == 0 || buffer->depth == 0)
                        return -EINVAL;

		if (enable) {
			rivatv_video_stop (info);
			rivatv_set_overlay_format (info);
			rivatv_video_start (info);
			rivatv_overlay_start (info);
			info->overlay.enabled = 1;
		}
		else {
			rivatv_overlay_stop (info);
			rivatv_video_stop (info);
			info->overlay.enabled = 0;
		}
		break;
	}

	/* Get the video overlay window */
	case VIDIOCGWIN:
	{
		struct video_window *v = &info->overlay.window;

		DPRINTK ("VIDIOCGWIN\n");
		COPY_TO_USER (*v);
		break;
	}

	/* Set the video overlay window - passes clip list for hardware smarts, chromakey etc. */
	case VIDIOCSWIN:
	{
		struct video_window window;
		struct video_window *win = &info->overlay.window;
		int restart = 0;

		COPY_FROM_USER (window);
		DPRINTK ("VIDIOCSWIN: %dx%d @ (%d,%d) (key: 0x%08X)\n",
			 window.width, window.height, window.x, window.y, window.chromakey);

		/* check for a change in overlay arguments */
		if ((window.width != win->width || window.height != win->height ||
		     window.x	  != win->x	|| window.y	 != win->y ||
		     window.chromakey != win->chromakey) && info->overlay.enable ) {
			restart = 1;
		}

		/* save given overlay window arguments */
		memcpy (win, &window, sizeof (window));
		win->flags = 0;
		win->clipcount = 0;
		win->clips = NULL;

		if (restart && (info->chip.arch == NV_ARCH_04 || info->chip.arch == NV_ARCH_03 ||
				info->chip.arch == NV_ARCH_10 || info->chip.arch == NV_ARCH_20 ||
				info->chip.arch == NV_ARCH_30)) {
			rivatv_overlay_stop (info);
			rivatv_video_stop (info);
			rivatv_set_overlay_format (info);
			rivatv_video_start (info);
			rivatv_overlay_start (info);
		}
		break;
	}

	/* Get frame buffer */
	case VIDIOCGFBUF:
	{
		struct video_buffer *buffer = &info->overlay.buffer;

		buffer->base = (void *) info->base1_region;

		DPRINTK ("VIDIOCGFBUF: %dx%d (%d bits, %d bpl) @ 0x%08lX\n",
			 buffer->width, buffer->height, buffer->depth, 
			 buffer->bytesperline, (unsigned long) buffer->base);

		COPY_TO_USER (*buffer);
		break;
	}

	/* Set frame buffer - root only */
	case VIDIOCSFBUF:
	{
		struct video_buffer *buffer = &info->overlay.buffer;

		if (!capable (CAP_SYS_ADMIN))
			return -EPERM;

		COPY_FROM_USER (*buffer);
		DPRINTK ("VIDIOCSFBUF: %dx%d (%d bits, %d bpl) @ 0x%08lX\n",
			 buffer->width, buffer->height, buffer->depth, 
			 buffer->bytesperline, (unsigned long) buffer->base);
		break;
	}
	
	/* Video key event, cuts capture on all DMA windows with this key (0xFFFFFFFF == all) */
	case VIDIOCKEY:
	{
		DPRINTK ("VIDIOCKEY\n");
		return -ENOIOCTLCMD;
	}

	/* Get tuner */
	case VIDIOCGFREQ:
	{
		unsigned long v = info->tuner_frequency;

		DPRINTK ("VIDIOCGFREQ\n");
		COPY_TO_USER (v);
		break;
	}

	/* Set tuner */
	case VIDIOCSFREQ:
	{
		unsigned long v;

		COPY_FROM_USER (v);
		DPRINTK ("VIDIOCSFREQ: %ld\n", v);
		info->tuner_frequency = v;

		if (info->tuner_required == -1 || tuner == NULL)
			return -ENODEV;
		if (info->tuner_frequency != 0)
			tuner->driver->command (tuner, cmd, &v);
		break;
	}

	/* Get audio info */
	case VIDIOCGAUDIO:
	{
		struct video_audio v;

		DPRINTK ("VIDIOCGAUDIO\n");
		v = info->audio;
		v.flags |= VIDEO_AUDIO_MUTABLE;
		strcpy (v.name, "TV");

		if (info->audio_required == -1 || audio_processor == NULL)
			return -ENODEV;
		audio_processor->driver->command (audio_processor, cmd, &v);
		if (audio_decoder != NULL)
			audio_decoder->driver->command (audio_decoder, cmd, &v);

		v.mode = info->audio.mode;
		COPY_TO_USER (v);
		break;
	}

	/* Audio source, mute etc */
	case VIDIOCSAUDIO:
	{
		struct video_audio v;
		int input;

		COPY_FROM_USER (v);
		DPRINTK ("VIDIOCSAUDIO: [%u:%s] VOL=%u BAS=%u TRE=%u BAL=%u, 0x%08X 0x%04X %u\n",
			 v.audio, v.name, v.volume, v.bass, v.treble, v.balance, v.flags, v.mode, v.step);

		if (info->audio_required == -1 || audio_processor == NULL)
			return -ENODEV;

		input = (info->channel == RIVATV_TUNER_SOURCE) ? AUDIO_TUNER : AUDIO_EXTERN;
		if (v.flags & VIDEO_AUDIO_MUTE)
			input |= AUDIO_MUTE;
		audio_processor->driver->command (audio_processor, AUDC_SET_INPUT, &input);
		audio_processor->driver->command (audio_processor, cmd, &v);
		if (audio_decoder != NULL)
			audio_decoder->driver->command (audio_decoder, cmd, &v);

		info->audio = v;
		break;
	}

	/* Sync with mmap grabbing */
	case VIDIOCSYNC:
	{
		int frame, ret;

		COPY_FROM_USER (frame);
		DPRINTK2 ("VIDIOCSYNC: %d\n", frame);
		ret = rivatv_sync (info, frame);
		return ret;
	}

	/* Grab frames */
	case VIDIOCMCAPTURE:
	{
		struct video_mmap v;
		int ret, restart, format;

		COPY_FROM_USER (v);
		DPRINTK2 ("VIDIOCMCAPTURE: %d\n", v.frame);

		restart = rivatv_check_format (info, &v);
		if (restart < 0)
			return -EINVAL;
		
		format = rivatv_check_source (info, &v);
		if (format) {
			rivatv_decoder_output (info, format);
			rivatv_configureDMA (info);
		}

		ret = rivatv_add_queue (info, &v);
		if (ret)
			return ret;

		/* start video here */
		if (restart)
			rivatv_video_stop (info);
		rivatv_video_start (info);
		break;
	}

	/* Memory map buffer info */
	case VIDIOCGMBUF:
	{
		struct video_mbuf v;
		int i;

		DPRINTK ("VIDIOCGMBUF\n");
		memset (&v, 0 , sizeof (v));
		v.size = RIVATV_CAPTURE_BUFSIZE * capbuffers;
		v.frames = capbuffers;
		for (i = 0; i < capbuffers; i++)
			v.offsets[i] = i * RIVATV_CAPTURE_BUFSIZE;
		COPY_TO_USER (v);
		break;
	}

	/* Get attached units */
	case VIDIOCGUNIT:
	{
		struct video_unit v;

		DPRINTK ("VIDIOCGUNIT\n");
		v.video = info->video->minor;
		v.vbi = info->vbi->minor;
		v.radio = VIDEO_NO_UNIT;
		v.audio = VIDEO_NO_UNIT;
		v.teletext = VIDEO_NO_UNIT;
		COPY_TO_USER (v);
		break;
	}

	case VIDIOCGCAPTURE:	/* Get subcapture */
	case VIDIOCSCAPTURE:	/* Set subcapture */
	case VIDIOCSPLAYMODE:	/* Set output video mode/feature */
	case VIDIOCSWRITEMODE:	/* Set write mode */
	case VIDIOCGPLAYINFO:	/* Get current playback info from hardware */
	case VIDIOCSMICROCODE:	/* Load microcode into hardware */
	case VIDIOCGVBIFMT:	/* Get VBI information */
	case VIDIOCSVBIFMT:	/* Set VBI information */
	default:
		DPRINTK ("V4L: Requested IOCTL (0x%08X) not implemented\n", cmd);
		return -ENOIOCTLCMD;
	}
	return 0;
}

/* if it is a RH ver and the version is on or above 2.4.20 */
/* use the 2.5.0 inputs and outputs */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0) && !defined(RIVATV_ISREDHAT)
static int rivatv_video_mmap (struct video_device *dev, const char *adr, unsigned long size)
{
#elif defined(RIVATV_ISREDHAT)
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 20)
static int rivatv_video_mmap (struct video_device *dev, const char *adr, unsigned long size)
{
#else
static int rivatv_video_mmap (struct vm_area_struct *vma, struct video_device *dev, const char *adr, unsigned long size)
{
#endif
#else
static int rivatv_video_mmap (struct file *file, struct vm_area_struct *vma)
{
        struct video_device *dev = video_devdata (file);
#endif
	struct rivatv_info *info = video_get_drvdata (dev);
	int ret;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0) && !defined(RIVATV_ISREDHAT)
	ret = rivatv_mmap (info, NULL, (ulong) adr, size);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 20) && defined(RIVATV_ISREDHAT)
	ret = rivatv_mmap (info, NULL, (ulong) adr, size);
#else
	ret = rivatv_mmap (info, vma, vma->vm_start, vma->vm_end - vma->vm_start);
#endif
	return ret;
}

/*----------------------------------------------------------------------
 *
 * Video4Linux VBI device driver part
 *
 *----------------------------------------------------------------------*/

/* Standard character-device-driver functions */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static int rivatv_vbi_open (struct video_device *dev, int nb)
{
#else
static int rivatv_vbi_open  (struct inode *inode, struct file *file)
{
        struct video_device *dev = video_devdata (file);
#endif
	struct rivatv_info *info = video_get_drvdata (dev);

	if (info->i2c == NULL || info->i2c->video_decoder == NULL)
		return -ENODEV;

	kcompat_i2c_client_get (info->i2c->video_decoder);

	return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static void rivatv_vbi_close (struct video_device *dev)
{
#else
static int rivatv_vbi_close  (struct inode *inode, struct file *file)
{
        struct video_device *dev = video_devdata (file);
#endif
	struct rivatv_info *info = video_get_drvdata (dev);

	kcompat_i2c_client_put (info->i2c->video_decoder);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
	return;
#else
	return 0;
#endif
}

#if EVER_IMPLEMENTED
static long rivatv_vbi_read (struct video_device *vd, char *buf, unsigned long count, int nb)
{
	struct rivatv_info *info = video_get_drvdata (vd);
	struct i2c_client *client = info->i2c->video_decoder;
	char i2c_buf[256];
	int c;

	if (client == NULL)
		return -ENODEV;

	c = client->driver->command (client, READ_REGISTERS, i2c_buf);
	if (c < 0)
		return -EINVAL;

	if (count > c)
		count = c;
	
	if (copy_to_user (buf, i2c_buf, count))
		return -EFAULT;

	return count;
}

static long rivatv_vbi_write (struct video_device *vd, const char *buf, unsigned long count, int nb)
{
	struct rivatv_info *info = video_get_drvdata (vd);
	struct i2c_client *client = info->i2c->video_decoder;
	char i2c_buf[256];
	int c;
	
	if (client == NULL)
		return -ENODEV;

	c = client->driver->command (client, GET_NR_OF_REGISTERS, NULL);
	if (count != c || count > 256)
		return -EINVAL;

	if (copy_from_user (i2c_buf, buf, count))
		return -EFAULT;

	if (client->driver->command (client, WRITE_REGISTERS, i2c_buf) != c)
		return -EINVAL;

	return count;
}
#endif /* EVER_IMPLEMENTED */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static int rivatv_vbi_ioctl (struct video_device *dev, unsigned int cmd, void *arg) 
{
#else
static int rivatv_vbi_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long a)
{
	void *arg = (void *) a;
#endif

	switch (cmd) {
	case VIDIOCGCAP:
	{
		struct video_capability v;
		v.type = VID_TYPE_TELETEXT;
		v.channels = 2;
		v.audios = 0;
		v.maxwidth = v.minwidth = 0;
		v.maxheight = v.minheight = 0;
		strcpy (v.name, "rivatv vbi");
		COPY_TO_USER (v);
		break;
	}
	case VIDIOCGVBIFMT:
	{
		struct vbi_format f;
		f.sampling_rate = 14750000UL; /* these are foobars */
		f.sample_format = VIDEO_PALETTE_RAW;
		f.flags = VBI_INTERLACED;
		COPY_TO_USER (f);
		break;
	}
	case VIDIOCSVBIFMT:
	{
		struct vbi_format f;
		COPY_FROM_USER (f);
		DPRINTK ("VIDIOCSVBIINFO (%d,%d,%d,%d,%d,%d,%d,%x)\n", 
			 f.sampling_rate, f.samples_per_line, f.sample_format,
			 f.start[0], f.start[1], f.count[0], f.count[1],
			 f.flags);
		break;
	}
	default:
		DPRINTK ("VBI: requested IOCTL (0x%08X) not implemented\n", cmd);
		return -ENOIOCTLCMD;
	}
	return 0;
}

/*---------------------------------------------------------------------- 
 *
 * Modularization
 *
 *----------------------------------------------------------------------*/

MODULE_AUTHOR	    ("Ferenc Bakonyi <fero@drama.obuda.kando.hu>");
MODULE_DESCRIPTION  ("Video4Linux driver for NVIDIA cards");
MODULE_LICENSE	    ("GPL");

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (capbuffers, int, 0644);
#else
MODULE_PARM	 (capbuffers, "1-" __stringify (RIVATV_MAX_CAPTURE_BUFFERS) "i");
#endif
MODULE_PARM_DESC (capbuffers, "Number of capture buffers (default = 4).");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (autoload, bool, 0644);
#else
MODULE_PARM	 (autoload, "i");
#endif
MODULE_PARM_DESC (autoload, "Turn decoder chip driver autoloading off (= 0) (On by default).");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (card, int, 0644);
#else
MODULE_PARM      (card, "1-" __stringify (RIVATV_MAX_CARDS) "i");
#endif
MODULE_PARM_DESC (card, "Specify card model, see CARDLIST file for a list.");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (debug, int, 0644);
#else
MODULE_PARM	 (debug, "i");
#endif
MODULE_PARM_DESC (debug, "Debug level: 0 = silent, 1 = verbose, 2 = very verbose (default = 1).");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (tvbox, int, 0644);
#else
MODULE_PARM	 (tvbox, "i");
#endif
MODULE_PARM_DESC (tvbox, "Indicate usage of a TV-Box (0 = off, see README file for a list) (Off by default).");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (dma, bool, 0644);
#else
MODULE_PARM	 (dma, "i");
#endif
MODULE_PARM_DESC (dma, "Indicate usage of DMA transfers (1 = on) (Off by default).");
#ifndef RIVATV_DISABLE_AGP
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (agp, bool, 0644);
#else
MODULE_PARM	 (agp, "i");
#endif
MODULE_PARM_DESC (agp, "Indicate usage of AGP (1 = on) (Off by default).");
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (mmx, bool, 0644);
#else
MODULE_PARM	 (mmx, "i");
#endif
MODULE_PARM_DESC (mmx, "Indicate usage of MMX code if possible (1 = on) (On by default).");
#ifndef RIVATV_DISABLE_CONVERSION
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (conversion, bool, 0644);
#else
MODULE_PARM	 (conversion, "i");
#endif
MODULE_PARM_DESC (conversion, "Enable colour conversion code (0 = off) (On by default).");
#endif
#ifdef CONFIG_MTRR
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param     (mtrr, bool, 0644);
#else
MODULE_PARM	 (mtrr, "i");
#endif
MODULE_PARM_DESC (mtrr, "Turn off (= 0) MTRR usage (default = 1).");
#endif

/* Definition of the video grabber device functionality. */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)

struct __initdata video_device rivatv_video = { 
	name:		"rivatv video",
	type:		VID_TYPE_CAPTURE,
	hardware:	VID_HARDWARE_RIVA128,
	open:		rivatv_video_open,
	close:		rivatv_video_close,
	read:		NULL, // rivatv_video_read,
	write:		NULL, // rivatv_video_write,
	ioctl:		rivatv_video_ioctl,
	mmap:		rivatv_video_mmap,
};

#else

static struct file_operations rivatv_video_fops = {
        owner:          THIS_MODULE,
        open:           rivatv_video_open,
        release:        rivatv_video_close,
	read:		NULL, // rivatv_video_read,
	write:		NULL, // rivatv_video_write,
        mmap:           rivatv_video_mmap,
        ioctl:          rivatv_video_ioctl,
        llseek:         no_llseek,
};

static struct video_device rivatv_video = {
        owner:          THIS_MODULE,
        name:           "rivatv video",
        type:           VID_TYPE_CAPTURE,
        hardware:       VID_HARDWARE_RIVA128,
        fops:           &rivatv_video_fops,
};

#endif

/* Definition of the video teletext device. */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)

struct __initdata video_device rivatv_vbi = { 
	name:		"rivatv vbi",
	type:		VID_TYPE_TELETEXT,
	hardware:	VID_HARDWARE_RIVA128,
	open:		rivatv_vbi_open,
	close:		rivatv_vbi_close,
	read:		NULL, // rivatv_vbi_read,
	write:		NULL, // rivatv_vbi_write,
	ioctl:		rivatv_vbi_ioctl,
};

#else

static struct file_operations rivatv_vbi_fops = {
        owner:          THIS_MODULE,
        open:           rivatv_vbi_open,
        release:        rivatv_vbi_close,
	read:		NULL, // rivatv_vbi_read,
	write:		NULL, // rivatv_vbi_write,
        ioctl:          rivatv_vbi_ioctl,
        llseek:         no_llseek,
};

static struct video_device rivatv_vbi = {
        owner:          THIS_MODULE,
        name:           "rivatv vbi",
        type:           VID_TYPE_TELETEXT,
        hardware:       VID_HARDWARE_RIVA128,
        fops:           &rivatv_vbi_fops,
};

#endif

/*---------------------------------------------------------------------- 
 *
 * Driver startup code
 *
 *----------------------------------------------------------------------*/

struct __initdata rivatv_initdata rivatv_driver_init = {
	video:	&rivatv_video,
	vbi:	&rivatv_vbi
};

int __init rivatv_init (void)
{
	PRINTK_INFO ("Video4Linux driver for NVIDIA cards\n");
#if 1
	PRINTK_INFO ("Version 0.8.6\n");
#endif

	/* detect MMX processor extension */
	rivatv_detect_mmx ();

	/* reset device count */
	rivatv_devices = 0;

	return pci_module_init (&rivatv_driver);
}

void __exit rivatv_cleanup (void) 
{
	pci_unregister_driver (&rivatv_driver);
}

module_init (rivatv_init);
module_exit (rivatv_cleanup);
