#ifndef __RIVATV_H
#define __RIVATV_H

#include <linux/config.h>
#include <linux/types.h>
#include <linux/videodev.h>
#include <linux/video_decoder.h>

#include "rivatv-kcompat.h"
#include "dma-riva.h"

/* Debugging definitions */

#define RIVATV_PFX "rivatv: "

#define PRINTK(format, args...)			       \
	printk (RIVATV_PFX format, ## args)
#define PRINTK_ERR(format, args...)		       \
	printk (RIVATV_PFX format, ## args)
#define PRINTK_INFO(format, args...)		       \
	printk (RIVATV_PFX format, ## args)
#define DPRINTK(format, args...) if (debug >= 1)       \
	printk (RIVATV_PFX format, ## args)
#define DPRINTK2(format, args...) if (debug >= 2)      \
	printk (RIVATV_PFX format, ## args)

/* Quick and dirty hack for extra debugging commands in I2C drivers */

#define READ_REGISTERS		0xBF01
#define WRITE_REGISTERS		0xBF02
#define GET_NR_OF_REGISTERS	0xBF03
#define READ_REGISTER		0xBF04
#define WRITE_REGISTER		0xBF05

/* Video channel selection for tuner via IR chip commands. */
#define RIVATV_TUNER_COMPOSITE 0xE0
#define RIVATV_TUNER_SVIDEO    0xA0
#define RIVATV_TUNER_TV        0x60

/*
 * PCI-Memory IO access macros.
 */
#define VID_WR08(p,i,val)  (((uint8_t *)(p))[(i)]=(val))
#define VID_RD08(p,i)	   (((uint8_t *)(p))[(i)])

#define VID_WR32(p,i,val)  (((uint32_t *)(p))[(i)/4]=(val))
#define VID_RD32(p,i)	   (((uint32_t *)(p))[(i)/4])

#ifndef USE_RMW_CYCLES
/*
 * Can be used to inhibit READ-MODIFY-WRITE cycles. On by default.
 */

#define MEM_BARRIER() __asm__ __volatile__ ("" : : : "memory")

#undef	VID_WR08
#define VID_WR08(p,i,val) ({ MEM_BARRIER(); ((uint8_t *)(p))[(i)]=(val); })
#undef	VID_RD08
#define VID_RD08(p,i)     ({ MEM_BARRIER(); ((uint8_t *)(p))[(i)]; })

#undef	VID_WR32
#define VID_WR32(p,i,val) ({ MEM_BARRIER(); ((uint32_t *)(p))[(i)/4]=(val); })
#undef	VID_RD32
#define VID_RD32(p,i)     ({ MEM_BARRIER(); ((uint32_t *)(p))[(i)/4]; })
#endif /* USE_RMW_CYCLES */

#define VID_AND32(p,i,val) VID_WR32(p,i,VID_RD32(p,i)&(val))
#define VID_OR32(p,i,val)  VID_WR32(p,i,VID_RD32(p,i)|(val))
#define VID_XOR32(p,i,val) VID_WR32(p,i,VID_RD32(p,i)^(val))


/*
 * Define supported architectures.
 */
#define NV_ARCH_03  0x03
#define NV_ARCH_04  0x04
#define NV_ARCH_10  0x10
#define NV_ARCH_20  0x20
#define NV_ARCH_30  0x30

/*
 * Additional architectures, used for i2c detection.
 */
#define NV_ARCH_05  0x05	/* Riva TNT2	   */
#define NV_ARCH_11  0x11	/* GeForce 2 MX    */
#define NV_ARCH_17  0x17	/* GeForce 4 MX    */
#define NV_ARCH_18  0x18	/* GeForce 4 MX    */
#define NV_ARCH_15  0x15	/* GeForce 2	   */
#define NV_ARCH_25  0x25	/* GeForce 4 Ti    */
#define NV_ARCH_28  0x28	/* GeForce 4 Ti    */
#define NV_ARCH_31  0x31        /* GeForce FX 5600 */
#define NV_ARCH_32  0x32        /* GeForce FX 5200 */
#define NV_ARCH_33  0x33        /* GeForce FX 5900 */
#define NV_ARCH_34  0x34        /* GeForce FX 5700 */

#define RIVATV_MIN_VLD_WIDTH  64  /* minimal valid width     */
#define RIVATV_MIN_VLD_HEIGHT 32  /* minimal valid height    */
#define RIVATV_MAX_VLD_WIDTH  704 /* maximum valid width     */
#define RIVATV_MAX_VLD_HEIGHT 576 /* maximum valid height    */
#define RIVATV_MAX_ORG_WIDTH  720 /* maximum original width  */
#define RIVATV_MAX_ORG_HEIGHT 576 /* maximum original height */

/* threshold for (hot)plugged devices */
#define RIVATV_MAX_DEVICES 99	/* keep in sync with space for "rivatv%d" */

/* Maximum capture queue buffers. */
#define RIVATV_MAX_CAPTURE_BUFFERS 32
/* Default capture queue buffers. */
#define RIVATV_CAPTURE_BUFFERS 4
/* Default capture buffer size. */
#define RIVATV_CAPTURE_BUFSIZE \
  PAGE_ALIGN (RIVATV_MAX_ORG_WIDTH * RIVATV_MAX_ORG_HEIGHT * 4)
/* Default raw (UYVY) capture buffer size. */
#define RIVATV_RAW_CAPTURE_BUFSIZE \
  PAGE_ALIGN (RIVATV_MAX_ORG_WIDTH * RIVATV_MAX_ORG_HEIGHT * 2)

/* Size of the nicename char-array. DO NOT WRITE BEYOND END OF BUFFER. */
#define RIVATV_NICENAME_BUFSIZE 51

/* The native palette format for all nVidia chips (though video decoders 
   may be able to output other formats). */
#define RIVATV_PALETTE_NATIVE VIDEO_PALETTE_UYVY

#define VIDEO_INTERLACED     8
#define VIDEO_FLAG_1	     1

/* Structure defining a video format. */

struct rivatv_format {
	int norm;      /* one of the VIDEO_MODE_ defines    */
	int width;     /* width in pixels		    */
	int height;    /* height in pixels		    */
	int size;      /* size of requested capture buffer  */
	int format;    /* one of the VIDEO_PALETTE_ defines */
	int overlay;   /* overlay indicator flag	    */
};

/* Structure defining a video port. */

struct rivatv_port {
	int max_width;	/* maximum width in pixels		   */
	int max_height; /* maximum height per field in pixels	   */
	int org_width;	/* decoder pixels in framebuffer	   */
	int org_height; /* decoder lines in framebuffer		   */
	int vld_width;	/* number of valid pixels in a line	   */
	int vld_height; /* valid lines				   */
	int width;	/* size of a video line in pixels	   */
	int height;	/* number of video lines		   */
	int flags;	/* VIDEO_INTERLACED, VIDEO_FLAG_1	   */
	int size;	/* size of maximum capture region in bytes */
	int offset;	/* buffer offset			   */
};

/* Structure defining a video source. */

struct rivatv_source {
	int id;		/* decoder chip id				      */
	int mode[3];	/* input modes: 0 - Composite, 1 - S-Video, 2 - Tuner */
	int formats[8]; /* possible video formats (VIDEO_PALETTE_)	      */
	int format;	/* current video format				      */
};

/* Structure defining a single capture queue buffer. */

struct rivatv_cap_buffer {
	int status;  /* status of a capture queue buffer */

#define RIVATV_UNUSED	1 /* not used			  */
#define RIVATV_GRABBING 2 /* currently under construction */
#define RIVATV_ERROR	4 /* error occurred		  */
#define RIVATV_READY	8 /* buffer is ready		  */

	int width;   /* size of a single line	  */
	int height;  /* number of lines to grab	  */
	int format;  /* VIDEO_PALETTE_XXX define  */
	void *addr;  /* virtual address in kernel */
	ulong size;  /* size of this buffer	  */
};

/* Structure defining the capture queue. */

struct rivatv_cap_queue {
	wait_queue_head_t wait;     /* the wait queue	      */
	struct tasklet_struct task; /* task queue	      */
	int index[RIVATV_MAX_CAPTURE_BUFFERS]; /* capture buffer indices */
	int head;		    /* queue start	      */
	int tail;		    /* end of queue	      */
	int fill;		    /* number of used entries */
};

/* List of v4l devices supported by this driver. */

struct rivatv_initdata {
	struct video_device *video;
	struct video_device *vbi;
};

/* Chip architecture specific info. */

struct rivatv_chip {

	volatile uint32_t *PMC;	   /* general control			*/
	volatile uint32_t *PME;	   /* multimedia port			*/
	volatile uint32_t *PFB;	   /* framebuffer control		*/
	volatile uint32_t *PVIDEO; /* overlay control			*/
	volatile uint8_t *PCIO;	   /* SVGA (CRTC, ATTR) registers	*/
	volatile uint8_t *PVIO;	   /* SVGA (MISC, GRAPH, SEQ) registers */
	volatile uint32_t *PRAMIN; /* instance memory			*/
	volatile uint32_t *PRAMHT; /* hash table			*/
	volatile uint32_t *PRAMFC; /* fifo context table		*/
	volatile uint32_t *PRAMRO; /* fifo runout table			*/
	volatile uint32_t *PFIFO;  /* fifo control region		*/
	volatile uint32_t *FIFO;   /* fifo channels (USER)		*/
	volatile uint32_t *PGRAPH; /* graphics engine                   */

	ulong fbsize;		   /* framebuffer size		   */
	int arch;		   /* compatible NV_ARCH_XX define */
	int realarch;		   /* real architecture		   */
	void (* lock) (struct rivatv_chip *, int);
};

/* Card identfication info. */

#define RIVATV_MAX_CARDS 142 /* Maximum number of supported cards, keep in sync with CARDLIST file. */

struct rivatv_card {
	unsigned short vendor;		    /* PCI vendor id					  */
	unsigned short device;		    /* PCI device id					  */
	unsigned short subsystem_vendor;    /* PCI subsystem vendor id				  */
	unsigned short subsystem_device;    /* PCI subsystem device id				  */
	char name[RIVATV_NICENAME_BUFSIZE]; /* complete name					  */
	int id;				    /* decoder chip id					  */
	char decoder[16];		    /* decoder chip module name				  */
	int mode[3];			    /* input modes: 0 - Composite, 1 - S-Video, 2 - Tuner */
	int tuner_required;		    /* tuner required					  */
	int audio_required;		    /* audio required					  */
	int identified;                     /* has card been identified correctly ?               */
};

struct rivatv_vendor {
	unsigned short subsystem_vendor;    /* PCI subsystem vendor id				  */
	char name[16];			    /* vendor name (company)				  */
};

/* Overlay definitions. */

struct rivatv_overlay {
	struct video_window window; /* X screen definitions	  */
	struct video_buffer buffer; /* overlay window definitions */
	int enable;		    /* overlay enable flag	  */
	int enabled;		    /* overall overlay flag	  */
	int virtualX;		    /* virtual width of screen	  */
	int virtualY;		    /* virtual height of screen	  */
	int physicalX;		    /* physical width of screen	  */
	int physicalY;		    /* physical height of screen  */
	int bytesPerLine;	    /* byte pitch per line	  */
	int bitsPerPixel;	    /* physical bits per pixel	  */
	u32 colorkey;		    /* saved colour key		  */
};

/* Interrupt count statistics */

struct rivatv_stats {
	unsigned long interrupts;         /* total number of interrupts */
	unsigned long decoder_interrupts; /* number of decoder interrupts */
	unsigned long overlay_interrupts; /* number of overlay interrupts */
	unsigned long dma_interrupts;     /* number of DMA interrupts */
};

/* Interrupt scheduler state (including statistics) */

#define RIVATV_SCHED_OVERLAY    1
#define RIVATV_SCHED_DMA        2
#define RIVATV_SCHED_DMA_ZC     4
#define RIVATV_SCHED_INTERLACED 8

struct rivatv_sched {
	spinlock_t sched_lock;
	unsigned long flags;
	u32 state_buf_notify;
	u32 state_buf_usage;
	int dma_zc_count;		/* Number of ready planes. */
	int dma_zc_planes;		/* Total number of planes. */
	unsigned long last_jiffies;
	struct rivatv_stats stats;
};

/* Overall video device structure. */

struct rivatv_info {
#ifdef CONFIG_MTRR
	int MTRR;			 /* flag if MTRR is enabled		       */
	int MTRR_range;			 /* MTRR index				       */
#endif /* CONFIG_MTRR */
	int transfer_active;		 /* set while picture transfer		       */
	struct rivatv_format format;	 /*					       */
	ulong vbi_region_size;		 /* size of teletext region		       */
	struct rivatv_source *source;	 /* decoder chip video source		       */
	int device_busy;		 /* is video capture device used	       */
	char *capture_buffer;		 /* aligned capture buffer		       */
	ulong capture_buffer_size;	 /* size of each capture buffer		       */
	int capture_buffers;		 /* number of capture buffers		       */
	ulong buffer_size;		 /* size of the raw capture buffer	       */
	char *data;			 /* capture data buffer			       */
	struct rivatv_cap_buffer capbuf[RIVATV_MAX_CAPTURE_BUFFERS];
	struct rivatv_cap_queue capture_queue;
	struct video_picture picture;	 /* video picture values		       */
	int capture_started;		 /* is video capture running ?		       */
	struct rivatv_card card;	 /* card info				       */
	struct rivatv_chip chip;	 /* NV architecture structure		       */
	struct i2c_riva_info *i2c;	 /* interface to I2C clients		       */
	unsigned int nr;		 /* rivatv device counter		       */
	struct video_device *video;	 /* V4L (capture) driver		       */
	struct video_device *vbi;	 /* VBI (teletext) driver		       */
	struct pci_dev *pci;		 /* pointer to board's pci driver	       */
	const char *driver;		 /* textual description of the PCI card	       */
	char nicecardname[RIVATV_NICENAME_BUFSIZE];
					 /* proper name of the card, if detected       */
	unsigned long base0_region_size; /* length of control region in bytes	       */
	unsigned long base1_region_size; /* length of fb region in bytes	       */
	unsigned long base0_region;	 /* physical address of control region	       */
	unsigned long base1_region;	 /* physical address of fb region	       */
	caddr_t video_base;		 /* virtual address of control region	       */
	caddr_t control_base;		 /* virtual address of fb region	       */
	struct rivatv_port port;	 /* video port settings (in framebuffer)       */
	unsigned long picture_base;	 /* direct pointer to video picture	       */
	unsigned long picture_offset;	 /* offset of video picture in frame buffer    */
	spinlock_t video_lock;		 /* video lock				       */
	spinlock_t queue_lock;		 /* capture queue lock			       */
	struct semaphore decoder_lock;	 /* video decoder lock			       */
	struct rivatv_overlay overlay;	 /* overlay settings			       */
	int tuner_required;		 /* flag indicating if tuner chip requested    */
	int tuner_selected;		 /* flag indicating if tuner is selected       */
	int ir_required;                 /* flag indicating if IR chip requested       */
	unsigned long tuner_frequency;	 /* cuurent tuner frequency		       */
	int tuner_used;			 /* flag indicating if tuner is used	       */
	int audio_required;		 /* flag indicating if audio is requested      */
	struct video_audio audio;	 /* current audio settings		       */ 
	int audio_decoder_used;		 /* flag indicating if audio decoder is used   */
	int audio_processor_used;	 /* flag indicating if audio processor is used */
	int sources;			 /* number of video inputs		       */
	int channel;                     /* currently selected video source            */
	struct rivatv_dma dma;           /* DMA structure                              */
	struct rivatv_sched sched;
	u32 last_RM, last_SU;
        int last_buffer_processed;
};

/* Image conversion definitions. */

typedef void (* line_convert_t)	 (char *src, char *dst, int width);
typedef void (* plane_convert_t) (char *src, char *dst, int width, int height);

struct rivatv_conversion {
	int bpp;	     /* number of bits per pixel of source format      */
	int planar;	     /* is the image format planar (1) or packed (0) ? */
	char *name;	     /* short name				       */
	void *converter[17]; /* image converter for each destination format    */
};

extern struct __initdata rivatv_card rivatv_cards[];
extern struct __initdata rivatv_vendor rivatv_vendors[];
extern struct rivatv_conversion rivatv_convert[17];
extern unsigned int rivatv_devices;
extern int isMMX;

/* Configurable module parameters. */

extern int capbuffers;
extern int autoload;
extern int debug;
extern int tvbox;
extern int mtrr;
extern int card;
extern int dma;
extern int agp;
extern int mmx;
extern int conversion;

/* Internal interfaces */

extern unsigned int rivatv_devices;
extern struct pci_driver rivatv_driver;
extern struct __initdata rivatv_initdata rivatv_driver_init;

extern int  rivatv_register_i2c (struct rivatv_info *);
extern void rivatv_unregister_i2c (struct rivatv_info *);
extern int  rivatv_video_start (struct rivatv_info *);
extern int  rivatv_video_stop (struct rivatv_info *);
extern void rivatv_overlay_start (struct rivatv_info *);
extern void rivatv_overlay_stop (struct rivatv_info *);
extern void rivatv_reset_queue (struct rivatv_info *);
extern int  rivatv_add_queue (struct rivatv_info *, struct video_mmap *);
extern int  rivatv_sync (struct rivatv_info *, int);
extern int  rivatv_check_format (struct rivatv_info *, struct video_mmap *);
extern int  rivatv_check_source (struct rivatv_info *, struct video_mmap *);
extern void rivatv_get_capability (struct rivatv_info *, struct video_capability *);
extern void rivatv_set_overlay_format (struct rivatv_info *);
extern int  rivatv_mmap (struct rivatv_info *, struct vm_area_struct *, ulong, ulong);
extern int  rivatv_register_procfs (struct rivatv_info *);
extern void rivatv_unregister_procfs (struct rivatv_info *);
extern unsigned long vmalloc_to_bus_addr (unsigned long);
extern void rivatv_schedule_next (struct rivatv_info *);
extern void rivatv_detect_mmx (void);
extern void rivatv_enable_tuner (struct i2c_client *, unsigned char);
extern int  rivatv_palette_supported (struct rivatv_info *, int);

#endif /* __RIVATV_H */
