/* ------------------------------------------------------------------------- */
/*   saa7108e.c I2C driver for Philips SAA7108E video decoder chips	     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2002, 2003, 2004 Stefan Jahn <stefan@lkcc.org>	     
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>

#include "rivatv-kcompat.h"
#include "saa7108e.h"

#define SAA7108E_I2C_VERSION_ID	 0x7108
#define SAA7114H_I2C_VERSION_ID	 0x7114

/* Philips SAA7114H and SAA7108AE video decoder chip is identical */

/* Debug option */
static int debug = 0;

#define SAA7108E_PREFIX SAA7108E_DEVICE_NAME ": "
#define SAA7108E_PRINTK(format, args...)			\
	printk (SAA7108E_PREFIX format, ## args)
#define SAA7108E_DEBUG_PRINTK(format, args...) if (debug >= 1)	\
	printk (SAA7108E_PREFIX format, ## args)
#define SAA7108E_DEBUG_PRINTK2(format, args...) if (debug >= 2) \
	printk (SAA7108E_PREFIX format, ## args)
#undef	PRINTK
#define PRINTK(format, args...)					\
	printk (format, ## args)
#undef	DPRINTK
#define DPRINTK(format, args...) if (debug >= 1)		\
	printk (format, ## args)
#undef	DPRINTK2
#define DPRINTK2(format, args...) if (debug >= 2)		\
	printk (format, ## args)

#define SAA7108E_REG_ADDR(client, ptr) \
	((ulong) ptr - (ulong) &((saa7108e_t *) i2c_get_clientdata (client))->reg.field)

#define SAA7108E_BYTEVAL(a) (*((__u8 *)&(a)))

/*----------------------------------------------------------------------
 *	SAA7108E I2C 'driver' driver
 *----------------------------------------------------------------------*/

static int saa7108e_attach   (struct i2c_adapter *adapter);
static int saa7108e_detach   (struct i2c_client *client);
static int saa7108e_command  (struct i2c_client *client, 
			      unsigned int cmd, void *arg);

/* Possible SAA7108E addresses: 0x20, 0x21 */
static __u16 normal_i2c[] = { 0x20, 0x21, I2C_CLIENT_END };
static __u16 normal_i2c_range[] = { I2C_CLIENT_END };

/* Magic definition of all other I2C variables and things */
I2C_CLIENT_INSMOD;

/* SAA7108E instance */
typedef struct {
	union {
		saa7108e_reg_map_t field;
		u8		   byte[SAA7108E_NR_REGISTER];
	} reg;
	__u16 bright;
	__u16 contrast;
	__u16 sat;
	__u16 hue;
} saa7108e_t;

/* Initial mode (from datasheet) */
static __u8 init_regs[SAA7108E_NR_REGISTER] = {
	/* 0x00 */
	0x00, 0x08, 0xc0, 0x00, 0x90, 0x90, 0xeb, 0xe0,
	0xb8, 0x40, 0x80, 0x44, 0x40, 0x00, 0x81, 0x29,
	
	/* 0x10 */
	0x04, 0x00, 0x00, 0x00, 0x00, 0x11, 0xfe, 0xc0,
	0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa1,

	/* 0x20 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* 0x30 */
	0xbc, 0xdf, 0x02, 0x00, 0xcd, 0xcc, 0x3a, 0x00,
	0x03, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* 0x40 */
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,

	/* 0x50 */
	0xff, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff,
	0x00, 0x47, 0x06, 0x83, 0x00, 0x3e, 0x00, 0x00,

	/* 0x60 */
	0x00, 0x07, 0xaf, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* 0x70 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* 0x80 */
	0x30, 0x00, 0x00, 0x00, 0xa0, 0x10, 0x45, 0x00,
	0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b,

	/* 0x90 */
	0x00, 0x48, 0x00, 0x84, 0x02, 0x00, 0xd0, 0x02,
	0x00, 0x00, 0x0d, 0x00, 0x00, 0x06, 0x0d, 0x00,

	/* 0xa0 */
	0x01, 0x00, 0x00, 0x00, 0x80, 0x40, 0x40, 0x00,
	0x40, 0x02, 0x00, 0x00, 0x20, 0x01, 0x00, 0x00,

	/* 0xb0 */
	0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* 0xc0 */
	0x00, 0x08, 0x00, 0x80, 0x10, 0x00, 0xd0, 0x02,
	0x16, 0x00, 0xf2, 0x00, 0xd0, 0x02, 0xf2, 0x00,

	/* 0xd0 */
	0x01, 0x00, 0x00, 0x00, 0x80, 0x40, 0x40, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,

	/* 0xe0 */
	0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* 0xf0 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* For registering the I2C 'driver' driver */
static struct i2c_driver i2c_driver_saa7108e = {
	.name		= "SAA7108E",
	.id		= I2C_DRIVERID_SAA7108,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter = saa7108e_attach,
	.detach_client	= saa7108e_detach,
	.command	= saa7108e_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use	= i2c_client_inc_use,
	.dec_use	= i2c_client_dec_use,
#else
	.owner		= THIS_MODULE,
#endif
};

/* Template for new clients */
static struct i2c_client saa7108e_client = {
	I2C_DEVNAME ("Philips SAA7108E"),
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	.id      = -1,
#endif
	.flags   = I2C_CLIENT_ALLOW_USE,
	.addr    = 0,
	.adapter = NULL,
	.driver  = &i2c_driver_saa7108e,
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
/* Unique ID allocation */
static int saa7108e_id = 0;
#endif

/* Read from a register */
static int saa7108e_read (struct i2c_client *client, void *reg)
{
	int val = i2c_smbus_read_byte_data (client, SAA7108E_REG_ADDR (client, reg));

	if (val < 0) {
		SAA7108E_PRINTK ("failed reading register 0x%02lx\n",
				 SAA7108E_REG_ADDR (client, reg));
	} else {
		(*((u8*)(reg))) = val;
	}
 
	return val;
}

/* Write to a register */
static int saa7108e_write (struct i2c_client *client, void *reg)
{
	int val = i2c_smbus_write_byte_data (client, SAA7108E_REG_ADDR (client, reg),
					     *(u8*)reg);

	if (val < 0) {
		SAA7108E_PRINTK ("failed writing register 0x%02lx\n",
				 SAA7108E_REG_ADDR (client, reg));
	}

	return saa7108e_read (client, reg);
}

/* Read all registers */
static int saa7108e_read_block (struct i2c_client *client)
{
	saa7108e_t *saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	int i;

	for (i = 0 ; i < SAA7108E_NR_REGISTER ; i++) {
		int error;
		if ((error = saa7108e_read (client, &saa7108e->reg.byte[i])) < 0)
			return error;
	}
	return 0;
}

/* Write to a register block */
static int saa7108e_write_block (struct i2c_client *client)
{
	saa7108e_t *saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	int i;

	for (i = 0 ; i < SAA7108E_NR_REGISTER ; i++) {
		int error;
		if ((error = saa7108e_write (client, &saa7108e->reg.byte[i])) < 0)
			return error;
	}
	return saa7108e_read_block (client);
}


/* Dump all registers */
static void saa7108e_dump_regs (struct i2c_client *client) {

	saa7108e_t * saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	__u8 * bmap = saa7108e->reg.byte;
	__u32 l, c;

	if (debug >= 2) {
		/* Refresh client registers */
		saa7108e_read_block (client);

		for (l = 0; l < SAA7108E_NR_REGISTER / 16; l++) {
			SAA7108E_DEBUG_PRINTK2 ("%02x |", l * 16);
			for (c = 0; c < 16 && (l * 16 + c) < SAA7108E_NR_REGISTER; c++) {
				DPRINTK2 (" %02x", bmap[l * 16 + c]);
			}
			DPRINTK2 ("\n");
		}
	}
}


/* I2C driver functions */

/* Called when a 'SAA7108E like' device found */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
static int saa7108e_detect (struct i2c_adapter *adap, int addr, unsigned short flags, int kind)
#else
static int saa7108e_detect (struct i2c_adapter *adap, int addr, int kind)
#endif
{
	struct i2c_client * client;
	saa7108e_t * saa7108e;
	int err = 0, version, ret, idx;

	if (!i2c_check_functionality (adap, I2C_FUNC_SMBUS_READ_BYTE_DATA | 
				      I2C_FUNC_SMBUS_WRITE_BYTE_DATA))
		goto err_out;

	/*
	 * Allocate an I2C client structure and setup it's fields
	 */
	client = kmalloc (sizeof (*client), GFP_KERNEL);

	/* check client structure allocation error */
	if (client == NULL) {
		err = -ENOMEM;
		goto err_out;
	}
	
	/* Zero client structure fields */
	memcpy (client, &saa7108e_client, sizeof (*client));

	/* Set client's adapter */
	client->adapter = adap;

	/* Set client's address */
	client->addr = addr;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	/* Set SAA7108E client ID, update for next client */
	client->id = saa7108e_id++;
#endif

	if (kind < 0) {
		/* Write version index register 2, 3, 4 and 5 */
		for (version = 0, idx = 2; idx <= 5; idx++) {
			if ((ret = i2c_smbus_write_byte_data (client, 0x00, idx)) < 0) {
				err = -EIO;
				goto err_out_kfree_client;
			}
			/* If we could not get version, bail out */
			if ((ret = i2c_smbus_read_byte_data (client, 0x00)) < 0) {
				err = -EIO;
				goto err_out_kfree_client;
			}
			version <<= 4;
			version |= ret & 0x0f;
		}

		/* If not a SAA7108E or SAA7114H, bail out */
		switch (version) {
		case SAA7108E_I2C_VERSION_ID:
			SAA7108E_PRINTK ("video decoder chip SAA7108E found, chip version: %#x\n", version);
			break;
		case SAA7114H_I2C_VERSION_ID:
			SAA7108E_PRINTK ("video decoder chip SAA7114H found, chip version: %#x\n", version);
			break;
		default:
			SAA7108E_PRINTK ("unknown SAA71xx chip found, chip version: %#x\n", version);
			err = -ENODEV;
			goto err_out_kfree_client;
		}
	}
	else {
		SAA7108E_PRINTK ("detection skipped\n");
	}
	
	saa7108e = kmalloc (sizeof(saa7108e_t), GFP_KERNEL);

	if (saa7108e == NULL) {
		SAA7108E_PRINTK ("failed allocating client data\n");
		err = -ENOMEM;
		goto err_out_kfree_client;
	}

	/* store client data pointer into client */
	i2c_set_clientdata (client, saa7108e);

	/* Initialize client registers and write them into chip */
	saa7108e->reg.field = *((saa7108e_reg_map_t *)init_regs);

	if (saa7108e_write_block (client)) {
		SAA7108E_PRINTK ("failed to write initial register values\n");
		err = -EIO;
		goto err_out_kfree_decoder;
	}

	if ((err = i2c_attach_client (client))) {
		SAA7108E_PRINTK ("failed attaching client\n");
		goto err_out_kfree_decoder;
	}

	saa7108e_dump_regs (client);
	return 0;
	
 err_out_kfree_decoder:
	kfree (saa7108e);
 err_out_kfree_client:
	kfree (client);
 err_out:
	return err;
}

/* Called when a new I2C bus found */
static int saa7108e_attach (struct i2c_adapter *adapter)
{
	return i2c_probe (adapter, &addr_data, saa7108e_detect);
}

/* Called on exit */
static int saa7108e_detach (struct i2c_client *client)
{
	int err;

	/* Try to detach client, bail on error */
	if ((err = i2c_detach_client (client)))
		return err;

	/* Free client data */
	kfree (i2c_get_clientdata (client));

	/* Free client */
	kfree (client);

	return 0;
}

typedef struct {
	/* colour standard identifier */
	int standard;
	/* textual colour standard description */
	char *name;
} saa7108e_standard_t;

static saa7108e_standard_t saa7108e_standard[2][8] = {
	/* 50 Hz standards */
	{ { SAA7108E_CSTD_PALB,	  "PAL-B"	    },
	  { SAA7108E_CSTD_NTSC50, "NTSC-4.43(50Hz)" },
	  { SAA7108E_CSTD_PALN,	  "PAL-N"	    },
	  { SAA7108E_CSTD_NTSCN,  "NTSC-N"	    },
	  { -1,			  "<Invalid 50Hz>"  },
	  { SAA7108E_CSTD_SECAM,  "SECAM"	    },
	  { -1,			  "<Invalid 50Hz>"  },
	  { -1,			  "<Invalid 50Hz>"  } },

	/* 60 Hz standards */
	{ { SAA7108E_CSTD_NTSCM,  "NTSC-M"	    },
	  { SAA7108E_CSTD_PAL60,  "PAL-4.43(60Hz)"  },
	  { SAA7108E_CSTD_NTSC60, "NTSC-4.43(60Hz)" },
	  { SAA7108E_CSTD_PALM,	  "PAL-M"	    },
	  { SAA7108E_CSTD_NTSCJ,  "NTSC-Japan"	    },
	  { -1,			  "<Invalid 60Hz>"  },
	  { -1,			  "<Invalid 60Hz>"  },
	  { -1,			  "<Invalid 60Hz>"  } }
};

/* Autodetection of colour standard */
static void saa7108e_detection (struct i2c_client *client)
{
	saa7108e_t *	     saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	saa7108e_reg_map_t * map      = &saa7108e->reg.field;
	int n, idx, old, detected;

	/* disable adaptive luminance comb filter */
	map->luminance_control.YCOMB = 1;
	saa7108e_write (client, &map->luminance_control);

	/* enable automatic gain control */
	map->chrominance_gain_control.ACGC = 0;
	saa7108e_write (client, &map->chrominance_gain_control);

	/* detect 50/60Hz */
	map->sync_control.AUFD = SAA7108E_AUFD_AUTO;
	saa7108e_write (client, &map->sync_control);

	/* stay here a little eternity */
	set_current_state (TASK_UNINTERRUPTIBLE);
	schedule_timeout (HZ/10);
	set_current_state (TASK_RUNNING);
			
	saa7108e_read (client, &map->decoder_status);

	/* set explicitely the detected frequency */
	idx = map->decoder_status.FIDT ? 1 : 0;
	map->sync_control.AUFD = SAA7108E_AUFD_FIXED;
	map->sync_control.FSEL = map->decoder_status.FIDT;
	saa7108e_write (client, &map->sync_control);
	
	/* save old colour standard */
	saa7108e_read (client, &map->chrominance_control);
	old = map->chrominance_control.CSTD;

	/* set old status byte */
	map->analog_compatibility_control.OLDSB = 1;
	saa7108e_write (client, &map->analog_compatibility_control);
	
	for (detected = n = 0; n < 8; n++) {
		if (saa7108e_standard[idx][n].standard != -1) {
			/* set colour standard */
			map->chrominance_control.CSTD = saa7108e_standard[idx][n].standard;
			saa7108e_write (client, &map->chrominance_control);
			
			/* stay here a little eternity */
			set_current_state (TASK_UNINTERRUPTIBLE);
			schedule_timeout (HZ/10);
			set_current_state (TASK_RUNNING);
			
			/* check status byte */
			saa7108e_read (client, &map->decoder_status);
			if (map->decoder_status.RDCAP) {
				DPRINTK (" %s", saa7108e_standard[idx][n].name);
				detected++;
				break;
			}
		}
	}

	/* restore old colour standard */
	if (!detected) {
		map->chrominance_control.CSTD = old;
		saa7108e_write (client, &map->chrominance_control);
	}

	/* set default status byte */
	map->analog_compatibility_control.OLDSB = 0;
	saa7108e_write (client, &map->analog_compatibility_control);
}

/* Enable or disable the scaler unit */
static void saa7108e_enable_scaler (struct i2c_client *client, int enable)
{
	saa7108e_t *	     saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	saa7108e_reg_map_t * map      = &saa7108e->reg.field;

	if (enable) {
		map->global_control.TEA = (enable & 0x01) ? 1 : 0;
		map->global_control.TEB = (enable & 0x02) ? 1 : 0;
		map->power_save.SWRST = 1;
		map->power_save.DPROG = 1;
	}
	else {
		map->global_control.TEA = 0;
		map->global_control.TEB = 0;
		map->power_save.SWRST = 0;
		map->power_save.DPROG = 0;
	}
	saa7108e_write (client, &map->global_control);
	saa7108e_write (client, &map->power_save);
}

/* Setup one of the scalers */
static void saa7108e_setup_scaler (struct i2c_client *client, saa7108e_scaler_t *scaler,
				   int src_x, int src_y, int src_width, int src_height,
				   int dst_width, int dst_height)
{
	int scale;

	/* x port definition */
	scaler->xport_format_config.SCRQE = 1;
	scaler->xport_format_config.SCSRC = 0;
	scaler->xport_format_config.FSC0 = 0;
	scaler->xport_format_config.FSC1 = 0;
	scaler->xport_format_config.CONLV = 0;
	scaler->xport_input_signal.XDV0 = 0;
	saa7108e_write (client, &scaler->xport_format_config);
	saa7108e_write (client, &scaler->xport_input_signal);

	/* input port definition */
	scaler->iport_format_config.FOI = 0;
	scaler->iport_format_config.FYSK = 0;
	scaler->iport_format_config.I8_16 = 0;
	scaler->iport_format_config.ICODE = 1;
	saa7108e_write (client, &scaler->iport_format_config);

	/* input window */
	scaler->horizontal_iwindow_start_1.XO = src_x & 0xff;
	scaler->horizontal_iwindow_start_2.XO_MSB = src_x >> 8;
	scaler->horizontal_iwindow_length_1.XS = src_width & 0xff;
	scaler->horizontal_iwindow_length_2.XS_MSB = src_width >> 8;
	scaler->vertical_iwindow_start_1.YO = src_y & 0xff;
	scaler->vertical_iwindow_start_2.YO_MSB = src_y >> 8;
	scaler->vertical_iwindow_length_1.YS = src_height & 0xff;
	scaler->vertical_iwindow_length_2.YS_MSB = src_height >> 8;
	saa7108e_write (client, &scaler->horizontal_iwindow_start_1);
	saa7108e_write (client, &scaler->horizontal_iwindow_start_2);
	saa7108e_write (client, &scaler->horizontal_iwindow_length_1);
	saa7108e_write (client, &scaler->horizontal_iwindow_length_2);
	saa7108e_write (client, &scaler->vertical_iwindow_start_1);
	saa7108e_write (client, &scaler->vertical_iwindow_start_2);
	saa7108e_write (client, &scaler->vertical_iwindow_length_1);
	saa7108e_write (client, &scaler->vertical_iwindow_length_2);

	/* output window */
	scaler->horizontal_owindow_length_1.XD = dst_width & 0xff;
	scaler->horizontal_owindow_length_2.XD_MSB = dst_width >> 8;
	scaler->vertical_owindow_length_1.YD = dst_height & 0xff;
	scaler->vertical_owindow_length_2.YD_MSB = dst_height >> 8;
	saa7108e_write (client, &scaler->horizontal_owindow_length_1);
	saa7108e_write (client, &scaler->horizontal_owindow_length_2);
	saa7108e_write (client, &scaler->vertical_owindow_length_1);
	saa7108e_write (client, &scaler->vertical_owindow_length_2);

	/* no prescaling */
	scaler->horizontal_prescale_ratio.XPSC = 1;
	saa7108e_write (client, &scaler->horizontal_prescale_ratio);

	/* horizontal scaler */
	scale = 1024 * src_width / dst_width;
	scaler->horizontal_luminance_scaler_1.XSCY = scale & 0xff;
	scaler->horizontal_luminance_scaler_2.XSCY_MSB = scale >> 8;
	scale >>= 1;
	scaler->horizontal_chrominance_scaler_1.XSCC = scale & 0xff;
	scaler->horizontal_chrominance_scaler_2.XSCC_MSB = scale >> 8;
	saa7108e_write (client, &scaler->horizontal_luminance_scaler_1);
	saa7108e_write (client, &scaler->horizontal_luminance_scaler_2);
	saa7108e_write (client, &scaler->horizontal_chrominance_scaler_1);
	saa7108e_write (client, &scaler->horizontal_chrominance_scaler_2);

	/* vertical scaler */
	scale = 1024 * src_height / dst_height;
	scaler->vertical_luminance_scaler_1.YSCY = scale & 0xff;
	scaler->vertical_luminance_scaler_2.YSCY_MSB = scale >> 8;
	scaler->vertical_chrominance_scaler_1.YSCC = scale & 0xff;
	scaler->vertical_chrominance_scaler_2.YSCC_MSB = scale >> 8;
	saa7108e_write (client, &scaler->vertical_luminance_scaler_1);
	saa7108e_write (client, &scaler->vertical_luminance_scaler_2);
	saa7108e_write (client, &scaler->vertical_chrominance_scaler_1);
	saa7108e_write (client, &scaler->vertical_chrominance_scaler_2);
}

static void saa7108e_setup_scaler_A (struct i2c_client *, int, int, int, int, int, int, int) __attribute__ ((unused));

/* Setup scaler task A */
static void saa7108e_setup_scaler_A (struct i2c_client *client,
				     int src_x, int src_y, int src_width, int src_height,
				     int dst_width, int dst_height, int enable)
{
	saa7108e_t *	     saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	saa7108e_reg_map_t * map      = &saa7108e->reg.field;
	saa7108e_scaler_t  * scaler   = &map->taska;

	/* global settings */
	map->global_control.TEA = enable ? 1 : 0;
	map->xport_enable_output_clock.XPE = 0;
	map->iport_enable_output_clock.IPE = 1;
	map->iport_fifo_signal.VITX = 1;
	map->iport_signal.IDG0 = 2;
	map->iport_fifo_signal.IDG02 = 0;
	map->iport_signal.IDG1 = 2;
	map->iport_fifo_signal.IDG12 = 0;
	map->power_save.CH2EN = 1;
	map->power_save.CH4EN = 1;
	map->power_save.SWRST = 1;
	map->power_save.DPROG = 1;
	saa7108e_write (client, &map->global_control);
	saa7108e_write (client, &map->xport_enable_output_clock);
	saa7108e_write (client, &map->iport_enable_output_clock);
	saa7108e_write (client, &map->power_save);
	saa7108e_write (client, &map->iport_fifo_signal);
	saa7108e_write (client, &map->iport_signal);

	saa7108e_setup_scaler (client, scaler, 
			       src_x, src_y, src_width, src_height, dst_width, dst_height);
}

/* Setup scaler task B */
static void saa7108e_setup_scaler_B (struct i2c_client *client,
				     int src_x, int src_y, int src_width, int src_height,
				     int dst_width, int dst_height, int enable)
{
	saa7108e_t *	     saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	saa7108e_reg_map_t * map      = &saa7108e->reg.field;
	saa7108e_scaler_t  * scaler   = &map->taskb;

	/* global settings */
	map->global_control.TEB = enable ? 1 : 0;
	map->xport_enable_output_clock.XPE = 0;
	map->iport_enable_output_clock.IPE = 1;
	map->iport_fifo_signal.VITX = 1;
	map->iport_signal.IDG0 = 2;
	map->iport_fifo_signal.IDG02 = 0;
	map->iport_signal.IDG1 = 2;
	map->iport_fifo_signal.IDG12 = 0;
	map->power_save.CH2EN = 1;
	map->power_save.CH4EN = 1;
	map->power_save.SWRST = 1;
	map->power_save.DPROG = 1;
	saa7108e_write (client, &map->global_control);
	saa7108e_write (client, &map->xport_enable_output_clock);
	saa7108e_write (client, &map->iport_enable_output_clock);
	saa7108e_write (client, &map->power_save);
	saa7108e_write (client, &map->iport_fifo_signal);
	saa7108e_write (client, &map->iport_signal);

	saa7108e_setup_scaler (client, scaler, 
			       src_x, src_y, src_width, src_height, dst_width, dst_height);
}

/* Interface to the world */
static int saa7108e_command (struct i2c_client *client, unsigned int cmd, void *arg)
{
	saa7108e_t * saa7108e = (saa7108e_t *) i2c_get_clientdata (client);
	saa7108e_reg_map_t * map = &saa7108e->reg.field;

	switch (cmd) {
		
#ifdef READ_REGISTERS
	case READ_REGISTERS:
		/* Read to client register map */
		saa7108e_read_block (client);

		/* Copy client map to passed arg */
		*((saa7108e_reg_map_t *)arg) = *map;

		SAA7108E_DEBUG_PRINTK ("READ_REGISTERS\n");

		saa7108e_dump_regs (client);

		return SAA7108E_NR_REGISTER;

	case WRITE_REGISTERS:
		/* Write passed map to client map */
		*map = *((saa7108e_reg_map_t *)arg);

		/* Write client map */
		saa7108e_write_block (client);

		SAA7108E_DEBUG_PRINTK ("WRITE_REGISTERS\n");

		saa7108e_dump_regs (client);

		return SAA7108E_NR_REGISTER;

	case GET_NR_OF_REGISTERS:
		SAA7108E_DEBUG_PRINTK ("GET_NR_OF_REGISTERS = %d\n",
				       SAA7108E_NR_REGISTER);

		return SAA7108E_NR_REGISTER;
#endif /* READ_REGISTERS */

#ifdef DECODER_DUMP
	case DECODER_DUMP:

		SAA7108E_DEBUG_PRINTK ("DECODER_DUMP\n");

		saa7108e_dump_regs (client);

		break;
#endif /* DECODER_DUMP */

#ifdef WRITE_REGISTER
	case WRITE_REGISTER:
	{
		u16 *sarg = arg;

		SAA7108E_DEBUG_PRINTK ("WRITE_REGISTER\n");
		saa7108e->reg.byte[*sarg >> 8] = *sarg & 0xff;
		saa7108e_write (client, &saa7108e->reg.byte[*sarg >> 8]);
		break;
	}
#endif /* WRITE_REGISTER */

	case DECODER_GET_CAPABILITIES:
	{
		struct video_decoder_capability *cap = arg;

		SAA7108E_DEBUG_PRINTK ("DECODER_GET_CAPABILITIES\n");
		cap->flags
			= VIDEO_DECODER_PAL
			| VIDEO_DECODER_SECAM
			| VIDEO_DECODER_NTSC
			| VIDEO_DECODER_AUTO
			| VIDEO_DECODER_CCIR;
		cap->inputs = 10;
		cap->outputs = 2;
		break;
	}


	case DECODER_GET_STATUS:
	{
		int *iarg = arg;
		int res;

		saa7108e_read (client, &map->decoder_status);
		saa7108e_read (client, &map->sync_control);
		SAA7108E_DEBUG_PRINTK ("DECODER_GET_STATUS = 0x%02x ->",
				       SAA7108E_BYTEVAL (map->decoder_status));

		res = 0;

		/* Get horizontal locking state */
		if (map->decoder_status.HLVLN == 0) {
			DPRINTK (" GOOD");
			res = DECODER_STATUS_GOOD;
		} else {
			DPRINTK (" BAD");
		}

		/* Get or detect decoder norm */
		if (map->sync_control.AUFD == SAA7108E_AUFD_AUTO) {
			DPRINTK (" AUTODETECT");
			if (map->decoder_status.FIDT) { /* 60/50 hz */
				res |= DECODER_STATUS_NTSC;
			} else {
				res |= DECODER_STATUS_PAL;
			}
		} else {
			if (map->sync_control.FSEL == SAA7108E_FSEL_50HZ) {
				res |= DECODER_STATUS_PAL;
			} else {
				res |= DECODER_STATUS_NTSC;
			}
		}

		if (res & DECODER_STATUS_PAL) {
			switch (map->chrominance_control.CSTD) {
			case SAA7108E_CSTD_PALB:
				DPRINTK (" PAL-B");
				break;

			case SAA7108E_CSTD_NTSC50:
				DPRINTK (" NTSC-4.43(50Hz)");
				break;

			case SAA7108E_CSTD_PALN:
				DPRINTK (" PAL-N");
				break;

			case SAA7108E_CSTD_PALM:
				DPRINTK (" PAL-M");
				break;

			case SAA7108E_CSTD_SECAM:
				res = (res & ~DECODER_STATUS_PAL) | DECODER_STATUS_SECAM;
				DPRINTK (" SECAM");
				break;
			}
		} else {
			switch (map->chrominance_control.CSTD) {
			case SAA7108E_CSTD_NTSCM:
				DPRINTK (" NTSC-M");
				break;

			case SAA7108E_CSTD_PAL60:
				DPRINTK (" PAL-4.43(60Hz)");
				break;

			case SAA7108E_CSTD_NTSC60:
				DPRINTK (" NTSC-4.43(60Hz)");
				break;

			case SAA7108E_CSTD_NTSCJ:
				DPRINTK (" NTSC-Japan");
				break;
			}      
		}

		if (map->decoder_status.WIPA) {
			DPRINTK (" WHITE_PEAK");
		}
      
		if (map->decoder_status.COPRO) {
			DPRINTK (" COPY_PROTECT");
		}

		if (map->decoder_status.RDCAP) {
			DPRINTK (" READY_FOR_CAPTURE");
		}

		if (map->mode_delay_control.COLO) {
			DPRINTK (" COLOR");
			res |= DECODER_STATUS_COLOR;
		}

		DPRINTK ("\n");
		*iarg = res;
		break;
	}

	case DECODER_SET_NORM: 
	{
		int *iarg = arg;

		SAA7108E_DEBUG_PRINTK ("DECODER_SET_NORM: ");

		switch (*iarg)
		{

		case VIDEO_MODE_NTSC:
			DPRINTK ("NTSC");
			map->sync_control.AUFD = SAA7108E_AUFD_FIXED;
			map->sync_control.FSEL = SAA7108E_FSEL_60HZ;
			map->chrominance_control.CSTD = SAA7108E_CSTD_NTSCM;
			map->chrominance_gain_control.ACGC = 0;
			map->luminance_control.YCOMB = 1;
			saa7108e_write (client, &map->sync_control);
			saa7108e_write (client, &map->chrominance_control);
			saa7108e_write (client, &map->chrominance_gain_control);
			saa7108e_write (client, &map->luminance_control);
			break;

		case VIDEO_MODE_PAL:
			DPRINTK ("PAL");
			map->sync_control.AUFD = SAA7108E_AUFD_FIXED;
			map->sync_control.FSEL = SAA7108E_FSEL_50HZ;
			map->chrominance_control.CSTD = SAA7108E_CSTD_PALB;
			map->chrominance_gain_control.ACGC = 0;
			map->luminance_control.YCOMB = 1;
			saa7108e_write (client, &map->sync_control);
			saa7108e_write (client, &map->chrominance_control);
			saa7108e_write (client, &map->chrominance_gain_control);
			saa7108e_write (client, &map->luminance_control);
			break;

		case VIDEO_MODE_SECAM:
			DPRINTK ("SECAM");
			map->sync_control.AUFD = SAA7108E_AUFD_FIXED;
			map->sync_control.FSEL = SAA7108E_FSEL_50HZ;
			map->chrominance_control.CSTD = SAA7108E_CSTD_SECAM;
			map->chrominance_gain_control.ACGC = 1;
			map->chrominance_gain_control.CGAIN = SAA7108E_CGAIN_NOMINAL;
			map->luminance_control.YCOMB = map->luminance_control.BYPS;
			saa7108e_write (client, &map->sync_control);
			saa7108e_write (client, &map->chrominance_control);
			saa7108e_write (client, &map->chrominance_gain_control);
			saa7108e_write (client, &map->luminance_control);
			break;

		case VIDEO_MODE_AUTO:
			DPRINTK ("AUTODETECT");
			saa7108e_detection (client);
			break;

		default:
			DPRINTK ("<invalid value %d>\n", *iarg);
			return -EINVAL;

		}
		DPRINTK ("\n");
		break;
	}

	case DECODER_SET_INPUT:
	{
		int *iarg = arg;

		SAA7108E_DEBUG_PRINTK ("DECODER_SET_INPUT: %d", *iarg);

		map->analog_input_control_1.MODE = *iarg;
		saa7108e_write (client, &map->analog_input_control_1);

		/* bypass chrominance trap for modes 6..9 */
		if (*iarg >= 6 && *iarg <= 9) {
			map->luminance_control.BYPS = 1;
			saa7108e_write (client, &map->luminance_control);
			DPRINTK (" S-Video");
		} else {
			map->luminance_control.BYPS = 0;
			saa7108e_write (client, &map->luminance_control);
			DPRINTK (" CVBS");
		}
		DPRINTK ("\n");
		break;
	}

	case DECODER_SET_OUTPUT:
	{
		int *iarg = arg;

		SAA7108E_DEBUG_PRINTK ("DECODER_SET_OUTPUT: %d\n", *iarg);

		/* not much choice of outputs */
		if (*iarg != 0) {
			return -EINVAL;
		}
		break;
	}

	case DECODER_ENABLE_OUTPUT:
	{
		int *iarg = arg;

		SAA7108E_DEBUG_PRINTK ("DECODER_ENABLE_OUTPUT: %d\n", *iarg);

		if (*iarg) {
			/* Lock horizontal PLL */
			map->sync_control.HPLL = SAA7108E_HPLL_CLOSED;

			/* Enable YUV output data */
			map->xport_output_control.OFTS = 0;

		} else {
			/* Unlock horizontal PLL */
			map->sync_control.HPLL = SAA7108E_HPLL_OPEN;
	    
			/* Disable YUV output data */
			map->xport_output_control.OFTS = 0;
		}

		/* Write modified data to chip */
		saa7108e_write (client, &map->sync_control);
		saa7108e_write (client, &map->xport_output_control);

		/* Setup scaler tasks */
		saa7108e_read (client, &map->decoder_status);
		if (map->decoder_status.FIDT) { /* 60 Hz */
			/*saa7108e_setup_scaler_A (client, 10, 18, 704, 240, 720, 240, 0);*/
			saa7108e_setup_scaler_B (client, 10, 18, 704, 240, 720, 240, 1);
		}
		else { /* 50 Hz */
			/*saa7108e_setup_scaler_A (client, 10, 18, 704, 288, 720, 288, 0);*/
			saa7108e_setup_scaler_B (client, 10, 18, 704, 288, 720, 288, 1);
		}
		saa7108e_enable_scaler (client, 0);
		saa7108e_enable_scaler (client, 2);

		map->output_control.RTSE0 = 0;
		map->output_control.RTSE1 = 0;
		saa7108e_write (client, &map->output_control);
		break;
	}

	case DECODER_SET_PICTURE:
	{
		struct video_picture *pic = arg;

		SAA7108E_DEBUG_PRINTK ("DECODER_SET_PICTURE\n");

		if (saa7108e->bright != pic->brightness) {
			/* We want 0 to 255 we get 0-65535 */
			saa7108e->bright = pic->brightness;
			map->luminance_brightness.BRIG = saa7108e->bright >> 8;
			saa7108e_write (client, &map->luminance_brightness);
		}
		if (saa7108e->contrast != pic->contrast) {
			/* We want 0 to 127 we get 0-65535 */
			saa7108e->contrast = pic->contrast;
			map->luminance_contrast.CONT = saa7108e->contrast >> 9;
			saa7108e_write (client, &map->luminance_contrast);
		}
		if (saa7108e->sat != pic->colour) {
			/* We want 0 to 127 we get 0-65535 */
			saa7108e->sat = pic->colour;
			map->chrominance_saturation.SATN = saa7108e->sat >> 9;
			saa7108e_write (client, &map->chrominance_saturation);
		}
		if (saa7108e->hue != pic->hue) {
			/* We want -128 to 127 we get 0-65535 */
			saa7108e->hue = pic->hue;
			map->chrominance_hue_control.HUE = (saa7108e->hue - 32768) >> 8;
			saa7108e_write (client, &map->chrominance_hue_control);
		}
		break;
	}

	default:
		SAA7108E_DEBUG_PRINTK ("<invalid function 0x%08X>\n", cmd);
		return -EINVAL;
	}

	saa7108e_dump_regs (client);
	return 0;
}


/*---------------------------------------------------------------------- 
 *	Modularization
 *----------------------------------------------------------------------*/

MODULE_AUTHOR	   ("Stefan Jahn <stefan@lkcc.org>");
MODULE_DESCRIPTION ("SAA7108E driver");
MODULE_LICENSE	   ("GPL");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param       (debug, int, 0644);
#else
MODULE_PARM	   (debug, "i");
#endif
MODULE_PARM_DESC   (debug, "Debug level: 0 = silent, 1 = verbose, 2 = very verbose (default = 0).");

int __init init_saa7108e (void) 
{
	return i2c_add_driver (&i2c_driver_saa7108e);
}

void __exit cleanup_saa7108e (void) 
{
	i2c_del_driver (&i2c_driver_saa7108e);
}

module_init (init_saa7108e);
module_exit (cleanup_saa7108e);
