#ifndef __SAA7108E_H
#define __SAA7108E_H

#include <linux/i2c.h>
#include <linux/videodev.h>
#include <linux/video_decoder.h>

#ifndef I2C_DRIVERID_SAA7108
#define I2C_DRIVERID_SAA7108 46
#endif

#define SAA7108E_DEVICE_NAME "SAA7108E"

/*----------------------------------------------------------------------
 *	SAA7108E register map definition
 *----------------------------------------------------------------------*/

/* number of SAA7108E registers */
#define SAA7108E_NR_REGISTER	256

typedef struct {
	__u8 ID; /* chip version */
} saa7108e_r00_t, saa7108e_chip_version_t;

typedef struct {
	__u8 IDEL : 4; /*
			 bit0:	disable increment delay
			 other: delay value (NOTE 000 is max, 111 is min)
			 1000: recommended
		       */
	__u8 _X_  : 4;
} saa7108e_r01_t, saa7108e_increment_delay_t;

typedef struct {
	__u8 MODE   : 4; /*
			   select: operational mode
			   0000: CVBS from analog input A11 (automatic gain)
			   0001: CVBS from analog input A12 (automatic gain)
			   0010: CVBS from analog input A21 (automatic gain)
			   0011: CVBS from analog input A22 (automatic gain)
			   0100: CVBS from analog input A23 (automatic gain)
			   0101: CVBS from analog input A24 (automatic gain)
			   0110: luminance from A11 (auto gain), chrominance from
			   .	  A21 (gain adjustable via GAI2). BYPS should be set.
			   0111: luminance from A12 (auto gain), chrominance from
			   .	  A22 (gain adjustable via GAI2). BYPS should be set.
			   1000: luminance from A11 (auto gain), chrominance from
			   .	  A21 (gain adapted from lum.). BYPS should be set
			   1001: luminance from A12 (auto gain), chrominance from
			   .	  A22 (gain adapted from lum.). BYPS should be set
			   XXXX: Other values reserved
			 */

	__u8 GUDL   : 2; /* val: update hysterisis for 9 bit gain: +/- 0/1/2/3 LSB */

	__u8 FUSE   : 2; /*
			   select: amplifier & anti-alias activation
			   00: bypassed
			   01: bypassed
			   10: amplifier active
			   11: amplifier & anti-alias both active
			 */
} saa7108e_r02_t, saa7108e_analog_input_control_1_t;

typedef struct {
	__u8 GAI18    : 1; /* MSB (sign) of static gain control for channel 1 */
	__u8 GAI28    : 1; /* MSB (sign) of static gain control for channel 2 */
	__u8 GAFIX    : 1; /*
			     select: gain control adjustment
			     0 : gain controlled by MODE3 to MODE0
			     1 : gain is user programmable
			   */

	__u8 HOLDG    : 1; /*
			     select: automatic gain integration
			     0: active
			     1: freeze
			   */

	__u8 WPOFF    : 1; /*
			     select: white peak off
			     0: white peak active
			     1: white peak off
			   */

	__u8 VBSL     : 1; /*
			     select: AGC hold during vertical blanking period
			     0: short, AGC disabled during equal and serat. pulses
			     1: long,  AGC disabled until start of video
			     .	line 22 for NTSC, line 24 for PAL
			   */

	__u8 HLNRS    : 1; /*
			     select: HL not reference select
			     0: normal clamping if decoder is in unlocked state
			     1: reference select if decoder is in unlocked state
			   */

	__u8 _X_      : 1;
} saa7108e_r03_t, saa7108e_analog_input_control_2_t;

typedef struct {
	__u8 GAI1 : 8; /*
			 val: gain control analog; static gain channel 1
			 .    MSB set by GAI18
			 . 0: ~ -3dB
			 127: ~ 0
			 511: ~ +6dB
		       */
} saa7108e_r04_t, saa7108e_analog_input_control_3_t;

typedef struct {
	__u8 GAI2 : 8; /*
			 val: gain control analog; static gain channel 2
			 .    MSB set by GAI28
			 . 0: ~ -3dB
			 127: ~ 0
			 511: ~ +6dB
		       */
} saa7108e_r05_t, saa7108e_analog_input_control_4_t;

typedef struct {
	__u8 HSB : 8; /*
			val: horizontal sync start (step size = 8/LLC)
			.    valid range is limited to [-108;+108]
		      */
} saa7108e_r06_t, saa7108e_horizontal_sync_start_t;

typedef struct {
	__u8 HSS : 8; /*
			val: horizontal sync stop (step size = 8/LLC)
			.    valid range is limited to [-108;+108]
		      */
} saa7108e_r07_t, saa7108e_horizontal_sync_stop_t;

typedef struct {
	__u8 VNOI : 2; /*
			 select: vertical noise reduction
			 00: normal mode (recommended)
			 01: fast mode (for stable sources). AUFD must be disabled.
			 10: free running mode
			 11: vertical noise reduction bypassed
		       */

#define SAA7108E_HPLL_CLOSED 0
#define SAA7108E_HPLL_OPEN   1
	__u8 HPLL : 1; /*
			 select: horizontal PLL mode
			 0: closed
			 1: open, horizontal frequency fixed
		       */

#define SAA7108E_HTC_FAST 3
	__u8 HTC  : 2; /*
			 select: horizontal time constant selection
			 00: TV mode (poor quality signals)
			 01: VTR mode (recommended with a detection control circuit
			 .   is connected directly to SAA7108E
			 10: reserved
			 11: fast locking mode (recommended)
		       */

	__u8 FOET : 1; /*
			 select: forced odd/even toggle
			 0: odd/even signal toggles only with interlaced sources
			 1: odd/even signal toggles even if source in non interlaced
		       */

#define SAA7108E_FSEL_50HZ 0
#define SAA7108E_FSEL_60HZ 1
	__u8 FSEL : 1; /*
			 select: field selection
			 0: PAL/SECAM (50Hz, 625 lines)
			 1: NTSC (60Hz, 525 lines)
		       */

#define SAA7108E_AUFD_FIXED 0
#define SAA7108E_AUFD_AUTO  1
	__u8 AUFD : 1; /*
			 select: automatic field (PAL/SECAM/NTSC) detection
			 0: field state fixed via FSEL
			 1: field state directly controlled via FSEL (recommended)
		       */
} saa7108e_r08_t, saa7108e_sync_control_t;

typedef struct {
	__u8 LUFI  : 4; /*
			  select: sharpness, luminance filter
			  0001: resolution enhancement filter; 8.0dB at 4.1MHz
			  0010: resolution enhancement filter; 6.8dB at 4.1MHz
			  0011: resolution enhancement filter; 5.1dB at 4.1MHz
			  0100: resolution enhancement filter; 4.1dB at 4.1MHz
			  0101: resolution enhancement filter; 3.0dB at 4.1MHz
			  0110: resolution enhancement filter; 2.3dB at 4.1MHz
			  0111: resolution enhancement filter; 1.6dB at 4.1MHz
			  0000: plain
			  1000: low-pass filter; 2dB at 4.1MHz
			  1001: low-pass filter; 3dB at 4.1MHz
			  1010: low-pass filter; 3dB at 3.3MHz; 4dB at 4.1MHz
			  1011: low-pass filter; 2.6dB at 3.3MHz; 8dB at 4.1MHz
			  1100: low-pass filter; 3dB at 2.4MHz; 14dB at 4.1MHz
			  1101: low-pass filter; 3dB at 2.2MHz; notch at 3.4MHz
			  1110: low-pass filter; 3dB at 1.9MHz; notch at 3.0MHz
			  1111: low-pass filter; 3dB at 1.7MHz; notch at 2.5MHz
			*/

	__u8 LUBW  : 1; /*
			  select: remodulation bandwidth for luminance
			  0: small remodulation (narrow chrominance, higher luminance)
			  1: large remodulation (wider chrominance, smaller luminance)
			*/

	__u8 LDEL  : 1; /*
			  select: delay in non-comb filter mode
			  0: processing delay is internal pipelining delay
			  1: 1 (NTSC) or 2 (PAL) video lines additional delay
			*/

	__u8 YCOMB : 1; /*
			  select: adaptive luminance comb filter
			  0: disabled (= chrominace trap enabled, if BYPS = 0)
			  1: active, if BYPS = 0
			*/

	__u8 BYPS  : 1; /*
			  select:
			  0: chrominance trap active; default for CVBS mode
			  1: chrominance trap bypassed; default for S-Video
			*/
} saa7108e_r09_t, saa7108e_luminance_control_t;

typedef struct {
#define SAA7108E_BRIG_ITU  0x80
	__u8 BRIG : 8; /* 
			  val: brightness level 
			  255: bright
			  128: ITU
			  0  : dark
		       */
} saa7108e_r0a_t, saa7108e_luminance_brightness_t;

typedef struct {
#define SAA7108E_CONT_ITU 0x44
	__u8 CONT : 8; /* 
			  val: contrast gain
			  10000000: -2 (inverse luminance)
			  11000000: -1 (inverse luminance)
			  0:	     0 (luminance off)
			  64->127:   1->1.999
			  01000000:  1.0
		       */
} saa7108e_r0b_t, saa7108e_luminance_contrast_t;

typedef struct {
#define SAA7108E_SATN_ITU 0x40
	__u8 SATN : 8; /* 
			  val: chrominance saturation control gain
			  10000000: -2 (inverse chrominance)
			  11000000: -1 (inverse chrominance)
			  0:	     0 (colour off)
			  64->127:   0->1.999
			  01000000:  1.0 (ITU value)
		 */
} saa7108e_r0c_t, saa7108e_chrominance_saturation_t;

typedef struct {
	__u8 HUE : 8; /*
			val: chrominance hue control (phase). SIGNED
			-128: -180 degrees
			+127: +178.6 degrees
		      */
} saa7108e_r0d_t, saa7108e_chrominance_hue_control_t;

typedef struct {
	__u8 CCOMB : 1; /*
			  select: adaptive chrominace filter
			  0: disabled
			  1: active
			*/

	__u8 _X_  : 1; /* reserved */

	__u8 FCTC : 1; /*
			 select: fast colour time constant
			 0: nominal time constant
			 1: fast time constant
		       */

	__u8 DCVF : 1; /*
			 select: disable chrominance vertical filter
			 0: filter on (during active video lines)
			 1: filter permanently off
		       */

#define SAA7108E_CSTD_PALB   0
#define SAA7108E_CSTD_NTSCM  0
#define SAA7108E_CSTD_NTSC50 1
#define SAA7108E_CSTD_PAL60  1
#define SAA7108E_CSTD_PALN   2
#define SAA7108E_CSTD_NTSC60 2
#define SAA7108E_CSTD_NTSCN  3
#define SAA7108E_CSTD_PALM   3
#define SAA7108E_CSTD_NTSCJ  4
#define SAA7108E_CSTD_SECAM  5
	__u8 CSTD : 3; /* 
			  select: colour standard definition
			  000: PAL BGHIN	 / NTSC-M
			  001: NTSC 4.43 (50 Hz) / PAL 4.43 (50 Hz)
			  010: Combination PAL-N / NTSC 4.43 (60Hz)
			  011: NTSC-N		 / PAL M
			  100:			 / NTSC-Japan
			  101: SECAM		 /
			  xxx: other values should not be set
		       */

	__u8 CDT0 : 1; /*
			 select: clear DTO
			 0: disabled
			 1: every time CDTO carrier is set, the internal subcarrier
			 .  DTO phase is reset to 0 (degrees) and the RTCO output
			 .  generates a logic 0 at time slot 68...
		       */
} saa7108e_r0e_t, saa7108e_chrominance_control_1_t;

typedef struct {

#define SAA7108E_CGAIN_MINIMUM 0x00
#define SAA7108E_CGAIN_NOMINAL 0x24
#define SAA7108E_CGAIN_MAXIMUM 0x7F
	__u8 CGAIN : 7; /*
			  val: chrominance gain control 0.5 -> 7.5
			  0000000: minimum gain
			  0100100: nominal gain
			  1111111: maximum gain
			*/
		  
	__u8 ACGC  : 1; /*
			  select: automatic gain control
			  0: on
			  1: programmable gain via CGAIN (for SECAM)
			*/
} saa7108e_r0f_t, saa7108e_chrominance_gain_control_t;

typedef struct {
	__u8 LCBW : 3; /*
			 select: combined luminace/chrominance bandwidth
			 000: smallest chrominace/largest luminance
			 111: largest chrominace/smallest luminance
		       */

	__u8 CHBW : 1; /*
			 select: chrominance bandwidth
			 0: small
			 1: wide
		       */
	
	__u8 OFFV : 2; /*
			 select: fine offset adjustment R - Y component
			 0/1/2/3 LSB
		       */

	__u8 OFFU : 2; /*
			 select: fine offset adjustment B - Y component
			 0/1/2/3 LSB
		       */
} saa7108e_r10_t, saa7108e_chrominance_control_2_t;

typedef struct {
	__u8 YDEL : 3; /* val: luminance delay compensation: -4 -> +3 */

	__u8 RTP0 : 1; /*
			 select: polarity of RTS0 output signal
			 0: non inverted
			 1: inverted
		       */

	__u8 HDEL : 2; /* val: fine position of HS (step size 2/LLC): 0->3 */

	__u8 RTP1 : 1; /*
			 select: polarity of RTS1 output signal
			 0: non inverted
			 1: inverted
		       */

	__u8 COLO : 1; /* 
			  select: colour on
			  0: automatic colour killer enabled
			  1: colour forced on
		       */
} saa7108e_r11_t, saa7108e_mode_delay_control_t;

typedef struct {
	__u8 RTSE0 : 4; /*
			  select: RTS0 output control
			  0000: 3-state
			  0001: constant LOW
			  0010: CREF, 13.5MHz toggling pulse
			  0011: CREF2, 6.75MHz toggling pulse
			  0100: HL, horizontal lock indicator
			  .	HL = 0: unlocked
			  .	HL = 1; locked
			  0101: VL, vertical and horizontal lock indicator
			  .	VL = 0: unlocked
			  .	VL = 1; locked
			  0110: DL, vertical and horizontal lock and colour detected
			  .	DL = 0: unlocked
			  .	DL = 1; locked
			  0111: reserved
			  1000: HREF, horizontal reference signal, 720 pixel valid data
			  1001: HS
			  1010: HQ, HREF gated with VGATE
			  1011: reserved
			  1100: V123, vertical sync
			  1101: VGATE, programmable
			  1110: LSBs of the 9-Bit ADC's
			  1111: FID, position programmable
			*/
	__u8 RTSE1 : 4; /* see above */
} saa7108e_r12_t, saa7108e_output_control_t;

typedef struct {
	__u8 OFTS : 3; /*
			 select: output format selection
			 000: standard ITU 656 format
			 001: modified ITU 656 format, according to VGATE
			 010: YUV422 8Bit (no SAV/EAV)
			 011: reseved
			 100: multiplexed AD2/AD1 bypass (bits 8 to 1)
			 101: multiplexed AD2/AD1 bypass (bits 7 to 0)
			 110: reserved
			 111: multiplexed ADC MSB/LSB bypass
		       */

	__u8 HLSEL : 1; /*
			  select: selection of horizontal lock indicator
			  0: copy of inverted HLCK satus bit
			  1: fast lock indicator
			*/

	__u8 XRVS : 2; /*
			 select: X port XRV output
			 00: V123
			 01: ITU 656 related field ID
			 10: inverted V123
			 11: inverted ITU 656 related field ID
		       */

	__u8 XRHS : 1; /*
			 select: X port XRH output
			 0: HREF
			 1: HS programmable
		       */

	__u8 RTCE : 1; /*
			 select: RTC0 output enable
			 0: 3-state
			 1: enabled
		       */
} saa7108e_r13_t, saa7108e_xport_output_control_t;

typedef struct {
	__u8 APCK   : 2; /* application dependent: sample clock phase delay */
	__u8 OLDSB  : 1; /* bool: old status byte (compatibility with SAA7112) */
	__u8 XTOUTE : 1; /* bool: pin P4 (XTOUTd) enabled */

	__u8 AOSL   : 2; /*
			  select: AOUT connected to
			  00 - internal test point 1
			  01 - input AD1
			  02 - input AD2
			  03 - internal test point 2
			 */

	__u8 UPTCV  : 1; /*
			   select: update time interval for AGC value
			   0: horizontal (once per line)
			   1: vertical (once per field)
			 */

	__u8 CM99  : 1; /*
			  select: CM99 compatibility to SAA7199
			  0: off, default value
			  1: to be set only if SAA7199 present
			*/
} saa7108e_r14_t, saa7108e_analog_compatibility_control_t;

typedef struct {
	__u8 VSTA : 8;
} saa7108e_r15_t, saa7108e_VGATE_start_t;

typedef struct {
	__u8 VSTO : 8;
} saa7108e_r16_t, saa7108e_VGATE_stop_t;

typedef struct {
	__u8 VSTA_MSB : 1; /* bit: MSB of VSTA */
	__u8 VSTO_MSB : 1; /* bit: MSB of VSTO */
	__u8 VGPS     : 1; /*
			     select: alternative VGATE position
			     0: normal
			     1: occurs 1 line earlier during field 2
			   */
	__u8 _X_      : 3; /* reserved */
	__u8 LLC2E    : 1; /* inverse bool: LLC2 output enable */
	__u8 LLCE     : 1; /* inverse bool: LLC output enable */
} saa7108e_r17_t, saa7108e_VGATE_MSB_miscellaneous_t;

typedef struct {
	__u8 RAWG : 8; /* value: raw data gain 0..255 */
} saa7108e_r18_t, saa7108e_rawdata_gain_t;

typedef struct {
	__u8 RAWO : 8; /* signed value: raw data offset -128..+127 */
} saa7108e_r19_t, saa7108e_rawdata_offset_t;

/* reserved */
typedef __u8 saa7108e_r1a_t;
typedef __u8 saa7108e_r1b_t;
typedef __u8 saa7108e_r1c_t;
typedef __u8 saa7108e_r1d_t;
typedef __u8 saa7108e_r1e_t;

typedef struct {
	__u8 RDCAP : 1; /* bool: ready to capture (PLLs locked)			    */
	__u8 COPRO : 1; /* bool: colour signal in accordance with selected standard */
	__u8 WIPA  : 1; /* bool: white peak loop active				    */
	__u8 GLIMB : 1; /* bool: channel's luminance gain has a minimum limit	    */
	__u8 GLIMT : 1; /* bool: channel's luminance gain has a maximum limit	    */
	__u8 FIDT  : 1; /* flag: detected frequency: 0 = 50Hz / 1 = 60Hz	    */
	__u8 HLVLN : 1; /* bool: horiz & vert PLLs locked			    */
	__u8 INTL  : 1; /* bool: interlace detected				    */
} saa7108e_r1f_t, saa7108e_decoder_status_t;

/* reserved */
typedef __u8 saa7108e_r20_t;
typedef __u8 saa7108e_r21_t;
typedef __u8 saa7108e_r22_t;
typedef __u8 saa7108e_r23_t;
typedef __u8 saa7108e_r24_t;
typedef __u8 saa7108e_r25_t;
typedef __u8 saa7108e_r26_t;
typedef __u8 saa7108e_r27_t;
typedef __u8 saa7108e_r28_t;
typedef __u8 saa7108e_r29_t;
typedef __u8 saa7108e_r2a_t;
typedef __u8 saa7108e_r2b_t;
typedef __u8 saa7108e_r2c_t;
typedef __u8 saa7108e_r2d_t;
typedef __u8 saa7108e_r2e_t;
typedef __u8 saa7108e_r2f_t;

typedef struct {
	__u8 ACPF0 : 8; /* val: bit 0..7 of audio master clock (cycles per field) */
} saa7108e_r30_t, saa7108e_audio_masterclock_0_t;

typedef struct {
	__u8 ACPF8 : 8; /* val: bit 8..15 of audio master clock (cycles per field) */
} saa7108e_r31_t, saa7108e_audio_masterclock_1_t;

typedef struct {
	__u8 ACPF16 : 2; /* val: bit 16..17 of audio master clock (cycles per field) */
	__u8 _X_    : 6;
} saa7108e_r32_t, saa7108e_audio_masterclock_2_t;

/* reserved */
typedef __u8 saa7108e_r33_t;

typedef struct {
	__u8 ACNI0 : 8; /* val: bit 0..7 of audio master clock (nominal increment) */
} saa7108e_r34_t, saa7108e_audio_masterclock_inc_0_t;

typedef struct {
	__u8 ACNI8 : 8; /* val: bit 8..15 of audio master clock (nominal increment) */
} saa7108e_r35_t, saa7108e_audio_masterclock_inc_1_t;

typedef struct {
	__u8 ACNI16 : 6; /* val: bit 16..21 of audio master clock (nominal increment) */
	__u8 _X_    : 2;
} saa7108e_r36_t, saa7108e_audio_masterclock_inc_2_t;

/* reserved */
typedef __u8 saa7108e_r37_t;

typedef struct {
	__u8 SDIV : 6; /* val: clock ratio audio master clock to serial bit clock */
	__u8 _X_  : 2;
} saa7108e_r38_t, saa7108e_audio_ratio_master_serial_t;

typedef struct {
	__u8 LRDIV : 6; /* val: clock ratio serial bit clock to channel select clock */
	__u8 _X_   : 2;
} saa7108e_r39_t, saa7108e_audio_ratio_serial_channel_t;

typedef struct {
	__u8 SCPH : 1; /* 
			  select: ASCLK phase
			  0: triggered by falling edges of AMCLK
			  1: triggered by rising edges of AMCLK
		       */

	__u8 LRPH : 1; /*
			 select: ALRCLK phase
			 0: triggered by falling edges of ASCLK
			 1: triggered by riing edges of ASCLK
		       */

	__u8 AMVR : 1; /*
			 select: audio master clock vertical reference
			 0: taken from internal decoder
			 1: taken from XRV input (expansion port)
		       */

	__u8 APLL : 1; /*
			 select: audio PLL mode
			 0: PLL active, AMCLK is field locked
			 1: PLL open, AMCLK is free-running
		       */

	__u8 _X_  : 4;
} saa7108e_r3a_t, saa7108e_audio_clock_control_t;

/* reserved */
typedef __u8 saa7108e_r3b_t;
typedef __u8 saa7108e_r3c_t;
typedef __u8 saa7108e_r3d_t;
typedef __u8 saa7108e_r3e_t;
typedef __u8 saa7108e_r3f_t;

typedef struct {
	__u8 _XXXX_ : 4;
	__u8 HUNT_N : 1; /* bool: amplitude searching active (default: 0 active) */
	__u8 FCE    : 1; /* bool: framing code error not allowed */
	__u8 HAM_N  : 1; /* bool: hamming check (default: 0 on) */
	__u8 _X_    : 1;
} saa7108e_r40_t, saa7108e_slicer_control_1_t;

/*
  Line Control Registers LCR2..LCR24

  0000: WST625	  - teletext EuroWST, CCST
  0001: CC625	  - European Close Caption
  0010: VPS	  - video programming service
  0011: WSS	  - wide screen signalling bits
  0100: WST525	  - US teletext (WST)
  0101: CC525	  - US Closed Caption (line21)
  0110: TEST	  - video component signal, VBS region
  0111: INTERCAST - raw data
  1000: GENERAL	  - teletext
  1001: VITC625	  - VITC/EBU time codes (Europe)
  1010: VITC525	  - VITC/SMPTE time codes (USA)
  1011: reserved
  1100: NABTS	  - US NABTS
  1101: JAPTEXT	  - MOJI (Japanese)
  1110: JFS	  - Japanese format switch (L20/L22)
  1111: ACTIVEVID - video component signal, active video region (default)
*/

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r41_t, saa7108e_line_control_2_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r42_t, saa7108e_line_control_3_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r43_t, saa7108e_line_control_4_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r44_t, saa7108e_line_control_5_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r45_t, saa7108e_line_control_6_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r46_t, saa7108e_line_control_7_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r47_t, saa7108e_line_control_8_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r48_t, saa7108e_line_control_9_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r49_t, saa7108e_line_control_10_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r4a_t, saa7108e_line_control_11_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r4b_t, saa7108e_line_control_12_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r4c_t, saa7108e_line_control_13_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r4d_t, saa7108e_line_control_14_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r4e_t, saa7108e_line_control_15_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r4f_t, saa7108e_line_control_16_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r50_t, saa7108e_line_control_17_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r51_t, saa7108e_line_control_18_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r52_t, saa7108e_line_control_19_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r53_t, saa7108e_line_control_20_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r54_t, saa7108e_line_control_21_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r55_t, saa7108e_line_control_22_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r56_t, saa7108e_line_control_23_t;

typedef struct {
	__u8 LCR : 8; /* val: line control register */
} saa7108e_r57_t, saa7108e_line_control_24_t;

typedef struct {
	__u8 FC : 8; /* val: framing code for programmable data types (default: 0x40) */
} saa7108e_r58_t, saa7108e_framing_code_t;

typedef struct {
	__u8 HOFF : 8; /* val: horizontal offset for slicer (default: 0x47) */
} saa7108e_r59_t, saa7108e_slicer_horizontal_offset_t;

typedef struct {
	__u8 VOFF : 8; /* val: vertical offset for slicer */
} saa7108e_r5a_t, saa7108e_slicer_vertical_offset_t;

typedef struct {
	__u8 HOFF_MSB : 3; /* val: horizonal offset for slicer (default: 0x03) */
	__u8 _X_      : 1;
	__u8 VOFF_MSB : 1; /* val: vertical offset for slicer */
	__u8 _XX_     : 1;

	__u8 RECODE   : 1; /*
			     select: recode
			     0: let data unchanged (recommended)
			     1: convert 00->03 and FF->FC
			   */

	__u8 FOFF     : 1; /*
			     select: field offset
			     0: no modification (default for 50Hz/625 lines)
			     1: invert field indicator (default for 60Hz/525 lines)
			   */
} saa7108e_r5b_t, saa7108e_slicer_set_t;

/* reserved */
typedef __u8 saa7108e_r5c_t;

typedef struct {
	__u8 DID   : 6; /* 
			   val: data identification
			   000000: ANC header framing (default)
			   111110: = 0x3E SAV/EAV framing, FVREF = 1
			   111111: = 0x3F SAV/EAV framing, FVREF = 0
			*/

	__u8 _X_   : 1;
	__u8 FVREF : 1; /*
			  select: field ID and V-blank selection for text output
			  0: F and V output of slicer is LCR table dependent
			  1: F and V output taken from decoder real time signals
			  .  EVEN_ITU and VBLNK_ITU
			*/
} saa7108e_r5d_t, saa7108e_header_did_code_control_t;

typedef struct {
	__u8 SDID : 6; /* val: sliced data identification (default: 0x00) */
	__u8 _X_  : 2;
} saa7108e_r5e_t, saa7108e_slicer_data_identification_t;

/* reserved */
typedef __u8 saa7108e_r5f_t;

typedef struct {
	__u8 _X_  : 2;
	__u8 CCV  : 1; /* bool: closed caption valid */
	__u8 PPV  : 1; /* bool: PALplus detected */
	__u8 VPSV : 1; /* VPS detected */
	__u8 FC7V : 1; /* framing code with 1 error detected */
	__u8 FC8V : 1; /* framing code with 0 error detected */
	__u8 _XX_ : 1;
} saa7108e_r60_t, saa7108e_slicer_status_0_t;

typedef struct {
	__u8 LN4   : 5; /* val: bit 4..8 of line number */
	__u8 F21_N : 1; /* bool: field ID as seen by VBI slicer; for field 1 = 0 */
	__u8 _X_   : 2;
} saa7108e_r61_t, saa7108e_slicer_status_1_t;

typedef struct {
	__u8 DT	 : 4; /* val: data type; see LCR table */
	__u8 LN0 : 4; /* val: bit 0..3 of line number */
} saa7108e_r62_t, saa7108e_slicer_status_2_t;

typedef __u8 saa7108e_r63_t;
typedef __u8 saa7108e_r64_t;
typedef __u8 saa7108e_r65_t;
typedef __u8 saa7108e_r66_t;
typedef __u8 saa7108e_r67_t;
typedef __u8 saa7108e_r68_t;
typedef __u8 saa7108e_r69_t;
typedef __u8 saa7108e_r6a_t;
typedef __u8 saa7108e_r6b_t;
typedef __u8 saa7108e_r6c_t;
typedef __u8 saa7108e_r6d_t;
typedef __u8 saa7108e_r6e_t;
typedef __u8 saa7108e_r6f_t;
typedef __u8 saa7108e_r70_t;
typedef __u8 saa7108e_r71_t;
typedef __u8 saa7108e_r72_t;
typedef __u8 saa7108e_r73_t;
typedef __u8 saa7108e_r74_t;
typedef __u8 saa7108e_r75_t;
typedef __u8 saa7108e_r76_t;
typedef __u8 saa7108e_r77_t;
typedef __u8 saa7108e_r78_t;
typedef __u8 saa7108e_r79_t;
typedef __u8 saa7108e_r7a_t;
typedef __u8 saa7108e_r7b_t;
typedef __u8 saa7108e_r7c_t;
typedef __u8 saa7108e_r7d_t;
typedef __u8 saa7108e_r7e_t;
typedef __u8 saa7108e_r7f_t;

typedef struct {
	__u8 ICKS  : 4; /*
			  select: I port and scaler backend clock selection
			  XX00: ICLK output and backend clock is line-locked clock LLC from decoder
			  XX01: ICLK output and backend clock is XCLK from X port
			  XX10: ICLK output is LL and backend clock is LLC2 clock
			  XX11: backend clock is ICLK input
			  X0XX: IDQ pin carries the data quilifier
			  X1XX: IDQ pin carries a gated backend clock
			  0XXX: IDQ generation only for valid data
			  1XXX: IDQ qualifies valid data inside the scaling region and all data
			  .	outside the scaling region
			*/
	__u8 TEA   : 1; /* bool: enable task of register set A */
	__u8 TEB   : 1; /* bool: enable task of register set B */
	__u8 SMOD  : 1; /*
			  select: defintion of F and V timing of the scaler output
			  0: scaler window
			  1: VBI-data slicer
			*/
	__u8 _XX_ : 1;
} saa7108e_r80_t, saa7108e_global_control_t;

typedef __u8 saa7108e_r81_t;
typedef __u8 saa7108e_r82_t;

typedef struct {
	__u8 XPE  : 2; /* 
			  select: X port output enable
			  00: X port output is disabled by software
			  01: X port output is enabled by software
			  10: X port output is enabled by pin XTRI at logic 0
			  11: X port output is enabled by pin XTRI at logic 1
		       */
	__u8 XRQT : 1; /*
			 select: signal ready
			 0: XRDY output signal is A/B task flag from event handler (A=1)
			 1: XRDY output signal is ready signal from scaler path
		       */
	__u8 _X_  : 1;
	__u8 XPCK : 2; /*
			 select: output clock phase
			 00: default output phase, recommended
			 01: XCLK output inverted
			 10: XCLK phase shifted by about 3ns
			 11: XCLK output inverted and shifted by about 3ns
		       */
	__u8 _XX_ : 2;
} saa7108e_r83_t, saa7108e_xport_enable_output_clock_t;

typedef struct {
	__u8 IDH  : 2; /*
			 select: I port signal definition
			 00: IGPH is H-gate signal, framing the scaler output
			 01: IGPH is extended H-gate
			 10: IGPH is horizontal trigger, on active going edge of H-gate
			 11: IGPH is horizontal trigger, on active going edge of extended H-gate
		       */
	__u8 IDV  : 2; /*
			 select: I port signal definition
			 00: IGPV is V-gate signal, framing scaled output lines
			 01: IGPV is V-reference signal from scaler input
			 10: IGPV is vertical trigger, derived from V-gate
			 11: IGPV is vertical trigger, derived from input V-reference
		       */
	__u8 IDG1 : 2; /*
			 select: I port signal defintion (MSB in 0x86[4])
			 000: IGP1 is output field ID as defined by OFIDC
			 001: IGP1 is A/B task flag as defined by CONLH
			 010: IGP1 is sliced data flag, framing the sliced VBI-data at the I port
			 011: IGP1 is set to logic 0 (default polarity)
			 100: IGP1 is output FIFO almost empty flag
			 100: IGP1 is output FIFO overflow flag
			 100: IGP1 is output FIFO almost full flag, level programmable by 0x86
			 100: IGP1 is output FIFO almost empty flag, level programmable by 0x86
		       */
	__u8 IDG0 : 2; /*
			 select: I port signal defintion (MSB in 0x86[5])
			 000: IGP0 is output field ID as defined by OFIDC
			 001: IGP0 is A/B task flag as defined by CONLH
			 010: IGP0 is sliced data flag, framing the sliced VBI-data at the I port
			 011: IGP0 is set to logic 0 (default polarity)
			 100: IGP0 is output FIFO almost empty flag
			 100: IGP0 is output FIFO overflow flag
			 100: IGP0 is output FIFO almost full flag, level programmable by 0x86
			 100: IGP0 is output FIFO almost empty flag, level programmable by 0x86
		       */
} saa7108e_r84_t, saa7108e_iport_signal_definition_t;

typedef struct {
	__u8 IDQP : 1; /* bool: IDQ  polarity inverted */
	__u8 IRHP : 1; /* bool: IGPH polarity inverted */
	__u8 IRVP : 1; /* bool: IGPV polarity inverted */
	__u8 IG1P : 1; /* bool: IGP1 polarity inverted */
	__u8 IG0P : 1; /* bool: IGP0 polarity inverted */
	__u8 ILLV : 1; /*
			 select: video data limit
			 0: limited to 1..254
			 1: limited to 8..247
		       */
	__u8 ISWP : 2; /*
			 select: dword byte swap, influences serial output timing
			 00: D0 D1 D2 D3 = FF  00  00 SAV Cb0  Y0 Cr0  Y1
			 01: D1 D0 D3 D2 = 00  FF SAV  00  Y0 Cb0  Y1 Cr0
			 10: D2 D3 D0 D1 = 00 SAV  FF  00 Cr0  Y1 Cb0  Y0
			 11: D3 D2 D1 D0 = SAV 00  00  FF  Y1 Cr0  Y0 Cb0
		       */
} saa7108e_r85_t, saa7108e_iport_xport_signal_t;

typedef struct {
	__u8 FEL   : 2; /*
			  select: FIFO almost empty level
			  00: < 16 dwords
			  01: < 8 dwords
			  10: < 4 dwords
			  11: 0 dwords
			*/
	__u8 FFL   : 2; /*
			  select: FIFO almost full level
			  00: >= 16 dwords
			  01: >= 24 dwords
			  10: >= 28 dwords
			  11: 32 dwords
			*/
	__u8 IDG12 : 1; /* MSB of IDG1 (see 0x84) */
	__u8 IDG02 : 1; /* MSB of IDG0 (see 0x84) */
	__u8 VITX  : 2; /*
			  select: I port signal definition
			  00: I port data output inhibited
			  01: only video data are transferred
			  10: only text data are transferred (no EAV,SAV will occur)
			  11: text and video data are transferred, text has priority
			*/
} saa7108e_r86_t, saa7108e_iport_fifo_signal_t;

typedef struct {
	__u8 IPE  : 2; /* 
			  select: I port output enable
			  00: I port output is disabled by software
			  01: I port output is enabled by software
			  10: I port output is enabled by pin XTRI at logic 0
			  11: I port output is enabled by pin XTRI at logic 1
		       */
	__u8 _X_  : 2;
	__u8 IPCK : 4; /*
			 select: output clock phase
			 XX00: ICLK default output phase
			 XX01: ICLK phase shifted by 1/2 cycle + ca. 3ns (recommended for ICKS=XX10)
			 XX10: ICLK phase shifted by about 3ns
			 XX11: ICLK phase shifted by 1/2 cycle + ca. 3ns (recommended for ICKS=XX10)

			 only affects gated clock if ICKS=X1XX:
			 00XX: IDQ default output phase
			 01XX: IDQ phase shifted by 1/2 cycle (recommended)
			 10XX: IDQ phase shifted by about 3ns
			 11XX: IDQ phase shifted by 1/2 cycle (recommended, alternative)
		       */
} saa7108e_r87_t, saa7108e_iport_enable_output_clock_t;

typedef struct {
	__u8 SLM0  : 1; /* bool: decoder and VBI slicer power-down
			   .	 scaler only operates if scaler input and ICLK source is X port */
	__u8 SLM1  : 1; /* bool: scaler power-down (stops I port output) */
	__u8 _X_   : 1;
	__u8 SLM3  : 1; /* bool: audio clock generation power-down */
	__u8 DPROG : 1; /* bool: 0=after reset; 1=device programmed (PRDON status) */
	__u8 SWRST : 1; /* bool: scaler path operation */
	__u8 CH2EN : 1; /* bool: AD1x analog channel is active */
	__u8 CH4EN : 1; /* bool: AD2x analog channel is active */
} saa7108e_r88_t, saa7108e_power_save_control_t;

typedef __u8 saa7108e_r89_t;
typedef __u8 saa7108e_r8a_t;
typedef __u8 saa7108e_r8b_t;
typedef __u8 saa7108e_r8c_t;
typedef __u8 saa7108e_r8d_t;
typedef __u8 saa7108e_r8e_t;

typedef struct {
	__u8 FIDSCO : 1; /* bool: status of field sequence ID at scaler output */
	__u8 FIDSCI : 1; /* bool: status of field sequence ID at scaler input */
	__u8 ERROF  : 1; /* bool: error flag of scaler output formatter */
	__u8 PRDON  : 1; /* bool: copy of DPROG; detect power-down and start-up fails */
	__u8 FFOV   : 1; /* bool: FIFO overflow */
	__u8 FFIL   : 1; /* bool: FIFO almost filled */
	__u8 ITRI   : 1; /* bool: input pin ITRI */
	__u8 XTRI   : 1; /* bool: input pin XTRI */
} saa7108e_r8f_t, saa7108e_scaler_status_t;

typedef struct {
	__u8 STRC  : 2; /*
			  select: event handler trigger
			  00: immediately
			  01: with next V-sync
			  10: field ID = 0
			  11: field ID = 1
			*/
	__u8 RPTSK : 1; /* bool: repeat actice task once */
	__u8 FSKP  : 3; /* val: number of skipped fields before active task starts (0..7) */
	__u8 OFIDC : 1; /* 
			   select: output field
			   0: field ID from scaler input
			   1: task status flag, changes every time an selected task is
			   .  activated
			*/
	__u8 CONLH : 1; /* bool: scaler SAV/EAV byte bit D7 and task flag (0 = default) */
} saa7108e_r90_t, saa7108e_event_handler_control_t;

typedef struct {
	__u8 FSC0  : 1; /*
			  select: input scheme
			  0: Y-Cb-Cr 4:2:2
			  1: Y-Cb-Cr 4:1:1
			*/
	__u8 FSC1  : 2; /* val: chroma provided every n-th line (default: 0=every) */
	__u8 SCRQE : 1; /*
			  select: scaler input configuration
			  0: only if XRQT=1; scaler input source reacts on SAA7108E request
			  1: continous data stream (if SAA7108E is source or XRQT=0)
			*/
	__u8 SCSRC : 2; /*
			  select: scaler input source
			  00: from decoder
			  01: is Y-Cb-Cr data from X port
			  10: raw digital CVBS from selected analog input channel
			  11: raw digital CVBS from X port
			*/
	__u8 HLDFV : 1; /*
			  select: SAV/EAV code bits D6 and D5 (F and V)
			  0: may change between SAV and EAV
			  1: synchronized to scaler output line start
			*/
	__u8 CONLV : 1; /*
			  select: SAV/EAV code bit D5 (V) and V-gate
			  0: as generated by internal processing
			  1: inverted
			*/
} saa7108e_r91_t, saa7108e_xport_format_configuration_t;

typedef struct {
	__u8 XCKS  : 1; /*
			  select: clock selection
			  0: XCLK input clock and XDQ input quilifier are needed
			  1: data rate is defined by XCLK only, no XDQ signal used
			*/
	__u8 XDQ   : 1; /* bool: data rate quilified at XDQ input logic (0 = logic 1) */
	__u8 XDH   : 1; /*
			  select: XRH input horizontal reference
			  0: rising edge
			  1: falling edge
			*/
	__u8 XCODE : 1; /*
			  select: reference signal
			  0: taken from XRH and XRV
			  1: decoded from EAV and SAV
			*/
	__u8 XDV0  : 1; /*
			  select: XRV input and decoder V123 vertical reference
			  0: riding edge
			  1: falling edge
			*/
	__u8 XDV1  : 1; /*
			  select: XRV
			  0: V-sync or V-gate signal
			  1: frame sync, V-pulse are generated internally on 
			  .  both edges of FS input
			*/
	__u8 XFDH  : 1; /* bool: X port field ID polarity (1 = inverted) */
	__u8 XFDV  : 1; /*
			  select: reference edge for field detection
			  0: falling edge of XRV
			  1: rising edge of XRV
			*/
} saa7108e_r92_t, saa7108e_xport_input_signal_definition_t;

typedef struct {
	__u8 FSI   : 3; /*
			  select: I port output format
			  000: 4:2:2 dword formatting
			  001: 4:1:1 dword formatting
			  010: 4:2:0, only every 2nd line Y + Cb-Cr output, in between Y only
			  011: 4:1:0, only every 4th line Y + Cb-Cr output, in between Y only
			  100: Y only
			  other: not defined
			*/
	__u8 FOI   : 2; /* val: leading Y only lines before 1st Y + Cb-Cr line */
	__u8 FYSK  : 1; /* bool: skip leading Y lines as FOI, (0 = output all lines) */
	__u8 I8_16 : 1; /* bool: 16-bit transfer (0 = 8-bit) */
	__u8 ICODE : 1; /* bool: insert ITU 656 like SAV/EAV codes */
} saa7108e_r93_t, saa7108e_iport_format_configuration_t;

typedef struct {
	__u8 XO : 8; /* val: bit 0..7 of horizontal window start */
} saa7108e_r94_t, saa7108e_horizontal_input_window_start_1_t;

typedef struct {
	__u8 XO_MSB : 4; /* val: bit 8..11 of horizontal window start */
	__u8 _X_    : 4;
} saa7108e_r95_t, saa7108e_horizontal_input_window_start_2_t;

typedef struct {
	__u8 XS : 8; /* val: bit 0..7 of horizontal window length */
} saa7108e_r96_t, saa7108e_horizontal_input_window_length_1_t;

typedef struct {
	__u8 XS_MSB : 4; /* val: bit 8..11 of horizontal window length */
	__u8 _X_    : 4;
} saa7108e_r97_t, saa7108e_horizontal_input_window_length_2_t;

typedef struct {
	__u8 YO : 8; /* val: bit 0..7 of vertical window start */
} saa7108e_r98_t, saa7108e_vertical_input_window_start_1_t;

typedef struct {
	__u8 YO_MSB : 4; /* val: bit 8..11 of vertical window start */
	__u8 _X_    : 4;
} saa7108e_r99_t, saa7108e_vertical_input_window_start_2_t;

typedef struct {
	__u8 YS : 8; /* val: bit 0..7 of vertical window length */
} saa7108e_r9a_t, saa7108e_vertical_input_window_length_1_t;

typedef struct {
	__u8 YS_MSB : 4; /* val: bit 8..11 of vertical window length */
	__u8 _X_    : 4;
} saa7108e_r9b_t, saa7108e_vertical_input_window_length_2_t;

typedef struct {
	__u8 XD : 8; /* val: bit 0..7 of horizontal output pixels */
} saa7108e_r9c_t, saa7108e_horizontal_output_window_length_1_t;

typedef struct {
	__u8 XD_MSB : 4; /* val: bit 8..11 of horizontal output pixels */
	__u8 _X_    : 4;
} saa7108e_r9d_t, saa7108e_horizontal_output_window_length_2_t;

typedef struct {
	__u8 YD : 8; /* val: bit 0..7 of vertical output pixels */
} saa7108e_r9e_t, saa7108e_vertical_output_window_length_1_t;

typedef struct {
	__u8 YD_MSB : 4; /* val: bit 8..11 of vertical output pixels */
	__u8 _X_    : 4;
} saa7108e_r9f_t, saa7108e_vertical_output_window_length_2_t;

typedef struct {
	__u8 XPSC : 6; /* val: horizontal integer prescaling ratio - downscale 1/(1..63) */
	__u8 _X_  : 2;
} saa7108e_ra0_t, saa7108e_horizontal_prescale_ratio_t;

typedef struct {
	__u8 XACL : 6; /* val: horizontal prescale length (1..64) */
	__u8 _X_  : 2;
} saa7108e_ra1_t, saa7108e_horizontal_prescale_length_t;

typedef struct {
	__u8 XDCG  : 3; /* val: prescaler output renormalized by factor 1/(1..128) */
	__u8 XC2_1 : 1; /*
			  select: weighting of samples inside sequence
			  0: e.g. 1 + 1 + 1 + 1
			  1: e.g. 1 + 2 + 2 + 1
			*/
	__u8 PFY   : 2; /*
			  select: luminace FIR filter
			  00: bypassed
			  01: Y = (1 2 1) / 4
			  10: Y = (-1 1 1.75 4.5 1.75 1 -1) / 8
			  11: Y = (1 2 2 2 1) / 8
			*/
	__u8 PFUV  : 2; /*
			  select: chrominace FIR filer
			  00: bypassed
			  01: UV = (1 2 1) / 4
			  10: UV = (3 8 10 8 3) / 32
			  11: UV = (1 2 2 2 1) / 8
			*/
} saa7108e_ra2_t, saa7108e_prescale_dcgain_fir_control_t;

typedef __u8 saa7108e_ra3_t;

typedef struct {
	__u8 BRIG : 8; /*
			 val: prescale brightness
			 0: dark
			 128: nominal value
			 255: bright
		       */
} saa7108e_ra4_t, saa7108e_prescale_luminance_brightness_t;

typedef struct {
	__u8 CONT : 8; /* val: prescale contrast (0..127) / 64 */
} saa7108e_ra5_t, saa7108e_prescale_luminance_contrast_t;

typedef struct {
	__u8 SATN : 8; /* val: prescale saturation (0..127) / 64 */
} saa7108e_ra6_t, saa7108e_prescale_chrominance_saturation_t;

typedef __u8 saa7108e_ra7_t;

typedef struct {
	__u8 XSCY : 8; /* val: bit 0..7 of horizontal luminance scale increment 
			  .    scale = 1024 / 1..8191
		       */
} saa7108e_ra8_t, saa7108e_horizontal_luminance_scaler_1_t;

typedef struct {
	__u8 XSCY_MSB : 5; /* val: bit 8..12 of horizontal luminance scale increment 
			      .	   scale = 1024 / 1..8191
			   */
	__u8 _X_      : 3;
} saa7108e_ra9_t, saa7108e_horizontal_luminance_scaler_2_t;

typedef struct {
	__u8 XPHY : 8; /* val: horizontal luminance phase offset = x / 32 */
} saa7108e_raa_t, saa7108e_horizontal_luminance_phase_offset_t;

typedef __u8 saa7108e_rab_t;

typedef struct {
	__u8 XSCC : 8; /* val: horizontal chrominance scale increment (must be XSCY/2) */
} saa7108e_rac_t, saa7108e_horizontal_chrominance_scaler_1_t;

typedef struct {
	__u8 XSCC_MSB : 5; /* val: horizontal chrominance scale increment (must be XSCY/2) */
	__u8 _X_      : 3;
} saa7108e_rad_t, saa7108e_horizontal_chrominance_scaler_2_t;

typedef struct {
	__u8 XPHC : 8; /* horizontal chrominance phase offset (must be XPHY/2) */
} saa7108e_rae_t, saa7108e_horizontal_chrominance_phase_offset_t;

typedef __u8 saa7108e_raf_t;

typedef struct {
	__u8 YSCY : 8; /* val: bit 0..7 of vertical luminance scale increment 
			  .    scale = 1024 / 0..63999
		       */
} saa7108e_rb0_t, saa7108e_vertical_luminance_scaler_1_t;

typedef struct {
	__u8 YSCY_MSB : 8; /* val: bit 8..15 of vertical luminance scale increment 
			      .	   scale = 1024 / 0..63999
			   */
} saa7108e_rb1_t, saa7108e_vertical_luminance_scaler_2_t;

typedef struct {
	__u8 YSCC : 8; /* val: bit 0..7 of vertical chrominance scale increment 
			  .    must be YSCY
		       */
} saa7108e_rb2_t, saa7108e_vertical_chrominance_scaler_1_t;

typedef struct {
	__u8 YSCC_MSB : 8; /* val: bit 8..15 of vertical chrominance scale increment 
			      .	   must be YSCY
			   */
} saa7108e_rb3_t, saa7108e_vertical_chrominance_scaler_2_t;

typedef struct {
	__u8 YMODE : 1; /*
			  select: vertical scaling mode
			  0: linear interpolation between lines
			  1: higher order accumulation interpolation
			*/
	__u8 _X_   : 3;
	__u8 YMIR  : 1; /* bool: mirror lines */
	__u8 _XX_  : 3;
} saa7108e_rb4_t, saa7108e_vertical_scaler_mode_control_t;

typedef __u8 saa7108e_rb5_t;
typedef __u8 saa7108e_rb6_t;
typedef __u8 saa7108e_rb7_t;

typedef struct {
	__u8 YPC0 : 8; /* val: vertical chrominance phase offset (0..255) / 32 lines */
} saa7108e_rb8_t, saa7108e_vertical_chrominance_phase_offset_1_t;

typedef struct {
	__u8 YPC1 : 8; /* val: vertical chrominance phase offset (0..255) / 32 lines */
} saa7108e_rb9_t, saa7108e_vertical_chrominance_phase_offset_2_t;

typedef struct {
	__u8 YPC2 : 8; /* val: vertical chrominance phase offset (0..255) / 32 lines */
} saa7108e_rba_t, saa7108e_vertical_chrominance_phase_offset_3_t;

typedef struct {
	__u8 YPC3 : 8; /* val: vertical chrominance phase offset (0..255) / 32 lines */
} saa7108e_rbb_t, saa7108e_vertical_chrominance_phase_offset_4_t;

typedef struct {
	__u8 YPY0 : 8; /* val: vertical luminance phase offset (0..255) / 32 lines */
} saa7108e_rbc_t, saa7108e_vertical_luminance_phase_offset_1_t;

typedef struct {
	__u8 YPY1 : 8; /* val: vertical luminance phase offset (0..255) / 32 lines */
} saa7108e_rbd_t, saa7108e_vertical_luminance_phase_offset_2_t;

typedef struct {
	__u8 YPY2 : 8; /* val: vertical luminance phase offset (0..255) / 32 lines */
} saa7108e_rbe_t, saa7108e_vertical_luminance_phase_offset_3_t;

typedef struct {
	__u8 YPY3 : 8; /* val: vertical luminance phase offset (0..255) / 32 lines */
} saa7108e_rbf_t, saa7108e_vertical_luminance_phase_offset_4_t;

typedef saa7108e_r90_t saa7108e_rc0_t;
typedef saa7108e_r91_t saa7108e_rc1_t;
typedef saa7108e_r92_t saa7108e_rc2_t;
typedef saa7108e_r93_t saa7108e_rc3_t;
typedef saa7108e_r94_t saa7108e_rc4_t;
typedef saa7108e_r95_t saa7108e_rc5_t;
typedef saa7108e_r96_t saa7108e_rc6_t;
typedef saa7108e_r97_t saa7108e_rc7_t;
typedef saa7108e_r98_t saa7108e_rc8_t;
typedef saa7108e_r99_t saa7108e_rc9_t;
typedef saa7108e_r9a_t saa7108e_rca_t;
typedef saa7108e_r9b_t saa7108e_rcb_t;
typedef saa7108e_r9c_t saa7108e_rcc_t;
typedef saa7108e_r9d_t saa7108e_rcd_t;
typedef saa7108e_r9e_t saa7108e_rce_t;
typedef saa7108e_r9f_t saa7108e_rcf_t;
typedef saa7108e_ra0_t saa7108e_rd0_t;
typedef saa7108e_ra1_t saa7108e_rd1_t;
typedef saa7108e_ra2_t saa7108e_rd2_t;
typedef saa7108e_ra3_t saa7108e_rd3_t;
typedef saa7108e_ra4_t saa7108e_rd4_t;
typedef saa7108e_ra5_t saa7108e_rd5_t;
typedef saa7108e_ra6_t saa7108e_rd6_t;
typedef saa7108e_ra7_t saa7108e_rd7_t;
typedef saa7108e_ra8_t saa7108e_rd8_t;
typedef saa7108e_ra9_t saa7108e_rd9_t;
typedef saa7108e_raa_t saa7108e_rda_t;
typedef saa7108e_rab_t saa7108e_rdb_t;
typedef saa7108e_rac_t saa7108e_rdc_t;
typedef saa7108e_rad_t saa7108e_rdd_t;
typedef saa7108e_rae_t saa7108e_rde_t;
typedef saa7108e_raf_t saa7108e_rdf_t;
typedef saa7108e_rb0_t saa7108e_re0_t;
typedef saa7108e_rb1_t saa7108e_re1_t;
typedef saa7108e_rb2_t saa7108e_re2_t;
typedef saa7108e_rb3_t saa7108e_re3_t;
typedef saa7108e_rb4_t saa7108e_re4_t;
typedef saa7108e_rb5_t saa7108e_re5_t;
typedef saa7108e_rb6_t saa7108e_re6_t;
typedef saa7108e_rb7_t saa7108e_re7_t;
typedef saa7108e_rb8_t saa7108e_re8_t;
typedef saa7108e_rb9_t saa7108e_re9_t;
typedef saa7108e_rba_t saa7108e_rea_t;
typedef saa7108e_rbb_t saa7108e_reb_t;
typedef saa7108e_rbc_t saa7108e_rec_t;
typedef saa7108e_rbd_t saa7108e_red_t;
typedef saa7108e_rbe_t saa7108e_ree_t;
typedef saa7108e_rbf_t saa7108e_ref_t;

typedef __u8 saa7108e_rf0_t;
typedef __u8 saa7108e_rf1_t;
typedef __u8 saa7108e_rf2_t;
typedef __u8 saa7108e_rf3_t;
typedef __u8 saa7108e_rf4_t;
typedef __u8 saa7108e_rf5_t;
typedef __u8 saa7108e_rf6_t;
typedef __u8 saa7108e_rf7_t;
typedef __u8 saa7108e_rf8_t;
typedef __u8 saa7108e_rf9_t;
typedef __u8 saa7108e_rfa_t;
typedef __u8 saa7108e_rfb_t;
typedef __u8 saa7108e_rfc_t;
typedef __u8 saa7108e_rfd_t;
typedef __u8 saa7108e_rfe_t;
typedef __u8 saa7108e_rff_t;

typedef struct {
	saa7108e_r90_t event_handler;
	saa7108e_r91_t xport_format_config;
	saa7108e_r92_t xport_input_signal;
	saa7108e_r93_t iport_format_config;
	saa7108e_r94_t horizontal_iwindow_start_1;
	saa7108e_r95_t horizontal_iwindow_start_2;
	saa7108e_r96_t horizontal_iwindow_length_1;
	saa7108e_r97_t horizontal_iwindow_length_2;
	saa7108e_r98_t vertical_iwindow_start_1;
	saa7108e_r99_t vertical_iwindow_start_2;
	saa7108e_r9a_t vertical_iwindow_length_1;
	saa7108e_r9b_t vertical_iwindow_length_2;
	saa7108e_r9c_t horizontal_owindow_length_1;
	saa7108e_r9d_t horizontal_owindow_length_2;
	saa7108e_r9e_t vertical_owindow_length_1;
	saa7108e_r9f_t vertical_owindow_length_2;

	saa7108e_ra0_t horizontal_prescale_ratio;
	saa7108e_ra1_t horizontal_prescale_length;
	saa7108e_ra2_t prescale_dcgain_fir_control;
	saa7108e_ra3_t reserved_ra3;
	saa7108e_ra4_t luminance_brightness;
	saa7108e_ra5_t luminance_contrast;
	saa7108e_ra6_t prescale_chrominance_saturation;
	saa7108e_ra7_t reserved_ra7;
	saa7108e_ra8_t horizontal_luminance_scaler_1;
	saa7108e_ra9_t horizontal_luminance_scaler_2;
	saa7108e_raa_t horizontal_luminance_phase_offset;
	saa7108e_rab_t reserved_rab;
	saa7108e_rac_t horizontal_chrominance_scaler_1;
	saa7108e_rad_t horizontal_chrominance_scaler_2;
	saa7108e_rae_t horizontal_chrominance_phase_offset;
	saa7108e_raf_t reserved_raf;

	saa7108e_rb0_t vertical_luminance_scaler_1;
	saa7108e_rb1_t vertical_luminance_scaler_2;
	saa7108e_rb2_t vertical_chrominance_scaler_1;
	saa7108e_rb3_t vertical_chrominance_scaler_2;
	saa7108e_rb4_t vertical_scaler_mode_control;
	saa7108e_rb5_t reserved_rb5;
	saa7108e_rb6_t reserved_rb6;
	saa7108e_rb7_t reserved_rb7;
	saa7108e_rb8_t vertical_chrominance_phase_offset_1;
	saa7108e_rb9_t vertical_chrominance_phase_offset_2;
	saa7108e_rba_t vertical_chrominance_phase_offset_3;
	saa7108e_rbb_t vertical_chrominance_phase_offset_4;
	saa7108e_rbc_t vertical_luminance_phase_offset_1;
	saa7108e_rbd_t vertical_luminance_phase_offset_2;
	saa7108e_rbe_t vertical_luminance_phase_offset_3;
	saa7108e_rbf_t vertical_luminance_phase_offset_4;

} saa7108e_scaler_t;

typedef struct {
	/* video decoder */
	saa7108e_r00_t chip_version;
	saa7108e_r01_t increment_delay;
	saa7108e_r02_t analog_input_control_1;
	saa7108e_r03_t analog_input_control_2;
	saa7108e_r04_t analog_input_control_3;
	saa7108e_r05_t analog_input_control_4;
	saa7108e_r06_t horizontal_sync_start;
	saa7108e_r07_t horizontal_sync_stop;
	saa7108e_r08_t sync_control;
	saa7108e_r09_t luminance_control;
	saa7108e_r0a_t luminance_brightness;
	saa7108e_r0b_t luminance_contrast;
	saa7108e_r0c_t chrominance_saturation;
	saa7108e_r0d_t chrominance_hue_control;
	saa7108e_r0e_t chrominance_control;
	saa7108e_r0f_t chrominance_gain_control;
	saa7108e_r10_t chrominance_control_2;
	saa7108e_r11_t mode_delay_control;
	saa7108e_r12_t output_control;
	saa7108e_r13_t xport_output_control;
	saa7108e_r14_t analog_compatibility_control;
	saa7108e_r15_t VGATE_start;
	saa7108e_r16_t VGATE_stop;
	saa7108e_r17_t VGATE_MSB_miscellaneous;
	saa7108e_r18_t rawdata_gain;
	saa7108e_r19_t rawdata_offset;
	saa7108e_r1a_t reserved_r1a;
	saa7108e_r1b_t reserved_r1b;
	saa7108e_r1c_t reserved_r1c;
	saa7108e_r1d_t reserved_r1d;
	saa7108e_r1e_t reserved_r1e;
	saa7108e_r1f_t decoder_status;

	/* reserved */
	saa7108e_r20_t reserved_r20;
	saa7108e_r21_t reserved_r21;
	saa7108e_r22_t reserved_r22;
	saa7108e_r23_t reserved_r23;
	saa7108e_r24_t reserved_r24;
	saa7108e_r25_t reserved_r25;
	saa7108e_r26_t reserved_r26;
	saa7108e_r27_t reserved_r27;
	saa7108e_r28_t reserved_r28;
	saa7108e_r29_t reserved_r29;
	saa7108e_r2a_t reserved_r2a;
	saa7108e_r2b_t reserved_r2b;
	saa7108e_r2c_t reserved_r2c;
	saa7108e_r2d_t reserved_r2d;
	saa7108e_r2e_t reserved_r2e;
	saa7108e_r2f_t reserved_r2f;

	/* audio clock generator */
	saa7108e_r30_t audio_masterclock_0;
	saa7108e_r31_t audio_masterclock_1;
	saa7108e_r32_t audio_masterclock_2;
	saa7108e_r33_t reserved_r33;
	saa7108e_r34_t audio_masterclock_inc_0;
	saa7108e_r35_t audio_masterclock_inc_1;
	saa7108e_r36_t audio_masterclock_inc_2;
	saa7108e_r37_t reserved_r37;
	saa7108e_r38_t audio_ratio_master_serial;
	saa7108e_r39_t audio_ratio_serial_channel;
	saa7108e_r3a_t audio_clock_control;

	/* reserved */
	saa7108e_r3b_t reserved_r3b;
	saa7108e_r3c_t reserved_r3c;
	saa7108e_r3d_t reserved_r3d;
	saa7108e_r3e_t reserved_r3e;
	saa7108e_r3f_t reserved_r3f;

	/* slicer */
	saa7108e_r40_t slicer_control;
	saa7108e_r41_t line_control_2;
	saa7108e_r42_t line_control_3;
	saa7108e_r43_t line_control_4;
	saa7108e_r44_t line_control_5;
	saa7108e_r45_t line_control_6;
	saa7108e_r46_t line_control_7;
	saa7108e_r47_t line_control_8;
	saa7108e_r48_t line_control_9;
	saa7108e_r49_t line_control_10;
	saa7108e_r4a_t line_control_11;
	saa7108e_r4b_t line_control_12;
	saa7108e_r4c_t line_control_13;
	saa7108e_r4d_t line_control_14;
	saa7108e_r4e_t line_control_15;
	saa7108e_r4f_t line_control_16;
	saa7108e_r50_t line_control_17;
	saa7108e_r51_t line_control_18;
	saa7108e_r52_t line_control_19;
	saa7108e_r53_t line_control_20;
	saa7108e_r54_t line_control_21;
	saa7108e_r55_t line_control_22;
	saa7108e_r56_t line_control_23;
	saa7108e_r57_t line_control_24;
	saa7108e_r58_t framing_code;
	saa7108e_r59_t slicer_horizontal_offset;
	saa7108e_r5a_t slicer_vertical_offset;
	saa7108e_r5b_t slicer_set;
	saa7108e_r5c_t reserved_r5c;
	saa7108e_r5d_t header_did_code_control;
	saa7108e_r5e_t slicer_did;
	saa7108e_r5f_t reserved_r5f;
	saa7108e_r60_t slicer_status_0;
	saa7108e_r61_t slicer_status_1;
	saa7108e_r62_t slicer_status_2;
	saa7108e_r63_t reserved_r63;
	saa7108e_r64_t reserved_r64;
	saa7108e_r65_t reserved_r65;
	saa7108e_r66_t reserved_r66;
	saa7108e_r67_t reserved_r67;
	saa7108e_r68_t reserved_r68;
	saa7108e_r69_t reserved_r69;
	saa7108e_r6a_t reserved_r6a;
	saa7108e_r6b_t reserved_r6b;
	saa7108e_r6c_t reserved_r6c;
	saa7108e_r6d_t reserved_r6d;
	saa7108e_r6e_t reserved_r6e;
	saa7108e_r6f_t reserved_r6f;
	saa7108e_r70_t reserved_r70;
	saa7108e_r71_t reserved_r71;
	saa7108e_r72_t reserved_r72;
	saa7108e_r73_t reserved_r73;
	saa7108e_r74_t reserved_r74;
	saa7108e_r75_t reserved_r75;
	saa7108e_r76_t reserved_r76;
	saa7108e_r77_t reserved_r77;
	saa7108e_r78_t reserved_r78;
	saa7108e_r79_t reserved_r79;
	saa7108e_r7a_t reserved_r7a;
	saa7108e_r7b_t reserved_r7b;
	saa7108e_r7c_t reserved_r7c;
	saa7108e_r7d_t reserved_r7d;
	saa7108e_r7e_t reserved_r7e;
	saa7108e_r7f_t reserved_r7f;

	/* X port and I port */
	saa7108e_r80_t global_control;
	saa7108e_r81_t reserved_r81;
	saa7108e_r82_t reserved_r82;
	saa7108e_r83_t xport_enable_output_clock;
	saa7108e_r84_t iport_signal;
	saa7108e_r85_t iport_xport_signal;
	saa7108e_r86_t iport_fifo_signal;
	saa7108e_r87_t iport_enable_output_clock;
	saa7108e_r88_t power_save;
	saa7108e_r89_t reserved_r89;
	saa7108e_r8a_t reserved_r8a;
	saa7108e_r8b_t reserved_r8b;
	saa7108e_r8c_t reserved_r8c;
	saa7108e_r8d_t reserved_r8d;
	saa7108e_r8e_t reserved_r8e;
	saa7108e_r8f_t scaler_status;

	/* scaler: task A definitions */
	saa7108e_scaler_t taska;

	/* scaler: task B definitions */
	saa7108e_scaler_t taskb;

	/* reserved */
	saa7108e_rf0_t reserved_rf0;
	saa7108e_rf1_t reserved_rf1;
	saa7108e_rf2_t reserved_rf2;
	saa7108e_rf3_t reserved_rf3;
	saa7108e_rf4_t reserved_rf4;
	saa7108e_rf5_t reserved_rf5;
	saa7108e_rf6_t reserved_rf6;
	saa7108e_rf7_t reserved_rf7;
	saa7108e_rf8_t reserved_rf8;
	saa7108e_rf9_t reserved_rf9;
	saa7108e_rfa_t reserved_rfa;
	saa7108e_rfb_t reserved_rfb;
	saa7108e_rfc_t reserved_rfc;
	saa7108e_rfd_t reserved_rfd;
	saa7108e_rfe_t reserved_rfe;
	saa7108e_rff_t reserved_rff;
} saa7108e_reg_map_t;

#endif /* __SAA7108E_H */
