/* ------------------------------------------------------------------------- */
/*   saa7111a.c I2C driver for Philips SAA7111A video decoder chips	     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2000 Ferenc Bakonyi <fero@drama.obuda.kando.hu>
 *   saa7111_command() stolen from saa7111.c by Dave Perks <dperks@ibm.net>
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/videodev.h>
#include <linux/video_decoder.h>

#include "rivatv-kcompat.h"

/* Debug option */
static int debug = 0;

#define SAA7111A_PREFIX "SAA7111A: "
#define SAA7111A_PRINTK(format, args...)                     \
        printk (SAA7111A_PREFIX format, ## args)
#define SAA7111A_DPRINTK(format, args...) if (debug >= 1)    \
        printk (SAA7111A_PREFIX format, ## args)
#define SAA7111A_DPRINTK2(format, args...) if (debug >= 2)   \
        printk (SAA7111A_PREFIX format, ## args)
#undef  PRINTK
#define PRINTK(format, args...)                              \
        printk (format, ## args)
#undef  DPRINTK
#define DPRINTK(format, args...) if (debug >= 1)             \
        printk (format, ## args)
#undef  DPRINTK2
#define DPRINTK2(format, args...) if (debug >= 2)            \
        printk (format, ## args)


/*----------------------------------------------------------------------
 *	SAA7111A I2C 'driver' driver
 *----------------------------------------------------------------------*/

static int  saa7111a_attach  (struct i2c_adapter *);
static int  saa7111a_detach  (struct i2c_client *);
static int  saa7111a_command (struct i2c_client *, unsigned int, void *);

/* Possible SAA7111A addresses: 0x24, 0x25 */
static unsigned short normal_i2c[] = {0x24, 0x25, I2C_CLIENT_END};
static unsigned short normal_i2c_range[] = {I2C_CLIENT_END};

/* Magic definition of all other I2C variables and things */
I2C_CLIENT_INSMOD;

/* Number of SAA7111A registers */
#define NR_REGISTER	32

/* SAA7111A instance */
struct saa7111a {
	unsigned char reg[NR_REGISTER];	/* local copy of saa7111a's registers */
	int norm;
	int input;
	int enable;
	int bright;
	int contrast;
	int hue;
	int sat;
};

/* Initial mode (PAL, works well with Asus V3000TV) */
static unsigned char init_regs[NR_REGISTER] =
{
	0x00,	/* 00 - ID byte */
	0x00,	/* 01 - reserved */

/* front end */
	0xd3,	/* 02 - FUSE=3, GUDL=2, MODE=3 */
	0x23,	/* 03 - HLNRS=0, VBSL=1, WPOFF=0, HOLDG=0, GAFIX=0, GAI1=256, GAI2=256 */
	0x00,	/* 04 - GAI1=256 */
	0x00,	/* 05 - GAI2=256 */

/* decoder */
	0x6c,	/* 06 - HSB=108(50Hz) */
	0x6c,	/* 07 - HSS=108(50Hz) */
	0x08,	/* 08 - AUFD=0, FSEL=0, EXFIL=0, VTRC=1, HPLL=0, VNOI=0 */
	0x00,	/* 09 - BYPS=0, PREF=0, BPSS=0, VBLB=0, UPTCV=0, APER=0 */
	0x80,	/* 0a - BRIG=128 */
	0x47,	/* 0b - CONT=1.109 */
	0x40,	/* 0c - SATN=1.0 */
	0x00,	/* 0d - HUE=0 */
	0x01,	/* 0e - CDTO=0, CSTD=0, DCCF=0, FCTC=0, CHBW=1 */
	0x00,	/* 0f - reserved */
	0xc8,	/* 10 - OFTS=3, HDEL=0, VRLN=1, YDEL=0 */
	0x0c,	/* 11 - GPSW=0, CM99=0, FECO=0, COMPO=0, OEYC=1, OEHV=1, VIPB=0, COLO=0 */
	0x80,	/* 12 - output control 2 */
	0x00,	/* 13 - output control 3 */
	0x00,	/* 14 - reserved */
	0x00,	/* 15 - VBI */
	0x00,	/* 16 - VBI */
	0x00,	/* 17 - VBI */
	0x00,   /* 18 - reserved */
	0x00,   /* 19 - reserved */
	0x00,   /* 1a - text slicer status */
	0x00,   /* 1b - text slicer byte 1 */
	0x00,   /* 1c - text slicer byte 2 */
	0x00,   /* 1d - reserved */
	0x00,   /* 1e - reserved */
	0x00    /* 1f - status byte */
};

/* For registering the I2C 'driver' driver */
static struct i2c_driver i2c_driver_saa7111a = {
	.name		= "SAA7111A",
	.id		= I2C_DRIVERID_SAA7111A,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter = saa7111a_attach,
	.detach_client	= saa7111a_detach,
	.command	= saa7111a_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use	= i2c_client_inc_use,
	.dec_use	= i2c_client_dec_use,
#else
	.owner		= THIS_MODULE,
#endif
};

/* Template for new clients */
static struct i2c_client saa7111a_client = {
	I2C_DEVNAME ("Philips SAA7111A"),
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	.id	 = -1,
#endif
	.flags	 = I2C_CLIENT_ALLOW_USE,
	.addr    = 0,
	.adapter = NULL,
	.driver  = &i2c_driver_saa7111a,
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
/* Unique ID allocation */
static int saa7111a_id = 0;
#endif

/* Helper functions */

/* Read from a register */
static int saa7111a_read (struct i2c_client *client, unsigned char subaddr)
{
	int val = i2c_smbus_read_byte_data (client, subaddr);
	struct saa7111a *dev = (struct saa7111a *) i2c_get_clientdata (client);

	if (val < 0)
		SAA7111A_PRINTK ("failed reading register 0x%02x\n", subaddr);
	else
		dev->reg[subaddr] = val;
	return val;
}

/* Write to a register */
static int saa7111a_write (struct i2c_client *client, unsigned char subaddr, unsigned char data)
{
	int val = i2c_smbus_write_byte_data (client, subaddr, data);
	struct saa7111a *dev = (struct saa7111a *) i2c_get_clientdata (client);

	if (val >= 0)
		dev->reg[subaddr] = data;
	return val;
}

/* Read all registers */
static int saa7111a_read_block (struct i2c_client *client, unsigned char *data)
{
	int i;

	for (i = 0 ; i < NR_REGISTER ; i++)
		data[i] = i2c_smbus_read_byte_data (client, i);
	return NR_REGISTER;
}

/* Write to a register block */
static int saa7111a_write_block (struct i2c_client *client, unsigned char *data, unsigned int len)
{
	struct saa7111a *dev = (struct saa7111a *) i2c_get_clientdata (client);
	int i, err;

	if (len > NR_REGISTER) len = NR_REGISTER;
	for (i = 0 ; i < len ; i++) {
		if ((err = i2c_smbus_write_byte_data (client, i, data[i])))
			return err;
    		dev->reg[i] = data[i];
	}
	return 0;
}

/* I2C driver functions */

/* Called when a 'SAA7111A like' device found */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
static int saa7111a_detect (struct i2c_adapter *adap, int addr, unsigned short flags, int kind)
#else
static int saa7111a_detect (struct i2c_adapter *adap, int addr, int kind)
#endif
{
	int err = 0;
	struct i2c_client *client;
	struct saa7111a *decoder;
	int version;

	if (!i2c_check_functionality (adap, I2C_FUNC_SMBUS_READ_BYTE_DATA | 
				      I2C_FUNC_SMBUS_WRITE_BYTE_DATA))
		goto err_out;

	client = kmalloc (sizeof (*client), GFP_KERNEL);
	if (client == NULL) {
		err = -ENOMEM;
		goto err_out;
	}
	
	memcpy (client, &saa7111a_client, sizeof (*client));
	client->adapter = adap;
        client->addr = addr;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
        client->id = saa7111a_id++;
#endif

	if (kind < 0) {
		if (i2c_smbus_write_byte_data (client, 0x00, 0x00) == -1)
			goto err_out_kfree_client;
		version = i2c_smbus_read_byte_data (client, 0x00);
		if (version == -1)
			goto err_out_kfree_client;
		if ((version & 0xf0) == 0x20 || (version & 0xf0) == 0x10) {
			SAA7111A_PRINTK ("video decoder chip found, chip version: %#x\n", version);
		} else {
			SAA7111A_PRINTK ("unknown SAA711x chip found, chip version: %#x\n", version);
			goto err_out_kfree_client;
		}
	} else {
		SAA7111A_PRINTK ("detection skipped\n");
	}
	
	decoder = kmalloc (sizeof (struct saa7111a), GFP_KERNEL);
	if (decoder == NULL) {
		err = -ENOMEM;
		goto err_out_kfree_client;
	}

	memset (decoder, 0, sizeof (struct saa7111a));
	decoder->norm = VIDEO_MODE_PAL;
	decoder->input = 3;
	decoder->enable = 1;
	decoder->bright = 32768;
	decoder->contrast = 32768;
	decoder->hue = 32768;
	decoder->sat = 32768;
	i2c_set_clientdata (client, decoder);

	if (saa7111a_write_block (client, init_regs, sizeof (init_regs)))
		goto err_out_kfree_decoder;

	if ((err = i2c_attach_client (client)))
		goto err_out_kfree_decoder;

	return 0;
	
err_out_kfree_decoder:
	kfree (decoder);
err_out_kfree_client:
	kfree (client);
err_out:
	return err;
}

/* Called when a new I2C bus found */
static int saa7111a_attach (struct i2c_adapter *adap)
{
	return i2c_probe (adap, &addr_data, saa7111a_detect);
}

/* Called on exit */
static int saa7111a_detach (struct i2c_client *client)
{
	int err;

	if ((err = i2c_detach_client (client)))
		return err;
	kfree (i2c_get_clientdata (client));
	kfree (client);
	return 0;
}

/* Register dump */
static void saa7111a_dump_regs (struct i2c_client *client)
{
	int i, j;

	if (debug >= 2) {
		for (i = 0; i < NR_REGISTER; i += 16) {
			SAA7111A_DPRINTK2 ("%02x |", i);
			for (j = 0; j < 16 && i + j < NR_REGISTER; j++)
				DPRINTK2 (" %02x", saa7111a_read (client, i + j));
			DPRINTK2 ("\n");
		}
	}
}

typedef struct {
	/* colour standard identifier */
	int standard;
	/* textual colour standard description */
	char *name;
} saa7111a_standard_t;

#define SAA7111A_CSTD_PALB   0
#define SAA7111A_CSTD_NTSCM  0
#define SAA7111A_CSTD_NTSC50 1
#define SAA7111A_CSTD_PAL60  1
#define SAA7111A_CSTD_PALN   2
#define SAA7111A_CSTD_NTSC60 2
#define SAA7111A_CSTD_NTSCN  3
#define SAA7111A_CSTD_PALM   3
#define SAA7111A_CSTD_SECAM  5

static saa7111a_standard_t saa7111a_standard[2][8] = {
	/* 50 Hz standards */
	{ { SAA7111A_CSTD_PALB,   "PAL-B"           },
	  { SAA7111A_CSTD_NTSC50, "NTSC-4.43(50Hz)" },
	  { SAA7111A_CSTD_PALN,   "PAL-N"           },
	  { SAA7111A_CSTD_NTSCN,  "NTSC-N"          },
	  { -1,                   "<Invalid 50Hz>"  },
	  { SAA7111A_CSTD_SECAM,  "SECAM"           },
	  { -1,                   "<Invalid 50Hz>"  },
	  { -1,                   "<Invalid 50Hz>"  } },

	/* 60 Hz standards */
	{ { SAA7111A_CSTD_NTSCM,  "NTSC-M"          },
	  { SAA7111A_CSTD_PAL60,  "PAL-4.43(60Hz)"  },
	  { SAA7111A_CSTD_NTSC60, "NTSC-4.43(60Hz)" },
	  { SAA7111A_CSTD_PALM,   "PAL-M"           },
	  { -1,                   "<Invalid 60Hz>"  },
	  { -1,                   "<Invalid 60Hz>"  },
	  { -1,                   "<Invalid 60Hz>"  },
	  { -1,                   "<Invalid 60Hz>"  } }
};

/* Autodetection of colour standard */
static void saa7111a_detection (struct i2c_client *client)
{
	struct saa7111a *decoder = (struct saa7111a *) i2c_get_clientdata (client);
	int n, idx, old, detected, reg;

	/* detect 50/60Hz */
	saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0x3f) | 0x80);

	/* stay here a little eternity */
	set_current_state (TASK_UNINTERRUPTIBLE);
	schedule_timeout (HZ/10);
	set_current_state (TASK_RUNNING);

	reg = saa7111a_read (client, 0x1f);

	/* set explicitely the detected frequency */
	idx = (reg & 0x20) ? 1 : 0;
	if (idx) {
		saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0x3f) | 0x40);
		DPRINTK (" 60Hz");
	}
	else {
		saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0x3f) | 0x00);
		DPRINTK (" 50Hz");
	}

	/* save old colour standard */
	old = saa7111a_read (client, 0x0e);

	for (detected = n = 0; n < 8; n++) {
		if (saa7111a_standard[idx][n].standard != -1) {
			/* set colour standard */
			saa7111a_write (client, 0x0e, (decoder->reg[0x0e] & 0x8f) | 
					(saa7111a_standard[idx][n].standard << 4));

			/* stay here a little eternity */
			set_current_state (TASK_UNINTERRUPTIBLE);
			schedule_timeout (HZ/10);
			set_current_state (TASK_RUNNING);

			/* check status byte */
			reg = saa7111a_read (client, 0x1f);
			if (reg & 0x01) {
				DPRINTK (" %s", saa7111a_standard[idx][n].name);
				detected++;
				break;
			}
		}
	}
  
	/* restore old colour standard */
	if (!detected)
		saa7111a_write (client, 0x0e, old);
}

/* Interface to the world */
static int saa7111a_command (struct i2c_client *client,
			     unsigned int cmd, void *arg)
{
	struct saa7111a *decoder = (struct saa7111a *) i2c_get_clientdata (client);

	switch (cmd) {
#ifdef READ_REGISTERS
	case READ_REGISTERS: /* for debugging */
	{
		SAA7111A_DPRINTK ("READ_REGISTERS\n");
		return saa7111a_read_block (client, arg); /* bytes read */
	}
	case WRITE_REGISTERS: /* for debugging */
	{
		SAA7111A_DPRINTK ("WRITE_REGISTERS\n");
		saa7111a_write_block (client, arg, NR_REGISTER);
		return NR_REGISTER;
	}
	case GET_NR_OF_REGISTERS: /* for debugging */
	{
		SAA7111A_DPRINTK ("GET_NR_OF_REGISTERS\n");
		return NR_REGISTER;
	}
#endif /* READ_REGISTERS */
#ifdef DECODER_DUMP
	case DECODER_DUMP:
	{
		SAA7111A_DPRINTK ("DECODER_DUMP\n");
		saa7111a_dump_regs (client);
		break;
	}
#endif /* DECODER_DUMP */
#ifdef WRITE_REGISTER
	case WRITE_REGISTER:
	{
		u16 *sarg = arg;
		SAA7111A_DPRINTK ("WRITE_REGISTER\n");
		saa7111a_write (client, *sarg >> 8, *sarg & 0xff);
		break;
	}
#endif /* WRITE_REGISTER */

	case DECODER_GET_CAPABILITIES:
	{
		struct video_decoder_capability *cap = arg;

		SAA7111A_DPRINTK ("DECODER_GET_CAPABILITIES\n");
		cap->flags
			= VIDEO_DECODER_PAL
			| VIDEO_DECODER_NTSC
			| VIDEO_DECODER_AUTO
			| VIDEO_DECODER_SECAM
			| VIDEO_DECODER_CCIR;
		cap->inputs = 8;
		cap->outputs = 1;
		break;
	}

	case DECODER_GET_STATUS:
	{
		int *iarg = arg;
		int status;
		int res = 0;

		status = saa7111a_read (client, 0x1f);
		SAA7111A_DPRINTK ("DECODER_GET_STATUS = 0x%02x ->", status);
		if ((status & (1 << 6)) == 0) {
			res |= DECODER_STATUS_GOOD;
			DPRINTK (" GOOD");
		}
		else {
			DPRINTK (" BAD");
		}
		switch (decoder->norm) {
		case VIDEO_MODE_NTSC:
			res |= DECODER_STATUS_NTSC;
			DPRINTK (" NTSC");
			break;
		case VIDEO_MODE_PAL:
			res |= DECODER_STATUS_PAL;
			DPRINTK (" PAL");
			break;
		case VIDEO_MODE_SECAM:
			res |= DECODER_STATUS_SECAM;
			DPRINTK (" SECAM");
			break;
		default:
		case VIDEO_MODE_AUTO:
			if ((status & (1 << 5)) != 0) {
				res |= DECODER_STATUS_NTSC;
				DPRINTK (" NTSC");
			} else {
				res |= DECODER_STATUS_PAL;
				DPRINTK (" PAL");
			}
			break;
		}
		if ((status & (1 << 0)) != 0) {
			res |= DECODER_STATUS_COLOR;
			DPRINTK (" COLOR");
		}
		DPRINTK ("\n");
		*iarg = res;
		break;
	}

	case DECODER_SET_NORM:
	{
		int *iarg = arg, status;

		SAA7111A_DPRINTK ("DECODER_SET_NORM: ");
		switch (*iarg) {

		case VIDEO_MODE_NTSC:
			DPRINTK ("NTSC");
			saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0x3f) | 0x40);
			saa7111a_write (client, 0x0e, (decoder->reg[0x0e] & 0x8f) | SAA7111A_CSTD_NTSCM << 4);
			break;

		case VIDEO_MODE_PAL:
			DPRINTK ("PAL");
			saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0x3f) | 0x00);
			saa7111a_write (client, 0x0e, (decoder->reg[0x0e] & 0x8f) | SAA7111A_CSTD_PALB << 4);
			break;

		case VIDEO_MODE_SECAM:
			DPRINTK ("SECAM");
			saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0x3f) | 0x00);
			saa7111a_write (client, 0x0e, (decoder->reg[0x0e] & 0x8f) | SAA7111A_CSTD_SECAM << 4);
			break;

		case VIDEO_MODE_AUTO:
			DPRINTK ("AUTO");
			saa7111a_detection (client);
			break;

		default:
			DPRINTK ("<invalid value %d>\n", *iarg);
			return -EINVAL;

		}
		DPRINTK ("\n");

		/* setup 50/60 Hz values */
		status = saa7111a_read (client, 0x1f);
		if (status & 0x20) {
			/* 60 Hz */
			saa7111a_write (client, 0x06, 0x6b);
			saa7111a_write (client, 0x07, 0x6b);
			saa7111a_write (client, 0x10, (decoder->reg[0x10] & (~0x04)) | 0x00);
		} else {
			/* 50 Hz */
			saa7111a_write (client, 0x06, 0x6c);
			saa7111a_write (client, 0x07, 0x6c);
			saa7111a_write (client, 0x10, (decoder->reg[0x10] & (~0x04)) | 0x04);
		}

		decoder->norm = *iarg;
		saa7111a_dump_regs (client);
		break;
	}

	case DECODER_SET_INPUT:
	{
		int *iarg = arg;

		SAA7111A_DPRINTK ("DECODER_SET_INPUT: %d\n", *iarg);
		if (*iarg < 0 || *iarg > 7) {
			return -EINVAL;
		}
		if (decoder->input != *iarg) {
			decoder->input = *iarg;
			/* select mode */
			saa7111a_write (client, 0x02, (decoder->reg[0x02] & 0xf8) | decoder->input);
			/* bypass chrominance trap for modes 4..7 */
			saa7111a_write (client, 0x09, (decoder->reg[0x09] & 0x7f) | ((decoder->input > 3) ? 0x80 : 0));
		}
		saa7111a_dump_regs (client);
		break;
	}

	case DECODER_SET_OUTPUT:
	{
		int *iarg = arg;

		SAA7111A_DPRINTK ("DECODER_SET_OUTPUT: %d\n", *iarg);
		if (*iarg != 0) {
			return -EINVAL;
		}
		break;
	}

	case DECODER_ENABLE_OUTPUT:
	{
		int *iarg = arg;
		int enable = (*iarg != 0);

		SAA7111A_DPRINTK ("DECODER_ENABLE_OUTPUT: %d\n", *iarg);
		decoder->enable = enable;

		if (decoder->enable) {
			/* set input mode */
			saa7111a_write (client, 0x02, (decoder->reg[0x02] & 0xf8) | decoder->input);
			/* HPLL closed */
			saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0xfb));
			/* output enable horizontal/vertical sync */
			saa7111a_write (client, 0x11, (decoder->reg[0x11] & 0xf3) | 0x0c);

			/* enable output */
			saa7111a_write (client, 0x12, 0x80);

			/* output format */
			switch (enable) {
			case VIDEO_PALETTE_YUV422:
			case VIDEO_PALETTE_YUYV:
				saa7111a_write (client, 0x10, (decoder->reg[0x10] & 0x3f) | 0x40);
				break;
			case VIDEO_PALETTE_YUV411:
				saa7111a_write (client, 0x10, (decoder->reg[0x10] & 0x3f) | 0x80);
				break;
			case VIDEO_PALETTE_RGB565:
				saa7111a_write (client, 0x10, (decoder->reg[0x10] & 0x3f) | 0x00);
				saa7111a_write (client, 0x12, (decoder->reg[0x12] & 0xf7) | 0x00);
				break;
			case VIDEO_PALETTE_RGB24:
				saa7111a_write (client, 0x10, (decoder->reg[0x10] & 0x3f) | 0x00);
				saa7111a_write (client, 0x12, (decoder->reg[0x12] & 0xf7) | 0x08);
				break;
			default:
			case VIDEO_PALETTE_UYVY:
				saa7111a_write (client, 0x10, (decoder->reg[0x10] & 0x3f) | 0xc0);
				break;
			}
		} else {
			/* reset input mode */
			saa7111a_write (client, 0x02, (decoder->reg[0x02] & 0xf8));
			/* HPLL closed */
			saa7111a_write (client, 0x08, (decoder->reg[0x08] & 0xfb) | 0x04);
			/* output enable horizontal/vertical sync */
			saa7111a_write (client, 0x11, (decoder->reg[0x11] & 0xf3));
		}

		saa7111a_dump_regs (client);
		break;
	}

	case DECODER_SET_PICTURE:
	{
		struct video_picture *pic = arg;

		SAA7111A_DPRINTK ("DECODER_SET_PICTURE\n");
		if (decoder->bright != pic->brightness) {
			/* We want 0 to 255 we get 0-65535 */
			decoder->bright = pic->brightness;
			saa7111a_write (client, 0x0a, decoder->bright >> 8);
		}
		if (decoder->contrast != pic->contrast) {
			/* We want 0 to 127 we get 0-65535 */
			decoder->contrast = pic->contrast;
			saa7111a_write (client, 0x0b, decoder->contrast >> 9);
		}
		if (decoder->sat != pic->colour) {
			/* We want 0 to 127 we get 0-65535 */
			decoder->sat = pic->colour;
			saa7111a_write (client, 0x0c, decoder->sat >> 9);
		}
		if (decoder->hue != pic->hue) {
			/* We want -128 to 127 we get 0-65535 */
			decoder->hue = pic->hue;
			saa7111a_write (client, 0x0d, (decoder->hue - 32768) >> 8);
		}
		saa7111a_dump_regs (client);
		break;
	}

	default:
		SAA7111A_DPRINTK ("<invalid function 0x%08X>\n", cmd);
		return -EINVAL;
	}

	return 0;
}


/*---------------------------------------------------------------------- 
 *	Modularization
 *----------------------------------------------------------------------*/

MODULE_AUTHOR      ("Ferenc Bakonyi <fero@drama.obuda.kando.hu>");
MODULE_DESCRIPTION ("SAA7111A driver");
MODULE_LICENSE     ("GPL");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param       (debug, int, 0644);
#else
MODULE_PARM	   (debug, "i");
#endif
MODULE_PARM_DESC   (debug, "Debug level: 0 = silent, 1 = verbose, 2 = very verbose (default = 0).");

int __init init_saa7111a (void) 
{
	return i2c_add_driver (&i2c_driver_saa7111a);
}

void __exit cleanup_saa7111a (void) 
{
	i2c_del_driver (&i2c_driver_saa7111a);
}

module_init (init_saa7111a);
module_exit (cleanup_saa7111a);
