/* ------------------------------------------------------------------------- */
/*   saa7174hl.c I2C driver for Philips SAA7174HL video decoder chips	     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2004, 2005 Stefan Jahn <stefan@lkcc.org>	     
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>

#include "rivatv-kcompat.h"
#include "saa7174hl.h"

#define SAA7174HL_I2C_VERSION_ID 0x7174
#define SAA7173HL_I2C_VERSION_ID 0x7173
#define SAA7134HL_I2C_VERSION_ID 0x7134
#define SAA7133HL_I2C_VERSION_ID 0x7133

/* Debug option */
static int debug = 1;

#define SAA7174HL_PREFIX SAA7174HL_DEVICE_NAME ": "
#define SAA7174HL_PRINTK(format, args...)			 \
	printk (SAA7174HL_PREFIX format, ## args)
#define SAA7174HL_DEBUG_PRINTK(format, args...) if (debug >= 1)	 \
	printk (SAA7174HL_PREFIX format, ## args)
#define SAA7174HL_DEBUG_PRINTK2(format, args...) if (debug >= 2) \
	printk (SAA7174HL_PREFIX format, ## args)
#undef	PRINTK
#define PRINTK(format, args...)					 \
	printk (format, ## args)
#undef	DPRINTK
#define DPRINTK(format, args...) if (debug >= 1)		 \
	printk (format, ## args)
#undef	DPRINTK2
#define DPRINTK2(format, args...) if (debug >= 2)		 \
	printk (format, ## args)

#define SAA7174HL_REG_ADDR(client, ptr) \
	((ulong) ptr - (ulong) &((saa7174hl_t *) i2c_get_clientdata (client))->reg.field)

#define SAA7174HL_BYTEVAL(a) (*((__u8 *)&(a)))

/*----------------------------------------------------------------------
 *	SAA7174HL I2C 'driver' driver
 *----------------------------------------------------------------------*/

static int saa7174hl_attach   (struct i2c_adapter *adapter);
static int saa7174hl_detach   (struct i2c_client *client);
static int saa7174hl_command  (struct i2c_client *client, 
			       unsigned int cmd, void *arg);

/* Possible SAA7174HL address: 0x21 only */
static __u16 normal_i2c[] = { 0x21, I2C_CLIENT_END };
static __u16 normal_i2c_range[] = { I2C_CLIENT_END };

/* Magic definition of all other I2C variables and things */
I2C_CLIENT_INSMOD;

/* SAA7174HL instance */
typedef struct {
	union {
		saa7174hl_reg_map_t field;
		u8		    byte[SAA7174HL_NR_REGISTER];
	} reg;
	__u16 bright;
	__u16 contrast;
	__u16 sat;
	__u16 hue;
} saa7174hl_t;

/* Collection of initial registers. */
static __u32 saa7174hl_init_bytes[] = {
	0x000, 0x00001005,
	0x004, 0x30010003,
	0x00c, 0x00000000,
	0x010, 0x40302010,
	0x014, 0x80706050,
	0x018, 0xc0b0a090,
	0x01c, 0xfff0e0d0,
	0x020, 0x40302010,
	0x024, 0x80706050,
	0x028, 0xc0b0a090,
	0x02c, 0xfff0e0d0,
	0x030, 0x40302010,
	0x034, 0x80706050,
	0x038, 0xc0b0a090,
	0x03c, 0xfff0e0d0,
	0x040, 0x0083010f,
	0x044, 0x06270088,	
	0x048, 0x00130007,
	0x04c, 0x000d0608,

	0x054, 0x0359008a,
	0x058, 0x01050016,
	0x05c, 0x00f002d0,
	0x060, 0x00010101,
	0x064, 0x00404080,
	0x068, 0x000003cf,
	0x06c, 0x00000400,
	0x070, 0x00000400,
	0x074, 0x00000000,

	0x080, 0x0003010f,
	0x084, 0x02cf0000,
	0x088, 0x000b0000,
	0x08c, 0x000c02d0,

	0x094, 0x02cf0000,
	0x098, 0x00ff000e,
	0x09c, 0x00f002d0,
	0x0a0, 0x00000001,
	0x0a4, 0x00404080,
	0x0a8, 0x00000400,
	0x0ac, 0x00000400,
	0x0b0, 0x00000400,
	0x0b4, 0x00000000,

	0x104, 0xe0eb9090,
	0x108, 0x448040b8,
	0x110, 0x00000000,
	0x114, 0x18fe1101,
	0x118, 0x00778040,
	0x11c, 0xb7a50000,

	0x15c, 0xb7b7b7b7,

	0x190, 0xac00b090,

	0x198, 0x00000000,

	0x1a0, 0x0148b96c,

	0x1b0, 0x00000000,
	0x1b4, 0x00010000,

	0x1c0, 0x00000003,

	0x1d0, 0x00000001,

	0x2a8, 0x0000fd00,

	0x2c0, 0x000005f0,

	0x2c8, 0x00000000,
	0x2cc, 0x003f3f80,

	0xfff, 0xffffffff /* end of list */
};

/* Initial mode (from datasheet) */
static __u8 init_regs[SAA7174HL_NR_REGISTER] = {
	/* task management */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* gamma correction */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* task A */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* task B */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* reserved */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* video decoder */
	0x00, 0x08, 0xc0, 0x00, 0x90, 0x90, 0xeb, 0xe0,
	0x18, 0x40, 0x80, 0x44, 0x3d, 0x00, 0x81, 0x2a,
	0x06, 0x00, 0x00, 0x00, 0x41, 0x11, 0xfe, 0x1c,
	0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* audio and sound processing */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* I2C-bus port */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* video ouput port */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	/* peripherals */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

/* For registering the I2C 'driver' driver */
static struct i2c_driver i2c_driver_saa7174hl = {
	.name		= "SAA7174HL",
	.id		= I2C_DRIVERID_SAA7174,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter = saa7174hl_attach,
	.detach_client	= saa7174hl_detach,
	.command	= saa7174hl_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use	= i2c_client_inc_use,
	.dec_use	= i2c_client_dec_use,
#else
	.owner		= THIS_MODULE,
#endif
};

/* Template for new clients */
static struct i2c_client saa7174hl_client = {
	I2C_DEVNAME ("Philips SAA7174HL"),
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	.id      = -1,
#endif
	.flags   = I2C_CLIENT_ALLOW_USE,
	.addr    = 0,
	.adapter = NULL,
	.driver  = &i2c_driver_saa7174hl,
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
/* Unique ID allocation */
static int saa7174hl_id = 0;
#endif

/* Receive byte through I2C bit-algo */
static int saa7174hl_recv (struct i2c_client *client, int addr)
{
	struct i2c_msg msg[2];
	u8 data[3];
	int ret;

	data[0] = (addr >> 8) & 0x0f; data[1] = addr & 0xff; data[2] = 0;
	msg[0].addr = client->addr;
	msg[0].flags = 0; /* WRITE */
	msg[0].len = 2;
	msg[0].buf = &data[0];
	msg[1].addr = client->addr;
	msg[1].flags = I2C_M_RD;
	msg[1].len = 1;
	msg[1].buf = &data[2];
	ret = i2c_transfer (client->adapter, msg, 2);
	if (ret < 0)
		return -1;
	return data[2] & 0xff;
}

/* Send byte through I2C bit-algo */
static int saa7174hl_send (struct i2c_client *client, int addr, int val)
{
	struct i2c_msg msg[1];
	u8 data[3];
	int ret;

	data[0] = (addr >> 8) & 0x0f; data[1] = addr & 0xff; data[2] = val & 0xff;
	msg[0].addr = client->addr;
	msg[0].flags = 0; /* WRITE */
	msg[0].len = 3;
	msg[0].buf = &data[0];
	ret = i2c_transfer (client->adapter, msg, 1);
	if (ret < 0)
		return -1;
	return data[2] & 0xff;
}

/* Send double-word through I2C bit-algo */
static void saa7174hl_send_32bit (struct i2c_client *client, int addr, u32 val)
{
	saa7174hl_send (client, addr+0, (val >>  0) & 0xff);
	saa7174hl_send (client, addr+1, (val >>  8) & 0xff);
	saa7174hl_send (client, addr+2, (val >> 16) & 0xff);
	saa7174hl_send (client, addr+3, (val >> 24) & 0xff);
}

/* Initialize extra registers. */
static void saa7174hl_send_regs (struct i2c_client *client)
{
	int i;
	for (i = 0; saa7174hl_init_bytes[i] != 0xfff; i += 2) {
		saa7174hl_send_32bit (client, saa7174hl_init_bytes[i], saa7174hl_init_bytes[i+1]);
	}
}

/* Read from a register */
static int saa7174hl_read (struct i2c_client *client, void *reg)
{
	int addr = SAA7174HL_REG_ADDR (client, reg);
	int val = saa7174hl_recv (client, addr);
	
	if (val < 0) {
		SAA7174HL_PRINTK ("failed reading register 0x%03lx\n",
				  SAA7174HL_REG_ADDR (client, reg));
	} else {
		(*((u8*)(reg))) = val;
	}
 
	return val;
}

/* Write to a register */
static int saa7174hl_write (struct i2c_client *client, void *reg)
{
	int addr = SAA7174HL_REG_ADDR (client, reg);
	int val = saa7174hl_send (client, addr, *(u8*)reg);

	if (val < 0) {
		SAA7174HL_PRINTK ("failed writing register 0x%03lx\n",
				  SAA7174HL_REG_ADDR (client, reg));
	}

	return saa7174hl_read (client, reg);
}

/* Read all registers */
static int saa7174hl_read_block (struct i2c_client *client)
{
	saa7174hl_t *saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	int i;

	for (i = 0 ; i < SAA7174HL_NR_REGISTER ; i++) {
		int error;
		if ((error = saa7174hl_read (client, &saa7174hl->reg.byte[i])) < 0)
			return error;
	}
	return 0;
}

/* Write to a register block */
static int saa7174hl_write_block (struct i2c_client *client)
{
	saa7174hl_t *saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	int i;

	for (i = 0 ; i < SAA7174HL_NR_REGISTER ; i++) {
		int error;
		if ((error = saa7174hl_write (client, &saa7174hl->reg.byte[i])) < 0)
			return error;
	}
	return saa7174hl_read_block (client);
}


/* Dump all registers */
static void saa7174hl_dump_regs (struct i2c_client *client) {

	saa7174hl_t * saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	__u8 * bmap = saa7174hl->reg.byte;
	__u32 l, c;

	if (debug >= 2) {
		/* Refresh client registers */
		saa7174hl_read_block (client);

		for (l = 0; l < SAA7174HL_NR_REGISTER / 16; l++) {
			SAA7174HL_DEBUG_PRINTK2 ("%03x |", l * 16);
			for (c = 0; c < 16 && (l * 16 + c) < SAA7174HL_NR_REGISTER; c++) {
				DPRINTK2 (" %02x", bmap[l * 16 + c]);
			}
			DPRINTK2 ("\n");
		}
	}
}

/* Reads and returns the chip revision. */
static char * saa7174hl_revision (struct i2c_client *client)
{
	static char rev[13];
	int ret, idx;
	for (idx = 0; idx < sizeof (rev) - 1; idx++) {
		saa7174hl_send (client, 0x100, idx);
		ret = saa7174hl_recv (client, 0x100) & 0x0f;
		rev[idx] = (ret < 10) ? (ret + '0') : (ret - 10 + 'a');
	}
	rev[idx] = '\0';
	return rev;
}

/* Reads and returns the audio chip software revision. */
static char * saa7174hl_soft_revision (struct i2c_client *client, u32 *id)
{
	static char rev[7];
	int ret, idx, b;
	if (id) {
		*id = 0;
	}
	for (idx = 0; idx < 3; idx++) {
		ret = saa7174hl_recv (client, 0x426 - idx) & 0xff;
		if (id) {
			*id <<= 8;
			*id |= ret;
		}
		b = (ret >> 4) & 0x0f;
		rev[2*idx+0] = (b < 10) ? (b + '0') : (b - 10 + 'a');
		b = (ret >> 0) & 0x0f;
		rev[2*idx+1] = (b < 10) ? (b + '0') : (b - 10 + 'a');
	}
	rev[6] = '\0';
	return rev;
}

/* I2C driver functions */

/* Called when a 'SAA7174HL like' device found */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
static int saa7174hl_detect (struct i2c_adapter *adap, int addr, unsigned short flags, int kind)
#else
static int saa7174hl_detect (struct i2c_adapter *adap, int addr, int kind)
#endif
{
	struct i2c_client * client;
	saa7174hl_t * saa7174hl;
	int err = 0, version, ret, idx;
	u32 audiorev;

	if (!i2c_check_functionality (adap, I2C_FUNC_SMBUS_READ_BYTE | 
				      I2C_FUNC_SMBUS_WRITE_BYTE))
		goto err_out;

	/*
	 * Allocate an I2C client structure and setup it's fields
	 */
	client = kmalloc (sizeof (*client), GFP_KERNEL);

	/* check client structure allocation error */
	if (client == NULL) {
		err = -ENOMEM;
		goto err_out;
	}
	
	/* Zero client structure fields */
	memcpy (client, &saa7174hl_client, sizeof (*client));

	/* Set client's adapter */
	client->adapter = adap;

	/* Set client's address */
	client->addr = addr;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	/* Set SAA7174HL client ID, update for next client */
	client->id = saa7174hl_id++;
#endif

	if (kind < 0) {
		/* Write version index register 2, 3, 4 and 5 */
		for (version = 0, idx = 2; idx <= 5; idx++) {
			if ((ret = saa7174hl_send (client, 0x100, idx)) < 0) {
				err = -EIO;
				goto err_out_kfree_client;
			}
			/* If we could not get version, bail out */
			if ((ret = saa7174hl_recv (client, 0x100)) < 0) {
				err = -EIO;
				goto err_out_kfree_client;
			}
			version <<= 4;
			version |= ret & 0x0f;
		}

		SAA7174HL_PRINTK ("audio revision id: %s\n", saa7174hl_soft_revision (client, &audiorev));
		/* Audio rev. should be: 0x00ac1011 for SAA7134 */

		/* If not a SAA7174HL, SAA7173HL or SAA7133HL, bail out */
		switch (version) {
		case SAA7174HL_I2C_VERSION_ID:
			SAA7174HL_PRINTK ("video decoder chip SAA7174HL found, chip version: %s\n",
					  saa7174hl_revision (client));
			break;
		case SAA7173HL_I2C_VERSION_ID:
			SAA7174HL_PRINTK ("video decoder chip SAA7173HL found, chip version: %s\n",
					  saa7174hl_revision (client));
			break;
		case SAA7134HL_I2C_VERSION_ID:
			SAA7174HL_PRINTK ("video decoder chip SAA7134HL found, chip version: %s\n",
					  saa7174hl_revision (client));
			break;
		case SAA7133HL_I2C_VERSION_ID:
			SAA7174HL_PRINTK ("video decoder chip SAA7133HL found, chip version: %s\n",
					  saa7174hl_revision (client));
			break;
		default:
			SAA7174HL_PRINTK ("unknown SAA71xx chip found, chip version: %s\n",
					  saa7174hl_revision (client));
			err = -ENODEV;
			goto err_out_kfree_client;
		}
	}
	else {
		SAA7174HL_PRINTK ("detection skipped\n");
	}
	
	saa7174hl = kmalloc (sizeof(saa7174hl_t), GFP_KERNEL);

	if (saa7174hl == NULL) {
		SAA7174HL_PRINTK ("failed allocating client data\n");
		err = -ENOMEM;
		goto err_out_kfree_client;
	}

	/* store client data pointer into client */
	i2c_set_clientdata (client, saa7174hl);

	/* initialize client registers and write them into chip */
	saa7174hl->reg.field = *((saa7174hl_reg_map_t *)init_regs);

	if (saa7174hl_write_block (client)) {
		SAA7174HL_PRINTK ("failed to write initial register values\n");
		err = -EIO;
		goto err_out_kfree_decoder;
	}

	if ((err = i2c_attach_client (client))) {
		SAA7174HL_PRINTK ("failed attaching client\n");
		goto err_out_kfree_decoder;
	}

	/* initialize additional registers */
	saa7174hl_send_regs (client);

	saa7174hl_dump_regs (client);
	return 0;
	
 err_out_kfree_decoder:
	kfree (saa7174hl);
 err_out_kfree_client:
	kfree (client);
 err_out:
	return err;
}

/* Called when a new I2C bus found */
static int saa7174hl_attach (struct i2c_adapter *adapter)
{
	return i2c_probe (adapter, &addr_data, saa7174hl_detect);
}

/* Called on exit */
static int saa7174hl_detach (struct i2c_client *client)
{
	int err;

	/* Try to detach client, bail on error */
	if ((err = i2c_detach_client (client)))
		return err;

	/* Free client data */
	kfree (i2c_get_clientdata (client));

	/* Free client */
	kfree (client);

	return 0;
}

/* Autodetection of colour standard */
static void saa7174hl_detection (struct i2c_client *client)
{
	saa7174hl_t *	saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	saa7174hl_reg_map_t * map = &saa7174hl->reg.field;

	/* detect 50/60Hz */
	map->sync_control.AUFD = SAA7174HL_AUFD_AUTO;
	saa7174hl_write (client, &map->sync_control);

	map->chroma_control_1.FCTC = 0;
	map->chroma_control_1.AUTO0 = 1;
	map->chroma_control_1.CSTD = SAA7174HL_CSTD_PALB; /* prefer PAL-B and NTSC-M */
	map->luminance_control.LDEL = 0;
	map->analog_adc.AUTO1 = 0;
	saa7174hl_write (client, &map->luminance_control);
	saa7174hl_write (client, &map->analog_adc);
	saa7174hl_write (client, &map->chroma_control_1);

	/* stay here a little eternity */
	set_current_state (TASK_UNINTERRUPTIBLE);
	schedule_timeout (HZ/10);
	set_current_state (TASK_RUNNING);
			
	/* read status bytes */
	saa7174hl_read (client, &map->status_byte_1);
	saa7174hl_read (client, &map->status_byte_2);

	if (map->status_byte_2.INTL) {
		DPRINTK (" interlaced");
	} else {
		DPRINTK (" non-interlaced");
	}
	if (map->status_byte_2.FIDT) {
		DPRINTK (" 60Hz");
	} else {
		DPRINTK (" 50Hz");
	}
	switch (map->status_byte_1.DCSTD) {
	case 1: DPRINTK (" NTSC"); break;
	case 2: DPRINTK (" PAL"); break;
	case 3: DPRINTK (" SECAM"); break;
	}
}

/* Setup the video output port */
static void saa7174hl_video_port (struct i2c_client *client, int on) {
	saa7174hl_t *	saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	saa7174hl_reg_map_t * map = &saa7174hl->reg.field;
	if (on) {
		map->video_port_control_4.VP_EN_PORT = 1;
		map->video_port_control_4.VP_EN_VIDEO = SAA7174HL_VP_EN_VIDEO_PORT_A;
		map->video_port_control_4.VP_EN_XV = 0;
		map->video_port_control_4.VP_EN_SC_VBI = 1;
		map->video_port_control_4.VP_EN_SC_ACT = 1;
		map->video_port_control_2.VP_EN_CODE_A = 1;
	}
	else {
		map->video_port_control_4.VP_EN_PORT = 0;
	}
	saa7174hl_write (client, &map->video_port_control_2);
	saa7174hl_write (client, &map->video_port_control_4);
}

/* Setup one of the scalers */
static void saa7174hl_setup_scaler (struct i2c_client *client, saa7174hl_scaler_t *scaler,
				    int src_x, int src_y, int src_width, int src_height,
				    int dst_width, int dst_height)
{
	int scale;

	/* general settings */
	scaler->data_path_configuration.CSCON = 1;
	scaler->field_handling.FSKIP = 0;
	saa7174hl_write (client, &scaler->data_path_configuration);
	saa7174hl_write (client, &scaler->field_handling);

	/* input window */
	scaler->video_horizontal_window_start_1.VXF = src_x & 0xff;
	scaler->video_horizontal_window_start_2.VXF_MSB = src_x >> 8;
	scaler->video_horizontal_window_stop_1.VXL = (src_x + src_width) & 0xff;
	scaler->video_horizontal_window_stop_2.VXL_MSB = (src_x + src_width) >> 8;
	scaler->video_vertical_window_start_1.VYF = src_y & 0xff;
	scaler->video_vertical_window_start_2.VYF_MSB = src_y >> 8;
	scaler->video_vertical_window_stop_1.VYL = (src_y + src_height) & 0xff;
	scaler->video_vertical_window_stop_2.VYL_MSB = (src_y + src_height) >> 8;
	saa7174hl_write (client, &scaler->video_horizontal_window_start_1);
	saa7174hl_write (client, &scaler->video_horizontal_window_start_2);
	saa7174hl_write (client, &scaler->video_horizontal_window_stop_1);
	saa7174hl_write (client, &scaler->video_horizontal_window_stop_2);
	saa7174hl_write (client, &scaler->video_vertical_window_start_1);
	saa7174hl_write (client, &scaler->video_vertical_window_start_2);
	saa7174hl_write (client, &scaler->video_vertical_window_stop_1);
	saa7174hl_write (client, &scaler->video_vertical_window_stop_2);

	/* output window */
	scaler->video_number_of_output_pixels_1.VXO = dst_width & 0xff;
	scaler->video_number_of_output_pixels_2.VXO_MSB = dst_width >> 8;
	scaler->video_number_of_lines_1.VYO = dst_height & 0xff;
	scaler->video_number_of_lines_2.VYO_MSB = dst_height >> 8;
	saa7174hl_write (client, &scaler->video_number_of_output_pixels_1);
	saa7174hl_write (client, &scaler->video_number_of_output_pixels_2);
	saa7174hl_write (client, &scaler->video_number_of_lines_1);
	saa7174hl_write (client, &scaler->video_number_of_lines_2);

	/* no prescaling */
	scaler->horizontal_prescaling.XPSC = 1;
	scaler->accumulation_length.XACL = 0;
	saa7174hl_write (client, &scaler->horizontal_prescaling);
	saa7174hl_write (client, &scaler->accumulation_length);

	/* default picture properties */
	scaler->lumina_brightness.BRIG = 0x80;
	scaler->lumina_contrast.CONT = 0x40;
	scaler->chroma_saturation.SATN = 0x40;
	saa7174hl_write (client, &scaler->lumina_brightness);
	saa7174hl_write (client, &scaler->lumina_contrast);
	saa7174hl_write (client, &scaler->chroma_saturation);

	/* horizontal scaler */
	scale = 1024 * src_width / dst_width;
	scaler->horizontal_scaling_increment_1.VXSC = scale & 0xff;
	scaler->horizontal_scaling_increment_2.VXSC_MSB = scale >> 8;
	saa7174hl_write (client, &scaler->horizontal_scaling_increment_1);
	saa7174hl_write (client, &scaler->horizontal_scaling_increment_2);

	/* vertical scaler */
	scale = 1024 * src_height / dst_height;
	scaler->vertical_scaling_ratio_1.YSC = scale & 0xff;
	scaler->vertical_scaling_ratio_2.YSC_MSB = scale >> 8;
	saa7174hl_write (client, &scaler->vertical_scaling_ratio_1);
	saa7174hl_write (client, &scaler->vertical_scaling_ratio_2);
}

/* Setup scaler task A */
static void saa7174hl_setup_scaler_tasks (struct i2c_client *client,
					  int src_x, int src_y, int src_width, int src_height,
					  int dst_width, int dst_height, int enable)
{
	saa7174hl_t         * saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	saa7174hl_reg_map_t * map      = &saa7174hl->reg.field;
	saa7174hl_scaler_t  * scalerA  = &map->taska;
	saa7174hl_scaler_t  * scalerB  = &map->taskb;

	/* reset scaler tasks, toggling reset */
	map->region_enable.SWRST = 0;
	saa7174hl_write (client, &map->region_enable);
	map->region_enable.SWRST = 1;
	saa7174hl_write (client, &map->region_enable);
	map->region_enable.SWRST = 0;
	saa7174hl_write (client, &map->region_enable);

	/* global settings */
	map->region_enable.VID_ENA = enable ? 1 : 0;
	map->region_enable.VID_ENB = enable ? 1 : 0;
	map->region_enable.VBI_ENA = 0;
	map->region_enable.VBI_ENB = 0;
	scalerA->task_condition.UODD = 1;
	scalerA->task_condition.UEVE = 1;
	scalerA->task_condition.U50 = 1;
	scalerA->task_condition.U60 = 1;
	scalerB->task_condition.UODD = 1;
	scalerB->task_condition.UEVE = 1;
	scalerB->task_condition.U50 = 1;
	scalerB->task_condition.U60 = 1;
	saa7174hl_write (client, &map->region_enable);
	saa7174hl_write (client, &scalerA->task_condition);
	saa7174hl_write (client, &scalerB->task_condition);

	if (enable) {
		/* setup the scalers themselves */
		saa7174hl_setup_scaler (client, scalerA,
					src_x, src_y, src_width, src_height, dst_width, dst_height);
		saa7174hl_setup_scaler (client, scalerB,
					src_x, src_y, src_width, src_height, dst_width, dst_height);
	}
}

/* Interface to the world */
static int saa7174hl_command (struct i2c_client *client, unsigned int cmd, void *arg)
{
	saa7174hl_t * saa7174hl = (saa7174hl_t *) i2c_get_clientdata (client);
	saa7174hl_reg_map_t * map = &saa7174hl->reg.field;

	switch (cmd) {
		
#ifdef READ_REGISTERS
	case READ_REGISTERS:
		/* Read to client register map */
		saa7174hl_read_block (client);

		/* Copy client map to passed arg */
		*((saa7174hl_reg_map_t *)arg) = *map;

		SAA7174HL_DEBUG_PRINTK ("READ_REGISTERS\n");

		saa7174hl_dump_regs (client);

		return SAA7174HL_NR_REGISTER;

	case WRITE_REGISTERS:
		/* Write passed map to client map */
		*map = *((saa7174hl_reg_map_t *)arg);

		/* Write client map */
		saa7174hl_write_block (client);

		SAA7174HL_DEBUG_PRINTK ("WRITE_REGISTERS\n");

		saa7174hl_dump_regs (client);

		return SAA7174HL_NR_REGISTER;

	case GET_NR_OF_REGISTERS:
		SAA7174HL_DEBUG_PRINTK ("GET_NR_OF_REGISTERS = %d\n",
				       SAA7174HL_NR_REGISTER);

		return SAA7174HL_NR_REGISTER;
#endif /* READ_REGISTERS */

#ifdef DECODER_DUMP
	case DECODER_DUMP:

		SAA7174HL_DEBUG_PRINTK ("DECODER_DUMP\n");

		saa7174hl_dump_regs (client);

		break;
#endif /* DECODER_DUMP */

#ifdef WRITE_REGISTER
	case WRITE_REGISTER:
	{
		u32 *sarg = arg;

		SAA7174HL_DEBUG_PRINTK ("WRITE_REGISTER\n");
		saa7174hl->reg.byte[(*sarg >> 8) & 0x0fff] = *sarg & 0xff;
		saa7174hl_write (client, &saa7174hl->reg.byte[(*sarg >> 8) & 0x0fff]);
		break;
	}
#endif /* WRITE_REGISTER */

	case DECODER_GET_CAPABILITIES:
	{
		struct video_decoder_capability *cap = arg;

		SAA7174HL_DEBUG_PRINTK ("DECODER_GET_CAPABILITIES\n");
		cap->flags
			= VIDEO_DECODER_PAL
			| VIDEO_DECODER_SECAM
			| VIDEO_DECODER_NTSC
			| VIDEO_DECODER_AUTO
			| VIDEO_DECODER_CCIR;
		cap->inputs = 10;
		cap->outputs = 2;
		break;
	}


	case DECODER_GET_STATUS:
	{
		int *iarg = arg;
		int res;

		saa7174hl_read (client, &map->status_byte_1);
		saa7174hl_read (client, &map->status_byte_2);
		saa7174hl_read (client, &map->sync_control);
		saa7174hl_read (client, &map->mode_delay_control);
		saa7174hl_read (client, &map->chroma_control_1);
		SAA7174HL_DEBUG_PRINTK ("DECODER_GET_STATUS = 0x%02x ->",
					SAA7174HL_BYTEVAL (map->status_byte_2));

		res = 0;

		/* Get horizontal locking state */
		if (map->status_byte_2.HLVLN == 0) {
			DPRINTK (" GOOD");
			res = DECODER_STATUS_GOOD;
		} else {
			DPRINTK (" BAD");
		}

		/* Get or detect decoder norm */
		if (map->sync_control.AUFD == SAA7174HL_AUFD_AUTO) {
			DPRINTK (" AUTODETECT");
			if (map->status_byte_2.FIDT) { /* 60/50 hz */
				res |= DECODER_STATUS_NTSC;
			} else {
				res |= DECODER_STATUS_PAL;
			}
			switch (map->status_byte_1.DCSTD) {
			case SAA7174HL_DCSTD_PAL:
				DPRINTK (" PAL");
				break;
			case SAA7174HL_DCSTD_NTSC:
				DPRINTK (" NTSC");
				break;
			case SAA7174HL_DCSTD_SECAM:
				DPRINTK (" SECAM");
				break;
			case SAA7174HL_DCSTD_NONE:
				DPRINTK (" NONE");
				break;
			}

		} else {
			if (map->sync_control.FSEL == SAA7174HL_FSEL_50HZ) {
				res |= DECODER_STATUS_PAL;
			} else {
				res |= DECODER_STATUS_NTSC;
			}
		}

		if (res & DECODER_STATUS_PAL) {
			switch (map->chroma_control_1.CSTD) {
			case SAA7174HL_CSTD_PALB:
				DPRINTK (" PAL-BGDHI");
				break;

			case SAA7174HL_CSTD_NTSC50:
				DPRINTK (" NTSC-4.43(50Hz)");
				break;

			case SAA7174HL_CSTD_PALN:
				DPRINTK (" PAL-N");
				break;

			case SAA7174HL_CSTD_NTSCN:
				DPRINTK (" NTSC-N");
				break;

			case SAA7174HL_CSTD_SECAM:
				res = (res & ~DECODER_STATUS_PAL) | DECODER_STATUS_SECAM;
				DPRINTK (" SECAM");
				break;
			}
		} else {
			switch (map->chroma_control_1.CSTD) {
			case SAA7174HL_CSTD_NTSCM:
				DPRINTK (" NTSC-M");
				break;

			case SAA7174HL_CSTD_PAL60:
				DPRINTK (" PAL-4.43(60Hz)");
				break;

			case SAA7174HL_CSTD_NTSC60:
				DPRINTK (" NTSC-4.43(60Hz)");
				break;

			case SAA7174HL_CSTD_PALM:
				DPRINTK (" PAL-M");
				break;

			case SAA7174HL_CSTD_NTSCJ:
				DPRINTK (" NTSC-Japan");
				break;
			}      
		}

		if (map->status_byte_1.WIPA) {
			DPRINTK (" WHITE_PEAK");
		}
      
		if (map->status_byte_2.COPRO) {
			DPRINTK (" COPY_PROTECT");
		}

		if (map->status_byte_2.RDCAP) {
			DPRINTK (" READY_FOR_CAPTURE");
		}

		if (map->mode_delay_control.COLO) {
			DPRINTK (" COLOR");
			res |= DECODER_STATUS_COLOR;
		}

		DPRINTK ("\n");
		*iarg = res;
		break;
	}

	case DECODER_SET_NORM: 
	{
		int *iarg = arg;

		SAA7174HL_DEBUG_PRINTK ("DECODER_SET_NORM: ");

		switch (*iarg)
		{

		case VIDEO_MODE_NTSC:
			DPRINTK ("NTSC");
			map->sync_control.AUFD = SAA7174HL_AUFD_FIXED;
			map->sync_control.FSEL = SAA7174HL_FSEL_60HZ;
			map->chroma_control_1.CSTD = SAA7174HL_CSTD_NTSCM;
			map->chroma_control_1.AUTO0 = 0;
			map->analog_adc.AUTO1 = 0;
			map->chroma_gain_control.ACGC = 0;
			saa7174hl_write (client, &map->sync_control);
			saa7174hl_write (client, &map->chroma_control_1);
			saa7174hl_write (client, &map->analog_adc);
			saa7174hl_write (client, &map->chroma_gain_control);
			break;

		case VIDEO_MODE_PAL:
			DPRINTK ("PAL");
			map->sync_control.AUFD = SAA7174HL_AUFD_FIXED;
			map->sync_control.FSEL = SAA7174HL_FSEL_50HZ;
			map->chroma_control_1.CSTD = SAA7174HL_CSTD_PALB;
			map->chroma_control_1.AUTO0 = 0;
			map->analog_adc.AUTO1 = 0;
			map->chroma_gain_control.ACGC = 0;
			saa7174hl_write (client, &map->sync_control);
			saa7174hl_write (client, &map->chroma_control_1);
			saa7174hl_write (client, &map->analog_adc);
			saa7174hl_write (client, &map->chroma_gain_control);
			break;

		case VIDEO_MODE_SECAM:
			DPRINTK ("SECAM");
			map->sync_control.AUFD = SAA7174HL_AUFD_FIXED;
			map->sync_control.FSEL = SAA7174HL_FSEL_50HZ;
			map->chroma_control_1.CSTD = SAA7174HL_CSTD_SECAM;
			map->chroma_control_1.AUTO0 = 0;
			map->analog_adc.AUTO1 = 0;
			map->chroma_gain_control.ACGC = 1;
			map->chroma_gain_control.CGAIN = SAA7174HL_CGAIN_NOMINAL;
			saa7174hl_write (client, &map->sync_control);
			saa7174hl_write (client, &map->chroma_control_1);
			saa7174hl_write (client, &map->analog_adc);
			saa7174hl_write (client, &map->chroma_gain_control);
			break;

		case VIDEO_MODE_AUTO:
			DPRINTK ("AUTODETECT");
			saa7174hl_detection (client);
			break;

		default:
			DPRINTK ("<invalid value %d>\n", *iarg);
			return -EINVAL;

		}
		DPRINTK ("\n");
		break;
	}

	case DECODER_SET_INPUT:
	{
		int *iarg = arg;

		SAA7174HL_DEBUG_PRINTK ("DECODER_SET_INPUT: %d", *iarg);

		map->analog_input_control_1.MODE = *iarg;
		saa7174hl_write (client, &map->analog_input_control_1);

		/* bypass chrominance trap for modes 6..9 */
		if (*iarg >= 6 && *iarg <= 9) {
			map->luminance_control.BYPS = 1;
			map->luminance_control.YCOMB = 0;
			map->chroma_control_1.CCOMB = 0;
			saa7174hl_write (client, &map->luminance_control);
			saa7174hl_write (client, &map->chroma_control_1);
			DPRINTK (" S-Video");
		} else {
			map->luminance_control.BYPS = 0;
			map->luminance_control.YCOMB = 1;
			map->chroma_control_1.CCOMB = 1;
			saa7174hl_write (client, &map->luminance_control);
			saa7174hl_write (client, &map->chroma_control_1);
			DPRINTK (" CVBS");
		}
		DPRINTK ("\n");
		break;
	}

	case DECODER_SET_OUTPUT:
	{
		int *iarg = arg;

		SAA7174HL_DEBUG_PRINTK ("DECODER_SET_OUTPUT: %d\n", *iarg);

		/* not much choice of outputs */
		if (*iarg != 0) {
			return -EINVAL;
		}
		break;
	}

	case DECODER_ENABLE_OUTPUT:
	{
		int *iarg = arg;

		SAA7174HL_DEBUG_PRINTK ("DECODER_ENABLE_OUTPUT: %d\n", *iarg);

		if (*iarg) {
			/* Lock horizontal PLL */
			map->sync_control.HPLL = SAA7174HL_HPLL_CLOSED;

		} else {
			/* Unlock horizontal PLL */
			map->sync_control.HPLL = SAA7174HL_HPLL_OPEN;
		}

		/* Write modified data to chip */
		saa7174hl_write (client, &map->sync_control);

		/* Setup scaler tasks */
		saa7174hl_read (client, &map->status_byte_2);
		if (map->status_byte_2.FIDT) { /* 60 Hz */
			saa7174hl_setup_scaler_tasks (client, 10, 18, 704, 240, 720, 240, *iarg);
		}
		else { /* 50 Hz */
			saa7174hl_setup_scaler_tasks (client, 10, 18, 704, 288, 720, 288, *iarg);
		}

		/* Enable the video port */
		saa7174hl_video_port (client, *iarg);

#if 1
		/* Print scaler status */
		saa7174hl_read (client, &map->scaler_status_1);
		SAA7174HL_DEBUG_PRINTK ("SCALER_STATUS: TR=%d CF=%d LD=%d\n", map->scaler_status_1.TRERR,
					map->scaler_status_1.CFERR, map->scaler_status_1.LDERR);
#endif

		break;
	}

	case DECODER_SET_PICTURE:
	{
		struct video_picture *pic = arg;

		SAA7174HL_DEBUG_PRINTK ("DECODER_SET_PICTURE\n");

		if (saa7174hl->bright != pic->brightness) {
			/* We want 0 to 255 we get 0-65535 */
			saa7174hl->bright = pic->brightness;
			map->luminance_brightness.DBRI = saa7174hl->bright >> 8;
			saa7174hl_write (client, &map->luminance_brightness);
		}
		if (saa7174hl->contrast != pic->contrast) {
			/* We want -128 to 127 we get 0-65535 */
			saa7174hl->contrast = pic->contrast;
			map->luminance_contrast.DCON = (saa7174hl->contrast - 32768) >> 8;
			saa7174hl_write (client, &map->luminance_contrast);
		}
		if (saa7174hl->sat != pic->colour) {
			/* We want -128 to 127 we get 0-65535 */
			saa7174hl->sat = pic->colour;
			map->chroma_saturation.DSAT = (saa7174hl->sat - 32768) >> 8;
			saa7174hl_write (client, &map->chroma_saturation);
		}
		if (saa7174hl->hue != pic->hue) {
			/* We want -128 to 127 we get 0-65535 */
			saa7174hl->hue = pic->hue;
			map->chroma_hue_control.HUEC = (saa7174hl->hue - 32768) >> 8;
			saa7174hl_write (client, &map->chroma_hue_control);
		}
		break;
	}

	default:
		SAA7174HL_DEBUG_PRINTK ("<invalid function 0x%08X>\n", cmd);
		return -EINVAL;
	}

	saa7174hl_dump_regs (client);
	return 0;
}


/*---------------------------------------------------------------------- 
 *	Modularization
 *----------------------------------------------------------------------*/

MODULE_AUTHOR	   ("Stefan Jahn <stefan@lkcc.org>");
MODULE_DESCRIPTION ("SAA7174HL driver");
MODULE_LICENSE	   ("GPL");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param       (debug, int, 0644);
#else
MODULE_PARM	   (debug, "i");
#endif
MODULE_PARM_DESC   (debug, "Debug level: 0 = silent, 1 = verbose, 2 = very verbose (default = 0).");

int __init init_saa7174hl (void) 
{
	return i2c_add_driver (&i2c_driver_saa7174hl);
}

void __exit cleanup_saa7174hl (void) 
{
	i2c_del_driver (&i2c_driver_saa7174hl);
}

module_init (init_saa7174hl);
module_exit (cleanup_saa7174hl);
