#ifndef __SAA7174HL_H
#define __SAA7174HL_H

#include <linux/i2c.h>
#include <linux/videodev.h>
#include <linux/video_decoder.h>

#ifndef I2C_DRIVERID_SAA7174
#define I2C_DRIVERID_SAA7174	0xF7
#endif

#define SAA7174HL_DEVICE_NAME "SAA7174HL"

/*----------------------------------------------------------------------
 *	SAA7174HL register map definition
 *----------------------------------------------------------------------*/

/* number of SAA7174HL registers */
#define SAA7174HL_NR_REGISTER 0x1E0

typedef struct {
	__u8 YPOS : 8; /*
			 value: this value gets loaded into line counter at vertical trigger event
		       */
} saa7174hl_r000_t, saa7174hl_source_timing_reference_1_t;

typedef struct {
	__u8 YPOS_MSB : 4; /* MSBs of YPOS */
	__u8 DHED     : 1; /*
			     select: define horizontal trigger event, using both edges of input horizontal
			     .	     sync-reference (i.e. HREF from decoder)
			     0: pixel counter will reset (zero) at rising edge, line counter will
			     .	increment at falling edge
			     1: pixel counter will reset (zero) at falling edge, line counter will
			     .	increment at rising edge
			   */
	__u8 DVED     : 1; /*
			     select: define vertical trigger event for line numbering definition
			     0: rising edge of input vertical sync-reference will load YPOS[11:0]
			     1: falling edge of input vertical sync-reference will load YPOS[11:0]
			   */
	__u8 DFID     : 1; /*
			     select: field ID polarity switch
			     0: input field ID is taken as provided
			     1: input field ID is inverted for interpretation by scaler
			   */
	__u8 _X_      : 1; /* reserved */
} saa7174hl_r001_t, saa7174hl_source_timing_reference_2_t;

typedef __u8 saa7174hl_r002_t;

typedef __u8 saa7174hl_r003_t;

typedef struct {
	__u8 VID_ENA : 1; /*
			    0: active video region of task A is disabled
			    1: active video region of task A is enabled
			  */
	__u8 VBI_ENA : 1; /*
			    0: VBI region of task A is disabled
			    1: VBI region of task A is enabled
			  */
	__u8 _XX_    : 2; /* reserved */
	__u8 VID_ENB : 1; /*
			    0: active Video region of task B is disabled
			    1: active Video region of task B is enabled
			  */
	__u8 VBI_ENB : 1; /*
			    0: VBI region of task B is disabled
			    1: VBI region of task B is enabled
			  */
	__u8 _X_     : 1; /* reserved */
	__u8 SWRST   : 1; /*
			    select: reset to scaling process; reset state-machines, keep programming
			    .	    for test and debug, must toggle 0 - 1 - 0 to operate again
			    0: normal operation
			    1: reset to scaling process. any running process is terminated and reset
			  */
} saa7174hl_r004_t, saa7174hl_region_enable_t;

typedef __u8 saa7174hl_r005_t;

typedef struct {
	__u8 VID_A : 1; /*
			  0: this region is currently inactive
			  1: this region is currently processing
			*/
	__u8 VBI_A : 1; /*
			  0: this region is currently inactive
			  1: this region is currently processing
			*/
	__u8 _XX_  : 2; /* reserved */
	__u8 VID_B : 1; /*
			  0: this region is currently inactive
			  1: this region is currently processing
			*/
	__u8 VBI_B : 1; /*
			  0: this region is currently inactive
			  1: this region is currently processing
			*/
	__u8 _X_   : 2; /* reserved */
} saa7174hl_r006_t, saa7174hl_scaler_status_0_t;

typedef struct {
	__u8 TRERR  : 1; /*
			   flag: 'trigger error' ; scaler task manager is failing to trigger any task
			   .	 for at least 2 fields; trigger conditions should be checked
			   0: normal operation
			   1: trigger error; this flag will get reset with the next input V-sync
			 */
	__u8 CFERR  : 1; /*
			   flag: 'Configuration error' ; scaler data path is not ready at start of next
			   .	 region; definition of I/O window should be checked for consistency
			   0: normal operation
			   1: configuration error; this flag will get reset with the next input V-sync
			 */
	__u8 LDERR  : 1; /*
			   flag: 'Load error' ; scaler data path is overloaded, due to mismatched I/O
			   .	 or internal data rates, e.g. too much zooming
			   0: normal operation
			   1: load error; this flag will get reset with the next input V-sync
			 */
	__u8 WASRST : 1; /*
			   0: normal operation
			   1: 'was reset' ; this flag will get reset with the next input V-sync
			 */
	__u8 FIDSCI : 1; /*
			   select: field ID as seen at the scaler input, as selected by DFID
			   0: field ID of input is 0 = 'odd'
			   1: field ID of input is 1 = 'even'
			 */
	__u8 FIDSCO : 1; /*
			   select: field ID as assigned by the scaler output, depend on FIDC
			   0: output field ID is 0 = upper (first DMA buffer)
			   1: output field ID is 1 = lower (second DMA buffer)
			 */
	__u8 D4XD5  : 1; /*
			   select: logical XOR of bits FIDSCO and FIDSCI, may be used to find permanent
			   .	   problem with the field ID interpretation
			   0: input FID and output FID are identical
			   1: input FID and output FID are not identical
			 */
	__u8 TASK   : 1; /*
			   select: VID_A, VBI_A, VID_B, VBI_B indicate which region is currently using
			   .	   the scaler resources for processing; only one of the four bits can
			   .	   be set at a time; if a presently active region gets disabled on the
			   .	   fly, it will be finished orderly and then reported as 'inactive'
			   0: task A is currently activated
			   1: task B is currently activated
			 */
} saa7174hl_r007_t, saa7174hl_scaler_status_1_t;

typedef __u8 saa7174hl_r008_t;

typedef __u8 saa7174hl_r009_t;

typedef __u8 saa7174hl_r00a_t;

typedef __u8 saa7174hl_r00b_t;

typedef struct {
	__u8 P0GST : 8; /*
			  value: look up table definition for GREEN path, all values straight binary
			  start point for curve generation, output value for input 00H
			*/
} saa7174hl_r00c_t, saa7174hl_start_point_green_path_t;

typedef struct {
	__u8 P1GST : 8; /*
			  value: look up table definition for BLUE path, all values straight binary
			  start point for curve generation, output value for input 00H
			*/
} saa7174hl_r00d_t, saa7174hl_start_point_blue_path_t;

typedef struct {
	__u8 P2GST : 8; /*
			  value: look up table definition for RED path, all values straight binary
			  start point for curve generation, output value for input 00H
			*/
} saa7174hl_r00e_t, saa7174hl_start_point_red_path_t;

typedef __u8 saa7174hl_r00f_t;

typedef struct {
	__u8 P0GA : 8; /* corner point of curve 0, output = P0GAy at input = y + 1 */
} saa7174hl_r010_t, saa7174hl_green_path_gamma_curve_t;

typedef struct {
	__u8 P1GA : 8; /* corner point of curve 1, output = P1GAy at input = y + 1 */
} saa7174hl_r020_t, saa7174hl_blue_path_gamma_curve_t;

typedef struct {
	__u8 P2GA : 8; /* corner point of curve 2, output = P2GAy at input = y + 1 */
} saa7174hl_r030_t, saa7174hl_red_path_gamma_curve_t;

typedef struct {
	__u8 UODD : 1; /*
			 0: if input FID = odd, then don't start this task; toggle to other task
			 1: if input FID = odd, then start this task
		       */
	__u8 UEVE : 1; /*
			 0: if input FID = even, then don't start this task; toggle to other task
			 1: if input FID = even, then start this task
		       */
	__u8 U50  : 1; /*
			 0: this task will not be worked, if signal source is 50 Hz
			 1: if signal source is 50 Hz, then work this task
		       */
	__u8 U60  : 1; /*
			 0: this task will not be worked, if signal source is 60 Hz
			 1: if signal source is 60 Hz, then work this task
		       */
	__u8 SLCK : 1; /*
			 select: process this task only, if video decoder is locked (status);
			 .	 if lock-status gets lost, the task will be terminated immediately
			 0: don't care the lock-status of the video decoder
			 1: process this task only, if video decoder is locked
		       */
	__u8 _X_  : 3; /* reserved */
} saa7174hl_r040_t, saa7174hl_task_condition_t;

typedef struct {
	__u8 FSKIP : 8; /*
			  value: number of fields with video region being processed, then end task
			  0: task ends after skip-count, no video region output is generated
			  1: 1 active video region processed, then end task
			  2: video region of 2 consecutive fields are processed, then end task
			  3: 3 fields to be processed, then end task

			  value: number of fields with active video region being skipped, but VBI
			  .	 region will be processed, if enabled by VBI_ENA/VBI_ENB (for field rate reduction)
			  00-3f: skip 0 to 63 fields without processing active video region, then do
			*/
} saa7174hl_r041_t, saa7174hl_field_handling_t;

typedef struct {
	__u8 MACON : 1;
	__u8 CSCON : 1; /*
			  0: YUV to RGB matrix is active
			  1: YUV to RGB matrix bypassed
			*/
	__u8 GACON : 1; /*
			  select: LUT transformation (e.g. gamma compensation)
			  0: transformation via LUT curve is disabled, bypassed
			  1: transformation via LUT curve is enabled
			*/
	__u8 PCSEL : 1; /*
			  select: signal to be provided to DMA
			  0: output of RGB-to-YUV matrix is provided to DMA and PCI
			  1: output from gamma curve is provided to DMA and PCI
			*/
	__u8 GPSEL : 1; /*
			  select: signal to be provided to video port (GPIO)
			  0: output of RGB-to-YUV matrix is provided to video port
			  1: output from gamma curve is provided to video port
			*/
	__u8 FIDC  : 1; /*
			  select: definition of output field ID (FID)
			  0: output FID is same as input FID; scaler output of odd input field is written to
			  .  first (upper) DMA buffer, scaler output of even input field is written to
			  .  second (lower) buffer
			  1: field ID cross conversion is enabled; output FID toggles 'free-running', i.e.
			  .  DMA buffer pointer toggles with each output field (per task and DMA channel)
			  .  vertical phase offsets are selected
			*/
	__u8 RAWS  : 2; /*
			  select: signal source for scaling/processing in RAW VBI region
			  00: 4:2:2 YUV data input from decoder
			  01: reserved
			  10: RAW samples input from ADC1 on Luma byte; Chroma undefined
			  11: RAW samples input from ADC2 on Luma byte; Chroma undefined
			*/
} saa7174hl_r042_t, saa7174hl_data_path_configuration_t;

typedef __u8 saa7174hl_r043_t;

typedef struct {
	__u8 RXF : 8; /*
			value: VBI region starts (horizontally) at pixel number XYZ
		      */
} saa7174hl_r044_t, saa7174hl_vbi_horizontal_window_start_1_t;

typedef struct {
	__u8 RXF_MSB : 4; /* MSBs of RXF */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r045_t, saa7174hl_vbi_horizontal_window_start_2_t;

typedef struct {
	__u8 RXL : 8; /*
			value: VBI region ends (horizontally) at pixel number XYZ
		      */
} saa7174hl_r046_t, saa7174hl_vbi_horizontal_window_stop_1_t;

typedef struct {
	__u8 RXL_MSB : 4; /* MSBs of RXL */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r047_t, saa7174hl_vbi_horizontal_window_stop_2_t;

typedef struct {
	__u8 RYF : 8; /*
			value: VBI region starts (in vertical direction) at line number XYZ
		      */
} saa7174hl_r048_t, saa7174hl_vbi_vertical_window_start_1_t;

typedef struct {
	__u8 RYF_MSB : 4; /* MSBs of RYF */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r049_t, saa7174hl_vbi_vertical_window_start_2_t;

typedef struct {
	__u8 RYL : 8; /*
			value: VBI region ends (in vertical direction) at line number XYZ
		      */
} saa7174hl_r04a_t, saa7174hl_vbi_vertical_window_stop_1_t;

typedef struct {
	__u8 RYL_MSB : 4; /* MSBs of RYL */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r04b_t, saa7174hl_vbi_vertical_window_stop_2_t;

typedef struct {
	__u8 RXO : 8; /*
			value: number MMM of RAW samples output per line in VBI region
		      */
} saa7174hl_r04c_t, saa7174hl_vbi_horizontal_output_length_1_t;

typedef struct {
	__u8 RXO_MSB : 4; /* MSBs of RXO */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r04d_t, saa7174hl_vbi_horizontal_output_length_2_t;

typedef struct {
	__u8 RYO; /*
		    value: number NNN of VBI output lines
		  */
} saa7174hl_r04e_t, saa7174hl_vbi_vertical_output_length_1_t;

typedef struct {
	__u8 RYO_MSB : 4; /* MSBs of RYO */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r04f_t, saa7174hl_vbi_vertical_output_length_2_t;

typedef __u8 saa7174hl_r050_t;

typedef __u8 saa7174hl_r051_t;

typedef __u8 saa7174hl_r052_t;

typedef __u8 saa7174hl_r053_t;

typedef struct {
	__u8 VXF : 8; /*
			value: video region starts (horizontally) at pixel number XYZ
		      */
} saa7174hl_r054_t, saa7174hl_video_horizontal_window_start_1_t;

typedef struct {
	__u8 VXF_MSB : 4; /* MSBs of VXF */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r055_t, saa7174hl_video_horizontal_window_start_2_t;

typedef struct {
	__u8 VXL : 8; /*
			value: video region ends (horizontally) at pixel number XYZ
		      */
} saa7174hl_r056_t, saa7174hl_video_horizontal_window_stop_1_t;

typedef struct {
	__u8 VXL_MSB : 4; /* MSBs of VXL */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r057_t, saa7174hl_video_horizontal_window_stop_2_t;

typedef struct {
	__u8 VYF : 8; /*
			value: video region starts (in vertical direction) at line number XYZ
		      */
} saa7174hl_r058_t, saa7174hl_video_vertical_window_start_1_t;

typedef struct {
	__u8 VYF_MSB : 4; /* MSBs of VYF */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r059_t, saa7174hl_video_vertical_window_start_2_t;

typedef struct {
	__u8 VYL : 8; /*
			value: video region ends (in vertical direction) at line number XYZ
		      */
} saa7174hl_r05a_t, saa7174hl_video_vertical_window_stop_1_t;

typedef struct {
	__u8 VYL_MSB : 4; /* MSBs of VYL */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r05b_t, saa7174hl_video_vertical_window_stop_2_t;

typedef struct {
	__u8 VXO : 8; /*
			value: number MMM of active video pixels output per line
		      */
} saa7174hl_r05c_t, saa7174hl_video_number_of_output_pixels_1_t;

typedef struct {
	__u8 VXO_MSB : 4; /* MSBs of VXO */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r05d_t, saa7174hl_video_number_of_output_pixels_2_t;

typedef struct {
	__u8 VYO : 8; /*
			value: number NNN of active video output lines, after scaling
		      */
} saa7174hl_r05e_t, saa7174hl_video_number_of_lines_1_t;

typedef struct {
	__u8 VYO_MSB : 4; /* MSBs of VYO */
	__u8 _X_     : 4; /* reserved */
} saa7174hl_r05f_t, saa7174hl_video_number_of_lines_2_t;

typedef struct {
	__u8 XPSC : 6; /*
			 value: horizontal prescale ratio, range 1/1 to 1/63
			 00: equal or greater value is not allowed, don't use
			 01: ratio = 1/1, i.e. pass through
			 nn: ratio = 1/NN, i.e. sample rate reduction by integer factor 'nn'
			 3f: ratio = 1/63
		       */
	__u8 _X_  : 2; /* reserved */
} saa7174hl_r060_t, saa7174hl_horizontal_prescaling_t;

typedef struct {
	__u8 XACL : 6; /*
			 value: prescale-filter (accumulation) length; MM pixels will be accumulated
			 .	and result must be attenuated (shift right by XDCG binary positions),
			 .	to get into nominal number range
			 00: length = 1; no accumulation
			 nn: length = nn + 1; weighting factor according XC2_1
			 3f: length = 64
		       */
	__u8 _X_  : 2; /* reserved */
} saa7174hl_r061_t, saa7174hl_accumulation_length_t;

typedef struct {
	__u8 XDCG  : 3; /*
			  value: DC gain adjustment, gain = 1/(2^[XDCG-1])
			*/
	__u8 XC2_1 : 1; /*
			  value: prescaling accumulation filter weighting factors
			  0: '0' = 1 + 1 + ... + 1 + 1
			  1: '1' = 1 + 2 + ... + 2 + 1
			*/
	__u8 _X_   : 4; /* reserved */
} saa7174hl_r062_t, saa7174hl_level_control_t;

typedef struct {
	__u8 VPFY : 2; /* 
			  value: prefilter for luminance path
			  00: delay compensated bypass
			  nn: other filter coefficients
		       */
	__u8 VPFC : 2; /* value: prefilter for chrominance path */
	__u8 RPFY : 2; /* value: pre-filter for VBI region, band limiting and for noise reduction */
	__u8 RPFC : 2; /* value: pre-filter-C for VBI region (program = RPFY) */
} saa7174hl_r063_t, saa7174hl_fir_prefilter_control_t;

typedef struct {
	__u8 BRIG : 8; /*
			 value: luminance brightness control
			 00: all black
			 80: nominal value = '128' (0 x 80)
			 ff: all white
		       */
} saa7174hl_r064_t, saa7174hl_lumina_brightness_t;

typedef struct {
	__u8 CONT : 8; /*
			 value: luminance contrast control, gain range 1/64 to 127/64
			 00: flat
			 40: nominal value = '64' = 0 x 40
			 ff: four-fold saturation
		       */
} saa7174hl_r065_t, saa7174hl_lumina_contrast_t;

typedef struct {
	__u8 SATN : 8; /*
			 value: luminance contrast control, gain range 1/64 to 127/64
			 00: no color
			 40: nominal value = '64' = 0 x 40
			 ff: four-fold saturation
		       */
} saa7174hl_r066_t, saa7174hl_chroma_saturation_t;

typedef __u8 saa7174hl_r067_t;

typedef struct {
	__u8 RXSC : 8; /*
			 value: VBI pixel re-sampling; scaling ratio = 1024d/RXSC
			*/
} saa7174hl_r068_t, saa7174hl_vbi_horizontal_scaling_increment_1_t;

typedef struct {
	__u8 RXSC_MSB : 4; /* MSBs of RXSC */
	__u8 _X_      : 4; /* reserved */
} saa7174hl_r069_t, saa7174hl_vbi_horizontal_scaling_increment_2_t;

typedef struct {
	__u8 RXPHY : 8; /*
			  value: start VBI re-sampling with horizontal phase offset = RXPHY x 2.3 ns
			*/
} saa7174hl_r06a_t, saa7174hl_vbi_phase_offset_t;

typedef struct {
	__u8 RXPHC : 8; /*
			  value: must be RXPHY/2
			*/
} saa7174hl_r06b_t, saa7174hl_vbi_phase_offset_chroma_t;

typedef struct {
	__u8 VXSC : 8; /*
			 value: horizontal fine scaling, scaling ratio = 1024d/VXSC
		       */
} saa7174hl_r06c_t, saa7174hl_horizontal_scaling_increment_1_t;

typedef struct {
	__u8 VXSC_MSB : 5; /* MSBs of VXSC */
	__u8 _X_      : 3; /* reserved */
} saa7174hl_r06d_t, saa7174hl_horizontal_scaling_increment_2_t;

typedef struct {
	__u8 VXPHY : 8; /*
			  value: start fine scaling with horizontal phase offset = VXPHY x 2.3 ns
			*/
} saa7174hl_r06e_t, saa7174hl_horizontal_phase_offset_luma_t;

typedef struct {
	__u8 VXPHC : 8; /*
			  value: must be VXPHY/2
			*/
} saa7174hl_r06f_t, saa7174hl_horizontal_phase_offset_chroma_t;

typedef struct {
	__u8 YSC : 8; /*
			value: vertical scaling ratio = 1024/YSC
		      */
} saa7174hl_r070_t, saa7174hl_vertical_scaling_ratio_1_t;

typedef struct {
	__u8 YSC_MSB : 8; /* MSBs of YSC */
} saa7174hl_r071_t, saa7174hl_vertical_scaling_ratio_2_t;

typedef struct {
	__u8 YMODE : 1; /*
			  0: bi-linear interpolation (LPI)
			  1: phase correct accumulation, multi-tap filter
			*/
	__u8 YMIR  : 1; /*
			  0: no mirroring
			  1: mirror video picture, flip left to right, e.g. for video phone or own picture
			*/
	__u8 _X_   : 6; /* reserved */
} saa7174hl_r072_t, saa7174hl_vertical_filter_mode_t;

typedef __u8 saa7174hl_r073_t;

typedef struct {
	__u8 YPH : 8; /*
			value: for phase accurate interlacing or cross conversion; 1-line of input = 20H
			vertical phase offset(00), input odd field -> output upper field
		      */
} saa7174hl_r074_t, saa7174hl_v_phase_offset_t;

typedef __u8 saa7174hl_r078_t;
typedef __u8 saa7174hl_r079_t;
typedef __u8 saa7174hl_r07a_t;
typedef __u8 saa7174hl_r07b_t;
typedef __u8 saa7174hl_r07c_t;
typedef __u8 saa7174hl_r07d_t;
typedef __u8 saa7174hl_r07e_t;
typedef __u8 saa7174hl_r07f_t;

typedef struct {
	saa7174hl_r040_t task_condition;
	saa7174hl_r041_t field_handling;
	saa7174hl_r042_t data_path_configuration;
	saa7174hl_r043_t reserved_r043;
	saa7174hl_r044_t vbi_horizontal_window_start_1;
	saa7174hl_r045_t vbi_horizontal_window_start_2;
	saa7174hl_r046_t vbi_horizontal_window_stop_1;
	saa7174hl_r047_t vbi_horizontal_window_stop_2;
	saa7174hl_r048_t vbi_vertical_window_start_1;
	saa7174hl_r049_t vbi_vertical_window_start_2;
	saa7174hl_r04a_t vbi_vertical_window_stop_1;
	saa7174hl_r04b_t vbi_vertical_window_stop_2;
	saa7174hl_r04c_t vbi_horizontal_output_length_1;
	saa7174hl_r04d_t vbi_horizontal_output_length_2;
	saa7174hl_r04e_t vbi_vertical_output_length_1;
	saa7174hl_r04f_t vbi_vertical_output_length_2;
	saa7174hl_r050_t reserved_r050;
	saa7174hl_r051_t reserved_r051;
	saa7174hl_r052_t reserved_r052;
	saa7174hl_r053_t reserved_r053;
	saa7174hl_r054_t video_horizontal_window_start_1;
	saa7174hl_r055_t video_horizontal_window_start_2;
	saa7174hl_r056_t video_horizontal_window_stop_1;
	saa7174hl_r057_t video_horizontal_window_stop_2;
	saa7174hl_r058_t video_vertical_window_start_1;
	saa7174hl_r059_t video_vertical_window_start_2;
	saa7174hl_r05a_t video_vertical_window_stop_1;
	saa7174hl_r05b_t video_vertical_window_stop_2;
	saa7174hl_r05c_t video_number_of_output_pixels_1;
	saa7174hl_r05d_t video_number_of_output_pixels_2;
	saa7174hl_r05e_t video_number_of_lines_1;
	saa7174hl_r05f_t video_number_of_lines_2;
	saa7174hl_r060_t horizontal_prescaling;
	saa7174hl_r061_t accumulation_length;
	saa7174hl_r062_t level_control;
	saa7174hl_r063_t fir_prefilter_control;
	saa7174hl_r064_t lumina_brightness;
	saa7174hl_r065_t lumina_contrast;
	saa7174hl_r066_t chroma_saturation;
	saa7174hl_r067_t reserved_r067;
	saa7174hl_r068_t vbi_horizontal_scaling_increment_1;
	saa7174hl_r069_t vbi_horizontal_scaling_increment_2;
	saa7174hl_r06a_t vbi_phase_offset;
	saa7174hl_r06b_t vbi_phase_offset_chroma;
	saa7174hl_r06c_t horizontal_scaling_increment_1;
	saa7174hl_r06d_t horizontal_scaling_increment_2;
	saa7174hl_r06e_t horizontal_phase_offset_luma;
	saa7174hl_r06f_t horizontal_phase_offset_chroma;
	saa7174hl_r070_t vertical_scaling_ratio_1;
	saa7174hl_r071_t vertical_scaling_ratio_2;
	saa7174hl_r072_t vertical_filter_mode;
	saa7174hl_r073_t reserved_r073;
	saa7174hl_r074_t v_phase_offset[4];
	saa7174hl_r078_t reserved_r078;
	saa7174hl_r079_t reserved_r079;
	saa7174hl_r07a_t reserved_r07a;
	saa7174hl_r07b_t reserved_r07b;
	saa7174hl_r07c_t reserved_r07c;
	saa7174hl_r07d_t reserved_r07d;
	saa7174hl_r07e_t reserved_r07e;
	saa7174hl_r07f_t reserved_r07f;
} saa7174hl_scaler_t;

typedef __u8 saa7174hl_r0c0_t;

typedef __u8 saa7174hl_r100_t;

typedef struct {
	__u8 IDEL  : 4; /*
			  select: pipeline delay adjustment for clock generation increment update
			  1000: default, don't use different values
			*/
	__u8 GUDL  : 2; /*
			  value: hysteresis for video AGC, inertia of x steps
			  00: no hysteresis, default (recommended)
			*/
	__u8 WPOFF : 1; /*
			  0: white peak control of video AGC is enabled; if luminance signal 
			  .  exceeds nominal white level, gain gets reduced (recommended)
			  1: white peak control is disabled
			*/
	__u8 _X_   : 1; /* reserved */
} saa7174hl_r101_t, saa7174hl_increment_delay_t;

typedef struct {
	__u8 MODE : 4; /* 
			  selection of input signal and routing
			  0-4: CVBS
			  6-9: S-Video
		       */
	__u8 _X_  : 2; /* reserved */
	__u8 FUSE : 2; /*
			 select: amplifier for gain control
			 00: bypassed and anti-alias-filter bypassed
			 01: reserved
			 10: active and anti-alias-filter bypassed (recommended)
			 11: active and anti-alias-filter active
			*/
} saa7174hl_r102_t, saa7174hl_analog_input_control_1_t;

typedef struct {
	__u8 GAI18 : 1; /* MSB of GAI1 */
	__u8 GAI28 : 1; /* MSB of GAI2 */
	__u8 GAFIX : 1; /*
			  0: Automatic Gain Control (AGC) is applied to CVBS and Y input signals; the 
			  .  selected input video signal is amplified or attenuated, so that sync bottom
			  .  is digitized as 1 by the ADC, and ADC does not overflow; gain control for
			  .  Chroma channel according setting of MODE[3:0] (recommended)
			  1: video AGC is disabled; gain adjustment is user programmable as fix gain
			  .  via GAI1[7:0] + GAI2[7:0]
			*/
	__u8 HOLDG : 1; /*
			  0: Automatic Gain Control (AGC) is active (recommended)
			  1: video AGC is on hold, last gain adjustment value is frozen (test or debug)
			*/
	__u8 CPOFF : 1; /*
			  0: color peak control of video AGC is enabled; if input composite video signal
			  .  exceeds ADC range, gain gets reduced (recommended)
			  1: color peak control is disabled
			*/
	__u8 VBSL  : 1; /*
			  select: automatic video clamp and gain control pauses during vertical blanking
			  .	  interval; 2 modes are available
			  0: short pause; during equalization- and serration pulses, i.e. line 1 to 9 for 
			  .  60 Hz systems, e.g. (recommended for unprotected sources)
			  1: long pause; from first pre-equalization pulses until begin of active video i.e 
			  .  line 1 to line 22 for 60 Hz systems, or till line 24 for 50 Hz systems 
			  .  (recommended for copy protected sources)
			*/
	__u8 HLNRS : 1; /*
			  0: clamping of input signal is operating in normal video tracking mode; the 
			  .  clamp control switches clamp current to and from the coupling capacitor to 
			  .  shift the DC offset of the input video signal, so that the video blanking 
			  .  level of CVBS/Y gets digitized as 120 dec by the 9-bit ADC; Chroma C-input
			  .  is clamped to 256 dec of 9-bit ADC (for CVBS or Y-C)
			  1: clamping in voltage reference mode, as long as decoder is not locked; a voltage
			  .   reference is applied to clamp (move) the average voltage of the input signal
			  .   to the centre of the ADC range; this mode must be used for digitizing DTV/DVB 
			  .   signals, e.g. VSB for DTV/DVB
			*/
	__u8 TEST  : 1; /*
			  0: normal device operation
			  1: test mode; some other control bits will change their meaning; don't use in 
			  .  normal application for test only
			*/
} saa7174hl_r103_t, saa7174hl_analog_input_control_2_t;

typedef struct {
	__u8 GAI2 : 8; /*
			 value: programmable gain adjustment for video input channel 1 (CV... and CV..);
			 .	valid only if video AGC is disabled (GAFIX = 1)
			 000h: -3 dB, input signal of 1 V(p-p) will get digitized with out clipping
			 091h: 0 dB, input signal of 0.7 V(p-p) will use ADC range optimally
			 1ffh: +6 dB, input signal of 0.35 V(p-p) will use ADC range optimally
		       */
} saa7174hl_r104_t, saa7174hl_analog_input_control_3_t;

typedef struct {
	__u8 GAI2 : 8;	/*
			 value: programmable gain adjustment for video input channel 2 (CV... and CV..);
			 .	valid only if video AGC is disabled (GAFIX = 1)
			 000h: -3 dB, input signal of 1 V(p-p) will get digitized with out clipping
			 091h: 0 dB, input signal of 0.7 V(p-p) will use ADC range optimally
			 1ffh: +6 dB, input signal of 0.35 V(p-p) will use ADC range optimally
		       */
} saa7174hl_r105_t, saa7174hl_analog_input_control_4_t;

typedef struct {
	__u8 HGB : 8; /*
			value: begin (HGB, rising edge) of HGATE; count unit is 8 LLC clock cycles,
			.      earlier < 00H < later
			range for 50 Hz/625 lines: 94H to FFH and 00H to 6BH; -108d to +108d
			range for 60 Hz/525 lines: 95H to FFH and 00H; 6CH; -107d to +107d
			other values are out of range, don't use
		      */
} saa7174hl_r106_t, saa7174hl_horizontal_sync_start_t;

typedef struct {
	__u8 HGE : 8; /*
			value: end (HGE, falling edge) of HGATE; count unit is 8 LLC clock cycles,
			.      earlier < 00H < later
			range for 50 Hz/625 lines: 94H to FFH and 00H to 6BH; -108d to +108d
			range for 60 Hz/525 lines: 95H to FFH and 00H; 6CH; -107d to +107d
			other values are out of range, don't use
		      */
} saa7174hl_r107_t, saa7174hl_horizontal_sync_stop_t;

typedef struct {
	__u8 VNOI : 2; /*
			 select: recovery of vertical synchronisation, field ID and field rate system
			 00: normal operation; tracking the incoming video signal with some inertia, surviving
			 .   noisy signal and missing sync pulses (recommended)
			 01: fast tracking; applicable for stable sources, e.g. TV channel hopping 
			 .   (AUFD must be switched off)
			 10: free running; regular vertical sync output with nominal timing, not tracking
			 .   the input signal (e.g. synthesize video timing)
			 11: immediate mode; no inertia, e.g. immediately catching next vertical sync after source switch
		       */
#define SAA7174HL_HPLL_CLOSED 0
#define SAA7174HL_HPLL_OPEN   1
	__u8 HPLL : 1; /*
			 0: horizontal PLL is enabled, to track incoming horizontal sync and
			 .  to lock video clock to line frequency (line locked clock = LLC) recommended
			 1: horizontal PLL is disabled, and not tracking the input signal; clock LLC 
			 .  assumes nominal frequency of 27 MHz and the generated output horizontal
			 .  sync signals (HREF and HGATE) will be regular
		       */
#define SAA7174HL_HTC_FAST 3
	__u8 HTC  : 2; /*
			 select: characteristic of HPLL (horizontal sync and line locked clock generation)
			 00: TV mode; for poor quality TV signals only, e.g. bad signal-to-noise ratio
			 01: VTR mode; fast timing tracking for low noise input signals, but frequency
			 .   deviations of output clock and sync signals are restricted; e.g. recommended
			 .   if frequency sensitive circuit is depend on this timing
			 10: reserved
			 11: fast tracking and locking of HPLL (recommended)
		       */
	__u8 FOET : 1; /*
			 select: generating field identification (odd/even) and FID
			 0: FID tracks video input, and toggles only if source is interlaced (recommended)
			 1: FID tracks input signal, but is forced to toggle, even if source is 
			 .  non-interlaced (no more than two neighboring fields carry same field ID)
		       */
#define SAA7174HL_FSEL_50HZ 0
#define SAA7174HL_FSEL_60HZ 1
	__u8 FSEL : 1; /*
			 field rate is forced by programming, only if AUFD is cleared to 0
			 0: field rate is set (fix) to 50 Hz, about 625 lines per frame
			 1: field rate is set (fix) to 60 Hz, about 525 lines per frame
		       */
#define SAA7174HL_AUFD_FIXED 0
#define SAA7174HL_AUFD_AUTO  1
	__u8 AUFD : 1; /*
			 0: field rate (50/60 Hz) is defined by FSEL
			 1: automatic field rate detection (ignoring FSEL) (recommended)
		       */
} saa7174hl_r108_t, saa7174hl_sync_control_t;

typedef struct {
	__u8 LUFI  : 4; /*
			  luminance filter for sharpness adjustment
			  0000-1111: various settings
			*/
	__u8 LUBW  : 1; /*
			  bandwidth of Chroma notch in luminance spectrum, or comb filter width
			  0: narrow Chroma notch, higher luminance bandwidth
			  1: wider Chroma notch, less luminance bandwidth
			*/
	__u8 LDEL  : 1; /*
			  select: additional vertical delay, restricted use
			  0: processing delay is equal to internal pipelining delay, default setting and
			  .  required for AUTO standard detection (normal)
			  1: one (NTSC-standards) or two (PAL-standards) video lines additional processing
			  .  delay -- for non-comb filter modes or SECAM only, for test/demo purpose only (debug)
			*/
	__u8 YCOMB : 1; /*
			  select: luminance Chroma trap ex-or comb filter, effective only if BYPS = 0
			  0: chrominance trap is active (no comb)
			  1: luminance comb filter is active. CCOMB must be set to 1
			*/
	__u8 BYPS  : 1; /*
			  select: suppression of chrominance spectra in luminance signal by subtraction;
			  .	  effected frequency range is defined by LUBW and LCBW
			  0: chrominance trap or luminance comb filter enabled, as selected by YCOMB (default for CVBS)
			  1: chrominance trap or luminance comb filter bypassed (default for S-video)
			*/
} saa7174hl_r109_t, saa7174hl_luminance_control_t;

typedef struct {
	__u8 DBRI : 8; /*
			 value: luminance brightness control applies offset to luminance signal
			 ff: 255 (bright)
			 80: 128 (ITU level)
			 00: 0 (dark)
		       */
} saa7174hl_r10a_t, saa7174hl_luminance_brightness_t;

typedef struct {
	__u8 DCON : 8; /*
			 value: luminance contrast control applies gain to luminance signal
			 7f: 1.984 (maximum)
			 44: 1.063 (ITU level)
			 40: 1.0
			 00: 0 (luminance off), results to flat medium grey, if brightness set nominal
			 c0: -1.0 (inverse luminance)
			 80: -2.0 (inverse luminance)
		       */
} saa7174hl_r10b_t, saa7174hl_luminance_contrast_t;

typedef struct {
	__u8 DSAT : 8; /*
			 value: chrominance saturation control applies gain to color difference signal
			 7f: 1.984 (maximum)
			 40: 1.0 (ITU level)
			 00: 0 (color off)
			 c0: -1.0 (inverse chrominance)
			 80: -2.0 (inverse chrominance)
		       */
} saa7174hl_r10c_t, saa7174hl_chrominance_saturation_t;

typedef struct {
	__u8 HUEC : 8; /*
			 value: Color hue (phase) control
			 7f: +178.6....
			 00: ....0....
			 80: ....-180.0
		       */
} saa7174hl_r10d_t, saa7174hl_chroma_hue_control_t;

typedef struct {
	__u8 CCOMB : 1; /*
			  select: the adaptive (logic) chrominance comb filter compares the signals of neighboring lines
			  0: no comb filter for Chroma signal; frequency spectrum of color difference signal is
			  .  defined by CHBW and LCBW
			  1: adaptive chrominance comb filter is enabled; frequency spectrum of combed color
			  .  difference signal is defined by CHBW and LCBW (recommended)
			*/
	__u8 AUTO0 : 1; /*
			  00: automatic color standard detection is disabled; selection of color
			  .   standard is selected directly via CSTD[2:0]
			  01: automatic color standard detection is enabled (auto level 3); the detected
			  .   standard will also determine default setting for filter (LUBW, LCBW[2:0],
			  .   CHBW and DCVF) and sharpness control (LUF[3:0]) (recommended)
			  10: automatic color standard detection is enabled (auto level 2); the detected
			  .   standard will determine default settings for filter only (LUBW, LCBW[2:0],
			  .   CHBW and DCVF but not for sharpness; sharpness control LUF[3:0] can be
			  .   programmed individually
			  11: automatic chrominance standard detection is enabled (auto level 1); no default
			  .   setting for filter or sharpness is applied with detected standard; filter
			  .   setting and sharpness must be programmed individually
			*/
	__u8 FCTC  : 1; /*
			  value: time constant for color identification (Chroma gain adjustment)
			  0: normal time constant (recommended)
			  1: fast color detection for special applications (video source in nominal quality,
			  .  low noise, fast Chroma lock required, e.g. observation cameras); AUTO[1:0] must be 00
			*/
	__u8 DCVF  : 1; /*
			  disable chrominance vertical filter and PAL phase error correction
			  0: chrominance vertical filter, PAL phase error correction on (during active video lines)
			  1: chrominance vertical filter, PAL phase error correction permanently off
			*/
#define SAA7174HL_CSTD_PALB   0
#define SAA7174HL_CSTD_NTSCM  0
#define SAA7174HL_CSTD_NTSC50 1
#define SAA7174HL_CSTD_PAL60  1
#define SAA7174HL_CSTD_PALN   2
#define SAA7174HL_CSTD_NTSC60 2
#define SAA7174HL_CSTD_NTSCN  3
#define SAA7174HL_CSTD_PALM   3
#define SAA7174HL_CSTD_NTSCJ  4
#define SAA7174HL_CSTD_SECAM  5
	__u8 CSTD  : 3; /*
			  value: direct color standard selection in non AUTO-mode
			  000: PAL BGDHI (4.434 MHz)	    / NTSC M (3.580 MHz)
			  001: NTSC 4.43 (50 Hz)	    / PAL 4.43 (60 Hz)
			  010: combination-PAL N (3.582MHz) / NTSC 4.43 (60 Hz)
			  011: NTSC N (3.58 MHz)	    / PAL M (3.576 MHz)
			  100: do not use		    / NTSC-Japan (3.58 MHz)
			  101: SECAM			    / do not use
			*/
	__u8 CDTO  : 1; /*
			  select: reset of color subcarrier oscillator (DTO), e.g. to synchronize the subcarrier
			  .	  generation in a peripherally connected encoder via RTCO (optionally available
			  .	  on GPIO pin AUX2)
			  0: color subcarrier oscillator operates continuously (normal)
			  1: if CDTO is set to 1, the internal color subcarrier DTO is restarted from phase 0 degree,
			  .  and the RTCO output protocol generates a logic 0 at time slot 68, signaling the DTO
			  .  phase reset; the completion of this reset procedure is reported by clearing the CDTO
			  .  bit after first read access
			*/
} saa7174hl_r10e_t, saa7174hl_chroma_control_1_t;

typedef struct {
#define SAA7174HL_CGAIN_MINIMUM 0x00
#define SAA7174HL_CGAIN_NOMINAL 0x24
#define SAA7174HL_CGAIN_MAXIMUM 0x7F
	__u8 CGAIN : 7; /*
			  00: ACGC = 1; minimum gain (0.5)
			  24: ACGC = 1; nominal gain (1.125)
			  7f: ACGC = 1; maximum gain (7.5)
			*/
	__u8 ACGC  : 1; /*
			  select: automatic color gain control, referenced to subcarrier burst amplitude
			  0: automatic Chroma gain control ACGC = ON (recommended)
			  1: programmable gain via CGAIN[6:0]
			*/
} saa7174hl_r10f_t, saa7174hl_chroma_gain_control_t;

typedef struct {
	__u8 LCBW : 3; /*
			 combined luminance and chrominance bandwidth adjustment
			 000: small chrominance bandwidth, large luminance bandwidth
			 111: large chrominance bandwidth, small luminance bandwidth
		       */
	__u8 CHBW : 1; /*
			 chrominance bandwidth selection, in conjunction LCBW
			 0: smaller bandwidth for color difference signals U and V
			 1: higher bandwidth
		       */
	__u8 OFFV : 2; /*
			 value: Chroma offset control allows fine adjustment of color balance (colorless)
			 00: 0 LSB
			 01: +1/4 LSB
			 10: +1/2 LSB
			 11: +3/4 LSB
		       */
	__u8 OFFU : 2; /* see OFFV */
} saa7174hl_r110_t, saa7174hl_chroma_control_2_t;

typedef struct {
	__u8 YDEL : 3; /*
			 adjustment of pixel position, luminance to chrominance association (steps in 2/LLC)
			 100: luminance is 4 pixel positioned earlier
			 000:
			 011: luminance is 3 pixel positioned later
		       */
	__u8 _X_  : 1; /* reserved */
	__u8 HDEL : 2; /*
			 value: fine position of both HGATE-edges in step size of 2 LLC clock cycles
			 00: no extra delay
			 01: HGATE delayed by 2 LLC cycles, i.e. by one pixel position
			 10: HGATE delayed by 4 LLC cycles, i.e. 2 pixels
			 11: HGATE delayed by 6 LLC cycles, i.e. 3 pixels
		       */
	__u8 _XX_ : 1; /* reserved */
	__u8 COLO : 1; /*
			 select: color ON switch
			 0: color output is enabled if color is successfully decoded; if color subcarrieris too
			 .  weak or instable, the internal color killer will switch off color output (recommended)
			 1: color output is forced on, independent of internal color killer detection (for debug only)
		       */
} saa7174hl_r111_t, saa7174hl_mode_delay_control_t;

typedef __u8 saa7174hl_r112_t;

typedef __u8 saa7174hl_r113_t;

typedef struct {
	__u8 APCK  : 2; /*
			  select: clock phase for video ADCs
			  01: default
			*/
	__u8 AUTO1 : 1; /* see AUTO0 */
	__u8 _X_   : 1; /* reserved */
	__u8 AOSL  : 2; /*
			  select: analog output select
			  00: default, don't use different values
			*/
	__u8 UPTCV : 1; /*
			  select: time constant of video AGC, to modify gain
			  0: once per line
			  1: once per field
			*/
	__u8 CM99  : 1; /*
			  select: switch for RTCO protocol format
			  0: normal RTCO protocol (default) (recommended)
			  1: RTCO in SAA7199 compliant format; use only in application with SAA7199
			*/
} saa7174hl_r114_t, saa7174hl_analog_adc_t;

typedef struct {
	__u8 VGB : 8; /*
			value: begin (VGB and rising edge) of VGATE; count-offset to ITU line number = -5,
			.      to SMPTE line number = -2 see Figs 22 and 23
			range for 50 Hz/625 lines: 000H to 138H, i.e 0 to 312d
			range for 60 Hz/525 lines: 000H to 106H, i.e 0 to 262d
			other values are out of range, don't use
*/
} saa7174hl_r115_t, saa7174hl_vgate_start_t;

typedef struct {
	__u8 VGE : 8; /*
			value: end (VGE and falling edge) of VGATE; count-offset to ITU line number = -5,
			.      to SMPTE line number = -2 see Figs 22 and 23
			range for 50 Hz/625 lines: 000H to 138H, i.e 0 to 312d
			range for 60 Hz/525 lines: 000H to 106H, i.e 0 to 262d
			other values are out of range, don't use
		      */
} saa7174hl_r116_t, saa7174hl_vgate_stop_t;

typedef struct {
	__u8 VGB_MSB : 1; /* MSB of VGB */
	__u8 VGE_MSB : 1; /* MSB of VGE */
	__u8 VGPS    : 1; /*
			    value: alternative VGATE position for second field
			    0: second field is lower field, as common practice in 60 Hz systems
			    1: VGATE position in 2nd field occurs one line earlier; this switch shifts
			    .  the second field to the upper (top) field, appropriate for 50 Hz systems
			  */
	__u8 LATY    : 3; /*
			    value: speed for color standard detection in time in number of field to
			    .	   investigate for a certain standard)
			    000: illegal, do not use
			    001: 1 field time
			    010: 2 field time
			    011: 3 field time (recommended)
			    100: 4 field time
			    101: 5 field time
			    110: 6 field time
			    111: 7 field time
			  */
	__u8 _X_     : 2; /* reserved */
} saa7174hl_r117_t, saa7174hl_msb_misc_t;

typedef struct {
	__u8 RAWG : 8; /*
			 value: gain adjustment for RAW sample stream during VBI
			 7f: 255 (double amplitude)
			 50: 128 (nominal amplitude)
			 00: 0 (off)
		       */
} saa7174hl_r118_t, saa7174hl_raw_data_gain_t;

typedef struct {
	__u8 RAWO : 8; /*
			 00: -128 LSB
			 80: 0 LSB
			 ff: +128 LSB
		       */
} saa7174hl_r119_t, saa7174hl_raw_data_offset_t;

typedef __u8 saa7174hl_r11a_t;

typedef __u8 saa7174hl_r11b_t;

typedef __u8 saa7174hl_r11c_t;

typedef __u8 saa7174hl_r11d_t;

typedef struct {
#define SAA7174HL_DCSTD_NONE  0
#define SAA7174HL_DCSTD_NTSC  1
#define SAA7174HL_DCSTD_PAL   2
#define SAA7174HL_DCSTD_SECAM 3
	__u8 DCSTD : 2; /*
			  value: detected color standard
			  00: no color signal could be detected; the video is assumed being B/W
			  01: NTSC signal detected
			  10: PAL signal detected
			  11: SECAM signal detected
			*/
	__u8 WIPA  : 1; /*
			  0: white or color peak control loop is not activated
			  1: white or color peak control was triggered in previous line
			*/
	__u8 GLIMB : 1; /*
			  0: video AGC is in normal operation mode
			  1: video AGC is on its bottom limit. Input signal is too small, out of AGC range
			*/
	__u8 GLIMT : 1; /*
			  0: video AGC is in normal operation mode
			  1: video AGC is on its top limit, i.e. input signal is too large, out of AGC range
			*/
	__u8 SLTCA : 1; /*
			  0: video AGC is in normal operation mode
			  1: video AGC is in gain recover mode (increase) after peak attack
			*/
	__u8 HLCK  : 1; /*
			  0: HPLL is locked; clock LLC is locked to line frequency
			  1: HPLL is not locked
			*/
	__u8 _X_   : 1;
} saa7174hl_r11e_t, saa7174hl_status_byte_1_t;

typedef struct {
	__u8 RDCAP  : 1; /*
			   0: one or more of the decoder control loops are not locked
			   1: video decoding provides output stream ready for capture; all control loops
			   .  are locked; horizontal, vertical, color subcarrier and Chroma gain control
			 */
	__u8 COPRO  : 1; /*
			   0: normal video signal; no Macrovision copy protection scheme detected
			   1: copy protection detected according Macrovision (including 7.01)
			 */
	__u8 COLSTR : 1; /*
			   0: normal video signal
			   1: Macrovision color stripe scheme detected
			 */
	__u8 TYPE3  : 1; /*
			   0: normal video signal
			   1: Macrovision copy protection type 3 detected
			 */
	__u8 _X_    : 1; /* reserved */
	__u8 FIDT   : 1; /*
			   0: video input is detected with 50 Hz field rate
			   1: video input is detected with 60 Hz field rate
			 */
	__u8 HLVLN  : 1; /*
			   0: horizontal and vertical synchronization is achieved
			   1: either horizontal or vertical synchronization is lost
			 */
	__u8 INTL   : 1; /*
			   0: video input is detected as non-interlaced
			   1: video input is detected as interlaced
			 */
} saa7174hl_r11f_t, saa7174hl_status_byte_2_t;

typedef __u8 saa7174hl_r120_t;
typedef __u8 saa7174hl_r140_t;

typedef struct {
	__u8 IIC_STATUS : 4; /*
			       value: status information of I2C-bus interface
			       0: no I2C-bus command pending
			       1: I2C-bus command done and executed STOP condition
			       2: executing I2C-bus command
			       3: executing I2C-bus command, time out on clock stretching
			       4: time out on arbitration trial, still trying
			       5: I2C-bus command done and awaiting next write command
			       6: I2C-bus command done and awaiting next read command
			       7: see 5, and time out on status echo
			       8: see 6, and time out on status echo
			       9: no acknowledge on device slave address
			       a: no acknowledge after data byte transfer
			       b: bus error
			       c: arbitration lost during transfer
			       d: erroneous programming sequence
			       e: wrong status echoing
			     */
	__u8 _X_	: 2; /* reserved */
	__u8 ATTR	: 2; /*
			       value: transfer attribute associated with this command; the byte following the
			       .      START condition is the START-BYTE
			       00: NOP; no operation on I2C-bus (default)
			       01: STOP; issue a STOP condition, no associated byte transfer
			       10: CONTINUE; continue with BYTE transfer
			       11: START; issue a START condition with BYTE transfer
			     */
} saa7174hl_r180_t, saa7174hl_iic_status_t;

typedef struct {
	__u8 BYTE : 8; /*
			 value: byte to transfer; read or write on I2C-bus according LSB of most recent
			 .	START-BYTE, i.e. device (slave) address
		       */
} saa7174hl_r181_t, saa7174hl_byte_read_write_t;

typedef struct {
	__u8 _XX_      : 6; /* reserved */
	__u8 CLOCK_SEL : 1; /*
			      select: nominal I2C-bus clock rate
			      0: 'slow' (below) 100 kHz, 33.3 MHz/400
			      1: 'fast' (below) 400 kHz, 33.3 MHz/100 (default)
			    */
	__u8 _X_       : 1; /* reserved */
} saa7174hl_r182_t, saa7174hl_clock_select_t;

typedef struct {
	__u8 _X_   : 4; /* reserved */
	__u8 TIMER : 4; /*
			  value: timeout parameter, applicable for various sequential conditions to check;
			  .	 bus arbitration, SCL stretching and DONE status echoing; timeout = I2C-bus
			  .	 clock period x 2^(timer)
			  0: timeout disabled: no check for any time (default)
			  f: max. timeout = IIC clock period x 2^15
			*/
} saa7174hl_r183_t, saa7174hl_timer_t;

typedef __u8 saa7174hl_r184_t;

typedef struct {
	__u8 _X_	 : 2; /* reserved */
	__u8 VP_V_CODE_P : 1; /*
				select: ITU/VIP code; V-bit polarity
				0: non inverted V-bit (default)
				1: inverted V-bit
			      */
	__u8 VP_V_CODE	 : 2; /*
				value: ITU/VIP code; V-bit content
				00: V_ITU (VBLNK_CCIR_L) from decoder (default)
				01: indicates VBI data at scaler output (VBI -> V = 1)
				10: indicates active video data at scaler output (active video -> V = 0)
				11: VGATE_L from decoder
			      */
	__u8 VP_F_CODE_P : 1; /*
				0: non inverted F-bit (default)
				1: inverted F-bit
			      */
	__u8 VP_F_CODE	 : 2; /*
				value: ITU/VIP code; F-bit content
				00: F_ITU from decoder (default)
				01: Scaler page: A -> F = 1, B -> F = 0
				10: F = scaler field ID (decoder dependant)
				11: F = scaler region field ID (Scaler dependant, corrected field ID in case
				.   of field rate conversion)
			      */
} saa7174hl_r190_t, saa7174hl_video_port_control_1_t;

typedef struct {
	__u8 VP_T_CODE_P   : 1; /*
				  select: VIP code; T-Bit polarity
				  0: non inverted T-bit
				  1: inverted T-bit
				*/
	__u8 VP_T_CODE	   : 2; /*
				  select: VIP code; T-bit content
				  00: the T-bit is constant '1'
				  01: results from the SCALER PAGE (page A -> T = 1 'b1)
				  10: indicates VBI data at scaler output (VBI -> T = 1 'b1)
				  11: VGATE_L from decoder
				*/
	__u8 VP_KEEP	   : 1; /*
				  select: keep last video data set stable at the video output until new valid data
				  .	  is send; required for e.g. VMI or DMSD Video formats
				  0: non qualified video data samples are carrying 00H (blanking) values during
				  .  horizontal active video (default)
				  1: non qualified video data samples are carrying the same value as the last qualified
				  .  video data sample during horiz. active video; additionally: the data cycle time of
				  .  active video data from scaler (NOT RAWVBI data) is set to two 27 MHz clock cycles
				  .  in case of a 16-bit output mode (mu_vp_en_video must be  11  bin) I.e. each active
				  .  video data set from the scaler is spread over two 27 MHz clock cycles
				*/
	__u8 VP_BLANK_8010 : 1; /*
				  select: blanking values
				  0: blanking values: Luma = 00H, Chroma = 00H (default)
				  1: blanking values: Luma = 10H, Chroma = 80H
				*/
	__u8 VP_GEN_HAMM   : 1; /*
				  select: parity code bits for ITU/VIP (T-), V-, F- and H bits in SAV/EAV codes
				  0: don't generate hamming code bits
				  1: generate hamming code bits
				*/
	__u8 VP_EN_CODE_C  : 1; /*
				  select: insert ITU/VIP codes on port C (i.e. GPIO[7:0])
				  0: don't insert EAV SAV codes on port C
				  1: insert EAV SAV codes on port C
				*/
	__u8 VP_EN_CODE_A  : 1; /*
				  select: ITU/VIP codes on port A (i.e. GPIO[15:8])
				  0: don't insert EAV SAV codes on port A (default)
				  1: insert EAV SAV codes on port A
				*/
} saa7174hl_r191_t, saa7174hl_video_port_control_2_t;

typedef struct {
	__u8 VP_DEC_RAW_VBI : 2; /*
				   select: video decoder (i.e. DMSD) source stream selector depending on the value
				   .	   of this 2-bit register, the V_ITU vertical reference signal and the
				   .	   programmable VGATE_L (programmed in the DMSD munit) the DMSD will output
				   .	   comb-filtered YUV data, straight (non comb-filtered) YUV data or RAW (non
				   .	   decoded) CVBS data (see Table 127); note: this effects the input of the scaler
				   .	   as well as the GPIO video port!
				   00: default
				 */
	__u8 _X_	    : 1; /* reserved */
	__u8 VP_EVAL_LOCK   : 1; /*
				   select: enable GPIO video out depending on the evaluation of decoder lock status
				   0: output video data from sources 'as is' (default)
				   1: video decoder lock status locked -> output video data 'as is', video decoder
				   .  lock status NOT locked -> output blanking values (as specified by 'VP_BLANK_8010')
				   .  and remove enabled sync signals (tie to 'inactive')
				 */
	__u8 _XX_	    : 4; /* reserved */
} saa7174hl_r192_t, saa7174hl_video_port_control_3_t;

typedef struct {
	__u8 VP_EN_DMSD	   : 1; /*
				  select: enables routing of decoder output to video output port, if no other stream
				  .	  is enabled and gated
				  0: routing disabled (default)
				  1: routing enabled
				*/
	__u8 VP_EN_XV	   : 1; /*
				  select: enables generating SAV/EAV codes according to programmable VGATE_L VIP 1.1
				  .	  requires to have (T-,) F- and V-bit for codes changed in an EAV ahead of first
				  .	  active (selected) video line (etc.); the video stream is taken from the decoder
				  0: SAV/EAV generation disabled (default)
				  1: SAV/EAV generation enabled
				*/
	__u8 VP_EN_SC_ACT  : 1; /*
				  select: enables routing of scaler output data to video output port, during VBI range
				  .	  from scaler only; during VBI range the data format is Y-only
				  0: routing disabled (default)
				  1: routing enabled
				*/
	__u8 VP_EN_SC_VBI  : 1; /*
				  select: enables routing of scaler output data to video output port, during VBI range
				  .	  from scaler only; during VBI range the data format is Y-only
				  0: routing disabled (default)
				  1: routing enabled
				*/
	__u8 VP_EN_LC_SWAP : 1; /*
				  select: swap Chroma and Luma data at GPIO video output port; this swap effects active
				  .	  video data as well as blanking values, but NOT THE SAV AND EAV CODES
				  0: 8-bit video modes (at port A or C): sequence: U-Y-V-Y- ... 16-bit video modes:
				  .  sequence: Y-Y- ... (at port A); U-V- ... (at port C) (default)
				  1: 8-bit video modes (at port A or C): sequence: Y-U-Y-V ... 16-bit video modes:
				  .  sequence: U-V ... (at port A); Y-Y- ... (at port C)
				*/
#define SAA7174HL_VP_EN_VIDEO_DISABLED 0
#define SAA7174HL_VP_EN_VIDEO_PORT_A   1
#define SAA7174HL_VP_EN_VIDEO_PORT_C   2
#define SAA7174HL_VP_EN_VIDEO_16BIT    3
	__u8 VP_EN_VIDEO   : 2; /*
				  select: GPIO video output port enable
				  00: GPIO video port disabled (no GPIO pin occupied for video data) (default)
				  01: 8-bit video output mode on port A (GPIO [15:8]) (port C not occupied)
				  10: 8-bit video output mode on port C (GPIO [7:0]) (port A not occupied)
				  11: 16-bit video output mode on port A (Luma) and C (Chroma)
				*/
	__u8 VP_EN_PORT	   : 1; /*
				  select: enable the video output port
				  0: video port is disabled (default)
				  1: video port is enabled
				*/
} saa7174hl_r193_t, saa7174hl_video_port_control_4_t;

typedef struct {
	__u8 VP_SQUEEZ_CNT : 8;
} saa7174hl_r194_t, saa7174hl_video_port_control_5_t;

typedef struct {
	__u8 VP_CLK_CTRL : 4;
	__u8 PICLK_OFF	 : 1;
	__u8 _X_	 : 3;
} saa7174hl_r195_t, saa7174hl_video_port_control_6_t;

typedef struct {
	__u8 VP_VS_TYPE : 3;
	__u8 VP_VS_P	: 1;
	__u8 VP_HS_TYPE : 2;
	__u8 VP_HS_P	: 1;
	__u8 _X_	: 1;
} saa7174hl_r196_t, saa7174hl_video_port_control_7_t;

typedef struct {
	__u8 _X_	 : 3;
	__u8 VP_AUX2_FKT : 2;
	__u8 VP_EN_AUX	 : 3;
} saa7174hl_r197_t, saa7174hl_video_port_control_8_t;

typedef struct {
	__u8 VSB_B_EN	 : 1;
	__u8 VSB_A_EN	 : 1;
	__u8 SWP_VSB_SRC : 1;
	__u8 _X_	 : 5;
} saa7174hl_r198_t, saa7174hl_video_port_control_9_t;

typedef __u8 saa7174hl_r199_t;
typedef __u8 saa7174hl_r1a0_t;

typedef struct {
	/* video scaler */
	saa7174hl_r000_t source_timing_reference_1;
	saa7174hl_r001_t source_timing_reference_2;
	saa7174hl_r002_t reserved_r002;
	saa7174hl_r003_t reserved_r003;
	saa7174hl_r004_t region_enable;
	saa7174hl_r005_t reserved_r005;
	saa7174hl_r006_t scaler_status_0;
	saa7174hl_r007_t scaler_status_1;
	saa7174hl_r008_t reserved_r008;
	saa7174hl_r009_t reserved_r009;
	saa7174hl_r00a_t reserved_r00a;
	saa7174hl_r00b_t reserved_r00b;
	saa7174hl_r00c_t start_point_green_path;
	saa7174hl_r00d_t start_point_blue_path;
	saa7174hl_r00e_t start_point_red_path;
	saa7174hl_r00f_t reserved_r00f;
	saa7174hl_r010_t green_path_gamma_curve[16];
	saa7174hl_r020_t blue_path_gamma_curve[16];
	saa7174hl_r030_t red_path_gamma_curve[16];
	/* task definitions */
	saa7174hl_scaler_t taska;
	saa7174hl_scaler_t taskb;
	/* reserved */
	saa7174hl_r0c0_t reserved_r0c0[0x40];
	/* video decoder */
	saa7174hl_r100_t reserved_r100;
	saa7174hl_r101_t increment_delay;
	saa7174hl_r102_t analog_input_control_1;
	saa7174hl_r103_t analog_input_control_2;
	saa7174hl_r104_t analog_input_control_3;
	saa7174hl_r105_t analog_input_control_4;
	saa7174hl_r106_t horizontal_sync_start;
	saa7174hl_r107_t horizontal_sync_stop;
	saa7174hl_r108_t sync_control;
	saa7174hl_r109_t luminance_control;
	saa7174hl_r10a_t luminance_brightness;
	saa7174hl_r10b_t luminance_contrast;
	saa7174hl_r10c_t chroma_saturation;
	saa7174hl_r10d_t chroma_hue_control;
	saa7174hl_r10e_t chroma_control_1;
	saa7174hl_r10f_t chroma_gain_control;
	saa7174hl_r110_t chroma_control_2;
	saa7174hl_r111_t mode_delay_control;
	saa7174hl_r112_t reserved_r112;
	saa7174hl_r113_t reserved_r113;
	saa7174hl_r114_t analog_adc;
	saa7174hl_r115_t vgate_start;
	saa7174hl_r116_t vgate_stop;
	saa7174hl_r117_t msb_misc;
	saa7174hl_r118_t raw_data_gain;
	saa7174hl_r119_t raw_data_offset;
	saa7174hl_r11a_t reserved_r11a;
	saa7174hl_r11b_t reserved_r11b;
	saa7174hl_r11c_t reserved_r11c;
	saa7174hl_r11d_t reserved_r11d;
	saa7174hl_r11e_t status_byte_1;
	saa7174hl_r11f_t status_byte_2;
	saa7174hl_r120_t reserved_r120[0x20];
	/* audio and sound processing */
	saa7174hl_r140_t audio[0x40];
	/* I2C-bus port */
	saa7174hl_r180_t iic_status;
	saa7174hl_r181_t byte_read_write;
	saa7174hl_r182_t clock_select;
	saa7174hl_r183_t timer;
	saa7174hl_r184_t reserved_r184[12];
	/* video output port */
	saa7174hl_r190_t video_port_control_1;
	saa7174hl_r191_t video_port_control_2;
	saa7174hl_r192_t video_port_control_3;
	saa7174hl_r193_t video_port_control_4;
	saa7174hl_r194_t video_port_control_5;
	saa7174hl_r195_t video_port_control_6;
	saa7174hl_r196_t video_port_control_7;
	saa7174hl_r197_t video_port_control_8;
	saa7174hl_r198_t video_port_control_9;
	saa7174hl_r199_t reserved_r199[7];
	/* peripherals */
	saa7174hl_r1a0_t peripherals[0x40];
} saa7174hl_reg_map_t;

#endif /* __SAA7174HL_H */
