 /*
    tw98.c - generic i2c-driver for the
     
     - tw98 Analog to Digital Video Decoder
    
    Copyright (C) 2000 H. S. Magnuski <hankm@mtinet.com>

    Modifications to work with RivaTV - 
      Steven Ellis <steven_ellis@users.sourceforge.net>
      Stefan Jahn <stefan@lkcc.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>

#include <linux/i2c.h>
#include <linux/videodev.h>
#include <linux/video_decoder.h>

#include "rivatv-kcompat.h"
#include "tw98.h"

#define TW98_I2C_VERSION_ID 0xC0
#define	I2C_TW98	    0x44
#define TW98_PREFIX         TW98_DEVICE_NAME ": "

/* Crzstal speed option */
static int clock = 1;

/* Debug option */
static int debug = 0;

#define TW98_PRINTK(format, args...)                        \
        printk (TW98_PREFIX format, ## args)
#define TW98_DEBUG_PRINTK(format, args...) if (debug >= 1)  \
        printk (TW98_PREFIX format, ## args)
#define TW98_DEBUG_PRINTK2(format, args...) if (debug >= 2) \
        printk (TW98_PREFIX format, ## args)
#undef	PRINTK
#define PRINTK(format, args...)				    \
	printk (format, ## args)
#undef	DPRINTK
#define DPRINTK(format, args...) if (debug >= 1)	    \
	printk (format, ## args)
#undef	DPRINTK2
#define DPRINTK2(format, args...) if (debug >= 2)	    \
	printk (format, ## args)


static int tw98_attach  (struct i2c_adapter *);
static int tw98_detach  (struct i2c_client *);
static int tw98_command (struct i2c_client *, unsigned int, void *);

/* Possible TW98 addresses to scan, found only at 0x88 and/or 0x8a */
static unsigned short normal_i2c[] = { I2C_TW98, I2C_TW98 + 1, I2C_CLIENT_END };
static unsigned short normal_i2c_range[] = { I2C_CLIENT_END };

/* Magic definition of all other variables and things */
I2C_CLIENT_INSMOD;

/* TW98 instance */
struct tw98 {
	int revision;
};

static struct i2c_driver i2c_driver_tw98 = {
	.name		= "TW98",
	.id		= I2C_DRIVERID_TW98,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter = tw98_attach,
	.detach_client	= tw98_detach,
	.command	= tw98_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use	= i2c_client_inc_use,
	.dec_use	= i2c_client_dec_use,
#else
	.owner		= THIS_MODULE,
#endif
};

static struct i2c_client tw98_client = {
	I2C_DEVNAME ("Techwell TW98"),
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	.id      = -1,
#endif
	.flags   = I2C_CLIENT_ALLOW_USE,
	.addr    = 0,
	.adapter = NULL,
	.driver  = &i2c_driver_tw98,
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
/* Unique ID allocation */
static int tw98_id = 0;
#endif

/* This functions sets partial bits of the register 'subaddress' from the
   i2c-client 'client'. Valid values of 'arg' may range between
   'range_begin' and 'range_end'; the corresponding bits to be set
   range from 'bit_begin' to 'bit_end' */
int tw98_set_register (struct i2c_client *client, int arg, int subaddress, 
		       int range_begin, int range_end,
		       int bit_begin, int bit_end) 
{
	s32 byte = 0x0;
	int i = 0, result = 0;

	/* be careful, check arguments :-) */
	if (arg < range_begin || arg > range_end)
		return -EINVAL;
	
	/* get the old register contents */
	if (-1 == (byte = i2c_smbus_read_byte_data (client, subaddress))) {
		TW98_PRINTK ("FAILED reading register 0x%02x\n", subaddress);
		return -EFAULT;
	}

	/* clear out corresponding bits */
	for (i = bit_begin; i <= bit_end; i++)
		byte &= ~(1 << i);

	/* set these bits new */
	byte |= (arg << bit_begin);

	/* write register contents back */
	if (0 != (result = i2c_smbus_write_byte_data (client, subaddress, byte))) {
		TW98_PRINTK ("FAILED writing register 0x%02x\n", subaddress);
		return -EFAULT;
	}
	
	/* everything's fine */
	return 0;
}

/* Read from a register */
int tw98_get_register (struct i2c_client *client, int subaddress)
{
	int val = i2c_smbus_read_byte_data (client, subaddress);

	if (val < 0) {
		TW98_PRINTK ("FAILED reading register 0x%02x\n", subaddress);
		return -EFAULT;
	}
	return val;
}

/* This functions sets a range of 5 registers, used to switch from NTSC to PAL */
static void tw98_set_range (struct i2c_client *client, int subaddress, int value_0,
			    int value_1, int value_2, int value_3, int value_4)
{
	tw98_set_register (client, value_0, subaddress + 0, 0, 255, 0, 7);
	tw98_set_register (client, value_1, subaddress + 1, 0, 255, 0, 7);
	tw98_set_register (client, value_2, subaddress + 2, 0, 255, 0, 7);
	tw98_set_register (client, value_3, subaddress + 3, 0, 255, 0, 7);
	tw98_set_register (client, value_4, subaddress + 4, 0, 255, 0, 7);
}

/* Used to setup crystal speed for different video standards */
static void tw98_set_clock (struct i2c_client *client, int norm)
{
	struct tw98 *decoder = (struct tw98 *) i2c_get_clientdata (client);

	switch (norm) {
	case VIDEO_MODE_NTSC:
		if (decoder->revision <= 4) {
			if (clock == 1) {
				/* Set Nominal, Maximum and Minimum TSC values */
				tw98_set_range (client, TW98_TSC1,  0x01, 0xeb, 0xae, 0xbf, 0x03);
				tw98_set_range (client, TW98_TSCM1, 0x01, 0xeb, 0xdc, 0x9f, 0x12);
				tw98_set_range (client, TW98_TSCL1, 0x01, 0xeb, 0x80, 0xde, 0xf8);
				tw98_set_register (client, 0xd4, TW98_SYNCL, 0, 255, 0, 7);
			}
			else if (clock == 0) {
				/* Set Nominal, Maximum and Minimum TSC values */
				tw98_set_range (client, TW98_TSC1,  0x01, 0xe2, 0xbe, 0x2f, 0xe7);
				tw98_set_range (client, TW98_TSCM1, 0x01, 0xe2, 0xe2, 0x46, 0x9b);
				tw98_set_range (client, TW98_TSCL1, 0x01, 0xe2, 0x9d, 0x39, 0x97);
				tw98_set_register (client, 0xb4, TW98_SYNCL, 0, 255, 0, 7);
			}
		}
		else if (decoder->revision == 5) {
			tw98_set_register (client, 0x38, TW98_BURST, 0, 255, 0, 7);
		}
		break;
	case VIDEO_MODE_SECAM:
	case VIDEO_MODE_PAL:
		if (decoder->revision <= 4) {
			if (clock == 1) {
				/* Set Nominal, Maximum and Minimum TSC values */
				tw98_set_range (client, TW98_TSC1,  0x01, 0x8c, 0xf7, 0x88, 0x5e);
				tw98_set_range (client, TW98_TSCM1, 0x01, 0x8d, 0x18, 0x4c, 0xfa);
				tw98_set_range (client, TW98_TSCL1, 0x01, 0x8c, 0xd6, 0xc3, 0xc3);
				tw98_set_register (client, 0xe0, TW98_SYNCL, 0, 255, 0, 7);
			}
			else if (clock == 0) {
				/* Set Nominal, Maximum and Minimum TSC values */
				tw98_set_range (client, TW98_TSC1,  0x01, 0x85, 0xbf, 0xd5, 0x04);
				tw98_set_range (client, TW98_TSCM1, 0x01, 0x85, 0xd6, 0x57, 0x70);
				tw98_set_range (client, TW98_TSCL1, 0x01, 0x85, 0xa9, 0x55, 0x32);
				tw98_set_register (client, 0xc0, TW98_SYNCL, 0, 255, 0, 7);
			}
		}
		else if (decoder->revision == 5) {
			tw98_set_register (client, 0x28, TW98_BURST, 0, 255, 0, 7);
		}
		break;
	}
}

/* Dump all registers */
static void tw98_dump_regs (struct i2c_client *client)
{
	int i, j;

	if (debug >= 2) {
		for (i = 0; i < TW98_NR_REGISTER; i += 16) {
			TW98_DEBUG_PRINTK2 ("0x%02x |", i);
			for (j = 0; j < 16 && i + j  < TW98_NR_REGISTER; j++) {
				DPRINTK2 (" %02x", tw98_get_register (client, i + j));
			}
			DPRINTK2 ("\n");
		}
	}
}

/* Setup the scaler */
static void tw98_setup_scaler (struct i2c_client *client,
			       int src_x, int src_y, int src_width, int src_height,
			       int dst_width, int dst_height)
{
	int scale;

	/* adjust vertical region */
	tw98_set_register (client, src_y & 0xff, TW98_VDELAY_LO, 0, 255, 0, 7);
	tw98_set_register (client, src_y >> 8,   TW98_CROP_HI,   0,   3, 6, 7);
	tw98_set_register (client, dst_height & 0xff, TW98_VACTIVE_LO, 0, 255, 0, 7);
	tw98_set_register (client, dst_height >> 8,   TW98_CROP_HI,    0,   3, 4, 5);
	scale = src_height * 256 / dst_height;
	tw98_set_register (client, scale & 0xff, TW98_VSCALE_LO, 0, 255, 0, 7);
	tw98_set_register (client, scale >> 8,   TW98_VSCALE_HI, 0,  15, 0, 4);
	
	/* adjust horizontal region */
	tw98_set_register (client, src_x & 0xff, TW98_HDELAY_LO, 0, 255, 0, 7);
	tw98_set_register (client, src_x >> 8,   TW98_CROP_HI,   0,   3, 2, 3);
	tw98_set_register (client, dst_width & 0xff, TW98_HACTIVE_LO, 0, 255, 0, 7);
	tw98_set_register (client, dst_width >> 8,   TW98_CROP_HI,    0,   3, 0, 1);
	scale = src_width * 256 / dst_width;
	tw98_set_register (client, scale & 0xff, TW98_HSCALE_LO, 0, 255, 0, 7);
	tw98_set_register (client, scale >> 8,   TW98_HSCALE_HI, 0, 255, 0, 7);
}

/* Interface to the world */
static int tw98_command (struct i2c_client *client, unsigned int cmd, void *arg)
{
	switch (cmd)
	{

	case DECODER_GET_CAPABILITIES:
	{
                struct video_decoder_capability *cap = arg;

                TW98_DEBUG_PRINTK ("DECODER_GET_CAPABILITIES\n");
		cap->flags
			= VIDEO_DECODER_PAL
                        | VIDEO_DECODER_SECAM
                        | VIDEO_DECODER_NTSC
                        | VIDEO_DECODER_AUTO
                        | VIDEO_DECODER_CCIR;
                cap->inputs = 4;
                cap->outputs = 0;
                break;
	}

	case DECODER_GET_STATUS:
	{
		int *iarg = arg;
                int res, reg;

		reg = tw98_get_register (client, TW98_CSTATUS);
                TW98_DEBUG_PRINTK ("DECODER_GET_STATUS = 0x%02x ->", reg);
                res = 0;

		if (reg & 0x80) {
			res |= DECODER_STATUS_GOOD;
			DPRINTK (" GOOD");
		} else {
			DPRINTK (" BAD");
		}

		reg = tw98_get_register (client, TW98_AUTODETECT);
		if (reg & 0x01) {
			DPRINTK (" AUTODETECT");
		}
		reg = tw98_get_register (client, TW98_INFORM);
		if ((reg & 0x02) == 0x00) {
			res |= DECODER_STATUS_NTSC;
			DPRINTK (" NTSC-M");
		} else if ((reg & 0x02) == 0x01) {
			res |= DECODER_STATUS_PAL;
			DPRINTK (" PAL-B");
		}
		else {
			DPRINTK (" <invalid PAL/NTSC>");
		}

		DPRINTK ("\n");
		tw98_dump_regs (client);
		*iarg = res;
		break;
	}

	case DECODER_SET_NORM:
	{
                int *iarg = arg;
		int result;

                TW98_DEBUG_PRINTK ("DECODER_SET_NORM: ");

                switch (*iarg)
                {

                case VIDEO_MODE_NTSC:
                        DPRINTK ("NTSC");
			result  = tw98_set_register (client, 0, TW98_AUTODETECT,
						     0,3,		/* valid values: 0 to 3 */
						     0,2);		/* bits to be used: 0-2 */
			result  = tw98_set_register (client, 0, TW98_INFORM,
						     0,1,		/* valid values: 0 to 1 */
						     0,1);		/* bits to be used: 0-1 */

			result  = tw98_set_register (client, 0x38, TW98_HSYNC,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
			result  = tw98_set_register (client, 0xd4, TW98_LLENGTH,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
			result  = tw98_set_register (client, 0x62, TW98_SYNCTRL,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
			result  = tw98_set_register (client, 3, TW98_VFORMAT,
						     0,3,		/* valid values: 0 to 3 */
						     6,7);		/* bits to be used: 6-7 */

			tw98_set_clock (client, *iarg);
			tw98_setup_scaler (client, 0, 24, 766, 480, 720, 480);
                        break;

                case VIDEO_MODE_PAL:
                        DPRINTK ("PAL");
			result  = tw98_set_register (client, 0, TW98_AUTODETECT,
						     0,3,		/* valid values: 0 to 3 */
						     0,2);		/* bits to be used: 0-2 */
			result  = tw98_set_register (client, 1, TW98_INFORM,
						     0,1,		/* valid values: 0 to 1 */
						     0,1);		/* bits to be used: 0-1 */

			result  = tw98_set_register (client, 0x28, TW98_HSYNC,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
			result  = tw98_set_register (client, 0xe0, TW98_LLENGTH,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
			result  = tw98_set_register (client, 0x00, TW98_SYNCTRL,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
			result  = tw98_set_register (client, 0, TW98_VFORMAT,
						     0,3,		/* valid values: 0 to 3 */
						     6,7);		/* bits to be used: 6-7 */

			tw98_set_clock (client, *iarg);
			tw98_set_register (client, 0x01, 0xc0, 0, 255, 0, 7);
			tw98_set_register (client, 0x01, 0xc6, 0, 255, 0, 7);
			tw98_set_register (client, 0x01, 0xcc, 0, 255, 0, 7);
			tw98_setup_scaler (client, 0, 24, 946, 576, 704, 576);
                        break;

                case VIDEO_MODE_SECAM:
                        DPRINTK ("SECAM");
			result  = tw98_set_register (client, 0, TW98_AUTODETECT,
						     0,3,		/* valid values: 0 to 3 */
						     0,2);		/* bits to be used: 0-2 */
			result  = tw98_set_register (client, 1, TW98_INFORM,
						     0,1,		/* valid values: 0 to 1 */
						     0,1);		/* bits to be used: 0-1 */

			tw98_set_clock (client, *iarg);
			tw98_setup_scaler (client, 0, 24, 946, 576, 704, 576);
                        break;

                case VIDEO_MODE_AUTO:
                        DPRINTK ("AUTODETECT");
			result  = tw98_set_register (client, 3, TW98_AUTODETECT,
						     0,3,		/* valid values: 0 to 3 */
						     0,2);		/* bits to be used: 0-2 */
                        break;

                default:
                        DPRINTK ("<invalid function>\n");
                        return -EINVAL;

                }
                PRINTK ("\n");
                break;
	}

	case DECODER_SET_INPUT:
	{
                int *iarg = arg;
		int result;

                TW98_DEBUG_PRINTK ("DECODER_SET_INPUT: %d", *iarg);

		result = tw98_set_register (client, *iarg, TW98_INFORM,
					    0,7,	/* valid values: 0-7 */
					    2,4);	/* bits to be used: 2-4 */

                /* set S-Video Mode */
		if (*iarg >= 4) {
			result = tw98_set_register (client, 1, TW98_INFORM,
						    0,1,	/* valid values: 0-1 */
						    4,4);	/* bit to be used: 4 */
                        DPRINTK (" S-Video");
                } else {
			result = tw98_set_register (client, 0, TW98_INFORM,
						    0,1,	/* valid values: 0-1 */
						    4,4);	/* bit to be used: 4 */
                        DPRINTK (" CVBS");
                }
                DPRINTK ("\n");
                break;
	}

	case DECODER_SET_OUTPUT:
	{
                int *iarg = arg;

                TW98_DEBUG_PRINTK ("DECODER_SET_OUTPUT: %d\n", *iarg);

                /* not much choice of outputs */
                if (*iarg != 0) {
                        return -EINVAL;
                }
                break;
	}

	case DECODER_ENABLE_OUTPUT:
	{
                int *iarg = arg;
		int result;

                TW98_DEBUG_PRINTK ("DECODER_ENABLE_OUTPUT: %d\n", *iarg);

		if (*iarg) {
			result  = tw98_set_register (client, 0xa2, TW98_OPFORM,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
		} else {
			result  = tw98_set_register (client, 0xa6, TW98_OPFORM,
						     0,255,		/* valid values: 0 to 255 */
						     0,7);		/* bits to be used: 0-7 */
		}
		break;
	}

	case DECODER_SET_PICTURE:
	{
                struct video_picture *pic = arg;
		int result;

                TW98_DEBUG_PRINTK ("DECODER_SET_PICTURE\n");

		result  = tw98_set_register (client, (pic->brightness - 32768) >> 8, TW98_BRIGHTNESS,
					     0,255,		/* valid values: 0 to 255 */
					     0,7);		/* bits to be used: 0-7 */

		result  = tw98_set_register (client, pic->contrast >> 10, TW98_CONTRAST,
					     0,63,		/* valid values: 0 to 63 */
					     0,5);		/* bits to be used: 0-5 */

		result  = tw98_set_register (client, pic->colour >> 8, TW98_SAT_U,
					     0,255,		/* valid values: 0 to 255 */
					     0,7);		/* bits to be used: 0-7 */

		result  = tw98_set_register (client, pic->colour >> 8, TW98_SAT_V,
					     0,255,		/* valid values: 0 to 255 */
					     0,7);		/* bits to be used: 0-7 */

		result  = tw98_set_register (client, pic->hue >> 10, TW98_HUE,
					     0,63,		/* valid values: 0 to 63 */
					     0,5);		/* bits to be used: 0-5 */
                break;
	}

#ifdef DECODER_DUMP
	case DECODER_DUMP:
                TW98_DEBUG_PRINTK ("DECODER_DUMP\n");
		tw98_dump_regs (client);
		break;
#endif

#ifdef GET_NR_OF_REGISTERS
        case GET_NR_OF_REGISTERS:
                TW98_DEBUG_PRINTK ("NUM_REGISTERS = %d\n", TW98_NR_REGISTER);
                return TW98_NR_REGISTER;
#endif

	default:
                TW98_DEBUG_PRINTK ("<invalid function> = 0x%08x\n", cmd);
		return -EINVAL;
	}
	return 0;
}

/* This function initalizes a TW98 to useful values */
static int tw98_initialize (struct i2c_client* client, int revision)
{
	int 	i = 0, result = 0;

#define RESET 32
	u8	s[RESET][2] = {	{ TW98_ACNTL,		0x80 }, /* 0x06 */
				{ TW98_INFORM,		0x61 }, /* 0x02 */
				{ TW98_OPFORM,		0xa2 }, /* 0x03 */ /* 0x86 for BRS direct */
				{ TW98_HSLEN,		0x40 }, /* 0x04 */
				{ TW98_POLARITY,	0x00 }, /* 0x05 */
				{ TW98_CROP_HI,		0x22 }, /* 0x07 */
				{ TW98_VDELAY_LO,	0x18 }, /* 0x08 */
				{ TW98_VACTIVE_LO,	0x40 }, /* 0x09 */
				{ TW98_HDELAY_LO,	0x00 }, /* 0x0a */
				{ TW98_HACTIVE_LO,	0xc0 }, /* 0x0b */
				{ TW98_VSCALE_HI,	0x01 }, /* 0x0c */
				{ TW98_VSCALE_LO,	0x00 }, /* 0x0d */
				{ TW98_HSCALE_HI,	0x01 }, /* 0x0e */
				{ TW98_HSCALE_LO,	0x58 }, /* 0x0f */
				{ TW98_BRIGHTNESS,	0xca }, /* 0x10 */
				{ TW98_CONTRAST,	0x2f }, /* 0x11 */
				{ TW98_SHARPNESS,	0x33 }, /* 0x12 */
				{ TW98_SAT_U,		0x71 }, /* 0x13 */
				{ TW98_SAT_V,		0x50 }, /* 0x14 */
				{ TW98_HUE,		0x20 }, /* 0x15 */
				{ TW98_CKILL,		0xd0 }, /* 0x16 */
				{ TW98_CORING,		0x02 }, /* 0x17 */
				{ TW98_VBICNTL,		0x18 }, /* 0x19 */
				{ TW98_CC_STATUS,	0xc0 }, /* 0x1a */
				{ TW98_AGCGAIN,		0x21 }, /* 0x1c */
				{ TW98_GPIO,		0xf0 }, /* 0x1d */
				{ TW98_NOISE,           0x05 }, /* 0x1f */
#if 1
				{ TW98_SYNCL,		0x14 }, /* 0xd3 */
				{ TW98_SHARP1,		0x18 }, /* 0xd8 */
				{ TW98_LLENGTH,		0xb4 }, /* 0xdf */
				{ TW98_HSK,		0x04 }, /* 0xe2 */
				{ TW98_SYNCTRL,		0x62 }  /* 0xe6 */
#endif
				};

	/* initial registers */
	TW98_DEBUG_PRINTK ("initializing\n");
	for (i = 0; i < RESET; i++) {
		if (0 != (result = i2c_smbus_write_byte_data (client, s[i][0], s[i][1]))) {
			TW98_PRINTK ("init error - reset on %d\n", i);
			return -EFAULT;
		}
	}

	/* initial registers depending on the chip revision */
	if (revision <= 4) {
		i2c_smbus_write_byte_data (client, TW98_HPLLC, 0x02);
		i2c_smbus_write_byte_data (client, TW98_COMPCTRL, 0x48);
		i2c_smbus_write_byte_data (client, TW98_SHARP1, 0x1f);
		i2c_smbus_write_byte_data (client, TW98_PLLC, 0x63);
		i2c_smbus_write_byte_data (client, TW98_OUTCTRL, 0x01);
		i2c_smbus_write_byte_data (client, TW98_HSYNC, 0x10);
	}
	else if (revision == 5) {
		i2c_smbus_write_byte_data (client, TW98_OUTCTRL, 0x01);
		i2c_smbus_write_byte_data (client, TW98_SHARP2, 0xf4);
		i2c_smbus_write_byte_data (client, TW98_HFK, 0x26);
		i2c_smbus_write_byte_data (client, TW98_SYNCTRL, 0x42);
		i2c_smbus_write_byte_data (client, 0xed, 0x42);
	}
	return 0;
}

/* This function is called by i2c_probe */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
int tw98_detect (struct i2c_adapter *adapter, int address, unsigned short flags, int kind)
#else
int tw98_detect (struct i2c_adapter *adapter, int address, int kind)
#endif
{
	struct i2c_client *client;
	int err = 0, version, revision = 0;
	struct tw98 *decoder;

	/* let's see whether this adapter can support what we need */
	if (!i2c_check_functionality (adapter, I2C_FUNC_SMBUS_READ_BYTE_DATA |
				      I2C_FUNC_SMBUS_WRITE_BYTE_DATA))
		return 0;

	/* allocate memory for client structure */
	client = kmalloc (sizeof (struct i2c_client), GFP_KERNEL);
        if (NULL == client)
		return -ENOMEM;
	
	/* fill client structure */
	memcpy (client, &tw98_client, sizeof (*client));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	client->id = tw98_id++;
#endif
	client->addr = address;
	client->adapter = adapter;

	/* chip detection */
	if (kind < 0) {
		/* Get chip version */
		version = i2c_smbus_read_byte_data (client, TW98_ID);

		/* If we could not get version, bail out */
		if (version == -1) {
			kfree (client);
			return 0;
		}

		/* If not a TW98, bail out */
		if ((version & 0xf8) != TW98_I2C_VERSION_ID) {
			TW98_PRINTK ("Unknown TWxx chip @ 0x%02x found."
				     " (Maybe TW99) Chip version: %#x\n", 2 * address, version);
			kfree (client);
			return 0;
		}

		/* Chip IS a TW98, log into syslog  */
		revision = version & 0x07;
		TW98_PRINTK ("video decoder chip found."
			     " Chip version: %#x (rev. %d)\n", version >> 3, revision);
	}
	else {
		TW98_PRINTK ("detection skipped\n");
	}

	/* allocate TW98 instance */
	decoder = kmalloc (sizeof (struct tw98), GFP_KERNEL);
	if (decoder == NULL) {
		err = -ENOMEM;
		goto err_out_kfree_client;
	}
	memset (decoder, 0, sizeof (struct tw98));
	decoder->revision = revision;
	i2c_set_clientdata (client, decoder);

	/* initialize registers */
	if (0 != tw98_initialize (client, revision)) {
		TW98_PRINTK ("could not initialize chipset, continuing...\n");
	}
	
	/* tell the i2c layer a client has arrived */
	if (0 != (err = i2c_attach_client (client))) {
		TW98_PRINTK ("FAILED attaching client\n");
		goto err_out_kfree_decoder;
	}

	TW98_PRINTK ("detected @ 0x%02x\n", 2 * address);
	return 0;

 err_out_kfree_decoder:
	kfree (decoder);
 err_out_kfree_client:
	kfree (client);
	return err;
}

/* Called when a new I2C bus found */
static int tw98_attach (struct i2c_adapter *adapter)
{
	return i2c_probe (adapter, &addr_data, &tw98_detect);
}

/* Called on exit */
static int tw98_detach (struct i2c_client *client)
{
	int err = 0;

	if (0 != (err = i2c_detach_client (client))) {
		TW98_PRINTK ("client deregistration failed, client not detached.\n");
		return err;
	}
	kfree (i2c_get_clientdata (client));
	kfree (client);
	return 0;
}

/*---------------------------------------------------------------------- 
 *      Modularization
 *----------------------------------------------------------------------*/

MODULE_AUTHOR      ("Hank Magnuski <hankm@mtinet.com>");
MODULE_DESCRIPTION ("Techwell 98 driver");
MODULE_LICENSE     ("GPL");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param       (debug, int, 0644);
#else
MODULE_PARM	   (debug, "i");
#endif
MODULE_PARM_DESC   (debug, "Debug level: 0 = silent, 1 = verbose, 2 = very verbose (default = 0).");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param       (clock, int, 0644);
#else
MODULE_PARM	   (clock, "i");
#endif
MODULE_PARM_DESC   (clock, "Crystal speed: 0 = 27MHz, 1 = 55MHz (default = 1).");

int __init tw98_init (void)
{
	int res = 0;

	if (0 != (res = i2c_add_driver (&i2c_driver_tw98))) {
		TW98_PRINTK ("driver registration failed, module not inserted.\n");
		return res;
	}
	return 0;
}

void __exit tw98_cleanup (void)
{	
	int res = 0;

	if (0 != (res = i2c_del_driver (&i2c_driver_tw98))) {
		TW98_PRINTK ("driver deregistration failed, module not removed.\n");
	}
}

module_init (tw98_init);
module_exit (tw98_cleanup);
