/* ------------------------------------------------------------------------- */
/*   v4l-riva.c V4L support functions for NVIDIA display adapters	     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 2000 Ferenc Bakonyi <fero@drama.obuda.kando.hu>
 *   Copyright (C) 2002, 2003, 2004 Stefan Jahn <stefan@lkcc.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */

#include <linux/config.h>
#include <linux/types.h>
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif
#ifdef CONFIG_MTRR
#include <asm/mtrr.h>
#endif
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/vmalloc.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <asm/processor.h>

#include "i2c-riva.h"
#include "tuner.h"
#include "rivatv.h"

/*----------------------------------------------------------------------
 *
 * Global data
 *
 *----------------------------------------------------------------------*/

/* (hot)plugged devices counter */ 
unsigned int rivatv_devices = 0;

/* configurable number of capture queue buffers */
int capbuffers = RIVATV_CAPTURE_BUFFERS;
/* configurable boolean indicating whether (!0) or not (0) to
   autoload a decoder chip module */
int autoload = 1;
/* debug level: 0 = silent, 1 = verbose, 2 = very verbose */
int debug = 1;
/* usage of tv-box: 0 = no tv-box, 1 = PAL tv-box, 2 = NTSC tv-box */
int tvbox = 0;
/* card type */
int card = -1;
/* enable colour conversion code */
int conversion = 1;
#ifdef CONFIG_MTRR
/* mtrr usage flag */
int mtrr = 1;
#endif

/* register values for weight register programming */
static unsigned char rivatv_weight_table [] = {
  /* 0x00000 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  /* 0x10000 */
  0xF2, 0xFF, 0xFF, 0xFF, 0x4E, 0x00, 0x00, 0x00, 0x4E, 0x00, 
  0x00, 0x00, 0xF2, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
  0xF6, 0xFF, 0xFF, 0xFF, 0x32, 0x00, 0x00, 0x00, 0x65, 0x00, 
  0x00, 0x00, 0xEC, 0xFF, 0xFF, 0xFF, 0x07, 0x00, 0x00, 0x00, 
  0xFD, 0xFF, 0xFF, 0xFF, 0x19, 0x00, 0x00, 0x00, 0x7B, 0x00, 
  0x00, 0x00, 0xEA, 0xFF, 0xFF, 0xFF, 0x05, 0x00, 0x00, 0x00,
  0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x87, 0x00, 
  0x00, 0x00, 0xED, 0xFF, 0xFF, 0xFF, 0x06, 0x00, 0x00, 0x00, 
  0x05, 0x00, 0x00, 0x00, 0xF5, 0xFF, 0xFF, 0xFF, 0x8C, 0x00, 
  0x00, 0x00, 0xF5, 0xFF, 0xFF, 0xFF, 0x05, 0x00, 0x00, 0x00,
  0x06, 0x00, 0x00, 0x00, 0xED, 0xFF, 0xFF, 0xFF, 0x87, 0x00, 
  0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 
  0x05, 0x00, 0x00, 0x00, 0xEA, 0xFF, 0xFF, 0xFF, 0x7B, 0x00, 
  0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0xFD, 0xFF, 0xFF, 0xFF,
  0x07, 0x00, 0x00, 0x00, 0xEC, 0xFF, 0xFF, 0xFF, 0x65, 0x00, 
  0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0xF6, 0xFF, 0xFF, 0xFF, 
  /* 0x12B85 */
  0xF6, 0xFF, 0xFF, 0xFF, 0x4A, 0x00, 0x00, 0x00, 0x4A, 0x00, 
  0x00, 0x00, 0xF6, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 
  0xF3, 0xFF, 0xFF, 0xFF, 0x3C, 0x00, 0x00, 0x00, 0x56, 0x00, 
  0x00, 0x00, 0xF9, 0xFF, 0xFF, 0xFF, 0x02, 0x00, 0x00, 0x00, 
  0xF5, 0xFF, 0xFF, 0xFF, 0x30, 0x00, 0x00, 0x00, 0x60, 0x00, 
  0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFA, 0xFF, 0xFF, 0xFF,
  0xF6, 0xFF, 0xFF, 0xFF, 0x21, 0x00, 0x00, 0x00, 0x65, 0x00, 
  0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0xFA, 0xFF, 0xFF, 0xFF, 
  0xF8, 0xFF, 0xFF, 0xFF, 0x14, 0x00, 0x00, 0x00, 0x68, 0x00, 
  0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0xF8, 0xFF, 0xFF, 0xFF, 
  0xFA, 0xFF, 0xFF, 0xFF, 0x0A, 0x00, 0x00, 0x00, 0x65, 0x00,
  0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0xF6, 0xFF, 0xFF, 0xFF, 
  0xFA, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x60, 0x00, 
  0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0xF5, 0xFF, 0xFF, 0xFF, 
  0x02, 0x00, 0x00, 0x00, 0xF9, 0xFF, 0xFF, 0xFF, 0x56, 0x00, 
  0x00, 0x00, 0x3C, 0x00, 0x00, 0x00, 0xF3, 0xFF, 0xFF, 0xFF,
  /* 0x187AE */
  0x05, 0x00, 0x00, 0x00, 0x3B, 0x00, 0x00, 0x00, 0x3B, 0x00, 
  0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x04, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0x42, 0x00,
  0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0xF7, 0xFF, 0xFF, 0xFF, 
  0xFF, 0xFF, 0xFF, 0xFF, 0x2F, 0x00, 0x00, 0x00, 0x44, 0x00, 
  0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0xFF, 
  0xFD, 0xFF, 0xFF, 0xFF, 0x27, 0x00, 0x00, 0x00, 0x48, 0x00, 
  0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0xFF,
  0xFC, 0xFF, 0xFF, 0xFF, 0x20, 0x00, 0x00, 0x00, 0x48, 0x00, 
  0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0xFF, 
  0xFC, 0xFF, 0xFF, 0xFF, 0x18, 0x00, 0x00, 0x00, 0x48, 0x00, 
  0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0xFD, 0xFF, 0xFF, 0xFF, 
  0xFC, 0xFF, 0xFF, 0xFF, 0x12, 0x00, 0x00, 0x00, 0x44, 0x00,
  0x00, 0x00, 0x2F, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 
  0xF7, 0xFF, 0xFF, 0xFF, 0x0C, 0x00, 0x00, 0x00, 0x42, 0x00, 
  0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 
  /* 0x28000 */
  0x16, 0x00, 0x00, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x2A, 0x00,
  0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0x12, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x2B, 0x00, 
  0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
  0x0F, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x2C, 0x00, 
  0x00, 0x00, 0x1A, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 
  0x0C, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x2C, 0x00,
  0x00, 0x00, 0x1D, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 
  0x0A, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x2C, 0x00, 
  0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00,
  0x08, 0x00, 0x00, 0x00, 0x1D, 0x00, 0x00, 0x00, 0x2C, 0x00, 
  0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 
  0x06, 0x00, 0x00, 0x00, 0x1A, 0x00, 0x00, 0x00, 0x2C, 0x00,
  0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x0F, 0x00, 0x00, 0x00, 
  0x04, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x2B, 0x00, 
  0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00
};


/* ------------------------------------------------------------------------- *
 *
 * V4L part
 *
 * ------------------------------------------------------------------------- */

/* Reprogram the Riva TV scale weight based upon a MAX_WIDTH and WIDTH. */
static int rivatv_program_weights (struct rivatv_info *info, int max_width, int width)
{
	int ratio, idx, n;
	u32 reg1, reg2, val1, val2, val3;

	if (width > max_width)
		return 0;

	if (max_width == 0)
		return 1;
  
	if (width == 0) {
		ratio = 0x10000;
	} else	{
		if (max_width == 0)
			ratio = 0x10000;
		else
			ratio = (max_width << 16) / width;
	}

	/* horizontal ratio > 2.5 */
	if (ratio > 0x28000)
		idx = 4;
	/* horizontal ratio > 1.53 */
	else if (ratio > 0x187AE)
		idx = 3;
	/* horizontal ratio > 1.17 */
	else if (ratio > 0x12B85)
		idx = 2;
	/* horizontal ratio > 1 */
	else if (ratio > 0x10000)
		idx = 1;
	/* horizontal ratio <= 1 */
	else
		idx = 0;

	for (n = 0; n < 8; n++) {
		val1 = rivatv_weight_table[160 * idx + n * 20 + 0];
		val2 = rivatv_weight_table[160 * idx + n * 20 + 4];
		val3 = rivatv_weight_table[160 * idx + n * 20 + 8];
		reg1 = (((val3 & 0xFF) << 0x10) & 0x00FF0000) |
			(((val2 & 0xFF) << 0x08) & 0x0000FF00) |
			((val1 & 0xFF) & 0x000000FF);
      
		val1 = rivatv_weight_table[160 * idx + n * 20 + 12];
		val2 = rivatv_weight_table[160 * idx + n * 20 + 16];
		reg2 = (((val2 & 0xFF) << 0x08) & 0x000FF00) |
			((val1 & 0xFF) & 0x000000FF);

		/* NV_PME_HORIZ_WGHTS_A */
		VID_WR32 (info->chip.PME, 0x2004B0 + n * 4, reg1);
		/* NV_PME_HORIZ_WGHTS_B */
		VID_WR32 (info->chip.PME, 0x2004D0 + n * 4, reg2);
	}
	return 1;
}

/* Excerpt of the SGS-THOMSON datasheet: 

   The RIVA 128 Video Port allows any arbitrary scale factor between 1
   and 31. For best results the scale factors of 1, 2, 3, 4, 6, 8, 12,
   16, and 24 are selected to avoid filtering losses. The Video Port
   decimates in the y-direction, dropping lines every few lines depending
   on the vertical scaling factor. The intention is to support filtered
   downscaling in the attached video decoder. 

   This piece of documentation apparently applies to NV3 and NV4
   architectures. This means the driver can setup the video port to 
   hardware scale to certain sizes. Any other sizes must be software scaled.
   This applies to horizontal scaling only. For vertical scaling the driver
   can use the hardware. */

static unsigned int rivatv_scale_table_1 [] = {
        /* 720     360     240     180     120      90      60      45      30         */
	/*   1       2       3       4       6       8      12      16      24     end */
	0x0001, 0x0002, 0x0003, 0x0004, 0x0006, 0x0008, 0x000C, 0x0010, 0x0018, 0xFFFF }; /* 5 Bits:  4:0 */
static unsigned int rivatv_scale_table_2 [] = {
	/*   0       1       2       2       3       3       4       4       5     end */
	0x0000, 0x0020, 0x0040, 0x0040, 0x0060, 0x0060, 0x0080, 0x0080, 0x00A0, 0xFFFF }; /* 3 Bits:  7:5 */
static unsigned int rivatv_scale_table_3 [] = {
	/*   0       1       1       2       2       3       3       4       4     end */
	0x0000, 0x0100, 0x0100, 0x0200, 0x0200, 0x0300, 0x0300, 0x0400, 0x0400, 0xFFFF }; /* 3 Bits: 10:8 */

/* Looks like the factors 4 = 2*2, 6 = 3*2, 8 = 2*2*2, 12 = 3*2*2, 16 = 2*2*2*2 and 
   24 = 3*2*2*2 are derived from the other factors. Those the values of scale table 
   2 and 3 are just added for these resolutions. 

   For scale factors derived from 3 the hardware produces strange colours. [FIXME] */

/* Setup nearest horiyontal resolutions for NV3 and NV4. */
static void rivatv_nearest_resolution (int width, int overlay, struct rivatv_port *port)
{
	int n = 0, res = 0;

	while (rivatv_scale_table_1[n] != 0xFFFF) {
		if (overlay) {
			if (RIVATV_MAX_VLD_WIDTH / rivatv_scale_table_1[n] <= width) {
				res = n;
				break;
			}
		} else {
			if (RIVATV_MAX_VLD_WIDTH / rivatv_scale_table_1[n] >= width) {
				res = n;
			}
		}
		n++;
	}
	port->org_width = RIVATV_MAX_ORG_WIDTH / rivatv_scale_table_1[res];
	port->vld_width = RIVATV_MAX_VLD_WIDTH / rivatv_scale_table_1[res];

	/* Fix the UYVY offset for scaler values derived from 3. */
	port->offset = 0;
	if ((rivatv_scale_table_1[res] % 3) == 0) {
		port->offset = 0;
		DPRINTK ("port offset set to %d\n", port->offset);
	}
}

/* Find an appropriate scale factor within one of the scale tables. */
static int rivatv_nearest_scale_factor (int hscale, int *idx)
{
	int n = 0;

	*idx = 0xFFFF;
	while (rivatv_scale_table_1[n] != 0xFFFF) {
		if (hscale >= rivatv_scale_table_1[n])
			*idx = n;
		n++;
	}
	if (*idx == 0xFFFF) {
		*idx = 1;
		return 0;
	}
	return 1;
}

/* Calculate scale factor for one of the Riva128 registers. */
static int rivatv_calculate_scaler_value (int max_width, int max_height, 
					  int width, int height, u32 *scaler)
{
	int idx, hscale, vscale;

	if (width == 0)
		width = max_width;
	hscale = max_width / width;
	if (hscale > 24)
		hscale = 24;
	if (rivatv_nearest_scale_factor (hscale, &idx) == 0)
		return 0;
  
	hscale = rivatv_scale_table_1[idx] | 
		rivatv_scale_table_2[idx] | 
		rivatv_scale_table_3[idx];
	if (height == 0)
		height = max_height;
	vscale = ((height - 1) << 10) / (max_height - 1);
	/* vertical ratio > 1 */
	if (vscale > 0x400)
		vscale = 0x400;
	*scaler = (vscale << 16) | hscale;
	return 1;
}

/* Start video transfer for NV04 architecture. */
static void rivatv_start_video_nv04 (struct rivatv_info *info)
{
	struct rivatv_port *port = &info->port;
	u32 fb_offset_a, fb_offset_b;
	u32 size, pitch, scaler, height;

	fb_offset_a = fb_offset_b = info->picture_offset;
	size = port->size;
	pitch = port->org_width << 1;
	height = port->org_height;
	
	/* Adjust values for interlaced video picture. */
	if (port->flags & VIDEO_INTERLACED) {
		fb_offset_b += pitch;
		pitch <<= 1;
		height >>= 1;
	}

	/* Calculate the scaler register value. */
	rivatv_calculate_scaler_value (port->max_width, port->max_height,
				       port->org_width, height, &scaler);

	/* Clear some registers. */
	VID_WR32 (info->chip.PME, 0x200418, 0);
	VID_WR32 (info->chip.PME, 0x20041C, 0);
	VID_WR32 (info->chip.PME, 0x200420, 0);

	/* Enable interrupts. */
	VID_OR32 (info->chip.PMC, 0x000140, 0x01);
	VID_OR32 (info->chip.PME, 0x200140, 0x01);

	/* Enable tasks. */
	VID_WR32 (info->chip.PME, 0x200200, 
		  (VID_RD32 (info->chip.PME, 0x200200) & 0xFFFFFFEC) | 0x12);

	/* Setup BUF0 of task A.*/
	VID_WR32 (info->chip.PME, 0x200400, fb_offset_a);
	VID_WR32 (info->chip.PME, 0x200408, pitch);
	VID_WR32 (info->chip.PME, 0x200410, size);
	VID_WR32 (info->chip.PME, 0x200424, scaler);

	/* Setup BUF1 of task A. */
	VID_WR32 (info->chip.PME, 0x200404, fb_offset_b);
	VID_WR32 (info->chip.PME, 0x20040C, pitch);
	VID_WR32 (info->chip.PME, 0x200414, size);
	VID_WR32 (info->chip.PME, 0x200428, scaler);

	/* Y-Crop lines (at top) for Task A. */
	VID_WR32 (info->chip.PME, 0x20042C, 0);

	/* Start BUF0 of Task A. */
	VID_WR32 (info->chip.PME, 0x20041C, 
		  (VID_RD32 (info->chip.PME, 0x20041C) & 0xFFFFFEFF) ^ 0x10000);
}

/* Stop video transfer for NV04 architecture. */
static void rivatv_stop_video_nv04 (struct rivatv_info *info)
{
	VID_AND32 (info->chip.PME, 0x200140, 0xFFFFFFFE);
	VID_AND32 (info->chip.PME, 0x200200, 0xFFFFFFEF);
	VID_WR32 (info->chip.PME, 0x200418, 0);
	VID_WR32 (info->chip.PME, 0x20041C, 0);
	VID_WR32 (info->chip.PME, 0x200420, 0);
}

/* Start video transfer for NV10 architecture. */
static void rivatv_start_video_nv10 (struct rivatv_info *info)
{
	struct rivatv_port *port = &info->port;
	u32 fb_offset_a, fb_offset_b;
	u32 size, pitch, height;

	fb_offset_a = fb_offset_b = info->picture_offset;
	size = port->size;
	pitch = port->org_width << 1;
	height = port->org_height;

	/* Adjust values for interlaced video picture. */
	if (port->flags & VIDEO_INTERLACED) {
		fb_offset_b += pitch;
		pitch <<= 1;
		height >>= 1;
	}

	/* NV_PME_TASKA_ME_STATE */
	VID_WR32 (info->chip.PME, 0x200458, 0);
	/* NV_PME_TASKA_SU_STATE */
	VID_WR32 (info->chip.PME, 0x20045C, info->last_SU = 0);
	/* NV_PME_TASKA_RM_STATE */
	VID_WR32 (info->chip.PME, 0x200460, info->last_RM = 0);
	/* NV_PME_656_CONFIG_TASKA_ONLY_ENABLED
	   NV_PME_656_CONFIG_TASKA_ENABLE */
	VID_OR32 (info->chip.PME, 0x200400, 0x41);

	VID_OR32 (info->chip.PMC, 0x000140, 0x01);
	/* NV_PME_INTR_EN_0_TASKA_NOTIFY_ENABLED */
	VID_OR32 (info->chip.PME, 0x200140, 0x01);

	rivatv_program_weights (info, port->max_width, port->org_width);

	/* NV_PME_TASKA_BUFF0_START_ADDRESS */
	VID_WR32 (info->chip.PME, 0x200440, fb_offset_a);
	/* NV_PME_TASKA_BUFF0_PITCH_VALUE */
	VID_WR32 (info->chip.PME, 0x200448, pitch);
	/* NV_PME_TASKA_BUFF0_LENGTH_VALUE */
	VID_WR32 (info->chip.PME, 0x200450, size);

	/* NV_PME_TASKA_BUFF1_START_ADDRESS */
	VID_WR32 (info->chip.PME, 0x200444, fb_offset_b);
	/* NV_PME_TASKA_BUFF1_PITCH_VALUE */
	VID_WR32 (info->chip.PME, 0x20044C, pitch);
	/* NV_PME_TASKA_BUFF1_LENGTH_VALUE */
	VID_WR32 (info->chip.PME, 0x200454, size);

	/* NV_PME_TASKA_Y_SCALE_INCR */
	VID_WR32 (info->chip.PME, 0x200468, 
		  ((height << 10) - 1) / (port->max_height - 1));
	/* NV_PME_TASKA_X_SCALE_FILTER_ENABLE
	   NV_PME_TASKA_X_SCALE_INCR */
	VID_WR32 (info->chip.PME, 0x20046C, 
		  ((port->max_width << 16) / port->org_width) | 0x80000000);

	/* NV_PME_TASKA_Y_CROP_STARTLINE */
	VID_WR32 (info->chip.PME, 0x200464, 0);

	/* NV_PME_TASKA_LINE_LENGTH_VALUE */
	if (port->org_width == 0)
		VID_WR32 (info->chip.PME, 0x2004F0, 0x04);
	else
		VID_WR32 (info->chip.PME, 0x2004F0, port->max_width << 1);
	/* NV_PME_TASKA_SU_STATE_BUFF0_IN_USE
	   NV_PME_TASKA_SU_STATE_BUFF0_FIELD */
	VID_WR32 (info->chip.PME, 0x20045C,
		  info->last_SU = ((VID_RD32 (info->chip.PME, 0x20045C) & 0xFFFFFEFF) ^ 0x10000));
}

/* Stop video transfer for NV10 architecture. */
static void rivatv_stop_video_nv10 (struct rivatv_info *info)
{
	/* NV_PME_INTR_EN_0_TASKA_NOTIFY_DISABLED 
	   NV_PME_INTR_EN_0_TASKB_NOTIFY_DISABLED */
	VID_AND32 (info->chip.PME, 0x200140, 0xFFFFFFEE);
	/* NV_PME_656_CONFIG_TASKA_ONLY_DISABLED
	   NV_PME_656_CONFIG_TASKA_DISABLED */
	VID_AND32 (info->chip.PME, 0x200400, 0xFFFFFFBE);
	/* NV_PME_TASKA_ME_STATE */
	VID_WR32 (info->chip.PME, 0x200458, 0);
	/* NV_PME_TASKA_SU_STATE */
	VID_WR32 (info->chip.PME, 0x20045C, info->last_SU = 0);
	/* NV_PME_TASKA_RM_STATE */
	VID_WR32 (info->chip.PME, 0x200460, info->last_RM = 0);
}

/* Compute chromakey depending on the colour depth. */
static u32 rivatv_overlay_colorkey (struct rivatv_overlay *overlay)
{
	u32 r, g, b, key = 0;

	r = (overlay->window.chromakey & 0x00FF0000) >> 16;
	g = (overlay->window.chromakey & 0x0000FF00) >> 8;
	b = overlay->window.chromakey & 0x000000FF;

	switch (overlay->buffer.depth) {
	case 15:
		key = ((r >> 3) << 10) | ((g >> 3) << 5) | ((b >> 3));
		break;
	case 16:
		key = ((r >> 3) << 11) | ((g >> 2) << 5) | ((b >> 3));
		break;
	case 24:
		key = overlay->window.chromakey & 0x00FFFFFF;
		break;
	case 32:
		key = overlay->window.chromakey;
		break;
	default:
		/* THINKME: Possible to pass a colour index for 8 bpp ? */
		DPRINTK ("invalid color depth: %d bpp\n", overlay->buffer.depth);
		break;
	}
	DPRINTK ("overlay colour key is: %08X\n", key);
	return key;
}

/* Get pan offset of the physical screen. */
static u32 rivatv_overlay_pan (struct rivatv_info *info)
{
	struct rivatv_overlay *overlay = &info->overlay;
	u32 pan;

	info->chip.lock (&info->chip, 0);
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x0D);
	pan = VID_RD08 (info->chip.PCIO, 0x3D5);
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x0C); 
	pan |= VID_RD08 (info->chip.PCIO, 0x3D5) << 8;
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x19);
	pan |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x1F) << 16;
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x2D);
	pan |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x60) << 16;
	pan <<= 2;
	PRINTK ("pan offset: %u byte @ (%u,%u)\n", pan,
		(pan % overlay->bytesPerLine) * 8 / overlay->bitsPerPixel,
		(pan / overlay->bytesPerLine));
	return pan;
}

/* Get resolution of FB screen. */
static void rivatv_fb_properties (struct rivatv_info *info)
{
	struct rivatv_overlay *overlay = &info->overlay;
	u32 bpp, x, y, val, dbl;

	/* Check using CRT registers. */
	info->chip.lock (&info->chip, 0);
	/* NV_PCRTC_PIXEL_FORMAT */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x28);
	bpp = 0x04 << (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x03);
	/* NV_PCRTC_HORIZ_DISPLAY_END */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x01);
	x = VID_RD08 (info->chip.PCIO, 0x3D5);
	/* NV_PCRTC_HORIZ_EXTRA_DISPLAY_END_8 */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x2D);
	x |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x02) << 7; x = (x + 1) << 3;
	/* NV_PCRTC_VERT_DISPLAY_END */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x12);
	y = VID_RD08 (info->chip.PCIO, 0x3D5);
	/* NV_PCRTC_OVERFLOW_VERT_DISPLAY_END_[89] */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x07);
	val = VID_RD08 (info->chip.PCIO, 0x3D5);
	y |= (val & 0x02) << 7;	y |= (val & 0x40) << 3;	y++;
	/* NV_PCRTC_EXTRA_VERT_DISPLAY_END_10 */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x25);
	y |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x02) << 9;
	/* NV_PCRTC_???_VERT_DISPLAY_END_11 */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x41);
	y |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x04) << 9;
	/* NV_PCRTC_MAX_SCAN_LINE_DOUBLE_SCAN */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x09);
	dbl = (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x80) >> 7; y >>= dbl;

	overlay->physicalX = x;
	overlay->physicalY = y;
	overlay->bitsPerPixel = bpp;
	PRINTK ("physical resolution: %ux%ux%u (%sdouble scan)\n", x, y, bpp, dbl ? "" : "no ");

	/* NV_PCRTC_OFFSET */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x13);
	x = VID_RD08 (info->chip.PCIO, 0x3D5);
	/* NV_PCRTC_REPAINT0_OFFSET_10_8 */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x19);
	x |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0xE0) << 3;
	/* NV_PCRTC_EXTRA_OFFSET_11 */
	VID_WR08 (info->chip.PCIO, 0x3D4, 0x25);
	x |= (VID_RD08 (info->chip.PCIO, 0x3D5) & 0x20) << 6; x <<= 3;

	overlay->virtualX = x;
	overlay->virtualY = y;
	overlay->bytesPerLine = overlay->virtualX * overlay->bitsPerPixel / 8;
	PRINTK ("virtual resolution: %ux%ux%u (%d bpl)\n", x, y, bpp, overlay->bytesPerLine);
}

/* Start overlay video. */
void rivatv_overlay_start (struct rivatv_info *info) {

	struct rivatv_port *port = &info->port;
	struct rivatv_overlay *overlay = &info->overlay;
	struct video_window *window = &overlay->window;
	u32 base, pitch, size, offset, xscale, yscale, pan;
	int x, y, width, height;

	if (overlay->enable)
		return;

	/* check framebuffer layout */
	if (overlay->buffer.depth == 0 || overlay->buffer.bytesperline == 0 || overlay->buffer.base == 0 ||
	    overlay->buffer.height == 0 || overlay->buffer.width == 0) {
		PRINTK_ERR ("previous VIDIOCSFBUF ioctl required for overlay\n");
		return;
	}

	/* get pan offset of the physical screen */
	rivatv_fb_properties (info);
	pan = rivatv_overlay_pan (info);

	/* adjust window position depending on the pan offset */
	window->x -= (pan % overlay->buffer.bytesperline) * 8 / overlay->buffer.depth;
	window->y -= (pan / overlay->buffer.bytesperline);

	DPRINTK ("starting overlay: %dx%d at (%d,%d)\n", 
		 window->width, window->height, window->x, window->y);

	base = info->picture_offset;
	size = port->size;
	pitch = port->org_width << 1;

	spin_lock_bh (&info->video_lock);
	spin_lock_irq (&info->sched.sched_lock);
	switch (info->chip.arch) {
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:

		/* adjust negative output window variables */
		offset = 0;
		x = window->x;
		y = window->y;
		width = window->width;
		height = window->height;
		if ((int32_t) window->x < 0) {
			x = 0;
			width = window->width + window->x;
			offset += (-window->x * port->vld_width / window->width) << 1; 
		}
		if ((int32_t) window->y < 0) {
			y = 0;
			height = window->height + window->y;
			offset += (-window->y * port->vld_height / window->height * port->org_width) << 1;
		}

		/* NV_PVIDEO_BASE */
		VID_WR32 (info->chip.PVIDEO, 0x900 + 0, base);
		//VID_WR32 (info->chip.PVIDEO, 0x900 + 4, base);
		/* NV_PVIDEO_LIMIT */
		VID_WR32 (info->chip.PVIDEO, 0x908 + 0, base + size - 1);
		//VID_WR32 (info->chip.PVIDEO, 0x908 + 4, base + size - 1);

		/* extra code for NV20 architectures */
		if (info->chip.arch == NV_ARCH_20 || info->chip.arch == NV_ARCH_30) {
			VID_WR32 (info->chip.PVIDEO, 0x800 + 0, base);
			//VID_WR32 (info->chip.PVIDEO, 0x800 + 4, base);
			VID_WR32 (info->chip.PVIDEO, 0x808 + 0, base + size - 1);
			//VID_WR32 (info->chip.PVIDEO, 0x808 + 4, base + size - 1);
		}

		/* NV_PVIDEO_LUMINANCE */
		VID_WR32 (info->chip.PVIDEO, 0x910 + 0, 0x00001000);
		//VID_WR32 (info->chip.PVIDEO, 0x910 + 4, 0x00001000);
		/* NV_PVIDEO_CHROMINANCE */
		VID_WR32 (info->chip.PVIDEO, 0x918 + 0, 0x00001000);
		//VID_WR32 (info->chip.PVIDEO, 0x918 + 4, 0x00001000);

		/* NV_PVIDEO_OFFSET */
		VID_WR32 (info->chip.PVIDEO, 0x920 + 0, offset + 0);
		//VID_WR32 (info->chip.PVIDEO, 0x920 + 4, offset + pitch);
		/* NV_PVIDEO_SIZE_IN */
		VID_WR32 (info->chip.PVIDEO, 0x928 + 0, ((port->org_height) << 16) | port->org_width);
		//VID_WR32 (info->chip.PVIDEO, 0x928 + 4, ((port->org_height/2) << 16) | port->org_width);
		/* NV_PVIDEO_POINT_IN */
		VID_WR32 (info->chip.PVIDEO, 0x930 + 0, 0x00000000);
		//VID_WR32 (info->chip.PVIDEO, 0x930 + 4, 0x00000000);
		/* NV_PVIDEO_DS_DX_RATIO */
		VID_WR32 (info->chip.PVIDEO, 0x938 + 0, (port->org_width << 20) / window->width);
		//VID_WR32 (info->chip.PVIDEO, 0x938 + 4, (port->org_width << 20) / window->width);
		/* NV_PVIDEO_DT_DY_RATIO */
		VID_WR32 (info->chip.PVIDEO, 0x940 + 0, ((port->org_height) << 20) / window->height);
		//VID_WR32 (info->chip.PVIDEO, 0x940 + 4, ((port->org_height/2) << 20) / window->height);

		/* NV_PVIDEO_POINT_OUT */
		VID_WR32 (info->chip.PVIDEO, 0x948 + 0, ((y + 0) << 16) | x);
		//VID_WR32 (info->chip.PVIDEO, 0x948 + 4, ((y + 0) << 16) | x);
		/* NV_PVIDEO_SIZE_OUT */
		VID_WR32 (info->chip.PVIDEO, 0x950 + 0, (height << 16) | width);
		//VID_WR32 (info->chip.PVIDEO, 0x950 + 4, (height << 16) | width);

		/* NV_PVIDEO_FORMAT */
		VID_WR32 (info->chip.PVIDEO, 0x958 + 0, (pitch << 0) | 0x00100000);
		//VID_WR32 (info->chip.PVIDEO, 0x958 + 4, (pitch << 1) | 0x00100000);
		/* NV_PVIDEO_COLOR_KEY */
		overlay->colorkey = VID_RD32 (info->chip.PVIDEO, 0xB00);
		DPRINTK ("overlay colour key has been: %08X\n", overlay->colorkey);
		VID_WR32 (info->chip.PVIDEO, 0xB00, rivatv_overlay_colorkey (overlay));

		/* NV_PVIDEO_INTR_EN_BUFFER */
		VID_OR32 (info->chip.PVIDEO, 0x140, 0x01/*0x11*/);
		/* NV_PVIDEO_STOP */
		VID_AND32 (info->chip.PVIDEO, 0x704, 0xFFFFFFEE);
		/* NV_PVIDEO_BUFFER */
		VID_OR32 (info->chip.PVIDEO, 0x700, 0x01/*0x11*/);
		break;

	case NV_ARCH_03:
	case NV_ARCH_04:

		/* adjust negative output window variables */
		offset = (port->org_width - port->vld_width) & ~1;
		x = window->x;
		y = window->y;
		width = window->width;
		height = window->height;
		if ((int32_t) window->x < 0) {
			x = 0;
			width = window->width + window->x;
			offset += (-window->x * port->vld_width / window->width) << 1; 
		}
		if ((int32_t) window->y < 0) {
			y = 0;
			height = window->height + window->y;
			offset += (-window->y * port->vld_height / window->height * port->org_width) << 1;
		}

		/* NV_PVIDEO_OE_STATE */
		VID_WR32 (info->chip.PVIDEO, 0x224, 0);
		/* NV_PVIDEO_SU_STATE */
		VID_WR32 (info->chip.PVIDEO, 0x228, 0);
		/* NV_PVIDEO_RM_STATE */
		VID_WR32 (info->chip.PVIDEO, 0x22C, 0);

		/* NV_PVIDEO_BUFF0_START_ADDRESS */
		VID_WR32 (info->chip.PVIDEO, 0x20C + 0, base + offset + 0);
		VID_WR32 (info->chip.PVIDEO, 0x20C + 4, base + offset + 0);
		/* NV_PVIDEO_BUFF0_PITCH_LENGTH */
		VID_WR32 (info->chip.PVIDEO, 0x214 + 0, pitch);
		VID_WR32 (info->chip.PVIDEO, 0x214 + 4, pitch);

		/* NV_PVIDEO_WINDOW_START */
		VID_WR32 (info->chip.PVIDEO, 0x230, (y << 16) | x);
		/* NV_PVIDEO_WINDOW_SIZE */
		VID_WR32 (info->chip.PVIDEO, 0x234, (height << 16) | width);
		/* NV_PVIDEO_STEP_SIZE */
		yscale = ((port->vld_height - 1) << 11) / (window->height - 1);
		xscale = ((port->vld_width - 1) << 11) / (window->width - 1);
		VID_WR32 (info->chip.PVIDEO, 0x200, (yscale << 16) | xscale);

		/* NV_PVIDEO_RED_CSC_OFFSET */
		VID_WR32 (info->chip.PVIDEO, 0x280, 0x69);
		/* NV_PVIDEO_GREEN_CSC_OFFSET */
		VID_WR32 (info->chip.PVIDEO, 0x284, 0x3e);
		/* NV_PVIDEO_BLUE_CSC_OFFSET */
		VID_WR32 (info->chip.PVIDEO, 0x288, 0x89);
		/* NV_PVIDEO_CSC_ADJUST */
		VID_WR32 (info->chip.PVIDEO, 0x28C, 0x00000); /* No colour correction! */

		/* NV_PVIDEO_CONTROL_Y (BLUR_ON, LINE_HALF) */
		VID_WR32 (info->chip.PVIDEO, 0x204, 0x001);
		/* NV_PVIDEO_CONTROL_X (WEIGHT_HEAVY, SHARPENING_ON, SMOOTHING_ON) */
		VID_WR32 (info->chip.PVIDEO, 0x208, 0x111);

		/* NV_PVIDEO_FIFO_BURST_LENGTH */
		VID_WR32 (info->chip.PVIDEO, 0x23C, 0x03);
		/* NV_PVIDEO_FIFO_THRES_SIZE */
		VID_WR32 (info->chip.PVIDEO, 0x238, 0x38);

		/* NV_PVIDEO_BUFF0_OFFSET */
		VID_WR32 (info->chip.PVIDEO, 0x21C + 0, 0);
		VID_WR32 (info->chip.PVIDEO, 0x21C + 4, 0);

		/* NV_PVIDEO_KEY */
		overlay->colorkey = VID_RD32 (info->chip.PVIDEO, 0x240);
		DPRINTK ("overlay colour key has been: %08X\n", overlay->colorkey);
		VID_WR32 (info->chip.PVIDEO, 0x240, rivatv_overlay_colorkey (overlay));

		/* NV_PVIDEO_INTR_EN_0_NOTIFY_ENABLED */
		VID_OR32 (info->chip.PVIDEO, 0x140, 0x01);
		/* NV_PVIDEO_OVERLAY (KEY_ON, VIDEO_ON, FORMAT_CCIR) */
		VID_WR32 (info->chip.PVIDEO, 0x244, 0x011);
		/* NV_PVIDEO_SU_STATE */
		VID_XOR32 (info->chip.PVIDEO, 0x228, 1 << 16);
		break;
	}
	info->sched.flags |= RIVATV_SCHED_OVERLAY;
	spin_unlock_irq (&info->sched.sched_lock);

	overlay->enable = 1;
	spin_unlock_bh (&info->video_lock);
}

/* Stop overlay video. */
void rivatv_overlay_stop (struct rivatv_info *info) {

	if (!info->overlay.enable)
		return;

	DPRINTK ("stopping overlay\n");

	spin_lock_bh (&info->video_lock);
	spin_lock_irq (&info->sched.sched_lock);
	switch (info->chip.arch ) {
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
		/* NV_PVIDEO_COLOR_KEY */
		/* Xv-Extension-Hack: Restore previously saved value. */
		VID_WR32 (info->chip.PVIDEO, 0xB00, info->overlay.colorkey);
		/* NV_PVIDEO_STOP */
		VID_OR32 (info->chip.PVIDEO, 0x704, 0x11);
		/* NV_PVIDEO_BUFFER */
		VID_AND32 (info->chip.PVIDEO, 0x700, ~0x11);
		/* NV_PVIDEO_INTR_EN_BUFFER */
		VID_AND32 (info->chip.PVIDEO, 0x140, ~0x11);
		break;
	case NV_ARCH_03:
	case NV_ARCH_04:
		/* NV_PVIDEO_KEY */
		VID_WR32 (info->chip.PVIDEO, 0x240, info->overlay.colorkey);
		/* NV_PVIDEO_OVERLAY_VIDEO_OFF */
		VID_AND32 (info->chip.PVIDEO, 0x244, ~0x01);
		/* NV_PVIDEO_INTR_EN_0_NOTIFY */
		VID_AND32 (info->chip.PVIDEO, 0x140, ~0x01);
		/* NV_PVIDEO_OE_STATE */
		VID_WR32 (info->chip.PVIDEO, 0x224, 0);
		/* NV_PVIDEO_SU_STATE */
		VID_WR32 (info->chip.PVIDEO, 0x228, 0);
		/* NV_PVIDEO_RM_STATE */
		VID_WR32 (info->chip.PVIDEO, 0x22C, 0);
		break;
	}
	info->sched.flags &= ~RIVATV_SCHED_OVERLAY;
	spin_unlock_irq (&info->sched.sched_lock);

	info->overlay.enable = 0;
	spin_unlock_bh (&info->video_lock);
}

/* Schedules the next video picture to be transfered from the video decoder
   into the framebuffer.  Thus we can ensure that no additional video picture
   runs into the currently captured (grabdisplay) or displayed (overlay).
   This hopefully fixes the interlace actefact problem for larger resolutions. */
void rivatv_schedule_next (struct rivatv_info *info)
{
	if (info->sched.state_buf_notify == 0)
		return;

	switch (info->chip.arch) {
	case NV_ARCH_03:
	case NV_ARCH_04:
		VID_XOR32 (info->chip.PME, 0x200420, info->sched.state_buf_notify);
		VID_XOR32 (info->chip.PME, 0x20041C, info->sched.state_buf_usage);
		break;
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
		/* toggle NV_PME_TASKA_RM_STATE_BUFF1_INTR_NOTIFY */
		VID_WR32 (info->chip.PME, 0x200460,
			  info->last_RM = (VID_RD32(info->chip.PME, 0x200460) ^ info->sched.state_buf_notify));
		/* toggle NV_PME_TASKA_SU_STATE_BUFF0_IN_USE */
		VID_WR32 (info->chip.PME, 0x20045C,
			  info->last_SU = (VID_RD32(info->chip.PME, 0x20045C) ^ info->sched.state_buf_usage));
		break;
	}
	info->sched.state_buf_notify = 0;
	info->sched.state_buf_usage = 0;
}

/* Interrupt handler for the overlay engine. */
static void rivatv_interrupt_overlay (struct rivatv_info *info)
{
        u32 intr_reg, state_OE, state_SU, state_RM, buffer_use_state, buffer_intr_state;

	switch (info->chip.arch) {
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
		
		/* Reset the overlay interrupt register. 
		   This interrupt happens to occur if overlay output is stopped. */

		/* NV_PVIDEO_INTR_BUFFER_X_PENDING */
		intr_reg = VID_RD32 (info->chip.PVIDEO, 0x100);
		if (intr_reg & 0x11) {
			info->sched.stats.overlay_interrupts++;
			if (VID_RD32 (info->chip.PVIDEO, 0x104)) {
				PRINTK_ERR ("protection fault in overlay buffer: 0x%02X\n",
					    VID_RD32 (info->chip.PVIDEO, 0x104));
			} else {
				/* NV_PVIDEO_INTR_BUFFER_X_RESET */
				VID_WR32 (info->chip.PVIDEO, 0x100, intr_reg & 0x11);
			}
			rivatv_schedule_next (info);
		}
		break;

	case NV_ARCH_03:
	case NV_ARCH_04:

		/* NV_PVIDEO_INTR_0_NOTIFY_PENDING */
		if (VID_RD32 (info->chip.PVIDEO, 0x100) & 0x01) {
			/* NV_PVIDEO_INTR_0_NOTIFY_RESET */
			VID_OR32 (info->chip.PVIDEO, 0x100, 0x01);
			info->sched.stats.overlay_interrupts++;

			/* Evaluate status registers. */
			state_OE = VID_RD32 (info->chip.PVIDEO, 0x224);
			state_SU = VID_RD32 (info->chip.PVIDEO, 0x228);
			state_RM = VID_RD32 (info->chip.PVIDEO, 0x22C);
			buffer_use_state = state_OE ^ state_SU;	 /* true if different */
			buffer_intr_state = state_OE ^ state_RM; /* true if different */
			if ((buffer_use_state & 0x10000) && (buffer_intr_state & 0x10)) {
				VID_XOR32 (info->chip.PVIDEO, 0x228, 0x110000);
				VID_XOR32 (info->chip.PVIDEO, 0x22C, 0x11);
			}
		}
		break;
	}
}


/* The PCI interrupt handler. */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
static void 
#else
static irqreturn_t
#endif
rivatv_interrupt (int irq, void * dev_id, struct pt_regs * regs)
{
	int both_fields_done = 0;
	struct rivatv_info *info;
	struct video_device *dev = dev_id;
	u32 state_SU, state_RM, buffer_use_state, state_ME, buffer_intr_state;
    
	if (dev == NULL) {
		PRINTK_ERR ("IRQ %d for invalid device\n", irq);
		return IRQ_RETVAL (0);
	}
	info = video_get_drvdata (dev);

	spin_lock (&info->sched.sched_lock);
	info->sched.stats.interrupts++;

	/* Handle DMA interrupts if necessary. */
	if (info->sched.flags & RIVATV_SCHED_DMA) {
		rivatv_interrupt_DMA (info);
	}

	/* Handle overlay interrupts here. */
	rivatv_interrupt_overlay (info);

	switch (info->chip.arch) {
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
		
		/* NV_PME_INTR_EN_0_TASKA_NOTIFY_ENABLED */
		if (!(VID_RD32 (info->chip.PME, 0x200140) & 0x01)) {
			spin_unlock (&info->sched.sched_lock);
			return IRQ_RETVAL (0);
		}

		/* NV_PME_INTR_0_TASKA_NOTIFY_RESET */
		VID_OR32 (info->chip.PME, 0x200100, 0x01);

		/* Evaluate task A status registers. */

		/* NV_PME_TASKA_ME_STATE */
		state_ME = VID_RD32 (info->chip.PME, 0x200458);
		/* NV_PME_TASKA_SU_STATE */
		state_SU = VID_RD32 (info->chip.PME, 0x20045C);
		/* NV_PME_TASKA_RM_STATE */
		state_RM = VID_RD32 (info->chip.PME, 0x200460);

		if (state_SU != info->last_SU) {
			DPRINTK2 ("divine interference on SU\n");
			/* not sure what to do, but this works */
			state_SU &= ~0x110000;
			state_SU |= info->last_SU & 0x110000;
			VID_WR32 (info->chip.PME, 0x20045C, state_SU);
		}

		if (state_RM != info->last_RM) {
			DPRINTK2 ("divine interference on RM\n");
			/* not sure what to do, but this works */
			state_RM &= ~0x11;
			state_RM |= info->last_RM & 0x11;
			VID_WR32 (info->chip.PME, 0x200460, state_RM);
		}

		/* NV_PME_TASKA_ME_STATE_BUFF01_IN_USE ^ NV_PME_TASKA_SU_STATE_BUFF01_IN_USE */
		buffer_use_state = state_ME ^ state_SU;
		/* NV_PME_TASKA_ME_STATE_BUFF01_INTR_NOTIFY ^ NV_PME_TASKA_RM_STATE_BUFF01_INTR_NOTIFY */
		buffer_intr_state = state_ME ^ state_RM;

		/* BUF0 in use or not in use and BUF0 notified */
		if (!(buffer_use_state & 0x10000) && (buffer_intr_state & 0x01)) {
			info->sched.stats.decoder_interrupts++;
			if (info->sched.flags & RIVATV_SCHED_INTERLACED) {
				info->sched.state_buf_notify = 0x01;
				info->sched.state_buf_usage  = 0x100000;
				both_fields_done = 1;
			} else {
				info->sched.state_buf_notify = 0x01;
				info->sched.state_buf_usage  = 0x100000;
				both_fields_done = 1;
			}
		}
		/* BUF1 in use or not in use and BUF1 notified */
		else if (!(buffer_use_state & 0x100000) && (buffer_intr_state & 0x10)) {
			info->sched.stats.decoder_interrupts++;
			info->sched.state_buf_notify = 0x10;
			info->sched.state_buf_usage  = 0x010000;
			both_fields_done = 1;
		}
		if (info->sched.flags & RIVATV_SCHED_OVERLAY)
			rivatv_schedule_next (info);
			
		break;

	case NV_ARCH_03:
	case NV_ARCH_04:

		if (!(VID_RD32 (info->chip.PME, 0x200140) & 0x01)) {
			spin_unlock (&info->sched.sched_lock);
			return IRQ_RETVAL (0);
		}

		/* Reset the interrupt register. */
		VID_OR32 (info->chip.PME, 0x200100, 0x01);

		/* Evaluate task A status registers. */
		state_ME = VID_RD32 (info->chip.PME, 0x200418);
		state_SU = VID_RD32 (info->chip.PME, 0x20041C);
		state_RM = VID_RD32 (info->chip.PME, 0x200420);

		buffer_use_state = state_ME ^ state_SU;
		buffer_intr_state = state_ME ^ state_RM;

#ifdef NEW_INTERRUPTS
#warning "Using new interrupt routine."
                if (info->last_buffer_processed) {
                        if (!(buffer_use_state & 0x10000) && (buffer_intr_state & 0x01)) {
				info->sched.stats.decoder_interrupts++;
                                VID_XOR32 (info->chip.PME, 0x200420, 0x01);
                                VID_XOR32 (info->chip.PME, 0x20041C, 0x100000);
                                info->last_buffer_processed = 0;
                                if (!(info->port.flags & VIDEO_INTERLACED))
                                        both_fields_done = 1;
                                both_fields_done = 1;
                        } else if (!(buffer_use_state & 0x100000) && (buffer_intr_state & 0x10)) {
				info->sched.stats.decoder_interrupts++;
                                VID_XOR32 (info->chip.PME, 0x200420, 0x10);
                                VID_XOR32 (info->chip.PME, 0x20041C, 0x10000);
                                info->last_buffer_processed = 1;
                                both_fields_done = 1;
                        }
                } else {
                        if (!(buffer_use_state & 0x100000) && (buffer_intr_state & 0x10)) {
				info->sched.stats.decoder_interrupts++;
                                VID_XOR32 (info->chip.PME, 0x200420, 0x10);
                                VID_XOR32 (info->chip.PME, 0x20041C, 0x10000);
                                info->last_buffer_processed = 1;
                                both_fields_done = 1;
                        } else if (!(buffer_use_state & 0x10000) && (buffer_intr_state & 0x01)) {
				info->sched.stats.decoder_interrupts++;
                                VID_XOR32 (info->chip.PME, 0x200420, 0x01);
                                VID_XOR32 (info->chip.PME, 0x20041C, 0x100000);
                                info->last_buffer_processed = 0;
                                if (!(info->port.flags & VIDEO_INTERLACED))
                                        both_fields_done = 1;
                                both_fields_done = 1;
                        }
                }
#else
		if (!(buffer_use_state & 0x10000) && (buffer_intr_state & 0x01)) {
			info->sched.stats.decoder_interrupts++;
			if (info->sched.flags & RIVATV_SCHED_INTERLACED) {
				info->sched.state_buf_notify = 0x01;
				info->sched.state_buf_usage  = 0x100000;
				both_fields_done = 1;
			} else {
				info->sched.state_buf_notify = 0x01;
				info->sched.state_buf_usage  = 0x100000;
				both_fields_done = 1;
			}
		} 
		else if (!(buffer_use_state & 0x100000) && (buffer_intr_state & 0x10)) {
			info->sched.stats.decoder_interrupts++;
			info->sched.state_buf_notify = 0x10;
			info->sched.state_buf_usage  = 0x010000;
			both_fields_done = 1;
		}
#endif
		if (info->sched.flags & RIVATV_SCHED_OVERLAY)
			rivatv_schedule_next (info);

		break;
	}

	/* Return here for overlay'ing. */
	if (info->sched.flags & RIVATV_SCHED_OVERLAY) {
		spin_unlock (&info->sched.sched_lock);
		return IRQ_RETVAL (1);
	}

	/* Schedule the bottom half here. */
	if (both_fields_done) {
		tasklet_hi_schedule (&info->capture_queue.task);
	}
	else if (jiffies - info->sched.last_jiffies > HZ / 10) {
		if (info->capture_buffer != NULL) {
			tasklet_hi_schedule (&info->capture_queue.task);
		}
	}

	spin_unlock (&info->sched.sched_lock);
	return IRQ_RETVAL (1);
}

/* Enable PFB (Framebuffer), PVIDEO (Overlay unit) and PME (Mediaport) if neccessary. */
static void __init rivatv_enable_PMEDIA (struct rivatv_info *info)
{
	u32 reg;

	/* switch off interrupts once for a while */
	VID_WR32 (info->chip.PME, 0x200140, 0x00);
	VID_WR32 (info->chip.PMC, 0x000140, 0x00);

	reg = VID_RD32 (info->chip.PMC, 0x000200);

	/* NV3 (0x10100010): NV03_PMC_ENABLE_PMEDIA, NV03_PMC_ENABLE_PFB, NV03_PMC_ENABLE_PVIDEO */

	if ((reg & 0x10100010) != 0x10100010) {
		PRINTK_INFO ("PMEDIA, PVIDEO and PFB disabled, enabling...\n");
		VID_OR32 (info->chip.PMC, 0x000200, 0x10100010);
	}

	/* re-enable interrupts again */
	VID_WR32 (info->chip.PMC, 0x000140, 0x01);
	VID_WR32 (info->chip.PME, 0x200140, 0x01);
}

/* Install the interrupt routine for video. */
static int __init rivatv_install_irq (struct rivatv_info *info)
{
	int err;

	err = request_irq (info->pci->irq, rivatv_interrupt,
			   SA_INTERRUPT | SA_SHIRQ, "rivatv", info->video);
	if (err < 0) {
		PRINTK_ERR ("unable to get IRQ %d (errno = %d)\n", info->pci->irq, err);
		return err;
	}

	/* Enable IRQs on PME and PMC. */
	rivatv_enable_PMEDIA (info);

	DPRINTK ("successfully requested IRQ %d\n", info->pci->irq);
	return 0;
}

/* Uninstall the interrupt routine for video. */
static int __exit rivatv_deinstall_irq (struct rivatv_info *info)
{
	VID_WR32 (info->chip.PME, 0x200140, 0);
	free_irq (info->pci->irq, info->video);
	DPRINTK ("successfully freed IRQ %d\n", info->pci->irq);
	return 0;
}

struct rivatv_video_standard {
	/* video standard constant */
	int standard;
	/* maximum width in pixels */
	int width;
	/* maximum height in pixels */
	int height;
};

static struct rivatv_video_standard rivatv_standard[3] = {
	{ VIDEO_MODE_PAL,   RIVATV_MAX_VLD_WIDTH, 576 },
	{ VIDEO_MODE_NTSC,  RIVATV_MAX_VLD_WIDTH, 480 },
	{ VIDEO_MODE_SECAM, RIVATV_MAX_VLD_WIDTH, 576 }
};

void rivatv_get_capability (struct rivatv_info *info, struct video_capability *cap)
{
	if (info->overlay.enable) {
		cap->maxwidth = info->overlay.buffer.width;
		cap->maxheight = info->overlay.buffer.height;
	} else {
		cap->maxwidth = rivatv_standard[info->format.norm].width;
		cap->maxheight = rivatv_standard[info->format.norm].height;
	}
	if (info->tuner_required != -1 && info->i2c->tuner != NULL) {
		cap->type |= VID_TYPE_TUNER;
	}
	if (info->audio_required != -1 && info->i2c->audio_processor != NULL) {
		cap->audios = 1;
	}
}

/* Setup size of video decoder image in framebuffer for overlay. */
void rivatv_set_overlay_format (struct rivatv_info *info)
{
	info->format.overlay = 1;
	info->format.width = rivatv_standard[info->format.norm].width;
	info->format.height = rivatv_standard[info->format.norm].height;

	switch (info->chip.arch) {
	case NV_ARCH_03:
	case NV_ARCH_04:

		/* FIXME: at least these archs don't really support interlaced heights */
		info->format.height /= 2;

		if (info->overlay.window.width < info->format.width)
			info->format.width = info->overlay.window.width;
		if (info->overlay.window.height < info->format.height)
			info->format.height = info->overlay.window.height;
		break;
	}
}

/* Scales a UYVY video buffer DOWN. */
static void rivatv_soft_scale (char *src_buf, char *dst_buf,
			       int src_width, int src_height, int src_offset, int src_pitch,
			       int dst_width)
{
#if (defined (__i386__) || defined (__x86_64__)) && RIVATV_ENABLE_ASM

	__asm__ __volatile__(
	"   mov	      %0, %%esi		\n" /* ESI = src_buf		       */
	"   shll      $1, %5		\n" /* src_offset <<= 1		       */
	"   shll      $1, %6		\n" /* src_pitch <<= 1		       */
	"   add	      %5, %%esi		\n" /* ESI += src_offset	       */
	"   mov	      %1, %%edi		\n" /* EDI = dst_buf		       */
	"   mov	      %2, %%edx		\n" /* EDX = src_width		       */
	"   mov	      %4, %%ebx		\n" /* EBX = dst_width		       */
	"1: mov	      %%ebx, %%ecx	\n" /* ECX = EBX		       */
	"   mov	      %%edx, %%eax	\n" /* EAX = EDX		       */
	"   shr	      $1, %%ecx		\n" /* ECX >>= 1		       */
	"   shr	      $1, %%eax		\n" /* EAX >>= 1		       */
	"   push      %%esi		\n" /* save ESI			       */
	"2: add	      %%ebx, %%eax	\n" /* EAX += EBX		       */
	"   cmp	      %%edx, %%eax	\n" /* EAX >= EDX		       */
	"   jae	      3f		\n" /* yes, goto 3		       */
	"   add	      $4, %%esi		\n" /* ESI += 4			       */
	"   jmp	      2b		\n" /* goto 2			       */
	"3: movsl			\n" /* *EDI = *ESI, EDI += 4, ESI += 4 */
	"   sub	      %%edx, %%eax	\n" /* EAX -= EDX		       */
	"   dec	      %%ecx		\n" /* ECX--			       */
	"   jnz	      2b		\n" /* ECX == 0, no, goto 2	       */
	"   pop	      %%esi		\n" /* restore ESI		       */
	"   add	      %6, %%esi		\n" /* ESI += src_pitch		       */
	"   decl      %3		\n" /* src_height--		       */
	"   jnz	      1b		\n" /* src_height == 0, no, goto 1     */
	/* output */		:
	/* input */		: "g" (src_buf),	"g" (dst_buf),
				  "g" (src_width),	"m" (src_height),
				  "g" (dst_width),	"g" (src_offset),
				  "g" (src_pitch)
	/* clobber registers */ : "cc", "eax", "ebx", "ecx", "edx", "esi", "edi");
#else
	int width, w;
	u32 *src = (u32 *) src_buf, *dst = (u32 *) dst_buf, *buf;
	
	src += src_offset >> 1;
	while (src_height--) {
		w = dst_width >> 1;
		width = src_width >> 1;
		buf = src;
		while (w) {
			width += dst_width;
			if (width >= src_width) {
				w--;
				*dst++ = *buf;
				width -= src_width;
			}
			buf++;
		}
		src += src_pitch >> 1;
	}
#endif /* __i386__ */
}

/* Apply colour conversion routine depending on source and destination format. */
#define rivatv_get_converters(line, plane, src, dst)				 \
	line_convert_t	line  = rivatv_convert[src].converter[dst];		 \
	plane_convert_t plane = (plane_convert_t) line;				 \
	if (line == NULL) {							 \
		PRINTK_ERR ("no such converter: %s -> %s\n",			 \
			    rivatv_convert[src].name, rivatv_convert[dst].name); \
		return;								 \
	}

/* Transfer a single frame with DMA switched on and no zero copying possibly. */
static void rivatv_transfer_frame_DMA (struct rivatv_info *info, char *dst,
				       int src_width, int src_height, 
				       int dst_width, int dst_height,
				       int src_fmt, int dst_fmt)
{
	struct rivatv_port *port = &info->port;
	unsigned char *src = info->data + port->offset;
	rivatv_get_converters (line_convert, plane_convert, src_fmt, dst_fmt);

	/* for NV3 and NV4 we must scale 'in place' if the height/width does not match,
	   the NV10 and NV20 archs can scale by hardware */
	if (info->chip.arch == NV_ARCH_03 || info->chip.arch == NV_ARCH_04) {
		if (port->vld_width != dst_width || port->vld_height != dst_height) {
			rivatv_soft_scale (src, info->data, 
					   port->vld_width, port->vld_height, 0, 
					   port->vld_width, dst_width);
			src = info->data;
		}
	}
	/* do colour conversion */
	if (rivatv_convert[dst_fmt].planar)
		plane_convert (src, dst, dst_width, dst_height);
	else
		line_convert (src, dst, dst_width * dst_height);
}

/* Transfer a single frame from framebuffer into the userland capture buffer. */
static void rivatv_transfer_frame (struct rivatv_info *info, char *src, char *dst,
				   int src_width, int src_height, 
				   int dst_width, int dst_height,
				   int src_fmt, int dst_fmt)
{
	rivatv_get_converters (line_convert, plane_convert, src_fmt, dst_fmt);

	/* again for NV3 and NV4 we may scale it by software */
	if (info->chip.arch == NV_ARCH_03 || info->chip.arch == NV_ARCH_04) {
		rivatv_soft_scale (src, info->data, info->port.vld_width, src_height, 
				   (info->port.org_width - info->port.vld_width) >> 1, 
				   info->port.org_width, dst_width);
		src = info->data;
	}
	/* do colour conversion */
	if (rivatv_convert[dst_fmt].planar)
		plane_convert (src, dst, dst_width, dst_height);
	else
		line_convert (src, dst, dst_width * dst_height);
}

/* Checks whether the active video decoder can output requested palette format. */
int rivatv_check_source (struct rivatv_info *info, struct video_mmap *vm) {

	struct rivatv_source *source = info->source;
	struct rivatv_format *format = &info->format;
	int result;

	/* check whether the capture format has been changed */
	if (format->format == vm->format)
		return 0;
	if (vm->format < 1 || vm->format > 16) {
		PRINTK_ERR ("invalid capture palette: %d\n", vm->format);
		return 0;
	}
	DPRINTK ("capture palette changed: %s -> %s\n",
		 rivatv_convert[format->format].name, rivatv_convert[vm->format].name);
	format->format = vm->format;

	/* check wether the video decoder format should be changed */
	if (source->format == format->format)
		return 0;

#if NVIDIA_SUPPORTS_OTHER_FORMATS
	/* nVidia cards handle UYVY only, possibly they change it in future */
	for (n = 0; source->formats[n] != -1; n++) {
		if (source->formats[n] == format->format) {
			result = source->format = format->format;
			DPRINTK ("decoder output set to %s\n", rivatv_convert[result].name);
			return result;
		}
	}
#endif

	/* set default UYVY format if necessary */
	result = source->format = source->formats[0];
	DPRINTK ("decoder output set to default %s\n", rivatv_convert[result].name);
	return result;
}

/* Checks whether the given video format is supported. */
int rivatv_palette_supported (struct rivatv_info *info, int palette)
{
	struct rivatv_source *source = info->source;

#ifndef RIVATV_DISABLE_CONVERSION
	if (conversion) {
		/* use colour conversion code in kernel */
		if (rivatv_convert[source->format].converter[palette] != NULL)
			return 1;
	} else 
#endif /* RIVATV_DISABLE_CONVERSION */
	{
		/* if DMA is available, then check for hardware conversion */
		if (info->sched.flags & RIVATV_SCHED_DMA) {
			if (rivatv_convert_DMA[palette].valid)
				return 1;
		} 
		/* otherwise only support native format */
		else {
			if (source->format == palette)
				return 1;
		}
	}
	return 0;
}

/* Forward declarations. */
static void rivatv_bh (struct rivatv_info *info);
static void rivatv_bh_DMA (struct rivatv_info *info);

/* Initialize the capture queue. */
void rivatv_reset_queue (struct rivatv_info *info)
{
	int n;

	/* zero out everything */
	memset (&info->capture_queue, 0, sizeof (struct rivatv_cap_queue));

	/* initialize the capture tasklet */
	tasklet_init (&info->capture_queue.task,(void (*) (unsigned long)) rivatv_bh, (unsigned long) info);
	/* initialize the DMA tasklet */
	tasklet_init (&info->dma.task, (void (*) (unsigned long)) rivatv_bh_DMA, (unsigned long) info);

	/* initialize capture queue */
	init_waitqueue_head (&info->capture_queue.wait);

	/* initialize capture buffers */
	memset (&info->capbuf, 0, capbuffers * sizeof (struct rivatv_cap_buffer));
	for (n = 0; n < capbuffers; n++)
		info->capbuf[n].status = RIVATV_UNUSED;
}

/* Add a buffer to capture queue. */
int rivatv_add_queue (struct rivatv_info *info, struct video_mmap *vm)
{
	struct rivatv_cap_buffer *cap;
	struct rivatv_cap_queue *queue = &info->capture_queue;
	struct rivatv_format *format = &info->format;

	/* did the user already mmap() ? */
	if (info->capture_buffer == NULL) {
		DPRINTK ("MMAP buffer %d not available\n", vm->frame);
		return -EAGAIN;
	}

	/* check frame range */
	if (vm->frame >= info->capture_buffers || vm->frame < 0)
		return -EINVAL;

	/* check width, height and palette format */
	if (vm->height < RIVATV_MIN_VLD_HEIGHT || vm->width < RIVATV_MIN_VLD_WIDTH ||
	    vm->height > RIVATV_MAX_VLD_HEIGHT || vm->width > RIVATV_MAX_VLD_WIDTH)
		return -EINVAL;
	if (!rivatv_palette_supported (info, vm->format))
		return -EINVAL;

	spin_lock_bh (&info->queue_lock);
	if (queue->fill >= info->capture_buffers) {
		spin_unlock_bh (&info->queue_lock);
		return -EAGAIN;
	}

	/* initialize capture structure */
	cap = &info->capbuf[vm->frame];
	cap->format = vm->format;
	cap->width = vm->width;
	cap->height = vm->height;
	cap->status = RIVATV_GRABBING;
	cap->size = info->capture_buffer_size;
	cap->addr = info->capture_buffer + info->capture_buffer_size * vm->frame;

	/* setup format buffer */
	format->format = vm->format;
	format->width = vm->width;
	format->height = vm->height;
	format->size = (format->width * format->height * rivatv_convert[format->format].bpp) >> 3;

	/* update capture queue */
	if (queue->fill)
		queue->head = (queue->head + 1) % RIVATV_MAX_CAPTURE_BUFFERS;
	queue->index[queue->head] = vm->frame;
	queue->fill++;
	spin_unlock_bh (&info->queue_lock);
	return 0;
}

/* Remove a buffer from capture queue at tail position. */
static void rivatv_del_queue (struct rivatv_info *info)
{
	struct rivatv_cap_queue *queue = &info->capture_queue;

	if (queue->fill == 0) {
		DPRINTK ("assert (queue->fill != 0) failed\n");
		return;
	}

	/* update queue */
	queue->fill--;
	if (queue->fill)
		queue->tail = (queue->tail + 1) % RIVATV_MAX_CAPTURE_BUFFERS;
}

/* Capture a single frame. */
static int rivatv_capture_frame (struct rivatv_info *info, ulong io_base)
{
	char *buffer;
	int frame;
	struct rivatv_cap_queue *queue = &info->capture_queue;
	struct rivatv_cap_buffer *cap;
	struct rivatv_port *port = &info->port;

	spin_lock (&info->queue_lock);
	if (queue->fill == 0) {
		/* nothing to do */
		spin_unlock (&info->queue_lock);
		return 0;
	}

	frame = queue->index[queue->tail];
	cap = &info->capbuf[frame];

	if (io_base) {
		int ret;

		spin_lock_irq (&info->sched.sched_lock);
		ret = rivatv_startDMA (info, frame);
		spin_unlock_irq (&info->sched.sched_lock);

		if (ret) {
			spin_unlock (&info->queue_lock);
			return 1;
		}
		buffer = info->capture_buffer +	frame * info->capture_buffer_size;

		rivatv_transfer_frame (info, (void *) (io_base + info->vbi_region_size + port->offset),
				       buffer,
				       port->org_width, port->org_height,
				       cap->width, cap->height,
				       info->source->format, cap->format);
	}

	cap->status = RIVATV_READY;
	rivatv_del_queue (info);
	spin_unlock (&info->queue_lock);
	return 0;
}

/* Setup the capture buffer. */
static int __init rivatv_setup_capture (struct rivatv_info *info)
{
	if (!info->data) {
		info->data = pci_alloc_consistent (info->pci, RIVATV_RAW_CAPTURE_BUFSIZE, &info->dma.addr);
		if (info->data == NULL) {
			PRINTK_ERR ("unable to allocate YUV capture buffer (%ld kb)\n",
				    RIVATV_RAW_CAPTURE_BUFSIZE / 1024);
			return -ENOMEM;
		}
		PRINTK_INFO ("allocated YUV capture buffer (%ld kb)\n", RIVATV_RAW_CAPTURE_BUFSIZE / 1024);
	}
	return 0;
}

/* Cleanup the capture buffer. */
static void __exit rivatv_cleanup_capture (struct rivatv_info *info)
{
	if (info->data) {
		PRINTK_INFO ("freeing YUV capture buffer (%ld kb)\n", RIVATV_RAW_CAPTURE_BUFSIZE / 1024);
		pci_free_consistent (info->pci, RIVATV_RAW_CAPTURE_BUFSIZE, info->data, info->dma.addr);
		info->data = NULL;
		info->dma.addr = 0;
	}
}

/* Configure the VBI region size depending on the video format. */
static void rivatv_config_vbi (struct rivatv_info *info)
{
	info->vbi_region_size = 0; // info->port.org_width << 1;
}

/* Setup default video format. */
static void __init rivatv_set_default_format (struct rivatv_format *format)
{
	format->norm = VIDEO_MODE_PAL;
	format->width = 352;
	format->height = 288;
}

/* Startup video. */
int rivatv_video_start (struct rivatv_info *info) 
{
	int width, height;
	struct rivatv_port *port = &info->port;

	if (info->capture_started)
		return 0;

	width = info->format.width;
	height = info->format.height;
	memset (port, 0, sizeof (struct rivatv_port));

	if (info->format.norm == VIDEO_MODE_NTSC) {
		port->max_width = RIVATV_MAX_ORG_WIDTH;
		port->max_height = 240;
		port->size = RIVATV_MAX_ORG_WIDTH * 502 * 2;
		port->org_height = 60;
		port->vld_height = 60;
	} else if (info->format.norm == VIDEO_MODE_PAL || 
		   info->format.norm == VIDEO_MODE_SECAM) {
		port->max_width = RIVATV_MAX_ORG_WIDTH;
		port->max_height = 288;
		port->size = RIVATV_MAX_ORG_WIDTH * 576 * 2;
		port->org_height = 72;
		port->vld_height = 72;
	} else {
		PRINTK_ERR ("No valid video standard set: %d\n", info->format.norm);
	}

	/* Ensure alignment of decoder width and height. Necessary for NV03 and NV04. */
	if (info->chip.arch == NV_ARCH_03 || info->chip.arch == NV_ARCH_04) {

		rivatv_nearest_resolution (width, info->format.overlay, port);
		port->org_width &= ~7;
		port->org_height = port->vld_height = height;
		DPRINTK ("configured port: %dx%d -> %dx%d (%dx%d)\n", width, height, 
			 port->org_width, port->org_height, port->vld_width, port->vld_height);
	} else {
		port->org_height = port->vld_height = height;
		port->org_width = port->vld_width = width;
	}

	port->height = height;
	port->width = width;

	/* Adjust field values. */
	spin_lock_irq (&info->sched.sched_lock);
	if (port->org_height > port->max_height) {
		port->flags |= VIDEO_INTERLACED;
		info->sched.flags |= RIVATV_SCHED_INTERLACED;
	} else {
		port->flags &= ~VIDEO_INTERLACED;
		info->sched.flags &= ~RIVATV_SCHED_INTERLACED;
	}
	spin_unlock_irq (&info->sched.sched_lock);

	info->format.overlay = 0;
	rivatv_config_vbi (info);

	DPRINTK ("starting video capture\n");
	spin_lock_bh (&info->video_lock);
	spin_lock_irq (&info->sched.sched_lock);
        info->last_buffer_processed = 1;
	switch (info->chip.arch) {
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
		rivatv_start_video_nv10 (info);
		break;
	case NV_ARCH_03:
	case NV_ARCH_04:
		rivatv_start_video_nv04 (info);
		break;
	}
	rivatv_configureDMA (info);
	info->sched.last_jiffies = jiffies;
	spin_unlock_irq (&info->sched.sched_lock);

	info->capture_started = 1;
	spin_unlock_bh (&info->video_lock);

	return 0;
}

/* Stop the video. */
int rivatv_video_stop (struct rivatv_info *info)
{
	struct rivatv_port *port = &info->port;

	if (info->capture_started == 0)
		return 0;

	memset (port, 0, sizeof (struct rivatv_port));
	DPRINTK ("stopping video capture\n");
	spin_lock_bh (&info->video_lock);
	spin_lock_irq (&info->sched.sched_lock);
	switch (info->chip.arch) {
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
		rivatv_stop_video_nv10 (info);
		break;
	case NV_ARCH_03:
	case NV_ARCH_04:
		rivatv_stop_video_nv04 (info);
		break;
	}
	spin_unlock_irq (&info->sched.sched_lock);
	info->capture_started = 0;
	spin_unlock_bh (&info->video_lock);
	return 0;
}

/* Checks whether the the current capture format is valid or has been changed. */
int rivatv_check_format (struct rivatv_info *info, struct video_mmap *vm)
{
	struct rivatv_format *format = &info->format;

	if (vm->format < 1 || vm->format > 16) {
		PRINTK_ERR ("invalid capture palette: %d\n", vm->format);
		return -EINVAL;
	}
	if (format->width != vm->width || format->height != vm->height) {
		DPRINTK ("capture resolution changed: %dx%d -> %dx%d\n", 
			 format->width, format->height, vm->width, vm->height);
		format->width = vm->width;
		format->height = vm->height;
		return 1;
	}
	return 0;
}

/* The bottom half of the interrupt service routine for DMA. */
void rivatv_bh_DMA (struct rivatv_info *info)
{
	char *dstbuffer;
	int frame;
	struct rivatv_cap_queue *queue = &info->capture_queue;
	struct rivatv_cap_buffer *cap;
	struct rivatv_port *port = &info->port;

	spin_lock (&info->queue_lock);
	frame = queue->index[queue->tail];
	cap = &info->capbuf[frame];

	/* if no zero copy DMA is possibly then transfer data into final 
	   location (userland) here */
	if (!(info->sched.flags & RIVATV_SCHED_DMA_ZC)) {
		dstbuffer = info->capture_buffer + frame * info->capture_buffer_size;
		rivatv_transfer_frame_DMA (info, dstbuffer,
					   port->org_width, port->org_height,
					   cap->width, cap->height,
					   info->source->format, cap->format);
	}

	/* signal that this buffer is ready and remove it from the capture queue */
	cap->status = RIVATV_READY;
	rivatv_del_queue (info);
	spin_unlock (&info->queue_lock);

	/* remember the timestamp when this occurred */
	spin_lock_irq (&info->sched.sched_lock);
	info->sched.last_jiffies = jiffies;
	spin_unlock_irq (&info->sched.sched_lock);

	/* wake up the VIDIOCSYNC ioctl */
	wake_up_interruptible (&info->capture_queue.wait);
}

/* The bottom half of the interrupt service routine. */
static void rivatv_bh (struct rivatv_info *info)
{
	int ret;

	/* capture this frame */
	spin_lock_irq (&info->sched.sched_lock);
	if (jiffies - info->sched.last_jiffies > HZ / 10) {
		spin_unlock_irq (&info->sched.sched_lock);

		spin_lock (&info->video_lock);
		ret = rivatv_capture_frame (info, (ulong) NULL);
		spin_unlock (&info->video_lock);
	} else {
		spin_unlock_irq (&info->sched.sched_lock);

		spin_lock (&info->video_lock);
		ret = rivatv_capture_frame (info, (ulong) info->picture_base);
		spin_unlock (&info->video_lock);
	}

	/* for none-DMA transfers */
	if (ret == 0) {
		spin_lock_irq (&info->sched.sched_lock);
		info->sched.last_jiffies = jiffies;
		rivatv_schedule_next (info);
		spin_unlock_irq (&info->sched.sched_lock);

		/* finally wake up the VIDIOCSYNC ioctl */
		wake_up_interruptible (&info->capture_queue.wait);
	}
}

/* Converts a kernel virtual address returned by vmalloc_32() into the
   a physical address. */
static unsigned long vmalloc_to_phys_addr (unsigned long address)
{
	unsigned long ret;

        /* get the address the page is refering to */
	ret = (unsigned long) page_address (vmalloc_to_page ((void *) address));
	/* add the offset within the page to the page address */
	ret |= (address & ~PAGE_MASK);
	return virt_to_phys ((void *) ret);
}

/* Converts a kernel virtual address returned by vmalloc_32() into the
   a (pci) bus address. */
unsigned long vmalloc_to_bus_addr (unsigned long address)
{
	unsigned long ret;

        /* get the address the page is refering to */
	ret = (unsigned long) page_address (vmalloc_to_page ((void *) address));
	/* add the offset within the page to the page address */
	ret |= (address & ~PAGE_MASK);
	return virt_to_bus ((void *) ret);
}

/* Allocate an arena of size bytes. Return non-zero on errors. */
static int rivatv_alloc_arena (struct rivatv_info *info, ulong size)
{
	ulong virt_addr;
	void *ptr;
	
	/* get a memory area that is only virtually continuous */
	if ((ptr = vmalloc_32 (size)) == NULL)
		return -1;

	DPRINTK ("allocated %ld kb MMAP buffer space\n", size / 1024);

	/* reserve all pages and initialize these */
	virt_addr = (ulong) ptr;
	memset (ptr, 0, size);
	while ((long) size > 0) {
		SetPageReserved (vmalloc_to_page ((void *) virt_addr));
		size -= PAGE_SIZE;
		virt_addr += PAGE_SIZE;
	}

	/* save this address */
	info->capture_buffer = ptr;
	return 0;
}

/* Free an arena previously reserved by rivatv_alloc_arena(). */
static void __exit rivatv_free_arena (struct rivatv_info *info)
{
	ulong virt_addr, size;

	/* unreserve all pages */
	if (info->capture_buffer) {
		size = info->capture_buffer_size * info->capture_buffers;
		virt_addr = (ulong) info->capture_buffer;
		DPRINTK ("freeing MMAP buffer space (%ld kb)\n", size / 1024);
		while ((long) size > 0) {
			ClearPageReserved (vmalloc_to_page ((void *) virt_addr));
			size -= PAGE_SIZE;
			virt_addr += PAGE_SIZE;
		}
		vfree (info->capture_buffer);
	}

	/* and free the area */
	info->capture_buffer = NULL;
	info->capture_buffer_size = 0;
}

/* Unmaps the capture buffer in kernel from virtual user space. */
static void __exit rivatv_munmap (struct rivatv_info *info)
{
	rivatv_free_arena (info);
}

/* Maps the capture buffer in kernel to the virtual user space address ADR. 
   Allocate the capture buffer first if necessary. */
int rivatv_mmap (struct rivatv_info *info, struct vm_area_struct *vma, ulong adr, ulong size)
{
	ulong start = adr;
	ulong vsize, virt_addr;

	/* check the size */
	vsize = capbuffers * RIVATV_CAPTURE_BUFSIZE;
	if (size > vsize)
		return -EINVAL;

	if (!info->capture_buffer) {
		/* allocate capture buffer */
		if (rivatv_alloc_arena (info, vsize) != 0) {
			PRINTK_ERR ("unable to allocate capture buffer (%ld kb)\n", vsize);
			return -ENOMEM;
		}
		info->capture_buffers = capbuffers;
		info->capture_buffer_size = RIVATV_CAPTURE_BUFSIZE;
	}

	/* map the capture buffer to user space page by page */
	virt_addr = (ulong) info->capture_buffer;
	vsize = size;
	while ((long) size > 0) {
		if (remap_page_range (vma, start, vmalloc_to_phys_addr (virt_addr),
				      PAGE_SIZE, RIVATV_PAGE_SHARED)) {
			PRINTK_ERR ("failed to put MMAP buffer to user space\n");
			return -EAGAIN;
		}
		start += PAGE_SIZE;
		virt_addr += PAGE_SIZE;
		size -= PAGE_SIZE;
	}

	/* state this in kernel messages */
	DPRINTK ("MMAP buffer available in user space (%ld kb)\n", vsize / 1024);
	return 0;
}

/* Synchronize function for mmap() grabbing. */
int rivatv_sync (struct rivatv_info *info, int frame)
{
	int ret = 0;
	struct rivatv_cap_buffer *cap;
	struct rivatv_cap_queue *queue;

	/* check the frames range */
	if (frame < 0 || frame >= info->capture_buffers)
		return -EINVAL;

	cap = &info->capbuf[frame];
	queue = &info->capture_queue;

sync_loop:
	spin_lock_bh (&info->queue_lock);
	switch (cap->status) {
	case RIVATV_UNUSED:		/* grabbing buffer unused */
		ret = -EINVAL;
		break;
	case RIVATV_GRABBING:		/* grabbing buffer currently in use */
		spin_unlock_bh (&info->queue_lock);
		if (wait_event_interruptible (queue->wait, cap->status != RIVATV_GRABBING)) {
			return -EINTR;
		}
		goto sync_loop;
	case RIVATV_READY:		/* grabbing buffer ready */
		break;
	case RIVATV_ERROR:		/* error in grabbing buffer */
		ret = -EIO;
		break;
	default:
		DPRINTK ("this should never happen\n");
		ret = -EIO;
		break;
	}
	if (!ret)
		cap->status = RIVATV_UNUSED;
	spin_unlock_bh (&info->queue_lock);
	return ret;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)

/* SysFS support. */
static ssize_t rivatv_print_card (struct class_device *cd, char *buf)
{
        struct video_device *dev = to_video_device (cd);
        struct rivatv_info *info = video_get_drvdata (dev);
        return sprintf (buf, "nVidia %s: %s\n", info->driver, info->nicecardname);
}
static CLASS_DEVICE_ATTR (card, S_IRUGO, rivatv_print_card, NULL);

static ssize_t rivatv_print_dev (struct class_device *cd, char *buf)
{
	struct video_device *dev = to_video_device (cd);
	return print_dev_t (buf, MKDEV (VIDEO_MAJOR, dev->minor));
}
static CLASS_DEVICE_ATTR (dev, S_IRUGO, rivatv_print_dev, NULL);

static ssize_t rivatv_print_decoder (struct class_device *cd, char *buf)
{
        struct video_device *dev = to_video_device (cd);
        struct rivatv_info *info = video_get_drvdata (dev);
        return sprintf (buf, "%s\n", info->i2c->video_decoder ? i2c_clientname (info->i2c->video_decoder) : "unavailable");
}
static CLASS_DEVICE_ATTR (decoder, S_IRUGO, rivatv_print_decoder, NULL);

static void rivatv_register_sysfs (struct rivatv_info *info)
{
        video_device_create_file (info->video, &class_device_attr_card);
        video_device_create_file (info->video, &class_device_attr_dev);
        video_device_create_file (info->video, &class_device_attr_decoder);
}

static void rivatv_unregister_sysfs (struct rivatv_info *info)
{
        video_device_remove_file (info->video, &class_device_attr_decoder);
        video_device_remove_file (info->video, &class_device_attr_dev);
        video_device_remove_file (info->video, &class_device_attr_card);
}

#else
# define rivatv_register_sysfs(x)
# define rivatv_unregister_sysfs(x)
#endif /* SysFS */

static int __init riva_setup_video_dev (struct rivatv_info *info, 
					const struct rivatv_initdata *v4l_init)
{
	int err;
	
	/* prepare device structures and copy templates */
	if ((info->video = video_device_alloc ()) == NULL)
		goto err_out;
	if ((info->vbi = video_device_alloc ()) == NULL)
		goto err_out_kfree_video;
	*info->video = *v4l_init->video;
	*info->vbi = *v4l_init->vbi;

	/* initialize some fields */
	video_set_drvdata (info->video, info);
	video_set_drvdata (info->vbi, info);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	info->video->release = video_device_release;
	info->vbi->release = video_device_release;
#endif

	if ((err = video_register_device (info->video, VFL_TYPE_GRABBER, -1)) < 0)
		goto err_out_kfree_vbi;
	if ((err = video_register_device (info->vbi, VFL_TYPE_VBI, -1)) < 0)
		goto err_out_unregister_video;
	rivatv_register_sysfs (info);

	if ((err = rivatv_setup_capture (info)) != 0)
		goto err_out_unregister_sysfs;

	rivatv_checkAGP (info);
	rivatv_checkDMA (info);
	rivatv_set_default_format (&info->format);
	rivatv_reset_queue (info);
	if ((err = rivatv_install_irq (info)) != 0)
		goto err_out_cleanup_capture;

	PRINTK_INFO ("Video4Linux device driver registered\n");
	return 0;

 err_out_cleanup_capture:
	rivatv_cleanup_capture (info);
 err_out_unregister_sysfs:
	rivatv_unregister_sysfs (info);
	if (info->vbi) {
		video_unregister_device (info->vbi);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
		info->vbi = NULL;
#endif
	}
 err_out_unregister_video:
	if (info->video) {
		video_unregister_device (info->video);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
		info->video = NULL;
#endif
	}
 err_out_kfree_vbi:
	if (info->vbi) {
		kfree (info->vbi);
		info->vbi = NULL;
	}
 err_out_kfree_video:
	if (info->video) {
		kfree (info->video);
		info->video = NULL;
	}
 err_out:
	return -1;
}

static void __exit riva_release_video_dev (struct rivatv_info *info)
{
	rivatv_deinstall_irq (info);
	rivatv_video_stop (info);
	rivatv_reset_queue (info);
	rivatv_munmap (info);
	rivatv_cleanup_capture (info);
	rivatv_cleanupDMA (info);

	rivatv_unregister_sysfs (info);
	video_unregister_device (info->vbi);
	video_unregister_device (info->video);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
	kfree (info->vbi);
	kfree (info->video);
#endif
}


/* ------------------------------------------------------------------------- *
 *
 * basic chip specific functions
 *
 * ------------------------------------------------------------------------- */

static void rivatv_lock_nv03 (struct rivatv_chip *chip, int LockUnlock)
{
	VID_WR08 (chip->PVIO, 0x3C4, 0x06);
	VID_WR08 (chip->PVIO, 0x3C5, LockUnlock ? 0x99 : 0x57);
}

static void rivatv_lock_nv04 (struct rivatv_chip *chip, int LockUnlock)
{
	VID_WR08 (chip->PCIO, 0x3C4, 0x06);
	VID_WR08 (chip->PCIO, 0x3C5, LockUnlock ? 0x99 : 0x57);
	VID_WR08 (chip->PCIO, 0x3D4, 0x1F);
	VID_WR08 (chip->PCIO, 0x3D5, LockUnlock ? 0x99 : 0x57);
}

#define rivatv_lock_nv10 rivatv_lock_nv04

static ulong __init rivatv_fbsize_nv03 (struct rivatv_chip *chip)
{
	if (VID_RD32 (chip->PFB, 0) & 0x00000020) {
		if (((VID_RD32 (chip->PMC, 0) & 0xF0) == 0x20)
		    && ((VID_RD32 (chip->PMC, 0) & 0x0F) >= 0x02)) {
			/* SDRAM 128 ZX. */
			return ((1 << (VID_RD32 (chip->PFB, 0) & 0x03)) * 1024 * 1024);
		}
		else {
			return 1024 * 1024 * 8;
		}
	}
	else {
		/* SGRAM 128. */
		switch (VID_RD32 (chip->PFB, 0) & 0x03) {
		case 0:
			return 1024 * 1024 * 8;
			break;
		case 2:
			return 1024 * 1024 * 4;
			break;
		default:
			return 1024 * 1024 * 2;
			break;
		}
	}
}

static ulong __init rivatv_fbsize_nv04 (struct rivatv_chip *chip)
{
	if (VID_RD32 (chip->PFB, 0) & 0x00000100) {
		return ((VID_RD32 (chip->PFB, 0) >> 12) & 0x0F) * 1024 * 1024 * 2
			+ 1024 * 1024 * 2;
	} else {
		switch (VID_RD32 (chip->PFB, 0) & 0x00000003) {
		case 0:
			return 1024 * 1024 * 32;
			break;
		case 1:
			return 1024 * 1024 * 4;
			break;
		case 2:
			return 1024 * 1024 * 8;
			break;
		case 3:
		default:
			return 1024 * 1024 * 16;
			break;
		}
	}
}

static ulong __init rivatv_fbsize_nv10 (struct rivatv_chip *chip)
{
	return ((VID_RD32 (chip->PFB, 0x20C) >> 20) & 0x000000FF) * 1024 * 1024;
}

static ulong __init rivatv_fbsize_nv30 (struct rivatv_chip *chip)
{
	return ((VID_RD32 (chip->PFB, 0x20C) >> 20) & 0x000001FF) * 1024 * 1024;
}

/* ------------------------------------------------------------------------- *
 *
 * PCI card identification
 *
 * ------------------------------------------------------------------------- */

enum rivatv_chips {
	CH_RIVA_128 = 0,
	CH_RIVA_TNT,
	CH_RIVA_TNT2,
	CH_RIVA_UTNT2,
	CH_RIVA_VTNT2,
	CH_RIVA_UVTNT2,
	CH_RIVA_ITNT2,
	CH_GEFORCE_SDR,
	CH_GEFORCE_DDR,
	CH_QUADRO,
	CH_GEFORCE2_MX,
	CH_QUADRO2_MXR,
	CH_GEFORCE2_GTS,
	CH_GEFORCE2_TI,
	CH_GEFORCE2_ULTRA,
	CH_QUADRO2_PRO,
	CH_GEFORCE3,
	CH_GEFORCE3_1,
	CH_GEFORCE3_2,
	CH_GEFORCE4_MX,
	CH_GEFORCE4_MX_8X,
	CH_GEFORCE4_TI,
	CH_GEFORCE4_TI_4800,
	CH_GEFORCE4_TI_8X,
	CH_GEFORCE4_TI_SE,
	CH_GEFORCE_FX_5800_ULTRA,
	CH_GEFORCE_FX_5800,
	CH_GEFORCE_FX_5600_ULTRA,
	CH_GEFORCE_FX_5600,
	CH_GEFORCE_FX_5600_XT,
	CH_GEFORCE_FX_5200_1,
	CH_GEFORCE_FX_5200_ULTRA,
	CH_GEFORCE_FX_5200,
	CH_GEFORCE_FX_5200_SE,
	CH_GEFORCE_FX_5900_ULTRA,
	CH_GEFORCE_FX_5900,
	CH_GEFORCE_FX_5900_XT,
	CH_GEFORCE_FX_5950_ULTRA,
	CH_GEFORCE_FX_5700_ULTRA,
	CH_GEFORCE_FX_5700,
};

/* Directly indexed by rivatv_chips, above. */
static struct rivatv_chip_info {
	const char *description;
	int arch;
	int realarch;
} rivatv_chip_info[] __devinitdata = {
	{ "RIVA 128",	           NV_ARCH_03, NV_ARCH_03 },
	{ "RIVA TNT",	           NV_ARCH_04, NV_ARCH_04 },
	{ "RIVA TNT2",	           NV_ARCH_04, NV_ARCH_05 },
	{ "RIVA TNT2 Ultra",       NV_ARCH_04, NV_ARCH_05 },
	{ "Vanta",                 NV_ARCH_04, NV_ARCH_05 },
	{ "RIVA TNT2 Model 64",    NV_ARCH_04, NV_ARCH_05 },
	{ "Aladdin TNT2",          NV_ARCH_04, NV_ARCH_05 },
	{ "GeForce 256",           NV_ARCH_10, NV_ARCH_10 },
	{ "GeForce DDR",           NV_ARCH_10, NV_ARCH_10 },
	{ "Quadro",	           NV_ARCH_10, NV_ARCH_10 },
	{ "GeForce2 MX",           NV_ARCH_10, NV_ARCH_11 },
	{ "Quadro2 MXR",           NV_ARCH_10, NV_ARCH_11 },
	{ "GeForce2 GTS",          NV_ARCH_10, NV_ARCH_15 },
	{ "GeForce2 Ti",           NV_ARCH_10, NV_ARCH_15 },
	{ "GeForce2 Ultra",        NV_ARCH_10, NV_ARCH_15 },
	{ "Quadro2 Pro",           NV_ARCH_10, NV_ARCH_15 },
	{ "GeForce3",	           NV_ARCH_20, NV_ARCH_20 },
	{ "GeForce3 Ti200",        NV_ARCH_20, NV_ARCH_20 },
	{ "GeForce3 Ti500",        NV_ARCH_20, NV_ARCH_20 },
	{ "GeForce4 MX4x0",        NV_ARCH_10, NV_ARCH_17 },
	{ "GeForce4 MX4x0 8xAGP",  NV_ARCH_10, NV_ARCH_18 },
	{ "GeForce4 Ti4x00",       NV_ARCH_20, NV_ARCH_25 },
	{ "GeForce4 Ti4800",       NV_ARCH_20, NV_ARCH_28 },
	{ "GeForce4 Ti4200 8xAGP", NV_ARCH_20, NV_ARCH_28 },
	{ "GeForce4 Ti4800 SE",    NV_ARCH_20, NV_ARCH_28 },
	{ "GeForce FX 5800 Ultra", NV_ARCH_30, NV_ARCH_30 },
	{ "GeForce FX 5800",       NV_ARCH_30, NV_ARCH_30 },
	{ "GeForce FX 5600 Ultra", NV_ARCH_30, NV_ARCH_31 },
	{ "GeForce FX 5600",       NV_ARCH_30, NV_ARCH_31 },
	{ "GeForce FX 5600 XT",    NV_ARCH_30, NV_ARCH_31 },
	{ "GeForce FX 5200",       NV_ARCH_30, NV_ARCH_32 },
	{ "GeForce FX 5200 Ultra", NV_ARCH_30, NV_ARCH_32 },
	{ "GeForce FX 5200",       NV_ARCH_30, NV_ARCH_32 },
	{ "GeForce FX 5200 SE",    NV_ARCH_30, NV_ARCH_32 },
	{ "GeForce FX 5900 Ultra", NV_ARCH_30, NV_ARCH_33 },
	{ "GeForce FX 5900",       NV_ARCH_30, NV_ARCH_33 },
	{ "GeForce FX 5900 XT",    NV_ARCH_30, NV_ARCH_33 },
	{ "GeForce FX 5950 Ultra", NV_ARCH_30, NV_ARCH_33 },
	{ "GeForce FX 5700 Ultra", NV_ARCH_30, NV_ARCH_34 },
	{ "GeForce FX 5700",       NV_ARCH_30, NV_ARCH_34 },
};

/* List of supported PCI devices. */
static struct pci_device_id rivatv_pci_table[] __devinitdata = {
	/* NV_ARCH_03 */
	{ PCI_VENDOR_ID_NVIDIA_SGS, PCI_DEVICE_ID_NVIDIA_SGS_RIVA128,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_128 },
	/* NV_ARCH_04 */
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_TNT,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_TNT },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_TNT2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_TNT2 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_UTNT2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_UTNT2 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_VTNT2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_VTNT2 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_UVTNT2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_UVTNT2 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_ITNT2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_RIVA_ITNT2 },
	/* NV_ARCH_10 */
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_SDR,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_SDR },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_DDR,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_DDR },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_QUADRO,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_QUADRO },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE2_MX,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE2_MX },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE2_MX2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE2_MX },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_QUADRO2_MXR,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_QUADRO2_MXR },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE2_GTS,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE2_GTS },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE2_GTS2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE2_TI },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE2_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE2_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_QUADRO2_PRO,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_QUADRO2_PRO },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_MX_460,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_MX },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_MX_440,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_MX },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_MX_420,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_MX },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_MX_440_8X,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_MX_8X },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_MX_440SE_8X,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_MX_8X },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_MX_420_8X,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_MX_8X },
	/* NV_ARCH_20 */
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE3,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE3 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE3_1,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE3_1 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE3_2,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE3_2 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_TI_4600,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_TI },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_TI_4400,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_TI },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_TI_4200,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_TI },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_TI_4800,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_TI_4800 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_TI_4200_8X,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_TI_8X },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE4_TI_4800_SE,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE4_TI_SE },
	/* NV_ARCH_30 */
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5800_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5800_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5800,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5800 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5600_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5600_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5600,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5600 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5600_XT,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5600_XT },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5200_1,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5200_1 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5200_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5200_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5200,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5200 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5200_SE,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5200_SE },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5900_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5900_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5900,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5900 },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5900_XT,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5900_XT },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5950_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5950_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5700_ULTRA,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5700_ULTRA },
	{ PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_GEFORCE_FX_5700,
	  PCI_ANY_ID, PCI_ANY_ID, 0, 0, CH_GEFORCE_FX_5700 },
	{ 0, } /* end of list */
};

/* rivatv_identify_pci() tries to identify which board we're supposed to drive
   and which video decoder chip goes with it. If it cannot identify the board,
   or cannot load the decoder module, it fails. */
static int __init rivatv_identify_pci (struct pci_dev *pd, struct rivatv_info *info)
{
	int i;
	struct rivatv_card *rcard = NULL;

	/* check the list of supported cards */
	if (pd->subsystem_vendor != 0x0000) {
		for (i = 0; rivatv_cards[i].vendor != (unsigned short) PCI_ANY_ID; i++) {
			if (rivatv_cards[i].vendor == pd->vendor &&
			    rivatv_cards[i].device == pd->device &&
			    rivatv_cards[i].subsystem_vendor == pd->subsystem_vendor &&
			    rivatv_cards[i].subsystem_device == pd->subsystem_device) {
				rcard = &rivatv_cards[i];
				memcpy (&info->card, rcard, sizeof (info->card));
				memcpy (info->nicecardname, rcard->name, sizeof (info->nicecardname));
				if (rcard->audio_required != -1)
					info->audio_required = rcard->audio_required;
				if (rcard->tuner_required != -1)
					info->tuner_required = rcard->tuner_required;
				break;
			}
		}
	}

	if (rcard == NULL) {

		/* card module parameter not set */
		if (card == -1) {
			struct rivatv_vendor *vendor = NULL;

			/* check list of supported vendors */
			for (i = 0; rivatv_vendors[i].subsystem_vendor != (unsigned short) PCI_ANY_ID; i++) {
				if (rivatv_vendors[i].subsystem_vendor == pd->subsystem_vendor) {
					vendor = &rivatv_vendors[i];
					sprintf (info->nicecardname, "%s [UNKNOWN]", vendor->name);
					PRINTK_INFO ("Unknown %s board: card not in database\n", vendor->name);
					break;
				}
			}

			if (vendor == NULL) {
				strcpy (info->nicecardname, "[UNKNOWN]");
				if (pd->subsystem_vendor == 0x0000)
					PRINTK_INFO ("It is impossible to identify your board uniquely, sorry\n");
				else
					PRINTK_INFO ("Could not identify your board\n");
			}

			PRINTK_INFO ("PCI identifiers: %04X %04X %04X %04X\n",
				     pd->vendor, pd->device, pd->subsystem_vendor, pd->subsystem_device);

#ifdef CONFIG_KMOD
			if (autoload) {
				PRINTK_INFO ("Attempting to load most common decoder (SAA7108E)\n");
				return request_module ("saa7108e");
			}
#endif /* CONFIG_KMOD */
			return 0;
		}

		/* card module parameter set */
		rcard = &rivatv_cards[card - 1];
		memcpy (&info->card, rcard, sizeof (info->card));
		memcpy (info->nicecardname, rcard->name, sizeof (info->nicecardname));
		if (rcard->audio_required != -1)
			info->audio_required = rcard->audio_required;
		if (rcard->tuner_required != -1)
			info->tuner_required = rcard->tuner_required;
	}

	PRINTK_INFO ("Identified your board as %s\n", info->nicecardname);
	info->card.identified = 1;
#ifdef CONFIG_KMOD
	if (autoload) {
		if (info->tuner_required != -1) {
			PRINTK_INFO ("Attempting to load tuner module\n");
			request_module ("tuner");
		}
		if (info->ir_required != -1) {
			PRINTK_INFO ("Attempting to load lirc_i2c module\n");
			request_module ("lirc_i2c");
		}
		if (info->audio_required != -1) {
			PRINTK_INFO ("Attempting to load tvaudio module\n");
			request_module ("tvaudio");
			PRINTK_INFO ("Attempting to load tvmixer module\n");
			request_module ("tvmixer");
		}
		PRINTK_INFO ("Attempting to load module %s\n", rcard->decoder);
		return request_module (rcard->decoder);
	}
#endif /* CONFIG_KMOD */
	return 0;
}

/* Checks which driver owns the given resource address. */
static char * __init rivatv_check_resource (struct resource *entry, unsigned long addr)
{
	struct resource * e = entry;
	char * res = NULL;
	while (e) {
		if (addr >= e->start && addr <= e->end && (e->flags & IORESOURCE_BUSY)) {
			res = (char *) (e->name ? e->name : "<BAD>");
			PRINTK_INFO ("0x%08lx - 0x%08lx is reserved by %s\n", e->start, e->end, res);
			return res;
		}
		if (e->child) {
			res = rivatv_check_resource (e->child, addr);
			if (res) return res;
		}
		e = e->sibling;
	}
	return res;
}

/* ------------------------------------------------------------------------- *
 *
 * PCI bus
 *
 * ------------------------------------------------------------------------- */

int __init rivatv_init_pci (struct pci_dev *pd,
			    const struct pci_device_id *ent)
{
	struct rivatv_info *info;
	struct rivatv_chip_info *rci = &rivatv_chip_info[ent->driver_data];
	unsigned short command_word;
	int err;

	if (rivatv_devices > RIVATV_MAX_DEVICES)
		return -EPERM;

	/* initialize private data */
	info = kmalloc (sizeof (struct rivatv_info), GFP_KERNEL);
	if (info == NULL) {
		PRINTK_ERR ("unable to allocate driver private data\n");
		return -ENOMEM;
	}
	memset (info, 0, sizeof (struct rivatv_info));

	info->nr = rivatv_devices;
	PRINTK_INFO ("nVidia card found - rivatv%d\n", info->nr);

	/* setup spin locks */
	info->video_lock = SPIN_LOCK_UNLOCKED;
	info->queue_lock = SPIN_LOCK_UNLOCKED;
	info->sched.sched_lock = SPIN_LOCK_UNLOCKED;
	init_MUTEX (&info->decoder_lock);

	info->driver = rci->description;
	info->chip.arch = rci->arch;
	info->chip.realarch = rci->realarch;

	/* make sure this buffer is always NULL-terminated */
	info->nicecardname[RIVATV_NICENAME_BUFSIZE - 1] = '\0';

	info->tuner_required = -1;
	info->ir_required = -1;

	/* initialize some audio variables */
	info->audio_required = -1;
	info->audio.balance = info->audio.treble = info->audio.bass = 32768;
	info->audio.volume = 0xffff;
	info->audio.mode = VIDEO_SOUND_MONO | VIDEO_SOUND_STEREO | VIDEO_SOUND_LANG2 | VIDEO_SOUND_LANG1;

	/* initialize picture variables */
	info->picture.brightness = 32768;
	info->picture.hue	 = 32768;
	info->picture.colour	 = 32768;
	info->picture.contrast	 = 32768;
	info->picture.whiteness	 = 32768;
	info->picture.depth	 = 24;
	info->picture.palette	 = VIDEO_PALETTE_RGB24;

	/* check if TV-Box should be used */
	switch (tvbox) {
		/* Philips FI1216 and compatibles */
	case 1: info->tuner_required = TUNER_PHILIPS_PAL; break;
		/* Philips FI1236, FM1236 and compatibles */
	case 2: info->tuner_required = TUNER_PHILIPS_NTSC; break;
		/* Philips FI1236 MK2 */
	case 3: info->tuner_required = TUNER_PHILIPS_NTSC_M; break;
		/* Philips FI1256 and compatibles */
	case 4: info->tuner_required = TUNER_PHILIPS_PAL_DK; break;
		/* Philips FQ1216ME */
	case 5: info->tuner_required = TUNER_PHILIPS_FQ1216ME; break;
		/* Philips FI1216MF, FM1216MF, FR1216MF */
	case 6: info->tuner_required = TUNER_PHILIPS_SECAM; break;
		/* Philips FI1246 and compatibles */
	case 7: info->tuner_required = TUNER_PHILIPS_PAL_I; break;
		/* invalid */
	default:
		tvbox = 0; break;
	}
	if (tvbox) {
		info->audio_required = 1;
		info->ir_required = 1;
	}

	strcpy (info->nicecardname, "[UNDETERMINED]");
	if (rivatv_identify_pci (pd,info) < 0)
		PRINTK_ERR ("decoder chip module autoloading failed\n");

	/* enable PCI device and set Bus-Mastering */
	if (pci_enable_device (pd) < 0) {
		PRINTK_ERR ("failed to enable PCI device\n");
		goto err_out_kfree;
	}
	pci_read_config_word (pd, PCI_COMMAND, &command_word);
	if (!(command_word & PCI_COMMAND_MEMORY) || !(command_word & PCI_COMMAND_MASTER)) {
		PRINTK_INFO ("device not PCI enabled; enabling\n");
		command_word |= (PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
		pci_write_config_word (pd, PCI_COMMAND, command_word);
	}

	info->pci = pd;
	info->base0_region_size = pci_resource_len (pd, 0);
	info->base1_region_size = pci_resource_len (pd, 1);
	info->base0_region = pci_resource_start (pd, 0);
	info->base1_region = pci_resource_start (pd, 1);

	if (!request_mem_region (info->base0_region,
				 info->base0_region_size, "rivatv")) {
		PRINTK_ERR ("cannot reserve MMIO region at 0x%08lx\n", info->base0_region);
		rivatv_check_resource (&iomem_resource, info->base0_region);
		goto err_out_kfree;
	}

	if (!request_mem_region (info->base1_region,
				 info->base1_region_size, "rivatv")) {
		char * foo;
		PRINTK_ERR ("cannot reserve FB region at 0x%08lx\n",
			    info->base1_region);
		foo = rivatv_check_resource (&iomem_resource, info->base1_region);
		if (!foo || strcmp (foo, "vesafb"))
			goto err_out_free_base0;
	}

	info->control_base = ioremap_nocache (info->base0_region,
					      info->base0_region_size);
	if (!info->control_base) {
		PRINTK_ERR ("cannot ioremap MMIO base at 0x%08lx\n",
			    info->base0_region);
		goto err_out_free_base1;
	}

	info->chip.PFIFO  = (uint32_t *) (info->control_base + 0x00002000);
	info->chip.FIFO   = (uint32_t *) (info->control_base + 0x00800000);
	info->chip.PMC    = (uint32_t *) (info->control_base + 0x00000000);
	info->chip.PFB    = (uint32_t *) (info->control_base + 0x00100000);
	info->chip.PME    = (uint32_t *) (info->control_base + 0x00000000);
	info->chip.PCIO   = (uint8_t *)  (info->control_base + 0x00601000);
	info->chip.PVIO   = (uint8_t *)  (info->control_base + 0x000C0000);
	info->chip.PGRAPH = (uint32_t *) (info->control_base + 0x00400000);

	/* setup chip specific functions */
	switch (info->chip.arch) {
	case NV_ARCH_03:
		info->chip.lock = rivatv_lock_nv03;
		info->chip.fbsize = rivatv_fbsize_nv03 (&info->chip);
		info->chip.PVIDEO = (uint32_t *) (info->control_base + 0x00680000);
		break;
	case NV_ARCH_04:
		info->chip.lock = rivatv_lock_nv04;
		info->chip.fbsize = rivatv_fbsize_nv04 (&info->chip);
		info->chip.PRAMIN = (uint32_t *) (info->control_base + 0x00700000);
		info->chip.PVIDEO = (uint32_t *) (info->control_base + 0x00680000);
		break;
	case NV_ARCH_10:
	case NV_ARCH_20:
		info->chip.lock = rivatv_lock_nv10;
		info->chip.fbsize = rivatv_fbsize_nv10 (&info->chip);
		info->chip.PRAMIN = (uint32_t *) (info->control_base + 0x00700000);
		info->chip.PVIDEO = (uint32_t *) (info->control_base + 0x00008000);
		break;
	case NV_ARCH_30:
		info->chip.lock = rivatv_lock_nv10;
		info->chip.fbsize = rivatv_fbsize_nv30 (&info->chip);
		info->chip.PRAMIN = (uint32_t *) (info->control_base + 0x00700000);
		info->chip.PVIDEO = (uint32_t *) (info->control_base + 0x00008000);
		break;
	}

	switch (info->chip.arch) {
	case NV_ARCH_03:
	{
		/* This maps framebuffer @3MB, thus 1MB is left for video. */
		u32 offset = PAGE_ALIGN (1024 * 768 * 4);
		info->video_base = ioremap (info->base1_region, info->base1_region_size);
		if (!info->video_base) {
			PRINTK_ERR ("cannot ioremap FB base at 0x%08lx\n",
				    info->base1_region);
			goto err_out_iounmap_ctrl;
		}
		/* This may trash your screen for resolutions greater than 1024x768, sorry. */
		info->picture_offset = offset;
		info->picture_base = (unsigned long) info->video_base + offset;
		info->chip.PRAMIN = (uint32_t *) (info->video_base + 0x00C00000);
		break;
	}
	case NV_ARCH_04:
	case NV_ARCH_10:
	case NV_ARCH_20:
	case NV_ARCH_30:
	{
		u32 size = PAGE_ALIGN (RIVATV_MAX_ORG_WIDTH * 2000 * 4);
		u32 offset = info->chip.fbsize - size;
		info->picture_offset = offset;
		info->picture_base = (ulong) ioremap (info->base1_region + offset, size);
		if (!info->picture_base) {
			PRINTK_ERR ("cannot ioremap FB base at 0x%08lx\n",
				    info->base1_region);
			goto err_out_iounmap_ctrl;
		}
		break;
	}
	}

#ifdef CONFIG_MTRR
	if (mtrr) {
		info->MTRR_range = mtrr_add (info->base1_region,
					     info->base1_region_size, MTRR_TYPE_WRCOMB, 1);
		if (info->MTRR_range < 0) {
			PRINTK_ERR ("unable to setup MTRR\n");
		} else {
			info->MTRR = 1;
			PRINTK_INFO ("MTRR successfully enabled\n");
		}
	} else {
		PRINTK_INFO ("MTRR disabled\n");
	}
#endif /* CONFIG_MTRR */

	pci_set_drvdata (pd, info);
	PRINTK_INFO ("PCI nVidia NV%X card detected (%s [0x%X], %ldMB @ 0x%08lX)\n",
		     info->chip.arch, info->driver, ent->device,
		     info->chip.fbsize / 1024 / 1024, info->base1_region);

	err = rivatv_register_i2c (info);
	if (err) {
		PRINTK_ERR ("error initializing I2C access\n");
		goto err_out_iounmap_ctrl;
	}

#ifdef CONFIG_PROC_FS
	err = rivatv_register_procfs (info);
	if (err) {
		PRINTK_ERR ("error initializing procfs file\n");
		goto err_out_iounmap_ctrl;
	}
#endif

	err = riva_setup_video_dev (info, &rivatv_driver_init);
	if (err)
		goto err_out_unregister_procfs;

	rivatv_devices++;
	return 0;

err_out_unregister_procfs:
	rivatv_unregister_procfs (info);
err_out_iounmap_ctrl:
	iounmap (info->control_base);
err_out_free_base1:
	release_mem_region (info->base1_region, info->base1_region_size);
err_out_free_base0:
	release_mem_region (info->base0_region, info->base0_region_size);
err_out_kfree:
	kfree (info);
	return -ENODEV;
}


void __exit rivatv_cleanup_pci (struct pci_dev *pd)
{
	struct rivatv_info *info = pci_get_drvdata (pd);

	if (info == NULL)
		return;

	riva_release_video_dev (info);
#ifdef CONFIG_PROC_FS
	rivatv_unregister_procfs (info);
#endif
	rivatv_unregister_i2c (info);

#ifdef CONFIG_MTRR
	if (mtrr) {
		if (info->MTRR)
			mtrr_del (info->MTRR_range, info->base1_region,
				  info->base1_region_size);
	}
#endif /* CONFIG_MTRR */

	iounmap (info->control_base);
	if (info->chip.arch != NV_ARCH_03)
		iounmap ((caddr_t) info->picture_base);
	else
		iounmap (info->video_base);
	release_mem_region (info->base0_region, info->base0_region_size);
	release_mem_region (info->base1_region, info->base1_region_size);
	kfree (info);

	pci_set_drvdata (pd, NULL);
}

struct pci_driver rivatv_driver = {
	name:		"rivatv",
	id_table:	rivatv_pci_table,
	probe:		rivatv_init_pci,
	remove:		rivatv_cleanup_pci,
};

MODULE_DEVICE_TABLE (pci, rivatv_pci_table);
