/*
    vpx32xx.c  MICRONAS INTERMETALL VPX32XX video pixel decoder driver

    (c) 1999-2001 Peter Schlaf <peter.schlaf@org.chemie.uni-giessen.de>
        project started by Jens Seidel <jens.seidel@hrz.tu-chemnitz.de>
	modifications for RivaTV by Stefan Jahn <stefan@lkcc.org>
                   
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/module.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/videodev.h>
#include <linux/video_decoder.h>
#include <linux/i2c.h>
#include <linux/init.h>

#include "rivatv-kcompat.h"

/* Debug option */
static int debug = 1;

#define VPX32XX_PREFIX "VPX32XX: "
#define VPX32XX_PRINTK(format, args...)			        \
	printk (VPX32XX_PREFIX format, ## args)
#define VPX32XX_DEBUG_PRINTK(format, args...) if (debug >= 1)	\
	printk (VPX32XX_PREFIX format, ## args)
#define VPX32XX_DEBUG_PRINTK2(format, args...) if (debug >= 2)  \
	printk (VPX32XX_PREFIX format, ## args)
#undef	PRINTK
#define PRINTK(format, args...)				        \
	printk (format, ## args)
#undef	DPRINTK
#define DPRINTK(format, args...) if (debug >= 1)		\
	printk (format, ## args)
#undef	DPRINTK2
#define DPRINTK2(format, args...) if (debug >= 2)		\
	printk (format, ## args)

#ifndef I2C_DRIVERID_VPX32XX
#define I2C_DRIVERID_VPX32XX	42      /* video decoder+vbi/vtxt */
#endif

#define VPX32XX_MAX_INPUT	 8	/* 3 CVBS + 5 SVHS modes are possible */
#define VPX32XX_MAX_OUTPUT	 0	/* these chips can only decode */

struct vpx32xx {
	int	part_number;
#define VPX3220A 0x4680      /* 16-bit part numbers */
#define VPX3216B 0x4260
#define VPX3214C 0x4280
#define VPX3225D 0x7230
#define VPX3224D 0x7231

	unsigned char	fprd;
	unsigned char	fpwr;
	unsigned char	fpdat;
	unsigned char	fpsta;

	int	norm;
#define VPX_PALBGHI	0x00
#define VPX_PALM	0x04
#define VPX_PALN	0x05
#define VPX_PAL60	0x06
#define VPX_NTSCM	0x01
#define VPX_NTSC44	0x03
#define VPX_NTSCCOMB	0x07
#define VPX_SECAM	0x02
	int	input;

	__u16	hue;
	__u16	saturation;
	__u16	brightness1;
	__u16	brightness2;
	__u16	contrast1;
	__u16	contrast2;
};

#ifndef I2C_VPX32XX
#define I2C_VPX32XX	(0x86 / 2)	/* I2C device address */
#endif

/* Possible VPX32XX addresses: 0x43, 0x47 */
static unsigned short int normal_i2c[] = { I2C_VPX32XX, 0x47, I2C_CLIENT_END };
static unsigned short int normal_i2c_range[] = { I2C_CLIENT_END };

/* Magic definition of all other I2C variables and things */
I2C_CLIENT_INSMOD;

static int vpx32xx_attach  (struct i2c_adapter *);
static int vpx32xx_detach  (struct i2c_client *);
static int vpx32xx_command (struct i2c_client *, unsigned int, void *);

static struct i2c_driver i2c_driver_vpx32xx = {
	.name		= "vpx32xx",
	.id		= I2C_DRIVERID_VPX32XX,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter = vpx32xx_attach,
	.detach_client	= vpx32xx_detach,
	.command	= vpx32xx_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 54) && !defined (I2C_PEC)
	.inc_use	= i2c_client_inc_use,
	.dec_use	= i2c_client_dec_use
#else
	.owner		= THIS_MODULE,
#endif
};

static struct i2c_client vpx32xx_client = {
	I2C_DEVNAME ("Micronas VPX32xx"),
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	.id      = -1,
#endif
	.flags   = I2C_CLIENT_ALLOW_USE,
	.addr    = 0,
	.adapter = NULL,
	.driver  = &i2c_driver_vpx32xx,
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
/* Unique ID allocation */
static int vpx32xx_id = 0;
#endif

/* Write one byte to i2c chip register */
static int vpx32xx_writeI2C (struct i2c_client *client, unsigned char subaddr, unsigned char data)
{
	unsigned char buffer[2] = { subaddr, data };

	if (sizeof (buffer) != i2c_master_send (client, buffer, sizeof (buffer))) {
		VPX32XX_PRINTK ("failed writing register 0x%02x (I/O error)\n", subaddr);
		return -EIO;
	}
	VPX32XX_DEBUG_PRINTK2 ("writeI2C (subaddr=0x%02x, data=0x%02x) OK\n", subaddr, data);
	return 0;
}

/* Read and return one byte from i2c chip register */
static unsigned char vpx32xx_readI2C (struct i2c_client *client, unsigned char subaddr)
{
	unsigned char read;
	struct i2c_msg msg[] = {
		{ client->addr, 0,        1, &subaddr },
		{ client->addr, I2C_M_RD, 1, &read    }
	};
	int size = sizeof (msg) / sizeof (msg[0]); 
	
	if (size != i2c_transfer (client->adapter, msg, size)) {
		VPX32XX_PRINTK ("failed reading register 0x%02x (I/O error)\n", subaddr);
		return -EIO;
	}
	VPX32XX_DEBUG_PRINTK2 ("readI2C (subaddr=0x%02x, read=0x%02x) OK\n", subaddr, read);
	return read;
}


/* Write to vpx FP-RAM register */
static int vpx32xx_writeFP (struct i2c_client *client, int subaddr, int data)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);

	unsigned char load[3];
	unsigned char write = decoder->fpsta;
	unsigned char read;
	struct i2c_msg msg[2] = {
		{ client->addr, 0,        1, &write },
		{ client->addr, I2C_M_RD, 1, &read  }
	};
	int size = sizeof (msg) / sizeof (msg[0]); 
	int i;

	load[0] = decoder->fpwr;
	load[1] = (unsigned char) (subaddr >> 8);
	load[2] = (unsigned char) (subaddr & 0x00FF);

	if (sizeof (load) != i2c_master_send (client, load, sizeof (load))) {
		VPX32XX_PRINTK ("writeFP failed (I/O error) [1]\n");
		return -EIO;
	}
	
	load[0] = decoder->fpdat;
	load[1] = (unsigned char) (data >> 8);
	load[2] = (unsigned char) (data & 0x00FF);

	for (i = 0; i < 10; i++) {	/* FP busy ? */
		if (size != i2c_transfer (client->adapter, msg, size)) {
			VPX32XX_PRINTK ("writeFP failed (I/O error) [2]\n");
			return -EIO;
		}
							
		if (!(read & 0x04)) {
			if (sizeof (load) != i2c_master_send (client, load, sizeof (load))) {
				VPX32XX_PRINTK ("writeFP failed (I/O error) [3]\n");
				return -EIO;
			}
			VPX32XX_DEBUG_PRINTK2 ("writeFP (subaddr=0x%03x, data=0x%03x) OK\n", subaddr, data);
			return 0;
		}
	}
	VPX32XX_PRINTK ("writeFP: FP processor is busy\n");
	return -EBUSY;
}

/* Read from vpx FP-RAM register */
static int vpx32xx_readFP (struct i2c_client *client, int subaddr)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);

	unsigned char load[3];
	unsigned char write = decoder->fpsta;
	unsigned char read[2];
	struct i2c_msg msg[2] = {
		{ client->addr, 0,        1, &write },
		{ client->addr, I2C_M_RD, 1, read   }
	};
	int size = sizeof (msg) / sizeof (msg[0]); 
	int i;
	
	load[0] = decoder->fprd;
	load[1] = (unsigned char) (subaddr >> 8);
	load[2] = (unsigned char) (subaddr & 0x00FF);

	if (sizeof (load) != i2c_master_send (client, load, sizeof (load))) {
		VPX32XX_PRINTK ("readFP failed (I/O error) [1]\n");
		return -EIO;
	}
	
	for (i = 0; i < 10; i++) {	/* FP busy ? */
		msg[1].len = 1;
		write = decoder->fpsta;
		if (size != i2c_transfer (client->adapter, msg, size)) {
			VPX32XX_PRINTK ("readFP failed (I/O error) [2]\n");
			return -EIO;
		}
							
		if (!(read[0] & 0x04)) {
			msg[1].len = 2;
			write = decoder->fpdat;

			if (size != i2c_transfer (client->adapter, msg, size)) {
				VPX32XX_PRINTK ("readFP failed (I/O error) [3]\n");
				return -EIO;
			}
			VPX32XX_DEBUG_PRINTK2 ("readFP (subaddr=0x%03x, read=0x%03x) OK\n",
					       subaddr, ((read[0] << 8) | read[1]));
			return ((read[0] << 8) | read[1]);
		}
	}
	VPX32XX_PRINTK ("readFP: FP processor is busy\n");
	return -EBUSY;
}

/* Set picture properties for VPX3214, VPX3216 or VPX3220 */
static int vpx32xx_set_picture (struct i2c_client *client)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int ret = 0;
	
	/* set brightness */
	ret |= vpx32xx_writeI2C (client, 0xE6, ((decoder->brightness1 - 32768) >> 8) & 0xFF);

	/* set contrast */
	ret |= vpx32xx_writeI2C (client, 0xE7, ((decoder->contrast1 >> 10) & 0x003F));

	/* set saturation */
	ret |= vpx32xx_writeFP (client, 0x0A0, (decoder->saturation >> 4) | 1);

	/* set hue (for NTSC only) */
	ret |= vpx32xx_writeFP (client, 0x01C, ((decoder->hue - 32768) >> 6) & 0x03FF); /* ntsc tint angle: +-512 */

	return ret;
}

/* Set picture properties for VPX3224 or VPX3225 */
static int vpx322x_set_picture (struct i2c_client *client)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int ret = 0;
	
	/* set brightness */
	ret |= vpx32xx_writeFP (client, 0x127, ((decoder->brightness1 - 32768) >> 8) & 0xFF);
	ret |= vpx32xx_writeFP (client, 0x131, ((decoder->brightness2 - 32768) >> 8) & 0xFF);

	/* set contrast */
	ret |= vpx32xx_writeFP (client, 0x128, ((decoder->contrast1 >> 10) | 0x0100)); /* set clamping level=16 */
	ret |= vpx32xx_writeFP (client, 0x132, ((decoder->contrast2 >> 10) | 0x0100)); /* set clamping level=16 */

	/* set saturation */
	ret |= vpx32xx_writeFP (client, 0x030, (decoder->saturation >> 4) | 1);

	/* set hue (for NTSC only) */
	ret |= vpx32xx_writeFP (client, 0x0DC, ((decoder->hue - 32768) >> 6) & 0x03FF); /* ntsc tint angle: +-512 */

	/* latching */
	ret |= vpx32xx_writeFP (client, 0x140, 0x860);	/* set latch Timing mode and WinTable #1 + #2 */
	
	return ret;
}

/* Set video standard for VPX3214, VPX3216 or VPX3220 */
static int vpx32xx_set_video_norm (struct i2c_client *client, int video_norm)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int i, choice, ret = 0;

	const int init0[][2] = {  /* FP init */
	/* addr    value                     */
	{ 0x00A0, 0x0001 },	/*           */
	{ 0x008F, 0x0000 },	/*           */
	{ 0x004B, 0x0298 },	/*           */
	{ 0x001C, 0x01F4 }	/*           */
	}; 

	const int init1[][2] = {  /* I2C init  */
	/* addr    value                       */
	{ 0x00D8, 0x00A6 },	/* old: 0x00A4 */
	{ 0x00E8, 0x00F8 },	/*             */
	{ 0x00EA, 0x0010 },	/*             */
	{ 0x00F0, 0x000B },	/*             */
	{ 0x00F1, 0x008C },	/*             */
	{ 0x00F2, 0x001B },	/*             */
	{ 0x00F8, 0x0002 },	/*             */
	{ 0x00F9, 0x0025 }	/*             */
	}; 

	const int norms[3][11] = {
	/*                  88      89      8a      8b      8c      8d      f0      f2      c1      e7      a0    */
	/* PAL   */	{ 0x0007, 0x0130, 0x0130, 0x0000, 0x0330, 0x0330, 0x0077, 0x0011, 0x0058, 0x026E, 0x0B32 },
	/* NTSC  */	{ 0x000A, 0x00F0, 0x00F0, 0x0000, 0x0290, 0x02B0, 0x0077, 0x0013, 0x0060, 0x020A, 0x0B32 },
	/* SECAM */	{ 0x0007, 0x0130, 0x0130, 0x0000, 0x0330, 0x0330, 0x0077, 0x0015, 0x006C, 0x026E, 0x0219 } };

	switch (video_norm) {
		case VIDEO_MODE_PAL: 	choice = 0; break;	/* PAL B,G,H,I 625/50	*/
		case VIDEO_MODE_NTSC: 	choice = 1; break;	/* NTSC-M 525/60 	*/
		case VIDEO_MODE_SECAM: 	choice = 2; break;	/* SECAM 625/50 	*/
		default:
			VPX32XX_PRINTK ("video standard invalid (norm=%d)\n", video_norm);
			return -EINVAL;
	}

	if (decoder->part_number == VPX3214C)
		ret |= vpx32xx_writeI2C (client, 0xAA, 0x00);	/* normal power mode */

	/*	FP init0 	*/
	for (i = 0; i < (sizeof (init0) / (2 * sizeof (int))); i++)
		ret |= vpx32xx_writeFP (client, init0[i][0], init0[i][1]);

	/*	I2C init1 	*/
	for (i = 0; i < (sizeof (init1) / (2 * sizeof (int))); i++)
		ret |= vpx32xx_writeI2C (client, (unsigned char) (init1[i][0]), (unsigned char) (init1[i][1]));

	ret |= vpx32xx_writeFP (client, 0x088, norms[choice][0]); 
	ret |= vpx32xx_writeFP (client, 0x089, norms[choice][1]);
	ret |= vpx32xx_writeFP (client, 0x08A, norms[choice][2]);
	ret |= vpx32xx_writeFP (client, 0x08B, norms[choice][3]); 
	ret |= vpx32xx_writeFP (client, 0x08C, norms[choice][4]); 
	ret |= vpx32xx_writeFP (client, 0x08D, norms[choice][5]); 

	ret |= vpx32xx_writeFP (client, 0x0F0, norms[choice][6]); 
	ret |= vpx32xx_writeFP (client, 0x0F2, norms[choice][7]); 
	ret |= vpx32xx_writeFP (client, 0x0C1, norms[choice][8]); 
	ret |= vpx32xx_writeFP (client, 0x0E7, norms[choice][9]); 
	ret |= vpx32xx_writeFP (client, 0x0A0, norms[choice][10]); 

	if (0 == ret)
		decoder->norm = video_norm;
	return ret;
}

/* Sets up scalar tasks. */
static int vpx322x_setup_scaler (struct i2c_client *client,
				 int src_x, int src_y, int src_width, int src_height,
				 int dst_width, int dst_height, int scaler, int enable)
{
	int ret = 0, idx = (scaler - 1) * 0x00a;

	/* Start of active video */
	ret |= vpx32xx_writeFP (client, 0x022, 0x0E);
	/* Start position of VACT */
	ret |= vpx32xx_writeFP (client, 0x151, 0);
	/* End position of VACT */
	ret |= vpx32xx_writeFP (client, 0x152, 800);
	/* HREF and VREF control */
	ret |= vpx32xx_writeFP (client, 0x153, 0x000);

	/* Vertical Begin */
	ret |= vpx32xx_writeFP (client, 0x120 + idx, src_y);
	/* Vertical Lines In */
	ret |= vpx32xx_writeFP (client, 0x121 + idx, src_height | (enable ? (0 << 10) : (3 << 10)));
	/* Vertical Lines Out */
	ret |= vpx32xx_writeFP (client, 0x122 + idx, dst_height);
	/* Horizontal Begin */
	ret |= vpx32xx_writeFP (client, 0x123 + idx, src_x);
	/* Horizontal Length */
	ret |= vpx32xx_writeFP (client, 0x124 + idx, dst_width);
	/* Number of Pixels */
	ret |= vpx32xx_writeFP (client, 0x125 + idx, src_x + src_width);
	/* Selection for peaking/coring */
	ret |= vpx32xx_writeFP (client, 0x126 + idx, 0x100);
	/* Brightness */
	ret |= vpx32xx_writeFP (client, 0x127 + idx, 0x000);
	/* Contrast */
	ret |= vpx32xx_writeFP (client, 0x128 + idx, 0x600);
	/* Register for control and latching: Latch Window # */
	ret |= vpx32xx_writeFP (client, 0x140, (scaler << 5));
	return ret;
}

/* Sets up additional PAL/NTSC and SECAM. */
static void vpx322x_setup_standard (struct i2c_client *client, int std)
{
	switch (std) {
	case VIDEO_MODE_PAL:
		vpx322x_setup_scaler (client, 7, 23, 720, 288, 720, 288, 1, 1);
		vpx322x_setup_scaler (client, 7, 23, 720, 288, 720, 288, 2, 0);
		vpx32xx_writeFP (client, 0x030, 2070);
		break;
	case VIDEO_MODE_NTSC:
		vpx322x_setup_scaler (client, 10, 20, 720, 240, 720, 240, 1, 1);
		vpx322x_setup_scaler (client, 10, 20, 720, 240, 720, 240, 2, 0);
		vpx32xx_writeFP (client, 0x030, 2070);
		break;
	case VIDEO_MODE_SECAM:
		vpx322x_setup_scaler (client, 7, 23, 768, 288, 720, 288, 1, 1);
		vpx322x_setup_scaler (client, 7, 23, 768, 288, 720, 288, 2, 0);
		vpx32xx_writeFP (client, 0x030, 0);
		vpx32xx_writeFP (client, 0x032, 1155);
		vpx32xx_writeFP (client, 0x033, 1496);
		break;
	}
}


typedef struct {
	/* colour standard identifier */
	int standard;
	/* textual colour standard description */
	char *name;
} vpx322x_standard_t;

static vpx322x_standard_t vpx322x_standard[2][5] = {
	/* 50 Hz standards */
	{ { VPX_PALBGHI,  "PAL-B"           },
	  { VPX_SECAM,    "SECAM"           },
	  { VPX_PALN,     "PAL-N"           },
	  { -1,           "<Invalid 50Hz>"  },
	  { -1,           "<Invalid 50Hz>"  } },

	/* 60 Hz standards */
	{ { VPX_NTSCM,    "NTSC-M"          },
	  { VPX_NTSC44,   "NTSC44"          },
	  { VPX_PALM,     "PAL-M"           },
	  { VPX_PAL60,    "PAL-60"          },
	  { VPX_NTSCCOMB, "NTSC-COMB"       } }
};

#define VPX322X_50HZ 312
#define VPX322X_60HZ 262

/* Autodetection of colour standard for VPX3224 and VPX3225 */
static void vpx322x_detection (struct i2c_client *client)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int lines, status, old, detected, i = 0, n, idx, data;

	/* save old colour standard */
	old = vpx32xx_readFP (client, 0x020);

	/* detect 50/60Hz */
	lines = vpx32xx_readFP (client, 0x0CB);
	DPRINTK (" %dHz", lines == VPX322X_50HZ ? 50 : 60);
	idx = (lines == VPX322X_50HZ ? 0 : 1);

	if (lines == VPX322X_50HZ) {
		vpx322x_setup_scaler (client, 7, 23, 720, 288, 720, 288, 1, 1);
		vpx322x_setup_scaler (client, 7, 23, 720, 288, 720, 288, 2, 0);
	} else {
		vpx322x_setup_scaler (client, 10, 20, 720, 240, 720, 240, 1, 1);
		vpx322x_setup_scaler (client, 10, 20, 720, 240, 720, 240, 2, 0);
	}

	for (detected = n = 0; n < 5; n++) {
		if (vpx322x_standard[idx][n].standard != -1) {
			/* set colour standard */
			vpx32xx_writeFP (client, 0x020, (old & ~0xC4F) | vpx322x_standard[idx][n].standard);

			/* stay here a little eternity */
			set_current_state (TASK_UNINTERRUPTIBLE);
			schedule_timeout (HZ/10);
			set_current_state (TASK_RUNNING);

			/* check status bits */
			data = vpx32xx_readFP (client, 0x020);
			status = vpx32xx_readFP (client, 0x013);
			DPRINTK (" %03x", status);
			if ((data & 0x800) && ((status & 0x2F) == 3)) {
				DPRINTK (" %s", vpx322x_standard[idx][n].name);
				detected++;
				i = n;
			}
		}
	}
  
	/* restore old colour standard */
	if (!detected) {
		vpx32xx_writeFP (client, 0x020, old);
		DPRINTK (" no video source detected");
	} else {
		vpx32xx_writeFP (client, 0x020, (old & ~0xC4F) | vpx322x_standard[idx][i].standard);
		switch (vpx322x_standard[idx][i].standard) {
		case VPX_PALBGHI:
		case VPX_PALM:
		case VPX_PALN:
		case VPX_PAL60:
			decoder->norm = VIDEO_MODE_PAL;
			break;
		case VPX_NTSCM:
		case VPX_NTSC44:
		case VPX_NTSCCOMB:
			decoder->norm = VIDEO_MODE_NTSC;
			break;
		case VPX_SECAM:
			decoder->norm = VIDEO_MODE_SECAM;
			break;
		}
	}
	vpx322x_setup_standard (client, decoder->norm);
	DPRINTK ("\n");
}

/* Set video_norm for VPX3224 and VPX3225 */
static int vpx322x_set_video_norm (struct i2c_client *client, __u16 video_norm)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int ret = 0, choice;
	const int norms[8][11] = {
	/*                ver.b.  ver.li. ver.lo. hor.b.  hor.l.  nr.px.  sel.p/c st.v.a. e.v.a.  col.pr. */
	/* PAL       */ { 0x0007, 0x0130, 0x0130, 0x0000, 0x0330, 0x0330, 0x0100, 0x0008, 0x0330, 0x07FF, 0x0000 }, 
	/* NTSC      */ { 0x000A, 0x00F0, 0x00F0, 0x0000, 0x0290, 0x02B0, 0x0100, 0x0030, 0x02A8, 0x07FF, 0x0001 },
	/* SECAM     */ { 0x0007, 0x0130, 0x0130, 0x0000, 0x0330, 0x0330, 0x0100, 0x0008, 0x0330, 0x017F, 0x0002 },

	/* PAL 60    */ { 0x000A, 0x00F0, 0x00F0, 0x0000, 0x0290, 0x02B0, 0x0100, 0x0030, 0x02A8, 0x07FF, 0x0006 },
	/* PAL M     */ { 0x000A, 0x00F0, 0x00F0, 0x0000, 0x0290, 0x02B0, 0x0100, 0x0030, 0x02A8, 0x07FF, 0x0004 },
	/* PAL N     */ { 0x0007, 0x0130, 0x0130, 0x0000, 0x0330, 0x0330, 0x0100, 0x0008, 0x0330, 0x07FF, 0x0005 }, 
	/* NTSC44    */ { 0x000A, 0x00F0, 0x00F0, 0x0000, 0x0290, 0x02B0, 0x0100, 0x0030, 0x02A8, 0x07FF, 0x0003 },
	/* NTSC COMP */ { 0x000A, 0x00F0, 0x00F0, 0x0000, 0x0290, 0x02B0, 0x0100, 0x0030, 0x02A8, 0x07FF, 0x0007 } };

	switch (video_norm) {
		case VIDEO_MODE_PAL: 	choice = 0; break;
		case VIDEO_MODE_NTSC: 	choice = 1; break;
		case VIDEO_MODE_SECAM: 	choice = 2; break;
		default:
			VPX32XX_PRINTK ("video standard invalid: %d\n", video_norm);
			return -EINVAL;
	}

	/* Output */
	ret |= vpx32xx_writeI2C (client, 0xAA, 0x00);		/* Low Power Mode, bit 4 on TDO-pin and 0 */
	ret |= vpx32xx_writeI2C (client, 0xF2, 0x0D);		/* Output Enable */

	/* Formatter */
	ret |= vpx32xx_writeFP (client, 0x150, 0x601);		/* Format Selection */

	/* Output Multiplexer */
	ret |= vpx32xx_writeFP (client, 0x154, 0x200);		/* Multiplexer/Multipurpose output */ 

	/* Standard Selection */	
	ret |= vpx32xx_writeFP (client, 0x020, norms[choice][10]); /* tv norm setting */

	/* Load Table for VBI-Window */
	ret |= vpx32xx_writeFP (client, 0x138, 0x0800);            /* Control VBI-Window (vbi window disabled) */ 

	if (0 == ret) {
		decoder->norm = video_norm;
		vpx322x_setup_standard (client, decoder->norm);
	}
	return ret;
}

/* Input select for VPX3214, VPX3216, VPX3218, VPX3224 and VPX3225 */
static int vpx32xx_selmux (struct i2c_client *client, int video_input)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int data, ret = 0;
	const int input_vpx322x[VPX32XX_MAX_INPUT][2] = {
	/*  0x21 , s-vhs                              */
	{  0x0224,   0 },   /* CVBS (VIN3)            */
	{  0x0225,   0 },   /* CVBS (VIN2)            */
	{  0x0226,   0 },   /* CVBS (VIN1)            */

	{  0x0224,   1 },   /* S-VHS (Y-VIN3, C-CIN)  */
	{  0x0225,   1 },   /* S-VHS (Y-VIN2, C-CIN)  */
	{  0x0226,   1 },   /* S-VHS (Y-VIN1, C-CIN)  */
	{  0x0220,   1 },   /* S-VHS (Y-VIN3, C-VIN1) */
	{  0x0221,   1 }};  /* S-VHS (Y-VIN2, C-VIN1) */

	const int input_vpx32xx[VPX32XX_MAX_INPUT][2] = {
	/* 0x33	 , s-vhs                              */
	{  0x0000,   0x11 },  /* CVBS (VIN3)            */
	{  0x0001,   0x11 },  /* CVBS (VIN2)            */
	{  0x0002,   0x11 },  /* CVBS (VIN1)            */

	{  0x000C,   0x31 },   /* S-VHS (Y-VIN3, C-CIN)  */
	{  0x000D,   0x31 },   /* S-VHS (Y-VIN2, C-CIN)  */
	{  0x000E,   0x31 },   /* S-VHS (Y-VIN1, C-CIN)  */
	{  0x0008,   0x31 },   /* S-VHS (Y-VIN3, C-VIN1) */
	{  0x0009,   0x31 }};  /* S-VHS (Y-VIN2, C-VIN1) */

	switch (decoder->part_number) {
		case VPX3224D: case VPX3225D:
			data = vpx32xx_readFP (client, 0x020) & ~(0x0040);	/* read word and erase bit 6 */
			if (0 > data)
				return data; 
			ret |= vpx32xx_writeFP (client, 0x020, (data | (input_vpx322x[video_input][1] << 6)));
			ret |= vpx32xx_writeFP (client, 0x021, input_vpx322x[video_input][0]);
			break;
		case VPX3220A: case VPX3216B: case VPX3214C:
			ret |= vpx32xx_writeI2C (client, 0x33, (unsigned char) (input_vpx32xx[video_input][0]));
			ret |= vpx32xx_writeFP (client, 0x0F2, input_vpx32xx[video_input][1]);
			break;
		default:
			return -EINVAL;
	}
	if (0 == ret)
		decoder->input = video_input;
	return ret;
}

/* Dump VPX3224 or VPX3225 registers */
static int vpx322x_dump (struct i2c_client *client)
{
	int i, data_0, data_1, ret = 0;
	 
	if (debug < 2)
		return ret;

	VPX32XX_PRINTK ("I2C registers\n");
	for (i = 0; i < 0x100; i += 2) {
		ret |= data_0 = vpx32xx_readI2C (client, i);
		ret |= data_1 = vpx32xx_readI2C (client, i + 1);
		VPX32XX_PRINTK ("0x%02x = 0x%02x,  0x%02x = 0x%02x\n", i, data_0, i + 1, data_1);
	}

	VPX32XX_PRINTK ("FP-RAM registers\n");
	for (i = 0; i < 0x180; i += 2) {
		ret |= data_0 = vpx32xx_readFP (client, i);
		ret |= data_1 = vpx32xx_readFP (client, i + 1);
		VPX32XX_PRINTK ("0x%03x = 0x%03x,  0x%03x = 0x%03x\n", i, data_0, i + 1, data_1);
	}
	return ret;
}

#ifdef DECODER_SET_VBI
/* VBI sliced data only possible on VPX3225 */
static int vpx3225_init_vbi_sliced (struct i2c_client *client)
{
	int ret = 0;

	/*	start vbi-initialisation	*/
	/*                              addr   value                                                             */
	ret |= vpx32xx_writeFP (client, 0x121, 0xC00);   /* Vertical Lines (window #1 disable)                   */

	ret |= vpx32xx_writeI2C (client, 0xC7, 0x01);    /* accumulator mode: reset DC, AC, and FLT accu         */
	ret |= vpx32xx_writeI2C (client, 0xC7, 0x30);    /* accumulator mode: dc and ac adaptation enable        */
	ret |= vpx32xx_writeI2C (client, 0xCE, 0x15);    /* bit error tolerance                                  */
	ret |= vpx32xx_writeI2C (client, 0xB8, 0x00);    /* clock run-in and framing code dont care mask high   */
	ret |= vpx32xx_writeI2C (client, 0xB9, 0x00);    /* clock run-in and framing code dont care mask mid    */
	ret |= vpx32xx_writeI2C (client, 0xBA, 0x03);    /* clock run-in and framing code dont care mask low    */ 
	ret |= vpx32xx_writeI2C (client, 0xC8, 0x40);    /* sync slicer                                          */
	ret |= vpx32xx_writeI2C (client, 0xC0, 0x10);    /* soft slicer                                          */ 
	ret |= vpx32xx_writeI2C (client, 0xC6, 0x40);    /* data slicer                                          */ 
	ret |= vpx32xx_writeI2C (client, 0xC5, 0x07);    /* filter coefficient                                   */
	ret |= vpx32xx_writeI2C (client, 0xC9, 0x01);    /* standard ... pal, ttx                                */
	ret |= vpx32xx_writeI2C (client, 0xBB, 0x27);    /* clock run-in and framing code reference high         */
	ret |= vpx32xx_writeI2C (client, 0xBC, 0x55);    /* clock run-in and framing code reference mid          */
	ret |= vpx32xx_writeI2C (client, 0xBD, 0x55);    /* clock run-in and framing code reference low          */
	ret |= vpx32xx_writeI2C (client, 0xC1, 0xBE);    /* ttx bitslicer frequency LSB	... wst pal              */
	ret |= vpx32xx_writeI2C (client, 0xC2, 0x0A);    /* ttx bitslicer frequency MSB                          */
	ret |= vpx32xx_writeI2C (client, 0xCF, 0xAB);    /* output mode	vid.lin data output, 64-byte mode disable */

	ret |= vpx32xx_writeFP (client, 0x134, 0x13F);   /* Start line even field  319  */
	ret |= vpx32xx_writeFP (client, 0x135, 0x14F);   /* End line even field    335  */
	ret |= vpx32xx_writeFP (client, 0x136, 0x006);   /* Start line odd field     6  */
	ret |= vpx32xx_writeFP (client, 0x137, 0x016);   /* End line odd field      22  */

	ret |= vpx32xx_writeFP (client, 0x139, 0x000);   /* Slicer data size 0 (corresp. to 64)     */
	ret |= vpx32xx_writeFP (client, 0x140, 0xC60);   /* Register for Control and Latching       */ 
	ret |= vpx32xx_writeFP (client, 0x150, 0x140);   /* Format selection 	disable splitting   */
	ret |= vpx32xx_writeFP (client, 0x154, 0x000);   /* Multiplexer/Multipurpose output         */ 

	ret |= vpx32xx_writeI2C (client, 0xAA, 0x10);    /* Low power mode, bit 4 connected to TDO-pin  */
	
	ret |= vpx32xx_writeFP (client, 0x138, 0x803);   /* Control VBI-Window, sliced data, vbi enable */ 
	ret |= vpx32xx_writeFP (client, 0x030, 0x666);   /* Color Processing                            */
	return ret; 
}

/* VBI raw data on VPX3225 */
static int vpx3225_init_vbi_raw (struct i2c_client *client)
{
	int ret = 0;

	/* vbi-raw initialisation */

	/* Load Table for Window #1 and #2 */
	ret |= vpx32xx_writeFP (client, 0x121, 0xC00);   /* Vertical Lines In (window #1 disabled) */
	ret |= vpx32xx_writeFP (client, 0x12B, 0xC00);   /* Vertical Lines In (window #2 disabled) */

	/* Load Table for VBI-Window */
	ret |= vpx32xx_writeFP (client, 0x134, 0x13F);   /* Start line even field 319            */
	ret |= vpx32xx_writeFP (client, 0x135, 0x14F);   /* End line even field   335            */
	ret |= vpx32xx_writeFP (client, 0x136, 0x006);   /* Start line odd field    6            */
	ret |= vpx32xx_writeFP (client, 0x137, 0x016);   /* End line odd field     22            */
	ret |= vpx32xx_writeFP (client, 0x139, 0x000);   /* Slicer data size 0 (corresp. to 64)  */

	/* Control Word */
	ret |= vpx32xx_writeFP (client, 0x140, 0xC60);   /* Register for Control and Latching    */ 

	/* Formatter */
	ret |= vpx32xx_writeFP (client, 0x150, 0x004);   /* Format selection                     */

	/* Length/Polarity of HREF, VREF, FIELD   */	
	ret |= vpx32xx_writeFP (client, 0x153, 0x027);

	/* Output Multiplexer */
	ret |= vpx32xx_writeFP (client, 0x154, 0x000);   /* Multiplexer/Multipurpose output      */ 

	/* Output */
	ret |= vpx32xx_writeI2C (client, 0xAA, 0x10);    /* Low power mode, bit 4 connected to TDO-pin */
	ret |= vpx32xx_writeI2C (client, 0xF2, 0x0F);    /* Output enable (A,B,pixclk,href,vref,field,vact,llc,llc2 */

	ret |= vpx32xx_writeFP (client, 0x138, 0x801);   /* Control VBI-Window, raw data, vbi enable */ 

	/* Color Processing */	
	ret |= vpx32xx_writeFP (client, 0x030, 0x666);   /* disable acc */

	return ret; 
}
#endif /* DECODER_SET_VBI */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
static int vpx32xx_detect (struct i2c_adapter *adap, int addr, unsigned short flags, int kind)
#else
static int vpx32xx_detect (struct i2c_adapter *adap, int addr, int kind)
#endif
{
	struct i2c_client *client;
	struct vpx32xx *decoder;
	int ret = 0;

	if (NULL == (client = kmalloc (sizeof (struct i2c_client) + sizeof (struct vpx32xx), GFP_KERNEL))) 
		return -ENOMEM;
	memset (client, 0, sizeof (struct i2c_client) + sizeof (struct vpx32xx));

	decoder = (struct vpx32xx *) &client[1];
	memcpy (client, &vpx32xx_client, sizeof (struct i2c_client));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 12)
	client->id = vpx32xx_id++;
#endif
	client->addr = addr;
	client->adapter = adap;
	i2c_set_clientdata (client, decoder);

	if (0 == (ret = i2c_attach_client (client))) {
		VPX32XX_PRINTK ("vpx32xx attached\n");
	} else {
		VPX32XX_PRINTK ("vpx32xx attach failed\n");
		kfree (client);
		return ret;
	}

	/* chip type detection */
	if (0xEC == vpx32xx_readI2C (client, 0x00)) {
		decoder->part_number = (vpx32xx_readI2C (client, 0x02) << 8) + vpx32xx_readI2C (client, 0x01);

		VPX32XX_PRINTK ("chip detection: MICRONAS INTERMETALL ");

		switch (decoder->part_number) {
			case VPX3225D: PRINTK ("VPX3225D "); break;
			case VPX3224D: PRINTK ("VPX3224D "); break;
			case VPX3220A: PRINTK ("VPX3220A "); break;
			case VPX3216B: PRINTK ("VPX3216B "); break;
			case VPX3214C: PRINTK ("VPX3214C "); break;
			default:
				PRINTK ("[unsupported]\n");
				i2c_detach_client (client);
				kfree(client);
				return -1;
		}
		PRINTK ("detected\n");
	} else {
		VPX32XX_PRINTK ("chip detection: no MICRONAS INTERMETALL chip found\n");
		i2c_detach_client (client);
		kfree (client);
		return -1;
	}

	/* load most common values into the parameter-list */
	decoder->norm = VIDEO_MODE_PAL;
	decoder->brightness1 = decoder->brightness2 = 32768;
	decoder->contrast1   = decoder->contrast2   = 32768;
	decoder->saturation  = 32768;
	decoder->hue         = 32768;
	decoder->input       = 2;

	switch (decoder->part_number) {	/* set fp-ram register addresses */
		case VPX3225D: case VPX3224D: 
			decoder->fpsta = 0x35; decoder->fprd  = 0x36;
			decoder->fpwr  = 0x37; decoder->fpdat = 0x38;
			ret |= vpx322x_set_video_norm (client, decoder->norm);
			ret |= vpx32xx_selmux (client, decoder->input);
			ret |= vpx322x_set_picture (client);
			break;
		case VPX3220A: case VPX3216B: case VPX3214C:
			decoder->fprd  = 0x26; decoder->fpwr  = 0x27;
			decoder->fpdat = 0x28; decoder->fpsta = 0x29;
			ret |= vpx32xx_set_video_norm (client, decoder->norm);
			ret |= vpx32xx_selmux (client, decoder->input);
			ret |= vpx32xx_set_picture (client);
			break;
		default:
			return -EINVAL;
	}

	if(0 != ret) {
		VPX32XX_PRINTK ("vpx32xx attach failed\n");
		i2c_detach_client (client);
		kfree (client);
		return ret;
	}
	return 0;
}

static int vpx32xx_detach (struct i2c_client *client)
{
	int ret = 0;

	/* stop further output */
	vpx32xx_writeI2C (client, 0xF2, 0x00);

	if (0 == (ret = i2c_detach_client (client))) {
		VPX32XX_PRINTK ("vpx32xx detached\n");
	} else {
		VPX32XX_PRINTK ("vpx32xx detach failed\n");
	}
	kfree (client);
	return ret;
}

static int vpx32xx_command (struct i2c_client *client, unsigned int cmd, void *arg)
{
	struct vpx32xx *decoder = i2c_get_clientdata (client);
	int ret = 0;

	switch (cmd) {
	case DECODER_GET_CAPABILITIES: {	/* get decoder capabilities */
		struct video_decoder_capability *dc = arg;
		VPX32XX_DEBUG_PRINTK ("DECODER_GET_CAPABILITIES\n");
		dc->flags = 
			VIDEO_DECODER_PAL   |
			VIDEO_DECODER_NTSC  |
			VIDEO_DECODER_SECAM |
			VIDEO_DECODER_AUTO;
		dc->inputs  = VPX32XX_MAX_INPUT;
		dc->outputs = VPX32XX_MAX_OUTPUT;
		return ret;
	}
				
	case DECODER_GET_STATUS: {	/* get decoder status */
		int res = 0, status = 0, lines = 0, *iarg = arg;

		if (VPX3225D == decoder->part_number || VPX3224D == decoder->part_number) {
			status = vpx32xx_readFP (client, 0x013);
			lines = vpx32xx_readFP (client, 0x0CB);
		}
		VPX32XX_DEBUG_PRINTK ("DECODER_GET_STATUS = 0x%03x ->", status);

		if ((status & 0x07) == 0x03) { /* signal, horizontally and vertically locked */
			DPRINTK (" GOOD");
			res = DECODER_STATUS_GOOD;
		} else {
			DPRINTK (" BAD");
		}

		if (status & 0x80) {
			DPRINTK (" INTERLACED");
		} else {
			DPRINTK (" NON-INTERLACED");
		}

		DPRINTK (" NLPF:%d %dHz", lines, lines == 312 ? 50 : 60);

		switch (decoder->norm) {
		case VIDEO_MODE_PAL:
			DPRINTK (" PAL");
			res |= DECODER_STATUS_PAL;
			break;
		case VIDEO_MODE_NTSC:
			DPRINTK (" NTSC");
			res |= DECODER_STATUS_NTSC;
			break;
		case VIDEO_MODE_SECAM:
			DPRINTK (" SECAM");
			res |= DECODER_STATUS_SECAM;
			break;
		}

		DPRINTK ("\n");
		*iarg = res;
		return ret;
	}
	
	case DECODER_SET_NORM: {	/* set video norm */
		int *iarg = arg;

		VPX32XX_DEBUG_PRINTK ("DECODER_SET_NORM: ");
		switch (*iarg) {
		case VIDEO_MODE_NTSC:
			DPRINTK ("NTSC");
			break;
		case VIDEO_MODE_PAL:
			DPRINTK ("PAL");
			break;
		case VIDEO_MODE_SECAM:
			DPRINTK ("SECAM");
			break;
		case VIDEO_MODE_AUTO:
			DPRINTK ("AUTO");
			vpx322x_detection (client);
			return 0;
		default:
			DPRINTK (" <invalid value %d>\n", *iarg);
			return -EINVAL;
		}
		DPRINTK ("\n");

		switch (decoder->part_number) {
		case VPX3225D: case VPX3224D:
			return vpx322x_set_video_norm (client, *iarg);
		case VPX3220A: case VPX3216B: case VPX3214C:
			return vpx32xx_set_video_norm (client, *iarg);
		default:
			return -EINVAL;
		}
	}
		
	case DECODER_SET_INPUT: {	/* set video channel input */
		int video_input = *(int *) arg;
		VPX32XX_DEBUG_PRINTK ("DECODER_SET_INPUT: video channel input %d\n", video_input);
		if ((VPX32XX_MAX_INPUT <= video_input) || (video_input < 0))
			return -EINVAL;
		return vpx32xx_selmux (client, video_input);
	}

	case DECODER_SET_OUTPUT: {	/* set decoder output - not supported here */
		VPX32XX_DEBUG_PRINTK ("DECODER_SET_OUTPUT: not supported\n");
		return -EOPNOTSUPP;
	}
		
	case DECODER_ENABLE_OUTPUT: {	/* decoder enable output */
		int enable = * (int *) arg;
		VPX32XX_DEBUG_PRINTK ("DECODER_ENABLE_OUTPUT: %d\n", enable);
		if (enable) {
			vpx32xx_writeI2C (client, 0xAA, 0x00);  /* Power mode                      */
			vpx32xx_writeI2C (client, 0xF2, 0x0D);  /* Output Enable Video Port A only */
			vpx32xx_writeI2C (client, 0xF8, 0x24);  /* Driver strength Typ A           */
			vpx32xx_writeI2C (client, 0xF9, 0x04);  /* Driver strength Typ B           */
			vpx32xx_writeFP (client, 0x153, 0x000); /* HREF and VREF control           */
			vpx32xx_writeFP (client, 0x150, 0x601); /* enable ITU-R656                 */
			vpx32xx_writeFP (client, 0x154, 0x200); /* Output Multiplexer              */
			vpx32xx_writeFP (client, 0x140, 0x800); /* Control/Latching                */
		} else {
			vpx32xx_writeI2C (client, 0xF2, 0x00);
		}
		return 0;
	}
		
	case DECODER_SET_PICTURE: {	/* set brightness, contrast, saturation, and hue = 0, ..., 65535 */
		struct video_picture *pic = arg;
		VPX32XX_DEBUG_PRINTK ("DECODER_SET_PICTURE\n");
		decoder->brightness1 = pic->brightness;
		decoder->brightness2 = pic->brightness;
		decoder->contrast1   = pic->contrast;
		decoder->contrast2   = pic->contrast;
		decoder->saturation  = pic->colour;
		decoder->hue         = pic->hue;

		switch (decoder->part_number) {
		case VPX3225D: case VPX3224D:
			return vpx322x_set_picture (client);
		case VPX3220A: case VPX3216B: case VPX3214C:
			return vpx32xx_set_picture (client);
		default:
			return -EINVAL;
		}
	}

#ifdef DECODER_SET_VBI
	case DECODER_SET_VBI: {		/* set parameters for vbi mode */
		VPX32XX_DEBUG_PRINTK ("DECODER_SET_VBI\n");
		if (VPX3225D == decoder->part_number)	/* vbi_sliced only for VPX3225 */
			return vpx3225_init_vbi_sliced (client);
		return vpx3225_init_vbi_raw (client);
		return -EINVAL;
	}
#endif

	case DECODER_DUMP: {
		VPX32XX_DEBUG_PRINTK ("DECODER_DUMP\n");
		if ((VPX3225D == decoder->part_number) || (VPX3224D == decoder->part_number))
			return vpx322x_dump (client);
		return -EINVAL;
	}

	default:
		VPX32XX_DEBUG_PRINTK ("<invalid function 0x%08X>\n", cmd);
		return -ENOIOCTLCMD;
	}
}

static int vpx32xx_attach (struct i2c_adapter *adap)
{
	return i2c_probe (adap, &addr_data, vpx32xx_detect); 
}

/*---------------------------------------------------------------------- 
 *      Modularization
 *----------------------------------------------------------------------*/

MODULE_DESCRIPTION ("MICRONAS INTERMETALL VPX32XX video pixel decoder driver");
MODULE_AUTHOR      ("Peter Schlaf <peter.schlaf@org.chemie.uni-giessen.de> and Stefan Jahn <stefan@lkcc.org>");
MODULE_LICENSE     ("GPL");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
module_param       (debug, int, 0644);
#else
MODULE_PARM	   (debug, "i");
#endif
MODULE_PARM_DESC   (debug, "Debug level: 0 = silent, 1 = verbose, 2 = very verbose (default = 1).");

int __init vpx32xx_init (void)
{
	return i2c_add_driver (&i2c_driver_vpx32xx);
}

void __exit vpx32xx_cleanup (void)
{
	i2c_del_driver (&i2c_driver_vpx32xx);
}

module_init (vpx32xx_init);
module_exit (vpx32xx_cleanup);
