/** \ingroup header
 * \file rpmdb/formats.c
 */

#include "system.h"

#include "rpmio_internal.h"
#include <rpmbc.h>	/* XXX beecrypt base64 */
#include <rpmcb.h>	/* XXX rpmIsVerbose */
#include <rpmmacro.h>	/* XXX for %_i18ndomains */

#define	_RPMTAG_INTERNAL
#include <rpmtag.h>

#include <rpmlib.h>	/* XXX rpmfi, rpmMkdirPath */
#include <rpmfi.h>	/* XXX RPMFILE_FOO */

#define _RPMEVR_INTERNAL
#include <rpmevr.h>	/* XXX RPMSENSE_FOO */

#include "legacy.h"
#include "argv.h"
#include "misc.h"

#include "debug.h"

/*@unchecked@*/
extern int _hdr_debug;

/*@access pgpDig @*/
/*@access pgpDigParams @*/
/*@access headerSprintfExtension @*/
/*@access headerTagTableEntry @*/
/*@access Header @*/	/* XXX debugging msgs */

/**
 * Convert tag data representation.
 * @param he		tag container
 * @param fmt		output radix (NULL or "" assumes %d)
 * @return		formatted string
 */
static char * intFormat(HE_t he, const char *fmt)
	/*@*/
{
    uint32_t ix = (he->ix > 0 ? he->ix : 0);
    uint64_t ival = 0;
    const char * istr = NULL;
    char * b;
    size_t nb = 0;
    int xx;

    if (fmt == NULL || *fmt == '\0')
	fmt = "d";

    switch (he->t) {
    default:
	return xstrdup(_("(not a number)"));
	/*@notreached@*/ break;
    case RPM_UINT8_TYPE:
	ival = (uint64_t) he->p.ui8p[ix];
	break;
    case RPM_UINT16_TYPE:
	ival = (uint64_t) he->p.ui16p[ix];
	break;
    case RPM_UINT32_TYPE:
	ival = (uint64_t) he->p.ui32p[ix];
	break;
    case RPM_UINT64_TYPE:
	ival = he->p.ui64p[ix];
	break;
    case RPM_STRING_TYPE:
	istr = he->p.str;
	break;
    case RPM_STRING_ARRAY_TYPE:
	istr = he->p.argv[ix];
	break;
    case RPM_BIN_TYPE:
	{   static char hex[] = "0123456789abcdef";
	    const char * s = he->p.str;
	    rpmTagCount c = he->c;
	    char * t;

	    nb = 2 * c + 1;
	    t = b = alloca(nb+1);
	    while (c-- > 0) {
		unsigned i;
		i = (unsigned) *s++;
		*t++ = hex[ (i >> 4) & 0xf ];
		*t++ = hex[ (i     ) & 0xf ];
	    }
	    *t = '\0';
	}   break;
    }

    if (istr) {		/* string */
	b = (char *)istr;	/* NOCAST */
    } else
    if (nb == 0) {	/* number */
	char myfmt[] = "%llX";
	myfmt[3] = ((fmt != NULL && *fmt != '\0') ? *fmt : 'd');
	nb = 64;
	b = alloca(nb);
/*@-formatconst@*/
	xx = snprintf(b, nb, myfmt, ival);
/*@=formatconst@*/
	b[nb-1] = '\0';
    }

    return xstrdup(b);
}

/**
 * Return octal formatted data.
 * @param he		tag container
 * @return		formatted string
 */
static char * octFormat(HE_t he)
	/*@*/
{
    return intFormat(he, "o");
}

/**
 * Return hex formatted data.
 * @param he		tag container
 * @return		formatted string
 */
static char * hexFormat(HE_t he)
	/*@*/
{
    return intFormat(he, "x");
}

/**
 * Return decimal formatted data.
 * @param he		tag container
 * @return		formatted string
 */
static char * decFormat(HE_t he)
	/*@*/
{
    return intFormat(he, "d");
}

/**
 * Return strftime formatted data.
 * @param he		tag container
 * @param strftimeFormat strftime(3) format
 * @return		formatted string
 */
static char * realDateFormat(HE_t he, const char * strftimeFormat)
	/*@*/
{
    char * val;

    if (he->t != RPM_UINT64_TYPE) {
	val = xstrdup(_("(not a number)"));
    } else {
	struct tm * tstruct;
	char buf[50];

	/* this is important if sizeof(uint64_t) ! sizeof(time_t) */
	{   time_t dateint = he->p.ui64p[0];
	    tstruct = localtime(&dateint);
	}
	buf[0] = '\0';
	if (tstruct)
	    (void) strftime(buf, sizeof(buf) - 1, strftimeFormat, tstruct);
	buf[sizeof(buf) - 1] = '\0';
	val = xstrdup(buf);
    }

    return val;
}

/**
 * Return date formatted data.
 * @param he		tag container
 * @return		formatted string
 */
static char * dateFormat(HE_t he)
	/*@*/
{
    return realDateFormat(he, _("%c"));
}

/**
 * Return day formatted data.
 * @param he		tag container
 * @return		formatted string
 */
static char * dayFormat(HE_t he)
	/*@*/
{
    return realDateFormat(he, _("%a %b %d %Y"));
}

/**
 * Return shell escape formatted data.
 * @param he		tag container
 * @return		formatted string
 */
static char * shescapeFormat(HE_t he)
	/*@*/
{
    char * val;
    size_t nb;
    int xx;

    /* XXX one of these integer types is unnecessary. */
    if (he->t == RPM_UINT32_TYPE) {
	nb = 20;
	val = xmalloc(nb);
	xx = snprintf(val, nb, "%u", (unsigned) he->p.ui32p[0]);
	val[nb-1] = '\0';
    } else if (he->t == RPM_UINT64_TYPE) {
	nb = 40;
	val = xmalloc(40);
/*@-duplicatequals@*/
	xx = snprintf(val, nb, "%llu", (unsigned long long)he->p.ui64p[0]);
/*@=duplicatequals@*/
	val[nb-1] = '\0';
    } else if (he->t == RPM_STRING_TYPE) {
	const char * s = he->p.str;
	char * t;
	int c;

	nb = 0;
	for (s = he->p.str; (c = (int)*s) != 0; s++)  {
	    nb++;
	    if (c == (int)'\'')
		nb += 3;
	}
	nb += 3;
	t = val = xmalloc(nb);
	*t++ = '\'';
	for (s = he->p.str; (c = (int)*s) != 0; s++)  {
	    if (c == (int)'\'') {
		*t++ = '\'';
		*t++ = '\\';
		*t++ = '\'';
	    }
	    *t++ = (char) c;
	}
	*t++ = '\'';
	*t = '\0';
    } else
	val = xstrdup(_("invalid type"));

    return val;
}

/*@-type@*/ /* FIX: cast? */
static struct headerSprintfExtension_s _headerDefaultFormats[] = {
    { HEADER_EXT_FORMAT, "octal",
	{ .fmtFunction = octFormat } },
    { HEADER_EXT_FORMAT, "oct",
	{ .fmtFunction = octFormat } },
    { HEADER_EXT_FORMAT, "hex",
	{ .fmtFunction = hexFormat } },
    { HEADER_EXT_FORMAT, "decimal",
	{ .fmtFunction = decFormat } },
    { HEADER_EXT_FORMAT, "dec",
	{ .fmtFunction = decFormat } },
    { HEADER_EXT_FORMAT, "date",
	{ .fmtFunction = dateFormat } },
    { HEADER_EXT_FORMAT, "day",
	{ .fmtFunction = dayFormat } },
    { HEADER_EXT_FORMAT, "shescape",
	{ .fmtFunction = shescapeFormat } },
    { HEADER_EXT_LAST, NULL, { NULL } }
};
/*@=type@*/

headerSprintfExtension headerDefaultFormats = &_headerDefaultFormats[0];
/* XXX FIXME: static for now, refactor from manifest.c later. */
static char * rpmPermsString(int mode)
	/*@*/
{
    char *perms = xstrdup("----------");
   
    if (S_ISREG(mode)) 
	perms[0] = '-';
    else if (S_ISDIR(mode)) 
	perms[0] = 'd';
    else if (S_ISLNK(mode))
	perms[0] = 'l';
    else if (S_ISFIFO(mode)) 
	perms[0] = 'p';
    /*@-unrecog@*/
    else if (S_ISSOCK(mode)) 
	perms[0] = 's';
    /*@=unrecog@*/
    else if (S_ISCHR(mode))
	perms[0] = 'c';
    else if (S_ISBLK(mode))
	perms[0] = 'b';
    else
	perms[0] = '?';

    if (mode & S_IRUSR) perms[1] = 'r';
    if (mode & S_IWUSR) perms[2] = 'w';
    if (mode & S_IXUSR) perms[3] = 'x';
 
    if (mode & S_IRGRP) perms[4] = 'r';
    if (mode & S_IWGRP) perms[5] = 'w';
    if (mode & S_IXGRP) perms[6] = 'x';

    if (mode & S_IROTH) perms[7] = 'r';
    if (mode & S_IWOTH) perms[8] = 'w';
    if (mode & S_IXOTH) perms[9] = 'x';

    if (mode & S_ISUID)
	perms[3] = ((mode & S_IXUSR) ? 's' : 'S'); 

    if (mode & S_ISGID)
	perms[6] = ((mode & S_IXGRP) ? 's' : 'S'); 

    if (mode & S_ISVTX)
	perms[9] = ((mode & S_IXOTH) ? 't' : 'T');

    return perms;
}

/**
 * Identify type of trigger.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * triggertypeFormat(HE_t he)
{
    int ix = (he->ix > 0 ? he->ix : 0);
    char * val;

assert(ix == 0);
    if (he->t != RPM_UINT64_TYPE)
	val = xstrdup(_("(invalid type)"));
    else {
	uint64_t anint = he->p.ui64p[ix];
	if (anint & RPMSENSE_TRIGGERPREIN)
	    val = xstrdup("prein");
	else if (anint & RPMSENSE_TRIGGERIN)
	    val = xstrdup("in");
	else if (anint & RPMSENSE_TRIGGERUN)
	    val = xstrdup("un");
	else if (anint & RPMSENSE_TRIGGERPOSTUN)
	    val = xstrdup("postun");
	else
	    val = xstrdup("");
    }
    return val;
}

/**
 * Format file permissions for display.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * permsFormat(HE_t he)
{
    int ix = (he->ix > 0 ? he->ix : 0);
    char * val;

assert(ix == 0);
    if (he->t != RPM_UINT64_TYPE) {
	val = xstrdup(_("(invalid type)"));
    } else {
	uint64_t anint = he->p.ui64p[0];
	val = rpmPermsString((int)anint);
    }

    return val;
}

/**
 * Format file flags for display.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * fflagsFormat(HE_t he)
{
    int ix = (he->ix >= 0 ? he->ix : 0);
    char * val;

assert(ix == 0);
    if (he->t != RPM_UINT64_TYPE) {
	val = xstrdup(_("(invalid type)"));
    } else {
	char buf[15];
	uint64_t anint = he->p.ui64p[ix];
	buf[0] = '\0';
	if (anint & RPMFILE_DOC)
	    strcat(buf, "d");
	if (anint & RPMFILE_CONFIG)
	    strcat(buf, "c");
	if (anint & RPMFILE_SPECFILE)
	    strcat(buf, "s");
	if (anint & RPMFILE_MISSINGOK)
	    strcat(buf, "m");
	if (anint & RPMFILE_NOREPLACE)
	    strcat(buf, "n");
	if (anint & RPMFILE_GHOST)
	    strcat(buf, "g");
	if (anint & RPMFILE_LICENSE)
	    strcat(buf, "l");
	if (anint & RPMFILE_README)
	    strcat(buf, "r");
	val = xstrdup(buf);
    }

    return val;
}

/**
 * Wrap a pubkey in ascii armor for display.
 * @todo Permit selectable display formats (i.e. binary).
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * armorFormat(HE_t he)
	/*@*/
{
    int ix = (he->ix > 0 ? he->ix : 0);
    const char * enc;
    const unsigned char * s;
    size_t ns;
    int atype;
    char * val;

assert(ix == 0);
    switch (he->t) {
    case RPM_BIN_TYPE:
	s = (unsigned char *) he->p.ui8p;
	ns = he->c;
	atype = PGPARMOR_SIGNATURE;	/* XXX check pkt for signature */
	break;
    case RPM_STRING_TYPE:
    case RPM_STRING_ARRAY_TYPE:
	enc = he->p.str;
	s = NULL;
	ns = 0;
/*@-moduncon@*/
	if (b64decode(enc, (void **)&s, &ns))
	    return xstrdup(_("(not base64)"));
/*@=moduncon@*/
	atype = PGPARMOR_PUBKEY;	/* XXX check pkt for pubkey */
	break;
    case RPM_UINT8_TYPE:
    case RPM_UINT16_TYPE:
    case RPM_UINT32_TYPE:
    case RPM_UINT64_TYPE:
    case RPM_I18NSTRING_TYPE:
    default:
	return xstrdup(_("(invalid type)"));
	/*@notreached@*/ break;
    }

    val = pgpArmorWrap(atype, s, ns);
    if (atype == PGPARMOR_PUBKEY)
	s = _free(s);
    return val;
}

/**
 * Encode binary data in base64 for display.
 * @todo Permit selectable display formats (i.e. binary).
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * base64Format(HE_t he)
	/*@*/
{
    int ix = (he->ix > 0 ? he->ix : 0);
    char * val;

assert(ix == 0);
    if (!(he->t == RPM_BIN_TYPE)) {
	val = xstrdup(_("(not a blob)"));
    } else {
	const char * enc;
	char * t;
	int lc;
	size_t ns = he->c;
	size_t nt = ((ns + 2) / 3) * 4;

	/*@-globs@*/
	/* Add additional bytes necessary for eol string(s). */
	if (b64encode_chars_per_line > 0 && b64encode_eolstr != NULL) {
	    lc = (nt + b64encode_chars_per_line - 1) / b64encode_chars_per_line;
	if (((nt + b64encode_chars_per_line - 1) % b64encode_chars_per_line) != 0)
	    ++lc;
	    nt += lc * strlen(b64encode_eolstr);
	}
	/*@=globs@*/

	val = t = xcalloc(1, nt + 1);
	*t = '\0';

    /* XXX b64encode accesses uninitialized memory. */
    { 	unsigned char * _data = xcalloc(1, ns+1);
	memcpy(_data, he->p.ptr, ns);
/*@-moduncon@*/
	if ((enc = b64encode(_data, ns)) != NULL) {
	    t = stpcpy(t, enc);
	    enc = _free(enc);
	}
/*@=moduncon@*/
	_data = _free(_data);
    }
    }

/*@-globstate@*/
    return val;
/*@=globstate@*/
}

/**
 * Return length of string represented with xml characters substituted.
 * @param s		string
 * @return		length of xml string
 */
static size_t xmlstrlen(const char * s)
	/*@*/
{
    size_t len = 0;
    int c;

    while ((c = (int) *s++) != (int) '\0')
    {
	switch (c) {
	case '<':
	case '>':	len += sizeof("&lt;") - 1;	/*@switchbreak@*/ break;
	case '&':	len += sizeof("&amp;") - 1;	/*@switchbreak@*/ break;
	default:	len += 1;			/*@switchbreak@*/ break;
	}
    }
    return len;
}

/**
 * Copy source string to target, substituting for  xml characters.
 * @param t		target xml string
 * @param s		source string
 * @return		target xml string
 */
static char * xmlstrcpy(/*@returned@*/ char * t, const char * s)
	/*@modifies t @*/
{
    char * te = t;
    int c;

    while ((c = (int) *s++) != (int) '\0') {
	switch (c) {
	case '<':	te = stpcpy(te, "&lt;");	/*@switchbreak@*/ break;
	case '>':	te = stpcpy(te, "&gt;");	/*@switchbreak@*/ break;
	case '&':	te = stpcpy(te, "&amp;");	/*@switchbreak@*/ break;
	default:	*te++ = (char) c;		/*@switchbreak@*/ break;
	}
    }
    *te = '\0';
    return t;
}

/**
 * Wrap tag data in simple header xml markup.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * xmlFormat(HE_t he)
	/*@*/
{
    int ix = (he->ix > 0 ? he->ix : 0);
    const char * xtag = NULL;
    size_t nb;
    char * val;
    const char * s = NULL;
    char * t, * te;
    uint64_t anint = 0;
    int freeit = 0;
    int xx;

assert(ix == 0);
assert(he->t == RPM_STRING_TYPE || he->t == RPM_UINT64_TYPE || he->t == RPM_BIN_TYPE);
    switch (he->t) {
    case RPM_STRING_ARRAY_TYPE:	/* XXX currently never happens */
	s = he->p.argv[ix];
	xtag = "string";
	/* XXX Force utf8 strings. */
	s = xstrdup(s);
	s = xstrtolocale(s);
	freeit = 1;
	break;
    case RPM_I18NSTRING_TYPE:	/* XXX currently never happens */
    case RPM_STRING_TYPE:
	s = he->p.str;
	xtag = "string";
	/* XXX Force utf8 strings. */
	s = xstrdup(s);
	s = xstrtolocale(s);
	freeit = 1;
	break;
    case RPM_BIN_TYPE:
/*@-globs -mods@*/
    {	int cpl = b64encode_chars_per_line;
	b64encode_chars_per_line = 0;
/*@-formatconst@*/
	s = base64Format(he);
/*@=formatconst@*/
	b64encode_chars_per_line = cpl;
	xtag = "base64";
	freeit = 1;
    }	break;
/*@=globs =mods@*/
    case RPM_UINT8_TYPE:
	anint = he->p.ui8p[ix];
	break;
    case RPM_UINT16_TYPE:
	anint = he->p.ui16p[ix];
	break;
    case RPM_UINT32_TYPE:
	anint = he->p.ui32p[ix];
	break;
    case RPM_UINT64_TYPE:
	anint = he->p.ui64p[ix];
	break;
    default:
	return xstrdup(_("(invalid xml type)"));
	/*@notreached@*/ break;
    }

    if (s == NULL) {
	int tlen = 64;
	t = memset(alloca(tlen+1), 0, tlen+1);
/*@-duplicatequals@*/
	if (anint != 0)
	    xx = snprintf(t, tlen, "%llu", (unsigned long long)anint);
/*@=duplicatequals@*/
	s = t;
	xtag = "integer";
    }

    nb = xmlstrlen(s);
    if (nb == 0) {
	nb += strlen(xtag) + sizeof("\t</>");
	te = t = alloca(nb);
	te = stpcpy( stpcpy( stpcpy(te, "\t<"), xtag), "/>");
    } else {
	nb += 2 * strlen(xtag) + sizeof("\t<></>");
	te = t = alloca(nb);
	te = stpcpy( stpcpy( stpcpy(te, "\t<"), xtag), ">");
	te = xmlstrcpy(te, s);
	te += strlen(te);
	te = stpcpy( stpcpy( stpcpy(te, "</"), xtag), ">");
    }

    if (freeit)
	s = _free(s);

    val = xstrdup(t);

    return val;
}

/**
 * Return length of string represented with yaml indentation.
 * @param s		string
 * @param lvl		indentation level
 * @return		length of yaml string
 */
static size_t yamlstrlen(const char * s, int lvl)
	/*@*/
{
    size_t len = 0;
    int indent = (lvl > 0);
    int c;

    while ((c = (int) *s++) != (int) '\0')
    {
	if (indent) {
	    len += 2 * lvl;
	    indent = 0;
	}
	if (c == (int) '\n')
	    indent = (lvl > 0);
	len++;
    }
    return len;
}

/**
 * Copy source string to target, indenting for yaml.
 * @param t		target yaml string
 * @param s		source string
 * @param lvl		indentation level
 * @return		target yaml string
 */
static char * yamlstrcpy(/*@out@*/ /*@returned@*/ char * t, const char * s, int lvl)
	/*@modifies t @*/
{
    char * te = t;
    int indent = (lvl > 0);
    int c;

    while ((c = (int) *s++) != (int) '\0') {
	if (indent) {
	    int i;
	    for (i = 0; i < lvl; i++) {
		*te++ = ' ';
		*te++ = ' ';
	    }
	    indent = 0;
	}
	if (c == (int) '\n')
	    indent = (lvl > 0);
	*te++ = (char) c;
    }
    *te = '\0';
    return t;
}

/**
 * Wrap tag data in simple header yaml markup.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * yamlFormat(HE_t he)
	/*@*/
{
    int element = he->ix;
    int ix = (he->ix > 0 ? he->ix : 0);
    const char * xtag = NULL;
    size_t nb;
    char * val;
    const char * s = NULL;
    char * t, * te;
    uint64_t anint = 0;
    int freeit = 0;
    int lvl = 0;
    int xx;
    int c;

assert(ix == 0);
assert(he->t == RPM_STRING_TYPE || he->t == RPM_UINT64_TYPE || he->t == RPM_BIN_TYPE);
    xx = 0;
    switch (he->t) {
    case RPM_STRING_ARRAY_TYPE:	/* XXX currently never happens */
    case RPM_I18NSTRING_TYPE:	/* XXX currently never happens */
    case RPM_STRING_TYPE:
	s = (he->t == RPM_STRING_ARRAY_TYPE ? he->p.argv[ix] : he->p.str);
	if (strchr("[", s[0]))	/* leading [ */
	    xx = 1;
	if (xx == 0)
	while ((c = (int) *s++) != (int) '\0') {
	    switch (c) {
	    default:
		continue;
	    case '\n':	/* multiline */
		xx = 1;
		/*@switchbreak@*/ break;
	    case '-':	/* leading "- \"" */
	    case ':':	/* embedded ": " or ":" at EOL */
		if (s[0] != ' ' && s[0] != '\0' && s[1] != '"')
		    continue;
		xx = 1;
		/*@switchbreak@*/ break;
	    }
	    /*@loopbreak@*/ break;
	}
	if (xx) {
	    if (element >= 0) {
		xtag = "- |-\n";
		lvl = 3;
	    } else {
		xtag = "|-\n";
		lvl = 2;
		if (he->ix < 0) lvl++;	/* XXX extra indent for array[1] */
	    }
	} else {
	    xtag = (element >= 0 ? "- " : NULL);
	}

	/* XXX Force utf8 strings. */
	s = xstrdup(he->p.str);
	s = xstrtolocale(s);
	freeit = 1;
	break;
    case RPM_BIN_TYPE:
/*@-globs -mods@*/
    {	int cpl = b64encode_chars_per_line;
	b64encode_chars_per_line = 0;
/*@-formatconst@*/
	s = base64Format(he);
	element = -element; 	/* XXX skip "    " indent. */
/*@=formatconst@*/
	b64encode_chars_per_line = cpl;
	xtag = "!!binary ";
	freeit = 1;
    }	break;
/*@=globs =mods@*/
    case RPM_UINT8_TYPE:
	anint = he->p.ui8p[ix];
	break;
    case RPM_UINT16_TYPE:
	anint = he->p.ui16p[ix];
	break;
    case RPM_UINT32_TYPE:
	anint = he->p.ui32p[ix];
	break;
    case RPM_UINT64_TYPE:
	anint = he->p.ui64p[ix];
	break;
    default:
	return xstrdup(_("(invalid yaml type)"));
	/*@notreached@*/ break;
    }

    if (s == NULL) {
	int tlen = 64;
	t = memset(alloca(tlen+1), 0, tlen+1);
/*@-duplicatequals@*/
	xx = snprintf(t, tlen, "%llu", (unsigned long long)anint);
/*@=duplicatequals@*/
	s = t;
	xtag = (element >= 0 ? "- " : NULL);
    }

    nb = yamlstrlen(s, lvl);
    if (nb == 0) {
	if (element >= 0)
	    nb += sizeof("    ") - 1;
	nb += sizeof("- ~") - 1;
	nb++;
	te = t = alloca(nb);
	if (element >= 0)
	    te = stpcpy(te, "    ");
	te = stpcpy(te, "- ~");
    } else {
	if (element >= 0)
	    nb += sizeof("    ") - 1;
	if (xtag)
	    nb += strlen(xtag);
	nb++;
	te = t = alloca(nb);
	if (element >= 0)
	    te = stpcpy(te, "    ");
	if (xtag)
	    te = stpcpy(te, xtag);
	te = yamlstrcpy(te, s, lvl);
	te += strlen(te);
    }

    /* XXX s was malloc'd */
    if (freeit)
	s = _free(s);

    val = xstrdup(t);

    return val;
}

/**
 * Display signature fingerprint and time.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * pgpsigFormat(HE_t he)
	/*@globals fileSystem, internalState @*/
	/*@modifies fileSystem, internalState @*/
{
    int ix = (he->ix > 0 ? he->ix : 0);
    char * val, * t;

assert(ix == 0);
    if (!(he->t == RPM_BIN_TYPE)) {
	val = xstrdup(_("(not a blob)"));
    } else {
	unsigned char * pkt = (unsigned char *) he->p.ui8p;
	unsigned int pktlen = 0;
	unsigned int v = (unsigned int) *pkt;
	pgpTag tag = 0;
	unsigned int plen;
	unsigned int hlen = 0;

	if (v & 0x80) {
	    if (v & 0x40) {
		tag = (v & 0x3f);
		plen = pgpLen((byte *)pkt+1, &hlen);
	    } else {
		tag = (v >> 2) & 0xf;
		plen = (1 << (v & 0x3));
		hlen = pgpGrab((byte *)pkt+1, plen);
	    }
	
	    pktlen = 1 + plen + hlen;
	}

	if (pktlen == 0 || tag != PGPTAG_SIGNATURE) {
	    val = xstrdup(_("(not an OpenPGP signature)"));
	} else {
	    pgpDig dig = pgpDigNew(0);
	    pgpDigParams sigp = pgpGetSignature(dig);
	    size_t nb = 0;
	    const char *tempstr;

	    (void) pgpPrtPkts((byte *)pkt, pktlen, dig, 0);

	    val = NULL;
	again:
	    nb += 100;
	    val = t = xrealloc(val, nb + 1);

	    switch (sigp->pubkey_algo) {
	    case PGPPUBKEYALGO_DSA:
		t = stpcpy(t, "DSA");
		break;
	    case PGPPUBKEYALGO_RSA:
		t = stpcpy(t, "RSA");
		break;
	    default:
		(void) snprintf(t, nb - (t - val), "%u", (unsigned)sigp->pubkey_algo);
		t += strlen(t);
		break;
	    }
	    if (t + 5 >= val + nb)
		goto again;
	    *t++ = '/';
	    switch (sigp->hash_algo) {
	    case PGPHASHALGO_MD5:
		t = stpcpy(t, "MD5");
		break;
	    case PGPHASHALGO_SHA1:
		t = stpcpy(t, "SHA1");
		break;
	    default:
		(void) snprintf(t, nb - (t - val), "%u", (unsigned)sigp->hash_algo);
		t += strlen(t);
		break;
	    }
	    if (t + strlen (", ") + 1 >= val + nb)
		goto again;

	    t = stpcpy(t, ", ");

	    /* this is important if sizeof(uint32_t) ! sizeof(time_t) */
	    {	time_t dateint = pgpGrab(sigp->time, sizeof(sigp->time));
		struct tm * tstruct = localtime(&dateint);
		if (tstruct)
 		    (void) strftime(t, (nb - (t - val)), "%c", tstruct);
	    }
	    t += strlen(t);
	    if (t + strlen (", Key ID ") + 1 >= val + nb)
		goto again;
	    t = stpcpy(t, ", Key ID ");
	    tempstr = pgpHexStr(sigp->signid, sizeof(sigp->signid));
	    if (t + strlen (tempstr) > val + nb)
		goto again;
	    t = stpcpy(t, tempstr);

	    dig = pgpDigFree(dig);
	}
    }

    return val;
}

/**
 * Format dependency flags for display.
 * @param he		tag container
 * @return		formatted string
 */
static /*@only@*/ char * depflagsFormat(HE_t he)
	/*@*/
{
    int ix = (he->ix > 0 ? he->ix : 0);
    char * val;

assert(ix == 0);
    if (he->t != RPM_UINT64_TYPE) {
	val = xstrdup(_("(invalid type)"));
    } else {
	uint64_t anint = he->p.ui64p[ix];
	char *t, *buf;

	t = buf = alloca(32);
	*t = '\0';

#ifdef	NOTYET	/* XXX appending markers breaks :depflags format. */
	if (anint & RPMSENSE_SCRIPT_PRE)
	    t = stpcpy(t, "(pre)");
	else if (anint & RPMSENSE_SCRIPT_POST)
	    t = stpcpy(t, "(post)");
	else if (anint & RPMSENSE_SCRIPT_PREUN)
	    t = stpcpy(t, "(preun)");
	else if (anint & RPMSENSE_SCRIPT_POSTUN)
	    t = stpcpy(t, "(postun)");
#endif
	if (anint & RPMSENSE_SENSEMASK)
	    *t++ = ' ';
	if (anint & RPMSENSE_LESS)
	    *t++ = '<';
	if (anint & RPMSENSE_GREATER)
	    *t++ = '>';
	if (anint & RPMSENSE_EQUAL)
	    *t++ = '=';
	if (anint & RPMSENSE_SENSEMASK)
	    *t++ = ' ';
	*t = '\0';

	val = xstrdup(buf);
    }

    return val;
}

/**
 * Retrieve install prefixes.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int instprefixTag(Header h, HE_t he)
	/*@modifies he @*/
{
    he->tag = RPMTAG_INSTALLPREFIX;
    if (headerGet(h, he, 0))
	return 0;

    he->tag = RPMTAG_INSTPREFIXES;
    if (headerGet(h, he, 0)) {
	rpmTagData array = { .argv = he->p.argv };
	he->t = RPM_STRING_TYPE;
	he->c = 1;
	he->p.str = xstrdup(array.argv[0]);
	he->freeData = 1;
	array.ptr = _free(array.ptr);
	return 0;
    }
    return 1;
}

/**
 * Retrieve trigger info.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int triggercondsTag(Header h, HE_t he)
	/*@modifies he @*/
{
    HE_t _he = memset(alloca(sizeof(*_he)), 0, sizeof(*_he));
    rpmTagData flags = { .ptr = NULL };
    rpmTagData indices = { .ptr = NULL };
    rpmTagData names = { .ptr = NULL };
    rpmTagData versions = { .ptr = NULL };
    rpmTagData s;
    rpmTagCount numNames;
    rpmTagCount numScripts;
    unsigned i, j;
    int rc = 1;		/* assume failure */
    int xx;

    he->freeData = 0;

/*@-compmempass@*/
    _he->tag = RPMTAG_TRIGGERNAME;
    xx = headerGet(h, _he, 0);
    names.argv = _he->p.argv;
    numNames = _he->c;
    if (!xx) {		/* no triggers, succeed anyways */
	rc = 0;
	goto exit;
    }

    _he->tag = RPMTAG_TRIGGERINDEX;
    xx = headerGet(h, _he, 0);
    indices.ui32p = _he->p.ui32p;
    if (!xx) goto exit;

    _he->tag = RPMTAG_TRIGGERFLAGS;
    xx = headerGet(h, _he, 0);
    flags.ui32p = _he->p.ui32p;
    if (!xx) goto exit;

    _he->tag = RPMTAG_TRIGGERVERSION;
    xx = headerGet(h, _he, 0);
    versions.argv = _he->p.argv;
    if (!xx) goto exit;

    _he->tag = RPMTAG_TRIGGERSCRIPTS;
    xx = headerGet(h, _he, 0);
    s.argv = _he->p.argv;
    numScripts = _he->c;
    if (!xx) goto exit;
/*@=compmempass@*/

    _he->tag = he->tag;
    _he->t = RPM_UINT32_TYPE;
    _he->p.ui32p = NULL;
    _he->c = 1;
    _he->freeData = 0;

    he->t = RPM_STRING_ARRAY_TYPE;
    he->c = numScripts;

    he->freeData = 1;
    he->p.argv = xmalloc(sizeof(*he->p.argv) * he->c);
    for (i = 0; i < (unsigned) he->c; i++) {
	char * item, * flagsStr;
	char * chptr;

	chptr = xstrdup("");

	for (j = 0; j < (unsigned) numNames; j++) {
	    if (indices.ui32p[j] != i)
		/*@innercontinue@*/ continue;

	    item = xmalloc(strlen(names.argv[j]) + strlen(versions.argv[j]) + 20);
/*@-compmempass@*/
	    if (flags.ui32p[j] & RPMSENSE_SENSEMASK) {
		_he->p.ui32p = &flags.ui32p[j];
		flagsStr = depflagsFormat(_he);
		sprintf(item, "%s %s %s", names.argv[j], flagsStr, versions.argv[j]);
		flagsStr = _free(flagsStr);
	    } else
		strcpy(item, names.argv[j]);
/*@=compmempass@*/

	    chptr = xrealloc(chptr, strlen(chptr) + strlen(item) + 5);
	    if (*chptr != '\0') strcat(chptr, ", ");
	    strcat(chptr, item);
	    item = _free(item);
	}

	he->p.argv[i] = chptr;
    }
    rc = 0;

exit:
    indices.ptr = _free(indices.ptr);
    flags.ptr = _free(flags.ptr);
    names.ptr = _free(names.ptr);
    versions.ptr = _free(versions.ptr);
    s.ptr = _free(s.ptr);

    return rc;
}

/**
 * Retrieve trigger type info.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int triggertypeTag(Header h, HE_t he)
	/*@modifies he @*/
{
    HE_t _he = memset(alloca(sizeof(*_he)), 0, sizeof(*_he));
    rpmTagData indices;
    rpmTagData flags;
    rpmTagData s;
    rpmTagCount numNames;
    rpmTagCount numScripts;
    unsigned i, j;
    int rc = 1;		/* assume failure */
    int xx;

    he->freeData = 0;

/*@-compmempass@*/
    _he->tag = RPMTAG_TRIGGERINDEX;
    xx = headerGet(h, _he, 0);
    indices.ui32p = _he->p.ui32p;
    numNames = _he->c;
    if (!xx) goto exit;

    _he->tag = RPMTAG_TRIGGERFLAGS;
    xx = headerGet(h, _he, 0);
    flags.ui32p = _he->p.ui32p;
    if (!xx) goto exit;

    _he->tag = RPMTAG_TRIGGERSCRIPTS;
    xx = headerGet(h, _he, 0);
    s.argv = _he->p.argv;
    numScripts = _he->c;
    if (!xx) goto exit;
/*@=compmempass@*/

    he->t = RPM_STRING_ARRAY_TYPE;
    he->c = numScripts;

    he->freeData = 1;
    he->p.argv = xmalloc(sizeof(*he->p.argv) * he->c);
    for (i = 0; i < (unsigned) he->c; i++) {
	for (j = 0; j < (unsigned) numNames; j++) {
	    if (indices.ui32p[j] != i)
		/*@innercontinue@*/ continue;

	    /* XXX FIXME: there's memory leaks here. */
	    if (flags.ui32p[j] & RPMSENSE_TRIGGERPREIN)
		he->p.argv[i] = xstrdup("prein");
	    else if (flags.ui32p[j] & RPMSENSE_TRIGGERIN)
		he->p.argv[i] = xstrdup("in");
	    else if (flags.ui32p[j] & RPMSENSE_TRIGGERUN)
		he->p.argv[i] = xstrdup("un");
	    else if (flags.ui32p[j] & RPMSENSE_TRIGGERPOSTUN)
		he->p.argv[i] = xstrdup("postun");
	    else
		he->p.argv[i] = xstrdup("");
	    /*@innerbreak@*/ break;
	}
    }
    rc = 0;

exit:
    indices.ptr = _free(indices.ptr);
    flags.ptr = _free(flags.ptr);
    s.ptr = _free(s.ptr);
    return 0;
}

/* I18N look aside diversions */

#if defined(ENABLE_NLS)
/*@-exportlocal -exportheadervar@*/
/*@unchecked@*/
extern int _nl_msg_cat_cntr;	/* XXX GNU gettext voodoo */
/*@=exportlocal =exportheadervar@*/
#endif
/*@observer@*/ /*@unchecked@*/
static const char * language = "LANGUAGE";

/*@observer@*/ /*@unchecked@*/
static const char * _macro_i18ndomains = "%{?_i18ndomains}";

/**
 * Retrieve i18n text.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int i18nTag(Header h, HE_t he)
	/*@globals rpmGlobalMacroContext, h_errno @*/
	/*@modifies he, rpmGlobalMacroContext @*/
{
    char * dstring = rpmExpand(_macro_i18ndomains, NULL);
    int rc = 1;		/* assume failure */

    he->t = RPM_STRING_TYPE;
    he->p.str = NULL;
    he->c = 0;
    he->freeData = 0;

    if (dstring && *dstring) {
	char *domain, *de;
	const char * langval;
	const char * msgkey;
	const char * msgid;

	{   HE_t nhe = memset(alloca(sizeof(*nhe)), 0, sizeof(*nhe));
	    const char * tn = tagName(he->tag);
	    char * mk;
	    size_t nb = sizeof("()");
	    int xx;
	    nhe->tag = RPMTAG_NAME;
	    xx = headerGet(h, nhe, 0);
	    if (tn)	nb += strlen(tn);
	    if (nhe->p.str)	nb += strlen(nhe->p.str);
	    mk = alloca(nb);
	    sprintf(mk, "%s(%s)", (nhe->p.str ? nhe->p.str : ""), (tn ? tn : ""));
	    nhe->p.ptr = _free(nhe->p.ptr);
	    msgkey = mk;
	}

	/* change to en_US for msgkey -> msgid resolution */
	langval = getenv(language);
	(void) setenv(language, "en_US", 1);
#if defined(ENABLE_NLS)
/*@i@*/	++_nl_msg_cat_cntr;
#endif

	msgid = NULL;
	for (domain = dstring; domain != NULL; domain = de) {
	    de = strchr(domain, ':');
	    if (de) *de++ = '\0';
	    msgid = /*@-unrecog@*/ dgettext(domain, msgkey) /*@=unrecog@*/;
	    if (msgid != msgkey) break;
	}

	/* restore previous environment for msgid -> msgstr resolution */
	if (langval)
	    (void) setenv(language, langval, 1);
	else
	    unsetenv(language);
#if defined(ENABLE_NLS)
/*@i@*/	++_nl_msg_cat_cntr;
#endif

	if (domain && msgid) {
	    const char * s = /*@-unrecog@*/ dgettext(domain, msgid) /*@=unrecog@*/;
	    if (s) {
		rc = 0;
		he->p.str = xstrdup(s);
		he->c = 1;
		he->freeData = 1;
	    }
	}
    }

/*@-dependenttrans@*/
    dstring = _free(dstring);
/*@=dependenttrans@*/
    if (!rc)
	return rc;

    rc = headerGet(h, he, HEADERGET_NOEXTENSION);
    if (rc) {
	rc = 0;
	he->p.str = xstrtolocale(he->p.str);
	he->freeData = 1;
/*@-nullstate@*/
	return rc;
/*@=nullstate@*/
    }

    he->t = RPM_STRING_TYPE;
    he->p.str = NULL;
    he->c = 0;
    he->freeData = 0;

    return 1;
}

/**
 * Retrieve text and convert to locale.
 */
static int localeTag(Header h, HE_t he)
	/*@modifies he @*/
{
    int rc;

    rc = headerGet(h, he, HEADERGET_NOEXTENSION);
    if (!rc || he->p.str == NULL || he->c == 0) {
	he->t = RPM_STRING_TYPE;
	he->freeData = 0;
	return 1;
    }

    switch (he->t) {
    default:
	he->freeData = 0;
	break;
    case RPM_STRING_TYPE:
	he->p.str = xstrtolocale(he->p.str);
	he->freeData = 1;
	break;
    case RPM_STRING_ARRAY_TYPE:
    {	const char ** argv;
	char * te;
	size_t l = 0;
	unsigned i;
	for (i = 0; i < (unsigned) he->c; i++) {
	    he->p.argv[i] = xstrdup(he->p.argv[i]);
	    he->p.argv[i] = xstrtolocale(he->p.argv[i]);
assert(he->p.argv[i] != NULL);
	    l += strlen(he->p.argv[i]) + 1;
	}
	argv = xmalloc(he->c * sizeof(*argv) + l);
	te = (char *)&argv[he->c];
	for (i = 0; i < (unsigned) he->c; i++) {
	    argv[i] = te;
	    te = stpcpy(te, he->p.argv[i]);
	    te++;
	    he->p.argv[i] = _free(he->p.argv[i]);
	}
	he->p.ptr = _free(he->p.ptr);
	he->p.argv = argv;
	he->freeData = 1;
    }	break;
    }

    return 0;
}

/**
 * Retrieve summary text.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int summaryTag(Header h, HE_t he)
	/*@globals rpmGlobalMacroContext, h_errno @*/
	/*@modifies he, rpmGlobalMacroContext @*/
{
    he->tag = RPMTAG_SUMMARY;
    return i18nTag(h, he);
}

/**
 * Retrieve description text.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int descriptionTag(Header h, HE_t he)
	/*@globals rpmGlobalMacroContext, h_errno @*/
	/*@modifies he, rpmGlobalMacroContext @*/
{
    he->tag = RPMTAG_DESCRIPTION;
    return i18nTag(h, he);
}

static int changelognameTag(Header h, HE_t he)
	/*@modifies he @*/
{
    he->tag = RPMTAG_CHANGELOGNAME;
    return localeTag(h, he);
}

static int changelogtextTag(Header h, HE_t he)
	/*@modifies he @*/
{
    he->tag = RPMTAG_CHANGELOGTEXT;
    return localeTag(h, he);
}

/**
 * Retrieve group text.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int groupTag(Header h, HE_t he)
	/*@globals rpmGlobalMacroContext, h_errno @*/
	/*@modifies he, rpmGlobalMacroContext @*/
{
    he->tag = RPMTAG_GROUP;
    return i18nTag(h, he);
}

/**
 * Retrieve db instance from header.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
/*@-globuse@*/
static int dbinstanceTag(Header h, HE_t he)
	/*@globals rpmGlobalMacroContext, h_errno,
		fileSystem, internalState @*/
	/*@modifies he, rpmGlobalMacroContext,
		fileSystem, internalState @*/
{
    he->tag = RPMTAG_DBINSTANCE;
    he->t = RPM_UINT32_TYPE;
    he->p.ui32p = xmalloc(sizeof(*he->p.ui32p));
    he->p.ui32p[0] = headerGetInstance(h);
    he->freeData = 1;
    he->c = 1;
    return 0;
}
/*@=globuse@*/

/**
 * Return (malloc'd) header name-version-release.arch string.
 * @param h		header
 * @return		name-version-release.arch string
 */
/*@only@*/
static char * hGetNVRA(Header h)
	/*@modifies h @*/
{
    const char * N = NULL;
    const char * V = NULL;
    const char * R = NULL;
    const char * A = NULL;
    size_t nb = 0;
    char * NVRA, * t;

    (void) headerNEVRA(h, &N, NULL, &V, &R, &A);
    if (N)	nb += strlen(N);
    if (V)	nb += strlen(V) + 1;
    if (R)	nb += strlen(R) + 1;
#if defined(RPM_VENDOR_OPENPKG) /* no-architecture-expose */
    /* do not expose the architecture as this is too less
       information, as in OpenPKG the "platform" is described by the
       architecture+operating-system combination. But as the whole
       "platform" information is actually overkill, just revert to the
       RPM 4 behaviour and do not expose any such information at all. */
#else
    if (A)	nb += strlen(A) + 1;
#endif
    nb++;
    NVRA = t = xmalloc(nb);
    *t = '\0';
    if (N)	t = stpcpy(t, N);
    if (V)	t = stpcpy( stpcpy(t, "-"), V);
    if (R)	t = stpcpy( stpcpy(t, "-"), R);
#if defined(RPM_VENDOR_OPENPKG) /* no-architecture-expose */
    /* do not expose the architecture as this is too less
       information, as in OpenPKG the "platform" is described by the
       architecture+operating-system combination. But as the whole
       "platform" information is actually overkill, just revert to the
       RPM 4 behaviour and do not expose any such information at all. */
#else
    if (A)	t = stpcpy( stpcpy(t, "."), A);
#endif
    N = _free(N);
    V = _free(V);
    R = _free(R);
    A = _free(A);
    return NVRA;
}

/**
 * Retrieve N-V-R.A compound string from header.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
/*@-globuse@*/
static int nvraTag(Header h, HE_t he)
	/*@globals rpmGlobalMacroContext, h_errno,
		fileSystem, internalState @*/
	/*@modifies h, he, rpmGlobalMacroContext,
		fileSystem, internalState @*/
{
    he->t = RPM_STRING_TYPE;
    he->p.str = hGetNVRA(h);
    he->c = 1;
    he->freeData = 1;
    return 0;
}
/*@=globuse@*/

/**
 * Retrieve file names from header.
 *
 * The representation of file names in package headers changed in rpm-4.0.
 * Originally, file names were stored as an array of absolute paths.
 * In rpm-4.0, file names are stored as separate arrays of dirname's and
 * basename's, * with a dirname index to associate the correct dirname
 * with each basname.
 *
 * This function is used to retrieve file names independent of how the
 * file names are represented in the package header.
 * 
 * @param h		header
 * @param tagN		RPMTAG_BASENAMES | PMTAG_ORIGBASENAMES
 * @retval *fnp		array of file names
 * @retval *fcp		number of files
 */
static void rpmfiBuildFNames(Header h, rpmTag tagN,
		/*@null@*/ /*@out@*/ const char *** fnp,
		/*@null@*/ /*@out@*/ rpmTagCount * fcp)
	/*@modifies *fnp, *fcp @*/
{
    HE_t he = memset(alloca(sizeof(*he)), 0, sizeof(*he));
    rpmTag dirNameTag = 0;
    rpmTag dirIndexesTag = 0;
    rpmTagData baseNames;
    rpmTagData dirNames;
    rpmTagData dirIndexes;
    rpmTagData fileNames;
    rpmTagCount count;
    size_t size;
    char * t;
    unsigned i;
    int xx;

    if (tagN == RPMTAG_BASENAMES) {
	dirNameTag = RPMTAG_DIRNAMES;
	dirIndexesTag = RPMTAG_DIRINDEXES;
    } else if (tagN == RPMTAG_ORIGBASENAMES) {
	dirNameTag = RPMTAG_ORIGDIRNAMES;
	dirIndexesTag = RPMTAG_ORIGDIRINDEXES;
    } else {
	if (fnp) *fnp = NULL;
	if (fcp) *fcp = 0;
	return;		/* programmer error */
    }

/*@-compmempass@*/
    he->tag = tagN;
    xx = headerGet(h, he, 0);
    baseNames.argv = he->p.argv;
    count = he->c;

    if (!xx) {
	if (fnp) *fnp = NULL;
	if (fcp) *fcp = 0;
	return;		/* no file list */
    }

    he->tag = dirNameTag;
    xx = headerGet(h, he, 0);
    dirNames.argv = he->p.argv;

    he->tag = dirIndexesTag;
    xx = headerGet(h, he, 0);
    dirIndexes.ui32p = he->p.ui32p;
    count = he->c;
/*@=compmempass@*/

    size = sizeof(*fileNames.argv) * count;
    for (i = 0; i < (unsigned)count; i++) {
	const char * dn = NULL;
	(void) urlPath(dirNames.argv[dirIndexes.ui32p[i]], &dn);
	size += strlen(baseNames.argv[i]) + strlen(dn) + 1;
    }

    fileNames.argv = xmalloc(size);
    t = (char *)&fileNames.argv[count];
    for (i = 0; i < (unsigned)count; i++) {
	const char * dn = NULL;
	(void) urlPath(dirNames.argv[dirIndexes.ui32p[i]], &dn);
	fileNames.argv[i] = t;
	t = stpcpy( stpcpy(t, dn), baseNames.argv[i]);
	*t++ = '\0';
    }
    baseNames.ptr = _free(baseNames.ptr);
    dirNames.ptr = _free(dirNames.ptr);
    dirIndexes.ptr = _free(dirIndexes.ptr);

/*@-onlytrans@*/
    if (fnp)
	*fnp = fileNames.argv;
    else
	fileNames.ptr = _free(fileNames.ptr);
/*@=onlytrans@*/
    if (fcp) *fcp = count;
}

/**
 * Retrieve file paths.
 * @param h		header
 * @retval *he		tag container
 * @return		0 on success
 */
static int _fnTag(Header h, HE_t he)
	/*@modifies he @*/
{
    he->t = RPM_STRING_ARRAY_TYPE;
    rpmfiBuildFNames(h, he->tag, &he->p.argv, &he->c);
    he->freeData = 1;
    return 0;
}

static int filepathsTag(Header h, HE_t he)
	/*@modifies he @*/
{
    he->tag = RPMTAG_BASENAMES;
    return _fnTag(h, he);
}

static int origpathsTag(Header h, HE_t he)
	/*@modifies he @*/
{
    he->tag = RPMTAG_ORIGBASENAMES;
    return _fnTag(h, he);
}

/*@-type@*/ /* FIX: cast? */
static struct headerSprintfExtension_s _headerCompoundFormats[] = {
    { HEADER_EXT_TAG, "RPMTAG_CHANGELOGNAME",
	{ .tagFunction = changelognameTag } },
    { HEADER_EXT_TAG, "RPMTAG_CHANGELOGTEXT",
	{ .tagFunction = changelogtextTag } },
    { HEADER_EXT_TAG, "RPMTAG_DESCRIPTION",
	{ .tagFunction = descriptionTag } },
    { HEADER_EXT_TAG, "RPMTAG_GROUP",
	{ .tagFunction = groupTag } },
    { HEADER_EXT_TAG, "RPMTAG_INSTALLPREFIX",
	{ .tagFunction = instprefixTag } },
    { HEADER_EXT_TAG, "RPMTAG_SUMMARY",
	{ .tagFunction = summaryTag } },
    { HEADER_EXT_TAG, "RPMTAG_TRIGGERCONDS",
	{ .tagFunction = triggercondsTag } },
    { HEADER_EXT_TAG, "RPMTAG_TRIGGERTYPE",
	{ .tagFunction = triggertypeTag } },
    { HEADER_EXT_TAG, "RPMTAG_DBINSTANCE",
	{ .tagFunction = dbinstanceTag } },
    { HEADER_EXT_TAG, "RPMTAG_NVRA",
	{ .tagFunction = nvraTag } },
    { HEADER_EXT_TAG, "RPMTAG_FILENAMES",
	{ .tagFunction = filepathsTag } },
    { HEADER_EXT_TAG, "RPMTAG_FILEPATHS",
	{ .tagFunction = filepathsTag } },
    { HEADER_EXT_TAG, "RPMTAG_ORIGPATHS",
	{ .tagFunction = origpathsTag } },
    { HEADER_EXT_FORMAT, "armor",
	{ .fmtFunction = armorFormat } },
    { HEADER_EXT_FORMAT, "base64",
	{ .fmtFunction = base64Format } },
    { HEADER_EXT_FORMAT, "depflags",
	{ .fmtFunction = depflagsFormat } },
    { HEADER_EXT_FORMAT, "fflags",
	{ .fmtFunction = fflagsFormat } },
    { HEADER_EXT_FORMAT, "perms",
	{ .fmtFunction = permsFormat } },
    { HEADER_EXT_FORMAT, "permissions",	
	{ .fmtFunction = permsFormat } },
    { HEADER_EXT_FORMAT, "pgpsig",
	{ .fmtFunction = pgpsigFormat } },
    { HEADER_EXT_FORMAT, "triggertype",	
	{ .fmtFunction = triggertypeFormat } },
    { HEADER_EXT_FORMAT, "xml",
	{ .fmtFunction = xmlFormat } },
    { HEADER_EXT_FORMAT, "yaml",
	{ .fmtFunction = yamlFormat } },
    { HEADER_EXT_MORE, NULL,		{ (void *) &headerDefaultFormats } }
} ;
/*@=type@*/

headerSprintfExtension headerCompoundFormats = &_headerCompoundFormats[0];

/*====================================================================*/

void rpmDisplayQueryTags(FILE * fp, headerTagTableEntry _rpmTagTable, headerSprintfExtension _rpmHeaderFormats)
{
    const struct headerTagTableEntry_s * t;
    headerSprintfExtension exts;
    headerSprintfExtension ext;
    int extNum;

    if (fp == NULL)
	fp = stdout;
    if (_rpmTagTable == NULL)
	_rpmTagTable = rpmTagTable;

    /* XXX this should use rpmHeaderFormats, but there are linkage problems. */
    if (_rpmHeaderFormats == NULL)
	_rpmHeaderFormats = headerCompoundFormats;

    for (t = _rpmTagTable; t && t->name; t++) {
	/*@observer@*/
	static const char * tagtypes[] = {
		"", "char", "uint8", "uint16", "uint32", "uint64",
		"string", "octets", "argv", "i18nstring",
	};
	uint32_t ttype;

	if (rpmIsVerbose()) {
	    fprintf(fp, "%-20s %6d", t->name + 7, t->val);
	    ttype = t->type & RPM_MASK_TYPE;
	    if (ttype < RPM_MIN_TYPE || ttype > RPM_MAX_TYPE)
		continue;
	    if (t->type & RPM_OPENPGP_RETURN_TYPE)
		fprintf(fp, " openpgp");
	    if (t->type & RPM_X509_RETURN_TYPE)
		fprintf(fp, " x509");
	    if (t->type & RPM_ASN1_RETURN_TYPE)
		fprintf(fp, " asn1");
	    if (t->type & RPM_OPAQUE_RETURN_TYPE)
		fprintf(fp, " opaque");
	    fprintf(fp, " %s", tagtypes[ttype]);
	    if (t->type & RPM_ARRAY_RETURN_TYPE)
		fprintf(fp, " array");
	    if (t->type & RPM_MAPPING_RETURN_TYPE)
		fprintf(fp, " mapping");
	    if (t->type & RPM_PROBE_RETURN_TYPE)
		fprintf(fp, " probe");
	    if (t->type & RPM_TREE_RETURN_TYPE)
		fprintf(fp, " tree");
	} else
	    fprintf(fp, "%s", t->name + 7);
	fprintf(fp, "\n");
    }

    exts = _rpmHeaderFormats;
    for (ext = exts, extNum = 0; ext != NULL && ext->type != HEADER_EXT_LAST;
	ext = (ext->type == HEADER_EXT_MORE ? *ext->u.more : ext+1), extNum++)
    {
	if (ext->name == NULL || ext->type != HEADER_EXT_TAG)
	    continue;

	/* XXX don't print header tags twice. */
	if (tagValue(ext->name) > 0)
	    continue;
	fprintf(fp, "%s\n", ext->name + 7);
    }
}

/*====================================================================*/

#define PARSER_BEGIN 	0
#define PARSER_IN_ARRAY 1
#define PARSER_IN_EXPR  2

/** \ingroup header
 */
typedef /*@abstract@*/ struct sprintfTag_s * sprintfTag;

/** \ingroup header
 */
struct sprintfTag_s {
    HE_s he;
/*@null@*/
    headerTagFormatFunction fmt;
/*@null@*/
    headerTagTagFunction ext;   /*!< NULL if tag element is invalid */
    int extNum;
    rpmTag tagno;
    int justOne;
    int arrayCount;
/*@kept@*/
    char * format;
/*@kept@*/ /*@null@*/
    char * type;
    unsigned pad;
};

/** \ingroup header
 */
typedef /*@abstract@*/ struct sprintfToken_s * sprintfToken;

/** \ingroup header
 */
struct sprintfToken_s {
    enum {
        PTOK_NONE       = 0,
        PTOK_TAG        = 1,
        PTOK_ARRAY      = 2,
        PTOK_STRING     = 3,
        PTOK_COND       = 4
    } type;
    union {
	struct sprintfTag_s tag;	/*!< PTOK_TAG */
	struct {
	/*@only@*/
	    sprintfToken format;
	    size_t numTokens;
	} array;			/*!< PTOK_ARRAY */
	struct {
	/*@dependent@*/
	    char * string;
	    size_t len;
	} string;			/*!< PTOK_STRING */
	struct {
	/*@only@*/ /*@null@*/
	    sprintfToken ifFormat;
	    size_t numIfTokens;
	/*@only@*/ /*@null@*/
	    sprintfToken elseFormat;
	    size_t numElseTokens;
	    struct sprintfTag_s tag;
	} cond;				/*!< PTOK_COND */
    } u;
};

/** \ingroup header
 */
typedef /*@abstract@*/ struct headerSprintfArgs_s * headerSprintfArgs;

/** \ingroup header
 */
struct headerSprintfArgs_s {
    Header h;
    char * fmt;
/*@observer@*/ /*@temp@*/
    headerTagTableEntry tags;
/*@observer@*/ /*@temp@*/
    headerSprintfExtension exts;
/*@observer@*/ /*@null@*/
    const char * errmsg;
    HE_t ec;			/*!< Extension data cache. */
    int nec;			/*!< No. of extension cache items. */
    sprintfToken format;
/*@relnull@*/
    HeaderIterator hi;
/*@owned@*/
    char * val;
    size_t vallen;
    size_t alloced;
    size_t numTokens;
    size_t i;
};

/*@access sprintfTag @*/
/*@access sprintfToken @*/
/*@access headerSprintfArgs @*/

/**
 */
static char escapedChar(const char ch)	/*@*/
{
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\t\t\\%c\n", ch);
/*@=modfilesys@*/
    switch (ch) {
    case 'a': 	return '\a';
    case 'b': 	return '\b';
    case 'f': 	return '\f';
    case 'n': 	return '\n';
    case 'r': 	return '\r';
    case 't': 	return '\t';
    case 'v': 	return '\v';
    default:	return ch;
    }
}

/**
 * Clean a tag container, free'ing attached malloc's.
 * @param he		tag container
 */
/*@relnull@*/
static HE_t rpmheClean(/*@returned@*/ /*@null@*/ HE_t he)
	/*@modifies he @*/
{
    if (he) {
	if (he->freeData && he->p.ptr != NULL)
	    he->p.ptr = _free(he->p.ptr);
	memset(he, 0, sizeof(*he));
    }
    return he;
}

/**
 * Destroy headerSprintf format array.
 * @param format	sprintf format array
 * @param num		number of elements
 * @return		NULL always
 */
static /*@null@*/ sprintfToken
freeFormat( /*@only@*/ /*@null@*/ sprintfToken format, size_t num)
	/*@modifies *format @*/
{
    unsigned i;

    if (format == NULL) return NULL;

    for (i = 0; i < (unsigned) num; i++) {
	switch (format[i].type) {
	case PTOK_TAG:
	    (void) rpmheClean(&format[i].u.tag.he);
	    /*@switchbreak@*/ break;
	case PTOK_ARRAY:
	    format[i].u.array.format =
		freeFormat(format[i].u.array.format,
			format[i].u.array.numTokens);
	    /*@switchbreak@*/ break;
	case PTOK_COND:
	    format[i].u.cond.ifFormat =
		freeFormat(format[i].u.cond.ifFormat, 
			format[i].u.cond.numIfTokens);
	    format[i].u.cond.elseFormat =
		freeFormat(format[i].u.cond.elseFormat, 
			format[i].u.cond.numElseTokens);
	    (void) rpmheClean(&format[i].u.cond.tag.he);
	    /*@switchbreak@*/ break;
	case PTOK_NONE:
	case PTOK_STRING:
	default:
	    /*@switchbreak@*/ break;
	}
    }
    format = _free(format);
    return NULL;
}

/**
 * Initialize an hsa iteration.
 * @param hsa		headerSprintf args
 * @return		headerSprintf args
 */
static headerSprintfArgs hsaInit(/*@returned@*/ headerSprintfArgs hsa)
	/*@modifies hsa */
{
    sprintfTag tag =
	(hsa->format->type == PTOK_TAG
	    ? &hsa->format->u.tag :
	(hsa->format->type == PTOK_ARRAY
	    ? &hsa->format->u.array.format->u.tag :
	NULL));

    if (hsa != NULL) {
	hsa->i = 0;
	if (tag != NULL && tag->tagno == -2)
	    hsa->hi = headerInit(hsa->h);
    }
/*@-nullret@*/
    return hsa;
/*@=nullret@*/
}

/**
 * Return next hsa iteration item.
 * @param hsa		headerSprintf args
 * @return		next sprintfToken (or NULL)
 */
/*@null@*/
static sprintfToken hsaNext(/*@returned@*/ headerSprintfArgs hsa)
	/*@modifies hsa */
{
    sprintfToken fmt = NULL;
    sprintfTag tag =
	(hsa->format->type == PTOK_TAG
	    ? &hsa->format->u.tag :
	(hsa->format->type == PTOK_ARRAY
	    ? &hsa->format->u.array.format->u.tag :
	NULL));

    if (hsa != NULL && hsa->i < hsa->numTokens) {
	fmt = hsa->format + hsa->i;
	if (hsa->hi == NULL) {
	    hsa->i++;
	} else {
	    HE_t he = rpmheClean(&tag->he);
	    if (!headerNext(hsa->hi, he, 0))
	    {
		tag->tagno = 0;
		return NULL;
	    }
	    he->avail = 1;
	    tag->tagno = he->tag;
	}
    }

/*@-dependenttrans -onlytrans@*/
    return fmt;
/*@=dependenttrans =onlytrans@*/
}

/**
 * Finish an hsa iteration.
 * @param hsa		headerSprintf args
 * @return		headerSprintf args
 */
static headerSprintfArgs hsaFini(/*@returned@*/ headerSprintfArgs hsa)
	/*@modifies hsa */
{
    if (hsa != NULL) {
	hsa->hi = headerFini(hsa->hi);
	hsa->i = 0;
    }
/*@-nullret@*/
    return hsa;
/*@=nullret@*/
}

/**
 * Reserve sufficient buffer space for next output value.
 * @param hsa		headerSprintf args
 * @param need		no. of bytes to reserve
 * @return		pointer to reserved space
 */
/*@dependent@*/ /*@exposed@*/
static char * hsaReserve(headerSprintfArgs hsa, size_t need)
	/*@modifies hsa */
{
    if ((hsa->vallen + need) >= hsa->alloced) {
	if (hsa->alloced <= need)
	    hsa->alloced += need;
	hsa->alloced <<= 1;
	hsa->val = xrealloc(hsa->val, hsa->alloced+1);	
    }
    return hsa->val + hsa->vallen;
}

/**
 * Return tag name from value.
 * @param tbl		tag table
 * @param val		tag value to find
 * @retval *typep	tag type (or NULL)
 * @return		tag name, NULL on not found
 */
/*@observer@*/ /*@null@*/
static const char * myTagName(headerTagTableEntry tbl, uint32_t val,
		/*@null@*/ uint32_t *typep)
	/*@modifies *typep @*/
{
    static char name[128];	/* XXX Ick. */
    const char * s;
    char *t;

    /* XXX Use bsearch on the "normal" rpmTagTable lookup. */
    if (tbl == NULL || tbl == rpmTagTable) {
	s = tagName(val);
	if (s != NULL && typep != NULL)
	    *typep = tagType(val);
	return s;
    }

    for (; tbl->name != NULL; tbl++) {
	if (tbl->val == val)
	    break;
    }
    if ((s = tbl->name) == NULL)
	return NULL;
    s += sizeof("RPMTAG_") - 1;
    t = name;
    *t++ = *s++;
    while (*s != '\0')
	*t++ = (char)xtolower((int)*s++);
    *t = '\0';
    if (typep)
	*typep = tbl->type;
    return name;
}

/**
 * Return tag value from name.
 * @param tbl		tag table
 * @param name		tag name to find
 * @return		tag value, 0 on not found
 */
static uint32_t myTagValue(headerTagTableEntry tbl, const char * name)
	/*@*/
{
    uint32_t val = 0;

    /* XXX Use bsearch on the "normal" rpmTagTable lookup. */
    if (tbl == NULL || tbl == rpmTagTable)
	val = tagValue(name);
    else
    for (; tbl->name != NULL; tbl++) {
	if (xstrcasecmp(tbl->name, name))
	    continue;
	val = tbl->val;
	break;
    }
    return val;
}

/**
 * Search extensions and tags for a name.
 * @param hsa		headerSprintf args
 * @param token		parsed fields
 * @param name		name to find
 * @return		0 on success, 1 on not found
 */
static int findTag(headerSprintfArgs hsa, sprintfToken token, const char * name)
	/*@modifies token @*/
{
    headerSprintfExtension exts = hsa->exts;
    headerSprintfExtension ext;
    sprintfTag stag = (token->type == PTOK_COND
	? &token->u.cond.tag : &token->u.tag);
    int extNum;

    stag->fmt = NULL;
    stag->ext = NULL;
    stag->extNum = 0;
    stag->tagno = -1;

    if (!strcmp(name, "*")) {
	stag->tagno = -2;
	goto bingo;
    }

    if (strncmp("RPMTAG_", name, sizeof("RPMTAG_")-1)) {
	char * t = alloca(strlen(name) + sizeof("RPMTAG_"));
	(void) stpcpy( stpcpy(t, "RPMTAG_"), name);
	name = t;
    }

    /* Search extensions for specific tag override. */
    for (ext = exts, extNum = 0; ext != NULL && ext->type != HEADER_EXT_LAST;
	ext = (ext->type == HEADER_EXT_MORE ? *ext->u.more : ext+1), extNum++)
    {
	if (ext->name == NULL || ext->type != HEADER_EXT_TAG)
	    continue;
	if (!xstrcasecmp(ext->name, name)) {
	    stag->ext = ext->u.tagFunction;
	    stag->extNum = extNum;
	    goto bingo;
	}
    }

    /* Search tag names. */
    stag->tagno = myTagValue(hsa->tags, name);
    if (stag->tagno != 0)
	goto bingo;

    return 1;

bingo:
    /* Search extensions for specific format. */
    if (stag->type != NULL)
    for (ext = exts; ext != NULL && ext->type != HEADER_EXT_LAST;
	    ext = (ext->type == HEADER_EXT_MORE ? *ext->u.more : ext+1))
    {
	if (ext->name == NULL || ext->type != HEADER_EXT_FORMAT)
	    continue;
	if (!strcmp(ext->name, stag->type)) {
	    stag->fmt = ext->u.fmtFunction;
	    break;
	}
    }
    return 0;
}

/* forward ref */
/**
 * Parse a headerSprintf expression.
 * @param hsa		headerSprintf args
 * @param token
 * @param str
 * @retval *endPtr
 * @return		0 on success
 */
static int parseExpression(headerSprintfArgs hsa, sprintfToken token,
		char * str, /*@out@*/char ** endPtr)
	/*@modifies hsa, str, token, *endPtr @*/
	/*@requires maxSet(endPtr) >= 0 @*/;

/**
 * Parse a headerSprintf term.
 * @param hsa		headerSprintf args
 * @param str
 * @retval *formatPtr
 * @retval *numTokensPtr
 * @retval *endPtr
 * @param state
 * @return		0 on success
 */
static int parseFormat(headerSprintfArgs hsa, /*@null@*/ char * str,
		/*@out@*/ sprintfToken * formatPtr,
		/*@out@*/ size_t * numTokensPtr,
		/*@null@*/ /*@out@*/ char ** endPtr, int state)
	/*@modifies hsa, str, *formatPtr, *numTokensPtr, *endPtr @*/
	/*@requires maxSet(formatPtr) >= 0 /\ maxSet(numTokensPtr) >= 0
		/\ maxSet(endPtr) >= 0 @*/
{
/*@observer@*/
static const char *pstates[] = {
"NORMAL", "ARRAY", "EXPR", "WTF?"
};
    char * chptr, * start, * next, * dst;
    sprintfToken format;
    sprintfToken token;
    size_t numTokens;
    unsigned i;
    int done = 0;

/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "-->     parseFormat(%p, \"%s\", %p, %p, %p, %s)\n", hsa, str, formatPtr, numTokensPtr, endPtr, pstates[(state & 0x3)]);
/*@=modfilesys@*/

    /* upper limit on number of individual formats */
    numTokens = 0;
    if (str != NULL)
    for (chptr = str; *chptr != '\0'; chptr++)
	if (*chptr == '%') numTokens++;
    numTokens = numTokens * 2 + 1;

    format = xcalloc(numTokens, sizeof(*format));
    if (endPtr) *endPtr = NULL;

/*@-infloops@*/ /* LCL: can't detect done termination */
    dst = start = str;
    numTokens = 0;
    token = NULL;
    if (start != NULL)
    while (*start != '\0') {
	switch (*start) {
	case '%':
	    /* handle %% */
	    if (*(start + 1) == '%') {
		if (token == NULL || token->type != PTOK_STRING) {
		    token = format + numTokens++;
		    token->type = PTOK_STRING;
		    /*@-temptrans -assignexpose@*/
		    dst = token->u.string.string = start;
		    /*@=temptrans =assignexpose@*/
		}
		start++;
		*dst++ = *start++;
		/*@switchbreak@*/ break;
	    } 

	    token = format + numTokens++;
	    *dst++ = '\0';
	    start++;

	    if (*start == '|') {
		char * newEnd;

		start++;
		if (parseExpression(hsa, token, start, &newEnd))
		{
		    format = freeFormat(format, numTokens);
		    return 1;
		}
		start = newEnd;
		/*@switchbreak@*/ break;
	    }

	    /*@-assignexpose@*/
	    token->u.tag.format = start;
	    /*@=assignexpose@*/
	    token->u.tag.pad = 0;
	    token->u.tag.justOne = 0;
	    token->u.tag.arrayCount = 0;

	    chptr = start;
	    while (*chptr && *chptr != '{' && *chptr != '%') chptr++;
	    if (!*chptr || *chptr == '%') {
		hsa->errmsg = _("missing { after %");
		format = freeFormat(format, numTokens);
		return 1;
	    }

/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\tchptr *%p = NUL\n", chptr);
/*@=modfilesys@*/
	    *chptr++ = '\0';

	    while (start < chptr) {
		if (xisdigit((int)*start)) {
		    i = strtoul(start, &start, 10);
		    token->u.tag.pad += i;
		    start = chptr;
		    /*@innerbreak@*/ break;
		} else {
		    start++;
		}
	    }

	    if (*start == '=') {
		token->u.tag.justOne = 1;
		start++;
	    } else if (*start == '#') {
		token->u.tag.justOne = 1;
		token->u.tag.arrayCount = 1;
		start++;
	    }

	    next = start;
	    while (*next && *next != '}') next++;
	    if (!*next) {
		hsa->errmsg = _("missing } after %{");
		format = freeFormat(format, numTokens);
		return 1;
	    }
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\tnext *%p = NUL\n", next);
/*@=modfilesys@*/
	    *next++ = '\0';

	    chptr = start;
	    while (*chptr && *chptr != ':') chptr++;

	    if (*chptr != '\0') {
		*chptr++ = '\0';
		if (!*chptr) {
		    hsa->errmsg = _("empty tag format");
		    format = freeFormat(format, numTokens);
		    return 1;
		}
		/*@-assignexpose@*/
		token->u.tag.type = chptr;
		/*@=assignexpose@*/
	    } else {
		token->u.tag.type = NULL;
	    }
	    
	    if (!*start) {
		hsa->errmsg = _("empty tag name");
		format = freeFormat(format, numTokens);
		return 1;
	    }

	    i = 0;
	    token->type = PTOK_TAG;

	    if (findTag(hsa, token, start)) {
		hsa->errmsg = _("unknown tag");
		format = freeFormat(format, numTokens);
		return 1;
	    }

	    dst = start = next;
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\tdst = start = next %p\n", dst);
/*@=modfilesys@*/
	    /*@switchbreak@*/ break;

	case '[':
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\t%s => %s *%p = NUL\n", pstates[(state & 0x3)], pstates[PARSER_IN_ARRAY], start);
/*@=modfilesys@*/
	    *start++ = '\0';
	    token = format + numTokens++;

	    if (parseFormat(hsa, start,
			    &token->u.array.format,
			    &token->u.array.numTokens,
			    &start, PARSER_IN_ARRAY))
	    {
		format = freeFormat(format, numTokens);
		return 1;
	    }

	    if (!start) {
		hsa->errmsg = _("] expected at end of array");
		format = freeFormat(format, numTokens);
		return 1;
	    }

	    dst = start;
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\tdst = start %p\n", dst);
/*@=modfilesys@*/

	    token->type = PTOK_ARRAY;

	    /*@switchbreak@*/ break;

	case ']':
	    if (state != PARSER_IN_ARRAY) {
		hsa->errmsg = _("unexpected ]");
		format = freeFormat(format, numTokens);
		return 1;
	    }
	    *start++ = '\0';
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\t<= %s %p[-1] = NUL\n", pstates[(state & 0x3)], start);
/*@=modfilesys@*/
	    if (endPtr) *endPtr = start;
	    done = 1;
	    /*@switchbreak@*/ break;

	case '}':
	    if (state != PARSER_IN_EXPR) {
		hsa->errmsg = _("unexpected }");
		format = freeFormat(format, numTokens);
		return 1;
	    }
	    *start++ = '\0';
/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\t<= %s %p[-1] = NUL\n", pstates[(state & 0x3)], start);
/*@=modfilesys@*/
	    if (endPtr) *endPtr = start;
	    done = 1;
	    /*@switchbreak@*/ break;

	default:
	    if (token == NULL || token->type != PTOK_STRING) {
		token = format + numTokens++;
		token->type = PTOK_STRING;
		/*@-temptrans -assignexpose@*/
		dst = token->u.string.string = start;
		/*@=temptrans =assignexpose@*/
	    }

/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "\t*%p = *%p \"%s\"\n", dst, start, start);
/*@=modfilesys@*/
	    if (*start == '\\') {
		start++;
		*dst++ = escapedChar(*start);
		*start++ = '\0';
	    } else {
		*dst++ = *start++;
	    }
	    /*@switchbreak@*/ break;
	}
	if (done)
	    break;
    }
/*@=infloops@*/

    if (dst != NULL)
        *dst = '\0';

    for (i = 0; i < (unsigned) numTokens; i++) {
	token = format + i;
	switch(token->type) {
	default:
	    /*@switchbreak@*/ break;
	case PTOK_STRING:
	    token->u.string.len = strlen(token->u.string.string);
	    /*@switchbreak@*/ break;
	}
    }

    if (numTokensPtr != NULL)
	*numTokensPtr = numTokens;
    if (formatPtr != NULL)
	*formatPtr = format;

    return 0;
}

static int parseExpression(headerSprintfArgs hsa, sprintfToken token,
		char * str, /*@out@*/ char ** endPtr)
{
    char * chptr;
    char * end;

/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "-->   parseExpression(%p, %p, \"%s\", %p)\n", hsa, token, str, endPtr);
/*@=modfilesys@*/

    hsa->errmsg = NULL;
    chptr = str;
    while (*chptr && *chptr != '?') chptr++;

    if (*chptr != '?') {
	hsa->errmsg = _("? expected in expression");
	return 1;
    }

    *chptr++ = '\0';

    if (*chptr != '{') {
	hsa->errmsg = _("{ expected after ? in expression");
	return 1;
    }

    chptr++;

    if (parseFormat(hsa, chptr, &token->u.cond.ifFormat, 
		    &token->u.cond.numIfTokens, &end, PARSER_IN_EXPR)) 
	return 1;

    /* XXX fix segfault on "rpm -q rpm --qf='%|NAME?{%}:{NAME}|\n'"*/
    if (!(end && *end)) {
	hsa->errmsg = _("} expected in expression");
	token->u.cond.ifFormat =
		freeFormat(token->u.cond.ifFormat, token->u.cond.numIfTokens);
	return 1;
    }

    chptr = end;
    if (*chptr != ':' && *chptr != '|') {
	hsa->errmsg = _(": expected following ? subexpression");
	token->u.cond.ifFormat =
		freeFormat(token->u.cond.ifFormat, token->u.cond.numIfTokens);
	return 1;
    }

    if (*chptr == '|') {
	if (parseFormat(hsa, NULL, &token->u.cond.elseFormat, 
		&token->u.cond.numElseTokens, &end, PARSER_IN_EXPR))
	{
	    token->u.cond.ifFormat =
		freeFormat(token->u.cond.ifFormat, token->u.cond.numIfTokens);
	    return 1;
	}
    } else {
	chptr++;

	if (*chptr != '{') {
	    hsa->errmsg = _("{ expected after : in expression");
	    token->u.cond.ifFormat =
		freeFormat(token->u.cond.ifFormat, token->u.cond.numIfTokens);
	    return 1;
	}

	chptr++;

	if (parseFormat(hsa, chptr, &token->u.cond.elseFormat, 
			&token->u.cond.numElseTokens, &end, PARSER_IN_EXPR)) 
	    return 1;

	/* XXX fix segfault on "rpm -q rpm --qf='%|NAME?{a}:{%}|{NAME}\n'" */
	if (!(end && *end)) {
	    hsa->errmsg = _("} expected in expression");
	    token->u.cond.ifFormat =
		freeFormat(token->u.cond.ifFormat, token->u.cond.numIfTokens);
	    return 1;
	}

	chptr = end;
	if (*chptr != '|') {
	    hsa->errmsg = _("| expected at end of expression");
	    token->u.cond.ifFormat =
		freeFormat(token->u.cond.ifFormat, token->u.cond.numIfTokens);
	    token->u.cond.elseFormat =
		freeFormat(token->u.cond.elseFormat, token->u.cond.numElseTokens);
	    return 1;
	}
    }
	
    chptr++;

    *endPtr = chptr;

    token->type = PTOK_COND;

    (void) findTag(hsa, token, str);

    return 0;
}

/**
 * Call a header extension only once, saving results.
 * @param hsa		headerSprintf args
 * @param fn		function
 * @retval he		tag container
 * @retval ec		extension cache
 * @return		1 on success, 0 on failure
 */
static int getExtension(headerSprintfArgs hsa, headerTagTagFunction fn,
		HE_t he, HE_t ec)
	/*@modifies he, ec @*/
{
    int rc = 0;
    if (!ec->avail) {
	he = rpmheClean(he);
	rc = fn(hsa->h, he);
	*ec = *he;	/* structure copy. */
	if (!rc)
	    ec->avail = 1;
    } else
	*he = *ec;	/* structure copy. */
    he->freeData = 0;
    rc = (rc == 0);	/* XXX invert getExtension return. */
    return rc;
}

/**
 * Format a single item's value.
 * @param hsa		headerSprintf args
 * @param tag		tag
 * @param element	element index
 * @return		end of formatted string (NULL on error)
 */
/*@observer@*/ /*@null@*/
static char * formatValue(headerSprintfArgs hsa, sprintfTag tag,
		uint32_t element)
	/*@modifies hsa, tag @*/
{
    HE_t vhe = memset(alloca(sizeof(*vhe)), 0, sizeof(*vhe));
    HE_t he = &tag->he;
    char * val = NULL;
    size_t need = 0;
    char * t, * te;
    uint64_t ival = 0;
    rpmTagCount countBuf;
    int xx;

    if (!he->avail) {
	if (tag->ext)
	    xx = getExtension(hsa, tag->ext, he, hsa->ec + tag->extNum);
	else {
	    he->tag = tag->tagno;	/* XXX necessary? */
	    xx = headerGet(hsa->h, he, 0);
	}
	if (!xx) {
	    (void) rpmheClean(he);
	    he->t = RPM_STRING_TYPE;	
	    he->p.str = xstrdup("(none)");
	    he->c = 1;
	    he->freeData = 1;
	}
	he->avail = 1;
    }

    if (tag->arrayCount) {
	countBuf = he->c;
	he = rpmheClean(he);
	he->t = RPM_UINT32_TYPE;
	he->p.ui32p = &countBuf;
	he->c = 1;
	he->freeData = 0;
    }

    vhe->tag = he->tag;

    if (he->p.ptr)
    switch (he->t) {
    default:
	val = xstrdup("(unknown type)");
	need = strlen(val) + 1;
	goto exit;
	/*@notreached@*/ break;
    case RPM_I18NSTRING_TYPE:
    case RPM_STRING_ARRAY_TYPE:
	vhe->t = RPM_STRING_TYPE;
	vhe->p.str = he->p.argv[element];
	vhe->c = he->c;
	/* XXX TODO: force array representation? */
	vhe->ix = (he->c > 1 ? 0 : -1);
	break;
    case RPM_STRING_TYPE:
	vhe->p.str = he->p.str;
	vhe->t = RPM_STRING_TYPE;
	vhe->c = he->c;
	vhe->ix = -1;
	break;
    case RPM_UINT8_TYPE:
    case RPM_UINT16_TYPE:
    case RPM_UINT32_TYPE:
    case RPM_UINT64_TYPE:
	switch (he->t) {
	default:
assert(0);	/* XXX keep gcc quiet. */
	    /*@innerbreak@*/ break;
	case RPM_UINT8_TYPE:
	    ival = he->p.ui8p[element];
	    /*@innerbreak@*/ break;
	case RPM_UINT16_TYPE:
	    ival = he->p.ui16p[element];	/* XXX note unsigned. */
	    /*@innerbreak@*/ break;
	case RPM_UINT32_TYPE:
	    ival = he->p.ui32p[element];
	    /*@innerbreak@*/ break;
	case RPM_UINT64_TYPE:
	    ival = he->p.ui64p[element];
	    /*@innerbreak@*/ break;
	}
	vhe->t = RPM_UINT64_TYPE;
	vhe->p.ui64p = &ival;
	vhe->c = he->c;
	/* XXX TODO: force array representation? */
	vhe->ix = (he->c > 1 ? 0 : -1);
	break;

    case RPM_BIN_TYPE:
	vhe->t = RPM_BIN_TYPE;
	vhe->p.ptr = he->p.ptr;
	vhe->c = he->c;
	vhe->ix = -1;
	break;
    }

/*@-compmempass@*/	/* vhe->p.ui64p is stack, not owned */
    if (tag->fmt) {
	val = tag->fmt(vhe);
assert(val != NULL);
    } else {
	val = intFormat(vhe, NULL);
assert(val != NULL);
    }
/*@=compmempass@*/
    if (val)
	need = strlen(val) + 1;

exit:
    if (val && need > 0) {
	if (tag->format && *tag->format && tag->pad > 0) {
	    size_t nb;
	    nb = strlen(tag->format) + sizeof("%s");
	    t = alloca(nb);
	    (void) stpcpy( stpcpy( stpcpy(t, "%"), tag->format), "s");
	    nb = tag->pad + strlen(val) + 1;
	    te = xmalloc(nb);
/*@-formatconst@*/
	    (void) snprintf(te, nb, t, val);
/*@=formatconst@*/
	    te[nb-1] = '\0';
	    val = _free(val);
	    val = te;
	    need += tag->pad;
	}
	t = hsaReserve(hsa, need);
	te = stpcpy(t, val);
	hsa->vallen += (te - t);
	val = _free(val);
    }

    return (hsa->val + hsa->vallen);
}

/**
 * Format a single headerSprintf item.
 * @param hsa		headerSprintf args
 * @param token		item to format
 * @param element	element index
 * @return		end of formatted string (NULL on error)
 */
/*@observer@*/
static char * singleSprintf(headerSprintfArgs hsa, sprintfToken token,
		uint32_t element)
	/*@modifies hsa, token @*/
{
    char numbuf[64];	/* XXX big enuf for "Tag_0x01234567" */
    char * t, * te;
    uint32_t i, j;
    uint32_t numElements;
    sprintfToken spft;
    sprintfTag tag = NULL;
    HE_t he = NULL;
    uint32_t condNumFormats;
    size_t need;
    int xx;

    /* we assume the token and header have been validated already! */

    switch (token->type) {
    case PTOK_NONE:
	break;

    case PTOK_STRING:
	need = token->u.string.len;
	if (need == 0) break;
	t = hsaReserve(hsa, need);
	te = stpcpy(t, token->u.string.string);
	hsa->vallen += (te - t);
	break;

    case PTOK_TAG:
	t = hsa->val + hsa->vallen;
/*@-modobserver@*/	/* headerCompoundFormats not modified. */
	te = formatValue(hsa, &token->u.tag,
			(token->u.tag.justOne ? 0 : element));
/*@=modobserver@*/
	if (te == NULL)
	    return NULL;
	break;

    case PTOK_COND:
	if (token->u.cond.tag.ext
	 || headerIsEntry(hsa->h, token->u.cond.tag.tagno))
	{
	    spft = token->u.cond.ifFormat;
	    condNumFormats = token->u.cond.numIfTokens;
	} else {
	    spft = token->u.cond.elseFormat;
	    condNumFormats = token->u.cond.numElseTokens;
	}

	need = condNumFormats * 20;
	if (spft == NULL || need == 0) break;

	t = hsaReserve(hsa, need);
	for (i = 0; i < condNumFormats; i++, spft++) {
/*@-modobserver@*/	/* headerCompoundFormats not modified. */
	    te = singleSprintf(hsa, spft, element);
/*@=modobserver@*/
	    if (te == NULL)
		return NULL;
	}
	break;

    case PTOK_ARRAY:
	numElements = 0;
	spft = token->u.array.format;
	for (i = 0; i < token->u.array.numTokens; i++, spft++)
	{
	    tag = &spft->u.tag;
	    if (spft->type != PTOK_TAG || tag->arrayCount || tag->justOne)
		continue;
	    he = &tag->he;
	    if (!he->avail) {
		he->tag = tag->tagno;
		if (tag->ext)
		    xx = getExtension(hsa, tag->ext, he, hsa->ec + tag->extNum);
		else
		    xx = headerGet(hsa->h, he, 0);
		if (!xx) {
		    (void) rpmheClean(he);
		    continue;
		}
		he->avail = 1;
	    }

	    /* Check iteration arrays are same dimension (or scalar). */
	    switch (he->t) {
	    default:
		if (numElements == 0) {
		    numElements = he->c;
		    /*@switchbreak@*/ break;
		}
		if (he->c == numElements)
		    /*@switchbreak@*/ break;
		hsa->errmsg =
			_("array iterator used with different sized arrays");
		he = rpmheClean(he);
		return NULL;
		/*@notreached@*/ /*@switchbreak@*/ break;
	    case RPM_BIN_TYPE:
	    case RPM_STRING_TYPE:
		if (numElements == 0)
		    numElements = 1;
		/*@switchbreak@*/ break;
	    }
	}
	spft = token->u.array.format;

	if (numElements == 0) {
#ifdef	DYING	/* XXX lots of pugly "(none)" lines with --conflicts. */
	    need = sizeof("(none)\n") - 1;
	    t = hsaReserve(hsa, need);
	    te = stpcpy(t, "(none)\n");
	    hsa->vallen += (te - t);
#endif
	} else {
	    int isxml;
	    int isyaml;

	    need = numElements * token->u.array.numTokens;
	    if (need == 0) break;

	    tag = &spft->u.tag;

	    isxml = (spft->type == PTOK_TAG && tag->type != NULL &&
		!strcmp(tag->type, "xml"));
	    isyaml = (spft->type == PTOK_TAG && tag->type != NULL &&
		!strcmp(tag->type, "yaml"));

	    if (isxml) {
		const char * tagN;
		/* XXX display "Tag_0x01234567" for arbitrary tags. */
		if (tag->tagno & 0x40000000) {
		    (void) snprintf(numbuf, sizeof(numbuf), "Tag_0x%08x",
				(unsigned) tag->tagno);
		    numbuf[sizeof(numbuf)-1] = '\0';
		    tagN = numbuf;
		} else
		    tagN = myTagName(hsa->tags, tag->tagno, NULL);
		need = sizeof("  <rpmTag name=\"\">\n") + strlen(tagN);
		te = t = hsaReserve(hsa, need);
		te = stpcpy( stpcpy( stpcpy(te, "  <rpmTag name=\""), tagN), "\">\n");
		hsa->vallen += (te - t);
	    }
	    if (isyaml) {
		rpmTag tagT = 0;
		const char * tagN;
		/* XXX display "Tag_0x01234567" for arbitrary tags. */
		if (tag->tagno & 0x40000000) {
		    (void) snprintf(numbuf, sizeof(numbuf), "Tag_0x%08x",
				(unsigned) tag->tagno);
		    numbuf[sizeof(numbuf)-1] = '\0';
		    tagN = numbuf;
/*@-type@*/
		    tagT = numElements > 1
			?  RPM_ARRAY_RETURN_TYPE : RPM_SCALAR_RETURN_TYPE;
/*@=type@*/
		} else
		    tagN = myTagName(hsa->tags, tag->tagno, &tagT);
		need = sizeof("  :     - ") + strlen(tagN);
		te = t = hsaReserve(hsa, need);
		*te++ = ' ';
		*te++ = ' ';
		te = stpcpy(te, tagN);
		*te++ = ':';
/*@-type@*/
		*te++ = (((tagT & RPM_MASK_RETURN_TYPE) == RPM_ARRAY_RETURN_TYPE)
			? '\n' : ' ');
/*@=type@*/
		/* XXX Dirnames: in srpms need "    " indent */
/*@-type@*/
		if (((tagT & RPM_MASK_RETURN_TYPE) == RPM_ARRAY_RETURN_TYPE)
		 && numElements == 1)
/*@=type@*/
		{
		    te = stpcpy(te, "    ");
		    if (tag->tagno != 1118)
			te = stpcpy(te, "- ");
		}
		*te = '\0';
		hsa->vallen += (te - t);
	    }

	    need = numElements * token->u.array.numTokens * 10;
	    t = hsaReserve(hsa, need);
	    for (j = 0; j < numElements; j++) {
		spft = token->u.array.format;
		for (i = 0; i < token->u.array.numTokens; i++, spft++) {
/*@-modobserver@*/	/* headerCompoundFormats not modified. */
		    te = singleSprintf(hsa, spft, j);
/*@=modobserver@*/
		    if (te == NULL)
			return NULL;
		}
	    }

	    if (isxml) {
		need = sizeof("  </rpmTag>\n") - 1;
		te = t = hsaReserve(hsa, need);
		te = stpcpy(te, "  </rpmTag>\n");
		hsa->vallen += (te - t);
	    }
	    if (isyaml) {
#if 0
		need = sizeof("\n") - 1;
		te = t = hsaReserve(hsa, need);
		te = stpcpy(te, "\n");
		hsa->vallen += (te - t);
#endif
	    }

	}
	break;
    }

    return (hsa->val + hsa->vallen);
}

/**
 * Create an extension cache.
 * @param exts		headerSprintf extensions
 * @retval *necp	no. of elements (or NULL)
 * @return		new extension cache
 */
static /*@only@*/ HE_t
rpmecNew(const headerSprintfExtension exts, /*@null@*/ int * necp)
	/*@modifies *necp @*/
{
    headerSprintfExtension ext;
    HE_t ec;
    int extNum = 0;

    if (exts != NULL)
    for (ext = exts, extNum = 0; ext != NULL && ext->type != HEADER_EXT_LAST;
	ext = (ext->type == HEADER_EXT_MORE ? *ext->u.more : ext+1), extNum++)
    {
	;
    }
    if (necp)
	*necp = extNum;
    ec = xcalloc(extNum+1, sizeof(*ec));	/* XXX +1 unnecessary */
    return ec;
}

/**
 * Destroy an extension cache.
 * @param exts		headerSprintf extensions
 * @param ec		extension cache
 * @return		NULL always
 */
static /*@null@*/ HE_t
rpmecFree(const headerSprintfExtension exts, /*@only@*/ HE_t ec)
	/*@modifies ec @*/
{
    headerSprintfExtension ext;
    int extNum;

    for (ext = exts, extNum = 0; ext != NULL && ext->type != HEADER_EXT_LAST;
	ext = (ext->type == HEADER_EXT_MORE ? *ext->u.more : ext+1), extNum++)
    {
	(void) rpmheClean(&ec[extNum]);
    }

    ec = _free(ec);
    return NULL;
}

char * headerSprintf(Header h, const char * fmt,
		headerTagTableEntry tags,
		headerSprintfExtension exts,
		errmsg_t * errmsg)
{
    headerSprintfArgs hsa = memset(alloca(sizeof(*hsa)), 0, sizeof(*hsa));
    sprintfToken nextfmt;
    sprintfTag tag;
    char * t, * te;
    int isxml;
    int isyaml;
    int need;

/*@-modfilesys@*/
if (_hdr_debug)
fprintf(stderr, "==> headerSprintf(%p, \"%s\", %p, %p, %p)\n", h, fmt, tags, exts, errmsg);
/*@=modfilesys@*/

    /* Set some reasonable defaults */
    if (tags == NULL)
	tags = rpmTagTable;
    /* XXX this loses the extensions in lib/formats.c. */
    if (exts == NULL)
	exts = headerCompoundFormats;
 
    hsa->h = headerLink(h);
    hsa->fmt = xstrdup(fmt);
/*@-assignexpose -dependenttrans@*/
    hsa->exts = exts;
    hsa->tags = tags;
/*@=assignexpose =dependenttrans@*/
    hsa->errmsg = NULL;

    if (parseFormat(hsa, hsa->fmt, &hsa->format, &hsa->numTokens, NULL, PARSER_BEGIN))
	goto exit;

    hsa->nec = 0;
    hsa->ec = rpmecNew(hsa->exts, &hsa->nec);
    hsa->val = xstrdup("");

    tag =
	(hsa->format->type == PTOK_TAG
	    ? &hsa->format->u.tag :
	(hsa->format->type == PTOK_ARRAY
	    ? &hsa->format->u.array.format->u.tag :
	NULL));
    isxml = (tag != NULL && tag->tagno == -2 && tag->type != NULL && !strcmp(tag->type, "xml"));
    isyaml = (tag != NULL && tag->tagno == -2 && tag->type != NULL && !strcmp(tag->type, "yaml"));

    if (isxml) {
	need = sizeof("<rpmHeader>\n") - 1;
	t = hsaReserve(hsa, need);
	te = stpcpy(t, "<rpmHeader>\n");
	hsa->vallen += (te - t);
    }
    if (isyaml) {
	need = sizeof("- !!omap\n") - 1;
	t = hsaReserve(hsa, need);
	te = stpcpy(t, "- !!omap\n");
	hsa->vallen += (te - t);
    }

    hsa = hsaInit(hsa);
    while ((nextfmt = hsaNext(hsa)) != NULL) {
/*@-modobserver@*/	/* headerCompoundFormats not modified. */
	te = singleSprintf(hsa, nextfmt, 0);
/*@=modobserver@*/
	if (te == NULL) {
	    hsa->val = _free(hsa->val);
	    break;
	}
    }
    hsa = hsaFini(hsa);

    if (isxml) {
	need = sizeof("</rpmHeader>\n") - 1;
	t = hsaReserve(hsa, need);
	te = stpcpy(t, "</rpmHeader>\n");
	hsa->vallen += (te - t);
    }
    if (isyaml) {
	need = sizeof("\n") - 1;
	t = hsaReserve(hsa, need);
	te = stpcpy(t, "\n");
	hsa->vallen += (te - t);
    }

    if (hsa->val != NULL && hsa->vallen < hsa->alloced)
	hsa->val = xrealloc(hsa->val, hsa->vallen+1);	

    hsa->ec = rpmecFree(hsa->exts, hsa->ec);
    hsa->nec = 0;
    hsa->format = freeFormat(hsa->format, hsa->numTokens);

exit:
/*@-dependenttrans -observertrans @*/
    if (errmsg)
	*errmsg = hsa->errmsg;
/*@=dependenttrans =observertrans @*/
    hsa->h = headerFree(hsa->h);
    hsa->fmt = _free(hsa->fmt);
/*@-retexpose@*/
    return hsa->val;
/*@=retexpose@*/
}
