/*
 * $Id: vscan-antivir.c,v 1.1.2.4 2005/02/23 02:49:22 reniar Exp $
 *
 * virusscanning VFS module for samba.  Log concerning files via syslog
 * facility and block access using the H+BEDV AntiVir scanner.
 *
 * Copyright (C) Rainer Link, 2001-2005
 *               OpenAntiVirus.org <rainer@openantivirus.org>
 *               Dariusz Markowicz <dariusz@markowicz.net>, 2003
 * Copyright (C) Stefan (metze) Metzmacher, 2003
 *               <metze@metzemix.de>
 * Copyright (C) H+BEDV Datentechnik GmbH, 2004
 *               <unix_support@antivir.de>
 *
 * based on the audit VFS module by
 * Copyright (C) Tim Potter, 1999-2000
 * Copyright (C) Alexander Bokovoy, 2002
 *
 *
 * Credits to
 * - Dave Collier-Brown for his VFS tutorial (http://www.geocities.com/orville_torpid/papers/vfs_tutorial.html)
 * - REYNAUD Jean-Samuel for helping me to solve some general Samba VFS issues at the first place
 * - Simon Harrison for his solution without Samba VFS (http://www.smh.uklinux.net/linux/sophos.html)
 * - the whole Samba Team :)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "vscan-global.h"
#include "vscan-antivir.h"

#include "vscan-vfs.h"

#define VSCAN_MODULE_STR "vscan-antivir"

vscan_config_struct vscan_config; /* contains the vscan module configuration */

BOOL verbose_file_logging;  	/* log ever file access */
BOOL send_warning_message;	/* send a warning message using the windows
				   messenger service? */

fstring	antivir_program_name;	/* name of antivir executable */

BOOL antivir_arch_scan_enable;  /* archive related settings for the AntiVir scanner */
int antivir_arch_max_ratio;
ssize_t antivir_arch_max_size;
int antivir_arch_max_recursion;

BOOL antivir_detect_dialer;       /* non virus type alerts related settings for the AntiVir scanner */
BOOL antivir_detect_game;
BOOL antivir_detect_joke;
BOOL antivir_detect_pms;
BOOL antivir_detect_spy;
BOOL antivir_detect_alltypes;

/* module version */
static const char module_id[]=VSCAN_MODULE_STR" "SAMBA_VSCAN_VERSION_STR;


static BOOL do_parameter(const char *param, const char *value)
{
	if ( do_common_parameter(&vscan_config, param, value) == False ) {
                /* parse VFS module specific configuration values */
		if ( StrCaseCmp("antivir program name", param) == 0) {
			fstrcpy(antivir_program_name, value);
			DEBUG(3, ("antivir program name is %s\n", antivir_program_name));
		} else if ( StrCaseCmp("antivir scan in archive", param) == 0) {
			set_boolean(&antivir_arch_scan_enable, value);
			DEBUG(3, ("antivir scan in archive is %d\n", antivir_arch_scan_enable));
		} else if ( StrCaseCmp("antivir max ratio in archive", param) == 0) {
			antivir_arch_max_ratio = atoi(value);
			DEBUG(3, ("antivir max ratio in archive is %d\n", antivir_arch_max_ratio));
		} else if ( StrCaseCmp("antivir max archived file size", param) == 0) {
			antivir_arch_max_size = atoll(value);
			DEBUG(3, ("antivir max archived file size is %lld\n", (long long)antivir_arch_max_size));
		} else if ( StrCaseCmp("antivir max recursion level", param) == 0) {
			antivir_arch_max_recursion = atoi(value);
			DEBUG(3, ("antivir max recursion level is %d\n", antivir_arch_max_recursion));
		} else if ( StrCaseCmp("antivir detect dialer", param) == 0) {
			set_boolean(&antivir_detect_dialer, value);
			DEBUG(3, ("antivir detect dialer is %d\n", antivir_detect_dialer));
		} else if ( StrCaseCmp("antivir detect game", param) == 0) {
			set_boolean(&antivir_detect_game, value);
			DEBUG(3, ("antivir detect game is %d\n", antivir_detect_game));
		} else if ( StrCaseCmp("antivir detect joke", param) == 0) {
			set_boolean(&antivir_detect_joke, value);
			DEBUG(3, ("antivir detect joke is %d\n", antivir_detect_joke));
		} else if ( StrCaseCmp("antivir detect pms", param) == 0) {
			set_boolean(&antivir_detect_pms, value);
			DEBUG(3, ("antivir detect pms is %d\n", antivir_detect_pms));
		} else if ( StrCaseCmp("antivir detect spy", param) == 0) {
			set_boolean(&antivir_detect_spy, value);
			DEBUG(3, ("antivir detect spy is %d\n", antivir_detect_spy));
		} else if ( StrCaseCmp("antivir detect alltypes", param) == 0) {
			set_boolean(&antivir_detect_alltypes, value);
			DEBUG(3, ("antivir detect alltypes is %d\n", antivir_detect_alltypes));
		} else
			DEBUG(3, ("unknown parameter: %s\n", param));
	}

	return True;
}

static BOOL do_section(const char *section)
{
	/* simply return true, there's only one section :-) */
	return True;
}


/* Implementation of vfs_ops.  */

#if (SMB_VFS_INTERFACE_VERSION >= 6)
static int vscan_connect(vfs_handle_struct *handle, connection_struct *conn, const char *svc, const char *user)
#else
static int vscan_connect(struct connection_struct *conn, PROTOTYPE_CONST char *svc, PROTOTYPE_CONST char *user)
#endif
{
        fstring config_file;            /* location of config file, either
                                           PARAMCONF or as set via vfs options
                                        */

	#if (SAMBA_VERSION_MAJOR==2 && SAMBA_VERSION_RELEASE>=4) || SAMBA_VERSION_MAJOR==3
	 #if !(SMB_VFS_INTERFACE_VERSION >= 6)
	  pstring opts_str;
	  PROTOTYPE_CONST char *p;
	 #endif
	#endif
	int retval;

#if (SMB_VFS_INTERFACE_VERSION >= 6)
	vscan_syslog("samba-vscan (%s) connected (Samba 3.0), (c) by Rainer Link, OpenAntiVirus.org", module_id);
#endif

	/* set default value for configuration files */
	fstrcpy(config_file, PARAMCONF);

        /* set default values */
        set_common_default_settings(&vscan_config);

	/* name of AntiVir executable file */
	fstrcpy(antivir_program_name, VSCAN_ANTIVIR_PROGRAM_NAME);

	/* AntiVir archive related settings */
	antivir_arch_scan_enable = VSCAN_AVARCH_ENABLED;
	antivir_arch_max_ratio = VSCAN_AVARCH_MAXRATIO;
	antivir_arch_max_size = VSCAN_AVARCH_MAXFILESIZE;
	antivir_arch_max_recursion = VSCAN_AVARCH_MAXRECLEVEL;

	/* AntiVir non virus alerts related settings */
	antivir_detect_dialer = VSCAN_AVDETECT_DIALER;
	antivir_detect_game = VSCAN_AVDETECT_GAME;
	antivir_detect_joke = VSCAN_AVDETECT_JOKE;
	antivir_detect_pms = VSCAN_AVDETECT_PMS;
	antivir_detect_spy = VSCAN_AVDETECT_SPY;
	antivir_detect_alltypes = VSCAN_AVDETECT_ALLTYPES;

	vscan_syslog("INFO: connect to service %s by user %s",
	       svc, user);

	#if (SAMBA_VERSION_MAJOR==2 && SAMBA_VERSION_RELEASE>=4) || SAMBA_VERSION_MAJOR==3

          fstrcpy(config_file, get_configuration_file(conn, VSCAN_MODULE_STR, PARAMCONF));
          DEBUG(3, ("configuration file is: %s\n", config_file));

          retval = pm_process(config_file, do_section, do_parameter);
          DEBUG(10, ("pm_process returned %d\n", retval));

          /* FIXME: this is lame! */
          verbose_file_logging = vscan_config.common.verbose_file_logging;
          send_warning_message = vscan_config.common.send_warning_message;

	  if (!retval) vscan_syslog("ERROR: could not parse configuration file '%s'. File not found or not read-able. Using compiled-in defaults", config_file);
	#endif

	/* initialise lrufiles list */
	DEBUG(5, ("init lrufiles list\n"));
	lrufiles_init(vscan_config.common.max_lrufiles, vscan_config.common.lrufiles_invalidate_time);

	/* initialise filetype */
	DEBUG(5, ("init file type\n"));
	filetype_init(0, vscan_config.common.exclude_file_types);

	/* tell core we have a(nother) session, ignore return code */
	(void)vscan_antivir_connect();

	#if (SMB_VFS_INTERFACE_VERSION >= 6)
	 return SMB_VFS_NEXT_CONNECT(handle, conn, svc, user);
	#else
	 return default_vfs_ops.connect(conn, svc, user);
	#endif

}

#if (SMB_VFS_INTERFACE_VERSION >= 6)
static void vscan_disconnect(vfs_handle_struct *handle, connection_struct *conn)
#else/* Samba 3.0 alphaX */
static void vscan_disconnect(struct connection_struct *conn)
#endif
{

	/* tell core the (one) session has gone */
	vscan_antivir_disconnect();

	vscan_syslog("INFO: disconnected");

	lrufiles_destroy_all();
	filetype_close();

#if (SMB_VFS_INTERFACE_VERSION >= 6)
	SMB_VFS_NEXT_DISCONNECT(handle, conn);
#else
	default_vfs_ops.disconnect(conn);
#endif
}


#if (SMB_VFS_INTERFACE_VERSION >= 6)
static int vscan_open(vfs_handle_struct *handle, connection_struct *conn, const char *fname, int flags, mode_t mode)
#else
static int vscan_open(struct connection_struct *conn, PROTOTYPE_CONST char *fname, int flags, mode_t mode)
#endif
{
	int retval, must_be_checked;
	SMB_STRUCT_STAT stat_buf;
	int sockfd;
	pstring filepath;
	char client_ip[CLIENT_IP_SIZE];
	int rc;

	/* Assemble complete file path */
	pstrcpy(filepath, conn->connectpath);
	pstrcat(filepath, "/");
	pstrcat(filepath, fname);

	/* scan files while opening? */
	if ( !vscan_config.common.scan_on_open ) {
		DEBUG(3, ("samba-vscan - open: File '%s' not scanned as scan_on_open is not set\n", fname));
#if (SMB_VFS_INTERFACE_VERSION >= 6)
		return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
		return default_vfs_ops.open(conn, fname, flags, mode);
#endif
	}

#if (SMB_VFS_INTERFACE_VERSION >= 6)
	if ( (SMB_VFS_NEXT_STAT(handle, conn, fname, &stat_buf)) != 0 )    /* an error occured */
		return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
	if ( (default_vfs_ops.stat(conn, fname, &stat_buf)) != 0 )    /* an error occured */
		return default_vfs_ops.open(conn, fname, flags, mode);
#endif
	else if ( S_ISDIR(stat_buf.st_mode) ) 	/* is it a directory? */
#if (SMB_VFS_INTERFACE_VERSION >= 6)
		return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
		return default_vfs_ops.open(conn, fname, flags, mode);
#endif
	else if ( ( stat_buf.st_size > vscan_config.common.max_size ) && ( vscan_config.common.max_size > 0 ) ) /* file is too large */
		vscan_syslog("INFO: File %s is larger than specified maximum file size! Not scanned!", fname);
	else if ( stat_buf.st_size == 0 ) /* do not scan empty files */
#if (SMB_VFS_INTERFACE_VERSION >= 6)
		return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
		return default_vfs_ops.open(conn, fname, flags, mode);
#endif
	else if ( filetype_skipscan(filepath) == VSCAN_FT_SKIP_SCAN ) {
		if ( vscan_config.common.verbose_file_logging )
			vscan_syslog("File '%s' not scanned as file type is on exclude list", filepath);
#if (SMB_VFS_INTERFACE_VERSION >= 6)
		return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
		return default_vfs_ops.open(conn, fname, flags, mode);
#endif

	} else
	{
		/* prepare file scan, (re)open connection to the scanner */
		sockfd = vscan_antivir_init();
		if ( sockfd == -1 && vscan_config.common.deny_access_on_error ) {
			/* an error occured - can not communicate to daemon - deny access */
			vscan_syslog("ERROR: can not communicate to daemon - access denied");
			errno = EACCES;
			return -1;
		} else if ( sockfd >= 0 ) {
			/* we now have a scanner */
			safe_strcpy(client_ip, conn->client_address, CLIENT_IP_SIZE -1);

			/* must file actually be scanned? */
			must_be_checked = lrufiles_must_be_checked(filepath, stat_buf.st_mtime);
			if ( must_be_checked == VSCAN_LRU_DENY_ACCESS ) {
				/* file has already been checked and marked as containing an alert */
				/* deny access */
				if ( vscan_config.common.verbose_file_logging )
					vscan_syslog("File '%s' has already been scanned and marked as containing an alert. Not scanned any more. Access denied", filepath);

				/* postprocess file scan */
				vscan_antivir_end(sockfd);

				/* deny access */
				errno = EACCES;
				return -1;
			} else if ( must_be_checked == VSCAN_LRU_GRANT_ACCESS )  {
				/* file has already been checked, not marked as containing an alert and not modified */
				if ( vscan_config.common.verbose_file_logging )
					vscan_syslog("File '%s' has already been scanned, not marked as containing an alert and not modified. Not scanned anymore. Access granted", filepath);

				/* postprocess file scan */
				vscan_antivir_end(sockfd);

				/* grant access */
#if (SMB_VFS_INTERFACE_VERSION >= 6)
				return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
				return default_vfs_ops.open(conn, fname, flags, mode);
#endif
			}
			/* ok, we must check the file */

			/* scan file */
			retval = vscan_antivir_scanfile(sockfd, filepath, client_ip);
			if ( retval == VSCAN_SCAN_MINOR_ERROR && vscan_config.common.deny_access_on_minor_error ) {
				/* a minor error occured - deny access */
				vscan_syslog("ERROR: daemon failed with a minor error - access to file %s denied", fname);
				vscan_antivir_end(sockfd);

				/* to be safe, remove file from lrufiles */
				lrufiles_delete(filepath);

				/* deny access */
				errno = EACCES;
				return -1;
			} else if ( retval == VSCAN_SCAN_ERROR && vscan_config.common.deny_access_on_error ) {
			/* an error occured - can not communicate to daemon - deny access */
				vscan_syslog("ERROR: can not communicate to daemon - access to file %s denied", fname);
				/* to be safe, remove file from lrufiles */
				lrufiles_delete(filepath);

				/* deny access */

				errno = EACCES;
				return -1;
			} else if ( retval == VSCAN_SCAN_VIRUS_FOUND ) {
				/* an alert was found */
				vscan_antivir_end(sockfd);
				/* do action ... */

#if (SMB_VFS_INTERFACE_VERSION >= 6)
				rc = vscan_do_infected_file_action(handle, conn, filepath, vscan_config.common.quarantine_dir, vscan_config.common.quarantine_prefix, vscan_config.common.infected_file_action);
#else
				rc = vscan_do_infected_file_action(&default_vfs_ops, conn, filepath, vscan_config.common.quarantine_dir, vscan_config.common.quarantine_prefix, vscan_config.common.infected_file_action);
#endif

				/* add/update file. mark file as containing an alert! */
				lrufiles_add(filepath, stat_buf.st_mtime, True);

				/* alert found, deny acces */
				errno = EACCES;
				return -1;
			} else if ( retval == VSCAN_SCAN_OK ) {
				/* file is clean, add to lrufiles */
				lrufiles_add(filepath, stat_buf.st_mtime, False);
			}
		}

		/* scan done, post process file scan */
		vscan_antivir_end(sockfd);
	}
#if (SMB_VFS_INTERFACE_VERSION >= 6)
	return SMB_VFS_NEXT_OPEN(handle, conn, fname, flags, mode);
#else
	return default_vfs_ops.open(conn, fname, flags, mode);
#endif
}

#if (SMB_VFS_INTERFACE_VERSION >= 6)
static int vscan_close(vfs_handle_struct *handle, files_struct *fsp, int fd)
#else
static int vscan_close(struct files_struct *fsp, int fd)
#endif
{
	pstring filepath;
	int retval, rv, rc;
	int sockfd;
	char client_ip[CLIENT_IP_SIZE];

	/* First close the file */
#if (SMB_VFS_INTERFACE_VERSION >= 6)
	retval = SMB_VFS_NEXT_CLOSE(handle, fsp, fd);
#else
	retval = default_vfs_ops.close(fsp, fd);
#endif

	if ( !vscan_config.common.scan_on_close ) {
		DEBUG(3, ("samba-vscan - close: File '%s' not scanned as scan_on_close is not set\n", fsp->fsp_name));
		return retval;
	}


	/* get the file name */
	pstrcpy(filepath, fsp->conn->connectpath);
	pstrcat(filepath, "/");
	pstrcat(filepath, fsp->fsp_name);

	/* Don't scan directorys */
	if ( fsp->is_directory )
	    return retval;


	if ( !fsp->modified ) {
		if ( vscan_config.common.verbose_file_logging )
			vscan_syslog("INFO: file %s was not modified - not scanned", filepath);

		return retval;
	}

	/* don't scan files which are in the list of exclude file types */
	if ( filetype_skipscan(filepath) == VSCAN_FT_SKIP_SCAN ) {
		if ( vscan_config.common.verbose_file_logging )
			vscan_syslog("File '%s' not scanned as file type is on exclude list", filepath);
		return retval;
	}


	sockfd = vscan_antivir_init();
	if ( sockfd >= 0 ) {
		safe_strcpy(client_ip, fsp->conn->client_address, CLIENT_IP_SIZE -1);
		/* scan only file, do nothing */
		rv = vscan_antivir_scanfile(sockfd, filepath, client_ip);
		vscan_antivir_end(sockfd);
		if ( rv == VSCAN_SCAN_VIRUS_FOUND ) {
			/* alert was found */
#if (SMB_VFS_INTERFACE_VERSION >= 6)
			rc = vscan_do_infected_file_action(handle, fsp->conn, filepath, vscan_config.common.quarantine_dir, vscan_config.common.quarantine_prefix, vscan_config.common.infected_file_action);
#else
			rc = vscan_do_infected_file_action(&default_vfs_ops, fsp->conn, filepath, vscan_config.common.quarantine_dir, vscan_config.common.quarantine_prefix, vscan_config.common.infected_file_action);
#endif
		}

	}
	return retval;
}


#if (SMB_VFS_INTERFACE_VERSION >= 6)
/* Samba 3.0 */
NTSTATUS init_module(void)
{
	NTSTATUS ret;

	ret = smb_register_vfs(SMB_VFS_INTERFACE_VERSION, VSCAN_MODULE_STR, vscan_ops);
	openlog("smbd_"VSCAN_MODULE_STR, LOG_PID, SYSLOG_FACILITY);
	vscan_syslog("samba-vscan (%s) registered (Samba 3.0), (c) by Rainer Link, OpenAntiVirus.org", module_id);
	DEBUG(5,("samba-vscan (%s) registered (Samba 3.0), (c) by Rainer Link, OpenAntiVirus.org\n", module_id));

	return ret;
}
#else
/* VFS initialisation function.  Return initialised vfs_ops structure
   back to SAMBA. */
#if SAMBA_VERSION_MAJOR==3
 /* Samba 3.0 alphaX */
 vfs_op_tuple *vfs_init(int *vfs_version, struct vfs_ops *def_vfs_ops,
			struct smb_vfs_handle_struct *vfs_handle)
#else
 /* Samba 2.2.x */
 #if SAMBA_VERSION_RELEASE>=4
  /* Samba 2.2.4 */
  struct vfs_ops *vfs_init(int *vfs_version, struct vfs_ops *def_vfs_ops)
 #elif SAMBA_VERSION_RELEASE==2
  /* Samba 2.2.2 / Samba 2.2.3 !!! */
  struct vfs_ops *vfs_init(int* Version, struct vfs_ops *ops)
 #elif SAMBA_VERSION_RELEASE==1
  /* Samba 2.2.1 */
  struct vfs_ops *vfs_module_init(int *vfs_version)
 #else
  /* Samba 2.2.0 */
  struct vfs_ops *vfs_init(int *vfs_version)
 #endif
#endif
{
	#if SAMBA_VERSION_MAJOR!=3
 	 #if SAMBA_VERSION_RELEASE>=4
	  /* Samba 2.2.4 */
	  struct vfs_ops tmp_ops;
	 #endif
	#endif

	openlog("smbd_"VSCAN_MODULE_STR, LOG_PID, SYSLOG_FACILITY);

	#if SAMBA_VERSION_MAJOR==3
	 /* Samba 3.0 alphaX */
	 *vfs_version = SMB_VFS_INTERFACE_VERSION;
	 vscan_syslog("samba-vscan (%s) loaded (Samba 3.x), (c) by Rainer Link, OpenAntiVirus.org", module_id);
	#else
	 /* Samba 2.2.x */
	 #if SAMBA_VERSION_RELEASE>=4
	  /* Samba 2.2.4 */
	  *vfs_version = SMB_VFS_INTERFACE_VERSION;
	  vscan_syslog("samba-vscan (%s) loaded (Samba >=2.2.4), (c) by Rainer Link, OpenAntiVirus.org", module_id);
	 #elif SAMBA_VERSION_RELEASE==2
	  /* Samba 2.2.2 / Samba 2.2.3 !!! */
	  *Version = SMB_VFS_INTERFACE_VERSION;
	  vscan_syslog("samba-vscan (%s) loaded (Samba 2.2.2/2.2.3), (c) by Rainer Link, OpenAntiVirus.org", module_id);
	 #else
	  /* Samba 2.2.1 / Samba 2.2.0 */
	  *vfs_version = SMB_VFS_INTERFACE_VERSION;
	  vscan_syslog("samba-vscan (%s) loaded (Samba 2.2.0/2.2.1), (c) by Rainer Link, OpenAntiVirus.org",
	       module_id);
	 #endif
	#endif


	#if SAMBA_VERSION_MAJOR==3
	 /* Samba 3.0 alphaX */
	 DEBUG(3, ("Initialising default vfs hooks\n"));
	 memcpy(&default_vfs_ops, def_vfs_ops, sizeof(struct vfs_ops));

	 /* Remember vfs_handle for further allocation and referencing of
	    private information in vfs_handle->data
	 */
	 vscan_handle = vfs_handle;
	 return vscan_ops;
	#else
	 /* Samba 2.2.x */
	 #if SAMBA_VERSION_RELEASE>=4
	  /* Samba 2.2.4 */

	  *vfs_version = SMB_VFS_INTERFACE_VERSION;
	  memcpy(&tmp_ops, def_vfs_ops, sizeof(struct vfs_ops));
	  tmp_ops.connect = vscan_connect;
	  tmp_ops.disconnect = vscan_disconnect;
	  tmp_ops.open = vscan_open;
	  tmp_ops.close = vscan_close;
	  memcpy(&vscan_ops, &tmp_ops, sizeof(struct vfs_ops));
	  return(&vscan_ops);

	 #else
	  /* Samba 2.2.3-2.2.0 */
	  return(&vscan_ops);
	 #endif
	#endif
}


#if SAMBA_VERSION_MAJOR==3
/* VFS finalization function */
void vfs_done(connection_struct *conn)
{
	DEBUG(3, ("Finalizing default vfs hooks\n"));
}
#endif

#endif /* #if (SMB_VFS_INTERFACE_VERSION >= 6) */
