/*
 * $Id: vscan-antivir_core.c,v 1.1.2.3 2005/01/18 20:03:14 reniar Exp $
 *
 * Core Interface for the H+BEDV AntiVir Scanner
 *
 * Copyright (C) Rainer Link, 2001-2005
 *               OpenAntiVirus.org <rainer@openantivirus.org>
 *               Dariusz Markowicz <dariusz@markowicz.net>, 2003
 *               H+BEDV Datentechnik GmbH <unix_support@antivir.de>, 2004
 *
 * This software is licensed under the GNU General Public License (GPL)
 * See: http://www.gnu.org/copyleft/gpl.html
 *
 */

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* use select(2) or poll(2) to drain antivir(1)'s greeting after process
 * startup?  we default to poll(2) since we cannot guarantee the file
 * descriptor to be <= 1023 */
#undef USE_SELECT
#if defined USE_SELECT
  #include <sys/select.h>
#else
  #include <sys/poll.h>
#endif

#include "vscan-global.h"
#include "vscan-antivir_core.h"

/* used when scanning files */
extern BOOL verbose_file_logging;
extern BOOL send_warning_message;

/* used to startup the scanner process */
extern fstring antivir_program_name;
extern BOOL antivir_arch_scan_enable;
extern int antivir_arch_max_ratio;
extern ssize_t antivir_arch_max_size;
extern int antivir_arch_max_recursion;
extern int antivir_detect_dialer;
extern int antivir_detect_game;
extern int antivir_detect_joke;
extern int antivir_detect_pms;
extern int antivir_detect_spy;
extern int antivir_detect_alltypes;

/* used to manage the scanner process */
static int conn_count = 0;
static int antivir_fd_write = -1;
static int antivir_fd_read = -1;
static FILE *antivir_file_read = NULL;
static pid_t antivir_scanner_pid = 0;

static int connect_to_scanner(void);
static void disconnect_from_scanner(void);

/* user connects to a share (might startup scanner daemon) */
int vscan_antivir_connect(void) {

	/* only act on the very first connection for this smbd process */
	conn_count++;
	if (conn_count > 1)
		return(0);

	if (connect_to_scanner() < 0)
		return(-1);

	return(0);
}

/* user disconnects from a share (might shutdown scanner daemon) */
void vscan_antivir_disconnect(void) {
	conn_count--;
	if (conn_count == 0)
		disconnect_from_scanner();
}

/*
 * introduces a scan request, (re)connect to the scanner process;
 * returns -1 on error or 0 on success
 */
int vscan_antivir_init(void)
{
	int rc;

	rc = connect_to_scanner();

	return(rc);
}

/*
 * If an alert was found, logs the filename and concern into syslog
 */
void vscan_antivir_log_alert(char *concerning_file, char *results, char *client_ip) {
	vscan_syslog_alert("ALERT - Scan result: found '%s' in file '%s', client: '%s'", results, concerning_file, client_ip);
	if ( send_warning_message )
		vscan_send_warning_message(concerning_file, results, client_ip);
}

/*
 * avoid non printables in the scanner communication protocol
 * XXX: does the condition need further splitting on EBCDIC machines?
 * but OTOH isprint(3) does not really match what we would need here
 */
static int needsescape(const char c) {
	return(((c <= ' ') || (c > '~') || (c == '\\')) ? 1 : 0);
#if 0
	/* will this do better / more portable? */
	if (c == '\\')
		return(1);
	if (isspace((int)c))
		return(1);
	if (isprint((int)c))
		return(0);
	return(1);
#endif
}

/*
 * Scans a file (NOT a directory) for concerns
 *
 * Gets the vscan_antivir_init() return value (not really needed), the name of
 * the file to scan and an IP to send notification to
 *
 * Returns
 * -2 on minor error,
 * -1 on error,
 *  0 if nothing was found,
 *  1 if a concern was found
 */
int vscan_antivir_scanfile(int sockfd, char *scan_file, char *client_ip) {
	char *request = NULL;
	size_t len;
	int bEsc;
	char escbuff[5];	/* "\\xNN" plus NUL */
	char buff[1024];
	char *p1, *p2;

	/* (re)connect to the scanner process */
	if (connect_to_scanner() < 0)
		return VSCAN_SCAN_ERROR;

	/* prepare antivir command (SCAN request) */
	/* +1 is for '\0' termination --metze */
	/* +3 for every special (non printable) char (escape) */
	len = strlen("SCAN:") + strlen(scan_file) + strlen("\n") + 1;
	bEsc = 0;
	for (p1 = scan_file; (p1 != NULL) && (*p1 != '\0'); p1++) {
		if (needsescape(*p1)) {
			len += 3;
			bEsc++;
		}
	}
	request = (char *)malloc(len);
	if ( request == NULL ) {
		vscan_syslog("ERROR: can not allocate memory");
		return VSCAN_SCAN_ERROR; /* error allocating memory */
	}
	safe_strcpy(request, "SCAN:", len-1);
	if (! bEsc) {
		/* simple concatenation of the complete filename */
		safe_strcat(request, scan_file, len-1);
	} else {
		/* a more expensive approach to avoid non printable chars in the protocol */
		/* XXX TODO optimization (memcpy() with a fork?) */
		for (p1 = scan_file; (p1 != NULL) && (*p1 != '\0'); p1++) {
			if (needsescape(*p1)) {
				snprintf(escbuff, sizeof(escbuff), "\\x%02X", (*((unsigned char *)p1)) & 0xFF);
			} else {
				/* there must be an easier way to accomplish this :) */
				snprintf(escbuff, sizeof(escbuff), "%c", *p1);
			}
			safe_strcat(request, escbuff, len - 1);
		}
	}
	snprintf(escbuff, sizeof(escbuff), "\n");
	safe_strcat(request, escbuff, len - 1);

	if (verbose_file_logging)
		vscan_syslog("INFO: Scanning file : '%s'", scan_file);

	/* send request */
	len = write(antivir_fd_write, request, strlen(request));
	if (len != strlen(request)) {
		free(request);
		vscan_syslog("ERROR: can not write to the antivir socket");
		return VSCAN_SCAN_ERROR; /* error writing to the antivir socket */
	}
	free(request);
	request = NULL;

	/* get response (in a loop, we might have to skip some lines) */
	do {
		memset(buff, 0, sizeof(buff));
		p1 = fgets((char *)&buff, sizeof(buff), antivir_file_read);
		if (p1 == NULL)
			break;
		/* remove trailing spaces */
		p1 = buff + strlen(buff);
		while ((p1 > buff) && (isspace((int)*(p1-1)))) {
			p1--;
			*p1 = '\0';
		}

		/* an updater might have restarted our scanner process */
		if (strncmp(buff, "Running in DEMO mode.", strlen("Running in DEMO mode.")) == 0) {
			/* log a warning? */
			continue;
		}
		if (strncmp(buff, "BANNER ", strlen("BANNER ")) == 0) {
			continue;
		}

		/*
		 * we got a response to our request;
		 * do NOT close the scanner, reuse it for multiple scans
		 */

		/* split response into keyword and (optional) parameter */
		p1 = buff;
		p2 = strchr(p1, ':');
		if (p2 == NULL)
			p2 = p1 + strlen(p1);
		if (*p2 == ':') {
			*p2 = '\0';
			p2++;
		}
		while (isspace((int)*p2))
			p2++;

		/* found an alert */
		if (strcmp(p1, "FOUND") == 0) {
			vscan_antivir_log_alert(scan_file, p2, client_ip);
			return VSCAN_SCAN_VIRUS_FOUND;
		/* file good */
		} else if (strcmp(p1, "OK") == 0) {
			if (verbose_file_logging) {
				vscan_syslog("INFO: file %s is clean", scan_file);
			}
			return VSCAN_SCAN_OK;
		/* no explicit check for "ERROR", caught here, too */
		} else {
			if (verbose_file_logging) {
				vscan_syslog("ERROR: file %s not found, not readable or an error occured", scan_file);
			}
			/* FIXME: should this really reported as minor error? */
			return VSCAN_SCAN_MINOR_ERROR;
		}
	} while (0);

	/* read error or EOF, close the scanner process */
	disconnect_from_scanner();

	vscan_syslog("ERROR: can not get result from antivir");
	return VSCAN_SCAN_ERROR;
}


/*
 * ends a scan request
 *
 * gets the vscan_antivir_init() return value (not really needed)
 */
void vscan_antivir_end(int sockfd) {

	/* leave the connection open, we will reuse it for the
	 * next file to scan and close it on (last) disconnect
	 */
	/* EMPTY */
}

/*
 * connect or reconnect to a scanner process;
 * returns -1 on errors, non negative codes otherwise
 */
static int connect_to_scanner(void) {
	int execargc;
	char *execargv[16];
	int rc;
	int fdreq[2];
	int fdrsp[2];
	pid_t pid;
	char buff[256];
	struct timeval tv;
	int fd;

	/*
	 * check if the scanner process is alive (handles dangling
	 * connections); we do not strictly need this but I want to
	 * avoid SIGPIPEs (we're in a shared object, attached to an
	 * smbd(8) process -- would SIG_IGN be appropriate? read(2)
	 * and fgets(3) would return errors, anyway ...)
	 */
	if ((antivir_scanner_pid != 0) && (kill(antivir_scanner_pid, 0) != 0))
		disconnect_from_scanner();

	/* shortcut for "already (completely) open?" */
	if ((antivir_fd_write != -1) && (antivir_fd_read != -1) &&
	    (antivir_file_read != NULL) && (antivir_scanner_pid != 0))
		return(0);

	/* one of the parameters might be wrong, close what is left open (if any) */
	disconnect_from_scanner();

	/* open up a new scanner process ... */

#define DEL_CMD_WORDS do {			 \
	while (execargc > 0) {			 \
		execargc--;			 \
		free(execargv[execargc]);	 \
	}					 \
} while (0)

#define ADD_CMD_WORD(word) do {						 \
	char *copy;							 \
									 \
	if (word != NULL) {						 \
		copy = strdup(word);					 \
		if (copy == NULL) {					 \
			DEL_CMD_WORDS;					 \
			return(-1);					 \
		}							 \
	} else {							 \
		copy = NULL;						 \
	}								 \
	execargv[execargc] = copy;					 \
	execargc++;							 \
	if (execargc >= sizeof(execargv) / sizeof(execargv[0])) {	 \
		DEL_CMD_WORDS;						 \
		disconnect_from_scanner();				 \
		return(-1);						 \
	}								 \
	execargv[execargc] = NULL;					 \
} while (0)

	/* ... build a command line ... */
	{
		execargc = 0;
		ADD_CMD_WORD(antivir_program_name);
		ADD_CMD_WORD("--samba-vscan");
		if (antivir_arch_scan_enable) {
			char printbuff[40];

			ADD_CMD_WORD("--scan-in-archive");
			/* for some reason the %z format spec did not work :( */
			snprintf(printbuff, sizeof(printbuff), "--archive-max-size=%lld", (long long)antivir_arch_max_size);
			ADD_CMD_WORD(printbuff);
			snprintf(printbuff, sizeof(printbuff), "--archive-max-recursion=%d", antivir_arch_max_recursion);
			ADD_CMD_WORD(printbuff);
			snprintf(printbuff, sizeof(printbuff), "--archive-max-ratio=%d", antivir_arch_max_ratio);
			ADD_CMD_WORD(printbuff);
		}
		/* optionally add a "--temp=" option? */
		if (antivir_detect_alltypes)
			ADD_CMD_WORD("--alltypes");
		if (antivir_detect_dialer)
			ADD_CMD_WORD("--with-dialer");
		if (antivir_detect_game)
			ADD_CMD_WORD("--with-game");
		if (antivir_detect_joke)
			ADD_CMD_WORD("--with-joke");
		if (antivir_detect_pms)
			ADD_CMD_WORD("--with-pms");
		if (antivir_detect_spy)
			ADD_CMD_WORD("--with-spy");
	}

	/* ... setup a communication pipe ... */
	rc = pipe(fdreq);
	if (rc < 0) {
		DEL_CMD_WORDS;
		disconnect_from_scanner();
		return(-1);
	}
	rc = pipe(fdrsp);
	if (rc < 0) {
		DEL_CMD_WORDS;
		disconnect_from_scanner();
		return(-1);
	}

	/* ... fork off the process ... */
	pid = fork();
	switch (pid) {
	case -1:	/* failure */
		DEL_CMD_WORDS;
		disconnect_from_scanner();
		return(-1);
	case 0: 	/* the child (scanner) */
		/* redirect stdin */
		rc = dup2(fdreq[0], STDIN_FILENO);
		if (rc == -1)
			exit(1);
		close(fdreq[0]); /* redirected */
		fdreq[0] = -1;
		close(fdreq[1]); /* not needed */
		fdreq[1] = -1;
		/* redirect stdout */
		rc = dup2(fdrsp[1], STDOUT_FILENO);
		if (rc == -1)
			exit(1);
		close(fdrsp[1]); /* redirected */
		fdrsp[1] = -1;
		close(fdrsp[0]); /* not needed */
		fdrsp[0] = -1;
		/* redirect stderr */
		fclose(stderr); freopen("/dev/null", "w", stderr);
		/* close all fds */
		/* XXX how to portably get the max fd?
		 *   fd = open("/dev/null", O_RDONLY);
		 * does not work reliably in the smbd(8) case,
		 * since there is a gap in the fd sequence
		 */
		fd = 1024; /* aux upper limit */
		while (fd > STDERR_FILENO) {
			close(fd);
			fd--;
		}
		/* execute the scanner process */
		rc = execvp(execargv[0], execargv);
		/* UNREACH */
		exit(1);
	default:	/* parent (samba-vscan) */
		/* release the command line */
		DEL_CMD_WORDS;
		/* get the read/write handles to the scanner process */
		close(fdreq[0]); fdreq[0] = -1;
		antivir_fd_write = fdreq[1];
		close(fdrsp[1]); fdrsp[1] = -1;
		antivir_fd_read = fdrsp[0];
		/* drain banners / greetings / help or synopsis on errors / etc */
		tv.tv_sec = 1; tv.tv_usec = 0;
		do {
			ssize_t ct;
#if defined USE_SELECT
			fd_set r;

			FD_ZERO(&r);
			FD_SET(antivir_fd_read, &r);
			rc = select(antivir_fd_read + 1, &r, NULL, NULL, &tv);
			tv.tv_sec = 0; tv.tv_usec = 0;
			if (rc < 0) {
				vscan_syslog("ERROR: Can not read from scanner - %s", strerror(errno));
				disconnect_from_scanner();
				return -1;
			}
			if (rc == 0)
				break;
			if (! FD_ISSET(antivir_fd_read, &r))
				break;
#else /* USE_SELECT */
			struct pollfd pfd[1];

			memset(pfd, 0, sizeof(pfd));
			pfd[0].fd = antivir_fd_read;
			pfd[0].events = POLLIN;
			rc = poll(pfd, 1, tv.tv_sec * 1000);
			tv.tv_sec = 0; tv.tv_usec = 0;
			if (rc < 0) {
				vscan_syslog("ERROR: Can not read from scanner - %s", strerror(errno));
				disconnect_from_scanner();
				return -1;
			}
			if (rc == 0)
				break;
#endif /* USE_SELECT */
			ct = read(antivir_fd_read, buff, sizeof(buff));
			if (ct <= 0) {
				vscan_syslog("ERROR: Can not read from scanner - %s", strerror(errno));
				disconnect_from_scanner();
				return -1;
			}
		} while (1);
		/* convert "int fd" to "FILE *" for fgets(3) */
		antivir_file_read = fdopen(antivir_fd_read, "r");
		if (antivir_file_read == NULL) {
			vscan_syslog("ERROR: Can not read from scanner - %s", strerror(errno));
			disconnect_from_scanner();
			return -1;
		}
		/* scanner process gone after the greeting? */
		if (kill(pid, 0) != 0) {
			vscan_syslog("ERROR: scanner disappeared at connect time");
			disconnect_from_scanner();
			return(-1);
		}
		antivir_scanner_pid = pid;
		/* connection setup done, now we're happy */
		return(0);
	}
	/* UNREACH */
	return(-1);
}

/*
 * invalidate the scanner process
 * (detach and clear vars after use or failure)
 */
static void disconnect_from_scanner(void) {
	if (antivir_fd_write != -1) {
		close(antivir_fd_write);
		antivir_fd_write = -1;
	}
	/* XXX drain possible input? */
	if (antivir_file_read != NULL) {
		fclose(antivir_file_read);
		antivir_file_read = NULL;
		antivir_fd_read = -1;
	} else if (antivir_fd_read != -1) {
		close(antivir_fd_read);
		antivir_fd_read = -1;
	}
	antivir_scanner_pid = 0;
}
