/*
 * $Id: vscan-oav_core.c,v 1.18.2.4 2004/09/25 12:27:07 reniar Exp $
 * 
 * Core Interface for OpenAntiVirus ScannerDaemon			
 *
 * Copyright (C) Rainer Link, 2001-2004
 *               OpenAntiVirus.org <rainer@openantivirus.org>
 *
 * Credits to W. Richard Stevens - RIP
 * 
 * This software is licensed under the GNU General Public License (GPL)
 * See: http://www.gnu.org/copyleft/gpl.html
 *
*/

#include "vscan-oav_core.h"
#include "vscan-global.h"

/* hum, global vars ... */
extern BOOL verbose_file_logging;
extern BOOL send_warning_message;


/* initialise socket to ScannerDaemon
   returns -1 on error or the socket descriptor */
int vscan_oav_init(const char* oav_ip, const unsigned short int oav_port)
{

	return vscan_inet_socket_init("ScannerDaemon", oav_ip, oav_port); 

}

/*
  If virus is found, logs the filename/virusname into syslog
*/
void vscan_oav_log_virus(char *infected_file, char *result, char* client_ip)
{
        char *str;
        size_t len;

        /* remove "FOUND: " from the result string to get only the virus name */

	len = strlen(result);
        /* sanity check ... */
        if ( len < 7 ) {
            /* hum, sth went wrong ... */
            vscan_syslog_alert("ALERT - Scan result: '%s' infected with virus 'UNKOWN', client: '%s'", infected_file, client_ip);
	    if ( send_warning_message )
		vscan_send_warning_message(infected_file, "UNKNOWN", client_ip);

        } else {
            str = result;
            str+= 7;
            vscan_syslog_alert("ALERT - Scan result: '%s' infected with virus '%s', client: '%s'", infected_file, str, client_ip);
	    if ( send_warning_message )
		vscan_send_warning_message(infected_file, str, client_ip);
        }
}



/*
  Scans a file (*FILE*, not a directory - keep that in mind) for a virus
  Expects socket descriptor and file name to scan for
  Returns -2 on a minor error, -1 on error, 0 if no virus was found, 
  1 if a virus was found 
*/
int vscan_oav_scanfile(int sockfd, char *scan_file, char* client_ip)
{
	char recvline[MAXLINE + 1];
	pstring oavCommand;

	FILE *fpin, *fpout;

	fpin = fdopen(sockfd, "r");
	if ( fpin == NULL ) {
		vscan_syslog("ERROR: can not open stream for reading - %s", strerror(errno));
		return VSCAN_SCAN_ERROR;
	}

	fpout = fdopen(sockfd, "w");
	if ( fpout == NULL ) {
		/* close fpin */
		fclose(fpin);
		vscan_syslog("ERROR: can not open stream for writing - %s", strerror(errno));
		return VSCAN_SCAN_ERROR;
	}

	if ( verbose_file_logging )
	        vscan_syslog("INFO: Scanning file : '%s'", scan_file);

	/* ScannerDaemon expects "SCAN <filename>\n" */
	/* what about if the <filename> itself contains '\n'? */
        pstrcpy(oavCommand, "SCAN ");
        pstrcat(oavCommand, scan_file);
        pstrcat(oavCommand, "\n");

	/* NOTE: what happens if scan_file is very long? */

	/* write to socket-stream */
	if ( fputs(oavCommand, fpout) == EOF) {
		vscan_syslog("ERROR: can not send file name to ScannerDaemon!");
		/* closing streams */
		fclose(fpin);
		fclose(fpout);
		return VSCAN_SCAN_ERROR;
	}

	/* hum, instead of flush()ing, use setvbuf to set line-based buffering? */
	if ( fflush(fpout) == EOF ) {
		vscan_syslog("ERROR: can not flush output stream - %s", strerror(errno));
		/* FIXME: shouldn't we return -1 here? */
	}

	/* read from socket-stream */
	if ( fgets(recvline, MAXLINE, fpin) == NULL ) {
		/* close streams */
		fclose(fpin);
		fclose(fpout);
		vscan_syslog("ERROR: can not get result from ScannerDaemon!");
		return VSCAN_SCAN_ERROR;
	}

	/* close streams */
	fclose(fpin);
	fclose(fpout);

	/* Response from ScannerDaemon is either
	   - OK                  file scanned and not infected
	   - FOUND: <virusname>  file scanned and infected
	   - ERROR		 sth went wrong, i.e. file not found
	*/
	if ( strncmp("FOUND", recvline, 5) == 0   ) {
		/* virus found */
		vscan_oav_log_virus(scan_file, recvline, client_ip);
		return VSCAN_SCAN_VIRUS_FOUND;
	} else if ( strncmp("ERROR", recvline, 5) == 0  ) {
		/* ERROR */
		if ( verbose_file_logging )
			vscan_syslog("ERROR: file %s not found, not readable or an error occured", scan_file);
		return VSCAN_SCAN_MINOR_ERROR;
	} else if ( strncmp("OK", recvline, 2) == 0 ) {
		/* OK */
		if ( verbose_file_logging )
			vscan_syslog("INFO: file %s is clean", scan_file);
        } else {
		/* unknown response */
		vscan_syslog("ERROR: unknown response from ScannerDaemon while scanning %s!", scan_file);
		/* FIXME: should we really mark this as a minor error? */
		return VSCAN_SCAN_MINOR_ERROR;
	}	

	return VSCAN_SCAN_OK;
	
}

/*
  close socket
*/
void vscan_oav_end(int sockfd)
{
	vscan_socket_end(sockfd);
}
