#!/bin/bash

#######################################################################
# This is the DRI installation script. It compiles the kernel modules 
# and installs the kernel, DRI and libGL modules.
#
#   Frank Worsley   <franco@nettaxi.com>
#   Alan Hourihane  <alanh@fairlite.demon.co.uk>
#   Felix Kuehling  <fxkuehl@gmx.de>
#
#######################################################################

###########################################################
# VARIABLE DEFINITIONS
###########################################################

LOGFILE="dri.log"
LOGFILE_TMP="tmp.log"

# Defaults
XF86_DIR="/usr/X11R6"
KERNEL_VER=`uname -r`
KERNEL_LIB_DIR="/lib/modules/$KERNEL_VER"
OPTION=$1

# Kernel version and module filename extension
KERNEL_MAJOR=${KERNEL_VER%%.*}
KERNEL_TAIL=${KERNEL_VER#*.}
KERNEL_MINOR=${KERNEL_TAIL%%.*}
KERNEL_PATCH=${KERNEL_TAIL#.*}

if [ $KERNEL_MAJOR -eq 2 -a $KERNEL_MINOR -le 4 ]; then
    KERNEL_MOD_EXT=o
else
    # This may be incorrect for early 2.5 kernels.
    # Anyone still using one, add an extra check. ;-)
    KERNEL_MOD_EXT=ko
fi

###########################################################
# FUNCTION DECLARATIONS
###########################################################

print_logo() {

	################################################################
	# This function prints a DRI logo and a general text header.
	################################################################

	clear
	echo ""
	echo "DIRECT RENDERING OPEN SOURCE PROJECT  -  DRIVER INSTALLATION SCRIPT"
	echo ""
	echo "[ http://dri.freedesktop.org ]"
	echo ""
	echo "=========================================================================="
	echo ""
}

detect_xorg() {

    #####################################################################
    # Detects Xorg 7 or uses X11R6 defaults
    #####################################################################

    X_VERSION=6
    XF86_DIR="/usr/X11R6"
    XF86_MODULES_DIR="$XF86_DIR/lib/modules"
    XF86_GL_DIR="$XF86_DIR/lib"
    if which pkg-config > /dev/null 2>&1 && pkg-config xorg-server; then
	X_VERSION=7
	XORG_DIR=`pkg-config --variable=prefix xorg-server`
	XORG_MODULES_DIR=`pkg-config --variable=moduledir xorg-server`
	XORG_GL_DIR=`pkg-config --variable=libdir xorg-server`

	if [ -n "$XORG_DIR" ]; then
	    XF86_DIR=$XORG_DIR
	fi

	if [ -n "$XORG_MODULES_DIR" ]; then
	    XF86_MODULES_DIR=$XORG_MODULES_DIR
	else
	    XF86_MODULES_DIR="$XF86_DIR/lib/xorg/modules"
	fi

	if [ -n "$XORG_GL_DIR" ]; then
	    XF86_GL_DIR=$XORG_GL_DIR
	else
	    XF86_GL_DIR="$XF86_DIR/lib"
	fi
    fi
}

set_values() {

	#####################################################################
	# Sets up default values.
	#####################################################################

	# Determine driver to be installed
	DRV_NAME=`sed -n '1p' pkginfo`
	DRV_DESC=`sed -n '2p' pkginfo`
	DRV_ARCH=`sed -n '3p' pkginfo`
	DRV_DATE=`sed -n '4p' pkginfo`
	DRV_MODULE=`sed -n '5p' pkginfo`
	DRV_VERSION=`sed -n '6p' pkginfo`
	DRV_BUILD_DESC=`sed -n '7p' pkginfo`
	
	# Determine directories from default or user values
	XF86_DRI_DIR="$XF86_MODULES_DIR/dri"
	XF86_DRV_DIR="$XF86_MODULES_DIR/drivers"
	XF86_EXT_DIR="$XF86_MODULES_DIR/extensions"
	XF86_OS_DIR="$XF86_MODULES_DIR/linux"
	KERNEL_DRM_DIR="$KERNEL_LIB_DIR/kernel/drivers/char/drm"
}

check_values() {

	#####################################################################
	# Checks if default/user given values are correct.
	#####################################################################
	
	# Check if installation directories exist
	MSG=$'The following problems have occured:\n'
	FATAL=0
	ERR=0
	
	if [ \( -d $DRV_NAME -o -d core \) -a ! -d "$XF86_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg directory does not exist'
		ERR="1"
		FATAL="1";
	fi
	
	if [ -d drm -a ! -d "$KERNEL_LIB_DIR" ]; then
		MSG="$MSG"$'\n	- Kernel module directory does not exist'
		ERR="1"
		FATAL="1";
	fi

	if [ -d drm -a ! -d "$KERNEL_DRM_DIR" ]; then
		MSG="$MSG"$'\n	- Kernel DRM directory does not exist'
		ERR="1";
	fi
	
	if [ -d GL -a ! -d "$XF86_GL_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg library directory does not exist'
		ERR="1";
	fi	
	
	if [ -d $DRV_NAME -a ! -d "$XF86_DRI_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg DRI directory does not exist'
		ERR="1";
	fi		

	if [ -d $DRV_NAME -a ! -d "$XF86_DRV_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg driver directory does not exist'
		ERR="1";
	fi
	
	if [ -d core -a ! -d "$XF86_MODULES_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg module directory does not exist'
		ERR="1";
	fi	

	if [ -d core -a ! -d "$XF86_EXT_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg extensions directory does not exist'
		ERR="1";
	fi	
	
	if [ -d core -a ! -d "$XF86_OS_DIR" ]; then
		MSG="$MSG"$'\n	- Xorg Linux OS directory does not exist'
		ERR="1";
	fi	
	
	# No error so return
	if [ "$ERR" == "0" ]; then
		return 0;
	fi
	
	# Get out if we have a fatal error that requires user to re-enter values
	if [ "$FATAL" == "1" ]; then
		MSG="$MSG"$'\n\nPlease make sure you have entered all information correctly.\n\nPress ENTER to review the information.'
		print_logo
		echo "$MSG"
		read KEY
		return -1;
	fi
	
	#Fix what we can
	MSG="$MSG"$'\n\nThe script can create these directories for you.\n\nPress ENTER to continue or CTRL-C to abort.'
	print_logo
	echo "$MSG"
	read KEY
	
	# Now we just create all directories no matter what ...
	mkdir -p $XF86_GL_DIR
	mkdir -p $XF86_MODULES_DIR
	mkdir -p $XF86_DRI_DIR
	mkdir -p $XF86_DRV_DIR
	mkdir -p $XF86_EXT_DIR
	mkdir -p $XF86_OS_DIR
	mkdir -p $KERNEL_DRM_DIR
	
	return 0;
}

copy_files() {

	#####################################################################
	# This function copies files to their destination directories.
	#####################################################################
	
	echo "Installing files:"
	
	if [ -d $DRV_NAME ]; then
		echo -n "	DRI Xorg modules..."
		cd $DRV_NAME
		ls -1 *drv.so | while read FILE
		do
			mv -f $XF86_DRV_DIR/$FILE $XF86_DRV_DIR/dri-old.$FILE >& $LOGFILE_TMP;
		done
		ls -1 *dri.so | while read FILE
		do
			mv -f $XF86_DRI_DIR/$FILE $XF86_DRI_DIR/dri-old.$FILE >& $LOGFILE_TMP;
		done	
		cd ..
		cp -f $DRV_NAME/*drv.so $XF86_DRV_DIR
		cp -f $DRV_NAME/*dri.so $XF86_DRI_DIR	
		echo "done"
	fi

	if [ -d drm ]; then
		echo -n "	kernel modules..."
		cd drm
		UP=..
		if [ -d linux -a -d linux-core ]; then
		    if [ $KERNEL_MINOR == 6 ]; then
			cd linux-core
		    else
			cd linux
		    fi
		    UP=../..
		fi
		mkdir -p $KERNEL_DRM_DIR
		ls -1 $DRV_MODULE.$KERNEL_MOD_EXT drm.$KERNEL_MOD_EXT \
		    2> /dev/null | while read FILE
		do
			mv -f $KERNEL_DRM_DIR/$FILE $KERNEL_DRM_DIR/dri-old.$FILE >& $LOGFILE_TMP;
			cp -f $FILE $KERNEL_DRM_DIR
		done
		cd $UP
		unset UP
		/sbin/depmod -a
		echo "done"
	fi
	
	if [ -d GL ]; then
		echo -n "	GL & GLU libraries..."
		cd GL
		ls -1 | while read FILE
		do
			mv -f $XF86_GL_DIR/$FILE $XF86_GL_DIR/dri-old.$FILE >& $LOGFILE_TMP;
			cp -f $FILE $XF86_GL_DIR
		done
		cd ..
		echo "done"
	fi
	
	if [ -d core ]; then
		echo -n "	core libraries..."
		cd core
		for FILE in libdri.so libglx.so libGLcore.so; do
			if [ -e $FILE ]; then
				mv -f $XF86_EXT_DIR/$FILE $XF86_EXT_DIR/dri-old.$FILE >& $LOGFILE_TMP
				cp -f $FILE $XF86_EXT_DIR;
			fi;
		done
		for FILE in libdrm.so; do
			if [ -e $FILE ]; then
				mv -f $XF86_OS_DIR/$FILE $XF86_OS_DIR/dri-old.$FILE >& $LOGFILE_TMP;
				cp -f $FILE $XF86_OS_DIR
			fi;
		done
		for FILE in libshadow.so libexa.so; do
			if [ -e $FILE ]; then
				mv -f $XF86_MODULES_DIR/$FILE $XF86_MODULES_DIR/dri-old.$FILE >& $LOGFILE_TMP;
				cp -f $FILE $XF86_MODULES_DIR
			fi;
		done
		cd ..
		echo "done"
	fi

	if [ -e extras/extras.sh ]; then
		echo -n "	Copying extra files..."
		extras/extras.sh $XF86_DIR
		echo "done"
	fi
}

restore_files () {

	print_logo
	echo "Restoring files..."
	
	if [ -d $DRV_NAME ]; then
		echo -n "	Xorg modules..."
		cd $DRV_NAME
		ls -1 *drv.so | while read FILE
		do
			mv -f $XF86_DRV_DIR/dri-old.$FILE $XF86_DRV_DIR/$FILE >& $LOGFILE_TMP;
		done
		for FILE in *dri.so
		do
			mv -f $XF86_DRI_DIR/dri-old.$FILE $XF86_DRI_DIR/$FILE >& $LOGFILE_TMP;
		done	
		cd ..
		echo "done"
	fi
	
	if [ -d drm ]; then
		echo -n "	kernel modules..."
		cd drm
		ls -1 *.$KERNEL_MOD_EXT | while read FILE
		do
			mv -f $KERNEL_DRM_DIR/dri-old.$FILE $KERNEL_DRM_DIR/$FILE >& $LOGFILE_TMP;
		done
		cd ..
		/sbin/depmod -a
		echo "done"
	fi
	
	if [ -d GL ]; then
		echo -n "	system libraries..."
		cd GL
		ls -1 * | while read FILE
		do
			mv -f $XF86_GL_DIR/dri-old.$FILE $XF86_GL_DIR/$FILE >& $LOGFILE_TMP;
		done
		cd ..
		echo "done"
	fi
	
	if [ -d core ]; then
		echo -n "	core libraries..."
		mv -f $XF86_EXT_DIR/dri-old.libdri.so $XF86_EXT_DIR/libdri.so >& $LOGFILE_TMP;
		mv -f $XF86_EXT_DIR/dri-old.libglx.so $XF86_EXT_DIR/libglx.so >& $LOGFILE_TMP;
		mv -f $XF86_EXT_DIR/dri-old.libGLcore.so $XF86_EXT_DIR/libGLcore.so >& $LOGFILE_TMP;
		mv -f $XF86_OS_DIR/dri-old.libdrm.so $XF86_OS_DIR/libdrm.so >& $LOGFILE_TMP;
		mv -f $XF86_MODULES_DIR/dri-old.libshadow.so $XF86_MODULES_DIR/libshadow.so >& $LOGFILE_TMP;
		mv -f $XF86_MODULES_DIR/dri-old.libexa.so $XF86_MODULES_DIR/libexa.so >& $LOGFILE_TMP;
		echo "done"
	fi
	
	echo ""
	echo "Completed restoring files."
}

update_config() {

	#####################################################################
	# This function updates the system configuration.
	#####################################################################
	
	echo ""
	echo "Updating configuration:"

	if [ -d GL ]; then
		echo -n "	Running ldconfig..."
		grep "$XF86_DIR" /etc/ld.so.conf >& $LOGFILE_TMP
		if [ $? != 0 ]; then
			echo "$XF86_DIR/lib" >> /etc/ld.so.conf
			/sbin/ldconfig >& $LOGFILE_TMP
			if [ $? != 0 ]; then
				echo "ERROR";
				ERR=1
			else
				echo "done";
			fi
		fi
		cat $LOGFILE_TMP >> $LOGFILE
	fi

	if [ -d drm -a ! "$X_RUNNING" == "1" ]; then
	
		# Only try and update kernel modules if X server is not running

		for MOD in $DRV_MODULE
		do
			echo -n "	Removing old kernel module \"$MOD\"..."
			/sbin/modprobe -r $MOD >& $LOGFILE_TMP

			if [ $? != 0 ]; then
				echo "ERROR"
				ERR=1;
			else
				echo "done";
			fi
	
			cat $LOGFILE_TMP >> $LOGFILE
		
			echo -n "	Inserting new kernel module \"$MOD\"..."
			/sbin/modprobe agpgart >& $LOGFILE_TMP
			/sbin/modprobe $MOD >& $LOGFILE_TMP
		
			if [ $? != 0 ]; then
				echo "ERROR"
				ERR=1;
			else
				echo "done";
			fi
	
			cat $LOGFILE_TMP >> $LOGFILE;
		done;
	fi
	
	if [ "$ERR" == "1" ]; then
		echo ""
		echo "There were errors updating the system configuration."
		echo "See the $LOGFILE file for more information.";
	fi

	rm $LOGFILE_TMP
}

check_config() {

	#####################################################################
	# This function checks the system configuration.
	#####################################################################
	
	echo ""
	echo -n "Checking configuration..."
	
	if [ -d GL ]; then
		# Fix potential libGL problems
	        GLXINFO=""
	        [ -x $XF86_DIR/bin/glxinfo ] && GLXINFO=$XF86_DIR/bin/glxinfo \
		    || which glxinfo > /dev/null 2>&1 && GLXINFO=`which glxinfo`
		LIBGL1=`ldd $GLXINFO 2> /dev/null | grep libGL.so.1 | awk -F" " '{ printf "%s",$3 }'`
		LIBGL=`echo $LIBGL1 | sed -e 's/\.1//'`
		if [ -n "$LIBGL" -a "$LIBGL" != "$XF86_GL_DIR/libGL.so" ]; then
		        SEC_GL_DIR=`echo $LIBGL1 | sed -e 's/libGL.so.1//'`
			echo ""
			echo -n "	second copy of DRI libraries found in "
			echo $SEC_GL_DIR
		
			echo -n "	libraries have been backed up to dri-old.* in "
			echo $SEC_GL_DIR
			echo ""

		        for FILE in libGL.so libGL.so.1 libGL.so.1.2
			do
				mv -f $SEC_GL_DIR/$FILE $SEC_GL_DIR/dri-old.$FILE
				ln -s $XF86_GL_DIR/$FILE $SEC_GL_DIR/$FILE
			done
		fi

		# Make sure libGL and have correct links
		rm -f $XF86_GL_DIR/libGL.so
		rm -f $XF86_GL_DIR/libGL.so.1
		ln -s $XF86_GL_DIR/libGL.so.1.2 $XF86_GL_DIR/libGL.so
		ln -s $XF86_GL_DIR/libGL.so.1.2 $XF86_GL_DIR/libGL.so.1
	fi

	echo "done"
}

###########################################################
# MAIN SCRIPT
###########################################################

# Check if pkginfo file exists
if [ ! -e "pkginfo" ]; then
	echo "Could not locate 'pkginfo' file. Aborting."
	exit 127;
fi

# Check if we are running as root
if [ `whoami` != "root" ] && [ `whoami` != "ROOT" ]; then
	echo "You must be root to install the DRI drivers."
	exit 127;
fi

# Check if sed is installed
which sed &> /dev/null
if [ "$?" != "0" ]; then
	echo "Could not located 'sed' editor. Aborting."
	exit 127;
fi

### FIXME: We should check for matching architectures here!!! ###

# Figure out if we should restore files
if [ "$OPTION" == "restore" ]; then
	
	print_logo
	detect_xorg
	echo "This will restore your previous files."
	echo ""
	echo "Xorg Dir          : $XF86_DIR"
	echo "Xorg Modules Dir  : $XF86_MODULES_DIR"
	echo "Xorg Library Dir  : $XF86_GL_DIR"
	echo "Kernel Module Dir : $KERNEL_LIB_DIR"
	echo ""
	echo "Press ENTER to restore files or C to change defaults."
	read KEY
	
	if [ "$KEY" == "C" ] || [ "$KEY" == "c" ]; then
		print_logo
		echo "Enter new values and then press ENTER."
		echo ""
		echo -n "Xorg Dir          : "
		read XF86_DIR
		echo -n "Xorg Modules Dir  : "
		read XF86_MODULES_DIR
		echo -n "Xorg Library Dir  : "
		read XF86_GL_DIR
		echo -n "Kernel Module Dir : "
		read KERNEL_LIB_DIR
		echo ""
		echo "Press ENTER to restore files."
		read KEY;
	fi

	set_values	
	restore_files
	echo "";
	exit 0;
fi

# Print X is running message
if [ -d drm ]; then
	ps -C X > /dev/null
	if [ $? == 0 ]; then
		X_RUNNING=1
		print_logo
		echo "WARNING: YOUR X SERVER SEEMS TO BE RUNNING!"
		echo ""
		echo "The script can not update your kernel modules while the X server is running."
		echo ""
		echo "You can do one of the following:"
		echo ""
		echo "1. Log out of your X session now and then run this script."
		echo ""
		echo "2. Run the script now and restart your computer after the installation."
		echo "   If you exit your X session now you will not have to restart."
		echo ""
		echo "Press ENTER to continue or CTRL-C to exit."
		read KEY ;
	else
		X_RUNNING=0;
	fi
fi

# Do misc stuff
rm $LOGFILE
touch $LOGFILE

# Setup the defaults values
detect_xorg
set_values

# Print a welcome message
print_logo
echo "Welcome to the DRI Driver Installation Script"
echo ""
echo "The package you downloaded is for the following driver: "
echo ""
echo "Driver Name    : $DRV_NAME"
echo "Description    : $DRV_DESC"
echo "Architecture   : $DRV_ARCH"
echo "Build Date     : $DRV_DATE"
echo "Kernel Module  : $DRV_MODULE"
echo ""
echo "Optional Information"
echo ""
echo "Driver Version      : $DRV_VERSION"
echo "Special Description : $DRV_BUILD_DESC"
echo ""
echo "Press ENTER to continue or CTRL-C to exit."
read KEY

# Determine Xorg Directory and Kernel Module directory from user input
ERR=-1;
while [ "$ERR" == "-1" ]; do

	# Determine Xorg directory
	if [ -d $DRV_NAME -o -d core ]; then
		print_logo
		echo "The script will need to copy the DRI Xorg driver modules to"
		echo "your Xorg directories."
		echo ""
		echo "The script will use the following Xorg directories:"
		echo ""
		echo "Xorg Dir         : $XF86_DIR"
		echo "Xorg Modules Dir : $XF86_MODULES_DIR"
		echo "Xorg Library Dir : $XF86_GL_DIR"
		echo ""
		echo "If this is correct press ENTER, press C to change or CTRL-C to exit."
		read KEY

		if [ "$KEY" == "C" ] || [ "$KEY" == "c" ]; then
			print_logo
			echo "Please enter the Xorg directories you would like to use."
			echo ""
			echo "Press ENTER when done or CTRL-C to abort."
			echo ""
			echo -n "Xorg Dir         : "
			read XF86_DIR ;
			echo -n "Xorg Modules Dir : "
			read XF86_MODULES_DIR ;
			echo -n "Xorg Library Dir : "
			read XF86_GL_DIR ;
		fi
	fi

	# Determine Kernel module directory
	if [ -d drm ]; then
		print_logo
		echo "The script also needs to copy the DRM kernel modules to your"
		echo "kernel module directory."
		echo ""
		echo "This version of the script supports 2.4.x and 2.6.x kernels."
		echo ""
		echo "Kernel Version   : $KERNEL_VER"
		echo "Module Directory : $KERNEL_LIB_DIR"
		echo ""
		echo "If this is correct press ENTER, press C to change or CTRL-C to exit."
		read KEY

		if [ "$KEY" == "C" ] || [ "$KEY" == "c" ]; then
			print_logo
			echo "Please enter your kernel module directory."
			echo ""
			echo "Press ENTER when done or CTRL-C to abort."
			echo ""
			echo -n "Directory: "
			read KERNEL_LIB_DIR ;
		fi
	fi

	# Determine paths from user input or defaults
	set_values
	
	# Check if values are good
	check_values
	ERR="$?";
done

# Compile the kernel modules
if [ -d drm ]; then
	print_logo
	echo "The script will now compile the DRM kernel modules for your machine."
	echo ""
	echo "Press ENTER to continue or CTRL-C to exit."
	read KEY

	echo ""
	echo -n "Compiling..."
	cd drm
	UP=..
	if [ -d linux -a -d linux-core ]; then
	    if [ $KERNEL_MINOR == 6 ]; then
		cd linux-core
	    else
		cd linux
	    fi
	    UP=../..
	fi
	if [ -e Makefile.linux ]; then
		MAKEFILE=Makefile.linux
	else
		MAKEFILE=Makefile
	fi
	# Makefile.linux doesn't have a rule to build .ko-files for 2.6 kernels.
	# They are somehow created automagically after $DRV_MODULE.o is built.
	make -f $MAKEFILE $DRV_MODULE.o >& $UP/$LOGFILE_TMP
	if [ $? != 0 ]; then
		cd $UP
		cat $LOGFILE_TMP >> $LOGFILE
		echo ""
		echo "ERROR: Kernel modules did not compile"
		echo ""
		echo "The DRI drivers can not be installed without the latest kernel modules."
		echo "Installation will be aborted. See the $LOGFILE file for information on"
		echo "what went wrong."
		echo ""
		exit 127;
	else
		echo "done";
	fi
	cd $UP
	unset UP
	cat $LOGFILE_TMP >> $LOGFILE
fi

# Print a last message
print_logo
echo "The script is now ready to complete the installation."
echo ""
echo "Press ENTER to continue or CTRL-C to exit."
read KEY

# Copy files, update config, check config and fix problems
print_logo

copy_files
update_config
check_config

echo ""
echo "Press ENTER to continue."
read KEY

# Print last message
print_logo
echo "The DRI installation is complete."
echo ""
if [ -d drm -a "$X_RUNNING" == "1" ]; then
	echo "Stop your X server, reload the drm module and restart the X server"
	echo "to try the new DRI drivers."
	echo ""
elif [ -d $DRV_NAME -o -d core ]; then
	echo "Restart your X server to try the new DRI drivers."
	echo ""
fi
echo "If you have problems with the DRI after upgrading your drivers"
echo "please visit the DRI website and read the Help and FAQ section."
echo "The FAQ contains solutions to many common problems."
echo ""
echo "Report any bugs, problems and comments on the dri-devel mailing list."
echo ""
echo "Thank you for using the DRI."
echo ""
