package net.sf.saxon.expr;
import net.sf.saxon.om.*;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.type.ItemType;
import net.sf.saxon.type.Type;
import net.sf.saxon.value.*;

/**
* A RangeExpression is an expression that represents an integer sequence as
* a pair of end-points (for example "x to y").
* If the end-points are equal, the sequence is of length one.
 * <p>From Saxon 7.8, the sequence must be ascending; if the end-point is less
 * than the start-point, an empty sequence is returned. This is to allow
 * expressions of the form "for $i in 1 to count($seq) return ...." </p>
*/

public class RangeExpression extends BinaryExpression {

    /**
    * Construct a RangeExpression
    */

    public RangeExpression(Expression start, int op, Expression end) {
        super(start, op, end);
    }

    /**
    * Type-check the expression
    */

    public Expression analyze(StaticContext env, ItemType contextItemType) throws XPathException {
        operand0 = operand0.analyze(env, contextItemType);
        operand1 = operand1.analyze(env, contextItemType);

        RoleLocator role0 = new RoleLocator(RoleLocator.BINARY_EXPR, "to", 0, null);
        role0.setSourceLocator(this);
        operand0 = TypeChecker.staticTypeCheck(
                operand0, SequenceType.OPTIONAL_INTEGER, false, role0, env);

        RoleLocator role1 = new RoleLocator(RoleLocator.BINARY_EXPR, "to", 1, null);
        role1.setSourceLocator(this);
        operand1 = TypeChecker.staticTypeCheck(
                operand1, SequenceType.OPTIONAL_INTEGER, false, role1, env);

        if (operand0 instanceof IntegerValue && operand1 instanceof IntegerValue) {
            long i0 = ((IntegerValue)operand0).longValue();
            long i1 = ((IntegerValue)operand1).longValue();
            if (i0 > i1) {
                return EmptySequence.getInstance();
            } else {
                return new IntegerRange(i0, i1);
            }
        }
        return super.simplify(env);
    }

    /**
    * Get the data type of the items returned
    */

    public ItemType getItemType() {
        return Type.INTEGER_TYPE;
    }

    /**
    * Determine the static cardinality
    */

    public int computeCardinality() {
        return StaticProperty.ALLOWS_ZERO_OR_MORE;
    }

    /**
    * Return an iteration over the sequence
    */

    public SequenceIterator iterate(XPathContext context) throws XPathException {
        AtomicValue av1 = (AtomicValue)operand0.evaluateItem(context);
        if (av1 == null) {
            return new EmptyIterator();
        }
        NumericValue v1 = (NumericValue)av1.getPrimitiveValue();

        AtomicValue av2 = (AtomicValue)operand1.evaluateItem(context);
        if (av2 == null) {
            return new EmptyIterator();
        }
        NumericValue v2 = (NumericValue)av2.getPrimitiveValue();

        if (v1.compareTo(v2) > 0) {
            return new EmptyIterator();
        }
        return new RangeIterator(v1.longValue(), v2.longValue());
    }

    /**
    * Iterator that produces numeric values in a monotonic sequence,
    * ascending or descending. Although a range expression (N to M) is always
     * in ascending order, applying the reverse() function will produce
     * a RangeIterator that works in descending order.
    */

    public static class RangeIterator implements SequenceIterator,
                                                  ReversibleIterator,
                                                  LastPositionFinder,
                                                  LookaheadIterator,
                                                  GroundedIterator  {

        long start;
        long currentValue;
        int increment;   // always +1 or -1
        long limit;

        public RangeIterator(long start, long end) {
            this.start = start;
            increment = (start <= end ? +1 : -1);
            currentValue = start;
            limit = end;
        }

        public boolean hasNext() {
            if (increment>0) {
                return currentValue <= limit;
            } else if (increment<0) {
                return currentValue >= limit;
            } else {
                return false;
            }
        }

        public Item next() {
            if (!hasNext()) {
                increment = 0;
                return null;
            }
            long d = currentValue;
            currentValue += increment;
            return new IntegerValue(d);
        }

        public Item current() {
            if (increment == 0) {
                return null;
            } else {
                return new IntegerValue(currentValue - increment);
            }
        }

        public int position() {
            if (increment > 0) {
                return (int)(currentValue - start);
            } else if (increment < 0) {
                return (int)(start - currentValue);
            } else {
                return -1;
            }
        }

        public int getLastPosition() {
            return (int)((limit - start) * increment + 1);
        }

        public SequenceIterator getAnother() throws XPathException {
            return new RangeIterator(start, limit);
        }

        public SequenceIterator getReverseIterator() {
            return new RangeIterator(limit, start);
        }

        /**
         * Return a Value containing all the items in the sequence returned by this
         * SequenceIterator. This should be an "in-memory" value, not a Closure.
         *
         * @return the corresponding Value
         */

        public Value materialize() throws XPathException {
            if (increment == 1) {
                return new IntegerRange(start, limit);
            } else {
                return new SequenceExtent(getAnother());
            }
        }
    }
}

//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): none.
//
