package net.sf.saxon.instruct;

import net.sf.saxon.Controller;
import net.sf.saxon.pull.UnconstructedDocument;
import net.sf.saxon.event.Receiver;
import net.sf.saxon.event.SequenceReceiver;
import net.sf.saxon.expr.ExpressionTool;
import net.sf.saxon.expr.XPathContext;
import net.sf.saxon.expr.Expression;
import net.sf.saxon.expr.StaticContext;
import net.sf.saxon.om.*;
import net.sf.saxon.pattern.NodeKindTest;
import net.sf.saxon.style.StandardNames;
import net.sf.saxon.tinytree.TinyBuilder;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.type.ItemType;
import net.sf.saxon.type.SchemaType;
import net.sf.saxon.value.TextFragmentValue;

import java.io.PrintStream;


/**
 * An instruction to create a document node. This doesn't correspond directly
 * to any XSLT instruction. It is used to support the document node constructor
 * expression in XQuery, and is used as a sub-instruction within an xsl:variable
 * that constructs a temporary tree.
 *
 * <p>Conceptually it represents an XSLT instruction xsl:document-node,
 * with no attributes, whose content is a complex content constructor for the
 * children of the document node.</p>
 */

public class DocumentInstr extends ParentNodeConstructor {

    private static final int[] treeSizeParameters = {50, 10, 5, 200};
    // estimated size of a temporary tree: {nodes, attributes, namespaces, characters}

    private boolean textOnly;
    private String constantText;
    private String baseURI;


    public DocumentInstr(boolean textOnly,
                         String constantText,
                         String baseURI) {
        this.textOnly = textOnly;
        this.constantText = constantText;
        this.baseURI = baseURI;
    }

    /**
     * An implementation of Expression must provide at least one of the methods evaluateItem(), iterate(), or process().
     * This method indicates which of these methods is prefered. For instructions this is the process() method.
     */

    public int getImplementationMethod() {
        return Expression.EVALUATE_METHOD;
    }


    /**
     * Set the validation action
     */

    public void setValidationAction(int action) {
        validation = action;
    }

    /**
     * Set the SchemaType of the document element
     */

    public void setSchemaType(SchemaType type) {
        schemaType = type;
    }

    /**
     * Simplify an expression. This performs any static optimization (by rewriting the expression
     * as a different expression). The default implementation does nothing.
     *
     * @return the simplified expression
     * @throws net.sf.saxon.trans.XPathException
     *          if an error is discovered during expression rewriting
     */

    public Expression simplify(StaticContext env) throws XPathException {
        setLazyConstruction(env.getConfiguration().isLazyConstructionMode());
        return super.simplify(env);
    }

    /**
     * Perform static analysis of an expression and its subexpressions.
     * <p/>
     * <p>This checks statically that the operands of the expression have
     * the correct type; if necessary it generates code to do run-time type checking or type
     * conversion. A static type error is reported only if execution cannot possibly succeed, that
     * is, if a run-time type error is inevitable. The call may return a modified form of the expression.</p>
     * <p/>
     * <p>This method is called after all references to functions and variables have been resolved
     * to the declaration of the function or variable. However, the types of such functions and
     * variables will only be accurately known if they have been explicitly declared.</p>
     *
     * @param env the static context of the expression
     * @return the original expression, rewritten to perform necessary
     *         run-time type checks, and to perform other type-related
     *         optimizations
     * @throws net.sf.saxon.trans.XPathException
     *          if an error is discovered during this phase
     *          (typically a type error)
     */

    public Expression analyze(StaticContext env, ItemType contextItemType) throws XPathException {
        Expression res = super.analyze(env, contextItemType);
        verifyLazyConstruction();
        return res;
    }

    public ItemType getItemType() {
        return NodeKindTest.DOCUMENT;
    }

    public TailCall processLeavingTail(XPathContext context) throws XPathException {
        Item item = evaluateItem(context);
        if (item != null) {
            SequenceReceiver out = context.getReceiver();
            out.append(item, locationId, NodeInfo.ALL_NAMESPACES);
        }
        return null;
    }

    /**
     * Evaluate as an expression.
     */

    public Item evaluateItem(XPathContext context) throws XPathException {
        if (isLazyConstruction()) {
            return new UnconstructedDocument(this, context);
        } else {
            Controller controller = context.getController();
            DocumentInfo root;
            if (textOnly) {
                CharSequence textValue;
                if (constantText != null) {
                    textValue = constantText;
                } else {
                    FastStringBuffer sb = new FastStringBuffer(100);
                    SequenceIterator iter = content.iterate(context);
                    if (iter instanceof AtomizableIterator) {
                        ((AtomizableIterator)iter).setIsAtomizing(true);
                    }
                    while (true) {
                        Item item = iter.next();
                        if (item==null) break;
                        sb.append(item.getStringValueCS());
                    }
                    textValue = sb.condense();
                }
                root = new TextFragmentValue(textValue, baseURI);
                ((TextFragmentValue)root).setConfiguration(controller.getConfiguration());
            } else {
                XPathContext c2 = context.newMinorContext();
                c2.setOrigin(this);

                // TODO: use an Outputter that delayes the decision whether to build a
                // TextFragment or a TinyTree until the first element is encountered, to
                // avoid the overhead of using a TinyTree for text-only trees. This would
                // make the static analysis superfluous.

                TinyBuilder builder = new TinyBuilder();
                //System.err.println("Build doc " + builder);
                builder.setSizeParameters(treeSizeParameters);
                builder.setLineNumbering(controller.getConfiguration().isLineNumbering());

                Receiver receiver = builder;
                receiver.setSystemId(baseURI);
                receiver.setPipelineConfiguration(controller.makePipelineConfiguration());

                c2.changeOutputDestination(null,
                        receiver,
                        false,
                        validation,
                        schemaType);
                Receiver out = c2.getReceiver();
                out.open();
                out.startDocument(0);

                content.process(c2);

                out.endDocument();
                out.close();

                root = (DocumentInfo)builder.getCurrentRoot();
            }
            return root;
        }
    }


    /**
     * Get the name of this instruction for diagnostic and tracing purposes
     * (the string "document-constructor")
     */

    public int getInstructionNameCode() {
        return StandardNames.XSL_DOCUMENT;
    }

    /**
     * Display this instruction as an expression, for diagnostics
     */

    public void display(int level, NamePool pool, PrintStream out) {
        out.println(ExpressionTool.indent(level) + "document-constructor");
        content.display(level+1, pool, out);
    }
}

//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay.
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): none.
//
