# -*- coding: utf-8 -*-
# Copyright © 2005 Lateef Alabi-Oki
#
# This file is part of Scribes.
#
# Scribes is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Scribes is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Scribes; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

from gtk import FileChooserDialog
from gobject import SIGNAL_RUN_LAST, TYPE_NONE

class OpenDialog(FileChooserDialog):
	"""
	This class instantiates a dialog that allows clients to select and load
	files into the text editor. It inherites from ScribesDialog and the
	gtk.FileChooserDialog classes. See dialog.py and the GTK+ manual
	respectively for the properties of each class.
	"""

	def __init__(self, editor):
		"""
		Initialize the text editor's open dialog.

		@param self: Reference to the OpenDialog instance.
		@type self: A OpenDialog object.

		@param window: A possible parent window.
		@type window: A gtk.Window object.
		"""
		FileChooserDialog.__init__(self)
		self.__init_attributes(editor)
		self.__set_dialog_properties()
		self.__signal_id_1 = self.__editor.connect_after("show-dialog", self.__dialog_show_dialog_cb)
		self.__signal_id_2 = self.__editor.connect("loaded-document", self.__dialog_reset_flags_cb)
		self.__signal_id_3 = self.__editor.connect("renamed-document", self.__dialog_reset_flags_cb)

	def __init_attributes(self, editor):
		"""
		Initialize the dialog attributes.

		@param self: Reference to the OpenDialog instance.
		@type self: A OpenDialog object.

		@param window: A possible parent window.
		@type window: A gtk.Window object.
		"""
		self.__editor = editor
		self.encoding  = None
		from SCRIBES.encodingbox import ScribesEncodingComboBox
		from SCRIBES.utils import create_encoding_box
		self.__encoding_box  = create_encoding_box(ScribesEncodingComboBox(editor))
		self.__shortcut_folder_is_set = False
		self.__signal_id_1 = self.__signal_id_2 = self.__signal_id_3 = None
		self.__signal_id_4 = None
		return

	def __set_dialog_properties(self):
		"""
		Set the dialog properties.

		@param self: Reference to the OpenDialog instance.
		@type self: A OpenDialog object.
		"""
		from SCRIBES.utils import calculate_resolution_independence
		width, height = calculate_resolution_independence(self.__editor.window,
														1.6, 1.929648241)
		self.set_default_size(width, height)
		from i18n import msg0001
		self.set_property("title", msg0001)
		self.set_property("role", "scribes_open_dialog")
		self.set_property("icon-name", "stock_open")
		self.set_property("local-only", False)
		self.set_property("select-multiple", True)
		self.set_property("name", "OpenDialog")
		from gtk import STOCK_OPEN, STOCK_CANCEL, RESPONSE_OK, RESPONSE_CANCEL
		self.add_button(STOCK_CANCEL, RESPONSE_CANCEL)
		self.add_button(STOCK_OPEN, RESPONSE_OK)
		self.set_extra_widget(self.__encoding_box)
		self.set_default_response(RESPONSE_OK)
		self.set_transient_for(self.__editor.window)
		from SCRIBES.dialogfilter import create_filter_list
		for filter in create_filter_list():
			self.add_filter(filter)
		return

	def show_dialog(self):
		"""
		Show the text editor's open dialog.

		@param self: Reference to the OpenDialog instance.
		@type self: A OpenDialog object.
		"""
		self.__editor.emit("show-dialog", self)
		from i18n import msg0002
		status_id = self.__editor.feedback.set_modal_message(msg0002, "open")
		self.show_all()
		response = self.run()
		self.__editor.feedback.unset_modal_message(status_id)
		from gtk import RESPONSE_OK
		self.__editor.emit("hide-dialog", self)
		self.hide()
		from operator import eq
		if eq(response, RESPONSE_OK):
			# Load selected uri(s) into the text editor's buffer.
			uri_list = self.get_uris()
			self.__editor.instance_manager.open_files(uri_list)
		return

	def __dialog_show_dialog_cb(self, editor, dialog):
		if not dialog.get_property("name") in ["OpenDialog"]:
			return
		if not self.__editor.uri: return
		try:
			if self.__shortcut_folder_is_set is False:
				from gobject import GError
				from gnomevfs import URI
				self.add_shortcut_folder_uri(str(URI(self.__editor.uri).parent))
				self.__shortcut_folder_is_set = True
		except GError:
			self.__shortcut_folder_is_set = True
		self.select_uri(self.__editor.uri)
		return

	def __dialog_reset_flags_cb(self, *args):
		self.__shortcut_folder_is_set = False
		return

	def destroy_(self):
		self.__destroy_cb(self)
		return

	def __destroy_cb(self, dialog):
		"""
		Handles callback when the "delete" signal is emitted.

		@param self: Reference to the OpenDialog instance.
		@type self: An OpenDialog object.

		@param dialog: Reference to the OpenDialog instance.
		@type dialog: An OpenDialog object.
		"""
		from SCRIBES.utils import disconnect_signal, delete_attributes
		disconnect_signal(self.__signal_id_1, self.__editor)
		disconnect_signal(self.__signal_id_2, self.__editor)
		disconnect_signal(self.__signal_id_3, self.__editor)
		self.destroy()
		delete_attributes(self)
		del self
		self = None
		return
