# -*- coding: utf-8 -*-
# Copyright © 2005 Lateef Alabi-Oki
#
# This file is part of Scribes.
#
# Scribes is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Scribes is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Scribes; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

from gtk import FileChooserDialog

class SaveDialog(FileChooserDialog):
	"""
	This class instantiates a dialog that allows clients to change the name of
	a file and save it. It inherites from ScribesDialog and the
	gtk.FileChooserDialog classes. See dialog.py and the GTK+ manual
	respectively for the properties of each class.
	"""

	def __init__(self, editor):
		"""
		Initialize the text editor's save dialog.

		@param self: Reference to the SaveDialog instance.
		@type self: A SaveDialog object.

		@param window: A possible parent window.
		@type window: A gtk.Window object.
		"""
		FileChooserDialog.__init__(self)
		self.__init_attributes(editor)
		self.__set_dialog_properties()

	def __init_attributes(self, editor):
		"""
		Initialize the dialog attributes.

		@param self: Reference to the ScribesOpenDialog instance.
		@type self: A ScribesOpenDialog object.

		@param window: A possible parent window.
		@type window: A gtk.Window object.
		"""
		self.__editor = editor
		from SCRIBES.encodingbox import ScribesEncodingComboBox
		from SCRIBES.utils import create_encoding_box
		self.__encoding_box  = create_encoding_box(ScribesEncodingComboBox(editor))
		return

	def __set_dialog_properties(self):
		"""
		Set the dialog properties.

		@param self: Reference to the SaveDialog instance.
		@type self: A SaveDialog object.
		"""
		from gtk import FILE_CHOOSER_ACTION_SAVE
		self.set_action(FILE_CHOOSER_ACTION_SAVE)
		from i18n import msg0001
		self.set_property("title", msg0001)
		self.set_property("role", "scribes_save_dialog")
		self.set_property("icon-name", "stock_save")
		self.set_property("local-only", False)
		self.set_property("name", "SaveDialog")
		from gtk import STOCK_SAVE, STOCK_CANCEL, RESPONSE_OK, RESPONSE_CANCEL
		self.add_button(STOCK_CANCEL, RESPONSE_CANCEL)
		self.add_button(STOCK_SAVE, RESPONSE_OK)
		self.set_extra_widget(self.__encoding_box)
		self.set_default_response(RESPONSE_OK)
		self.set_transient_for(self.__editor.window)
		from SCRIBES.dialogfilter import create_filter_list
		for filter in create_filter_list():
			self.add_filter(filter)
		return

	def show_dialog(self):
		"""
		Show the text editor's open dialog.

		@param self: Reference to the SaveDialog instance.
		@type self: A SaveDialog object.
		"""
		self.__editor.emit("show-dialog", self)
		from i18n import msg0002
		status_id = self.__editor.feedback.set_modal_message(msg0002, "saveas")
		self.__set_current_folder_and_name()
		self.show_all()
		response = self.run()
		self.__editor.feedback.unset_modal_message(status_id)
		self.hide()
		self.__editor.emit("hide-dialog", self)
		from operator import eq
		from gtk import RESPONSE_OK
		if eq(response, RESPONSE_OK):
			# Rename the document.
			newuri = self.get_uri()
			self.__editor.emit("rename-document", newuri)
		return

################################################################################
#
#					FIXME:
#
################################################################################

	def __set_current_folder_and_name(self):
		"""
		Set the current folder and name in the save dialog.

		@param self: Reference to the SaveDialog instance.
		@type self: A SaveDialog object.
		"""
		if self.__editor.uri:
			from gnomevfs import URI
			self.set_current_folder_uri(str(URI(self.__editor.uri).parent))
			self.set_current_name(str(URI(self.__editor.uri).short_name))
		else:
			from i18n import msg0003
			self.set_current_name(msg0003)
			from SCRIBES.info import desktop_folder, home_folder
			try:
				self.set_current_folder(desktop_folder)
			except:
				self.set_current_folder(home_folder)
		return

	def destroy_(self):
		self.__destroy_cb(self)
		return

	def __destroy_cb(self, dialog):
		"""
		Handles callback when the "delete" signal is emitted.

		@param self: Reference to the SaveDialog instance.
		@type self: A SaveDialog object.

		@param dialog: Reference to the SaveDialog instance.
		@type dialog: A SaveDialog object.
		"""
		from SCRIBES.utils import delete_attributes
		self.destroy()
		delete_attributes(self)
		del self
		self = None
		return
