/*
 * Copyright (C) 2005 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/** Contain the functions for operating on the SensorsApplet structure
 *  (represents the applet itself, and its associated variables.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gnome.h>
#include <glib/gprintf.h>
#include "sensors-applet.h"
#include "sensors-applet-gconf.h"
#include "acpi-sensors-interface.h"
#include "i2c-sys-sensors-interface.h"
#include "i2c-proc-sensors-interface.h"
#include "i8k-sensors-interface.h"
#include "ibm-acpi-sensors-interface.h"
#include "omnibook-sensors-interface.h"
#include "pmu-sys-sensors-interface.h"
#include "hddtemp-sensors-interface.h"
#include "prefs-dialog.h"
#include "about-dialog.h"

#define SENSORS_APPLET_MENU_FILE "SensorsApplet.xml"
#define DEFAULT_TIMEOUT 2000

/* callbacks for panel menu */
static void prefs_cb(BonoboUIComponent *uic,
		     SensorsApplet    *sensors_applet,
		     const gchar       *verbname) {
	if (sensors_applet->prefs_dialog) {
		gtk_window_present(GTK_WINDOW(sensors_applet->prefs_dialog));
		return;
	}
	prefs_dialog_open(sensors_applet);
}

static void about_cb(BonoboUIComponent *uic,
		     SensorsApplet    *sensors_applet,
		     const gchar       *verbname) {
#ifndef HAVE_GTK_26
	if (sensors_applet->about_dialog) {
		gtk_window_present(GTK_WINDOW(sensors_applet->about_dialog));
		return;
	}
#endif /* HAVE_GTK_26 */
	about_dialog_open(sensors_applet);
}

static void help_cb (BonoboUIComponent *uic, 
		     SensorsApplet *sensors_applet,
		     const gchar *verbname) {
	   GError *error = NULL;
	   
	   gnome_help_display(PACKAGE, NULL,
			      &error);

	   if (error) {
			 g_printf("Could not open help document: %s\n ",error->message);
			 g_error_free (error);
	   }
}

static void destroy_cb(GtkWidget *widget, SensorsApplet *sensors_applet) {
	/* destory sensors tree and dialogs and finally the applet */
	if (sensors_applet->timeout_id) {
		g_source_remove(sensors_applet->timeout_id);
	}

	if (sensors_applet->sensors != NULL) {
		gtk_tree_store_clear(sensors_applet->sensors);
	}

	if (sensors_applet->prefs_dialog != NULL) {
		gtk_widget_destroy(GTK_WIDGET(sensors_applet->prefs_dialog));
	}
	gtk_widget_destroy(GTK_WIDGET(sensors_applet->applet));

	g_free(sensors_applet);
	return;
}

static void change_background_cb(PanelApplet *applet, 
				 PanelAppletBackgroundType type,
				 GdkColor *color, 
				 GdkPixmap *pixmap, 
				 SensorsApplet *sensors_applet) {
	GtkRcStyle *rc_style;
	GtkStyle *style;

	g_assert(applet == sensors_applet->applet);

	/* reset style */
	gtk_widget_set_style(GTK_WIDGET(applet), NULL);
	rc_style = gtk_rc_style_new();
	gtk_widget_modify_style(GTK_WIDGET(applet), rc_style);
	gtk_rc_style_unref(rc_style);

	switch(type) {
	case PANEL_COLOR_BACKGROUND:
		gtk_widget_modify_bg(GTK_WIDGET(applet),
				     GTK_STATE_NORMAL, color);
		break;

	case PANEL_PIXMAP_BACKGROUND:
		style = gtk_style_copy(GTK_WIDGET(applet)->style);
		if (style->bg_pixmap[GTK_STATE_NORMAL]) {
			g_object_unref(style->bg_pixmap[GTK_STATE_NORMAL]);
		}
		style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref(pixmap);
		gtk_widget_set_style(GTK_WIDGET(applet), style);
		g_object_unref(style);
		break;

	case PANEL_NO_BACKGROUND:
		/* fall through */
	default:
		break;
	}
}

static const BonoboUIVerb sensors_applet_menu_verbs[] = {
	BONOBO_UI_UNSAFE_VERB("Preferences", prefs_cb),
	BONOBO_UI_UNSAFE_VERB ("Help", help_cb),
	BONOBO_UI_UNSAFE_VERB("About", about_cb),
	BONOBO_UI_VERB_END
};


/* functions to be called by the different sensors intefaces */
void sensors_applet_register_sensors_interface(SensorsApplet *sensors_applet,
					       SensorInterface interface,
					       GetSensorValueFunction get_sensor_value_function) {
	sensors_applet->get_sensor_value[interface] = get_sensor_value_function;
}


   
  

gboolean sensors_applet_add_sensor_full_details(SensorsApplet *sensors_applet,
						const gchar *path, 
						const gchar *id, 
						const gchar *label, 
						SensorInterface interface, 
						SensorType type, 
						gboolean enable,
						gdouble alarm_value,
						AlarmType alarm_type,
						gboolean alarm_enable,
						const gchar *alarm_command,
						gint alarm_timeout,
						gdouble multiplier,
						gdouble offset,
						const gchar *icon_filename) {

					       
	GtkTreeIter interfaces_iter, sensors_iter;
	gboolean not_empty_tree;

	/* assign to UNUSED to ensure we dont accidently match an
	 * interface that doesnt actually exist */
	guint node_interface = UNUSED;
	gboolean not_end_of_interfaces = TRUE, interface_exists = FALSE;
	gboolean not_end_of_sensors = TRUE;
	gchar *sensor_id;
	GdkPixbuf *icon;
	GError *error = NULL;

	g_assert(sensors_applet != NULL);

	/* assume tree is not empty */
	not_empty_tree = TRUE;


	if (NULL == sensors_applet->sensors) {

		sensors_applet->sensors = gtk_tree_store_new(N_COLUMNS, 
							     G_TYPE_STRING, /* path */
							     G_TYPE_STRING, /* id */
							     G_TYPE_STRING, /* label */
							     G_TYPE_UINT, /* interface */
							     G_TYPE_UINT, /* sensor
									   * type */
							     G_TYPE_BOOLEAN, /* enable */
							     G_TYPE_BOOLEAN, /* visible */
							     G_TYPE_DOUBLE, /* alarm value */
							     G_TYPE_UINT, /* alarm type */
							     G_TYPE_BOOLEAN, /* alarm enable */
							     G_TYPE_STRING, /* alarm command */ 
							     G_TYPE_UINT, /* alarm timeout */
							     G_TYPE_INT, /* alarm
									  * timeout
									  * index */
							     G_TYPE_DOUBLE, /* multiplier */
							     G_TYPE_DOUBLE, /* offset */
							     G_TYPE_STRING, /* icon filename */
							     GDK_TYPE_PIXBUF); /* icon pixbuf */
		 
		

		/* we know tree is actually empty since we just created it */
		not_empty_tree = FALSE;
	}
	
	/* search sensor tree for the parent interface to place this
	 * sensor under */
	for (not_empty_tree = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(sensors_applet->sensors), &interfaces_iter); not_empty_tree && not_end_of_interfaces && !interface_exists; not_end_of_interfaces = gtk_tree_model_iter_next(GTK_TREE_MODEL(sensors_applet->sensors), &interfaces_iter)) {
		gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors), &interfaces_iter,
				   INTERFACE_COLUMN, &node_interface,
				   -1);
		if (interface == node_interface) {
			/* found interface in tree */
			interface_exists = TRUE;

			/* now see if this actual sensor already
			 * exists within this interface - don't want
			 * to add duplicates */
			/* see if have children */
			for (not_end_of_sensors = gtk_tree_model_iter_children(GTK_TREE_MODEL(sensors_applet->sensors), &sensors_iter,  &interfaces_iter); not_end_of_sensors; not_end_of_sensors = gtk_tree_model_iter_next(GTK_TREE_MODEL(sensors_applet->sensors), &sensors_iter)) {
				gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors), &sensors_iter,
						   ID_COLUMN, &sensor_id,
						   -1);
				if (g_ascii_strcasecmp(sensor_id, id) == 0) {
					/* sensor already exists so
					 * dont add a second time */
					g_free(sensor_id);
					return FALSE;
				}
				g_free(sensor_id);
			}
							    
			break;
		}
	}


	if (!interface_exists) {
		/* wasn't able to find interface root node so create it */
		gtk_tree_store_append(sensors_applet->sensors,
				      &interfaces_iter,
				      NULL);
		
		gtk_tree_store_set(sensors_applet->sensors,
				   &interfaces_iter,
				   ID_COLUMN, sensor_interface[interface],
				   INTERFACE_COLUMN, interface,
				   VISIBLE_COLUMN, FALSE,
				   -1);
		
	}

	/* try to load the icon */
	icon = gdk_pixbuf_new_from_file_at_size(icon_filename, DEFAULT_ICON_SIZE, DEFAULT_ICON_SIZE, &error);
	if (error) {
		 g_warning ("Could not load icon: %s\nTrying with default icon.\n", error->message);
		 g_error_free(error);
		 error = NULL;

		 /* try again with default name */
		 icon_filename = SENSORS_APPLET_ICON;
		 icon = gdk_pixbuf_new_from_file_at_size(icon_filename, DEFAULT_ICON_SIZE, DEFAULT_ICON_SIZE, &error);
		 if (error) {
			 g_warning ("Could not load icon: %s\n", error->message);
			 g_error_free(error);
			 error = NULL;
			 return;
		 }
	}

	/* then add sensor as a child under interface node - ie assume
	 * we either found it or created it - the inteface node that
	 * is */

	/* for now just add sensors all in a single list */
	gtk_tree_store_append(sensors_applet->sensors,
			      &sensors_iter,
			      &interfaces_iter);
	
	gtk_tree_store_set(sensors_applet->sensors,
			   &sensors_iter,
			   PATH_COLUMN, path,
			   ID_COLUMN, id,
			   LABEL_COLUMN, label,
			   INTERFACE_COLUMN, interface,
			   SENSOR_TYPE_COLUMN, type,
			   ENABLE_COLUMN, enable,
			   VISIBLE_COLUMN, TRUE,
			   ALARM_VALUE_COLUMN, alarm_value,
			   ALARM_TYPE_COLUMN, alarm_type,
			   ALARM_ENABLE_COLUMN, alarm_enable,
			   ALARM_COMMAND_COLUMN, alarm_command,
			   ALARM_TIMEOUT_COLUMN, alarm_timeout,
			   ALARM_TIMEOUT_INDEX_COLUMN, -1,
			   MULTIPLIER_COLUMN, multiplier,
			   OFFSET_COLUMN, offset,
			   ICON_FILENAME_COLUMN, icon_filename,
			   ICON_PIXBUF_COLUMN, icon,
			   -1);

	/* remove reference to icon as tree now has ref */
	g_object_unref(icon);

}	

/* path should be the full path to a file representing the sensor (eg
 * /dev/hda or /sys/devices/platform/i2c-0/0-0290/temp1_input) */
gboolean sensors_applet_add_sensor(SensorsApplet *sensors_applet,
				   const gchar *path, 
				   const gchar *id, 
				   const gchar *label, 
				   SensorInterface interface, 
				   gboolean enable,
				   SensorType type,
				   const gchar *icon_filename) {
	
	g_assert(sensors_applet != NULL);
	if (icon_filename == NULL) {
		icon_filename = SENSORS_APPLET_ICON;
	}

	return sensors_applet_add_sensor_full_details(sensors_applet,
						      path,
						      id,
						      label,
						      interface,
						      type,
						      enable,
						      0.0,
						      ALARM_WHEN_VALUE_GREATER_THAN_THRESHOLD,
						      FALSE,
						      "",
						      0,
						      1.0,
						      0.0,
						      icon_filename);
} 

	
	
/* internal helper functions for updating display etc*/
/* returns a constant string containing the pango fontsize, defaults
   to medium if error - this value is owned and cannot be freed by the
   caller */
static const gchar *sensors_applet_get_font_size(SensorsApplet *sensors_applet) {
	FontSize font_size;

	font_size = panel_applet_gconf_get_int(sensors_applet->applet, FONT_SIZE, NULL);
	switch (font_size) {
	case XX_LARGE:
		return XX_LARGE_TEXT;
	case X_LARGE:
		return X_LARGE_TEXT;
	case LARGE:
		return LARGE_TEXT;
	case MEDIUM:
		return MEDIUM_TEXT;
	case SMALL:
		return SMALL_TEXT;
	case X_SMALL:
		return X_SMALL_TEXT;
	case XX_SMALL:
		return XX_SMALL_TEXT;
	default:
		return MEDIUM_TEXT;
	}
}

 /* HELPER FUNCTIONS for managing labels and values lists */ 	 
 /* removes remainder of list links and frees widgets inside list */ 	 
 static void sensors_applet_crop_list(GSList **list, GSList **current_element) { 	 
         while (*current_element != NULL) { 	 
                 GSList *old_element = *current_element; 	 
                 gtk_widget_destroy(GTK_WIDGET(old_element->data)); 	 
                 *list = g_slist_remove_link(*list, old_element); 	 
                 *current_element = g_slist_next(*current_element); 	 
         } 	 
 } 	 
  	 
 /* take the given text and add if there is already a node at the
  * given element, change its text to this, or create a new node and
  * point current_element at the newly created node */ 	 
static void sensors_applet_add_label_to_list(GSList **list, GSList **current_element, const gchar *text) { 	 
	/* if have reached end of list, need to create a new node etc */ 	 
	if (*current_element == NULL) { 	 
		/* create a new label and add it to 	 
		   the list and the display */ 	 
		*list = g_slist_append(*list, g_object_new(GTK_TYPE_LABEL,
							   "use-markup", TRUE,
							   "label", text,
							   NULL)); 	 
  	 
		/* if we have just created the first 	 
		   node in the list, we will want to be 	 
		   pointing at it now */ 	 
		if (g_slist_length(*list) == 1) { 	 
			*current_element = *list; 	 
		} else { 	 
			/* if we just appended a new 	 
			   node, we will want to be 	 
			   pointing at this as the 	 
			   current node */ 	 
			*current_element = g_slist_next(*current_element); 	 
			
		} 	 
	} else {
		  gtk_label_set_label(GTK_LABEL((*current_element)->data), text);
	} 	 
}

static void sensors_applet_add_image_to_list(GSList **list, GSList **current_element, GdkPixbuf *pixbuf) { 	 

	GtkWidget *image;
         /* if have reached end of list, need to create a new node etc */ 	 
	if (*current_element == NULL) {
		/* create a new image and add it to 	 
		   the list and the display */ 	 
		 image = gtk_image_new_from_pixbuf(pixbuf);
                 *list = g_slist_append(*list, image); 	 
  	 
                 /* if we have just created the first 	 
                    node in the list, we will want to be 	 
                    pointing at it now */ 	 
                 if (g_slist_length(*list) == 1) { 	 
                         *current_element = *list; 	 
                 } else { 	 
                         /* if we just appended a new 	 
                            node, we will want to be 	 
                            pointing at this as the 	 
                            current node */ 	 
                         *current_element = g_slist_next(*current_element); 	 
  	 
                 } 	 
         } else { 	 
		 /* and set to new one if needed */
		 if (gtk_image_get_pixbuf(GTK_IMAGE((*current_element)->data)) != pixbuf) {
			 gtk_image_set_from_pixbuf(GTK_IMAGE((*current_element)->data),
						   pixbuf);
		 }
		 
		 
         } 	 
}

static gboolean sensors_applet_execute_alarm(const gchar *alarm_command) {
	gnome_execute_shell(NULL, alarm_command);
	return TRUE;
}

static guint sensors_applet_add_alarm(SensorsApplet *sensors_applet, const gchar *alarm_command, guint alarm_timeout) {
	gchar *command = g_strdup(alarm_command);
	sensors_applet->alarm_commands = g_slist_prepend(sensors_applet->alarm_commands, command);
	return g_timeout_add(alarm_timeout,
			     (GSourceFunc)sensors_applet_execute_alarm,
			     command);
}

static void sensors_applet_alarm_on(SensorsApplet *sensors_applet, GtkTreeIter *iter) {
	gchar *alarm_command;
	gint alarm_timeout_index;
	guint alarm_timeout;
	gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors),
			   iter,
			   ALARM_TIMEOUT_INDEX_COLUMN, &alarm_timeout_index,
			   -1);
	
	if (alarm_timeout_index == -1) {
		/* alarm is not currently on */

		gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors),
				   iter,
				   ALARM_COMMAND_COLUMN, &alarm_command,
				   ALARM_TIMEOUT_COLUMN, &alarm_timeout,
				   -1);
		/* execute alarm once, then add to time to
		   keep repeating it */
		gnome_execute_shell(NULL, alarm_command);
		if (alarm_timeout <= 0) {
			alarm_timeout_index = sensors_applet_add_alarm(sensors_applet, alarm_command, G_MAXINT);
		} else {
			alarm_timeout_index = sensors_applet_add_alarm(sensors_applet, alarm_command, alarm_timeout * 1000);
		}
		gtk_tree_store_set(sensors_applet->sensors,
				   iter,
				   ALARM_TIMEOUT_INDEX_COLUMN, alarm_timeout_index,
				   -1);
		g_free(alarm_command);
	} 
}

/* needs to be able to be called by the config dialog when the alarm
 * command changes */
void sensors_applet_alarm_off(SensorsApplet *sensors_applet, GtkTreeIter *iter) {
	gint alarm_timeout_index;
	gchar *alarm_command;
	GSList *current_alarm_command;

	gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors),
			   iter,
			   ALARM_TIMEOUT_INDEX_COLUMN, &alarm_timeout_index,
			   -1);
	if (alarm_timeout_index != -1) {
		/* alarm is on, cycle through list of alarm_commands
		 * and free this one */

		gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors),
				   iter,
				   ALARM_COMMAND_COLUMN, &alarm_command,
				   -1);
		for (current_alarm_command = sensors_applet->alarm_commands; current_alarm_command != NULL; current_alarm_command = g_slist_next(current_alarm_command)) {
			if (g_strrstr(alarm_command, ((gchar *)(current_alarm_command->data))) == alarm_command) {
				g_free(current_alarm_command->data);
				sensors_applet->alarm_commands = g_slist_remove_link(sensors_applet->alarm_commands, current_alarm_command);
				g_slist_free_1(current_alarm_command);
				break;
			}
		}
		g_free(alarm_command);
		if (!g_source_remove(alarm_timeout_index)) {
			g_printf("Error removing alarm source\n");
		}
		gtk_tree_store_set(sensors_applet->sensors,
				   iter,
				   ALARM_TIMEOUT_INDEX_COLUMN, -1,
				   -1);
	}
}

/* should be called as a g_container_foreach at the start of
 * pack_display if ythe table already exists to remove but keep alive
 * all children of the table before repacking it */
static void sensors_applet_pack_display_empty_table_cb(GtkWidget *widget,
						   gpointer data) {
	GtkContainer *container = GTK_CONTAINER(data);

	/* ref then remove widget */
	g_object_ref(widget);
	gtk_container_remove(container, widget);
}

/* should be called as a g_container_foreach at the end of
 * pack_display to unref any of the old children that we have readdded
 * to the table to stop reference creep from the g_object_ref called
 * on each child at the start of pack labels */
static void sensors_applet_pack_display_cleanup_refs_cb(GtkWidget *widget,
							gpointer data) {
						  
	GList *old_children = (GList *)data;
	if (g_list_find(old_children, widget)) {
		g_object_unref(widget);
	}
}
						 
static void sensors_applet_pack_display(SensorsApplet *sensors_applet) {
	static GtkLabel *no_sensors_enabled_label = NULL;
	int num_active_sensors = 0, num_sensors_per_group, cols, rows, i, j;
	GList *old_table_children = NULL;
	GList *current_child;

	GSList *current_value;
	GSList *current_label;
	GSList *current_icon;
	gint display_mode;

	num_active_sensors = g_slist_length(sensors_applet->values);
	display_mode = panel_applet_gconf_get_int(sensors_applet->applet, DISPLAY_MODE, NULL);

	num_sensors_per_group = panel_applet_gconf_get_int(sensors_applet->applet, NUM_SENSORS, NULL);

	if (panel_applet_get_orient(sensors_applet->applet) == PANEL_APPLET_ORIENT_UP || panel_applet_get_orient(sensors_applet->applet) == PANEL_APPLET_ORIENT_DOWN) {
		/* if oriented horizontally, want as many
		   sensors per column as user has defined, then
		   enough columns to hold all the values */
		rows = num_sensors_per_group;
		cols = num_active_sensors / num_sensors_per_group;
		while (rows * cols < num_active_sensors || cols == 0) {
			cols++;
		}
		
	} else {
		/* if oriented vertically, want as many
		   sensors per row as user has defined, then
		   enough rows to hold all the labels */
		cols = num_sensors_per_group;
		rows = num_active_sensors / num_sensors_per_group;
		while (rows * cols < num_active_sensors || rows == 0) {
			rows++;
		}
		
	}

	/* if displaying labels need to modify number of rows / colums
	   to accomodate this */
	 if (display_mode != DISPLAY_NONE) {
		 if (panel_applet_gconf_get_bool(sensors_applet->applet, LABELS_INLINE, NULL)) {
			 /* to display labels next to values need twice
			    as many columns */
			 cols *= 2;
		 } else {
			 /* to display labels above values, we need
			  * twice as many rows as without */
			 rows *= 2;
		 }
	 }	 

	if (sensors_applet->table == NULL) {
		/* create table and add to applet */
		sensors_applet->table = gtk_table_new(rows, cols, FALSE);
		gtk_table_set_col_spacings(GTK_TABLE(sensors_applet->table), 3);
		gtk_table_set_row_spacings(GTK_TABLE(sensors_applet->table), 0);
		gtk_container_add(GTK_CONTAINER(sensors_applet->applet), sensors_applet->table);
	} else {
		/* remove all children if table already exists so we can start
		 * again */
		/* save a list of the old children for later */
		old_table_children = gtk_container_get_children(GTK_CONTAINER(sensors_applet->table));

		gtk_container_foreach(GTK_CONTAINER(sensors_applet->table),
				      sensors_applet_pack_display_empty_table_cb,
				      sensors_applet->table);

		/* then resize table */
		gtk_table_resize(GTK_TABLE(sensors_applet->table), rows, cols);
	}
	
	/* if no active sensors, display a label to this effect and return */
	if (num_active_sensors == 0) {
		if (no_sensors_enabled_label == NULL) {
			no_sensors_enabled_label = g_object_new(GTK_TYPE_LABEL,
								"label", _("No sensors enabled!"),
								NULL);
		}
		gtk_table_attach_defaults(GTK_TABLE(sensors_applet->table),
					  GTK_WIDGET(no_sensors_enabled_label),
					  0, 1,
					  0, 1);

	}

	current_value = sensors_applet->values;
	current_label = sensors_applet->labels;
	current_icon = sensors_applet->icons;

	/* pack icons / labels and values into table */
	
	/* if showing labels need to pack these first */
	if (display_mode != DISPLAY_NONE) {
		/* loop through columns */
		for (i = 0; current_label != NULL && current_value != NULL && current_icon != NULL && i < cols; /* increments depends on how we lay them out - see below */) {
			
			/* loop through rows in a column */
			for (j = 0; current_label != NULL && current_value != NULL && current_icon != NULL && j < rows; /* see bottom of for loop*/) {
				/* attach label / icon at this point */
				if (display_mode == DISPLAY_ICON) {
					gtk_table_attach_defaults(GTK_TABLE(sensors_applet->table),
								  GTK_WIDGET(current_icon->data),
							  i, i + 1,
							  j, j + 1);
				} else {
					gtk_table_attach_defaults(GTK_TABLE(sensors_applet->table),
								  GTK_WIDGET(current_label->data),
							  i, i + 1,
							  j, j + 1);
				}				
				/* now attach sensor value to either
				   row below or column next to */
				if (panel_applet_gconf_get_bool(sensors_applet->applet, LABELS_INLINE, NULL)) { 
					/* left align labels */
                                        gtk_misc_set_alignment(GTK_MISC(current_icon->data), 0.0, 0.5);
					gtk_misc_set_alignment(GTK_MISC(current_label->data), 0.0, 0.5); 	 
					gtk_misc_set_alignment(GTK_MISC(current_value->data), 0.0, 0.5);
 


					 /* place value next to label */
					gtk_table_attach_defaults(GTK_TABLE(sensors_applet->table),
								  GTK_WIDGET(current_value->data),
								  i + 1, i + 2,
								  j, j + 1);
					
					j++;
				} else { /* place value below label */
					/* center align labels */ 	 
					gtk_misc_set_alignment(GTK_MISC(current_icon->data), 0.5, 0.5); 	 
					gtk_misc_set_alignment(GTK_MISC(current_label->data), 0.5, 0.5); 	 
					gtk_misc_set_alignment(GTK_MISC(current_value->data), 0.5, 0.5); 	 
 
					gtk_table_attach_defaults(GTK_TABLE(sensors_applet->table),
								  GTK_WIDGET(current_value->data),
								  i, i + 1,
								  j + 1, j + 2);
					j += 2;
				}
				current_label = g_slist_next(current_label);
				current_icon = g_slist_next(current_icon);
				current_value = g_slist_next(current_value);

			} /* end row loop */
			/* now increment column index as needed */
			if (panel_applet_gconf_get_bool(sensors_applet->applet, LABELS_INLINE, NULL)) { /* place value next to label */
				i += 2;
			} else {
				i++;
			}
			
			
		} /* end column loop	*/

		
	} else { /* not showing labels or icons so just pack values */
		for (i = 0; current_value != NULL && i < cols; ++i) {
			for (j = 0; current_value!= NULL && j < rows; ++j) {
				gtk_table_attach_defaults(GTK_TABLE(sensors_applet->table),
							  GTK_WIDGET(current_value->data),
							  i, i + 1,
							  j, j + 1);

				current_value = g_slist_next(current_value);
			}
		}
		
	}
	if (old_table_children != NULL) {
		gtk_container_foreach(GTK_CONTAINER(sensors_applet->table),
				      sensors_applet_pack_display_cleanup_refs_cb,
				      old_table_children);
		g_list_free(old_table_children);
	}

} 	    

static void sensors_applet_update_display_only(SensorsApplet *sensors_applet) {
	sensors_applet_update_display(sensors_applet,
				      FALSE);
}

void sensors_applet_update_display(SensorsApplet *sensors_applet,
				   gboolean repack_display) {
	/* walk through tree structure and for each enabled sensor
	 * call it's get_sensor_value function and then update labels
	 * etc */

	gchar *tooltip;
	gint number_of_active_sensors = 0;
	static gint last_number_of_active_sensors = 0;

	/* to iterate through the sensors data structure */
	GtkTreeIter interfaces_iter, sensors_iter;
	gboolean not_end_of_interfaces = TRUE, not_end_of_sensors = TRUE;

	gchar *sensor_path = NULL;
	gchar *sensor_id = NULL;
	gchar *sensor_label = NULL;
	SensorType sensor_type;
	SensorInterface sensor_interface;
	gboolean sensor_enabled;
	gdouble sensor_alarm_value;
	gboolean sensor_alarm_enabled;
	AlarmType sensor_alarm_type;
	gdouble sensor_multiplier;
	gdouble sensor_offset;
	gdouble sensor_value;

	/* to build the list of labels as we go */
	gchar *label_text = NULL;
	gchar *value_text = NULL;
	const gchar *font_size;
	gchar *old_value_text;
	GdkPixbuf *icon;

	GSList *current_label = NULL;
	GSList *current_value = NULL;
	GSList *current_icon = NULL;
	
	GError *error = NULL;

	g_assert(sensors_applet->sensors != NULL);

	/* get the first label in the list */
	current_label = sensors_applet->labels;
	current_value = sensors_applet->values;
	current_icon = sensors_applet->icons;
	
	/* now step through the GtkTreeStore sensors to
	   find which sensors are enabled */
	for (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(sensors_applet->sensors), &interfaces_iter); not_end_of_interfaces; not_end_of_interfaces = gtk_tree_model_iter_next(GTK_TREE_MODEL(sensors_applet->sensors), &interfaces_iter)) {
		/* reset sensors sentinel */
		not_end_of_sensors = TRUE;
		for (gtk_tree_model_iter_children(GTK_TREE_MODEL(sensors_applet->sensors), &sensors_iter, &interfaces_iter); not_end_of_sensors; not_end_of_sensors = gtk_tree_model_iter_next(GTK_TREE_MODEL(sensors_applet->sensors), &sensors_iter)) {
			gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors), 
					   &sensors_iter,
					   PATH_COLUMN, &sensor_path,
					   ID_COLUMN, &sensor_id,
					   LABEL_COLUMN, &sensor_label,
					   INTERFACE_COLUMN, &sensor_interface,
					   SENSOR_TYPE_COLUMN, &sensor_type,
					   ENABLE_COLUMN, &sensor_enabled,
					   ALARM_VALUE_COLUMN, &sensor_alarm_value,
					   ALARM_TYPE_COLUMN, &sensor_alarm_type,
					   ALARM_ENABLE_COLUMN, &sensor_alarm_enabled,
					   MULTIPLIER_COLUMN, &sensor_multiplier,
					   OFFSET_COLUMN, &sensor_offset,
					   ICON_PIXBUF_COLUMN, &icon,
					   -1);
			
			if (sensor_enabled) {
				/* get font size */
				font_size = sensors_applet_get_font_size(sensors_applet);
				
				sensor_value = sensors_applet->get_sensor_value[sensor_interface](sensor_path,
												  sensor_id,
												  sensor_type,
												  &error);
				
				
				if (error) {
					value_text = g_strdup(error->message);
					g_error_free(error);
					error = NULL;
				} else { 
					switch (sensor_type) {
					case TEMP_SENSOR: 
						if (panel_applet_gconf_get_bool(sensors_applet->applet, FARENHEIT, NULL)) {
							sensor_value = (9.0 * sensor_value / 5.0) + 32.0;
							sensor_value = (sensor_value * sensor_multiplier) + sensor_offset;
							value_text = g_strdup_printf("%2.0f%s", sensor_value, (panel_applet_gconf_get_bool(sensors_applet->applet, SHOW_UNITS, NULL) ? "\302\260F" : ""));
						} else {
							sensor_value = (sensor_value * sensor_multiplier) + sensor_offset;
							value_text = g_strdup_printf("%2.0f%s", sensor_value, (panel_applet_gconf_get_bool(sensors_applet->applet, SHOW_UNITS, NULL) ? "\302\260C" : ""));
						}
						break;
						
					case FAN_SENSOR:
						sensor_value = (sensor_value * sensor_multiplier) + sensor_offset;
						value_text = g_strdup_printf("%4.0f%s", sensor_value, (panel_applet_gconf_get_bool(sensors_applet->applet, SHOW_UNITS, NULL) ? _("RPM") : ""));
						break;
						
					case VOLTAGE_SENSOR:
						sensor_value = (sensor_value * sensor_multiplier) + sensor_offset;
						value_text = g_strdup_printf("%4.2f%s", sensor_value, (panel_applet_gconf_get_bool(sensors_applet->applet, SHOW_UNITS, NULL) ? _("V") : ""));
						break;
						
					case CURRENT_SENSOR:
						sensor_value = (sensor_value * sensor_multiplier) + sensor_offset;
						value_text = g_strdup_printf("%4.2f%s", sensor_value, (panel_applet_gconf_get_bool(sensors_applet->applet, SHOW_UNITS, NULL) ? _("A") : ""));
						break;
						
					default:
						g_assert_not_reached();	
						
					} /* end switch(sensor_type) */
				        old_value_text = value_text;
					
					if (sensor_alarm_enabled) {
						switch (sensor_alarm_type) {
						case ALARM_WHEN_VALUE_GREATER_THAN_THRESHOLD:
							if (sensor_value >= sensor_alarm_value) {
								/* make value
								 * text red with
								 correct font
								 size and do
								 alarm command */
								value_text = g_markup_printf_escaped("<span foreground=\"#FF0000\" size=\"%s\">%s</span>", font_size, old_value_text);
								sensors_applet_alarm_on(sensors_applet, &sensors_iter);
							} else {
								/* just do font size */
								value_text = g_markup_printf_escaped("<span size=\"%s\">%s</span>", font_size, old_value_text);
								sensors_applet_alarm_off(sensors_applet, &sensors_iter);
							}				
							break;
						case ALARM_WHEN_VALUE_LESS_THAN_THRESHOLD:
							if (sensor_value <= sensor_alarm_value) {
								/* make value
								 * text red with
								 correct font
								 size and do
								 alarm command */
								value_text = g_markup_printf_escaped("<span foreground=\"#FF0000\" size=\"%s\">%s</span>", font_size, old_value_text);
								sensors_applet_alarm_on(sensors_applet, &sensors_iter);
						} else {
								/* just do font size */
								value_text = g_markup_printf_escaped("<span size=\"%s\">%s</span>", font_size, old_value_text);
								sensors_applet_alarm_off(sensors_applet, &sensors_iter);
							}				
							break;
						default: 
							g_assert_not_reached();
						} /* end
						   * switch(sensor_alarm_type) */

					} else {
						/* just do font size */
						value_text = g_markup_printf_escaped("<span size=\"%s\">%s</span>", font_size, old_value_text);
						sensors_applet_alarm_off(sensors_applet, &sensors_iter);
					}
					
					g_free(old_value_text);
				} /* end of else from if (error) */
				sensors_applet_add_label_to_list(&(sensors_applet->values), &current_value, value_text);
				g_free(value_text);
				
				/* advance to next element in list */
				current_value = g_slist_next(current_value);
				
				/* create label text with font size
				 * formatting */
				label_text = g_markup_printf_escaped("<span size=\"%s\">%s</span>", font_size, sensor_label);
				sensors_applet_add_label_to_list(&(sensors_applet->labels), &current_label, label_text);
				g_free(label_text);
				current_label = g_slist_next(current_label);

				/* add icon to icons list */
				sensors_applet_add_image_to_list(&(sensors_applet->icons), &current_icon, icon);
				/* g_object_unref(icon) */
				current_icon = g_slist_next(current_icon);
				
			} /* end if (sensor_enabled) */
			g_free(sensor_path);
			g_free(sensor_id);
			g_free(sensor_label);
			g_object_unref(icon);

		} /* end for loop of sensors */
	} /* end for loop of interfaces */
	
	/* now remove any remaining elements in either list to free up
	   memory */
	sensors_applet_crop_list(&(sensors_applet->values), &current_value);
	sensors_applet_crop_list(&(sensors_applet->labels), &current_label);
	sensors_applet_crop_list(&(sensors_applet->icons), &current_icon);
	
	number_of_active_sensors = g_slist_length(sensors_applet->values);

	/* now repack labels into table update display of applets */
	if (repack_display || last_number_of_active_sensors != number_of_active_sensors) {
		sensors_applet_pack_display(sensors_applet);
	}
	last_number_of_active_sensors = number_of_active_sensors;


	tooltip = g_strdup_printf("%s\n%d %s",
				  _("Sensors Applet"),
				  number_of_active_sensors,
				  ngettext("sensor enabled", 
					   "sensors enabled",
					   number_of_active_sensors));
	gtk_tooltips_set_tip(sensors_applet->tooltips, GTK_WIDGET(sensors_applet->applet), tooltip, "");
	g_free(tooltip);

	gtk_widget_show_all(GTK_WIDGET(sensors_applet->applet));

	return;

}

static void sensors_applet_write_defaults_to_gconf(SensorsApplet *sensors_applet) {
	panel_applet_gconf_set_bool(sensors_applet->applet, FARENHEIT, FALSE, NULL);
	panel_applet_gconf_set_int(sensors_applet->applet, NUM_SENSORS, 1, NULL);
	panel_applet_gconf_set_int(sensors_applet->applet, DISPLAY_MODE, DISPLAY_LABEL, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, LABELS_INLINE, TRUE, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, SHOW_UNITS, TRUE, NULL);
	panel_applet_gconf_set_int(sensors_applet->applet, TIMEOUT, DEFAULT_TIMEOUT, NULL);
	panel_applet_gconf_set_int(sensors_applet->applet, FONT_SIZE, MEDIUM, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, IS_SETUP, FALSE, NULL);

}

static void sensors_applet_setup_sensors_interfaces(SensorsApplet *sensors_applet) {
	acpi_sensors_interface_init(sensors_applet);
	hddtemp_sensors_interface_init(sensors_applet);
	i2c_proc_sensors_interface_init(sensors_applet);
	i2c_sys_sensors_interface_init(sensors_applet);
	i8k_sensors_interface_init(sensors_applet);
	ibm_acpi_sensors_interface_init(sensors_applet);
	omnibook_sensors_interface_init(sensors_applet);
	pmu_sys_sensors_interface_init(sensors_applet);
}


void sensors_applet_init(SensorsApplet *sensors_applet) {
	
	g_assert(sensors_applet);
	g_assert(sensors_applet->applet);

	panel_applet_set_flags(sensors_applet->applet, PANEL_APPLET_EXPAND_MINOR);
	panel_applet_setup_menu_from_file(sensors_applet->applet,
					  DATADIR,
					  SENSORS_APPLET_MENU_FILE,
					  NULL,
					  sensors_applet_menu_verbs,
					  sensors_applet);

	g_signal_connect(sensors_applet->applet, 
			 "change_background",
			 G_CALLBACK(change_background_cb), 
			 sensors_applet);

	g_signal_connect(sensors_applet->applet, "destroy",
			 G_CALLBACK(destroy_cb),
			 sensors_applet);

		
	/* if not setup, write defaults to gconf */
	if (!panel_applet_gconf_get_bool(sensors_applet->applet, IS_SETUP, NULL)) {
		sensors_applet_write_defaults_to_gconf(sensors_applet);
	} else {
		/* setup sensors from stored gconf values */
		sensors_applet_gconf_setup_sensors(sensors_applet);
	}
	/* now do any setup needed manually */
	sensors_applet_setup_sensors_interfaces(sensors_applet);

		
	/* should have created sensors tree above, but if have
	   not was because we couldn't find any sensors */
	if (NULL == sensors_applet->sensors) {
		GtkWidget *label;	
		label = gtk_label_new(_("No sensors found!"));
		gtk_container_add(GTK_CONTAINER(sensors_applet->applet), label);
		gtk_widget_show_all(GTK_WIDGET(sensors_applet->applet));
		return;
	}
	

	sensors_applet->tooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(sensors_applet->tooltips, GTK_WIDGET(sensors_applet->applet), _("Sensors Applet"), "");
	sensors_applet_update_display(sensors_applet, TRUE);

	sensors_applet->timeout_id = g_timeout_add(panel_applet_gconf_get_int(sensors_applet->applet, TIMEOUT, NULL), (GSourceFunc)sensors_applet_update_display_only, sensors_applet);

	gtk_widget_show_all(GTK_WIDGET(sensors_applet->applet));
}



